;;; GNU Guix --- Functional package management for GNU
;;; Copyright © 2013, 2014, 2015 Ludovic Courtès <ludo@gnu.org>
;;; Copyright © 2015 Eric Bavier <bavier@member.fsf.org>
;;;
;;; This file is part of GNU Guix.
;;;
;;; GNU Guix is free software; you can redistribute it and/or modify it
;;; under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 3 of the License, or (at
;;; your option) any later version.
;;;
;;; GNU Guix is distributed in the hope that it will be useful, but
;;; WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with GNU Guix.  If not, see <http://www.gnu.org/licenses/>.

(define-module (gnu packages imagemagick)
  #:use-module (guix packages)
  #:use-module (guix build-system gnu)
  #:use-module (guix build-system perl)
  #:use-module (guix download)
  #:use-module ((guix licenses) #:select (fsf-free))
  #:use-module (gnu packages)
  #:use-module (gnu packages algebra)
  #:use-module (gnu packages compression)
  #:use-module (gnu packages fontutils)
  #:use-module (gnu packages ghostscript)
  #:use-module (gnu packages graphviz)
  #:use-module (gnu packages gtk)
  #:use-module (gnu packages image)
  #:use-module (gnu packages pkg-config)
  #:use-module (gnu packages xml)
  #:use-module (gnu packages xorg))

(define-public imagemagick
  (package
    (name "imagemagick")
    (version "6.9.2-1")
    (source (origin
             (method url-fetch)
             (uri (string-append "mirror://imagemagick/ImageMagick-"
                                 version ".tar.xz"))
             (sha256
              (base32
               "159afhqrj22jlz745ccbgnkdiwvn8pjcc96jic0iv9ms7gqxwln5"))
             (patches (list (search-patch "imagemagick-test-segv.patch")))))
    (build-system gnu-build-system)
    (arguments
     `(#:configure-flags '("--with-frozenpaths")
       #:phases (modify-phases %standard-phases
                  (add-before
                   'build 'pre-build
                   (lambda* (#:key outputs #:allow-other-keys)
                     (substitute* "Makefile"
                       ;; Clear the `LIBRARY_PATH' setting, which otherwise
                       ;; interferes with our own use.
                       (("^LIBRARY_PATH[[:blank:]]*=.*$")
                        "")

                       ;; Since the Makefile overrides $docdir, modify it to
                       ;; refer to what we want.
                       (("^DOCUMENTATION_PATH[[:blank:]]*=.*$")
                        (let ((doc (assoc-ref outputs "doc")))
                          (string-append "DOCUMENTATION_PATH = "
                                         doc "/share/doc/"
                                         ,name "-" ,version "\n"))))))
                  (add-before
                   'configure 'strip-configure-xml
                   (lambda _
                     (substitute* "config/configure.xml.in"
                       ;; Do not record 'configure' arguments in the
                       ;; configure.xml file that gets installed: That would
                       ;; include --docdir, and thus retain a reference to the
                       ;; 'doc' output.
                       (("@CONFIGURE_ARGS@")
                        "not recorded")))))))
    ;; TODO: Add Jasper etc.
    (inputs `(("fftw" ,fftw)
              ("graphviz" ,graphviz)
              ("ghostscript" ,ghostscript)
              ("lcms" ,lcms)
              ("libx11" ,libx11)
              ("zlib" ,zlib)
              ("libxml2" ,libxml2)
              ("libtiff" ,libtiff)
              ("libpng" ,libpng)
              ("libjpeg" ,libjpeg)
              ("pango" ,pango)
              ("freetype" ,freetype)
              ("bzip2" ,bzip2)
              ("xz" ,xz)))
    (native-inputs `(("pkg-config" ,pkg-config)))
    (outputs '("out"
               "doc"))                          ; 26 MiB of HTML documentation
    (home-page "http://www.imagemagick.org/")
    (synopsis "Create, edit, compose, or convert bitmap images")
    (description
     "ImageMagick® is a software suite to create, edit, compose, or convert
bitmap images.  It can read and write images in a variety of formats (over 100)
including DPX, EXR, GIF, JPEG, JPEG-2000, PDF, PhotoCD, PNG, Postscript, SVG,
and TIFF.  Use ImageMagick to resize, flip, mirror, rotate, distort, shear and
transform images, adjust image colors, apply various special effects, or draw
text, lines, polygons, ellipses and Bézier curves.")
    (license (fsf-free "http://www.imagemagick.org/script/license.php"))))

(define-public perl-image-magick
  (package
    (name "perl-image-magick")
    (version "6.89")
    (source
     (origin
       (method url-fetch)
       (uri (string-append "mirror://cpan/authors/id/J/JC/JCRISTY/"
                           "PerlMagick-" version "-1.tar.gz"))
       (sha256
        (base32
         "0n9afy1z5bhf9phrbahnkwhgcmijn8jggpbzwrivw1zhliliiy68"))))
    (build-system perl-build-system)
    (native-inputs `(("pkg-config" ,pkg-config)))
    (inputs `(("imagemagick" ,imagemagick)))
    (arguments
     `(#:phases
       (modify-phases %standard-phases
         (add-before
          'configure 'image-magick-flags
          (lambda* (#:key inputs #:allow-other-keys)
            (let ((im (assoc-ref inputs "imagemagick")))
              (substitute* "Makefile.PL"
                (("my \\$INC_magick = .*")
                 "my $INC_magick = `pkg-config --cflags ImageMagick`;\n")
                (("my \\$LIBS_magick = .*")
                 "my $LIBS_magick = `pkg-config --libs ImageMagick`;\n")))))
         (add-before
          'check 'skip-mpeg-tests
          (lambda _
            ;; TODO: MPEG tests fail even though our imagemagick supports
            ;; MPEG.  Has been reported elsewhere,
            ;; http://www.imagemagick.org/discourse-server/viewtopic.php?f=7&t=25036,
            ;; so skip for now.
            (delete-file "t/mpeg/read.t"))))))
    (home-page "http://search.cpan.org/dist/PerlMagick")
    (synopsis "Perl interface to ImageMagick")
    (description "This Perl extension allows the reading, manipulation and
writing of a large number of image file formats using the ImageMagick library.
Use it to create, edit, compose, or convert bitmap images from within a Perl
script.")
    ;; See Magick.pm
    (license (package-license imagemagick))))
