/**
 * <copyright> 
 * 
 * Copyright (c) 2004-2005 IBM Corporation and others. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License - v 1.0 
 * which accompanies this distribution, and is available at 
 * http://opensource.org/licenses/eclipse-1.0.txt 
 * 
 * Contributors: 
 *   IBM - Initial API and implementation 
 * 
 * </copyright> 
 * 
 * $Id: OWLXMLSaverImpl.java,v 1.4 2007/03/18 10:24:39 lzhang Exp $
 */

package org.eclipse.eodm.owl.resource;

import java.io.IOException;
import java.io.Writer;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Vector;

import org.eclipse.emf.common.util.EList;
import org.eclipse.eodm.exceptions.UnsupportedViewTypeException;
import org.eclipse.eodm.owl.owlbase.AllValuesFromRestriction;
import org.eclipse.eodm.owl.owlbase.CardinalityRestriction;
import org.eclipse.eodm.owl.owlbase.ComplementClass;
import org.eclipse.eodm.owl.owlbase.EnumeratedClass;
import org.eclipse.eodm.owl.owlbase.HasValueRestriction;
import org.eclipse.eodm.owl.owlbase.Individual;
import org.eclipse.eodm.owl.owlbase.IntersectionClass;
import org.eclipse.eodm.owl.owlbase.MaxCardinalityRestriction;
import org.eclipse.eodm.owl.owlbase.MinCardinalityRestriction;
import org.eclipse.eodm.owl.owlbase.OWLAnnotationProperty;
import org.eclipse.eodm.owl.owlbase.OWLClass;
import org.eclipse.eodm.owl.owlbase.OWLDataRange;
import org.eclipse.eodm.owl.owlbase.OWLDatatypeProperty;
import org.eclipse.eodm.owl.owlbase.OWLObjectProperty;
import org.eclipse.eodm.owl.owlbase.OWLOntology;
import org.eclipse.eodm.owl.owlbase.OWLOntologyProperty;
import org.eclipse.eodm.owl.owlbase.OWLRestriction;
import org.eclipse.eodm.owl.owlbase.SomeValuesFromRestriction;
import org.eclipse.eodm.owl.owlbase.UnionClass;
import org.eclipse.eodm.rdf.rdfbase.BlankNode;
import org.eclipse.eodm.rdf.rdfbase.PlainLiteral;
import org.eclipse.eodm.rdf.rdfbase.RDFProperty;
import org.eclipse.eodm.rdf.rdfbase.RDFSLiteral;
import org.eclipse.eodm.rdf.rdfbase.RDFSResource;
import org.eclipse.eodm.rdf.rdfbase.RDFStatement;
import org.eclipse.eodm.rdf.rdfbase.RDFXMLLiteral;
import org.eclipse.eodm.rdf.rdfbase.TypedLiteral;
import org.eclipse.eodm.rdf.rdfbase.URIReference;
import org.eclipse.eodm.rdf.rdfbase.URIReferenceNode;
import org.eclipse.eodm.rdf.rdfs.RDFAlt;
import org.eclipse.eodm.rdf.rdfs.RDFBag;
import org.eclipse.eodm.rdf.rdfs.RDFSClass;
import org.eclipse.eodm.rdf.rdfs.RDFSContainer;
import org.eclipse.eodm.rdf.rdfs.RDFSeq;
import org.eclipse.eodm.rdf.rdfweb.Document;
import org.eclipse.eodm.rdf.rdfweb.NamespaceDefinition;
import org.eclipse.eodm.util.Triple;
import org.eclipse.eodm.vocabulary.OWL;
import org.eclipse.eodm.vocabulary.RDF;
import org.eclipse.eodm.vocabulary.RDFS;

/**
 * OWLXMLSaverImpl
 */
public class OWLXMLSaverImpl {
    public static final String OWL_PREFIX = "owl:";

    public static final String RDF_PREFIX = "rdf:";

    public static final String RDFS_PREFIX = "rdfs:";

    //the output writer
    private Writer writer = null;

    //the ontology to be saved
    private OWLOntology ontology = null;
    private Document document=null;

    /**
     * Store and search all namespaces
     */
    protected Hashtable NamespaceHashTable = new Hashtable();
    public OWLXMLSaverImpl(Writer writer, Document doc) {
        super();
        this.writer = writer;
        this.document = doc;
        for (Iterator iter = document.getNamespaceDefinition().iterator(); iter.hasNext();) {
            NamespaceDefinition namespace = (NamespaceDefinition) iter.next();
            NamespaceHashTable.put(namespace.getNamespace().getNamespaceURIRef().getURIString(), namespace.getNamespacePrefix());
        }        
        if (NamespaceHashTable.get(RDF.NAMESPACE)==null)            
            NamespaceHashTable.put(RDF.NAMESPACE, "rdf");
        if (NamespaceHashTable.get(RDFS.NAMESPACE)==null)   
            NamespaceHashTable.put(RDFS.NAMESPACE, "rdfs");
        if (NamespaceHashTable.get(RDFS.XSD_NAMESPACE)==null)   
            NamespaceHashTable.put(RDFS.XSD_NAMESPACE, "xsd");
        if (NamespaceHashTable.get(OWL.NAMESPACE_URI)==null)   
            NamespaceHashTable.put(OWL.NAMESPACE_URI, "owl");
    }

    private static final int PLUS = 1;

    private static final int STAY = 0;

    private static final int REDUCTION = -1;

    private int indentLevel = 1;

    private void writeIndents(int oper) throws IOException {
        if (oper == REDUCTION)
            indentLevel--;
        for (int i = indentLevel; i > 0; i--)
            writer.write("\t");
        if (oper == PLUS)
            indentLevel++;
    }

    /* 1. All unused elements in the ontology */
    private List unusedElements = new Vector();

    /* 2. All elements used for serialization */
    private Map usedElements = new HashMap();

    /**
     * Save all the resources.
     * 
     * @throws IOException
     * @throws UnsupportedViewTypeException 
     */
    public void save() throws IOException, UnsupportedViewTypeException {
        
        Iterator graphIter=document.getComplementalGraph().getAllResources();
        
        //1. Ontology Header.
        List ontolist=null;
        ontolist = document.getComplementalGraph().getTypeResources(OWL.ONTOLOGY_URI );
        
        if (ontolist!=null && ontolist.size() >0){
            ontology=(OWLOntology)ontolist.get(0);
            writeOntoHeader(ontology, false, false);
        }
        
        //remove all redundant URIReferenceNode
        List allResources=new ArrayList();
        List ResourceWithSameURI=new ArrayList();
        String preURI="";
        RDFSResource resource;
        boolean bURIReferenceNode=false;
        for (Iterator iterator = graphIter; iterator.hasNext();) {
            Object object = iterator.next();
            resource = (RDFSResource) object;
            if (!resource.canAsType(RDFSLiteral.class)&& !resource.canAsType(BlankNode.class)
                    &&!resource.canAsType(OWLOntology.class)){
                if (!resource.getURI().equals(preURI)){
                    if (ResourceWithSameURI.size() >0){
                        //add list
                        allResources.addAll(ResourceWithSameURI);
                        ResourceWithSameURI.clear() ;
                    }
                    preURI=resource.getURI();
                    ResourceWithSameURI.add(resource);
                    bURIReferenceNode=false;
                }else{
                    if (ResourceWithSameURI.size()>0 && resource.eClass().getName().equals("URIReferenceNode") )
                        continue;
                    else if (!resource.eClass().getName().equals("URIReferenceNode") ){                        
                        if (!bURIReferenceNode){
                            for (Iterator iter=ResourceWithSameURI.iterator();iter.hasNext();){
                                RDFSResource res= (RDFSResource)iter.next();
                                if (res.eClass().getName().equals("URIReferenceNode") )
                                {
                                    ResourceWithSameURI.remove(res);
                                     iter=ResourceWithSameURI.iterator();
                                }
                            }
                        }
                        bURIReferenceNode=true;
                    }
                    ResourceWithSameURI.add(resource);                        
                }
            }
        }
        //process the last resource
        if (ResourceWithSameURI.size() >0){
            allResources.addAll(ResourceWithSameURI);
            ResourceWithSameURI.clear() ;
        }
        //3. All typed elements with URI.
        for (Iterator iterator = allResources.iterator(); iterator.hasNext();) {
            Object object = iterator.next();
            resource = (RDFSResource) object;
            if (!(resource.canAsType( RDFSLiteral.class ))) {                
                if (!isAnonymousElement(resource)){
                    System.out.println(resource.getURI()+" type: "+resource.eClass().getName()) ;
                    saveResource(resource);
                }
                else
                    unusedElements.add(resource);
            }
        }
        //3. Elements left unused.
        for (int i = 0, n = unusedElements.size(); i < n; i++) {
            resource = (RDFSResource) unusedElements.get(i);
            if (!resource.canAsType( BlankNode.class )){
                if (!usedElements.containsKey(resource.getURI()))
                    saveResource(resource);
            }
            else{
                if (!usedElements.containsKey(resource.getNodeID() ))
                    saveResource(resource);
            }
        }
    }

    /**
     * Save the resource : Class / ObjectProperty / DatatypeProperty /
     * Individual Statement / Container
     * 
     * @param resource
     * @throws IOException
     * @throws UnsupportedViewTypeException 
     */
    private void saveResource(RDFSResource resource) throws IOException, UnsupportedViewTypeException {
        indentLevel = 1;
        if (resource.canAsType( OWLClass.class ))
            writeClass((OWLClass) resource.asType( OWLClass.class ), false, false);
        else if (resource.canAsType( OWLObjectProperty.class ))
            writeObjProperty((OWLObjectProperty) resource.asType(OWLObjectProperty.class ), false, false);
        else if (resource.canAsType( OWLDatatypeProperty.class ))
            writeDatatypeProperty((OWLDatatypeProperty) resource.asType(OWLDatatypeProperty.class ), false, false);
        else if (resource.canAsType( OWLAnnotationProperty.class ))
            writeAnnoProperty((OWLAnnotationProperty) resource.asType( OWLAnnotationProperty.class ), false, false);
        else if (resource.canAsType( OWLOntologyProperty.class ))
            writeOntoProperty((OWLOntologyProperty) resource.asType(OWLOntologyProperty.class  ), false, false);
        else if (resource.canAsType( Individual.class ))
            writeIndividual((Individual) resource.asType(Individual.class ), false, false);
        else if (resource.canAsType( RDFSContainer.class ))
            writeContainer((RDFSContainer) resource.asType( RDFSContainer.class), false, false);
        else if (resource.canAsType( RDFStatement.class ))
            writeStatement((RDFStatement) resource.asType(RDFStatement.class ), false, false);
        else if (resource.canAsType( OWLDataRange.class ))
            writeDataRange((OWLDataRange) resource.asType( OWLDataRange.class), false, false);
        else if (resource.canAsType( URIReferenceNode.class ))
            writeURIReferenceNode((URIReferenceNode) resource.asType(URIReferenceNode.class ), false, false);
            
    }

    /** ************************* write method ********************************* */
    private void writeOntoProperty(OWLOntologyProperty ontoProperty,
            boolean isReferenced, boolean referencedAsNode) throws IOException {
        startElement(ontoProperty, OWL_PREFIX + OWL.ONTOLOGY_PROPERTY, isReferenced, referencedAsNode);
        endElement(OWL_PREFIX + OWL.ONTOLOGY_PROPERTY);
    }

    private void writeAnnoProperty(OWLAnnotationProperty annoProperty,
            boolean isReferenced, boolean referencedAsNode) throws IOException {
        startElement(annoProperty, OWL_PREFIX + OWL.ANNOTATION_PROPERTY, isReferenced, referencedAsNode);
        endElement(OWL_PREFIX + OWL.ANNOTATION_PROPERTY);
    }

    //owl:Ontology
    private void writeOntoHeader(OWLOntology onto, boolean isReferenced, boolean referencedAsNode)
            throws IOException, UnsupportedViewTypeException {
        startElement(onto, OWL_PREFIX + OWL.ONTOLOGY, isReferenced, referencedAsNode);
        if (!isReferenced)
            writeOntoAttrs(onto);
        endElement(OWL_PREFIX + OWL.ONTOLOGY);
    }

    //owl:priorVersion owl:backwardCompatibleWith owl:imcompatibleWith
    //owl:versionInfo owl:imports
    private void writeOntoAttrs(OWLOntology onto) throws IOException, UnsupportedViewTypeException {
        if (onto.getOWLpriorVersion() != null) {
            Iterator iter = onto.getOWLpriorVersion().iterator();
            while (iter.hasNext()) {
                writeIndents(PLUS);
                writer.write("<owl:priorVersion>\n");
                writeOntoHeader((OWLOntology) (iter.next()), true, true);
                writeIndents(REDUCTION);
                writer.write("</owl:priorVersion>\n");
            }
        }
        if (onto.getOWLbackwardCompatibleWith() != null) {
            Iterator iter = onto.getOWLbackwardCompatibleWith().iterator();
            while (iter.hasNext()) {
                writeIndents(PLUS);
                writer.write("<owl:backwardCompatibleWith>\n");
                writeOntoHeader((OWLOntology) (iter.next()), true, true);
                writeIndents(REDUCTION);
                writer.write("</owl:backwardCompatibleWith>\n");
            }
        }
        if (onto.getOWLincompatibleWith() != null) {
            Iterator iter = onto.getOWLincompatibleWith().iterator();
            while (iter.hasNext()) {
                writeIndents(PLUS);
                writer.write("<owl:incompatibleWith>\n");
                writeOntoHeader((OWLOntology) (iter.next()), true, true);
                writeIndents(REDUCTION);
                writer.write("</owl:incompatibleWith>\n");
            }
        }
        if (onto.getOWLversionInfo() != null) {
            Iterator iter = onto.getOWLversionInfo().iterator();
            while (iter.hasNext()) {
                writeAttribute(OWL_PREFIX + OWL.VERSION_INFO,
                        (RDFSResource) iter.next());
            }
        }
        writeAttributes(OWL_PREFIX + OWL.IMPORTS, onto.getOWLimports());
        writeRDFSResourceAttrs(ontology);
    }

    //owl:Class owl:Restriction
    private void writeClass(OWLClass cls, boolean isReferenced, boolean referencedAsNode) throws IOException, UnsupportedViewTypeException {
        if (cls.canAsType( BlankNode.class ))
            usedElements.put(cls.getNodeID(), cls);
        else
            usedElements.put(cls.getURI(), cls);

        if (isReferenced && !isAnonymousElement(cls)) {
            //if the class is referenced by other resources, the parameter isReferenced should be set to true
            startElement(cls, OWL_PREFIX + OWL.CLASS, isReferenced, referencedAsNode);
            endElement(OWL_PREFIX + OWL.CLASS);
            return;
        }
        
        //6.OWLRestriction
        if (cls.canAsType( OWLRestriction.class )) {
            writeRestriction((OWLRestriction) cls.asType( OWLRestriction.class), isReferenced, referencedAsNode);
            return;
        }

        //5.EnumeratedClass
        if (cls.canAsType( EnumeratedClass.class ))
            writeEnumeratedCls((EnumeratedClass) cls.asType( EnumeratedClass.class), isReferenced, referencedAsNode);
        //4.ComplementClass
        else if (cls.canAsType( ComplementClass.class ))
            writeComplementCls((ComplementClass) cls.asType( ComplementClass.class ), isReferenced, referencedAsNode);
        //3.UnionClass
        else if (cls.canAsType( UnionClass.class ))
            writeUnionCls((UnionClass) cls.asType(UnionClass.class), isReferenced, referencedAsNode);
        //2.IntersectionClass
        else if (cls.canAsType( IntersectionClass.class ))
            writeIntersectionCls((IntersectionClass) cls.asType( IntersectionClass.class), isReferenced, referencedAsNode);
        //1.OWLClass with URI
        else {
            startElement(cls, OWL_PREFIX + OWL.CLASS, isReferenced, referencedAsNode);
            if (!isReferenced || isAnonymousElement(cls))
                writeClsAttrs(cls);
            endElement(OWL_PREFIX + OWL.CLASS);
            //DeprecatedClass
            if ( cls.isIsDeprecated() ) {
                startElement(cls, OWL_PREFIX + OWL.DEPRECATED_CLASS, isReferenced, referencedAsNode);
                endElement(OWL_PREFIX + OWL.DEPRECATED_CLASS);
            }
        }
    }

    //owl:equivalentClass owl:disjointWith rdfs:subClassOf
    private void writeClsAttrs(OWLClass cls) throws IOException, UnsupportedViewTypeException {
        writeAttributes(OWL_PREFIX + OWL.EQUIVALENT_CLASS, cls
                .getEquivalentClass());
        writeAttributes(OWL_PREFIX + OWL.DISJOINT_WITH, cls
                .getOWLdisjointWith());
        writeAttributes(RDFS_PREFIX + RDFS.P_SUBCLASSOF, cls
                .getRDFSsubClassOf());
        writeRDFSResourceAttrs(cls);
    }

    //owl:Restriction
    private void writeRestriction(OWLRestriction restriction, boolean isReferenced, boolean referencedAsNode)
            throws IOException, UnsupportedViewTypeException {
        if ( !isReferenced && !isAnonymousElement(restriction) ) {
            startElement(restriction, OWL_PREFIX + OWL.CLASS, isReferenced, referencedAsNode);
            writeIndents(PLUS);
            writer.write("<owl:equivalentClass>\n");
            writeIndents(PLUS);
            writer.write("<owl:Restriction>\n");
        } else
        {
            writeIndents(PLUS);
//        	if ( restriction.getLocalName() == null || restriction.getLocalName().length() == 0 )
//                writer.write("<owl:Restriction>\n");
//        	else            
        	{
        	    if (isAnonymousElement(restriction))
        	    {
        	        writer.write("<owl:Restriction rdf:nodeID=\"" + restriction.getNodeID() + "\">\n");
        	    }
//        	    else if (restriction.getNamespace()==null)
//        	        writer.write("<owl:Restriction rdf:nodeID=\"" + restriction.getLocalName() + "\">");
        	    else                    
            	{  
                    URIReference uri=((URIReference)(restriction.getUriRef().get(0)));
                    String abbreviation = null;
                    abbreviation=(String) NamespaceHashTable.get(uri.getURIString());
                    if (abbreviation!=null ) {
                        if (uri.getURIString().indexOf("%")==-1)
                            writer.write( "<owl:Restriction rdf:about=\"&" + abbreviation + ";" + uri.getFragmentIdentifier().getName()+"\">");
                        else
                            writer.write ("<owl:Restriction rdf:about=\""+uri.getURIString()+"\">");
                    } 
                    else
                        writer.write ("<owl:Restriction rdf:about=\""+uri.getURIString()+"\">");
    
                 }           	        
	    
        	}
        	    //writer.write("<owl:Restriction rdf:nodeID=\"" + restriction.getLocalName() + "\">");
        }
        
        //owl:onProperty
        writeAttribute(OWL_PREFIX + OWL.ON_PROPERTY, restriction
                .getOWLonProperty());
        //owl:allValuesFrom
        if (restriction.canAsType(AllValuesFromRestriction.class )){
            if (((AllValuesFromRestriction)(restriction.asType( AllValuesFromRestriction.class))).getAllValuesFromClass()!=null)
            writeAttribute(OWL_PREFIX + OWL.ALL_VALUES_FROM,
                    ((AllValuesFromRestriction) restriction.asType( AllValuesFromRestriction.class))
                            .getAllValuesFromClass());
            else
                writeAttribute(OWL_PREFIX + OWL.ALL_VALUES_FROM,
                        ((AllValuesFromRestriction) restriction.asType( AllValuesFromRestriction.class))
                                .getAllValuesFromDataRange());
        }
        //owl:someValuesFrom
        else if (restriction.canAsType( SomeValuesFromRestriction.class )){
            if (((SomeValuesFromRestriction) restriction.asType(SomeValuesFromRestriction.class ))
                .getSomeValuesFromClass()!=null)
                writeAttribute(OWL_PREFIX + OWL.SOME_VALUES_FROM,
            ((SomeValuesFromRestriction) restriction.asType(SomeValuesFromRestriction.class )).getSomeValuesFromClass());
            else
                writeAttribute(OWL_PREFIX + OWL.SOME_VALUES_FROM,
                        ((SomeValuesFromRestriction) restriction.asType(SomeValuesFromRestriction.class )).getSomeValuesFromClass());
            }
        //owl:hasValue
        else if (restriction.canAsType( HasValueRestriction.class )){
            if (((HasValueRestriction) restriction.asType(HasValueRestriction.class )).getHasIndividualValue()!=null)
            writeAttribute(OWL_PREFIX + OWL.HAS_VALUE,
                    ((HasValueRestriction) restriction.asType(HasValueRestriction.class )).getHasIndividualValue());
            else
                writeAttribute(OWL_PREFIX + OWL.HAS_VALUE,
                        ((HasValueRestriction) restriction.asType(HasValueRestriction.class )).getHasLiteralValue());
        }
        //owl:maxCardinality
        else if (restriction.canAsType( MaxCardinalityRestriction.class ))
            writeAttribute(OWL_PREFIX + OWL.MAX_CARDINALITY,
                    ((MaxCardinalityRestriction) restriction.asType(MaxCardinalityRestriction.class ))
                            .getOWLmaxCardinality());
        //owl:minCardinality
        else if (restriction.canAsType(MinCardinalityRestriction.class ))
            writeAttribute(OWL_PREFIX + OWL.MIN_CARDINALITY,
                    ((MinCardinalityRestriction) restriction.asType(MinCardinalityRestriction.class ))
                            .getOWLminCardinality());
        //owl:cardinality
        else if (restriction.canAsType( CardinalityRestriction.class ))
            writeAttribute(OWL_PREFIX + OWL.CARDINALITY,
                    ((CardinalityRestriction) restriction.asType(CardinalityRestriction.class )).getOWLcardinality());
        writeClsAttrs(restriction);
        
        endElement(OWL_PREFIX + OWL.RESTRICTION);

        if ( !isReferenced && !isAnonymousElement(restriction) ) {
            writeIndents(REDUCTION);
            writer.write("</owl:equivalentClass>\n");
            endElement(OWL_PREFIX + OWL.CLASS);
        }
    }

    //owl:oneOf
    private void writeEnumeratedCls(EnumeratedClass cls, boolean isReferenced, boolean referencedAsNode)
            throws IOException, UnsupportedViewTypeException {
        startElement(cls, OWL_PREFIX + OWL.CLASS, isReferenced, referencedAsNode);
        EList list = cls.getOWLoneOf();
        writeIndents(PLUS);
        writer.write("<owl:oneOf rdf:parseType=\"Collection\">\n");
        for (int i = 0; i < list.size(); i++)
            writeIndividual((Individual) list.get(i), true, true);
        writeIndents(REDUCTION);
        writer.write("</owl:oneOf>\n");
        writeClsAttrs(cls);
        endElement(OWL_PREFIX + OWL.CLASS);
    }

    //owl:intersectionOf
    private void writeIntersectionCls(IntersectionClass cls, boolean isReferenced, boolean referencedAsNode)
            throws IOException, UnsupportedViewTypeException {
        startElement(cls, OWL_PREFIX + OWL.CLASS, isReferenced, referencedAsNode);
        writeIndents(PLUS);
        EList list = cls.getOWLintersectionOf();
        writer.write("<owl:intersectionOf rdf:parseType=\"Collection\">\n");
        for (int i = 0; i < list.size(); i++)
            writeClass((OWLClass) list.get(i), true, true);
        writeIndents(REDUCTION);
        writer.write("</owl:intersectionOf>\n");
        writeClsAttrs(cls);
        endElement(OWL_PREFIX + OWL.CLASS);
    }

    //owl:unionOf
    private void writeUnionCls(UnionClass cls, boolean isReferenced, boolean referencedAsNode)
            throws IOException, UnsupportedViewTypeException {
        startElement(cls, OWL_PREFIX + OWL.CLASS, isReferenced, referencedAsNode);
        writeIndents(PLUS);
        EList list = cls.getOWLunionOf();
        writer.write("<owl:unionOf rdf:parseType=\"Collection\">\n");
        for (int i = 0; i < list.size(); i++)
            writeClass((OWLClass) list.get(i), true, true);
        writeIndents(REDUCTION);
        writer.write("</owl:unionOf>\n");
        writeClsAttrs(cls);
        endElement(OWL_PREFIX + OWL.CLASS);
    }

    //owl:complementOf
    private void writeComplementCls(ComplementClass cls, boolean isReferenced, boolean referencedAsNode)
            throws IOException, UnsupportedViewTypeException {
        startElement(cls, OWL_PREFIX + OWL.CLASS, isReferenced, referencedAsNode);
        writeAttribute(OWL_PREFIX + OWL.COMPLEMENT_OF, cls.getOWLcomplementOf());
        writeClsAttrs(cls);
        endElement(OWL_PREFIX + OWL.CLASS);
    }

    //owl:ObjectProperty
    private void writeObjProperty(OWLObjectProperty property, boolean isReferenced, boolean referencedAsNode)
            throws IOException, UnsupportedViewTypeException {
        startElement(property, OWL_PREFIX + OWL.OBJECT_PROPERTY, isReferenced, referencedAsNode);

        writeAttributes(OWL_PREFIX + OWL.INVERSE_OF, property
                    .getOWLinverseOf());
        
        //owl:equivalentProperty
        writeAttributes(OWL_PREFIX + OWL.EQUIVALENT_PROPERTY, property
                .getOWLequivalentProperty());

        if (property.isIsTransitive()) {
            writeIndents(STAY);
            writer.write("<rdf:type"
                         + " rdf:resource= \"" + OWL.TRANSITIVE_PROPERTY_URI
                         + "\"/>\n");
        }
        if (property.isIsFunctional()) {
            writeIndents(STAY);
            writer.write("<rdf:type"
                         + " rdf:resource= \"" + OWL.FUNCTIONAL_PROPERTY_URI
                         + "\"/>\n");
        }
        if (property.isIsSymmetric()) {
            writeIndents(STAY);
            writer.write("<rdf:type"
                         + " rdf:resource= \"" + OWL.SYMMETRIC_PROPERTY_URI
                         + "\"/>\n");
        }
        if (property.isIsInverseFunctional()) {
            writeIndents(STAY);
            writer.write("<rdf:type"
                         + " rdf:resource= \""
                         + OWL.INVERSE_FUNCTIONAL_PROPERTY_URI + "\"/>\n");
        }
        writeRDFPropertyAttrs(property);
        endElement(OWL_PREFIX + OWL.OBJECT_PROPERTY);
        //DeprecatedProperty
        if ( property.isIsPropertyDeprecated() ) {
            startElement(property, OWL_PREFIX + OWL.DEPRECATED_PROPERTY, isReferenced, referencedAsNode);
            endElement(OWL_PREFIX + OWL.DEPRECATED_PROPERTY);
        }
    }

    //owl:DatatypeProperty
    private void writeDatatypeProperty(OWLDatatypeProperty property,
            boolean isReferenced, boolean referencedAsNode) throws IOException, UnsupportedViewTypeException {
        //deal with datatype property
        startElement(property, OWL_PREFIX + OWL.DATATYPE_PROPERTY, isReferenced, referencedAsNode);
        //owl:equivalentProperty
        if (property.getOWLequivalentProperty() != null)
            writeAttributes(OWL_PREFIX + OWL.EQUIVALENT_PROPERTY, property
                    .getOWLequivalentProperty());
        if (property.isIsFunctional()) {
            writeIndents(STAY);
            writer.write("<rdf:type"
                         + " rdf:resource= \"" + OWL.FUNCTIONAL_PROPERTY_URI
                         + "\"/>\n");
        }
        writeRDFPropertyAttrs(property);
        endElement(OWL_PREFIX + OWL.DATATYPE_PROPERTY);
        //DeprecatedProperty
        if ( property.isIsPropertyDeprecated() ) {
            startElement(property, OWL_PREFIX + OWL.DEPRECATED_PROPERTY, isReferenced, referencedAsNode);
            endElement(OWL_PREFIX + OWL.DEPRECATED_PROPERTY);
        }
    }

    //rdfs:subPropertyOf rdfs:range rdfs:domain
    private void writeRDFPropertyAttrs(RDFProperty property) throws IOException, UnsupportedViewTypeException {
        writeAttributes(RDFS_PREFIX + RDFS.P_SUBPROPERTYOF, property
                .getRDFSsubPropertyOf());
        writeAttributes(RDFS_PREFIX + RDFS.P_DOMAIN, property.getRDFSdomain());
        writeAttributes(RDFS_PREFIX + RDFS.P_RANGE, property.getRDFSrange());
        writeRDFSResourceAttrs(property);
    }

    //rdfs:isDefinedBy rdfs:seeAlso rdf:value rdf:type rdfs:comment rdf:li
    // rdfs:label
    private void writeRDFSResourceAttrs(RDFSResource resource)
            throws IOException, UnsupportedViewTypeException {
        writeAttributes(RDFS_PREFIX + RDFS.ISDEFINEDBY, resource
                .getRDFSisDefinedBy());
        writeAttributes(RDFS_PREFIX + RDFS.SEEALSO, resource.getRDFSseeAlso());
        //writeAttributes(RDF_PREFIX + RDF.P_VALUE, resource.getRDFvalue());
        writeAttributes(RDFS_PREFIX + RDFS.P_COMMENT, resource.getRDFScomment());
        writeAttributes(RDF_PREFIX + RDF.S_LI, resource.getRDFSmember());
        writeAttributes(RDFS_PREFIX + RDFS.P_LABEL, resource.getRDFSlabel());
    }

    //owl:Thing
    private void writeIndividual(Individual inv, boolean isReferenced, boolean referencedAsNode)
            throws IOException, UnsupportedViewTypeException {
        String type = null;
        String uri = null;
        for (int i = 0; i < inv.getRDFtype().size(); i++) {
            RDFSClass cls = (RDFSClass) inv.getRDFtype().get(i);
            if (!cls.getURI().equals(OWL.THING_URI)) {
            	String nsURI =  (String) NamespaceHashTable.get(((URIReference)cls.getUriRef().get(0)).getNamespace().getNamespaceURIRef().getURIString() );
                type = OWLXMLSaverImpl.replaceKeywords( nsURI )
                + ":" + ((URIReference)cls.getUriRef().get(0)).getFragmentIdentifier().getName();
                uri = cls.getURI();
                break;
            }
        }
        if (type == null) {
            type = OWL_PREFIX + OWL.THING;
            uri = OWL.THING_URI;
        }

        if (inv.getURI().indexOf("Medium")!=-1)
            System.out.print( "dads");
        startElement(inv, type, isReferenced, referencedAsNode);
        if (!isReferenced) {
            //owl:differentFrom
            writeAttributes(OWL_PREFIX + OWL.DIFFERENT_FROM, inv
                    .getOWLdifferentFrom());
            //owl:sameAs
            writeAttributes(OWL_PREFIX + OWL.SAME_AS, inv.getOWLsameAs());
            
            Iterator iter=inv.getCustomTriples().iterator();
            while(iter.hasNext() ){
                Triple triple=(Triple) iter.next() ;
                RDFSResource[] prop=document.getComplementalGraph().getRDFSResource(triple.getPredicate()).getAllTypeViews();
                for(int i=0;i<prop.length;i++){
                    if (prop[i] instanceof RDFProperty){
                        if (triple.isObjectLiteral()){
                            EList obj=(EList) new ArrayList();
                            obj.add(triple.getObjectLiteral());
                            writeAttributes(NamespaceHashTable.get (((URIReference)prop[i].getUriRef().get( 0)).getNamespace().getNamespaceURIRef().getURIString())
                                + ":" + ((URIReference)prop[i].getUriRef().get( 0)).getFragmentIdentifier().getName(), obj);   
                        }
                        else if (triple.isObjectBlankNode()){
                            EList obj=(EList) new ArrayList();
                            RDFSResource[] res=document.getComplementalGraph().getRDFSResource(triple.getObjectNodeID()).getAllTypeViews();
                            obj.add(res[0]);
                            writeAttributes(NamespaceHashTable.get (((URIReference)prop[i].getUriRef().get( 0)).getNamespace().getNamespaceURIRef().getURIString())
                                    + ":" + ((URIReference)prop[i].getUriRef().get( 0)).getFragmentIdentifier().getName(), obj);   
                        }
                        else{
                            EList obj=(EList) new ArrayList();
                            RDFSResource[] res=document.getComplementalGraph().getRDFSResource(triple.getObjectURI()).getAllTypeViews();
                            obj.add(res[0]);
                            writeAttributes(NamespaceHashTable.get (((URIReference)prop[i].getUriRef().get( 0)).getNamespace().getNamespaceURIRef().getURIString())
                                    + ":" + ((URIReference)prop[i].getUriRef().get( 0)).getFragmentIdentifier().getName(), obj);   
                        }
                    }
                }                
            }
                        
            for (int i = 0; i < inv.getRDFtype().size(); i++) {
                RDFSClass cls = (RDFSClass) inv.getRDFtype().get(i);
                if (cls.getURI().equals(OWL.THING_URI)
                    || cls.getURI().equals(uri))
                    continue;
                writeAttribute(RDF_PREFIX + RDF.P_TYPE, cls);
            }
            writeRDFSResourceAttrs(inv);
        }
        endElement(type);
    }
    
    //rdf:description
    private void writeURIReferenceNode(URIReferenceNode inv, boolean isReferenced, boolean referencedAsNode)
            throws IOException, UnsupportedViewTypeException {
        String type = null;
        String uri = null;
        for (int i = 0; i < inv.getRDFtype().size(); i++) {
            RDFSClass cls = (RDFSClass) inv.getRDFtype().get(i);
            if (!cls.getURI().equals(OWL.THING_URI)) {
                String nsURI = (String) NamespaceHashTable.get (((URIReference)cls.getUriRef().get(0)).getNamespace().getNamespaceURIRef().getURIString() );
                type = OWLXMLSaverImpl.replaceKeywords( nsURI )
                + ":" + ((URIReference)cls.getUriRef().get(0)).getFragmentIdentifier().getName();
                uri = cls.getURI();
                break;
            }
        }
        if (type == null) {
            type = RDF_PREFIX + RDF.S_DESCRIPTION;
            uri = RDF.S_DESCRIPTION_STR;
        }

        startElement(inv, type, isReferenced, referencedAsNode);
        if (!isReferenced) {           
            Iterator iter=inv.getCustomTriples().iterator();
            while(iter.hasNext() ){
                Triple triple=(Triple) iter.next() ;
                RDFSResource[] prop=document.getComplementalGraph().getRDFSResource(triple.getPredicate()).getAllTypeViews();
                for(int i=0;i<prop.length;i++){
                    if (prop[i] instanceof RDFProperty){
                        if (triple.isObjectLiteral()){
                            EList obj=(EList) new ArrayList();
                            obj.add(triple.getObjectLiteral());
                            writeAttributes(NamespaceHashTable.get (((URIReference)prop[i].getUriRef().get( 0)).getNamespace().getNamespaceURIRef().getURIString())
                                + ":" + ((URIReference)prop[i].getUriRef().get( 0)).getFragmentIdentifier().getName(), obj);   
                        }
                        else if (triple.isObjectBlankNode()){
                            EList obj=(EList) new ArrayList();
                            RDFSResource[] res=document.getComplementalGraph().getRDFSResource(triple.getObjectNodeID()).getAllTypeViews();
                            obj.add(res[0]);
                            writeAttributes(NamespaceHashTable.get (((URIReference)prop[i].getUriRef().get( 0)).getNamespace().getNamespaceURIRef().getURIString())
                                    + ":" + ((URIReference)prop[i].getUriRef().get( 0)).getFragmentIdentifier().getName(), obj);   
                        }
                        else{
                            EList obj=(EList) new ArrayList();
                            RDFSResource[] res=document.getComplementalGraph().getRDFSResource(triple.getObjectURI()).getAllTypeViews();
                            obj.add(res[0]);
                            writeAttributes(NamespaceHashTable.get (((URIReference)prop[i].getUriRef().get( 0)).getNamespace().getNamespaceURIRef().getURIString())
                                    + ":" + ((URIReference)prop[i].getUriRef().get( 0)).getFragmentIdentifier().getName(), obj);   
                        }
                    }
                }                
            }
                        
            for (int i = 0; i < inv.getRDFtype().size(); i++) {
                RDFSClass cls = (RDFSClass) inv.getRDFtype().get(i);
                if (cls.getURI().equals(OWL.THING_URI)
                    || cls.getURI().equals(uri))
                    continue;
                writeAttribute(RDF_PREFIX + RDF.P_TYPE, cls);
            }
            writeRDFSResourceAttrs(inv);
            
//            //owl:differentFrom
//            writeAttributes(OWL_PREFIX + OWL.DIFFERENT_FROM, inv
//                    .getOWLdifferentFrom());
//            //owl:sameAs
//            writeAttributes(OWL_PREFIX + OWL.SAME_AS, inv.getOWLsameAs());
        }
        endElement(type);
    }


    //owl:DataRange owl:oneOf
    private void writeDataRange(OWLDataRange range, boolean isReferenced, boolean referencedAsNode)
            throws IOException, UnsupportedViewTypeException {
        usedElements.put(range.getNodeID(), range);
        startElement(range, OWL_PREFIX + OWL.DATA_RANGE, isReferenced, referencedAsNode);
        writeIndents(PLUS);
        writer.write("<owl:oneOf>\n");
        writeRDFList(range.getOWLDataRangeOneOf());
        writeIndents(REDUCTION);
        writer.write("</owl:oneOf>\n");
        writeAttributes(RDFS_PREFIX + RDFS.P_SUBCLASSOF, range
                .getRDFSsubClassOf());
        writeRDFSResourceAttrs(range);
        endElement(OWL_PREFIX + OWL.DATA_RANGE);
    }

    //rdf:List rdf:first rdf:rest
    private void writeRDFList(EList list) throws IOException, UnsupportedViewTypeException {
        for (int i = 0; i < list.size(); i++) {
            writeIndents(PLUS);
            writer.write("<rdf:List>\n");
            writeAttribute("rdf:first", (RDFSLiteral) list.get(i));
            if (i == list.size() - 1) {
            	writeIndents(STAY);
                writer.write("<rdf:rest rdf:resource=\"&rdf;nil\"/>\n");
            } else {
                writeIndents(PLUS);
                writer.write("<rdf:rest>\n");
            }
        }
        for (int i = list.size()-1; i >= 0; i--) {
        	if (i < list.size() - 1) {
	            writeIndents(REDUCTION);
	            writer.write("</rdf:rest>\n");
        	}
            writeIndents(REDUCTION);
            writer.write("</rdf:List>\n");
        }
    }

    //rdf:Statement
    private void writeStatement(RDFStatement statement, boolean isReferenced, boolean referencedAsNode)
            throws IOException, UnsupportedViewTypeException {
        startElement(statement, RDF_PREFIX + RDF.C_STATEMENT, false, referencedAsNode);
        //rdf:subject
        writeAttribute(RDF_PREFIX + RDF.P_SUBJECT, statement.getRDFsubject());
        //rdf:predicate
        writeAttribute(RDF_PREFIX + RDF.P_PREDICATE, statement
                .getRDFpredicate());
        //rdf:object
        writeAttribute(RDF_PREFIX + RDF.P_OBJECT, statement.getRDFobject());
        writeRDFSResourceAttrs(statement);
        endElement(RDF_PREFIX + RDF.C_STATEMENT);
    }

    private void writeContainer(RDFSContainer container, boolean isReferenced, boolean referencedAsNode)
            throws IOException, UnsupportedViewTypeException {
        //rdf:Alt
        if (container.canAsType( RDFAlt.class)) {
            startElement(container, RDF_PREFIX + RDF.C_ALT, isReferenced, referencedAsNode);
            writeRDFSResourceAttrs(container);
            endElement(RDF_PREFIX + RDF.C_ALT);
            return;
        }
        //rdf:Bag
        if (container.canAsType( RDFBag.class )) {
            startElement(container, RDF_PREFIX + RDF.C_BAG, isReferenced, referencedAsNode);
            writeRDFSResourceAttrs(container);
            endElement(RDF_PREFIX + RDF.C_BAG);
            return;
        }
        //rdf:Seq
        if (container.canAsType( RDFSeq.class )) {
            startElement(container, RDF_PREFIX + RDF.C_SEQ, isReferenced, referencedAsNode);
            writeRDFSResourceAttrs(container);
            endElement(RDF_PREFIX + RDF.C_SEQ);
            return;
        }
    }

    /** ************************************************************************* */

    private void startElement(RDFSResource resource, String tag, boolean isReferenced, boolean referencedAsNode)
            throws IOException {
        writeIndents(PLUS);
        writer.write("<" + tag + " " + getIDAttr(resource, isReferenced, referencedAsNode) + ">\n");
    }

    private void endElement(String tag) throws IOException {
        writeIndents(REDUCTION);
        writer.write("</" + tag + ">\n");
    }

    private boolean isAnonymousElement(RDFSResource resource) {
        return resource.canAsType(BlankNode.class);
    }

    private String getIDAttr(RDFSResource resource, boolean isReferenced, boolean referencedAsNode) {
              
        if (isAnonymousElement(resource))
            if ( resource.getNodeID()  == null )
        		return "";
        	else
        		return "rdf:nodeID=\"" + resource.getNodeID() + "\"";
//        if (resource instanceof OWLClass || resource instanceof OWLDataRange){
//            if ( resource.getLocalName() == null || resource.getLocalName().length() == 0 )
//        		return "";
//        }
        if (resource.canAsType( OWLOntology.class )) {
            if (!isReferenced)
                return "rdf:about=\"" + resource.getURI() + "\"";
            else if ( referencedAsNode )
                return "rdf:about=\"" + resource.getURI() + "\"";
            else
            	return "rdf:resource=\"" + resource.getURI() + "\"";
        }
        String key = isReferenced ? ( referencedAsNode ? "rdf:about" : "rdf:resource" ) : "rdf:about";
        String value = null;
        String abbreviation = null;
        URIReference uri=(URIReference) resource.getUriRef().get(0);
        if (uri.getNamespace()!=null ) {
            abbreviation=(String) NamespaceHashTable.get(uri.getNamespace().getNamespaceURIRef().getURIString());
            if (abbreviation!=null && uri.getNamespace().getNamespaceURIRef().getURIString().indexOf("%")==-1)
                value = "&" + abbreviation + ";" + uri.getFragmentIdentifier().getName();
            else
                value = resource.getURI();
        } else
            value = resource.getURI();
        return key + "=\"" + value + "\"";
    }
    
    private void writeAttribute(String name, RDFSResource value)
            throws IOException, UnsupportedViewTypeException {
        if (value.canAsType( RDFSLiteral.class )) {
            writeIndents(STAY);
            if (value.canAsType( RDFXMLLiteral.class ))
                writer.write("<"
                             + name
                             + " rdf:parseType=\"Literal\">"
                             + ((RDFSLiteral) value)
                                     .getLexicalForm());
            else if (value.canAsType( TypedLiteral.class ))
                writer.write("<"
                             + name
                             + " rdf:datatype=\""
                             + ((TypedLiteral) value).getDatatypeURI().getURIString()
                             + "\">"
                             + ((RDFSLiteral) value)
                                     .getLexicalForm());
            else if (value.canAsType( PlainLiteral.class )) {
                String lang = ((PlainLiteral) value).getLanguage();
                if (lang != null && lang.length() > 0)
                    writer.write("<"
                                 + name
                                 + " xml:lang=\""
                                 + lang
                                 + "\">"
                                 + ((RDFSLiteral) value)
                                         .getLexicalForm());
                else
                    writer.write("<"
                                 + name
                                 + ">"
                                 + ((RDFSLiteral) value)
                                         .getLexicalForm());
            } else
                writer.write("<"
                             + name
                             + ">"
                             + ((RDFSLiteral) value)
                                     .getLexicalForm());
            writer.write("</" + name + ">\n");
            return;
        }

        if (!(value.canAsType( OWLClass.class )) && !( value.canAsType( OWLDataRange.class ))
                || isAnonymousElement(value) || (!isAnonymousElement(value) && value.getURI()  !=null && value.getURI().length() >0))
	        if (!isAnonymousElement(value)|| isAnonymousElement(value) && usedElements.containsKey(value.getNodeID() )) {                
                writeIndents(STAY);
	            writer.write("<" + name + " " + getIDAttr(value, true, false) + "/>\n");
	            return;
	        }

        writeIndents(PLUS);
        writer.write("<" + name + ">\n");
        if (value.canAsType( OWLClass.class ))
            writeClass((OWLClass) value.asType(OWLClass.class ), true, true);
        else if (value.canAsType( OWLObjectProperty.class ))
            writeObjProperty((OWLObjectProperty) value.asType(OWLObjectProperty.class ), true, true);
        else if (value.canAsType( OWLDatatypeProperty.class ))
            writeDatatypeProperty((OWLDatatypeProperty) value.asType(OWLDatatypeProperty.class), true, true);
        else if (value.canAsType( Individual.class ))
            writeIndividual((Individual) value.asType(Individual.class), true, true);
        else if (value.canAsType( RDFStatement.class ))
            writeStatement((RDFStatement) value.asType( RDFStatement.class), true, true);
        else if (value.canAsType( RDFSContainer.class ))
            writeContainer((RDFSContainer) value.asType( RDFSContainer.class), true, true);
        else if (value.canAsType( OWLDataRange.class ))
            writeDataRange((OWLDataRange) value.asType( OWLDataRange.class ), true, true);
        writeIndents(REDUCTION);
        writer.write("</" + name + ">\n");
    }

    private void writeAttributes(String name, EList values) throws IOException, UnsupportedViewTypeException {
        for (Iterator iterator = values.iterator(); iterator.hasNext();)
            writeAttribute(name, (RDFSResource) iterator.next());
    }

    static String replaceKeywords(String str) {
        str = str.replaceAll("&", "&amp;");
        str = str.replaceAll("<", "&lt;");
        str = str.replaceAll(">", "&gt;");
        str = str.replaceAll("'", "&apos;");
        str = str.replaceAll("\"", "&quot;");
        return str;
    }    
}