/*******************************************************************************
 * Copyright (c) 2000, 2011 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.team.ui;

import java.net.URI;
import java.util.*;

import org.eclipse.core.runtime.*;
import org.eclipse.jface.util.IPropertyChangeListener;
import org.eclipse.team.core.*;
import org.eclipse.team.internal.ui.TeamUIPlugin;
import org.eclipse.team.internal.ui.history.GenericHistoryView;
import org.eclipse.team.internal.ui.registry.TeamContentProviderManager;
import org.eclipse.team.ui.history.IHistoryPageSource;
import org.eclipse.team.ui.history.IHistoryView;
import org.eclipse.team.ui.mapping.ITeamContentProviderManager;
import org.eclipse.team.ui.synchronize.ISynchronizeManager;
import org.eclipse.ui.*;

/**
 * TeamUI contains public API for generic UI-based Team functionality.
 * <p>
 * This class is not intended to be subclassed or instantiated by clients
 */
public class TeamUI {
	
	/**
	 * Property constant indicating the global ignores list has changed. 
	 */
	public static final String GLOBAL_IGNORES_CHANGED = TeamUIPlugin.ID + "global_ignores_changed"; //$NON-NLS-1$
	
    /**
     * Property constant indicating the global file types list has changed.
     * @since 3.1
     */
	public static final String GLOBAL_FILE_TYPES_CHANGED = TeamUIPlugin.ID + "global_file_types_changed"; //$NON-NLS-1$
	
	/**
	 * Return the synchronize manager.
	 * 
	 * @return the synchronize manager
	 * @since 3.0
	 */
	public static ISynchronizeManager getSynchronizeManager() {
		return TeamUIPlugin.getPlugin().getSynchronizeManager();
	}

	/**
	 * Register for changes made to Team properties.
	 * 
	 * @param listener the listener to add
	 */
	public static void addPropertyChangeListener(IPropertyChangeListener listener) {
		TeamUIPlugin.addPropertyChangeListener(listener);
	}

	/**
	 * Remove the listener from Team property change listener list.
	 * 
	 * @param listener the listener to remove
	 */
	public static void removePropertyChangeListener(IPropertyChangeListener listener) {
		TeamUIPlugin.removePropertyChangeListener(listener);
	}
	
	/**
	 * Shows the history view and returns a handle to it. Note that in the case of many
	 * history views, the main history view is the one returned here.
	 * 
	 * @return an IHistoryView which is the main history view if it is found or null if it can't be found
	 * @since 3.2
	 */
	public static IHistoryView getHistoryView() {
		try {
			TeamUIPlugin.getActivePage().showView(IHistoryView.VIEW_ID);
			return (IHistoryView) TeamUIPlugin.getActivePage().findView(IHistoryView.VIEW_ID);
		} catch (PartInitException e) {
		}

		return null;
	}
	
	/**
	 * Shows a history view containing the given input and returns a handle to the view
	 * or <code>null</code> if no history was available for the given input. If an appropriate
	 * instance of a history view is already opened, its input will be changed and the view will
	 * be activated. Otherwise a new view will be opened.
	 * @param page the workbench page containing the history view
	 * @param input the input whose history is to be displayed
	 * @param pageSource the history page source from which to obtain the page or <code>null</code> 
	 * 	if the page source should be determined using the Adapter manager.
	 * 
	 * @return an IHistoryView which is the main history view if it is found or null if it can't be found
	 * @since 3.3
	 */
	public static IHistoryView showHistoryFor(IWorkbenchPage page, Object input, IHistoryPageSource pageSource) {
		try {
			IHistoryView view = (IHistoryView) page.findView(IHistoryView.VIEW_ID);
			if (view == null) {
				page.showView(IHistoryView.VIEW_ID);
				view = (IHistoryView) TeamUIPlugin.getActivePage().findView(IHistoryView.VIEW_ID);
				return showInputInView(page, input, view, pageSource);
			} else {
				view = ((GenericHistoryView)view).findAppropriateHistoryViewFor(input, pageSource);
				if (view == null) {
					view = (IHistoryView) page.showView(IHistoryView.VIEW_ID, IHistoryView.VIEW_ID + System.currentTimeMillis(), IWorkbenchPage.VIEW_CREATE);
					return showInputInView(page, input, view, pageSource);
				} else {
					return showInputInView(page, input, view, pageSource);
				}
			}
		} catch (PartInitException e) {
		}

		return null;
	}

	private static IHistoryView showInputInView(IWorkbenchPage page,
			Object input, IHistoryView view, IHistoryPageSource pageSource) {
		page.activate((IWorkbenchPart)view);
		((GenericHistoryView)view).showHistoryPageFor(input, true, false, pageSource);
		return view;
	}
	
	/**
	 * Return the team content provider manager which gives access to the team
	 * content providers registered with the
	 * <code>org.eclipse.team.ui.teamContentProviders</code> extension point.
	 * 
	 * @return the team content provider manager
	 * @since 3.2
	 */
	public static ITeamContentProviderManager getTeamContentProviderManager() {
		return TeamContentProviderManager.getInstance();
	}
	
	/**
	 * Return a set of wizard pages for the given descriptions. If no wizard
	 * page is registered for a SCM URL scheme from a description then a page
	 * will not be created. If an extension exits, a page will be created and
	 * initialized with a set of corresponding descriptions.
	 * 
	 * <strong>EXPERIMENTAL</strong>. This class has been added as part of
	 * a work in progress. There is no guarantee that this API will work or that
	 * it will remain the same. Please do not use this API without consulting
	 * with the Team team.
	 * 
	 * @param descriptions
	 *            descriptions with SCM URLs
	 * @return initialized wizard pages
	 * @throws CoreException
	 *             if an error occurs while trying to create a page extension
	 * @since 3.6
	 */
	public static IScmUrlImportWizardPage[] getPages(ScmUrlImportDescription[] descriptions) throws CoreException {
		// TODO: check scmUrls
		IConfigurationElement[] elements = Platform.getExtensionRegistry().getConfigurationElementsFor(TeamUIPlugin.PLUGIN_ID, IScmUrlImportWizardPage.ATT_EXTENSION);
		if (elements.length > 0) {
			Set/*<IScmUrlImportWizardPage>*/ pages = new HashSet();
			for (int i = 0; i < elements.length; i++) {
				String repository = elements[i].getAttribute(IScmUrlImportWizardPage.ATT_REPOSITORY);
				RepositoryProviderType providerType = RepositoryProviderType.getProviderType(repository);
				String providerScheme = providerType.getFileSystemScheme();
				Set/*<URI>*/ schemeUris = new HashSet();
				// filter out descriptions for which provider is not known
				for (int j = 0; j < descriptions.length; j++) {
					URI scmUri = descriptions[j].getUri();
					if (scmUri != null) {
						if (ProjectSetCapability.SCHEME_SCM.equals(scmUri.getScheme())) {
							if (scmUri.getSchemeSpecificPart().startsWith(providerScheme)) {
								schemeUris.add(scmUri);
							}
						}
					}
				}
				if (schemeUris.size() > 0) {
					Object ext = TeamUIPlugin.createExtension(elements[i], IScmUrlImportWizardPage.ATT_PAGE);
					if (ext instanceof IScmUrlImportWizardPage) {
						IScmUrlImportWizardPage page = (IScmUrlImportWizardPage) ext;
						page.setProvider(providerType);
						Set/*<ScmUrlImportDescription>*/ set = new HashSet();
						for (Iterator iterator = schemeUris.iterator(); iterator.hasNext();) {
							URI uri = (URI) iterator.next();
							for (int j = 0; j < descriptions.length; j++) {
								if (descriptions[j].getUri().equals(uri)) {
									set.add(descriptions[j]);
								}
							}
						}
						page.setSelection((ScmUrlImportDescription[]) set.toArray(new ScmUrlImportDescription[0]));
						pages.add(page);
					}
				}
			}
			return (IScmUrlImportWizardPage[]) pages.toArray(new IScmUrlImportWizardPage[0]);
		}
		return null;
	}
}
