%{
/* This file contains routines used to convert GtkArg structures and arrays
 * to and from Python objects.  This code is used for the callback marshalling
 * code, signal handling, and GtkObject properties (gtk_object_[gs]et).
 *
 * It is collected here, so I don't accidentally reproduce it elsewhere in
 * pygtk. */

/* destroy notify for PyObject */
static void PyGtk_DestroyNotify(gpointer data) {
  Py_DECREF((PyObject *)data);
}
static void PyGtk_CallbackMarshal(GtkObject *o, gpointer d, int nargs,
				  GtkArg *args);
static PyObject *GtkArgs_AsTuple(int nparams, GtkArg *args);
static int GtkArgs_FromSequence(GtkArg *args, int nparams, PyObject *seq);

/* create a GtkArg from a PyObject, using the GTK_VALUE_* routines.
 * returns -1 if it couldn't set the argument. */
static int GtkArg_FromPyObject(GtkArg *arg, PyObject *obj) {
  PyObject *tmp;
  switch (GTK_FUNDAMENTAL_TYPE(arg->type)) {
  case GTK_TYPE_NONE:
  case GTK_TYPE_INVALID:
    GTK_VALUE_INT(*arg) = 0;
    break;
  case GTK_TYPE_BOOL:
    if (tmp = PyNumber_Int(obj))
      GTK_VALUE_BOOL(*arg) = (PyInt_AsLong(tmp) != 0);
    else
      return -1;
    Py_DECREF(tmp);
    break;
  case GTK_TYPE_CHAR:
    if (tmp = PyObject_Str(obj))
      GTK_VALUE_CHAR(*arg) = PyString_AsString(tmp)[0];
    else 
      return -1;
    Py_DECREF(tmp);
    break;
  case GTK_TYPE_ENUM:
    if (PyGtkEnum_get_value(arg->type, obj, &(GTK_VALUE_ENUM(*arg))))
      return -1;
    break;
  case GTK_TYPE_FLAGS:
    if (PyGtkFlag_get_value(arg->type, obj, &(GTK_VALUE_FLAGS(*arg))))
      return -1;
    break;
  case GTK_TYPE_INT:
    if (tmp = PyNumber_Int(obj))
      GTK_VALUE_INT(*arg) = PyInt_AsLong(tmp);
    else
      return -1;
    Py_DECREF(tmp);
    break;
  case GTK_TYPE_UINT:
    if (tmp = PyNumber_Int(obj))
      GTK_VALUE_UINT(*arg) = PyInt_AsLong(tmp);
    else
      return -1;
    Py_DECREF(tmp);
    break;
  case GTK_TYPE_LONG:
    if (tmp = PyNumber_Int(obj))
      GTK_VALUE_LONG(*arg) = PyInt_AsLong(tmp);
    else
      return -1;
    Py_DECREF(tmp);
    break;
  case GTK_TYPE_ULONG:
    if (tmp = PyNumber_Int(obj))
      GTK_VALUE_ULONG(*arg) = PyInt_AsLong(tmp);
    else
      return -1;
    Py_DECREF(tmp);
    break;
  case GTK_TYPE_FLOAT:
    if (tmp = PyNumber_Float(obj))
      GTK_VALUE_FLOAT(*arg) = PyFloat_AsDouble(tmp);
    else
      return -1;
    Py_DECREF(tmp);
    break;
  case GTK_TYPE_STRING:
    if (tmp = PyObject_Str(obj))
      GTK_VALUE_STRING(*arg) = PyString_AsString(tmp);
    else
      return -1;
    Py_DECREF(tmp);
    break;
  case GTK_TYPE_OBJECT:
    if (PyGtk_Check(obj))
      GTK_VALUE_OBJECT(*arg) = PyGtk_Get(obj);
    else
      return -1;
    break;
  case GTK_TYPE_BOXED:
    if (arg->type == GTK_TYPE_ACCEL_GROUP) {
      if (PyGtkAccelGroup_Check(obj))
	GTK_VALUE_BOXED(*arg) = PyGtkAccelGroup_Get(obj);
      else
	return -1;
    } else if (arg->type == GTK_TYPE_STYLE) {
      if (PyGtkStyle_Check(obj))
	GTK_VALUE_BOXED(*arg) = PyGtkStyle_Get(obj);
      else
	return -1;
    } else if (arg->type == GTK_TYPE_GDK_EVENT) {
      if (PyGdkEvent_Check(obj))
	GTK_VALUE_BOXED(*arg) = PyGdkEvent_Get(obj);
      else
	return -1;
    } else if (arg->type == GTK_TYPE_GDK_FONT) {
      if (PyGdkFont_Check(obj))
	GTK_VALUE_BOXED(*arg) = PyGdkFont_Get(obj);
      else
	return -1;
    } else if (arg->type == GTK_TYPE_GDK_COLOR) {
      if (PyGdkColor_Check(obj))
	GTK_VALUE_BOXED(*arg) = PyGdkColor_Get(obj);
      else
	return -1;
    } else if (arg->type == GTK_TYPE_GDK_WINDOW) {
      if (PyGdkWindow_Check(obj))
	GTK_VALUE_BOXED(*arg) = PyGdkWindow_Get(obj);
      else
	return -1;
    } else if (arg->type == GTK_TYPE_GDK_COLORMAP) {
      if (PyGdkColormap_Check(obj))
	GTK_VALUE_BOXED(*arg) = PyGdkColormap_Get(obj);
      else
	return -1;
    } else
      return -1;
    break;
  case GTK_TYPE_POINTER:
    if (PyCObject_Check(obj))
      GTK_VALUE_BOXED(*arg) = PyCObject_AsVoidPtr(obj);
    else
      return -1;
    break;
  case GTK_TYPE_FOREIGN:
    Py_INCREF(obj);
    GTK_VALUE_FOREIGN(*arg).data = obj;
    GTK_VALUE_FOREIGN(*arg).notify = PyGtk_DestroyNotify;
    break;
  case GTK_TYPE_SIGNAL:
    if (PyCallable_Check(obj)) {
      Py_INCREF(obj);
      GTK_VALUE_SIGNAL(*arg).f = NULL;
      GTK_VALUE_SIGNAL(*arg).d = obj;
    } else
      return -1;
    break;
  case GTK_TYPE_CALLBACK:
    if (PyCallable_Check(obj)) {
      Py_INCREF(obj);
      GTK_VALUE_CALLBACK(*arg).marshal =
	(GtkCallbackMarshal)PyGtk_CallbackMarshal;
      GTK_VALUE_CALLBACK(*arg).data = obj;
      GTK_VALUE_CALLBACK(*arg).notify = PyGtk_DestroyNotify;
    } else
      return -1;
    break;
  case GTK_TYPE_ARGS:
  case GTK_TYPE_C_CALLBACK:
    fprintf(stderr, "unsupported type");
    g_assert_not_reached();
    return -1;
  }
  return 0;
}

/* convert a GtkArg structure to a PyObject, using the GTK_VALUE_* routines.
 * conversion is always possible. */
static PyObject *GtkArg_AsPyObject(GtkArg *arg) {
  switch (GTK_FUNDAMENTAL_TYPE(arg->type)) {
  case GTK_TYPE_INVALID:
  case GTK_TYPE_NONE:
    Py_INCREF(Py_None);
    return Py_None;
  case GTK_TYPE_CHAR:
    return PyString_FromStringAndSize(&GTK_VALUE_CHAR(*arg), 1);
  case GTK_TYPE_BOOL:
    return PyInt_FromLong(GTK_VALUE_BOOL(*arg));
  case GTK_TYPE_ENUM:
  case GTK_TYPE_FLAGS:
  case GTK_TYPE_INT:
    return PyInt_FromLong(GTK_VALUE_INT(*arg));
  case GTK_TYPE_UINT:
    return PyInt_FromLong(GTK_VALUE_UINT(*arg));
  case GTK_TYPE_LONG:
    return PyInt_FromLong(GTK_VALUE_LONG(*arg));
  case GTK_TYPE_ULONG:
    return PyInt_FromLong(GTK_VALUE_ULONG(*arg));
  case GTK_TYPE_FLOAT:
    return PyFloat_FromDouble(GTK_VALUE_FLOAT(*arg));
  case GTK_TYPE_DOUBLE:
    return PyFloat_FromDouble(GTK_VALUE_DOUBLE(*arg));
  case GTK_TYPE_STRING:
    if (GTK_VALUE_STRING(*arg) != NULL)
      return PyString_FromString(GTK_VALUE_STRING(*arg));
    else {
      Py_INCREF(Py_None);
      return Py_None;
    }
  case GTK_TYPE_ARGS:
    return GtkArgs_AsTuple(GTK_VALUE_ARGS(*arg).n_args,
			   GTK_VALUE_ARGS(*arg).args);
  case GTK_TYPE_OBJECT:
    return PyGtk_New(GTK_VALUE_OBJECT(*arg));
  case GTK_TYPE_POINTER:
    return PyCObject_FromVoidPtr(GTK_VALUE_POINTER(*arg), NULL);
  case GTK_TYPE_BOXED:
    if (arg->type == GTK_TYPE_ACCEL_GROUP)
      return PyGtkAccelGroup_New(GTK_VALUE_BOXED(*arg));
    else if (arg->type == GTK_TYPE_STYLE)
      return PyGtkStyle_New(GTK_VALUE_BOXED(*arg));
    else if (arg->type == GTK_TYPE_GDK_EVENT)
      return PyGdkEvent_New(GTK_VALUE_BOXED(*arg));
    else if (arg->type == GTK_TYPE_GDK_FONT)
      return PyGdkFont_New(GTK_VALUE_BOXED(*arg));
    else if (arg->type == GTK_TYPE_GDK_COLOR)
      return PyGdkColor_New(GTK_VALUE_BOXED(*arg));
    else if (arg->type == GTK_TYPE_GDK_WINDOW)
      return PyGdkWindow_New(GTK_VALUE_BOXED(*arg));
    else if (arg->type == GTK_TYPE_GDK_COLORMAP)
      return PyGdkColormap_New(GTK_VALUE_BOXED(*arg));
    else
      return PyCObject_FromVoidPtr(GTK_VALUE_BOXED(*arg), NULL);
  case GTK_TYPE_FOREIGN:
    Py_INCREF((PyObject *)GTK_VALUE_FOREIGN(*arg).data);
    return (PyObject *)GTK_VALUE_FOREIGN(*arg).data;
  case GTK_TYPE_CALLBACK:
    Py_INCREF((PyObject *)GTK_VALUE_CALLBACK(*arg).data);
    return (PyObject *)GTK_VALUE_CALLBACK(*arg).data;
  case GTK_TYPE_SIGNAL:
    Py_INCREF((PyObject *)GTK_VALUE_SIGNAL(*arg).d);
    return (PyObject *)GTK_VALUE_SIGNAL(*arg).d;
  default:
    g_assert_not_reached();
    break;
  }
  return NULL;
}

/* set a GtkArg structure's data from a PyObject, using the GTK_RETLOC_*
 * routines.  If it can't make the conversion, set the return to a zero
 * equivalent. */
static void GtkRet_FromPyObject(GtkArg *ret, PyObject *py_ret) {
  PyObject *tmp;
  switch (GTK_FUNDAMENTAL_TYPE(ret->type)) {
  case GTK_TYPE_NONE:
  case GTK_TYPE_INVALID:
    break;
  case GTK_TYPE_BOOL:
    if (tmp = PyNumber_Int(py_ret)) {
      *GTK_RETLOC_BOOL(*ret) = (PyInt_AsLong(tmp) != 0);
      Py_DECREF(tmp);
    } else
      *GTK_RETLOC_BOOL(*ret) = FALSE;
    break;
  case GTK_TYPE_CHAR:
    if (tmp = PyObject_Str(py_ret)) {
      *GTK_RETLOC_CHAR(*ret) = PyString_AsString(tmp)[0];
      Py_DECREF(tmp);
    } else
      *GTK_RETLOC_CHAR(*ret) = '\0';
    break;
  case GTK_TYPE_ENUM:
    if (PyGtkEnum_get_value(ret->type, py_ret, GTK_RETLOC_ENUM(*ret))) {
      PyErr_Clear();
      *GTK_RETLOC_ENUM(*ret) = 0;
    }
    break;
  case GTK_TYPE_FLAGS:
    if (PyGtkFlag_get_value(ret->type, py_ret, GTK_RETLOC_FLAGS(*ret))) {
      PyErr_Clear();
      *GTK_RETLOC_FLAGS(*ret) = 0;
    }
    break;
  case GTK_TYPE_INT:
    if (tmp = PyNumber_Int(py_ret)) {
      *GTK_RETLOC_INT(*ret) = PyInt_AsLong(tmp);
      Py_DECREF(tmp);
    } else
      *GTK_RETLOC_INT(*ret) = 0;
    break;
  case GTK_TYPE_UINT:
    if (tmp = PyNumber_Int(py_ret)) {
      *GTK_RETLOC_UINT(*ret) = PyInt_AsLong(tmp);
      Py_DECREF(tmp);
    } else
      *GTK_RETLOC_UINT(*ret) = 0;
    break;
  case GTK_TYPE_LONG:
    if (tmp = PyNumber_Int(py_ret)) {
      *GTK_RETLOC_LONG(*ret) = PyInt_AsLong(tmp);
      Py_DECREF(tmp);
    } else
      *GTK_RETLOC_LONG(*ret) = 0;
    break;
  case GTK_TYPE_ULONG:
    if (tmp = PyNumber_Int(py_ret)) {
      *GTK_RETLOC_ULONG(*ret) = PyInt_AsLong(tmp);
      Py_DECREF(tmp);
    } else
      *GTK_RETLOC_ULONG(*ret) = 0;
    break;
  case GTK_TYPE_FLOAT:
    if (tmp = PyNumber_Float(py_ret)) {
      *GTK_RETLOC_FLOAT(*ret) = PyFloat_AsDouble(tmp);
      Py_DECREF(tmp);
    } else
      *GTK_RETLOC_FLOAT(*ret) = 0;
    break;
  case GTK_TYPE_DOUBLE:
    if (tmp = PyNumber_Float(py_ret)) {
      *GTK_RETLOC_DOUBLE(*ret) = PyFloat_AsDouble(tmp);
      Py_DECREF(tmp);
    } else
      *GTK_RETLOC_DOUBLE(*ret) = 0;
    break;
  case GTK_TYPE_STRING:
    if (tmp = PyObject_Str(py_ret)) {
      *GTK_RETLOC_STRING(*ret) = g_strdup(PyString_AsString(py_ret));
      Py_DECREF(tmp);
    } else
      *GTK_RETLOC_STRING(*ret) = NULL;
    break;
  case GTK_TYPE_OBJECT:
    if (PyGtk_Check(py_ret))
      *GTK_RETLOC_OBJECT(*ret) = PyGtk_Get(py_ret);
    else
      *GTK_RETLOC_OBJECT(*ret) = NULL;
    break;
  case GTK_TYPE_BOXED:
    if (ret->type == GTK_TYPE_ACCEL_GROUP) {
      if (PyGtkAccelGroup_Check(py_ret))
	*GTK_RETLOC_BOXED(*ret) = PyGtkAccelGroup_Get(py_ret);
      else
	*GTK_RETLOC_BOXED(*ret) = NULL;
    } else if (ret->type == GTK_TYPE_STYLE) {
      if (PyGtkStyle_Check(py_ret))
	*GTK_RETLOC_BOXED(*ret) = PyGtkStyle_Get(py_ret);
      else
	*GTK_RETLOC_BOXED(*ret) = NULL;
    } else if (ret->type == GTK_TYPE_GDK_EVENT) {
      if (PyGdkEvent_Check(py_ret))
	*GTK_RETLOC_BOXED(*ret) = PyGdkEvent_Get(py_ret);
      else
	*GTK_RETLOC_BOXED(*ret) = NULL;
    } else if (ret->type == GTK_TYPE_GDK_FONT) {
      if (PyGdkFont_Check(py_ret))
	*GTK_RETLOC_BOXED(*ret) = PyGdkFont_Get(py_ret);
      else
	*GTK_RETLOC_BOXED(*ret) = NULL;
    } else if (ret->type == GTK_TYPE_GDK_COLOR) {
      if (PyGdkColor_Check(py_ret))
	*GTK_RETLOC_BOXED(*ret) = PyGdkColor_Get(py_ret);
      else
	*GTK_RETLOC_BOXED(*ret) = NULL;
    } else if (ret->type == GTK_TYPE_GDK_WINDOW) {
      if (PyGdkWindow_Check(py_ret))
	*GTK_RETLOC_BOXED(*ret) = PyGdkWindow_Get(py_ret);
      else
	*GTK_RETLOC_BOXED(*ret) = NULL;
    } else if (ret->type == GTK_TYPE_GDK_COLORMAP) {
      if (PyGdkColormap_Check(py_ret))
	*GTK_RETLOC_BOXED(*ret) = PyGdkColormap_Get(py_ret);
      else
	*GTK_RETLOC_BOXED(*ret) = NULL;
    } else {
      if (PyCObject_Check(py_ret))
	*GTK_RETLOC_BOXED(*ret) = PyCObject_AsVoidPtr(py_ret);
      else
	*GTK_RETLOC_BOXED(*ret) = NULL;
    }
    break;
  case GTK_TYPE_POINTER:
    if (PyCObject_Check(py_ret))
      *GTK_RETLOC_POINTER(*ret) = PyCObject_AsVoidPtr(py_ret);
    else
      *GTK_RETLOC_POINTER(*ret) = NULL;
  default:
    g_assert_not_reached();
    break;
  }
}

/* convert a GtkArg structure to a PyObject, using the GTK_RETLOC_* routines.
 * conversion is always possible. */
static PyObject *GtkRet_AsPyObject(GtkArg *arg) {
  switch (GTK_FUNDAMENTAL_TYPE(arg->type)) {
  case GTK_TYPE_INVALID:
  case GTK_TYPE_NONE:
    Py_INCREF(Py_None);
    return Py_None;
  case GTK_TYPE_CHAR:
    return PyString_FromStringAndSize(GTK_RETLOC_CHAR(*arg), 1);
  case GTK_TYPE_BOOL:
    return PyInt_FromLong(*GTK_RETLOC_BOOL(*arg));
  case GTK_TYPE_ENUM:
  case GTK_TYPE_FLAGS:
  case GTK_TYPE_INT:
    return PyInt_FromLong(*GTK_RETLOC_INT(*arg));
  case GTK_TYPE_UINT:
    return PyInt_FromLong(*GTK_RETLOC_UINT(*arg));
  case GTK_TYPE_LONG:
    return PyInt_FromLong(*GTK_RETLOC_LONG(*arg));
  case GTK_TYPE_ULONG:
    return PyInt_FromLong(*GTK_RETLOC_ULONG(*arg));
  case GTK_TYPE_FLOAT:
    return PyFloat_FromDouble(*GTK_RETLOC_FLOAT(*arg));
  case GTK_TYPE_DOUBLE:
    return PyFloat_FromDouble(*GTK_RETLOC_DOUBLE(*arg));
  case GTK_TYPE_STRING:
    if (*GTK_RETLOC_STRING(*arg) != NULL)
      return PyString_FromString(*GTK_RETLOC_STRING(*arg));
    else {
      Py_INCREF(Py_None);
      return Py_None;
    }
  case GTK_TYPE_ARGS:
    break;
  case GTK_TYPE_OBJECT:
    return PyGtk_New(*GTK_RETLOC_OBJECT(*arg));
  case GTK_TYPE_POINTER:
    return PyCObject_FromVoidPtr(*GTK_RETLOC_POINTER(*arg), NULL);
  case GTK_TYPE_BOXED:
    if (arg->type == GTK_TYPE_ACCEL_GROUP)
      return PyGtkAccelGroup_New(*GTK_RETLOC_BOXED(*arg));
    else if (arg->type == GTK_TYPE_STYLE)
      return PyGtkStyle_New(*GTK_RETLOC_BOXED(*arg));
    else if (arg->type == GTK_TYPE_GDK_EVENT)
      return PyGdkEvent_New(*GTK_RETLOC_BOXED(*arg));
    else if (arg->type == GTK_TYPE_GDK_FONT)
      return PyGdkFont_New(*GTK_RETLOC_BOXED(*arg));
    else if (arg->type == GTK_TYPE_GDK_COLOR)
      return PyGdkColor_New(*GTK_RETLOC_BOXED(*arg));
    else if (arg->type == GTK_TYPE_GDK_WINDOW)
      return PyGdkWindow_New(*GTK_RETLOC_BOXED(*arg));
    else if (arg->type == GTK_TYPE_GDK_COLORMAP)
      return PyGdkColormap_New(*GTK_RETLOC_BOXED(*arg));
    else
      return PyCObject_FromVoidPtr(*GTK_RETLOC_BOXED(*arg), NULL);
  default:
    g_assert_not_reached();
    break;
  }
  return NULL;
}

/* convert the GtkArg array ARGS to a python tuple */
static PyObject *GtkArgs_AsTuple(int nparams, GtkArg *args) {
  PyObject *tuple, *item;
  int i;

  tuple = PyTuple_New(nparams);
  for (i = 0; i < nparams; i++) {
    item = GtkArg_AsPyObject(&args[i]);
    if (item == NULL) {
      Py_INCREF(Py_None);
      item = Py_None;
    }
    PyTuple_SetItem(tuple, i, item);
  }
  return tuple;
}

/* converts a Python sequence to a GtkArg array.  Returns -1 if the sequence
 * doesn't match the specification in ARGS */
static int GtkArgs_FromSequence(GtkArg *args, int nparams, PyObject *seq) {
  PyObject *item;
  int i;
  if (!PySequence_Check(seq))
    return -1;
  for (i = 0; i < nparams; i++) {
    item = PySequence_GetItem(seq, i);
    if (GtkArg_FromPyObject(&args[i], item)) {
      gchar buf[512];
      if (args[i].name == NULL)
	g_snprintf(buf, 511, "argument %d: expected %s, %s found", i+1,
		   gtk_type_name(args[i].type), item->ob_type->tp_name);
      else
	g_snprintf(buf, 511, "argument %s: expected %s, %s found",
		   args[i].name, gtk_type_name(args[i].type),
		   item->ob_type->tp_name);
      PyErr_SetString(PyExc_TypeError, buf);
      return -1;
    }
  }
  return 0;
}

/* generic callback marshal */
static void PyGtk_CallbackMarshal(GtkObject *o, gpointer data, int nargs,
				  GtkArg *args) {
  PyObject *func = data, *ret, *a, *params;

  a = GtkArgs_AsTuple(nargs, args);
  if (a == NULL) {
    PyErr_Clear();
    fprintf(stderr, "can't decode params -- callback not run\n");
    return;
  }
  if (o == NULL)
    params = a;
  else {
    /* prepend object to argument list */
    ret = PyTuple_New(1);
    PyTuple_SetItem(ret, 0, PyGtk_New(o));
    params = PySequence_Concat(ret, a);
    Py_DECREF(ret); Py_DECREF(a);
  }
  ret = PyObject_CallObject(func, params);
  Py_DECREF(params);
  if (ret == NULL) {
    PyErr_Print();
    PyErr_Clear();
    return;
  }
  GtkRet_FromPyObject(&args[nargs], ret);
  Py_DECREF(ret);
}

%}
