/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.utils;

import com.sun.jna.LastErrorException;
import java.io.FileDescriptor;
import java.io.IOException;
import java.lang.reflect.Field;
import java.nio.channels.FileChannel;
import java.util.concurrent.TimeUnit;
import org.apache.cassandra.config.CassandraRelevantProperties;
import org.apache.cassandra.io.FSWriteError;
import org.apache.cassandra.io.util.File;
import org.apache.cassandra.io.util.FileInputStreamPlus;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.JVMStabilityInspector;
import org.apache.cassandra.utils.NativeLibraryDarwin;
import org.apache.cassandra.utils.NativeLibraryLinux;
import org.apache.cassandra.utils.NativeLibraryWrapper;
import org.apache.cassandra.utils.NoSpamLogger;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class NativeLibrary {
    private static final Logger logger = LoggerFactory.getLogger(NativeLibrary.class);
    private static final boolean REQUIRE = !CassandraRelevantProperties.IGNORE_MISSING_NATIVE_FILE_HINTS.getBoolean();
    public static final OSType osType;
    private static final int MCL_CURRENT;
    private static final int MCL_FUTURE;
    private static final int ENOMEM = 12;
    private static final int F_GETFL = 3;
    private static final int F_SETFL = 4;
    private static final int F_NOCACHE = 48;
    private static final int O_DIRECT = 16384;
    private static final int O_RDONLY = 0;
    private static final int POSIX_FADV_NORMAL = 0;
    private static final int POSIX_FADV_RANDOM = 1;
    private static final int POSIX_FADV_SEQUENTIAL = 2;
    private static final int POSIX_FADV_WILLNEED = 3;
    private static final int POSIX_FADV_DONTNEED = 4;
    private static final int POSIX_FADV_NOREUSE = 5;
    private static final NativeLibraryWrapper wrappedLibrary;
    private static boolean jnaLockable;
    private static final Field FILE_DESCRIPTOR_FD_FIELD;
    private static final Field FILE_CHANNEL_FD_FIELD;

    private NativeLibrary() {
    }

    private static OSType getOsType() {
        String osName = CassandraRelevantProperties.OS_NAME.getString().toLowerCase();
        if (osName.contains("linux")) {
            return OSType.LINUX;
        }
        if (osName.contains("mac")) {
            return OSType.MAC;
        }
        logger.warn("the current operating system, {}, is unsupported by Cassandra", (Object)osName);
        if (osName.contains("aix")) {
            return OSType.AIX;
        }
        return OSType.LINUX;
    }

    private static int errno(RuntimeException e) {
        assert (e instanceof LastErrorException);
        try {
            return ((LastErrorException)e).getErrorCode();
        }
        catch (NoSuchMethodError x) {
            if (REQUIRE) {
                logger.warn("Obsolete version of JNA present; unable to read errno. Upgrade to JNA 3.2.7 or later");
            }
            return 0;
        }
    }

    public static boolean isAvailable() {
        return wrappedLibrary.isAvailable();
    }

    public static boolean jnaMemoryLockable() {
        return jnaLockable;
    }

    public static void tryMlockall() {
        block5: {
            try {
                wrappedLibrary.callMlockall(MCL_CURRENT);
                jnaLockable = true;
                logger.info("JNA mlockall successful");
            }
            catch (UnsatisfiedLinkError unsatisfiedLinkError) {
            }
            catch (RuntimeException e) {
                if (!(e instanceof LastErrorException)) {
                    throw e;
                }
                if (NativeLibrary.errno(e) == 12 && osType == OSType.LINUX) {
                    logger.warn("Unable to lock JVM memory (ENOMEM). This can result in part of the JVM being swapped out, especially with mmapped I/O enabled. Increase RLIMIT_MEMLOCK.");
                }
                if (osType == OSType.MAC) break block5;
                logger.warn("Unknown mlockall error {}", (Object)NativeLibrary.errno(e));
            }
        }
    }

    public static void trySkipCache(String path, long offset, long len) {
        File f = new File(path);
        if (!f.exists()) {
            return;
        }
        try (FileInputStreamPlus fis = new FileInputStreamPlus(f);){
            NativeLibrary.trySkipCache(NativeLibrary.getfd(fis.getChannel()), offset, len, path);
        }
        catch (IOException e) {
            logger.warn("Could not skip cache", (Throwable)e);
        }
    }

    public static void trySkipCache(int fd, long offset, long len, String path) {
        if (len == 0L) {
            NativeLibrary.trySkipCache(fd, 0L, 0, path);
        }
        while (len > 0L) {
            int sublen = (int)Math.min(Integer.MAX_VALUE, len);
            NativeLibrary.trySkipCache(fd, offset, sublen, path);
            len -= (long)sublen;
            offset -= (long)sublen;
        }
    }

    public static void trySkipCache(int fd, long offset, int len, String path) {
        if (fd < 0) {
            return;
        }
        try {
            int result;
            if (osType == OSType.LINUX && (result = wrappedLibrary.callPosixFadvise(fd, offset, len, 4)) != 0) {
                NoSpamLogger.log(logger, NoSpamLogger.Level.WARN, 10L, TimeUnit.MINUTES, "Failed trySkipCache on file: {} Error: " + wrappedLibrary.callStrerror(result).getString(0L), path);
            }
        }
        catch (UnsatisfiedLinkError result) {
        }
        catch (RuntimeException e) {
            if (!(e instanceof LastErrorException)) {
                throw e;
            }
            logger.warn("posix_fadvise({}, {}) failed, errno ({}).", new Object[]{fd, offset, NativeLibrary.errno(e)});
        }
    }

    public static int tryFcntl(int fd, int command, int flags) {
        int result;
        block4: {
            result = -1;
            try {
                result = wrappedLibrary.callFcntl(fd, command, flags);
            }
            catch (UnsatisfiedLinkError unsatisfiedLinkError) {
            }
            catch (RuntimeException e) {
                if (!(e instanceof LastErrorException)) {
                    throw e;
                }
                if (!REQUIRE) break block4;
                logger.warn("fcntl({}, {}, {}) failed, errno ({}).", new Object[]{fd, command, flags, NativeLibrary.errno(e)});
            }
        }
        return result;
    }

    public static int tryOpenDirectory(String path) {
        int fd;
        block4: {
            fd = -1;
            try {
                return wrappedLibrary.callOpen(path, 0);
            }
            catch (UnsatisfiedLinkError unsatisfiedLinkError) {
            }
            catch (RuntimeException e) {
                if (!(e instanceof LastErrorException)) {
                    throw e;
                }
                if (!REQUIRE) break block4;
                logger.warn("open({}, O_RDONLY) failed, errno ({}).", (Object)path, (Object)NativeLibrary.errno(e));
            }
        }
        return fd;
    }

    public static void trySync(int fd) {
        block5: {
            if (fd == -1) {
                return;
            }
            try {
                wrappedLibrary.callFsync(fd);
            }
            catch (UnsatisfiedLinkError unsatisfiedLinkError) {
            }
            catch (RuntimeException e) {
                if (!(e instanceof LastErrorException)) {
                    throw e;
                }
                if (!REQUIRE) break block5;
                String errMsg = String.format("fsync(%s) failed, errno (%s) %s", fd, NativeLibrary.errno(e), e.getMessage());
                logger.warn(errMsg);
                throw new FSWriteError((Throwable)e, errMsg);
            }
        }
    }

    public static void tryCloseFD(int fd) {
        block5: {
            if (fd == -1) {
                return;
            }
            try {
                wrappedLibrary.callClose(fd);
            }
            catch (UnsatisfiedLinkError unsatisfiedLinkError) {
            }
            catch (RuntimeException e) {
                if (!(e instanceof LastErrorException)) {
                    throw e;
                }
                if (!REQUIRE) break block5;
                String errMsg = String.format("close(%d) failed, errno (%d).", fd, NativeLibrary.errno(e));
                logger.warn(errMsg);
                throw new FSWriteError((Throwable)e, errMsg);
            }
        }
    }

    public static int getfd(FileChannel channel) {
        try {
            return NativeLibrary.getfd((FileDescriptor)FILE_CHANNEL_FD_FIELD.get(channel));
        }
        catch (IllegalAccessException | IllegalArgumentException e) {
            if (REQUIRE) {
                logger.warn("Unable to read fd field from FileChannel", (Throwable)e);
            }
            return -1;
        }
    }

    public static int getfd(FileDescriptor descriptor) {
        try {
            return FILE_DESCRIPTOR_FD_FIELD.getInt(descriptor);
        }
        catch (Exception e) {
            if (REQUIRE) {
                JVMStabilityInspector.inspectThrowable(e);
                logger.warn("Unable to read fd field from FileDescriptor", (Throwable)e);
            }
            return -1;
        }
    }

    public static long getProcessID() {
        block3: {
            try {
                return wrappedLibrary.callGetpid();
            }
            catch (UnsatisfiedLinkError unsatisfiedLinkError) {
            }
            catch (Exception e) {
                if (!REQUIRE) break block3;
                logger.info("Failed to get PID from JNA", (Throwable)e);
            }
        }
        return -1L;
    }

    static {
        jnaLockable = false;
        FILE_DESCRIPTOR_FD_FIELD = FBUtilities.getProtectedField(FileDescriptor.class, "fd");
        try {
            FILE_CHANNEL_FD_FIELD = FBUtilities.getProtectedField(Class.forName("sun.nio.ch.FileChannelImpl"), "fd");
        }
        catch (ClassNotFoundException e) {
            throw new RuntimeException(e);
        }
        osType = NativeLibrary.getOsType();
        switch (osType) {
            case MAC: {
                wrappedLibrary = new NativeLibraryDarwin();
                break;
            }
            default: {
                wrappedLibrary = new NativeLibraryLinux();
            }
        }
        if (CassandraRelevantProperties.OS_ARCH.getString().toLowerCase().contains("ppc")) {
            if (osType == OSType.LINUX) {
                MCL_CURRENT = 8192;
                MCL_FUTURE = 16384;
            } else if (osType == OSType.AIX) {
                MCL_CURRENT = 256;
                MCL_FUTURE = 512;
            } else {
                MCL_CURRENT = 1;
                MCL_FUTURE = 2;
            }
        } else {
            MCL_CURRENT = 1;
            MCL_FUTURE = 2;
        }
    }

    public static enum OSType {
        LINUX,
        MAC,
        AIX,
        OTHER;

    }
}

