//----------------------------------------------------------------------------
//
// C++ Objects for Allegro's gui
//
// Douglas Eleveld (D.J.Eleveld@anest.azg.nl)
//
//----------------------------------------------------------------------------

//----------------------------------------------------------------------------
// Gui object descriptions class
//----------------------------------------------------------------------------
#include <stdio.h>
#include <string.h>
#include "descript.h"
#include "internal.h"

//----------------------------------------------------------------------------
// Some constants used for encoding and decoding objects
const char *box_type          = "box_object";
const char *shadow_box_type   = "shadow_box_object";
const char *panel_raised_type = "panel_raised_object";
const char *panel_sunken_type = "panel_sunken_object";
const char *panel_ridge_type  = "panel_ridge_object";
const char *panel_groove_type = "panel_groove_object";
const char *bitmap_type       = "bitmap_object";
const char *text_type         = "text_object";
const char *centerd_text_type = "centerd_text_object";
const char *button_type       = "button_object";
const char *checkbox_type     = "checkbox_object";
const char *radio_button_type = "radio_button_object";
const char *edittext_type     = "edittext_object";
const char *list_type         = "list_object";
const char *textbox_type      = "textbox_object";
const char *progress_bar_type = "progress_bar_object";
const char *window_type       = "window_object";
const char *hidden_list_type  = "hidden_list_object";

const char *illegal_type      = "illegal_object";

//----------------------------------------------------------------------------
// Dialog base class object functions
//----------------------------------------------------------------------------
// Constructor
object_description::object_description (void)
   :index(0),
   type(NULL),
   name(NULL),
   arguments(NULL)
   {
   }
//----------------------------------------------------------------------------
// Destructor
object_description::~object_description (void)
   {
   if(type!=NULL) delete type;
   if(name!=NULL) delete name;
   if(arguments!=NULL) delete arguments;
   }
//----------------------------------------------------------------------------
// Set the objects type
void object_description::set_type (const char *data)
   {
   // Possible remone old stuff
   if(type!=NULL) delete type;

   // Possibly remove stuff
   if((data==NULL)||(strlen(data)<1))
      {
      type = NULL;
      }
   // Change the stuff
   else
      {
      type = new char[strlen(data)+1];
      if(type==NULL) degui_no_memory();
      strcpy(type,data);
      }
   }
//----------------------------------------------------------------------------
// Set the objects name
void object_description::set_name (const char *data)
   {
   // Possible remone old stuff
   if(name!=NULL) delete name;

   // Possibly remove stuff
   if((data==NULL)||(strlen(data)<1))
      {
      name = NULL;
      }
   // Change the stuff
   else
      {
      name = new char[strlen(data)+1];
      if(name==NULL) degui_no_memory();
      strcpy(name,data);
      }
   }
//----------------------------------------------------------------------------
// Set the objects arguments
void object_description::set_arguments (const char *data)
   {
   // Possible remone old stuff
   if(arguments!=NULL) delete arguments;

   // Possibly remove stuff
   if((data==NULL)||(strlen(data)<1))
      {
      arguments = NULL;
      }
   // Change the stuff
   else
      {
      arguments = new char[strlen(data)+1];
      if(arguments==NULL) degui_no_memory();
      strcpy(arguments,data);
      }
   }
//----------------------------------------------------------------------------
// Get the objects name
const char* object_description::get_name (void)
   {
   // Some temporary space for retrned names
   static char data[255];

   // Build default name
   if(name==NULL)
      {
      sprintf(data,"object_%i",index);
      return data;
      }
   else return name;
   }
//----------------------------------------------------------------------------
// Encode an object
void object_description::encode (char *d)
   {
   strcpy(d,get_type());
   strcat(d,",");

   if(name!=NULL) strcat(d,get_name());
   else strcat(d,"NULL");
   strcat(d,",");

   if(arguments!=NULL) strcat(d,get_arguments());
   else strcat(d,"NULL");
   strcat(d,";");
   }
//----------------------------------------------------------------------------
// Helper function for decoding
char* object_description::remove_quotes (char *d)
   {
   char *end_quote = strchr(&d[1],'"');
   if(end_quote!=0)
      {
      end_quote[0] = '\0';
      return &d[1];
      }
   else return d;
   }
//----------------------------------------------------------------------------
// Decode an object
dialog_object* object_description::decode (char *d)
   {
   // The returned object
   dialog_object *ret;

   // Look at the first part, the type
   char *type = strtok(d,",");
   char *name = strtok(NULL,",");
   char *args = strtok(NULL,";");

   // The arguments will be the same
   if(stricmp(args,"NULL")==0) args = NULL;
   set_arguments(args);

   // Set other info about the object
   if(stricmp(name,"NULL")==0) name = NULL;
   set_name(name);

   // Decode a box_object
   if(stricmp(type,box_type)==0)
      {
      ret = new box_object;
      set_type(box_type);
      }
   // Decode a shadow box_object
   else if(stricmp(type,shadow_box_type)==0)
      {
      ret = new shadow_box_object;
      set_type(shadow_box_type);
      }
   // Decode panel objects
   else if(stricmp(type,panel_raised_type)==0)
      {
      ret = new panel_raised_object;
      set_type(panel_raised_type);
      }
   else if(stricmp(type,panel_sunken_type)==0)
      {
      ret = new panel_sunken_object;
      set_type(panel_sunken_type);
      }
   else if(stricmp(type,panel_ridge_type)==0)
      {
      ret = new panel_ridge_object;
      set_type(panel_ridge_type);
      }
   else if(stricmp(type,panel_groove_type)==0)
      {
      ret = new panel_groove_object;
      set_type(panel_groove_type);
      }
   // Decode a bitmap object
   else if(stricmp(type,bitmap_type)==0)
      {
      // Check for a filename
      if(args[0]=='"')
         {
         args = remove_quotes(args);
         ret = new helper_bitmap_object(args);
         }
      // Must be a pointer name
      else
         {
         ret = new helper_bitmap_object(args);
         }
      set_type(bitmap_type);
      }
   // Decode a text object
   else if(stricmp(type,text_type)==0)
      {
      // Check for text
      if(args[0]=='"')
         {
         args = remove_quotes(args);
         ret = new helper_text_object(args);
         }
      // Must be a pointer name
      else
         {
         ret = new helper_text_object(args);
         }
      set_type(text_type);
      }
   // Decode some centerd text
   else if(stricmp(type,centerd_text_type)==0)
      {
      // Check for text
      if(args[0]=='"')
         {
         args = remove_quotes(args);
         ret = new helper_centerd_text_object(args);
         }
      // Must be a pointer name
      else
         {
         ret = new helper_centerd_text_object(args);
         }
      set_type(centerd_text_type);
      }
   // Decode a button
   else if(stricmp(type,button_type)==0)
      {
      // Split off the possible arguments
      char *arg1 = strtok(args,",");
      char *arg2 = strtok(NULL,",");
      char *arg3 = strtok(NULL,";");

      // Get the shortcut key
      int shortcut = atoi(arg2);

      // Find out if the button is supposed to close the dialog
      bool close_dialog = false;
      if((arg3[0]=='t')||(arg3[0]=='T')) close_dialog = true;

      // Look at the first argument to pull out the string
      if(arg1[0]=='"')
         {
         arg1 = remove_quotes(arg1);
         ret = new helper_button_object(arg1,shortcut,close_dialog);
         }
      // Must be a pointer name
      else
         {
         ret = new helper_button_object(arg1,shortcut,close_dialog);
         }
      set_type(button_type);
      }
   // Decode a checkbox type
   else if(stricmp(type,checkbox_type)==0)
      {
      // Look at the first argument to pull out the string
      if(args[0]=='"')
         {
         args = remove_quotes(args);
         ret = new helper_checkbox_object(args);
         }
      // Must be a pointer name
      else
         {
         set_arguments(args);
         ret = new helper_checkbox_object(args);
         }
      set_type(checkbox_type);
      }
   // Decode a radio button type
   else if(stricmp(type,radio_button_type)==0)
      {
      // Split the arguments off
      char *arg1 = strtok(args,",");
      char *arg2 = strtok(NULL,",");
      char *arg3 = strtok(NULL,";");

      // Get the group number
      int group = atoi(arg2);

      // Find out if the button is supposed to be square or not
      bool square = false;
      if((arg3[0]=='t')||(arg3[0]=='T')) square = true;

      // Look at the first argument to pull out the string
      if(arg1[0]=='"')
         {
         arg1 = remove_quotes(arg1);
         ret = new helper_radio_button_object(arg1,group,square);
         }
      else
         {
         ret = new helper_radio_button_object(arg1,group,square);
         }
      set_type(radio_button_type);
      }
   // Decode a edittext object
   else if(stricmp(type,edittext_type)==0)
      {
      // Split the arguments off
      strtok(args,",");
      char *arg2 = strtok(NULL,";");

      int length = atoi(arg2);

      ret = new edittext_object(NULL,length);

      set_type(edittext_type);
      }
   // Decode a list object
   else if(stricmp(type,list_type)==0)
      {
      ret = new list_object(NULL);

      set_type(list_type);
      }
   // Decode a textbox object
   else if(stricmp(type,textbox_type)==0)
      {
      // Split the arguments off
      strtok(args,",");
      char *arg2 = strtok(NULL,",");
      char *arg3 = strtok(NULL,";");

      // Decode the wrapping
      bool wrap = false;
      bool wwrap = false;
      if((arg2[0]=='t')||(arg2[0]=='T')) wrap = true;
      if((arg3[0]=='t')||(arg3[0]=='T')) wwrap = true;

      ret = new textbox_object(NULL,wrap,wwrap);

      set_type(textbox_type);
      }
   // Decode a progress_bar object
   else if(stricmp(type,progress_bar_type)==0)
      {
      ret = new progress_bar_object;
      set_type(progress_bar_type);

      }
   // Decode window objects
   else if(stricmp(type,window_type)==0)
      {
      ret = new window_object;
      set_type(window_type);
      }
   // Decode hidden list objects
   else if(stricmp(type,hidden_list_type)==0)
      {
      ret = new hidden_list_object;
      set_type(hidden_list_type);
      }
   // Unknown type means a user object
   else
      {
      char temp[1000];
      sprintf(temp,"%s",type);
      if((args!=NULL)&&(strlen(args)>1))
         {
         strcat(temp,"(");
         strcat(temp,args);
         strcat(temp,")");
         }
      strcat(temp,";");

      ret = new helper_object(temp);

      set_type(type);
      }
   // Check for a good allocation
   if(ret==NULL) degui_no_memory();

   return ret;
   }

//----------------------------------------------------------------------------
// Helper object for the builder
//----------------------------------------------------------------------------
// Constructor
helper_object::helper_object (char *t)
   :text_object(t)
   {
   char *temp = new char[strlen(t)+1];
   if(temp==NULL) degui_no_memory();
   strcpy(temp,t);

   set_text(temp);

   set_x(100);
   set_y(100);
   set_w(100);
   set_h(100);
   }
//----------------------------------------------------------------------------
// Destructor
helper_object::~helper_object (void)
   {
   delete text;
   }
//----------------------------------------------------------------------------
// Deal with special cases for messages
void helper_object::msg_draw (void)
   {
   text_mode(color ? color->back() : degui_back_color);
   textout(screen, font, (unsigned char*)text, x(), y()+(h())/2, color ? color->fore() : degui_fore_color);
   rect(screen,x(),y(),x()+w(),y()+h(),color ? color->fore() : degui_fore_color);
   }

//----------------------------------------------------------------------------
// Helper bitmap object for the builder
//----------------------------------------------------------------------------
// Constructor
helper_bitmap_object::helper_bitmap_object (char *t)
   :bitmap_object(t)
   {
   // Remember the text
   text = new char[strlen(t)+1];
   if(text==NULL) degui_no_memory();
   strcpy(text,t);
   }
//----------------------------------------------------------------------------
// Destructor
helper_bitmap_object::~helper_bitmap_object (void)
   {
   delete text;
   }
//----------------------------------------------------------------------------
// Deal with special cases for messages
void helper_bitmap_object::msg_draw (void)
   {
   bitmap_object::msg_draw();

   if(b==NULL)
      {
      text_mode(color ? color->back() : degui_back_color);
      textout(screen, font, (unsigned char*)text, x(), y()+(h())/2, color ? color->fore() : degui_fore_color);
      rect(screen,x(),y(),x2(),y2(),color ? color->fore() : degui_fore_color);
      }
   }

//----------------------------------------------------------------------------
// Constructor
helper_text_object::helper_text_object (char *t)
   :text_object(t)
   {
   // Remember the text
   char *temp = new char[strlen(t)+1];
   if(temp==NULL) degui_no_memory();
   strcpy(temp,t);
   text = temp;
   }
//----------------------------------------------------------------------------
// Destructor
helper_text_object::~helper_text_object (void)
   {
   delete text;
   }

//----------------------------------------------------------------------------
// Helper ceneterd text object for the builder
//----------------------------------------------------------------------------
// Constructor
helper_centerd_text_object::helper_centerd_text_object (char *t)
   :centerd_text_object(t)
   {
   // Remember the text
   char *temp = new char[strlen(t)+1];
   if(temp==NULL) degui_no_memory();
   strcpy(temp,t);
   text = temp;
   }
//----------------------------------------------------------------------------
// Destructor
helper_centerd_text_object::~helper_centerd_text_object (void)
   {
   delete text;
   }

//----------------------------------------------------------------------------
// Helper button object for the builder
//----------------------------------------------------------------------------
// Constructor
helper_button_object::helper_button_object (char *t, const int k, const bool e)
   :button_object(t,k,e)
   {
   // Remember the text
   char *temp = new char[strlen(t)+1];
   if(temp==NULL) degui_no_memory();
   strcpy(temp,t);
   set_text(temp);
   }
//----------------------------------------------------------------------------
// Destructor
helper_button_object::~helper_button_object (void)
   {
   delete text;
   }

//----------------------------------------------------------------------------
// Helper checkbox object for the builder
//----------------------------------------------------------------------------
// Constructor
helper_checkbox_object::helper_checkbox_object (char *t)
   :checkbox_object(t)
   {
   // Remember the text
   char *temp = new char[strlen(t)+1];
   if(temp==NULL) degui_no_memory();
   strcpy(temp,t);
   text = temp;
   }
//----------------------------------------------------------------------------
// Destructor
helper_checkbox_object::~helper_checkbox_object (void)
   {
   delete text;
   }

//----------------------------------------------------------------------------
// Helper radio object for the builder
//----------------------------------------------------------------------------
// Constructor
helper_radio_button_object::helper_radio_button_object (char *t, const int g, const bool s)
   :radio_button_object(t,g,s)
   {
   // Remember the text
   char *temp = new char[strlen(t)+1];
   if(temp==NULL) degui_no_memory();
   strcpy(temp,t);
   text = temp;
   }
//----------------------------------------------------------------------------
// Destructor
helper_radio_button_object::~helper_radio_button_object (void)
   {
   delete text;
   }
//----------------------------------------------------------------------------
// Helper window object for the builder
//----------------------------------------------------------------------------
// Constructor
helper_window_object::helper_window_object (char *t, const int g)
   :window_object(t,g)
   {
   // Remember the text
   char *temp = new char[strlen(t)+1];
   if(temp==NULL) degui_no_memory();
   strcpy(temp,t);
   set_text(temp);
   }
//----------------------------------------------------------------------------
// Destructor
helper_window_object::~helper_window_object (void)
   {
   delete text;
   }
//----------------------------------------------------------------------------



