//----------------------------------------------------------------------------
//
// C++ Objects for Allegro's gui
//
// Douglas Eleveld (D.J.Eleveld@anest.azg.nl)
//
//----------------------------------------------------------------------------

//----------------------------------------------------------------------------
// Some C Objects for Allegro's gui
//----------------------------------------------------------------------------
#include "degui.h"
#include "internal.h"
#include <ctype.h>

//----------------------------------------------------------------------------
// Global helper functions for the menu object
//----------------------------------------------------------------------------
// Calculates the coordinates of an object within a top level menu bar.
static void get_menu_pos (MENU_INFO *m, int c, int *x, int *y, int *w)
   {
   int c2;

   if (m->bar)
      {
      *x = m->x+1;

      for (c2=0; c2<c; c2++)
         *x += degui_strlen(m->menu[c2].text) + 16;

      *y = m->y+1;
      *w = degui_strlen(m->menu[c].text) + 16;
      }
   else
      {
      *x = m->x+1;
      *y = m->y+c*(text_height(font)+4)+1;
      *w = m->w-2;
      }
   }
// Draws an item from a popup menu onto the screen.
static void draw_menu_item (MENU_INFO *m, int c)
   {
   int x, y, w;

   int bg = degui_deselect_color;
   if(c == m->sel) bg = degui_select_color;

   get_menu_pos(m, c, &x, &y, &w);

   rectfill(screen, x, y, x+w-1, y+text_height(font)+3, bg);
   text_mode(bg);
   if (m->menu[c].text[0])
      degui_textout(screen, m->menu[c].text, x+8, y+1, degui_fore_color, FALSE);
   else
      hline(screen, x, y+text_height(font)/2+2, x+w, degui_mid_color);
   }


/* draw_menu:
*  Draws a popup menu onto the screen.
*/
void _draw_menu (MENU_INFO *m)
   {
   int c;

//   rect(screen, m->x, m->y, m->x+m->w-1, m->y+m->h-1, gui_fg_color);
   draw_3d_frame(screen, m->x, m->y, m->x+m->w-1, m->y+m->h-1,-1,degui_light_shad_color,degui_dark_shad_color);

//   vline(screen, m->x+m->w, m->y+1, m->y+m->h, gui_fg_color);
//   hline(screen, m->x+1, m->y+m->h, m->x+m->w, gui_fg_color);

   for (c=0; m->menu[c].text; c++)
      {
      draw_menu_item(m, c);
      }
   }

/* menu_mouse_object:
 *  Returns the index of the object the mouse is currently on top of.
 */
static int menu_mouse_object(MENU_INFO *m)
{
   int c;
   int x, y, w;

   for (c=0; c<m->size; c++) {
      get_menu_pos(m, c, &x, &y, &w);

      if ((mouse_x >= x) && (mouse_x < x+w) &&
	  (mouse_y >= y) && (mouse_y < y+(text_height(font)+4)))
	 return (m->menu[c].text[0]) ? c : -1;
   }

   return -1;
}



/* mouse_in_parent_menu:
 *  Recursively checks if the mouse is inside a menu or any of its parents.
 */
static int mouse_in_parent_menu(MENU_INFO *m) 
{
   int c;

   if (!m)
      return FALSE;

   c = menu_mouse_object(m);
   if ((c >= 0) && (c != m->sel))
      return TRUE;

   return mouse_in_parent_menu(m->parent);
}



/* fill_menu_info:
 *  Fills a menu info structure when initialising a menu.
 */
void _fill_menu_info(MENU_INFO *m, MENU *menu, MENU_INFO *parent, int bar, int x, int y)
{
   int c;

   m->menu = menu;
   m->parent = parent;
   m->bar = bar;
   m->x = x;
   m->y = y;
   m->w = 2;
   m->h = (m->bar) ? (text_height(font)+6) : 2;
   m->proc = NULL;
   m->sel = -1;

   /* calculate size of the menu */
   for (m->size=0; m->menu[m->size].text; m->size++) {
      c = degui_strlen(m->menu[m->size].text);

      if (m->bar) {
	 m->w += c+16;
      }
      else {
	 m->h += text_height(font)+4;
	 m->w = MAX(m->w, c+16);
      }
   }
}



/* menu_key_shortcut:
 *  Returns true if c is indicated as a keyboard shortcut by a '&' character
 *  in the specified string.
 */
static int menu_key_shortcut(int c, char *s)
{
   while (*s) {
      if (*s == '&') {
	 s++;
	 if ((*s != '&') && (tolower(*s) == tolower(c & 0xff)))
	    return TRUE;
      }
      s++;
   }

   return FALSE;
}



/* menu_alt_key:
 *  Searches a menu for keyboard shortcuts, for the alt+letter to bring
 *  up a menu.
 */
int _menu_alt_key(int k, MENU *m)
{
   char *s;
   int c;

   if (k & 0xff)
      return 0;

   k = key_ascii_table[k>>8];

   for (c=0; m[c].text; c++) {
      s = m[c].text;
      while (*s) {
	 if (*s == '&') {
	    s++;
	    if ((*s != '&') && (tolower(*s) == tolower(k)))
	       return k;
	 }
	 s++;
      }
   }

   return 0;
}



/* _do_menu:
 *  The core menu control function, called by do_menu() and d_menu_proc().
 */
int _do_degui_menu(MENU *menu, MENU_INFO *parent, int bar, int x, int y, int repos, int *dret)
{
   MENU_INFO m;
   MENU_INFO *i;
   int c, c2;
   int ret = -1;
   int mouse_on = mouse_b;
   int old_sel;
   int mouse_sel;
   int _x, _y;
   int redraw = TRUE;
   show_mouse(NULL);

   _fill_menu_info(&m, menu, parent, bar, x, y);

   if (repos) {
      m.x = MID(0, m.x, SCREEN_W-m.w-1);
      m.y = MID(0, m.y, SCREEN_H-m.h-1);
   }

   /* save screen under the menu */
   m.saved = create_bitmap(m.w+1, m.h+1); 

   if (m.saved)
      blit(screen, m.saved, m.x, m.y, 0, 0, m.w+1, m.h+1);
   else
      errno = ENOMEM;

   m.sel = mouse_sel = menu_mouse_object(&m);
   if ((m.sel < 0) && (!mouse_b))
      m.sel = 0;

   show_mouse(screen);

   do {
      old_sel = m.sel;

      c = menu_mouse_object(&m);
      if ((mouse_b) || (c != mouse_sel))
	 m.sel = mouse_sel = c;

      if (mouse_b) {                                  /* if button pressed */
	 if ((mouse_x < m.x) || (mouse_x > m.x+m.w) ||
	     (mouse_y < m.y) || (mouse_y > m.y+m.h)) {
	    if (!mouse_on)                            /* dismiss menu? */
	       break;

	    if (mouse_in_parent_menu(m.parent))       /* back to parent? */
	       break;
	 }

	 if ((m.sel >= 0) && (m.menu[m.sel].child))   /* bring up child? */
	    ret = m.sel;

	 mouse_on = TRUE;
	 clear_keybuf();
      }
      else {                                          /* button not pressed */
	 if (mouse_on)                                /* selected an item? */
	    ret = m.sel;

	 mouse_on = FALSE;

	 if (keypressed()) {                          /* keyboard input */
	    c = readkey();

	    if ((c & 0xff) == 27) {
	       ret = -1;
	       goto getout;
	    }

	    switch (c >> 8) {

	       case KEY_LEFT:
		  if (m.parent) {
		     if (m.parent->bar) {
			simulate_keypress(KEY_LEFT<<8);
			simulate_keypress(KEY_DOWN<<8);
		     }
		     ret = -1;
		     goto getout;
		  }
		  /* fall through */

	       case KEY_UP:
		  if ((((c >> 8) == KEY_LEFT) && (m.bar)) ||
		      (((c >> 8) == KEY_UP) && (!m.bar))) {
		     c = m.sel;
		     do {
			c--;
			if (c < 0)
			   c = m.size - 1;
		     } while ((!(m.menu[c].text[0])) && (c != m.sel));
		     m.sel = c;
		  }
		  break;

	       case KEY_RIGHT:
		  if (((m.sel < 0) || (!m.menu[m.sel].child)) &&
		      (m.parent) && (m.parent->bar)) {
		     simulate_keypress(KEY_RIGHT<<8);
		     simulate_keypress(KEY_DOWN<<8);
		     ret = -1;
		     goto getout;
		  }
		  /* fall through */

	       case KEY_DOWN:
		  if ((m.sel >= 0) && (m.menu[m.sel].child) &&
		      ((((c >> 8) == KEY_RIGHT) && (!m.bar)) ||
		       (((c >> 8) == KEY_DOWN) && (m.bar)))) {
		     ret = m.sel;
		  }
		  else if ((((c >> 8) == KEY_RIGHT) && (m.bar)) ||
			   (((c >> 8) == KEY_DOWN) && (!m.bar))) {
		     c = m.sel;
		     do {
			c++;
			if (c >= m.size)
			   c = 0;
		     } while ((!(m.menu[c].text[0])) && (c != m.sel));
		     m.sel = c;
		  }
		  break;

	       case KEY_SPACE:
	       case KEY_ENTER:
		  if (m.sel >= 0)
		     ret = m.sel;
		  break;

	       default:
		  if ((!m.parent) && ((c & 0xff) == 0))
		     c = _menu_alt_key(c, m.menu);
		  for (c2=0; m.menu[c2].text; c2++) {
		     if (menu_key_shortcut(c, m.menu[c2].text)) {
			ret = m.sel = c2;
			break;
		     }
		  }
		  if (m.parent) {
		     i = m.parent;
		     for (c2=0; i->parent; c2++)
			i = i->parent;
		     c = _menu_alt_key(c, i->menu);
		     if (c) {
			while (c2-- > 0)
			   simulate_keypress(27);
			simulate_keypress(c);
			ret = -1;
			goto getout;
		     }
		  }
		  break;
	    }
	 }
      }

      if ((redraw) || (m.sel != old_sel)) {           /* selection changed? */
	 show_mouse(NULL);

	 if (redraw) {
	    _draw_menu(&m);
	    redraw = FALSE;
	 }
	 else {
	    if (old_sel >= 0)
	       draw_menu_item(&m, old_sel);

	    if (m.sel >= 0)
	       draw_menu_item(&m, m.sel);
	 }

	 show_mouse(screen);
      }

      if (ret >= 0) {                                 /* child menu? */
	 if (m.menu[ret].child) {
	    if (m.bar) {
	       get_menu_pos(&m, ret, &_x, &_y, &c);
	       _x += 6;
	       _y += text_height(font)+7;
	    }
	    else {
	       _x = m.x+m.w*2/3;
	       _y = m.y + (text_height(font)+4)*ret + text_height(font)/4+2;
	    }
	    c = _do_degui_menu(m.menu[ret].child, &m, FALSE, _x, _y, TRUE, NULL);
	    if (c < 0) {
	       ret = -1;
	       mouse_on = FALSE;
	       mouse_sel = menu_mouse_object(&m);
	    }
	 }
      }

      if ((m.bar) && (!mouse_b) && (!keypressed()) &&
	  ((mouse_x < m.x) || (mouse_x > m.x+m.w) ||
	   (mouse_y < m.y) || (mouse_y > m.y+m.h)))
	 break;

   } while (ret < 0);

   getout:

   if (dret)
      *dret = 0;

   /* callback function? */
   if (!m.proc)
      m.proc = m.menu[ret].proc;

   if (ret >= 0) {
      if (parent)
	 parent->proc = m.proc;
      else  {
	 if (m.proc) {
	    c = m.proc();
	    if (dret)
	       *dret = c;
	 }
      }
   }

   /* restore screen */
   if (m.saved) {
      show_mouse(NULL);
      blit(m.saved, screen, 0, 0, m.x, m.y, m.w+1, m.h+1);
      destroy_bitmap(m.saved);
   }

   show_mouse(screen);

   return ret;
}



/* do_menu:
 *  Displays and animates a popup menu at the specified screen position,
 *  returning the index of the item that was selected, or -1 if it was
 *  dismissed. If the menu crosses the edge of the screen it will be moved.
 */
int do_degui_menu(MENU *menu, int x, int y)
{
   int ret = _do_degui_menu(menu, NULL, FALSE, x, y, TRUE, NULL);

   do {
   } while (mouse_b);

   return ret;
}



/* d_degui_menu_proc:
 *  Dialog procedure for adding drop down menus to a GUI dialog. This 
 *  displays the top level menu items as a horizontal bar (eg. across the
 *  top of the screen), and pops up child menus when they are clicked.
 *  When it executes one of the menu callback routines, it passes the
 *  return value back to the dialog manager, so these can return D_O_K,
 *  D_CLOSE, D_REDRAW, etc.
 */
int d_degui_menu_proc(int msg, DIALOG *d, int c)
{ 
   MENU_INFO m;
   int ret = D_O_K;
   int x;

   switch (msg) {

      case MSG_START:
	 _fill_menu_info(&m, d->dp, NULL, TRUE, d->x-1, d->y-1);
	 d->w = m.w-2;
	 d->h = m.h-2;
	 break;

      case MSG_DRAW:
	 _fill_menu_info(&m, d->dp, NULL, TRUE, d->x-1, d->y-1);
	 _draw_menu(&m);
	 break;

      case MSG_XCHAR:
	 x = _menu_alt_key(c, d->dp);
	 if (!x)
	    break;

	 ret |= D_USED_CHAR;
	 simulate_keypress(x);
	 /* fall through */

      case MSG_GOTMOUSE:
      case MSG_CLICK:
	 _do_degui_menu(d->dp, NULL, TRUE, d->x-1, d->y-1, FALSE, &x);
	 ret |= x;
	 do {
	 } while (mouse_b);
	 break;
   }

   return ret;
}

//----------------------------------------------------------------------------

