/*
 * Copyright (C) Keith Whitwell 1996
 */
#include <Lib3d/Lib3d.H>
#include <Lib3d/Devices/XDevice.H>
#include <Lib3d/Devices/FileDevice.H>
#include <iostream.h>
#include <fstream.h>

extern "C" {
#include "interface.h"
}

ModelBuilder mb;
World *world;
Model *model;
Camera *camera;
Viewport *viewport;
XDevice *device;
Matrix34 transform;
Matrix34 cameraTransform;
Scene *scene;

FileDevice *fileDevice;
Viewport *fileViewport;
int currentFileDeviceNr = -1;

Pipeline *pipeline[4];
Pipeline *defaultPipeline;

int 
initialize( Widget w )
{
    device = XDevice::create(XtDisplay(w), XtWindow(w));
    viewport = Viewport::create(device);

    if (!viewport) return 0;

    device->setSyncBehaviour(true);
    world = new World;

    Vector3 colour(1,1,1);   
    Vector3 direction(0,1,.5);

    camera = new Camera(*world);
    camera->setParameters(1, 30, 15, 1);
    world->setActiveCamera( *camera );

    Light *light = new Light(*world);
    light->setParameters(colour, colour, direction);
    world->registerLight( *light );

    Vector3 colour2(1,0,0.5);
    Vector3 direction2(1,0,0.5);
    Light *light2 = new Light(*world);
    light2->setParameters(colour2, colour2, direction2);
    //world->registerLight( *light2 );

    cameraTransform.setIdentity();

    defaultPipeline = pipeline[0] = new FlatPipeline;
    pipeline[1] = new SmoothPipeline;
    pipeline[2] = new TexturePipeline;
    pipeline[3] = new WirePipeline;

    scene = new Scene(*device, *viewport, *world, *camera);
}

const char **
getFileTypeArray( int *nr )
{
    return FileDevice::getFileTypeArray( *nr );
}

int
renderToFile( int deviceNr, const char *filename )
{
    if (deviceNr != currentFileDeviceNr) {
	delete fileViewport;	// Deletes device.

	fileDevice = FileDevice::create(deviceNr, 640, 400, 8);
	fileViewport = Viewport::create(fileDevice);
	if (!fileViewport) return 0;
	currentFileDeviceNr = deviceNr;
    } 
    
    // The demo only supports single frame/file.  This restriction is
    // not present in (at least) the postscript device.

    if (!fileDevice->openFile(filename)) return 0;    

    world->renderHierarchy( *fileViewport );
    fileViewport->swapBuffers();
    fileDevice->closeFile();
    return 1;
}

void
shutdown()
{
    delete world;		// Deletes all children. This will change
    delete viewport;		// Deletes device. This will change
}

int
load_model( const char *filename, int type )
{
    ifstream in(filename);
    if (in.bad()) return 0;

/*
    if (model) {
	world.release(model);
	delete model;
    }
*/
    int i, j;

    Colour white = viewport->getColour(255,255,255);
    Colour blue = viewport->getColour(0,0,255);

    Colour *buf = new Colour[128*128];
    
    for (i = 0 ; i < 128 ; i++) {
	for (j = 0 ; j < 128 ; j++) {
	    if (((i/16) ^ (j/16)) & 1) {
		buf[i + j*128] = blue;
	    } else {
		buf[i + j*128] = white;
	    }
	}
    }


    Texture *tex = new Texture(buf, 7);	// 128=2^7


    mb.startModel();
    mb.setModelTexture(tex);
    //mb.setColourfulMode();
    switch(type) {
    case FiletypeNff:
	mb.readNFF(in);
	break;

    case FiletypeGeom:		// inactive.
	mb.readGeom(in);
	break;
    }

    model = mb.endModel();

    if (model->isBad()) {
	delete model;
	model = 0;
	return 0;
    }

    transform.setIdentity();
    model->setTransform(transform);
    world->adopt(model);

    model->usePipeline(*defaultPipeline);

    return 1;
}



void
render()
{
    scene->render();
}

void 
notifyExpose( int xmin, int ymin, int xmax, int ymax )
{
    scene->notifyExpose(xmin, ymin, xmax, ymax);
}

void
notifyResize()
{
    scene->notifyResize();
}

void
arcBall( int dx, int dy )
{
    if (!model) return;

    Matrix34 incr;
    incr.setArcBallIncrement(100.0, dx, dy);
    transform.premul(incr);
    model->setTransform(transform);
}

void
moveCamera( int dx, int dy, int dz )
{
    if (!model) return;

    Matrix34 incr;
    incr.setTranslation( dx/100.0, dy/100.0, dz/100.0 );
    cameraTransform.premul(incr);
    camera->setTransform(cameraTransform);
}

void
setPipeline( int nr )
{
    if (model) model->usePipeline( *pipeline[nr] );
    defaultPipeline = pipeline[nr];
}









