/*
 * Copyright (c) 2020, Oracle and/or its affiliates.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, version 2.0, as
 * published by the Free Software Foundation.
 *
 * This program is also distributed with certain software (including
 * but not limited to OpenSSL) that is licensed under separate terms,
 * as designated in a particular file or component or in included license
 * documentation.  The authors of MySQL hereby grant you an
 * additional permission to link the program and your derivative works
 * with the separately licensed software that they have included with
 * MySQL.
 *
 * Without limiting anything contained in the foregoing, this file,
 * which is part of MySQL Connector/Node.js, is also subject to the
 * Universal FOSS Exception, version 1.0, a copy of which can be found at
 * http://oss.oracle.com/licenses/universal-foss-exception.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License, version 2.0, for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301  USA
 */

'use strict';

const InsertStub = require('../../../Stubs/mysqlx_crud_pb').Insert;
const collection = require('./Collection');
const column = require('./Column');
const list = require('../../Traits/List');
const polyglot = require('../../Traits/Polyglot');
const scalar = require('../Datatypes/Scalar');
const serializable = require('../../Traits/Serializable');
const typedRow = require('./TypedRow');
const wraps = require('../../Traits/Wraps');

/**
 * @private
 * @alias module:adapters.Mysqlx.Crud.Insert
 * @param {proto.Mysqlx.Crud.Insert} proto - protobuf stub
 * @returns {module:adapters.Mysqlx.Crud.Insert}
 */
function Insert (proto) {
    return Object.assign({}, polyglot(proto), serializable(proto), wraps(proto), {
        /**
         * Serialize to JSON using a protobuf-like convention.
         * @function
         * @name module:adapters.Mysqlx.Crud.Insert#toJSON
         * @returns {Object} The JSON representation
         */
        toJSON () {
            return {
                collection: collection(proto.getCollection()).toJSON(),
                data_model: this.getDataModel(),
                projection: list(proto.getProjectionList().map(c => column(c))).toJSON(),
                row: list(proto.getRowList().map(row => typedRow(row))).toJSON(),
                args: list(proto.getArgsList().map(arg => scalar(arg))).toJSON(),
                upsert: proto.getUpsert()
            };
        }
    });
}

/**
 * Creates a wrapper of a generic Mysqlx.Crud.Insert instance for a given statement.
 * @param {module:CollectionAdd|module:TableInsert} statement
 * @returns {module:adapters.Mysqlx.Crud.Insert}
 */
Insert.create = function (statement) {
    const proto = new InsertStub();

    proto.setCollection(collection.create(statement.getTableName(), statement.getSchema()).valueOf());
    proto.setDataModel(statement.getCategory());
    proto.setProjectionList(statement.getColumns().map(c => column.create(c).valueOf()));
    proto.setRowList(statement.getItems().map(fields => typedRow.create(fields).valueOf()));
    proto.setUpsert(statement.isUpsert());

    return Insert(proto);
};

module.exports = Insert;
