/* Copyright (C) 2000, 2001  SWsoft, Singapore                                  
 *                                                                              
 *  This program is free software; you can redistribute it and/or modify        
 *  it under the terms of the GNU General Public License as published by        
 *  the Free Software Foundation; either version 2 of the License, or           
 *  (at your option) any later version.                                         
 *                                                                              
 *  This program is distributed in the hope that it will be useful,             
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of              
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               
 *  GNU General Public License for more details.                                
 *                                                                              
 *  You should have received a copy of the GNU General Public License           
 *  along with this program; if not, write to the Free Software                 
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA   
 */

#pragma once

#define nuint32   unsigned long int

// ODBC definitions
#include "sqltypes.h"
#include "sql.h"
#include "sqlext.h"

#define	 MAXSERVERNAME		_MAX_PATH	//30
#define  MAXUSERNAME	30 // Pervasive definition
#define  MAXPASSWORD 30
#define	 DBNAMMNG_MAXNAME	     20 /* max length of name of Named Database */
#define  MAXDDFPATH			(_MAX_PATH + _MAX_FNAME)
#define  MAXDATASOURCENAME	250 // should be not less than DBNAMMNG_MAXNAME
#define  MYSQL_MAXDATASOURCENAME	1024 // ODBC connection string. Should be more than MAXDATASOURCENAME
#define  MAXDESCRIPTION		250	// should be less than 255
#define  MAXOWNERNAME		9	//  
#define  BACK				-1
#define	 ODBC_KEY			"Software\\ODBC\\ODBC.INI"
#define	 ODBC_SUBKEY		ODBC_KEY##"\\%s"


// Enumerates data source types
enum DATASOURCE_TYPE
{
	// Special values
	DATASOURCE_NONE = 0,			// Not selected
	DATASOURCE_SAME,					// No changes
	
	// File system - based types
	DATASOURCE_DDF,					// Data dictionary
	DATASOURCE_FILEDSN,				// File Data Source Name
	
	// Register / db - based types
	DATASOURCE_NAMEDDB,				// Named  db
	DATASOURCE_USERDSN,				// User Data Source Name
	DATASOURCE_SYSTEMDSN,			// System Data Source Name

	DATASOURCE_MYSQL

};

// Buffer for Select Data Source Name dialog
struct DSNINFO
{
   START_CLASS();
public:
   DATASOURCE_TYPE m_type;	   // Type of enumerator
   char m_szServerName[ MAXSERVERNAME+1 ];  // Server name. Defauls (empty) string means local server
   char m_szUser[ MAXUSERNAME+1 ];  // User ID
   char m_szPassword[ MAXPASSWORD+1 ];  // Password
   char m_szDataSource[ MAXDATASOURCENAME+1 ];  // Data source name
   char m_szDDFPath[ MAXDDFPATH+1 ]; // DDF Path
   char m_szDataPath[ MAXDDFPATH+1 ]; // DDF Path
   char m_szDescription[ MAXDESCRIPTION+1 ]; // DDF Path
   char *m_pszMySqlStr; // MySQL ODBC connection string
   BYTE  m_iDriverID; // ID of ODBC driver for P.SQL 2000
   DSNINFO() { CLEAR_CONSTRUCT(DSNINFO); }
   FINISH_CLASS();
};

struct SelectDataSourceTypeTag
{
	char * m_lpszInitDataSource;
	BOOL m_bValid;
	SelectDataSourceTypeTag( char * lpsz, BOOL b ) { m_lpszInitDataSource = lpsz, m_bValid = b; }
};

// Select Data source type
DATASOURCE_TYPE SelectDataSourceType( HWND hWnd, char* lpszInitDataSource, BOOL bValid );

// Select database by path
BOOL SelectDDF( HWND hParent, LPSTR lpszDDFPath, LPSTR lpszDataPath );

// Select file DSN
BOOL SelectFileDSN( HWND hParent, LPSTR lpszFileDsnPath, LPSTR lpszDDFPath, LPSTR lpszDataPath, BYTE* piDriverType );

// Select Data Source Name
BOOL SelectDataSourceName( HWND hWnd, char* lpsDataSource, char* lpsDDFPath, char* lpsDataPath, DATASOURCE_TYPE type );

// Check name. Get path by name of any type. Get type, too
BOOL GetPathByAnyDatabaseName_Slow( char* lpszDataSource, char* lpszDDFPath, char* lpszDataPath, DATASOURCE_TYPE* type, BYTE* piDriverType );

// cached version of above
BOOL GetPathByAnyDatabaseName( char* lpszDataSource, char* lpszDDFPath, char* lpszDataPath, DATASOURCE_TYPE* type, BYTE* piDriverType );

// Base pure virtual class for database enumerators
class EnumDatabase
{
public:	
	// Enumerational part of functions
	virtual HRESULT Init( DSNINFO* pDSNINFO ) = 0;
	virtual HRESULT Done() = 0;
	virtual HRESULT Reset() = 0;
	virtual HRESULT Next( DSNINFO* pDSNINFO ) = 0;
	virtual HRESULT Skip() = 0;
	virtual HRESULT Count(int* count) = 0;
	virtual HRESULT PathByName( DSNINFO* pDSNINFO ) = 0;
	virtual HRESULT GetType( DATASOURCE_TYPE* type ) = 0;
};


// Enumerator for Named Databases
class EnumNamedDatabases : public EnumDatabase
{
	START_CLASS(); 

	union{ // internal connection number
		nuint32	m_ulConnectionNumber;	
		long	m_lConnectionNumber;
	};

	int		m_iDatabase;			// # of db in list

public:
	EnumNamedDatabases();

	HRESULT Init( DSNINFO* pDSNINFO );
	HRESULT Done();
	HRESULT Reset();
	HRESULT PathByName( DSNINFO* pDSNINFO );
	HRESULT Next( DSNINFO* pDSNINFO );
	HRESULT Skip();
	HRESULT Count(int* count );
	HRESULT GetType(DATASOURCE_TYPE* type) 
	{ if(!type) return E_INVALIDARG; *type = DATASOURCE_NAMEDDB; return S_OK; }

	FINISH_CLASS(); 
};


// Enumerator for Pervasive Mashine DSNs
class EnumMashineDSNs : public EnumDatabase
{
	START_CLASS(); 

	HKEY	m_hBaseKey;		// Base key for User DSN or System DSN
	HKEY	m_hKey;					// Key handle
	HKEY	m_hRemoteKey;			// Key handle opened on another computer
	DWORD	m_nSubKey;				// # of subkey

public:
	EnumMashineDSNs( HKEY hBaseKey) 
	{
		CLEAR_CONSTRUCT( EnumMashineDSNs );
		
		m_hBaseKey = hBaseKey;
		m_nSubKey = -1;
	}

	HRESULT Init( DSNINFO* pDSNINFO );
	HRESULT Done();
	HRESULT Reset();
	HRESULT Next( DSNINFO* pDSNINFO );
	HRESULT PathByName( DSNINFO* pDSNINFO );
	HRESULT Skip();
	HRESULT Count(int* count );

	FINISH_CLASS(); 
};

// Enumerator for Pervasive User DSNs
class EnumUserDSNs : public EnumMashineDSNs
{
public:
	EnumUserDSNs() : EnumMashineDSNs( HKEY_CURRENT_USER ) {}
	HRESULT Init( DSNINFO* pDSNINFO );
	
	HRESULT GetType(DATASOURCE_TYPE* type) 
	{ if(!type) return E_INVALIDARG; *type = DATASOURCE_USERDSN; return S_OK; }
};


// Enumerator for Pervasive System DSNs
class EnumSystemDSNs : public EnumMashineDSNs
{
public:
	EnumSystemDSNs() : EnumMashineDSNs( HKEY_LOCAL_MACHINE ) {}

	HRESULT GetType(DATASOURCE_TYPE* type) 
	{ if(!type) return E_INVALIDARG; *type = DATASOURCE_SYSTEMDSN; return S_OK; }
};

// Enumerator for DDFs (not supported)
class EnumDDFs : public EnumDatabase
{
public:
	HRESULT Init( DSNINFO* pDSNINFO ) { return S_OK; }
	HRESULT Done() { return S_OK; }
	HRESULT Reset() { return S_OK; }
	HRESULT Next( DSNINFO* pDSNINFO ) { return E_FAIL; }
	HRESULT PathByName( DSNINFO* pDSNINFO );
	HRESULT Skip() { return S_OK; }
	HRESULT Count(int* count ) { if( !count ) return E_INVALIDARG; *count = 0; return S_OK; }

	HRESULT GetType(DATASOURCE_TYPE* type) 
	{ if( !type ) return E_INVALIDARG; *type = DATASOURCE_DDF; return S_OK; }
};

// Enumerator for File DSNs (not supported)
class EnumFileDSNs : public EnumDatabase
{
public:
	HRESULT Init( DSNINFO* pDSNINFO ) { return S_OK; }
	HRESULT Done() { return S_OK; }
	HRESULT Reset() { return S_OK; }
	HRESULT Next( DSNINFO* pDSNINFO ) { return E_FAIL; }
	HRESULT PathByName( DSNINFO* pDSNINFO );
	HRESULT Skip() { return S_OK; }
	HRESULT Count(int* count ) { if( !count ) return E_INVALIDARG; *count = 0; return S_OK; }

	HRESULT GetType(DATASOURCE_TYPE* type) 
	{ if( !type ) return E_INVALIDARG; *type = DATASOURCE_DDF; return S_OK; }
};

// Base pure virtual class for database enumerators
class EnumOfEnumsDataSources: public EnumDatabase
{
	START_CLASS();
	DATASOURCE_TYPE m_nCurrent;
public:	
	EnumOfEnumsDataSources() 
	{
		CLEAR_CONSTRUCT( EnumOfEnumsDataSources );
		m_nCurrent = DATASOURCE_NONE;
	}
	HRESULT Init( DSNINFO* pDSNINFO ) { return S_OK; }
	HRESULT Done() { return S_OK; }
	HRESULT Reset()  { m_nCurrent = DATASOURCE_NONE; return S_OK; }
	HRESULT Next( DSNINFO* pDSNINFO );
	HRESULT Skip() { Next( NULL ); return S_OK;}
	HRESULT Count(int* count);
	HRESULT PathByName( DSNINFO* pDSNINFO ) { return E_FAIL; }
	HRESULT GetType( DATASOURCE_TYPE* type ) 
	{ if( type == NULL ) return E_INVALIDARG; *type = DATASOURCE_NONE; return S_OK; }
	
	static HRESULT GetNewObject( DATASOURCE_TYPE type, EnumDatabase** ppObj );

	FINISH_CLASS();
};

// Enumerator for MySQL databases
class EnumMySQLs : public EnumDatabase
{
	START_CLASS();
	SQLHENV	 m_henv; 
	SQLHDBC	 m_hdbc;
	SQLHSTMT m_hstmt;

public:
	EnumMySQLs() { CLEAR_CONSTRUCT(EnumMySQLs); }

	HRESULT Init( DSNINFO* pDSNINFO );
	HRESULT Done();
	HRESULT Reset();
	HRESULT Next( DSNINFO* pDSNINFO );
	HRESULT PathByName( DSNINFO* pDSNINFO );
	HRESULT Skip();
	HRESULT Count(int* count );

	HRESULT GetType(DATASOURCE_TYPE* type) 
	{ if( !type ) return E_INVALIDARG; *type = DATASOURCE_MYSQL; return S_OK; }

	FINISH_CLASS();
};

