/* Copyright (C) 2000, 2001  SWsoft, Singapore                                  
 *                                                                              
 *  This program is free software; you can redistribute it and/or modify        
 *  it under the terms of the GNU General Public License as published by        
 *  the Free Software Foundation; either version 2 of the License, or           
 *  (at your option) any later version.                                         
 *                                                                              
 *  This program is distributed in the hope that it will be useful,             
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of              
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               
 *  GNU General Public License for more details.                                
 *                                                                              
 *  You should have received a copy of the GNU General Public License           
 *  along with this program; if not, write to the Free Software                 
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA   
 */

// This file contains schema implementation


#include "hfiles.h"
#include "headers.h"
#include "Schema.h"
#include "mysql.h"

WCHAR INFORMATION_SCHEMA[] = L"INFORMATION_SCHEMA";

int CompareIndexNames( SWSTINDEX* pIndexLeft, SWSTINDEX* pIndexRight )
{
   	// Left name
	LPCSTR lpszLeft = pIndexLeft->m_pSwstRelateName ? pIndexLeft->m_pSwstRelateName->m_szName : 
			(pIndexLeft->m_pSwstFieldName ? pIndexLeft->m_pSwstFieldName->m_szName : NULL);
	
   	// Right name
	LPCSTR lpszRight = pIndexRight->m_pSwstRelateName ? pIndexRight->m_pSwstRelateName ->m_szName: 
			(pIndexRight->m_pSwstFieldName ? pIndexRight->m_pSwstFieldName->m_szName : NULL);
	
    // Number ?  Number
    if( !lpszLeft && !lpszRight )
	    return pIndexLeft->m_wNumber - pIndexRight->m_wNumber;
    // String < Number
    else if( lpszLeft && !lpszRight )
	    return -1;
    // Number > String
    else if( !lpszLeft && lpszRight )
	    return 1;
    // String ? String
    else 
	    return stricmp( lpszLeft, lpszRight );
}

// Wide output for index name
int WidePrint( SWSTINDEX* pIndex, void* pOut ) 
{
	LPWSTR pStr = (LPWSTR) pOut;
	
	if( pIndex->m_pSwstRelateName != NULL )
	{
		A2W( pIndex->m_pSwstRelateName->m_szName, pStr, MAXSTR(pIndex->m_pSwstRelateName->m_szName) );
		wcscat( pStr, L" (#" );
		_itow( pIndex->m_wNumber, pStr + wcslen( pStr ), 10 );
		wcscat( pStr, L")" );
	}
	else if( pIndex->m_pSwstFieldName != NULL )
	{
		A2W( pIndex->m_pSwstFieldName->m_szName, pStr, MAXSTR(pIndex->m_pSwstFieldName->m_szName) );
		wcscat( pStr, L" (#" );
		_itow( pIndex->m_wNumber, pStr + wcslen( pStr ), 10 );
		wcscat( pStr, L")" );
	}
	else
	{
		*pStr = L'#';
		_itow( pIndex->m_wNumber, pStr + 1, 10 );
		if( pIndex->IsPrimaryKey() )
			wcscat( pStr, L" (Primary key)" );
	}

	// Return real column size
	return wcslen( pStr ) * sizeof(WCHAR);
}

// Filters index names
bool PassFilter( SWSTINDEX* pIndex, LPCSTR lpszIndexNameRestr, int iIndexNameRestr ) 
{
	// It may be string or number representation
	bool bFieldName = pIndex->m_pSwstFieldName != NULL &&
		!stricmp(pIndex->m_pSwstFieldName->m_szName, lpszIndexNameRestr);
	bool bRelateName = pIndex->m_pSwstRelateName != NULL &&
		!stricmp(pIndex->m_pSwstRelateName->m_szName, lpszIndexNameRestr);
	bool bPrimaryKey = pIndex->IsPrimaryKey() &&
		!stricmp(lpszIndexNameRestr, "PRIMARY KEY" );
	bool bNumberName = iIndexNameRestr == pIndex->m_wNumber;

	return bFieldName || bRelateName || bNumberName || bPrimaryKey;
}


// CImpIDBSchemaRowset::GetRowset ------------------------------------------------
//
// @mfunc Returns a schema rowset
//
// @rdesc HRESULT
//      @flag S_OK                  | The method succeeded.
//      @flag E_INVALIDARG          | ppCommand was NULL
//      @flag E_NOINTERFACE         | Could not obtain requested interface on DBSession object
//      @flag E_OUTOFMEMORY         | Out of memory
//      @flag E_FAIL                | Could not initialize rowset
//      @flag DB_E_NOAGGREGATION    | pUnkOuter was not NULL (this object does not support
//                                    being aggregated)
//      @flag DB_E_OBJECTOPEN       | The provider would have to open a new connection to 
//									  support the operation and DBPROP_MULTIPLECONNECTIONS is set to VARIANT_FALSE
//		@flag DB_S_ERRORSOCCURED	| Rowset is opened but at least one property was not set
//		@flag DB_E_ERRORSOCCURED	| No rowset is opened because at least one property was not set
//		@flag DB_E_NOTSUPPORTED		| Restrictions are not supported
//
STDMETHODIMP CImpIDBSchemaRowset::GetRowset
    (
    IUnknown*		pUnkOuter,			//@parm IN		| Controlling IUnknown if being aggregated 
    REFGUID			rguidSchema,		//@parm IN		| The GUID of the schema rowset
	ULONG			cRestrictions,		//@parm IN		| Count of restriction values
	const VARIANT	rgRestrictions[],	//@parm IN		| Array of restriction values
	REFIID			riid,				//@parm IN		| IID of requested rowset interface
	ULONG			cProperties,		//@parm IN		| The number of properties
	DBPROPSET		rgProperties[],		//@parm IN/OUT	| Properties
    IUnknown**		ppRowset			//@parm OUT		| A pointer to memory in which to return the interface pointer
    )
{
	INTERFACE_METHOD_START("IDBSchemaRowset::GetRowset");

	HRESULT hr;
	HRESULT hrProp = S_OK;
	
    // check in-params and NULL out-params in case of error
    if (ppRowset == NULL || cProperties < 0)
        return E_INVALIDARG;

	*ppRowset = NULL;

    assert(m_pObj);

	CRowset *pRowset;
	ULONG ulRowset;

	// set the properties
	if (cProperties)
		hr = m_pObj->m_pUtilProp->SetProperties(PROPSET_ROWSET, cProperties, rgProperties);
	
	if( (hr == DB_E_ERRORSOCCURRED) || 
		(hr == DB_S_ERRORSOCCURRED) )
	{
		// If all the properties set were SETIFCHEAP then we can set 
		// our status to DB_S_ERRORSOCCURRED and continue.
		for(ULONG ul = 0; ul < cProperties; ul++)
		{
			for(ULONG ul2 = 0; ul2 < rgProperties[ul].cProperties; ul2++)
			{
				// Check for a required property that failed, if found, we must return
				// DB_E_ERRORSOCCURRED
				if( (rgProperties[ul].rgProperties[ul2].dwStatus != DBPROPSTATUS_OK) &&
					(rgProperties[ul].rgProperties[ul2].dwOptions != DBPROPOPTIONS_SETIFCHEAP) )
					return DB_E_ERRORSOCCURRED;
			}
		}
		hrProp = DB_S_ERRORSOCCURRED;
	}

	if (FAILED(hr) && hrProp == S_OK)
	{
		TRACE( "IDBSchemaRowset::GetRowset, failed: properties" );
		return hr;
	}


	//Get meta data
	CSwstMetaHolder *pSwstMetaHolder = NULL;
	hr = m_pObj->m_pCDataSource->GetSwstMeta(&pSwstMetaHolder);
    if (FAILED(hr))
	{
        TRACE( "IDBSchemaRowset::GetRowset, failed GetSwstMeta" );
		return hr;
	}
	
	// open and initialize a file object
    CSchema *pData = new CSchema( m_pObj );
    if (!pData)
		return E_OUTOFMEMORY;

    TRACE( "IDBSchemaRowset::GetRowset, CSchema - before init" );

	hr = pData->Init(rguidSchema, pSwstMetaHolder, cRestrictions, rgRestrictions );
    if (hr != S_OK)
    {
        delete pData;
		TRACE( "IDBSchemaRowset::GetRowset, failed pData->Init" );
		return hr;
    }

	TRACE( "IDBSchemaRowset::GetRowset, CSchema - inited" );

	// open and initialize a rowset\cursor object
	// Create Rowset:
	// Get number of first inactive rowset
	hr = m_pObj->GetFirstInactiveRowset(&ulRowset);
	if (hr == S_FALSE)
	{
        delete pData;
		TRACE( "IDBSchemaRowset::GetRowset, failed GetForstInactiveRowset" );
		return DB_E_OBJECTCREATIONLIMITREACHED;	
	}

	// shortcut for m_pObj->m_pRowsets[ulRowset]
	pRowset = m_pObj->m_pRowsets[ulRowset];
	// some cautions
	assert(ulRowset < NUM_SUPPORTED_ROWSETS_PER_SESSION);
	assert(pRowset == NULL); //it's clear yet

	// allocate memory for new rowset
    pRowset = new CRowset(pUnkOuter);
    if (!pRowset)
	{
        delete  pData;
		return E_OUTOFMEMORY;
	}

	// mark rowset # ulRowset as active
	hr = m_pObj->SetActiveRowset(ulRowset);
	if (hr == S_FALSE)
	{
        delete  pData;
        delete  pRowset;
		TRACE( "IDBSchemaRowset::GetRowset, failed SetActiveRowsets" );
		return 	E_FAIL;	
	}

	// refresh the number of active sessions
	hr = m_pObj->CountActiveRowsets();
	if (hr != S_OK)
	{
        delete  pData;
		delete  pRowset;
		TRACE( "IDBSchemaRowset::GetRowset, failed CountActiveRowsets" );
		return 	E_FAIL;	
	}

	// Initialize new rowset
    if (!pRowset->Init(ulRowset, pData, ROWSET_SCROLL))
	{
        delete  pRowset;
		TRACE( "IDBSchemaRowset::GetRowset, failed IRowset->Init" );
		return  E_FAIL;
	}

 
	//At this point we have handed off the pData pointer to the
	//provider so null it out.
	pData = NULL;

    // get requested interface pointer on rowset\cursor
    hr = pRowset->QueryInterface(riid, (void**)ppRowset);
    if (FAILED(hr))
    {
        delete pRowset;
		TRACE2( "IDBSchemaRowset::GetRowset, QI - hr=%d", hr );
		return hr;
    }

	//Assign creator pointer. Used for IRowsetInfo::GetSpecification
    pRowset->m_pCreator = m_pObj;     
	pRowset->m_pCreator->AddRef();

    //Assign session pointer
	pRowset->m_pSession = m_pObj;     
	pRowset->m_pSession->AddRef();
    m_pObj->m_fRowsetCreated = TRUE;

	return (hr == S_OK) ? hrProp : hr;

	INTERFACE_METHOD_END();
}

// CImpIDBSchemaRowset::GetSchemas ------------------------------------------------
//
// @mfunc Returns a list of schema rowsets accessible by CImpIDBSchemaRowset::GetRowset
//
// @rdesc HRESULT
//      @flag S_OK                  | The method succeeded.
//      @flag E_INVALIDARG          | at least one of parameters was NULL
//      @flag E_OUTOFMEMORY         | Out of memory
//      @flag E_FAIL                | Could not initialize rowset
//
STDMETHODIMP CImpIDBSchemaRowset::GetSchemas
    (
	ULONG*	pcSchemas,				//@parm OUT	| Number of returned schemas
	GUID**	prgSchemas,				//@parm OUT | Array of supported schema GUIDs
	ULONG**	prgRestrictionsSupport	//@parm OUT	| Array of supported restrictions
    )
{
	INTERFACE_METHOD_START("IDBSchemaRowset::GetSchemas");

    // check in-params and NULL out-params in case of error
    if (prgSchemas == NULL || pcSchemas == NULL)
        return E_INVALIDARG;
	
	assert(m_pObj);

	//Provider supports minimum + DBSHEMA_CATALOGS + DBSCHEMA_SCHEMATA
	*pcSchemas = 7+6;
	
	//Allocate memory for schema array
	*prgSchemas = (GUID*)g_pIMalloc->Alloc(*pcSchemas * sizeof(GUID));
	if (*prgSchemas == NULL)
		return E_OUTOFMEMORY;

	//Fill out guid array
	memcpy(*prgSchemas, &DBSCHEMA_COLUMNS, sizeof(GUID));
	memcpy(*prgSchemas + 1, &DBSCHEMA_PROVIDER_TYPES, sizeof(GUID));
	memcpy(*prgSchemas + 2, &DBSCHEMA_TABLES, sizeof(GUID));
	memcpy(*prgSchemas + 3, &DBSCHEMA_CATALOGS, sizeof(GUID));
	memcpy(*prgSchemas + 4, &DBSCHEMA_SCHEMATA, sizeof(GUID));
	memcpy(*prgSchemas + 5, &DBSCHEMA_STATISTICS, sizeof(GUID));
	memcpy(*prgSchemas + 6, &DBSCHEMA_INDEXES, sizeof(GUID));
	memcpy(*prgSchemas + 7, &DBSCHEMA_CONSTRAINT_COLUMN_USAGE, sizeof(GUID));
	memcpy(*prgSchemas + 8, &DBSCHEMA_CONSTRAINT_TABLE_USAGE, sizeof(GUID));
	memcpy(*prgSchemas + 9, &DBSCHEMA_TABLE_CONSTRAINTS, sizeof(GUID));
	memcpy(*prgSchemas + 10, &DBSCHEMA_REFERENTIAL_CONSTRAINTS, sizeof(GUID));
	memcpy(*prgSchemas + 11, &DBSCHEMA_PRIMARY_KEYS, sizeof(GUID));
	memcpy(*prgSchemas + 12, &DBSCHEMA_FOREIGN_KEYS, sizeof(GUID));

	if (prgRestrictionsSupport != NULL)
	{
		//Allocate memory for resctrictions array
		*prgRestrictionsSupport = (ULONG*)g_pIMalloc->Alloc(*pcSchemas * sizeof(ULONG));
		if (*prgRestrictionsSupport == NULL)
		{
			g_pIMalloc->Free(*prgSchemas);
			return E_OUTOFMEMORY;
		}

		//We support TABLE_CATALOG_NAME & TABLE_SCHEMA & TABLE_NAME & COLUMN_NAME restrictions for COLUMNS		
		(*prgRestrictionsSupport)[0] = 1+2+4+8;
		//We support DATA_TYPE & BEST_MATCH restrictions for PROVIDER_TYPES
		(*prgRestrictionsSupport)[1] = 1+2;
		//We support TABLE_CATALOG & TABLE_SCHEMA & TABLE_NAME & TABLE_TYPE restrictions for TABLES
		(*prgRestrictionsSupport)[2] = 1+2+4+8;
		//We support CATALOG_NAME restrictions for CATALOGS
		(*prgRestrictionsSupport)[3] = 1;
		//We support CATALOG_NAME & SCHEMA_NAME & SCHEMA_OWNER restrictions for SCHEMATA
		(*prgRestrictionsSupport)[4] = 1+2+4;
		//We support TABLE_CATALOG & TABLE_SCHEMA & TABLE_NAME restrictions for STATICSTICS
		(*prgRestrictionsSupport)[5] = 1+2+4;
		//We support TABLE_CATALOG & TABLE_SCHEMA & INDEX_NAME & TYPE & TABLE_NAME restrictions for INDEXES
		(*prgRestrictionsSupport)[6] = 1+2+4+8+16;
		//We support all 7 restrictions for DBSCHEMA_CONSTRAINT_COLUMN_USAGE
		(*prgRestrictionsSupport)[7] = 1+2+4+8+16+32+64;
		//We support all 6 restrictions for DBSCHEMA_CONSTRAINT_TABLE_USAGE
		(*prgRestrictionsSupport)[8] = 1+2+4+8+16+32;
		//We support all 7 restrictions for DBSCHEMA_TABLE_CONSTRAINTS
		(*prgRestrictionsSupport)[9] = 1+2+4+8+16+32+64;
		//We support all 3 restrictions for DBSCHEMA_REFERENTIAL_CONSTRAINTS
		(*prgRestrictionsSupport)[10] = 1+2+4;
		//We support all 3 restrictions for DBSCHEMA_PRIMARY_KEYS
		(*prgRestrictionsSupport)[11] = 1+2+4;
		//We support all 6 restrictions for DBSCHEMA_FOREIGN_KEYS
		(*prgRestrictionsSupport)[12] = 1+2+4+8+16+32;
	}
	
	return S_OK;

	INTERFACE_METHOD_END();
}


//Skip relative
HRESULT CSchema::GetFetchedData(LONG lRow, DWORD* pdwBmk)
{
	// Check if array was used
	if( m_plRowPositions==NULL )
		return DB_S_ENDOFROWSET;
	
	// Sign does no matter
	lRow = abs(lRow);

	// Get appropriate row position value from buffer
	HRESULT hr = m_plRowPositions[ lRow ].m_hr;
	m_lCurrentRow = m_plRowPositions[ lRow ].m_lRowPos;
	if( pdwBmk != NULL )
		*pdwBmk = Pos2Bmk( m_lCurrentRow );

	return hr;
}


// Invalidate buffers
HRESULT CSchema::SetMoveInfo( HRESULT hr )
{
	delete m_plRowPositions;
	m_plRowPositions = NULL;
	return hr;
}

//@cmember Move relative
HRESULT CSchema::MovePending(LONG nRows, BOOL bSaveCurrentPosition)
{
	// Allocate new buffer for row information
	delete m_plRowPositions;
	m_plRowPositions = new CRowInfo[ abs(nRows) + 1 ];
	if( m_plRowPositions == NULL )
		return E_OUTOFMEMORY;

	HRESULT hr = S_OK;
	LONG lIdx = 0;

	// Save first row position if asked
	if( bSaveCurrentPosition )
	{
		m_plRowPositions[ lIdx ].m_lRowPos = m_lCurrentRow;
		m_plRowPositions[ lIdx ].m_hr = ( m_lCurrentRow >= 0 && m_lCurrentRow < m_cRows ) ? S_OK : DB_S_ENDOFROWSET;
		lIdx++;
	}

	// Save other row positions
	for( LONG i = 1; i <= abs(nRows); i++, lIdx++ )
	{
		LONG lCurrentRow = m_lCurrentRow + sgn(nRows) * i;
		
		if( m_lCurrentRow >= m_cRows || lCurrentRow >= m_cRows )
		{
			m_plRowPositions[ lIdx ].m_lRowPos = m_cRows;
			hr = m_plRowPositions[ lIdx ].m_hr = DB_S_ENDOFROWSET; //EOF
		}
		else if( m_lCurrentRow < 0 || lCurrentRow < 0)
		{
			m_plRowPositions[ lIdx ].m_lRowPos = -1; 
			hr = m_plRowPositions[ lIdx ].m_hr = DB_S_ENDOFROWSET; //BOF
		}
		else
		{
			m_plRowPositions[ lIdx ].m_lRowPos = lCurrentRow; 
			m_plRowPositions[ lIdx ].m_hr = S_OK;
		}
	}

	return hr;
}


//@cmember Move to first row
HRESULT CSchema::MoveFirst()
{
	TRACE( "CSchema::MoveFirst" );
	m_lCurrentRow = 0;
	return S_OK;
}

//@cmember Move to last row
HRESULT CSchema::MoveLast()
{
	TRACE( "CSchema::MoveLast" );
	m_lCurrentRow = m_cRows - 1;
	return S_OK;
}

//@cmember Move to specified bookmark
HRESULT CSchema::MoveBookmark(ULONG* pulBookmark)
{
	TRACE( "CSchema::MoveBookmark" );
	ULONG ulPos = Bmk2Pos(*pulBookmark);
	if (ulPos >= m_cRows || ulPos < 0)
		return DB_E_BADBOOKMARK;

	m_lCurrentRow = ulPos;
	return S_OK;
}


//@cmember Find row from current position
HRESULT CSchema::Find(ULONG icol, PCOLUMNDATA pDst/*void* pvValue*/,	DBCOMPAREOP	CompareOp, DBTYPE dbType, bool bForward)
{
	TRACE( "CSchema::Find" );
	return E_FAIL;
}


//@cmember Return the number of rows in the table	
HRESULT CSchema::GetRowCnt(DWORD* pdwRows)
{
	TRACE( "CSchema::GetRowCnt" );
	*pdwRows = m_cRows;
	return S_OK;
}

//@cmember Get relative position (by percentage) of the row in  table
HRESULT CSchema::GetPercentage(ULONG* pulBookmark, double *pdfPPos)
{
	TRACE( "CSchema::GetPercentage" );
	ULONG ulPos = Bmk2Pos(*pulBookmark);

	if (ulPos >= m_cRows || ulPos < 0)
		return E_INVALIDARG;

	*pdfPPos = double(ulPos + 1) / double(m_cRows);
	return S_OK;
}

//@cmember Get bookmark for current row
HRESULT CSchema::GetBookmark(ULONG *pulBookmark)
{
	TRACE( "CSchema::GetBookmark" );
	
	*pulBookmark = Pos2Bmk(m_lCurrentRow);
	return S_OK;
}

//@cmember Retrieve number of columns
HRESULT CSchema::GetColumnCnt(DWORD* pdwCount)
{
	TRACE( "CSchema::GetColumnCnt" );

	*pdwCount = m_cCols;
	return S_OK;
}

//@cmember Retrieve column information
HRESULT CSchema::GetColumnInfo(DWORD dwCol, DBCOLUMNINFO* pInfo)
{
	TRACE( "CSchema::GetColumnInfo" );
	
	if (dwCol < 1 || dwCol > m_cCols)
		return E_INVALIDARG;
	
	COLINFO *pCol = m_pColumns + dwCol - 1;
	
	//Copy column info	
	pInfo->iOrdinal		= dwCol; 
    pInfo->pTypeInfo	= NULL; 
    pInfo->dwFlags		= pCol->dwFlags; 
    pInfo->ulColumnSize = pCol->ulSize; 
    pInfo->wType		= pCol->wType; 
    pInfo->bPrecision	= 0; 
    pInfo->bScale		= 0; 
    pInfo->columnid.eKind          = DBKIND_PROPID;
    pInfo->columnid.uGuid.guid     = GUID_NULL;
    pInfo->columnid.uName.ulPropid = pInfo->iOrdinal;

	// Copy name in special way
	pInfo->pwszName = new WCHAR[MAX_COLUMN_NAME_CHARS];
	wcscpy0(pInfo->pwszName, pCol->wszName, MAX_COLUMN_NAME_CHARS - 1);

	return S_OK;
}

	
//@cmember Fetch row data
HRESULT CSchema::GetRow(ULONG* pulOffset, BYTE* pbProvRow)
{
	INTERFACE_METHOD_START( "CSchema::GetRow" );

	switch (m_enumSchema)	
	{
	case TABLES_SCHEMA:
		return GetTablesRow(pulOffset, pbProvRow);

	case COLUMNS_SCHEMA:
		return GetColumnsRow(pulOffset, pbProvRow);
		
	case TYPES_SCHEMA:
		return GetTypesRow(pulOffset, pbProvRow);

	case CATALOGS_SCHEMA:
		return GetCatalogsRow(pulOffset, pbProvRow);

	case SCHEMATA_SCHEMA:
		return GetSchemataRow(pulOffset, pbProvRow);

	case STATISTICS_SCHEMA:
		return GetStatisticsRow(pulOffset, pbProvRow);

	case INDEXES_SCHEMA:
		return GetIndexesRow(pulOffset, pbProvRow);

	case CONSTRAINT_COLUMN_USAGE_SCHEMA:
		return GetConstraintColumnsRow(pulOffset, pbProvRow);

	case CONSTRAINT_TABLE_USAGE_SCHEMA:
		return GetConstraintTablesRow(pulOffset, pbProvRow);

	case TABLE_CONSTRAINTS_SCHEMA:
		return GetTableConstraintsRow(pulOffset, pbProvRow);

	case REFERENTIAL_CONSTRAINTS_SCHEMA:
		return GetReferentialConstraintsRow(pulOffset, pbProvRow);

	case PRIMARY_KEYS_SCHEMA:
		return GetPrimaryKeysRow(pulOffset, pbProvRow);

	case FOREIGN_KEYS_SCHEMA:
		return GetForeignKeysRow(pulOffset, pbProvRow);
	}
	
	return E_FAIL;

	INTERFACE_METHOD_END();
}


//Retrieve current row content for TABLES schema
HRESULT CSchema::GetTablesRow(ULONG* pulOffset, BYTE* pbProvRow)
{
	SWSTFILE *pFile = m_ppFiles[m_lCurrentRow];
	for (int i = 1; i <= m_cCols; i++)
	{
		COLUMNDATA *pCol = (COLUMNDATA*)((BYTE*)pbProvRow + pulOffset[i]);
		pCol->dwStatus = DBSTATUS_S_OK;

		switch (i)
		{
		case 1:	// TABLE_CATALOG
			wcscpy( (LPOLESTR)pCol->bData, pFile->m_pCatalogOwner->m_wszCatalog );
			pCol->dwLength = wcslen( pFile->m_pCatalogOwner->m_wszCatalog ) * sizeof(WCHAR);
			break;

		case 2:	// TABLE_SHEMA
			if( *pFile->m_pCatalogOwner->m_wszOwner )
			{
				wcscpy( (LPOLESTR)pCol->bData, pFile->m_pCatalogOwner->m_wszOwner );
				pCol->dwLength = wcslen( pFile->m_pCatalogOwner->m_wszOwner ) * sizeof(WCHAR);
			}
			else
			{
				wcscpy( (LPOLESTR)pCol->bData, INFORMATION_SCHEMA );
				pCol->dwLength = sizeof( INFORMATION_SCHEMA ) - sizeof( WCHAR );
			}
			break;
		 
		case 3:	//TABLE_NAME
			pCol->dwLength = sizeof(WCHAR) * 
					MultiByteToWideChar(CP_ACP, MB_PRECOMPOSED,
										pFile->m_szName,
										-1,
										(LPOLESTR)pCol->bData, 
										m_pColumns[i - 1].ulSize / sizeof(WCHAR));
			break;
		
		case 4:	//TABLE_TYPE
			if (0 == strnicmp(pFile->m_szName, "X$", 2))
			{
				wcscpy((LPOLESTR)pCol->bData, L"SYSTEM TABLE");
				pCol->dwLength = sizeof(L"SYSTEM TABLE") - sizeof( WCHAR );
			}
			else
			{
				wcscpy((LPOLESTR)pCol->bData, L"TABLE");
				pCol->dwLength = sizeof(L"TABLE") - sizeof( WCHAR );
			}

			break;

		default://All other columns are NULL
			pCol->dwStatus = DBSTATUS_S_ISNULL;
			pCol->dwLength = 0; 
		}
	}

	TRACE4("CSchema::GetTablesRow: %S %S %s", pFile->m_pCatalogOwner->m_wszCatalog, pFile->m_pCatalogOwner->m_wszOwner, pFile->m_szName );
    return S_OK;	
}


//Retrieve current row content for COLUMNS schema
HRESULT CSchema::GetColumnsRow(ULONG* pulOffset, BYTE* pbProvRow)
{
	SWSTFIELD*	pField		= m_pFieldInfos[m_lCurrentRow].m_pField;
	SWSTFILE*	pFile		= m_pFieldInfos[m_lCurrentRow].m_pFile;
	ULONG		ulPosition	= m_pFieldInfos[m_lCurrentRow].m_ulPosition;

	WORD	wOledbType;			//OLEDB Type name
	DWORD	dwOledbFlags;		//OLEDB Type flags
	DWORD	dwOledbLength;		//OLEDB Type length	
	WORD	wOledbPrecision;	//OLEDB Type precision
	WORD	wOledbScale;		//OLEDB Type scale

	HRESULT hr;
	//if( m_pCDataSource->m_pUtilProp->MySqlMode() )
	if( m_pCDataSource->m_pUtilProp->InternalSqlSupport() == ISS_MYSQL )
		hr = CSwstMeta::SwstType2OledbTypeHelper
						(
						pField->m_btDataType, 
						pField->m_wSize,
						pField->m_btDec,
						pField->m_bNullable,
						TRUE,
						&wOledbType,
						&dwOledbFlags,
						&dwOledbLength,
						&wOledbPrecision,
						&wOledbScale
						);
	
	if (hr != S_OK)
		return hr;

	for (int i = 1; i <= m_cCols; i++)
	{
		COLUMNDATA *pCol = (COLUMNDATA*)((BYTE*)pbProvRow + pulOffset[i]);
		pCol->dwLength = m_pColumns[i - 1].ulSize;
		pCol->dwStatus = DBSTATUS_S_OK;

		switch (i)
		{
		case 1:	// TABLE_CATALOG
			wcscpy( (LPOLESTR)pCol->bData, pFile->m_pCatalogOwner->m_wszCatalog );
			pCol->dwLength = wcslen( pFile->m_pCatalogOwner->m_wszCatalog ) * sizeof(WCHAR);
			break;

		case 2:	// TABLE_SHEMA
			if( *pFile->m_pCatalogOwner->m_wszOwner )
			{
				wcscpy( (LPOLESTR)pCol->bData, pFile->m_pCatalogOwner->m_wszOwner );
				pCol->dwLength = wcslen( pFile->m_pCatalogOwner->m_wszOwner ) * sizeof(WCHAR);
			}
			else
			{
				wcscpy( (LPOLESTR)pCol->bData, INFORMATION_SCHEMA );
				pCol->dwLength = sizeof( INFORMATION_SCHEMA ) - sizeof( WCHAR );
			}
			break;

		case 3:	//TABLE_NAME
			pCol->dwLength = sizeof(WCHAR) * 
						MultiByteToWideChar(CP_ACP, MB_PRECOMPOSED,
											pFile->m_szName,
											-1, //sizeof(pFile->m_szName),
											(LPOLESTR)pCol->bData, 
											m_pColumns[i - 1].ulSize / sizeof(WCHAR));
			break;

		case 4:	//COLUMN_NAME
			pCol->dwLength = sizeof(WCHAR) * 
						MultiByteToWideChar(CP_ACP, MB_PRECOMPOSED,
											pField->m_szName,
											-1, //sizeof(pField->m_szName),
											(LPOLESTR)pCol->bData, 
											m_pColumns[i - 1].ulSize / sizeof(WCHAR));
			break;
		case 7: //ORDINAL_POSITION
			*(ULONG*)pCol->bData = ulPosition + 1;
			break;

		case 8: //COLUMN_HAS_DEFAULT
			*(VARIANT_BOOL*)pCol->bData = VARIANT_FALSE;
			break;

		case 10: //COLUMNS_FLAGS
			*(ULONG*)pCol->bData = dwOledbFlags;
			break;

		case 11: //IS_NULLABLE
			*(VARIANT_BOOL*)pCol->bData = (dwOledbFlags & DBCOLUMNFLAGS_ISNULLABLE) ? VARIANT_TRUE : VARIANT_FALSE;
			break;

		case 12: //DATA_TYPE
			*(WORD*)pCol->bData = wOledbType;
			break;

		case 14: //CHARACTER_MAXIMUM_LENGTH
		case 15: //CHARACTER_OCTET_LENGTH
			if (wOledbType == DBTYPE_STR)
				*(WORD*)pCol->bData = dwOledbLength;
			else
				pCol->dwStatus = DBSTATUS_S_ISNULL;
			break;

		default://All other columns are NULL
			pCol->dwStatus = DBSTATUS_S_ISNULL;
		}
	}


	TRACE5("CSchema::GetColumnsRow: %S %S %s %s", pFile->m_pCatalogOwner->m_wszCatalog, pFile->m_pCatalogOwner->m_wszOwner, pFile->m_szName, pField->m_szName );
	
	return S_OK;
}

//Retrieve current row content for PROVIDER_TYPES schema
HRESULT CSchema::GetTypesRow(ULONG* pulOffset, BYTE* pbProvRow)
{
	TRACE( "CSchema::GetTypesRow" );

	assert(m_ppTypes != NULL);
	TYPEINFO* pType = m_ppTypes[m_lCurrentRow];
	for (int i = 1; i <= m_cCols; i++)
	{
		COLUMNDATA *pCol = (COLUMNDATA*)((BYTE*)pbProvRow + pulOffset[i]);
		pCol->dwLength = m_pColumns[i - 1].ulSize;
		pCol->dwStatus = DBSTATUS_S_OK;

		switch (i)
		{
		case 1:	 //TYPE_NAME
		case 13: //LOCAL_TYPE_NAME
			wcscpy((LPOLESTR)pCol->bData, pType->wszTypeName);
			pCol->dwLength = sizeof(WCHAR) * (1 + wcslen(pType->wszTypeName));
			break;

		case 2: //DATA_TYPE
			*(WORD*)pCol->bData = pType->wDataType;
			break;

		case 3: //COLUMN_SIZE
			*(WORD*)pCol->bData = pType->ulColumnSize;
			break;
		
		case 4:	//LITERAL_PREFIX
			wcscpy((LPOLESTR)pCol->bData, pType->wszLiteralPrefix);
			pCol->dwLength = sizeof(WCHAR) * (1 + wcslen(pType->wszLiteralPrefix));
			break;

		case 5:	//LITERAL_SUFFIX
			wcscpy((LPOLESTR)pCol->bData, pType->wszLiteralSuffix);
			pCol->dwLength = sizeof(WCHAR) * (1 + wcslen(pType->wszLiteralSuffix));
			break;

		case 6:	//CREATE_PARAMS
			wcscpy((LPOLESTR)pCol->bData, pType->wszCreateParams);
			pCol->dwLength = sizeof(WCHAR) * (1 + wcslen(pType->wszCreateParams));
			break;
		
		case 7: //IS_NULLABLE
			*(VARIANT_BOOL*)pCol->bData = 
				//( m_pCDataSource->m_pUtilProp->MySqlMode() || IsNullable( pType->wDataType ) ) 
				( m_pCDataSource->m_pUtilProp->InternalSqlSupport() == ISS_MYSQL || IsNullable( pType->wDataType ) ) 
					? VARIANT_TRUE 
					: VARIANT_FALSE;
			break;

		case 8: //CASE_SENSITIVE
			if (pType->wDataType == DBTYPE_STR)
				*(VARIANT_BOOL*)pCol->bData = VARIANT_TRUE;
			else
				*(VARIANT_BOOL*)pCol->bData = VARIANT_FALSE;
			break;

		case 9: //SEARCHABLE
			*(ULONG*)pCol->bData = DB_SEARCHABLE;
			break;
		
		case 10: //UNSIGNED_ATTRIBUTE
			if (pType->bUnsignedAttribute == VARIANT_TRUE || pType->bUnsignedAttribute == VARIANT_FALSE)
				*(VARIANT_BOOL*)pCol->bData = pType->bUnsignedAttribute;
			else
				pCol->dwStatus = DBSTATUS_S_ISNULL;
			break;

		case 11: //FIXED_PREC_SCALE
			*(VARIANT_BOOL*)pCol->bData = pType->bFixedPrecScale;
			break;

		case 12: //AUTO_UNIQUE_VALUE
			*(VARIANT_BOOL*)pCol->bData = pType->bAutoUniqueValue;
			break;

		case 18: //VERSION
			swprintf( (LPOLESTR)pCol->bData, L"0%.2f.0000", pType->fltVersion);
			break;

		case 19: //IS_LONG
			*(VARIANT_BOOL*)pCol->bData = pType->bIsLong/*VARIANT_FALSE*/;
			break;

		case 20: //BEST_MATCH
			*(VARIANT_BOOL*)pCol->bData = pType->bBestMatch;
			break;

		case 21: //IS_FIXEDLENGTH
			*(VARIANT_BOOL*)pCol->bData = pType->bFixedLength;
			break;

		default://All other columns are NULL
			pCol->dwStatus = DBSTATUS_S_ISNULL;
		}
	}

	return S_OK;
}


//Retrieve current row content for CATALOGS schema
HRESULT CSchema::GetCatalogsRow(ULONG* pulOffset, BYTE* pbProvRow)
{
	CATALOGINFO *pCatInfo = m_pCatalogInfos + m_lCurrentRow;
	
	for (int i = 1; i <= m_cCols; i++)
	{
		COLUMNDATA *pCol = (COLUMNDATA*)((BYTE*)pbProvRow + pulOffset[i]);
		pCol->dwStatus = DBSTATUS_S_OK;

		switch (i)
		{
		case 1:	//CATALOG_NAME
			wcscpy( (LPOLESTR)pCol->bData, pCatInfo->m_wszCatalog );
			pCol->dwLength = sizeof( WCHAR ) * wcslen( pCatInfo->m_wszCatalog );
			break;
		
		case 2:	//DESCRIPTION
			wcscpy( (LPOLESTR)pCol->bData, pCatInfo->m_wszDescription );
			pCol->dwLength = sizeof( WCHAR ) * wcslen( pCatInfo->m_wszDescription );
			break;

		default://All other columns are NULL
			pCol->dwStatus = DBSTATUS_S_ISNULL;
			pCol->dwLength = 0; 
		}
	}

	TRACE2("CSchema::GetCatalogsRow: %S", pCatInfo->m_wszCatalog );

    return S_OK;	
}


//Retrieve current row content for SCHEMATA schema
HRESULT CSchema::GetSchemataRow(ULONG* pulOffset, BYTE* pbProvRow)
{
	TRACE("CSchema::GetSchemataRow" );

	CatalogOwner *pSchemataInfo = m_pSchemataInfos[ m_lCurrentRow ];
	
	for (int i = 1; i <= m_cCols; i++)
	{
		COLUMNDATA *pCol = (COLUMNDATA*)((BYTE*)pbProvRow + pulOffset[i]);
		pCol->dwStatus = DBSTATUS_S_OK;

		switch (i)
		{
		case 1:	//CATALOG_NAME
			wcscpy( (LPOLESTR)pCol->bData, pSchemataInfo->m_wszCatalog );
			pCol->dwLength = sizeof(WCHAR) * wcslen( pSchemataInfo->m_wszCatalog );
			break;
		
		case 2:	//SCHEMA_NAME
		case 3:	//SCHEMA_OWNER
			if( *pSchemataInfo->m_wszOwner )
			{
				wcscpy( (LPOLESTR)pCol->bData, pSchemataInfo->m_wszOwner );
				pCol->dwLength = sizeof(WCHAR) * wcslen( pSchemataInfo->m_wszOwner );
			}
			else
			{
				wcscpy( (LPOLESTR)pCol->bData, INFORMATION_SCHEMA );
				pCol->dwLength = sizeof( INFORMATION_SCHEMA ) - sizeof( WCHAR );
			}

			break;

		default://All other columns are NULL
			pCol->dwStatus = DBSTATUS_S_ISNULL;
			pCol->dwLength = 0; 
		}
	}

    return S_OK;	
}


//Retrieve current row content for STATISTICS schema
HRESULT CSchema::GetStatisticsRow(ULONG* pulOffset, BYTE* pbProvRow)
{
	STATINFO *pStat = m_pStatistics + m_lCurrentRow;
	SWSTFILE *pFile = pStat->pFile; 
	for (int i = 1; i <= m_cCols; i++)
	{
		COLUMNDATA *pCol = (COLUMNDATA*)((BYTE*)pbProvRow + pulOffset[i]);
		pCol->dwStatus = DBSTATUS_S_OK;

		switch (i)
		{
		case 1:	// TABLE_CATALOG
			wcscpy( (LPOLESTR)pCol->bData, pFile->m_pCatalogOwner->m_wszCatalog );
			pCol->dwLength = wcslen( pFile->m_pCatalogOwner->m_wszCatalog ) * sizeof(WCHAR);
			break;

		case 2:	// TABLE_SHEMA
			if( *pFile->m_pCatalogOwner->m_wszOwner )
			{
				wcscpy( (LPOLESTR)pCol->bData, pFile->m_pCatalogOwner->m_wszOwner );
				pCol->dwLength = wcslen( pFile->m_pCatalogOwner->m_wszOwner ) * sizeof(WCHAR);
			}
			else
			{
				wcscpy( (LPOLESTR)pCol->bData, INFORMATION_SCHEMA );
				pCol->dwLength = sizeof( INFORMATION_SCHEMA ) - sizeof( WCHAR );
			}
			break;
		 
		case 3:	//TABLE_NAME
			pCol->dwLength = sizeof(WCHAR) * 
					MultiByteToWideChar(CP_ACP, MB_PRECOMPOSED,
										pFile->m_szName,
										-1,
										(LPOLESTR)pCol->bData, 
										m_pColumns[i - 1].ulSize / sizeof(WCHAR));
			break;
		
		case 4:	//CARDINALITY
			if( pStat->dwRows != -1 )
			{
				pCol->dwLength = sizeof(__int64);
				*(__int64*)pCol->bData = __int64( pStat->dwRows);
			}
			else
			{
				pCol->dwStatus = DBSTATUS_S_ISNULL;
				pCol->dwLength = 0; 
			}
			break;

		default://All other columns are NULL
			pCol->dwStatus = DBSTATUS_S_ISNULL;
			pCol->dwLength = 0; 
		}
	}

	TRACE("CSchema::GetStatisticsRow" );
    return S_OK;	
}


//Retrieve current row content for INDEXES schema
HRESULT CSchema::GetIndexesRow(ULONG* pulOffset, BYTE* pbProvRow)
{
	SWSTINDEX *pIndex = m_ppIndexes[m_lCurrentRow];
	SWSTFILE  *pFile = pIndex->m_pSwstField->m_pSwstFile;
	for (int i = 1; i <= m_cCols; i++)
	{
		COLUMNDATA *pCol = (COLUMNDATA*)((BYTE*)pbProvRow + pulOffset[i]);
		pCol->dwStatus = DBSTATUS_S_OK;

		switch (i)
		{
		case 1:	// TABLE_CATALOG
		case 4:	// INDEX_CATALOG
			wcscpy( (LPOLESTR)pCol->bData, pFile->m_pCatalogOwner->m_wszCatalog );
			pCol->dwLength = wcslen( pFile->m_pCatalogOwner->m_wszCatalog ) * sizeof(WCHAR);
			break;

		case 2:	// TABLE_SHEMA
		case 5:	// INDEX_SHEMA
			if( *pFile->m_pCatalogOwner->m_wszOwner )
			{
				wcscpy( (LPOLESTR)pCol->bData, pFile->m_pCatalogOwner->m_wszOwner );
				pCol->dwLength = wcslen( pFile->m_pCatalogOwner->m_wszOwner ) * sizeof(WCHAR);
			}
			else
			{
				wcscpy( (LPOLESTR)pCol->bData, INFORMATION_SCHEMA );
				pCol->dwLength = sizeof( INFORMATION_SCHEMA ) - sizeof( WCHAR );
			}
			break;
		 
		case 3:	//TABLE_NAME
			pCol->dwLength = sizeof(WCHAR) * 
					MultiByteToWideChar(CP_ACP, MB_PRECOMPOSED,
										pFile->m_szName,
										-1,
										(LPOLESTR)pCol->bData, 
										m_pColumns[i - 1].ulSize / sizeof(WCHAR));
			break;
		
		case 6:	// INDEX_NAME
			pCol->dwLength = WidePrint( pIndex, pCol->bData );
			break;

		case 7: // PRIMARY_KEYS
			pCol->dwLength = sizeof( VARIANT_BOOL );
			if( pIndex->IsPrimaryKey() )
				*(VARIANT_BOOL*)pCol->bData = VARIANT_TRUE;
			else
				*(VARIANT_BOOL*)pCol->bData = VARIANT_FALSE;
			break;
		
		case 8: // UNIQUE
			pCol->dwLength = sizeof( VARIANT_BOOL );
			if( pIndex->IsUnique() )
				*(VARIANT_BOOL*)pCol->bData = VARIANT_TRUE;
			else
				*(VARIANT_BOOL*)pCol->bData = VARIANT_FALSE;
			break;

		case 9: // CLUSTERED
			pCol->dwLength = sizeof( VARIANT_BOOL );
			*(VARIANT_BOOL*)pCol->bData = VARIANT_FALSE;
			break;

		case 10: // TYPE
			pCol->dwLength = sizeof(WORD); //2;
			*(WORD*)pCol->bData = DBPROPVAL_IT_BTREE;
			break;
		
		case 11:// FILL_FACTOR
			//if( m_pCDataSource->m_pUtilProp->MySqlMode() )
			if( m_pCDataSource->m_pUtilProp->InternalSqlSupport() == ISS_MYSQL )
			{
				pCol->dwStatus = DBSTATUS_S_ISNULL;
				pCol->dwLength = 0; 
			}
			
			break;

		case 14:// SORT_BOOKMARKS
			pCol->dwLength = sizeof(VARIANT_BOOL); 
			*(VARIANT_BOOL*)pCol->bData = VARIANT_TRUE;
			break;
		
		case 15:// AUTO_UPDATE
			pCol->dwLength = sizeof(VARIANT_BOOL); 
			*(VARIANT_BOOL*)pCol->bData = VARIANT_TRUE;
			break;

		case 17:// ORDINAL_POSITION
			pCol->dwLength = sizeof(ULONG); 
			*(ULONG*)pCol->bData = pIndex->m_wPart + 1;
			break;

		case 18:// COLUMN_NAME
			pCol->dwLength = A2W( pIndex->m_pSwstField->m_szName, (LPOLESTR)pCol->bData, 
				m_pColumns[i - 1].ulSize / sizeof(WCHAR) - 1 ) * sizeof(WCHAR);
			break;

		case 21:// COLLATION
			pCol->dwLength = sizeof(SHORT); // 2
			*(SHORT*)pCol->bData = pIndex->IsDescending() ? DB_COLLATION_DESC 
														 : DB_COLLATION_ASC;
			break;

		case 25: // INTEGRATED
			pCol->dwLength = sizeof( VARIANT_BOOL );
			*(VARIANT_BOOL*)pCol->bData = VARIANT_TRUE;
			break;

		default://All other columns are NULL
			pCol->dwStatus = DBSTATUS_S_ISNULL;
			pCol->dwLength = 0; 
		}
	}

	TRACE("CSchema::GetIndexesRow" );
    return S_OK;	
}


//Retrieve current row content for CONSTRAINT_COLUMN_USAGE schema
HRESULT CSchema::GetConstraintColumnsRow(ULONG* pulOffset, BYTE* pbProvRow)
{
	SWSTINDEX *pIndex = m_ppIndexes[ m_lCurrentRow ]; // Specially filled m_ppIndexes
	SWSTFILE  *pFile = pIndex->m_pSwstField->m_pSwstFile;
	for (int i = 1; i <= m_cCols; i++)
	{
		COLUMNDATA *pCol = (COLUMNDATA*)((BYTE*)pbProvRow + pulOffset[i]);
		pCol->dwStatus = DBSTATUS_S_OK;

		switch (i)
		{
		case 1:	// TABLE_CATALOG
		case 7:	// CONSTRAINT_CATALOG
			wcscpy( (LPOLESTR)pCol->bData, pFile->m_pCatalogOwner->m_wszCatalog );
			pCol->dwLength = wcslen( pFile->m_pCatalogOwner->m_wszCatalog ) * sizeof(WCHAR);
			break;

		case 2:	// TABLE_SCHEMA
		case 8:	// CONSTRAINT_SCHEMA
			if( *pFile->m_pCatalogOwner->m_wszOwner )
			{
				wcscpy( (LPOLESTR)pCol->bData, pFile->m_pCatalogOwner->m_wszOwner );
				pCol->dwLength = wcslen( pFile->m_pCatalogOwner->m_wszOwner ) * sizeof(WCHAR);
			}
			else
			{
				wcscpy( (LPOLESTR)pCol->bData, INFORMATION_SCHEMA );
				pCol->dwLength = sizeof( INFORMATION_SCHEMA ) - sizeof( WCHAR );
			}
			break;
		 
		case 3:	//TABLE_NAME
			pCol->dwLength = sizeof(WCHAR) * 
					MultiByteToWideChar(CP_ACP, MB_PRECOMPOSED,
										pFile->m_szName,
										-1,
										(LPOLESTR)pCol->bData, 
										m_pColumns[i - 1].ulSize / sizeof(WCHAR));
			break;
		
		case 4:	// COLUMN_NAME
			pCol->dwLength = A2W( pIndex->m_pSwstField->m_szName, (LPOLESTR)pCol->bData, 
				m_pColumns[i - 1].ulSize / sizeof(WCHAR) - 1 ) * sizeof(WCHAR);
			break;

		case 9:	// CONSTRAINT_NAME
			pCol->dwLength = WidePrint( pIndex, pCol->bData );
			break;

		default://All other columns are NULL
			pCol->dwStatus = DBSTATUS_S_ISNULL;
			pCol->dwLength = 0; 
		}
	}

	TRACE("CSchema::GetConstraintColumnsRow" );
    return S_OK;	
}


//Retrieve current row content for CONSTRAINT_TABLE_USAGE schema
HRESULT CSchema::GetConstraintTablesRow(ULONG* pulOffset, BYTE* pbProvRow)
{
	SWSTINDEX *pIndex = m_ppIndexes[ m_lCurrentRow ]; // Specially filled m_ppIndexes
	SWSTFILE  *pFile = pIndex->m_pSwstField->m_pSwstFile;
	for (int i = 1; i <= m_cCols; i++)
	{
		COLUMNDATA *pCol = (COLUMNDATA*)((BYTE*)pbProvRow + pulOffset[i]);
		pCol->dwStatus = DBSTATUS_S_OK;

		switch (i)
		{
		case 1:	// TABLE_CATALOG
		case 4:	// CONSTRAINT_CATALOG
			wcscpy( (LPOLESTR)pCol->bData, pFile->m_pCatalogOwner->m_wszCatalog );
			pCol->dwLength = wcslen( pFile->m_pCatalogOwner->m_wszCatalog ) * sizeof(WCHAR);
			break;

		case 2:	// TABLE_SCHEMA
		case 5:	// CONSTRAINT_SCHEMA
			if( *pFile->m_pCatalogOwner->m_wszOwner )
			{
				wcscpy( (LPOLESTR)pCol->bData, pFile->m_pCatalogOwner->m_wszOwner );
				pCol->dwLength = wcslen( pFile->m_pCatalogOwner->m_wszOwner ) * sizeof(WCHAR);
			}
			else
			{
				wcscpy( (LPOLESTR)pCol->bData, INFORMATION_SCHEMA );
				pCol->dwLength = sizeof( INFORMATION_SCHEMA ) - sizeof( WCHAR );
			}
			break;
		 
		case 3:	//TABLE_NAME
			pCol->dwLength = sizeof(WCHAR) * 
					MultiByteToWideChar(CP_ACP, MB_PRECOMPOSED,
										pFile->m_szName,
										-1,
										(LPOLESTR)pCol->bData, 
										m_pColumns[i - 1].ulSize / sizeof(WCHAR));
			break;
		
		case 6:	// CONSTRAINT_NAME
			pCol->dwLength = WidePrint( pIndex, pCol->bData );
			break;

		default://All other columns are NULL
			pCol->dwStatus = DBSTATUS_S_ISNULL;
			pCol->dwLength = 0; 
		}
	}

	TRACE("CSchema::GetConstraintTablesRow" );
    return S_OK;	
}


//Retrieve current row content for TABLE_CONSTRAINTS schema
HRESULT CSchema::GetTableConstraintsRow(ULONG* pulOffset, BYTE* pbProvRow)
{
	SWSTINDEX *pIndex = m_ppIndexes[ m_lCurrentRow ]; // Specially filled m_ppIndexes
	SWSTFILE  *pFile = pIndex->m_pSwstField->m_pSwstFile;
	for (int i = 1; i <= m_cCols; i++)
	{
		COLUMNDATA *pCol = (COLUMNDATA*)((BYTE*)pbProvRow + pulOffset[i]);
		pCol->dwStatus = DBSTATUS_S_OK;

		switch (i)
		{
		case 1:	// CONSTRAINT_CATALOG
		case 4:	// TABLE_CATALOG
			wcscpy( (LPOLESTR)pCol->bData, pFile->m_pCatalogOwner->m_wszCatalog );
			pCol->dwLength = wcslen( pFile->m_pCatalogOwner->m_wszCatalog ) * sizeof(WCHAR);
			break;

		case 2:	// CONSTRAINT_SCHEMA
		case 5:	// TABLE_SCHEMA
			if( *pFile->m_pCatalogOwner->m_wszOwner )
			{
				wcscpy( (LPOLESTR)pCol->bData, pFile->m_pCatalogOwner->m_wszOwner );
				pCol->dwLength = wcslen( pFile->m_pCatalogOwner->m_wszOwner ) * sizeof(WCHAR);
			}
			else
			{
				wcscpy( (LPOLESTR)pCol->bData, INFORMATION_SCHEMA );
				pCol->dwLength = sizeof( INFORMATION_SCHEMA ) - sizeof( WCHAR );
			}
			break;
		 
		case 3:	// CONSTRAINT_NAME
			pCol->dwLength = WidePrint( pIndex, pCol->bData );
			break;

		case 6:	//TABLE_NAME
			pCol->dwLength = sizeof(WCHAR) * 
					MultiByteToWideChar(CP_ACP, MB_PRECOMPOSED,
										pFile->m_szName,
										-1,
										(LPOLESTR)pCol->bData, 
										m_pColumns[i - 1].ulSize / sizeof(WCHAR));
			break;
		
		case 7:	//CONSTRAINT_TYPE
			if( pIndex->IsForeignKey() )
				wcscpy( (LPOLESTR)pCol->bData, L"FOREIGN KEY" );
			else if( pIndex->IsPrimaryKey() )
				wcscpy( (LPOLESTR)pCol->bData, L"PRIMARY KEY" );
			else if( pIndex->IsUnique() )
				wcscpy( (LPOLESTR)pCol->bData, L"UNIQUE" );
			else
				*(LPOLESTR)pCol->bData = L'\0';

			pCol->dwLength = wcslen( (LPOLESTR)pCol->bData ) * sizeof(WCHAR);
			break;

		case 8:	//IS_DEFERRABLE
			pCol->dwLength = sizeof( VARIANT_BOOL );
			*(VARIANT_BOOL*)pCol->bData = VARIANT_FALSE;
			break;

		case 9:	//INITIALLY_DEFERRED
			pCol->dwLength = sizeof( VARIANT_BOOL );
			*(VARIANT_BOOL*)pCol->bData = VARIANT_FALSE;
			break;

		default://All other columns are NULL
			pCol->dwStatus = DBSTATUS_S_ISNULL;
			pCol->dwLength = 0; 
		}
	}

	TRACE("CSchema::GetTableConstraintsRow" );
    return S_OK;	
}


//Retrieve current row content for REFERENTIAL_CONSTRAINTS schema
HRESULT CSchema::GetReferentialConstraintsRow(ULONG* pulOffset, BYTE* pbProvRow)
{
	SWSTRELATE *pRelate = m_ppRelations[ m_lCurrentRow ]; 
	
	for (int i = 1; i <= m_cCols; i++)
	{
		SWSTINDEX *pIndex = pRelate->m_ppParentIndex[0];
		
		COLUMNDATA *pCol = (COLUMNDATA*)((BYTE*)pbProvRow + pulOffset[i]);
		pCol->dwStatus = DBSTATUS_S_OK;

		switch (i)
		{
		case 1:	// CONSTRAINT_CATALOG
			pIndex = pRelate->m_ppDependIndex[0];
		case 4:	// UNIQUE_CONSTRAINT_CATALOG
			wcscpy( (LPOLESTR)pCol->bData, pIndex->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszCatalog );
			pCol->dwLength = wcslen( (LPOLESTR)pCol->bData ) * sizeof(WCHAR);
			break;

		case 2:	// CONSTRAINT_SCHEMA
			pIndex = pRelate->m_ppDependIndex[0];
		case 5:	// UNIQUE_CONSTRAINT_SCHEMA
			if( *pIndex->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszOwner )
			{
				wcscpy( (LPOLESTR)pCol->bData, pIndex->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszOwner );
				pCol->dwLength = wcslen( (LPOLESTR)pCol->bData ) * sizeof(WCHAR);
			}
			else
			{
				wcscpy( (LPOLESTR)pCol->bData, INFORMATION_SCHEMA );
				pCol->dwLength = sizeof( INFORMATION_SCHEMA ) - sizeof( WCHAR );
			}
			break;
		 
		case 3:	// CONSTRAINT_NAME
			pIndex = pRelate->m_ppDependIndex[0];
		case 6:	// UNIQUE_CONSTRAINT_NAME
			pCol->dwLength = WidePrint( pIndex, pCol->bData );
			break;

		case 7:	//MATCH_OPTION
			wcscpy( (LPOLESTR)pCol->bData, L"FULL" );
			pCol->dwLength = wcslen( (LPOLESTR)pCol->bData ) * sizeof(WCHAR);
			break;

		case 8:	//UPDATE_RULE
		case 9:	//DELETE_RULE
			switch( (i==8) ? pRelate->m_btUpdateRule : pRelate->m_btDeleteRule )
			{
			case 1:
				wcscpy( (LPOLESTR)pCol->bData, L"RESTRICT" );
				break;
			case 2:
				wcscpy( (LPOLESTR)pCol->bData, L"CASCADE" );
				break;
			default:
				wcscpy( (LPOLESTR)pCol->bData, L"NO ACTION" );
			}
			pCol->dwLength = wcslen( (LPOLESTR)pCol->bData ) * sizeof(WCHAR);
			break;

		case 10:	// DESCRIPTION
			wcscpy( (LPOLESTR)pCol->bData, L"Parent table is " );
			A2W( pRelate->m_ppParentIndex[0]->m_pSwstField->m_pSwstFile->m_szName, 
				(LPOLESTR)pCol->bData + wcslen( (LPOLESTR)pCol->bData ),
				MAXSTR(pRelate->m_ppParentIndex[0]->m_pSwstField->m_pSwstFile->m_szName) );
			wcscat( (LPOLESTR)pCol->bData, L", dependent table is " );
			A2W( pRelate->m_ppDependIndex[0]->m_pSwstField->m_pSwstFile->m_szName, 
				(LPOLESTR)pCol->bData + wcslen( (LPOLESTR)pCol->bData ),
				MAXSTR(pRelate->m_ppDependIndex[0]->m_pSwstField->m_pSwstFile->m_szName) );
			pCol->dwLength = wcslen( (LPOLESTR)pCol->bData ) * sizeof(WCHAR);
			break;

		default://All other columns are NULL
			pCol->dwStatus = DBSTATUS_S_ISNULL;
			pCol->dwLength = 0; 
		}
	}

	TRACE("CSchema::GetReferentialConstraintsRow" );
    return S_OK;	
}


//Retrieve current row content for PRIMARY_KEYS schema
HRESULT CSchema::GetPrimaryKeysRow(ULONG* pulOffset, BYTE* pbProvRow)
{
	SWSTINDEX *pIndex = m_ppIndexes[ m_lCurrentRow ]; // Specially filled m_ppIndexes
	SWSTFILE  *pFile = pIndex->m_pSwstField->m_pSwstFile;
	
	for (int i = 1; i <= m_cCols; i++)
	{
		COLUMNDATA *pCol = (COLUMNDATA*)((BYTE*)pbProvRow + pulOffset[i]);
		pCol->dwStatus = DBSTATUS_S_OK;

		switch (i)
		{
		case 1:	// PK_NAME
			pCol->dwLength = WidePrint( pIndex, pCol->bData );
			break;

		case 2:	// TABLE_CATALOG
			wcscpy( (LPOLESTR)pCol->bData, pFile->m_pCatalogOwner->m_wszCatalog );
			pCol->dwLength = wcslen( pFile->m_pCatalogOwner->m_wszCatalog ) * sizeof(WCHAR);
			break;

		case 3:	// TABLE_SCHEMA
			if( *pFile->m_pCatalogOwner->m_wszOwner )
			{
				wcscpy( (LPOLESTR)pCol->bData, pFile->m_pCatalogOwner->m_wszOwner );
				pCol->dwLength = wcslen( pFile->m_pCatalogOwner->m_wszOwner ) * sizeof(WCHAR);
			}
			else
			{
				wcscpy( (LPOLESTR)pCol->bData, INFORMATION_SCHEMA );
				pCol->dwLength = sizeof( INFORMATION_SCHEMA ) - sizeof( WCHAR );
			}
			break;
		 
		case 4:	//TABLE_NAME
			pCol->dwLength = sizeof(WCHAR) * 
					MultiByteToWideChar(CP_ACP, MB_PRECOMPOSED,
										pFile->m_szName,
										-1,
										(LPOLESTR)pCol->bData, 
										m_pColumns[i - 1].ulSize / sizeof(WCHAR));
			break;
		
		case 5:	//COLUMN_NAME
			pCol->dwLength = sizeof(WCHAR) * A2W( pIndex->m_pSwstField->m_szName, 
				(LPOLESTR)pCol->bData, m_pColumns[i - 1].ulSize / sizeof(WCHAR) - 1 );
			break;

		case 8:	//ORDINAL
			pCol->dwLength = sizeof(ULONG);
			*(ULONG*)pCol->bData = pIndex->m_wPart + 1;
			break;

		default://All other columns are NULL
			pCol->dwStatus = DBSTATUS_S_ISNULL;
			pCol->dwLength = 0; 
		}
	}

	TRACE("CSchema::GetPrimaryKeysRow" );
    return S_OK;	
}

//Retrieve current row content for FOREIGN_KEYS schema
HRESULT CSchema::GetForeignKeysRow(ULONG* pulOffset, BYTE* pbProvRow)
{
	FOREIGNKEYINFO* pInfo = m_pForeignKeyInfo + m_lCurrentRow;
	SWSTINDEX* pParentIndex = pInfo->pRelate->m_ppParentIndex[ pInfo->iIdx ];
	SWSTINDEX* pDependIndex = pInfo->pRelate->m_ppDependIndex[ pInfo->iIdx ];
	
	for (int i = 1; i <= m_cCols; i++)
	{
		COLUMNDATA *pCol = (COLUMNDATA*)((BYTE*)pbProvRow + pulOffset[i]);
		pCol->dwStatus = DBSTATUS_S_OK;

		switch (i)
		{
		case 1:	// PK_NAME
			pCol->dwLength = WidePrint( pParentIndex, pCol->bData );
			break;

		case 2:	// PK_TABLE_CATALOG
			wcscpy( (LPOLESTR)pCol->bData, pParentIndex->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszCatalog );
			pCol->dwLength = wcslen( (LPOLESTR)pCol->bData ) * sizeof(WCHAR);
			break;

		case 3:	// PK_TABLE_SCHEMA
			wcscpy( (LPOLESTR)pCol->bData, pParentIndex->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszOwner );
			pCol->dwLength = wcslen( (LPOLESTR)pCol->bData ) * sizeof(WCHAR);
			break;

		case 4:	// PK_TABLE_NAME
			pCol->dwLength = sizeof(WCHAR) * A2W( pParentIndex->m_pSwstField->m_pSwstFile->m_szName,
				(LPOLESTR)pCol->bData, m_pColumns[i - 1].ulSize / sizeof(WCHAR) - 1 );
			break;

		case 5:	// PK_COLUMN_NAME
			pCol->dwLength = sizeof(WCHAR) * A2W( pParentIndex->m_pSwstField->m_szName,
				(LPOLESTR)pCol->bData, m_pColumns[i - 1].ulSize / sizeof(WCHAR) - 1 );
			break;
			
		case 8:	// FK_NAME
			pCol->dwLength = WidePrint( pDependIndex, pCol->bData );
			break;

		case 9:	// FK_TABLE_CATALOG
			wcscpy( (LPOLESTR)pCol->bData, pDependIndex->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszCatalog );
			pCol->dwLength = wcslen( (LPOLESTR)pCol->bData ) * sizeof(WCHAR);
			break;

		case 10:	// FK_TABLE_SCHEMA
			wcscpy( (LPOLESTR)pCol->bData, pDependIndex->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszOwner );
			pCol->dwLength = wcslen( (LPOLESTR)pCol->bData ) * sizeof(WCHAR);
			break;

		case 11:	// FK_TABLE_NAME
			pCol->dwLength = sizeof(WCHAR) * A2W( pDependIndex->m_pSwstField->m_pSwstFile->m_szName,
				(LPOLESTR)pCol->bData, m_pColumns[i - 1].ulSize / sizeof(WCHAR) - 1 );
			break;

		case 12:	// FK_COLUMN_NAME
			pCol->dwLength = sizeof(WCHAR) * A2W( pDependIndex->m_pSwstField->m_szName,
				(LPOLESTR)pCol->bData, m_pColumns[i - 1].ulSize / sizeof(WCHAR) - 1 );
			break;

		case 15:	// ORDINAL
			pCol->dwLength = sizeof( ULONG );
			*(ULONG*)pCol->bData = pParentIndex->m_wPart + 1; // Same as in pDependent
			break;

		case 16:	//UPDATE_RULE
		case 17:	//DELETE_RULE
			switch( (i==16) ? pInfo->pRelate->m_btUpdateRule : pInfo->pRelate->m_btDeleteRule )
			{
			case 1:
				wcscpy( (LPOLESTR)pCol->bData, L"RESTRICT" );
				break;
			case 2:
				wcscpy( (LPOLESTR)pCol->bData, L"CASCADE" );
				break;
			default:
				wcscpy( (LPOLESTR)pCol->bData, L"NO ACTION" );
			}
			pCol->dwLength = wcslen( (LPOLESTR)pCol->bData ) * sizeof(WCHAR);
			break;

		case 18: // DEFERRABILITY
			pCol->dwLength = sizeof( SHORT );
			*(SHORT*)pCol->bData = DBPROPVAL_DF_NOT_DEFERRABLE;
			break;

		default://All other columns are NULL
			pCol->dwStatus = DBSTATUS_S_ISNULL;
			pCol->dwLength = 0; 
		}
	}

	TRACE("CSchema::GetForeignKeysRow" );
    return S_OK;	
}


//@cmember Update the current rows values
HRESULT CSchema::UpdateRow(
	ULONG*		pulBookmark,		//@parm IN | Bookmark of row to update	
    ULONG*      ulOffset,		//@parm IN | Array of offsets for the columns
    BYTE*       pbProvRow,		//@parm IN | Data to update row with.
	PACCESSOR	pAccessor,		//@parm IN | Accessor with bindings to update.
	BYTE*		pbProvRowOld
						   )
{
	TRACE( "CSchema::UpdateRow" );
	return E_FAIL;
}

//@cmember Insert new row
HRESULT CSchema::InsertRow(ULONG* ulOffset, BYTE* pbProvRow, PACCESSOR pAccessor)
{
	TRACE( "CSchema::InsertRow" );
	return E_FAIL;
}

//@cmember Remove row with the bmk
HRESULT CSchema::DeleteRow(ULONG* pulBookmark)
{
	TRACE( "CSchema::DeleteRow" );
	return E_FAIL;
}



CSchema::CSchema( CDBSession* pSession )
{
	CLEAR_CONSTRUCT( CSchema );
	
	m_lCurrentRow	= -1;

	assert( pSession );
	assert( pSession->m_pCDataSource );
	m_pCDataSource = pSession->m_pCDataSource;
}

CSchema::~CSchema()
{
	delete [] m_ppTypes;
	delete [] m_ppFiles;
	delete [] m_pFieldInfos;
	delete [] m_pCatalogInfos;
	delete [] m_pSchemataInfos;
	delete [] m_plRowPositions;
	delete [] m_pStatistics;
	delete [] m_ppIndexes;
	delete [] m_ppRelations;
	delete [] m_pForeignKeyInfo;
}


//Initialize object with schema data
HRESULT CSchema::Init
	(
	REFGUID				rgSchema,			//@parm IN	| Schema GUID
	CSwstMetaHolder*	pSwstMetaHolder,	//@parm IN	| Pointer to metadata descriptions
	ULONG				cRestrictions,		//@parm IN	| Count of restriction values
	const VARIANT		rgRestrictions[]	//@parm IN	| Array of restriction values
	)
{
	TRACE( "CSchema::Init" );

	m_pSwstMetaHolder = pSwstMetaHolder;

	HRESULT hr = DB_E_NOTSUPPORTED;

	try{ 
		if (memcmp(&rgSchema, &DBSCHEMA_COLUMNS, sizeof(GUID)) == 0)
			hr = InitColumns(cRestrictions, rgRestrictions);
		else if (memcmp(&rgSchema, &DBSCHEMA_PROVIDER_TYPES, sizeof(GUID)) == 0)
			hr = InitProviderTypes(cRestrictions, rgRestrictions);
		else if (memcmp(&rgSchema, &DBSCHEMA_TABLES, sizeof(GUID)) == 0)
			hr = InitTables(cRestrictions, rgRestrictions);
		else if (memcmp(&rgSchema, &DBSCHEMA_CATALOGS, sizeof(GUID)) == 0)
			hr = InitCatalogs(cRestrictions, rgRestrictions);
		else if (memcmp(&rgSchema, &DBSCHEMA_SCHEMATA, sizeof(GUID)) == 0)
			hr = InitSchemata(cRestrictions, rgRestrictions);
		else if (memcmp(&rgSchema, &DBSCHEMA_STATISTICS, sizeof(GUID)) == 0)
			hr = InitStatistics(cRestrictions, rgRestrictions);
		else if (memcmp(&rgSchema, &DBSCHEMA_INDEXES, sizeof(GUID)) == 0)
			hr = InitIndexes(cRestrictions, rgRestrictions);
		else if (memcmp(&rgSchema, &DBSCHEMA_CONSTRAINT_COLUMN_USAGE, sizeof(GUID)) == 0)
			hr = InitConstraintColumns(cRestrictions, rgRestrictions);
		else if (memcmp(&rgSchema, &DBSCHEMA_CONSTRAINT_TABLE_USAGE, sizeof(GUID)) == 0)
			hr = InitConstraintTables(cRestrictions, rgRestrictions);
		else if (memcmp(&rgSchema, &DBSCHEMA_TABLE_CONSTRAINTS, sizeof(GUID)) == 0)
			hr = InitTableConstraints(cRestrictions, rgRestrictions);
		else if (memcmp(&rgSchema, &DBSCHEMA_PRIMARY_KEYS, sizeof(GUID)) == 0)
			hr = InitPrimaryKeys(cRestrictions, rgRestrictions);
		else if (memcmp(&rgSchema, &DBSCHEMA_FOREIGN_KEYS, sizeof(GUID)) == 0)
			hr = InitForeignKeys(cRestrictions, rgRestrictions);
		else if (memcmp(&rgSchema, &DBSCHEMA_REFERENTIAL_CONSTRAINTS, sizeof(GUID)) == 0)
			hr = InitReferentialConstraints(cRestrictions, rgRestrictions);

		if (hr != S_OK)
			return hr;

		return MoveFirst();
	}
	catch(...)
	{
		TRACE("EXCEPTION ON CSchema::Init");
		return E_ABORT;
	}
}


// Compare files
int __cdecl CompareFiles(const void *left, const void *right ) 
{
	SWSTFILE* pLeft = *(SWSTFILE**)left;
	SWSTFILE* pRight = *(SWSTFILE**)right;
	
	BOOL bIsSystem1 = (0 == strnicmp(pLeft->m_szName, "X$", 2));
	BOOL bIsSystem2 = (0 == strnicmp(pRight->m_szName, "X$", 2));
	
	if( bIsSystem1 && !bIsSystem2 )
		return -1;
	if( !bIsSystem1 && bIsSystem2 )
		return 1;

	int iCat = wcsicmp(pLeft->m_pCatalogOwner->m_wszCatalog, pRight->m_pCatalogOwner->m_wszCatalog );
	if( iCat )
		return iCat;

	int iOwn = wcsicmp(pLeft->m_pCatalogOwner->m_wszOwner, pRight->m_pCatalogOwner->m_wszOwner );
	if( iOwn )
		return iOwn;

	int iNam = stricmp(pLeft->m_szName, pRight->m_szName);
	return iNam;
}


//Initialize object with columns schema
HRESULT CSchema::InitTables
	(
	ULONG			cRestrictions,		//@parm IN	| Count of restriction values
	const VARIANT	rgRestrictions[]	//@parm IN	| Array of restriction values
	)
{
	TRACE( "CSchema::InitTables" );

	m_enumSchema = TABLES_SCHEMA;

	//Init columns
	static COLINFO Columns[] = 
	{
		{L"TABLE_CATALOG",	DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"TABLE_SCHEMA",	DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"TABLE_NAME",		DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"TABLE_TYPE",		DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"TABLE_GUID",		DBTYPE_GUID,	sizeof(GUID),	DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"DESCRIPTION",	DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"TABLE_PROPID",	DBTYPE_UI4,		sizeof(LONG),	DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"DATE_CREATED",	DBTYPE_DATE,	sizeof(DBDATE),	DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"DATE_MODIFIED",	DBTYPE_DATE,	sizeof(DBDATE),	DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL}
	};
	m_pColumns = Columns;
	m_cCols = 9;

	//Check restrictions
	if (cRestrictions > 4)
	{
		TRACE2( "CSchema::InitTables: restrictions = %d", cRestrictions );
		return E_INVALIDARG;
	}

	WCHAR* pwszCatalogNameRestr = NULL; 
	WCHAR* pwszSchemaNameRestr = NULL; 
	char szTableTypeRestr[MAX_NAME_SIZE]; 
	char szTableNameRestr[MAX_NAME_SIZE]; 
	bool bCatalogNameRestr = false;
	bool bSchemaNameRestr = false;
	bool bTableTypeRestr = false;
	bool bTableNameRestr = false;

	//Only BSTR is accepted
	if ((cRestrictions >= 1 && rgRestrictions[0].vt != VT_BSTR && rgRestrictions[0].vt != VT_EMPTY) ||
		(cRestrictions >= 2 && rgRestrictions[1].vt != VT_BSTR && rgRestrictions[1].vt != VT_EMPTY) || 
		(cRestrictions >= 3 && rgRestrictions[2].vt != VT_BSTR && rgRestrictions[2].vt != VT_EMPTY) || 
		(cRestrictions >= 4 && rgRestrictions[3].vt != VT_BSTR && rgRestrictions[3].vt != VT_EMPTY))
	{
		TRACE( "CSchema::InitTables: restr fails" );
		return E_INVALIDARG;
	}

	//Extract CATALOG_NAME restriction
	if (cRestrictions >= 1 && rgRestrictions[0].vt == VT_BSTR)
	{
		pwszCatalogNameRestr = rgRestrictions[0].bstrVal ? rgRestrictions[0].bstrVal : L"";
		bCatalogNameRestr = true;
		TRACE2( "  Catalog [%S]", pwszCatalogNameRestr );
	}

	//Extract SCHEMA_NAME restriction
	if (cRestrictions >= 2 && rgRestrictions[1].vt == VT_BSTR)
	{
		pwszSchemaNameRestr = rgRestrictions[1].bstrVal ? rgRestrictions[1].bstrVal : L"";;
		if( !wcsicmp( pwszSchemaNameRestr, INFORMATION_SCHEMA ) )
			pwszSchemaNameRestr = L"";
		bSchemaNameRestr = true;
		TRACE2( "  Schema [%S]", pwszSchemaNameRestr );
	}

	// Load catalog information if necessary
	if( (bCatalogNameRestr || bSchemaNameRestr) && m_pCDataSource )
	{
		// May fail, Doesn't matter
		m_pCDataSource->CheckAndTrackOneDatabase( pwszCatalogNameRestr, pwszSchemaNameRestr );
	}

	//Extract TABLE_NAME restriction
	if (cRestrictions >= 3 && rgRestrictions[2].vt == VT_BSTR)
	{
		if( rgRestrictions[2].bstrVal == NULL )
			*szTableNameRestr = '\0';
		else
			WideCharToMultiByte(CP_ACP, 0,
							rgRestrictions[2].bstrVal, -1,
							szTableNameRestr, MAX_NAME_SIZE,
							NULL, NULL );
		bTableNameRestr = true;
		TRACE2( "  Table [%s]", szTableNameRestr );
	}

	//Extract TABLE_TYPE restriction
	if (cRestrictions >= 4 && rgRestrictions[3].vt == VT_BSTR)
	{
		if( rgRestrictions[3].bstrVal == NULL )
			*szTableTypeRestr = '\0';
		else
			WideCharToMultiByte(CP_ACP, 0,
							rgRestrictions[3].bstrVal, -1,
							szTableTypeRestr, MAX_NAME_SIZE,
							NULL, NULL );
		bTableTypeRestr = true;
		TRACE2( "  Type [%s]", szTableTypeRestr );
	}

	TRACE( "CSchema::InitTables: before getting data" );

	// Get total number of files
	int uiTotalFiles = 0;
	int iTotalMetas;
	CSwstMeta* pSwstMeta;
	if( FAILED( m_pSwstMetaHolder->GetMaxNumber( &iTotalMetas ) ) )
		return E_FAIL;
	for( int j = 0; j < iTotalMetas; j++ )
		if( m_pSwstMetaHolder->At( j, &pSwstMeta ) == S_OK &&
			(!bCatalogNameRestr || !wcsicmp( pwszCatalogNameRestr, pSwstMeta->m_CatalogOwner.m_wszCatalog ) ) &&
			(!bSchemaNameRestr || !wcsicmp( pwszSchemaNameRestr, pSwstMeta->m_CatalogOwner.m_wszOwner ) ) )
				uiTotalFiles += pSwstMeta->m_siFiles;
	
	//Init data
	m_ppFiles = new SWSTFILE*[ uiTotalFiles ? uiTotalFiles : 1];
	if( m_ppFiles == NULL )
		return E_OUTOFMEMORY;

#ifdef _DEBUG
			int maxNumb = -1;
			m_pSwstMetaHolder->GetMaxNumber(&maxNumb);
			TRACE( " >> There are %d dictionaries", maxNumb );
#endif

	//Copy file pointers to buffer
	m_cRows = 0;
	for( j = 0; j < iTotalMetas; j++ )
		if( m_pSwstMetaHolder->At( j, &pSwstMeta ) == S_OK &&
		   (!bCatalogNameRestr || !wcsicmp( pwszCatalogNameRestr, pSwstMeta->m_CatalogOwner.m_wszCatalog ) ) &&
		   (!bSchemaNameRestr || !wcsicmp( pwszSchemaNameRestr, pSwstMeta->m_CatalogOwner.m_wszOwner ) ) )
		{
				for (int i = 0; i < pSwstMeta->m_siFiles; i++)
				{
					//Check TABLE_NAME restriction
					if (bTableNameRestr)
					{
						if (0 != stricmp(pSwstMeta->m_pFiles[i].m_szName, szTableNameRestr))
							continue;
					}

					//Check TABLE_TYPE restriction
					if (bTableTypeRestr)
					{
						bool bIsSystem = (0 == strnicmp(pSwstMeta->m_pFiles[i].m_szName, "X$", 2));
						
						if (0 == stricmp("SYSTEM TABLE", szTableTypeRestr))
						{
							if (!bIsSystem)
								continue;
						}
						else
						if (0 == stricmp("TABLE", szTableTypeRestr))
						{
							if (bIsSystem)
								continue;
						}
						else
							continue;
					}

					//Copy file pointer
					m_ppFiles[m_cRows] = &pSwstMeta->m_pFiles[i];
					m_cRows++;
				}
		}
	
	//Sort file pointers by file name (system files first)
	qsort( m_ppFiles, m_cRows, sizeof(m_ppFiles[0]), CompareFiles );
/*	SWSTFILE *pTemp;
	for ( j = m_cRows - 1; j > 0; j--)
	{
		for (int k = 0; k < j; k++)
		{
			BOOL bIsSystem1 = (0 == strnicmp(m_ppFiles[k]->m_szName, "X$", 2));
			BOOL bIsSystem2 = (0 == strnicmp(m_ppFiles[k + 1]->m_szName, "X$", 2));
			BOOL bIsGreatThan = bIsSystem2;

			if ( ( bIsSystem1 && bIsSystem2 ) || (!bIsSystem1 && !bIsSystem2) )
			{
				int iCat = wcsicmp(m_ppFiles[k]->m_pCatalogOwner->m_wszCatalog, m_ppFiles[k + 1]->m_pCatalogOwner->m_wszCatalog );
				int iOwn = wcsicmp(m_ppFiles[k]->m_pCatalogOwner->m_wszOwner, m_ppFiles[k + 1]->m_pCatalogOwner->m_wszOwner );
				int iNam = stricmp(m_ppFiles[k]->m_szName, m_ppFiles[k + 1]->m_szName);
				bIsGreatThan = iCat > 0 || ( iCat==0 && iOwn > 0 ) || ( iCat==0 && iOwn==0 && iNam > 0);
			}

			if (bIsGreatThan)
			{
				pTemp = m_ppFiles[k];
				m_ppFiles[k] = m_ppFiles[k + 1];
				m_ppFiles[k + 1] = pTemp;
			}
		}
	}*/

	TRACE2( "CSchema::InitTables: completing (total %d)", m_cRows );
	return S_OK;
}


// Compare files for columns schema
int __cdecl CompareColumns(const void *left, const void *right ) 
{
	SWSTFILE* pLeft = *(SWSTFILE**)left;
	SWSTFILE* pRight = *(SWSTFILE**)right;

	int iCat = wcsicmp(pLeft->m_pCatalogOwner->m_wszCatalog, pRight->m_pCatalogOwner->m_wszCatalog );
	if( iCat )
		return iCat;

	int iOwn = wcsicmp(pLeft->m_pCatalogOwner->m_wszOwner, pRight->m_pCatalogOwner->m_wszOwner );
	if( iOwn )
		return iOwn;

	int iNam = stricmp(pLeft->m_szName, pRight->m_szName);
	return iNam;
}

	
//Initialize object with columns schema
HRESULT CSchema::InitColumns
	(
	ULONG			cRestrictions,		//@parm IN	| Count of restriction values
	const VARIANT	rgRestrictions[]	//@parm IN	| Array of restriction values
	)
{
	TRACE( "CSchema::InitColumns" );

	m_enumSchema = COLUMNS_SCHEMA;

	//Init columns
	m_cCols = 28;
	static COLINFO Columns[] = 
	{
		{L"TABLE_CATALOG",			DBTYPE_WSTR,	MAX_NAME_SIZE,			DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"TABLE_SCHEMA",			DBTYPE_WSTR,	MAX_NAME_SIZE,			DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"TABLE_NAME",				DBTYPE_WSTR,	MAX_NAME_SIZE,			DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"COLUMN_NAME",			DBTYPE_WSTR,	MAX_NAME_SIZE,			DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"COLUMN_GUID",			DBTYPE_GUID,	sizeof(GUID),			DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"COLUMN_PROPID",			DBTYPE_UI4,		sizeof(LONG),			DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"ORDINAL_POSITION",		DBTYPE_UI4,		sizeof(LONG),			DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},			
		{L"COLUMN_HASDEFAULT",		DBTYPE_BOOL,	sizeof(VARIANT_BOOL),	DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},			
		{L"COLUMN_DEFAULT",			DBTYPE_WSTR,	MAX_NAME_SIZE,			DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"COLUMN_FLAGS",			DBTYPE_UI4,		sizeof(LONG),			DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"IS_NULLABLE",			DBTYPE_BOOL,	sizeof(VARIANT_BOOL),	DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},			
		{L"DATA_TYPE",				DBTYPE_UI2,		sizeof(WORD),			DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"TYPE_GUID",				DBTYPE_GUID,	sizeof(GUID),			DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"CHARACTER_MAXIMUM_LENGTH",DBTYPE_UI4,	sizeof(LONG),			DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"CHARACTER_OCTET_LENGTH", DBTYPE_UI4,		sizeof(LONG),			DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"NUMERIC_PRECISION",		DBTYPE_UI2,		sizeof(WORD),			DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"NUMERIC_SCALE",			DBTYPE_UI2,		sizeof(WORD),			DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"DATETIME_PRECISION",		DBTYPE_UI4,		sizeof(LONG),			DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"CHARACTER_SET_CATALOG",	DBTYPE_WSTR,	MAX_NAME_SIZE,			DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"CHARACTER_SET_SCHEMA",	DBTYPE_WSTR,	MAX_NAME_SIZE,			DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"CHARACTER_SET_NAME",		DBTYPE_WSTR,	MAX_NAME_SIZE,			DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"COLLATION_CATALOG",		DBTYPE_WSTR,	MAX_NAME_SIZE,			DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"COLLATION_SCHEMA",		DBTYPE_WSTR,	MAX_NAME_SIZE,			DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"COLLATION_NAME",			DBTYPE_WSTR,	MAX_NAME_SIZE,			DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"DOMAIN_CATALOG",			DBTYPE_WSTR,	MAX_NAME_SIZE,			DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"DOMAIN_SCHEMA",			DBTYPE_WSTR,	MAX_NAME_SIZE,			DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"DOMAIN_NAME",			DBTYPE_WSTR,	MAX_NAME_SIZE,			DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"DESCRIPTION",			DBTYPE_WSTR,	MAX_NAME_SIZE,			DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL}
	};

	m_pColumns = Columns;

	//Check restrictions
	if (cRestrictions > 4)
	{
		TRACE2( "CSchema::InitColumns: restrictions = %d", cRestrictions );
		return E_INVALIDARG;
	}

	//extract all restrictions
	WCHAR* pwszCatalogNameRestr = NULL; 
	WCHAR* pwszSchemaNameRestr = NULL; 
	char szTableNameRestr[MAX_NAME_SIZE]; 
	char szColumnNameRestr[MAX_NAME_SIZE]; 
	bool bCatalogNameRestr = false;
	bool bSchemaNameRestr = false;
	bool bTableNameRestr = false; 
	bool bColumnNameRestr = false; 

	//Only BSTR is accepted
	if ((cRestrictions >= 1 && rgRestrictions[0].vt != VT_BSTR && rgRestrictions[0].vt != VT_EMPTY) ||
		(cRestrictions >= 2 && rgRestrictions[1].vt != VT_BSTR && rgRestrictions[1].vt != VT_EMPTY) || 
		(cRestrictions >= 3 && rgRestrictions[2].vt != VT_BSTR && rgRestrictions[2].vt != VT_EMPTY) || 
		(cRestrictions >= 4 && rgRestrictions[3].vt != VT_BSTR && rgRestrictions[3].vt != VT_EMPTY))
	{
		return E_INVALIDARG;
	}

	//Extract CATALOG_NAME restriction
	if (cRestrictions >= 1 && rgRestrictions[0].vt == VT_BSTR)
	{
		pwszCatalogNameRestr = rgRestrictions[0].bstrVal ? rgRestrictions[0].bstrVal : L"";
		bCatalogNameRestr = true;
	}

	//Extract SHEMA_NAME restriction
	if (cRestrictions >= 2 && rgRestrictions[1].vt == VT_BSTR)
	{
		pwszSchemaNameRestr = rgRestrictions[1].bstrVal ? rgRestrictions[1].bstrVal : L"";
		if( !wcsicmp( pwszSchemaNameRestr, INFORMATION_SCHEMA ) )
			pwszSchemaNameRestr = L"";;
		bSchemaNameRestr = true;
	}

	// Load catalog information if necessary
	if( (bCatalogNameRestr || bSchemaNameRestr) && m_pCDataSource )
	{
		m_pCDataSource->CheckAndTrackOneDatabase( pwszCatalogNameRestr, pwszSchemaNameRestr );
	}

	//Extract TABLE_NAME restriction
	if (cRestrictions >= 3 && rgRestrictions[2].vt == VT_BSTR)
	{
		if( rgRestrictions[2].bstrVal == NULL )
			*szTableNameRestr = '\0';
		else
			WideCharToMultiByte(CP_ACP, 0,
							rgRestrictions[2].bstrVal, -1,
							szTableNameRestr, MAX_NAME_SIZE,
							NULL, NULL );
		bTableNameRestr = true;
	}

	//Extract COLUMN_NAME restriction
	if (cRestrictions >= 4 && rgRestrictions[3].vt == VT_BSTR)
	{
		if( rgRestrictions[3].bstrVal == NULL )
			*szColumnNameRestr = '\0';
		else
			WideCharToMultiByte(CP_ACP, 0,
							rgRestrictions[3].bstrVal, -1,
							szColumnNameRestr, MAX_NAME_SIZE,
							NULL, NULL );
		bColumnNameRestr = true;
	}
	
	// Get total number of files and fields
	int uiTotalFiles = 0, uiTotalFields = 0;
	int iTotalMetas;
	CSwstMeta* pSwstMeta;
	if( FAILED( m_pSwstMetaHolder->GetMaxNumber( &iTotalMetas ) ) )
		return E_FAIL;
	for( int j = 0; j < iTotalMetas; j++ )
		if( m_pSwstMetaHolder->At( j, &pSwstMeta ) == S_OK &&
			(!bCatalogNameRestr || !wcsicmp( pwszCatalogNameRestr, pSwstMeta->m_CatalogOwner.m_wszCatalog ) ) &&
			(!bSchemaNameRestr || !wcsicmp( pwszSchemaNameRestr, pSwstMeta->m_CatalogOwner.m_wszOwner ) ) )
		{
			uiTotalFiles += pSwstMeta->m_siFiles;
			uiTotalFields += pSwstMeta->m_siFields;
		}
	
	//Init data
	m_ppFiles = new SWSTFILE*[ uiTotalFiles ? uiTotalFiles : 1];
	if( m_ppFiles == NULL )
		return E_OUTOFMEMORY;
	m_pFieldInfos	= new FIELDINFO[ uiTotalFields ? uiTotalFields : 1 ];
	if( m_pFieldInfos == NULL )
	{
		delete m_ppFiles;
		m_ppFiles = NULL;
		return E_OUTOFMEMORY;
	}

	//Copy file pointers to buffer
	int iFiles = 0;
	for( j = 0; j < iTotalMetas; j++ )
		if( m_pSwstMetaHolder->At( j, &pSwstMeta ) == S_OK &&
			(!bCatalogNameRestr || !wcsicmp( pwszCatalogNameRestr, pSwstMeta->m_CatalogOwner.m_wszCatalog ) ) &&
			(!bSchemaNameRestr || !wcsicmp( pwszSchemaNameRestr, pSwstMeta->m_CatalogOwner.m_wszOwner ) ) )
		{
			for (int i = 0; i < pSwstMeta->m_siFiles; i++)
			{
				//Check TABLE_NAME restriction
				if (bTableNameRestr)
				{
					if (0 != stricmp(pSwstMeta->m_pFiles[i].m_szName, szTableNameRestr))
						continue;
				}

				m_ppFiles[iFiles] = &pSwstMeta->m_pFiles[i];
				iFiles++;
			}
		}
	
	//Sort file pointers by file name
	qsort( m_ppFiles, iFiles, sizeof(m_ppFiles[0]), CompareColumns );
	/*SWSTFILE *pFile;
	for (j = iFiles - 1; j > 0; j--)
	{
		for (int k = 0; k < j; k++)
		{
			int iCat = wcsicmp(m_ppFiles[k]->m_pCatalogOwner->m_wszCatalog, m_ppFiles[k + 1]->m_pCatalogOwner->m_wszCatalog );
			int iOwn = wcsicmp(m_ppFiles[k]->m_pCatalogOwner->m_wszOwner, m_ppFiles[k + 1]->m_pCatalogOwner->m_wszOwner );
			int iNam = stricmp(m_ppFiles[k]->m_szName, m_ppFiles[k + 1]->m_szName);
				
			if( iCat > 0 || ( iCat==0 && iOwn > 0 ) || ( iCat==0 && iOwn==0 && iNam > 0) )
			{
				pFile = m_ppFiles[k];
				m_ppFiles[k] = m_ppFiles[k + 1];
				m_ppFiles[k + 1] = pFile;
			}
		}
	}*/

	//Fill out field pointer array
	m_cRows = 0;
	for (int i = 0; i < iFiles; i++)
	{
		SWSTFILE *pFile = m_ppFiles[i];
		for (j = 0; j < pFile->m_wFields; j++)
		{
			//Check COLUMN_NAME restriction
			if (bColumnNameRestr)
			{
				if (0 != stricmp(pFile->m_pFields[j]->m_szName, szColumnNameRestr))
					continue;
			}

			//Fill out new FIELDINFO structure
			m_pFieldInfos[m_cRows].m_pField		= pFile->m_pFields[j];
			m_pFieldInfos[m_cRows].m_pFile		= pFile;
			m_pFieldInfos[m_cRows].m_ulPosition = j;
			m_cRows++;
		}
	}

	return S_OK;
}

//Initialize object with provider types schema
HRESULT CSchema::InitProviderTypes
	(
	ULONG			cRestrictions,		//@parm IN	| Count of restriction values
	const VARIANT	rgRestrictions[]	//@parm IN	| Array of restriction values
	)
{
	TRACE( "CSchema::InitProviderTypes" );

	//Set schema type
	m_enumSchema = TYPES_SCHEMA;

	//Init column info
	static COLINFO Columns[] = 
	{
		{L"TYPE_NAME",			DBTYPE_WSTR,	MAX_NAME_SIZE,			DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"DATA_TYPE",			DBTYPE_UI2,		sizeof(WORD),			DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"COLUMN_SIZE",		DBTYPE_UI4,		sizeof(LONG),			DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"LITERAL_PREFIX",		DBTYPE_WSTR,	MAX_NAME_SIZE,			DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"LITERAL_SUFFIX",		DBTYPE_WSTR,	MAX_NAME_SIZE,			DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"CREATE_PARAMS",		DBTYPE_WSTR,	MAX_NAME_SIZE,			DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"IS_NULLABLE",		DBTYPE_BOOL,	sizeof(VARIANT_BOOL),	DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},			
		{L"CASE_SENSITIVE",		DBTYPE_BOOL,	sizeof(VARIANT_BOOL),	DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},			
		{L"SEARCHABLE",			DBTYPE_UI4,		sizeof(LONG),			DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},			
		{L"UNSIGNED_ATTRIBUTE",	DBTYPE_BOOL,	sizeof(VARIANT_BOOL),	DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},			
		{L"FIXED_PREC_SCALE",	DBTYPE_BOOL,	sizeof(VARIANT_BOOL),	DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},			
		{L"AUTO_UNIQUE_VALUE",	DBTYPE_BOOL,	sizeof(VARIANT_BOOL),	DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},			
		{L"LOCAL_TYPE_NAME",	DBTYPE_WSTR,	MAX_NAME_SIZE,			DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"MINIMUM_SCALE",		DBTYPE_UI2,		sizeof(WORD),			DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"MAXIMUM_SCALE",		DBTYPE_UI2,		sizeof(WORD),			DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"GUID",				DBTYPE_GUID,	sizeof(GUID),			DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"TYPELIB",			DBTYPE_WSTR,	MAX_NAME_SIZE,			DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"VERSION",			DBTYPE_WSTR,	MAX_NAME_SIZE,			DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"IS_LONG",			DBTYPE_BOOL,	sizeof(VARIANT_BOOL),	DBCOLUMNFLAGS_ISFIXEDLENGTH},			
		{L"BEST_MATCH",			DBTYPE_BOOL,	sizeof(VARIANT_BOOL),	DBCOLUMNFLAGS_ISFIXEDLENGTH},			
		{L"IS_FIXEDLENGTH",		DBTYPE_BOOL,	sizeof(VARIANT_BOOL),	DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL}
	};
    
	m_pColumns = Columns;
	m_cCols = 21;
	const float ver61 = (float)6.1, ver70 = (float)7.0, ver75 = (float)7.5, ver323 = (float) 3.23;

	static TYPEINFO TypesOlder[] =		//Sorted by OLEDB data type
	{
		//Name				Version	Data type			Size	Prefix	Suffix	Create				Unsigned		Fixed P&S		AutoUniqueValue	BestMatch		Fixed Length	Is Long
		{L"INTEGER(2)",		ver61,	DBTYPE_I2,			2, 		L"",	L"",	L"",				VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"AUTOINC(2)",		ver61,	DBTYPE_I2,			2, 		L"",	L"",	L"",				VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"INTEGER(4)",		ver61,	DBTYPE_I4,			4, 		L"",	L"",	L"",				VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"AUTOINC(4)",		ver61,	DBTYPE_I4,			4, 		L"",	L"",	L"",				VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"FLOAT(4)",		ver61,	DBTYPE_R4,			7,		L"",	L"",	L"",				VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"BFLOAT(4)",		ver61,	DBTYPE_R4,			7,		L"",	L"",	L"",				VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"FLOAT(8)",		ver61,	DBTYPE_R8,			15,		L"",	L"",	L"",				VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"BFLOAT(8)",		ver61,	DBTYPE_R8,			15,		L"",	L"",	L"",				VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"MONEY",			ver61,	DBTYPE_R8,			15,		L"",	L"",	L"precision,scale",	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"DECIMAL",		ver61,	DBTYPE_R8,			15,		L"",	L"",	L"precision,scale",	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"NUMERIC",		ver61,	DBTYPE_R8,			15,		L"",	L"",	L"precision,scale",	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_FALSE},		
		{L"NUMERICSA",		ver61,	DBTYPE_R8,			15,		L"",	L"",	L"precision,scale",	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_FALSE},		
		{L"NUMERICSTS",		ver61,	DBTYPE_R8,			15,		L"",	L"",	L"precision,scale",	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_FALSE},		
		{L"CURRENCY",		ver70,	DBTYPE_CY,			19,		L"",	L"",	L"",				VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"BIT",			ver61,	DBTYPE_BOOL,		1,		L"",	L"",	L"",				-100,			VARIANT_TRUE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"LOGICAL(1)",		ver61,	DBTYPE_BOOL,		1,		L"",	L"",	L"",				-100,			VARIANT_TRUE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"LOGICAL(2)",		ver61,	DBTYPE_BOOL,		2,		L"",	L"",	L"",				-100,			VARIANT_TRUE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"INTEGER(1)",		ver61,	DBTYPE_I1,			1, 		L"",	L"",	L"",				VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"UNSIGNED(1)",	ver70,	DBTYPE_UI1,			1, 		L"",	L"",	L"",				VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"UNSIGNED(2)",	ver70,	DBTYPE_UI2,			2, 		L"",	L"",	L"",				VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"UNSIGNED(4)",	ver70,	DBTYPE_UI4,			4, 		L"",	L"",	L"",				VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"INTEGER(8)",		ver70,	DBTYPE_I8,			8, 		L"",	L"",	L"",				VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"UNSIGNED(8)",	ver70,	DBTYPE_UI8,			8, 		L"",	L"",	L"",				VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE},
	
		{L"LVAR",			ver61,	DBTYPE_BYTES,		32765,	L"'",	L"'",	L"length",			-100,			VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_TRUE},
		{L"ZSTRING",   		ver61,	DBTYPE_STR,			255,	L"'",	L"'",	L"length",			-100,			VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_FALSE/*VARIANT_TRUE*/,	VARIANT_FALSE},	// For better compatibility with MS SQL Server
		{L"CHARACTER",		ver61,	DBTYPE_STR,			255,	L"'",	L"'",	L"length",			-100,			VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE/*VARIANT_TRUE*/,	VARIANT_FALSE},	// For better compatibility with MS SQL Server
		{L"LSTRING", 		ver61,	DBTYPE_STR,			255,	L"'",	L"'",	L"length",			-100,			VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE/*VARIANT_TRUE*/,	VARIANT_FALSE},	// For better compatibility with MS SQL Server
		{L"NOTE",			ver70,	DBTYPE_STR,			32765,	L"'",	L"'",	L"length",			-100,			VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_TRUE},

		{L"DATE",			ver61,	DBTYPE_DBDATE,		4,		L"'",	L"'",	L"",				-100,			VARIANT_TRUE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"TIME",			ver61,	DBTYPE_DBTIME,		4,		L"'",	L"'",	L"",				-100,			VARIANT_TRUE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"TIMESTAMP",		ver70,	DBTYPE_DBTIMESTAMP,	4,		L"'",	L"'",	L"",				-100,			VARIANT_TRUE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE}
	};
	
	static TYPEINFO Types2000[] =		// Types of P.SQL 2000. Sorted by OLEDB data type
	{
		//Name				Version	Data type			Size	Prefix	Suffix	Create				Unsigned		Fixed P&S		AutoUniqueValue	BestMatch		Fixed Length	Is Long
		{L"SMALLINT",		ver75,	DBTYPE_I2,			2, 		L"",	L"",	L"",				VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"SMALLIDENTITY",	ver75,	DBTYPE_I2,			2, 		L"",	L"",	L"",				VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"INT",			ver75,	DBTYPE_I4,			4, 		L"",	L"",	L"",				VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"IDENTITY",		ver75,	DBTYPE_I4,			4, 		L"",	L"",	L"",				VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"REAL",			ver75,	DBTYPE_R4,			7,		L"",	L"",	L"",				VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"FLOAT",			ver75,	DBTYPE_R4,			7,		L"",	L"",	L"",				VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"DOUBLE",			ver75,	DBTYPE_R8,			15,		L"",	L"",	L"",				VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"DECIMAL",		ver75,	DBTYPE_R8,			15,		L"",	L"",	L"precision,scale",	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"NUMERIC",		ver75,	DBTYPE_R8,			15,		L"",	L"",	L"precision,scale",	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_FALSE},		
		{L"CURRENCY",		ver75,	DBTYPE_CY,			19,		L"",	L"",	L"",				VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"BIT",			ver75,	DBTYPE_BOOL,		1,		L"",	L"",	L"",				-100,			VARIANT_TRUE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"TINYINT",		ver75,	DBTYPE_I1,			1, 		L"",	L"",	L"",				VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"UTINYINT",		ver75,	DBTYPE_UI1,			1, 		L"",	L"",	L"",				VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"USMALLINT",		ver75,	DBTYPE_UI2,			2, 		L"",	L"",	L"",				VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"UINTEGER",		ver75,	DBTYPE_UI4,			4, 		L"",	L"",	L"",				VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"BIGINT",			ver75,	DBTYPE_I8,			8, 		L"",	L"",	L"",				VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"UBIGINT",		ver75,	DBTYPE_UI8,			8, 		L"",	L"",	L"",				VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE},
	
		{L"BINARY",			ver75,	DBTYPE_BYTES,		255,	L"'",	L"'",	L"length",			-100,			VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_FALSE/*VARIANT_TRUE*/,	VARIANT_FALSE},	// For better compatibility with MS SQL Server
		{L"VARCHAR",   		ver75,	DBTYPE_STR,			254,	L"'",	L"'",	L"length",			-100,			VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_FALSE/*VARIANT_TRUE*/,	VARIANT_FALSE},	// For better compatibility with MS SQL Server
		{L"CHARACTER",		ver75,	DBTYPE_STR,			255,	L"'",	L"'",	L"length",			-100,			VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE/*VARIANT_TRUE*/,	VARIANT_FALSE},	// For better compatibility with MS SQL Server
		
		{L"LONGVARCHAR",	ver75,	DBTYPE_STR,			0x7FFFFFFFL, // Maxumum positive number on 4 bytes
																L"'",	L"'",	L"",			-100,			VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_TRUE},
		{L"LONG VARBINARY",	ver75,	DBTYPE_BYTES,		0x7FFFFFFFL, // Maxumum positive number on 4 bytes
																L"'",	L"'",	L"",			-100,			VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_TRUE},

		{L"DATE",			ver75,	DBTYPE_DBDATE,		4,		L"'",	L"'",	L"",				-100,			VARIANT_TRUE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"TIME",			ver75,	DBTYPE_DBTIME,		4,		L"'",	L"'",	L"",				-100,			VARIANT_TRUE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"TIMESTAMP",		ver75,	DBTYPE_DBTIMESTAMP,	4,		L"'",	L"'",	L"",				-100,			VARIANT_TRUE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE}
	};

	static TYPEINFO MySQL[] =		// Types of MySQL
	{
		//Name				Version	Data type			Size	Prefix	Suffix	Create				Unsigned		Fixed P&S		AutoUniqueValue	BestMatch		Fixed Length	Is Long
		{L"TINYINT",		ver323,	DBTYPE_I1,			1, 		L"",	L"",	L"",				VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"YEAR",			ver323,	DBTYPE_I1,			1, 		L"",	L"",	L"",				VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"TINYINT UNSIGNED",ver323,DBTYPE_UI1,			1, 		L"",	L"",	L"",				VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"SMALLINT",		ver323,	DBTYPE_I2,			2, 		L"",	L"",	L"",				VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"SMALLINT AUTO_INCREMENT",ver323,	DBTYPE_I2,	2, 		L"",	L"",	L"",				VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"SMALLINT UNSIGNED",ver323,DBTYPE_UI2,		2, 		L"",	L"",	L"",				VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE},		
		{L"INTEGER",		ver323,	DBTYPE_I4,			4, 		L"",	L"",	L"",				VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"INTEGER AUTO_INCREMENT",ver323,	DBTYPE_I4,	4, 		L"",	L"",	L"",				VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"INTEGER UNSIGNED",ver323,	DBTYPE_UI4,		4, 		L"",	L"",	L"",				VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"MEDIUMINT",		ver323,	DBTYPE_I4,			4, 		L"",	L"",	L"",				VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"MEDIUMINT UNSIGNED",ver323,DBTYPE_UI4,		4, 		L"",	L"",	L"",				VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"BIGINT",			ver323,	DBTYPE_I8,			8, 		L"",	L"",	L"",				VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"BIGINT UNSIGNED",ver323,	DBTYPE_UI8,			8, 		L"",	L"",	L"",				VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"FLOAT",			ver323,	DBTYPE_R4,			7,		L"",	L"",	L"",				VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"DOUBLE",			ver323,	DBTYPE_R8,			15,		L"",	L"",	L"",				VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE},		
		{L"DECIMAL",		ver323,	DBTYPE_R8,			15,		L"",	L"",	L"precision,scale",	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"NUMERIC",		ver323,	DBTYPE_R8,			15,		L"",	L"",	L"precision,scale",	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_FALSE},		
		{L"DATE",			ver323,	DBTYPE_DBDATE,		4,		L"'",	L"'",	L"",				-100,			VARIANT_TRUE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"TIME",			ver323,	DBTYPE_DBTIME,		4,		L"'",	L"'",	L"",				-100,			VARIANT_TRUE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"TIMESTAMP",		ver323,	DBTYPE_DBTIMESTAMP,	4,		L"'",	L"'",	L"length",			-100,			VARIANT_TRUE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_FALSE},
		{L"DATETIME",		ver323,	DBTYPE_DBTIMESTAMP,	4,		L"'",	L"'",	L"",				-100,			VARIANT_TRUE,	VARIANT_FALSE,	VARIANT_TRUE,	VARIANT_TRUE,	VARIANT_FALSE},
		
		{L"CHAR",			ver323,	DBTYPE_STR,			255,	L"'",	L"'",	L"length",			-100,			VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE},	
		{L"BINARY",			ver323,	DBTYPE_BYTES,		255,	L"'",	L"'",	L"length",			-100,			VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE},	
		{L"VARCHAR",		ver323,	DBTYPE_STR,			255,	L"'",	L"'",	L"length",			-100,			VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE},	
		{L"VARBINARY",		ver323,	DBTYPE_BYTES,		255,	L"'",	L"'",	L"length",			-100,			VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE},	

		{L"TINYTEXT",		ver323,	DBTYPE_STR,			255,	L"'",	L"'",	L"",				-100,			VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_TRUE},	
		{L"TINYBLOB",		ver323,	DBTYPE_BYTES,		255,	L"'",	L"'",	L"",				-100,			VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_TRUE},	
		{L"TEXT",			ver323,	DBTYPE_STR,			16*1024L /*65535L*/,	L"'",	L"'",	L"",				-100,			VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_TRUE},	
		{L"BLOB",			ver323,	DBTYPE_BYTES,		16*1024L /*65535L*/,	L"'",	L"'",	L"",				-100,			VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_TRUE},	
		{L"MEDIUMTEXT",		ver323,	DBTYPE_STR,			32*1024L /*16777215L*/,	L"'",	L"'",	L"",				-100,			VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_TRUE},	
		{L"MEDIUMBLOB",		ver323,	DBTYPE_BYTES,		32*1024L /*16777215L*/,	L"'",	L"'",	L"",				-100,			VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_TRUE},	
		{L"LONGTEXT",		ver323,	DBTYPE_STR,			63*1024L /*2147483647L*/,	L"'",	L"'",	L"",				-100,			VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_TRUE},	
		{L"LONGBLOB",		ver323,	DBTYPE_BYTES,		63*1024L /*2147483647L*/,	L"'",	L"'",	L"",				-100,			VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_FALSE,	VARIANT_TRUE}
	};

	// Set right pointer and count number of supported data types
	int nNumElemTypes;
	TYPEINFO* Types;
	float fltVersion;

	//if( m_pCDataSource->m_pUtilProp->MySqlMode() )
	if( m_pCDataSource->m_pUtilProp->InternalSqlSupport() == ISS_MYSQL )
	{
		nNumElemTypes = NUMELEM( MySQL );
		Types = MySQL;
		fltVersion = ver323;
	}

	//Check restrictions
	if (cRestrictions > 2)
	{
		TRACE2( "CSchema::InitProviderTypes: restrictions = %d", cRestrictions );
		return E_INVALIDARG;
	}
	
	//Extract all restrictions
	WORD wDataTypeRestr; 
	VARIANT_BOOL vbBestMatchRestr;
	bool bDataTypeRestr = false;
	bool bBestMatchRestr = false;

	//Extract DATA_TYPE restriction
	if( cRestrictions >= 1 )
	{
		switch (rgRestrictions[0].vt)
		{
		case VT_I2:
			wDataTypeRestr = rgRestrictions[0].iVal;
			bDataTypeRestr = true;
			break;

		case VT_I4:
			wDataTypeRestr = rgRestrictions[0].lVal;
			bDataTypeRestr = true;
			break;
		
		case VT_EMPTY:
			break;

		default:
			return E_INVALIDARG;
		}
	}

	//Extract BEST_MATCH restriction
	if( cRestrictions >= 2 )
	{
		switch (rgRestrictions[1].vt)
		{
		case VT_BOOL:
			vbBestMatchRestr = rgRestrictions[1].boolVal;
			bBestMatchRestr = true;
			break;

		case VT_EMPTY:
			break;

		default:
			return E_INVALIDARG;
		}
	}

	//Init data
	m_ppTypes = new TYPEINFO*[nNumElemTypes];

	//Fill out pointer array
	m_cRows = 0;
	for (int i = 0; i < nNumElemTypes; i++)
	{
		//Check  version (not all types are available in all versions)
		if (fltVersion < Types[i].fltVersion)
			continue;

		//Check TABLE_NAME restriction
		if (bDataTypeRestr)
		{
			if (Types[i].wDataType != wDataTypeRestr)
				continue;
		}

		//Check BEST_MATCH restriction
		if (bBestMatchRestr)
		{
			if (Types[i].bBestMatch != vbBestMatchRestr)
				continue;
		}

		m_ppTypes[m_cRows] = &Types[i];
		m_cRows++;
	}
		
	return S_OK;
}


// Compare metas for catalog schema
int __cdecl CompareCatalogs(const void *left, const void *right ) 
{
	CSwstMeta* pLeft = *(CSwstMeta**)left;
	CSwstMeta* pRight = *(CSwstMeta**)right;

	return wcsicmp(pLeft->m_CatalogOwner.m_wszCatalog, pRight->m_CatalogOwner.m_wszCatalog );
}



//Initialize object with catalogs schema
HRESULT CSchema::InitCatalogs
	(
	ULONG			cRestrictions,		//@parm IN	| Count of restriction values
	const VARIANT	rgRestrictions[]	//@parm IN	| Array of restriction values
	)
{
	TRACE( "CSchema::InitCatalogs" );

	HRESULT ret = E_FAIL;
	
	m_enumSchema = TABLES_SCHEMA;

	//Init columns
	static COLINFO Columns[] = 
	{
		{L"CATALOG_NAME",	DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"DESCRIPTION",	DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL}
	};
	m_pColumns = Columns;
	m_cCols = 2;

	//Set schema type
	m_enumSchema = CATALOGS_SCHEMA;
	
	//Check restrictions
	if (cRestrictions > 1)
		return E_INVALIDARG;
	
	//extract all restrictions
	WCHAR * pwszCatalogNameRestr = NULL; 
	bool bCatalogNameRestr = false; 

	//Only BSTR is accepted
	if ( cRestrictions >= 1 && rgRestrictions[0].vt != VT_BSTR && rgRestrictions[0].vt != VT_EMPTY )
		return E_INVALIDARG;

	//Extract CATALOG_NAME restriction
	if (cRestrictions >= 1 && rgRestrictions[0].vt == VT_BSTR)
	{
		pwszCatalogNameRestr = rgRestrictions[0].bstrVal ? rgRestrictions[0].bstrVal : L"";
		bCatalogNameRestr = true;
	}

	// Load catalog information if necessary
	if( bCatalogNameRestr && m_pCDataSource )
	{
		// May fail, Doesn't matter
		m_pCDataSource->CheckAndTrackOneDatabase( pwszCatalogNameRestr, NULL );
	}

	// Get total number of metas
	int iTotalMetas;
	if( FAILED( m_pSwstMetaHolder->GetMaxNumber( &iTotalMetas ) ) )
		return E_FAIL;
	
	// Collect metas to sort them by catalog name. Check restrictions
	CSwstMeta** ppSwstMeta = new CSwstMeta*[ iTotalMetas ? iTotalMetas : 1 ];
	if( ppSwstMeta == NULL )
		return E_OUTOFMEMORY;

	// Go though all metas and search for appropriate
	int iMeta = 0;
	CSwstMeta* pTemp;
	// Store pointers and check restrictions
	for( int j = 0; j < iTotalMetas; j++ )
		if( m_pSwstMetaHolder->At( j, &pTemp ) == S_OK &&
			(!bCatalogNameRestr || !wcsicmp( pwszCatalogNameRestr, pTemp->m_CatalogOwner.m_wszCatalog ) ) )
		{
			BOOL bIsThere = FALSE;

			// Catalogs may differ only by owner
			for( int k = 0; k < iMeta; k++ ) 
				if( !wcsicmp( pTemp->m_CatalogOwner.m_wszCatalog, ppSwstMeta[ k ]->m_CatalogOwner.m_wszCatalog ) )
				{
					bIsThere = TRUE;
					break;
				}
			
			if( !bIsThere )
			{
				ppSwstMeta[ iMeta ] = pTemp;
				iMeta++;
			}
		}

	// no more metas than iMeta
	iTotalMetas = iMeta; 

	// Sort metas by name
	qsort( ppSwstMeta, iTotalMetas, sizeof(ppSwstMeta[0]), CompareCatalogs );
	/*for ( j = iTotalMetas - 1; j > 0; j--)
	{
		for (int k = 0; k < j; k++)
		{
			if( wcsicmp(ppSwstMeta[k]->m_CatalogOwner.m_wszCatalog, ppSwstMeta[k + 1]->m_CatalogOwner.m_wszCatalog ) > 0 )
			{
				pTemp = ppSwstMeta[k];
				ppSwstMeta[k] = ppSwstMeta[ k + 1 ];
				ppSwstMeta[k + 1] = pTemp;
			}
		}
	}*/

	//Init data
	m_pCatalogInfos = new CATALOGINFO[ iTotalMetas ? iTotalMetas : 1];
	if( m_pCatalogInfos == NULL )
		return E_OUTOFMEMORY;

	//Copy catalogs information to buffer
	m_cRows = 0;
	for( j = 0; j < iTotalMetas; j++ )
	{
		// Get next meta
		CSwstMeta* pSwstMeta = ppSwstMeta[ j ];
		
		// Get catalog name
		wcscpy0( m_pCatalogInfos[ m_cRows ].m_wszCatalog, pSwstMeta->m_CatalogOwner.m_wszCatalog, MAXSTR(m_pCatalogInfos[ m_cRows ].m_wszCatalog) ); 

		// "" is catalog of current data source
		if( *m_pCatalogInfos[ m_cRows ].m_wszCatalog == L'\0' )
		{
			wcscpy( m_pCatalogInfos[ m_cRows ].m_wszDescription, L"Catalog of data source" );
			m_cRows++;
			continue;
		}

		// Get description
		// Create db by type
		EnumDatabase *pEnumDatabase = NULL;
		if( FAILED( EnumOfEnumsDataSources::GetNewObject( pSwstMeta->m_CatalogOwner.m_type, &pEnumDatabase ) ) )
			continue;

		// Save  all authentificational information
		DSNINFO dsnInfo;
		m_pCDataSource->GetAuthInfo( &dsnInfo );
		// Put pointer to MySQL ODBC coinnection string
		dsnInfo.m_pszMySqlStr = pSwstMeta->m_szMySqlStr;

		if( SUCCEEDED( pEnumDatabase->Init( &dsnInfo ) ) )
		{
			WideCharToMultiByte(CP_ACP, 0,
						m_pCatalogInfos[ m_cRows ].m_wszCatalog, -1,
						dsnInfo.m_szDataSource, MAXSTR(dsnInfo.m_szDataSource),
						NULL, NULL );
			
			// Get struct with description by catalog name
			if( SUCCEEDED( pEnumDatabase->PathByName( &dsnInfo ) ) &&
				MultiByteToWideChar(CP_ACP, MB_PRECOMPOSED, 
					dsnInfo.m_szDescription, -1, 
					m_pCatalogInfos[ m_cRows ].m_wszDescription, MAXSTR(m_pCatalogInfos[ m_cRows ].m_wszDescription) ) )
						m_cRows++;
			
			pEnumDatabase->Done();
		}

		delete pEnumDatabase; // it is not COM object
	}
					
	TRACE2( "  CSchema::InitCatalogs: completing (total %d)", m_cRows );
	return S_OK;
}
	

// Compare schemata
int __cdecl CompareSchemata(const void *left, const void *right ) 
{
	CatalogOwner* pLeft = *(CatalogOwner**)left;
	CatalogOwner* pRight = *(CatalogOwner**)right;

	int iCat = wcsicmp(pLeft->m_wszCatalog, pRight->m_wszCatalog );
	if( iCat )
		return iCat;
	
	int iSch = wcsicmp(pLeft->m_wszOwner, pRight->m_wszOwner );
	return iSch;
}


//Initialize object with schemata schema
HRESULT CSchema::InitSchemata
	(
	ULONG			cRestrictions,		//@parm IN	| Count of restriction values
	const VARIANT	rgRestrictions[]	//@parm IN	| Array of restriction values
	)
{
	TRACE( "CSchema::InitSchemata" );

	HRESULT ret = E_FAIL;
	
	//Set schema type
	m_enumSchema = SCHEMATA_SCHEMA;

	//Init columns
	static COLINFO Columns[] = 
	{
		{L"CATALOG_NAME",	DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"SCHEMA_NAME",	DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"SCHEMA_OWNER",	DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"DEFAULT_CHARACTER_SET_CATALOG",	DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"DEFAULT_CHARACTER_SET_SHEMA",	DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"DEFAULT_CHARACTER_SET_NAME",	DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL}
	};
	m_pColumns = Columns;
	m_cCols = 6;

	//Check restrictions
	if (cRestrictions > 3)
		return E_INVALIDARG;
	
	//extract all restrictions
	WCHAR* pwszCatalogNameRestr; 
	WCHAR* pwszSchemaNameRestr; 
	WCHAR* pwszSchemaOwnerRestr; 
	bool bCatalogNameRestr = false;
	bool bSchemaNameRestr = false;
	bool bSchemaOwnerRestr = false;

	//Only BSTR is accepted
	if ((cRestrictions >= 1 && rgRestrictions[0].vt != VT_BSTR && rgRestrictions[0].vt != VT_EMPTY) ||
		(cRestrictions >= 2 && rgRestrictions[1].vt != VT_BSTR && rgRestrictions[1].vt != VT_EMPTY) || 
		(cRestrictions >= 3 && rgRestrictions[2].vt != VT_BSTR && rgRestrictions[2].vt != VT_EMPTY))
	{
		return E_INVALIDARG;
	}

	//Extract CATALOG_NAME restriction
	if (cRestrictions >= 1 && rgRestrictions[0].vt == VT_BSTR)
	{
		pwszCatalogNameRestr = rgRestrictions[0].bstrVal ? rgRestrictions[0].bstrVal : L"";
		bCatalogNameRestr = true;
	}

	//Extract SHEMA_NAME restriction
	if (cRestrictions >= 2 && rgRestrictions[1].vt == VT_BSTR)
	{
		pwszSchemaNameRestr = rgRestrictions[1].bstrVal ? rgRestrictions[1].bstrVal : L"";
		if( !wcsicmp( pwszSchemaNameRestr, INFORMATION_SCHEMA ) )
			pwszSchemaNameRestr = L"";
		bSchemaNameRestr = true;
	}

	// Load catalog information if necessary
	if( (bCatalogNameRestr || bSchemaNameRestr) && m_pCDataSource )
	{
		// May fail, Doesn't matter
		m_pCDataSource->CheckAndTrackOneDatabase( pwszCatalogNameRestr, pwszSchemaNameRestr );
	}

	//Extract SHEMA_OWNER restriction
	if (cRestrictions >= 3 && rgRestrictions[2].vt == VT_BSTR)
	{
		pwszSchemaOwnerRestr = rgRestrictions[2].bstrVal ? rgRestrictions[2].bstrVal : L"";
		if( !wcsicmp( pwszSchemaOwnerRestr, INFORMATION_SCHEMA ) )
			pwszSchemaOwnerRestr = L"";
		bSchemaOwnerRestr = true;
	}

	// Get total number of metas
	CSwstMeta* pSwstMeta;
	int iTotalMetas;
	if( FAILED( m_pSwstMetaHolder->GetMaxNumber( &iTotalMetas ) ) )
		return E_FAIL;
	
	//Init data
	m_pSchemataInfos = new CatalogOwner*[ iTotalMetas ? iTotalMetas : 1];
	if( m_pSchemataInfos == NULL )
		return E_OUTOFMEMORY;

	//Copy file pointers to buffer. Check restrictions
	m_cRows = 0;
	for( int j = 0; j < iTotalMetas; j++ )
		if( m_pSwstMetaHolder->At( j, &pSwstMeta ) == S_OK &&
			(!bCatalogNameRestr || !wcsicmp( pwszCatalogNameRestr, pSwstMeta->m_CatalogOwner.m_wszCatalog ) ) &&
			(!bSchemaNameRestr || !wcsicmp( pwszSchemaNameRestr, pSwstMeta->m_CatalogOwner.m_wszOwner ) ) &&
			(!bSchemaOwnerRestr || !wcsicmp( pwszSchemaOwnerRestr, pSwstMeta->m_CatalogOwner.m_wszOwner ) ) )
				m_pSchemataInfos[ m_cRows++ ] = &pSwstMeta->m_CatalogOwner;
					
	//Sort file pointers by catalog + schema + owner
	qsort( m_pSchemataInfos, m_cRows, sizeof(m_pSchemataInfos[0]),  CompareSchemata );
	/*CatalogOwner* pTemp;
	for ( j = m_cRows - 1; j > 0; j--)
	{
		for (int k = 0; k < j; k++)
		{
			int iCat = wcsicmp(m_pSchemataInfos[k]->m_wszCatalog, m_pSchemataInfos[k + 1]->m_wszCatalog );
			int iSch = wcsicmp(m_pSchemataInfos[k]->m_wszOwner, m_pSchemataInfos[k + 1]->m_wszOwner );
			int iOwn = iSch;
			BOOL bIsGreatThan = iCat > 0 || ( iCat==0 && iSch > 0 ) || ( iCat==0 && iSch==0 && iOwn > 0);

			if (bIsGreatThan)
			{
				pTemp = m_pSchemataInfos[k];
				m_pSchemataInfos[k] = m_pSchemataInfos[k + 1];
				m_pSchemataInfos[k + 1] = pTemp;
			}
		}
	}*/

	TRACE2( "CSchema::InitSchemata: completing (total %d)", m_cRows );
	return S_OK;
}


// Compare statistics
int __cdecl CompareStatistics(const void *left, const void *right ) 
{
	STATINFO* pLeft = (STATINFO*)left;
	STATINFO* pRight = (STATINFO*)right;

	int iCat = wcsicmp(pLeft->pFile->m_pCatalogOwner->m_wszCatalog, pRight->pFile->m_pCatalogOwner->m_wszCatalog );
	if( iCat )
		return iCat;

	int iOwn = wcsicmp(pLeft->pFile->m_pCatalogOwner->m_wszOwner, pRight->pFile->m_pCatalogOwner->m_wszOwner );
	if( iOwn )
		return iOwn;
	
	int iNam = stricmp(pLeft->pFile->m_szName, pRight->pFile->m_szName);
	return iNam;
}


//Initialize object with columns schema
HRESULT CSchema::InitStatistics
	(
	ULONG			cRestrictions,		//@parm IN	| Count of restriction values
	const VARIANT	rgRestrictions[]	//@parm IN	| Array of restriction values
	)
{
	TRACE( "CSchema::InitStatistics" );

	m_enumSchema = STATISTICS_SCHEMA;

	//Init columns
	static COLINFO Columns[] = 
	{
		{L"TABLE_CATALOG",	DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"TABLE_SCHEMA",	DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"TABLE_NAME",		DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"CARDINALITY",	DBTYPE_UI8,		sizeof(__int64),	DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL}
	};
	m_pColumns = Columns;
	m_cCols = 4;

	//Check restrictions
	if (cRestrictions > 3)
	{
		TRACE2( "CSchema::InitStatistics: restrictions = %d", cRestrictions );
		return E_INVALIDARG;
	}

	WCHAR* pwszCatalogNameRestr = NULL; 
	WCHAR* pwszSchemaNameRestr = NULL; 
	char szTableNameRestr[MAX_NAME_SIZE]; 
	bool bCatalogNameRestr = false;
	bool bSchemaNameRestr = false;
	bool bTableNameRestr = false;

	//Only BSTR is accepted
	if ((cRestrictions >= 1 && rgRestrictions[0].vt != VT_BSTR && rgRestrictions[0].vt != VT_EMPTY) ||
		(cRestrictions >= 2 && rgRestrictions[1].vt != VT_BSTR && rgRestrictions[1].vt != VT_EMPTY) || 
		(cRestrictions >= 3 && rgRestrictions[2].vt != VT_BSTR && rgRestrictions[2].vt != VT_EMPTY) )
	{
		TRACE( "CSchema::InitStatistics: restr fails" );
		return E_INVALIDARG;
	}

	//Extract CATALOG_NAME restriction
	if (cRestrictions >= 1 && rgRestrictions[0].vt == VT_BSTR)
	{
		pwszCatalogNameRestr = rgRestrictions[0].bstrVal ? rgRestrictions[0].bstrVal : L"";
		bCatalogNameRestr = true;
		TRACE2( "  Catalog [%S]", pwszCatalogNameRestr );
	}

	//Extract SCHEMA_NAME restriction
	if (cRestrictions >= 2 && rgRestrictions[1].vt == VT_BSTR)
	{
		pwszSchemaNameRestr = rgRestrictions[1].bstrVal ? rgRestrictions[1].bstrVal : L"";;
		if( !wcsicmp( pwszSchemaNameRestr, INFORMATION_SCHEMA ) )
			pwszSchemaNameRestr = L"";
		bSchemaNameRestr = true;
		TRACE2( "  Schema [%S]", pwszSchemaNameRestr );
	}

	// Load catalog information if necessary
	if( (bCatalogNameRestr || bSchemaNameRestr) && m_pCDataSource )
	{
		// May fail, Doesn't matter
		m_pCDataSource->CheckAndTrackOneDatabase( pwszCatalogNameRestr, pwszSchemaNameRestr );
	}

	//Extract TABLE_NAME restriction
	if (cRestrictions >= 3 && rgRestrictions[2].vt == VT_BSTR)
	{
		if( rgRestrictions[2].bstrVal == NULL )
			*szTableNameRestr = '\0';
		else
			WideCharToMultiByte(CP_ACP, 0,
							rgRestrictions[2].bstrVal, -1,
							szTableNameRestr, MAX_NAME_SIZE,
							NULL, NULL );
		bTableNameRestr = true;
		TRACE2( "  Table [%s]", szTableNameRestr );
	}

	TRACE( "CSchema::InitStatistics: before getting data" );

	// Get total number of files
	int uiTotalFiles = 0;
	int iTotalMetas;
	CSwstMeta* pSwstMeta;
	if( FAILED( m_pSwstMetaHolder->GetMaxNumber( &iTotalMetas ) ) )
		return E_FAIL;
	for( int j = 0; j < iTotalMetas; j++ )
		if( m_pSwstMetaHolder->At( j, &pSwstMeta ) == S_OK &&
			(!bCatalogNameRestr || !wcsicmp( pwszCatalogNameRestr, pSwstMeta->m_CatalogOwner.m_wszCatalog ) ) &&
			(!bSchemaNameRestr || !wcsicmp( pwszSchemaNameRestr, pSwstMeta->m_CatalogOwner.m_wszOwner ) ) )
				uiTotalFiles += pSwstMeta->m_siFiles;
	
	//Init data
	m_pStatistics = new STATINFO[ uiTotalFiles ? uiTotalFiles : 1];
	if( m_pStatistics == NULL )
		return E_OUTOFMEMORY;

	//Copy file pointers to buffer
	m_cRows = 0;
	for( j = 0; j < iTotalMetas; j++ )
		if( m_pSwstMetaHolder->At( j, &pSwstMeta ) == S_OK &&
		   (!bCatalogNameRestr || !wcsicmp( pwszCatalogNameRestr, pSwstMeta->m_CatalogOwner.m_wszCatalog ) ) &&
		   (!bSchemaNameRestr || !wcsicmp( pwszSchemaNameRestr, pSwstMeta->m_CatalogOwner.m_wszOwner ) ) )
		{
				for (int i = 0; i < pSwstMeta->m_siFiles; i++)
				{
					// Check TABLE_NAME restriction
					if (bTableNameRestr)
					{
						if (0 != stricmp(pSwstMeta->m_pFiles[i].m_szName, szTableNameRestr))
							continue;
					}

					//Copy file pointer
					m_pStatistics[m_cRows].pFile = &pSwstMeta->m_pFiles[i];
					
					// Calculate number of rows
					HRESULT hr;

					//if( !m_pCDataSource->m_pUtilProp->MySqlMode() )
					switch( m_pCDataSource->m_pUtilProp->InternalSqlSupport() )
					{
					case ISS_WRONG:
						return E_UNEXPECTED;

					case ISS_MYSQL:
						{
						// 1) Create MySql object
						CMySql* pMySql = new CMySql();
						if( pMySql == NULL )
							return E_OUTOFMEMORY;
						
						// 2) Init  object with "select count(*)" query
						DWORD dummy;
						char szQuery[ 128 ];
						strcpy( szQuery, "select count(*) from " );
						strcat( szQuery, pSwstMeta->m_pFiles[i].m_szName );
						
						hr = pMySql->Init( m_pCDataSource, pSwstMeta->m_CatalogOwner.m_wszCatalog, szQuery, &dummy );
						if( FAILED( hr ) )
						{
							delete pMySql;
							continue;
						}

						// 3) Get first a value from first column of first row
						hr = pMySql->MoveFirst();
						if( FAILED( hr ) )
						{
							delete pMySql;
							continue;
						}

						ULONG offsets[2] = {0, 0};
						BYTE byteBuffer[ sizeof(COLUMNDATA)+sizeof(DWORD) ];
						PCOLUMNDATA colData = (PCOLUMNDATA) byteBuffer;

						hr = pMySql->GetRow( offsets, byteBuffer );
						if( FAILED( hr ) || colData->dwStatus == DBSTATUS_S_ISNULL )
							m_pStatistics[m_cRows].dwRows = -1;
						else
							m_pStatistics[m_cRows].dwRows = *(DWORD*)(colData->bData);

						// 4) Delete CData object
						delete pMySql;

						break;
						}
					}

					// Completed with this row. Go to next one.
					m_cRows++;
				}
		}
	
	//Sort statistics by file name
	qsort( m_pStatistics, m_cRows, sizeof(m_pStatistics[0]), CompareStatistics );
	/*STATINFO stat;
	for (j = m_cRows - 1; j > 0; j--)
	{
		for (int k = 0; k < j; k++)
		{
			int iCat = wcsicmp(m_pStatistics[k].pFile->m_pCatalogOwner->m_wszCatalog, m_pStatistics[k + 1].pFile->m_pCatalogOwner->m_wszCatalog );
			int iOwn = wcsicmp(m_pStatistics[k].pFile->m_pCatalogOwner->m_wszOwner, m_pStatistics[k + 1].pFile->m_pCatalogOwner->m_wszOwner );
			int iNam = stricmp(m_pStatistics[k].pFile->m_szName, m_pStatistics[k + 1].pFile->m_szName);
				
			if( iCat > 0 || ( iCat==0 && iOwn > 0 ) || ( iCat==0 && iOwn==0 && iNam > 0) )
			{
				stat = m_pStatistics[k];
				m_pStatistics[k] = m_pStatistics[k + 1];
				m_pStatistics[k + 1] = stat;
			}
		}
	}*/

	TRACE2( "CSchema::InitStatistics: completing (total %d)", m_cRows );
	return S_OK;
}

// Compare indexes
int __cdecl CompareIndexes(const void *left, const void *right ) 
{
	SWSTINDEX* pLeft = *(SWSTINDEX**)left;
	SWSTINDEX* pRight = *(SWSTINDEX**)right;

	if( pRight->IsUnique() && !pLeft->IsUnique() )
		return 1;
	if( !pRight->IsUnique() && pLeft->IsUnique() )
		return -1;
	
	int iCat = wcsicmp(pLeft->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszCatalog, 
					   pRight->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszCatalog );
	if( iCat )
		return iCat;
	
	int iOwn = wcsicmp(pLeft->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszOwner, 
					   pRight->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszOwner );
	if( iOwn )
		return iOwn;
	
	int iNam = CompareIndexNames( pLeft, pRight );
	if( iNam )
		return iNam;

	int iTab = stricmp( pLeft->m_pSwstField->m_pSwstFile->m_szName, 
					   pRight->m_pSwstField->m_pSwstFile->m_szName );
	if( iTab )
		return iTab;
	
	int iOrd = pLeft->m_wPart - pRight->m_wPart;
	return iOrd;
}


//Initialize object with columns schema
HRESULT CSchema::InitIndexes
	(
	ULONG			cRestrictions,		//@parm IN	| Count of restriction values
	const VARIANT	rgRestrictions[]	//@parm IN	| Array of restriction values
	)
{
	TRACE( "CSchema::InitIndexes" );

	m_enumSchema = INDEXES_SCHEMA;

	//Init columns
	static COLINFO Columns[] = 
	{
		{L"TABLE_CATALOG",	DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"TABLE_SCHEMA",	DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"TABLE_NAME",		DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"INDEX_CATALOG",	DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"INDEX_SCHEMA",	DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"INDEX_NAME",		DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"PRIMARY_KEY",	DBTYPE_BOOL,	sizeof(VARIANT_BOOL),	DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"UNIQUE",			DBTYPE_BOOL,	sizeof(VARIANT_BOOL),	DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"CLUSTERED",		DBTYPE_BOOL,	sizeof(VARIANT_BOOL),	DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"TYPE",			DBTYPE_UI2,		sizeof(WORD),	DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"FILL_FACTOR",	DBTYPE_I4,		sizeof(LONG),	DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"INITIAL_SIZE",	DBTYPE_I4,		sizeof(LONG),	DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"NULLS",			DBTYPE_I4,		sizeof(LONG),	DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"SORT_BOOKMARKS",	DBTYPE_BOOL,	sizeof(VARIANT_BOOL),	DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"AUTO_UPDATE",	DBTYPE_BOOL,	sizeof(VARIANT_BOOL),	DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"NULL_COLLATION",	DBTYPE_I4,		sizeof(LONG),	DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"ORDINAL_POSITION",DBTYPE_UI4,	sizeof(ULONG),	DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"COLUMN_NAME",	DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"COLUMN_GUID",	DBTYPE_GUID,	sizeof(GUID),	DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"COLUMN_PROPID",	DBTYPE_UI4,		sizeof(ULONG),	DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"COLLATION",		DBTYPE_I2,		sizeof(SHORT),	DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"CARDINALITY",	DBTYPE_UI8,		sizeof(__int64),DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"PAGES",			DBTYPE_I4,		sizeof(LONG),	DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"FILTER_CONDITION",DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"INTEGRATED",		DBTYPE_BOOL,	sizeof(VARIANT_BOOL),	DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL}
	};
	m_pColumns = Columns;
	m_cCols = 25;

	//Check restrictions
	if (cRestrictions > 5)
	{
		TRACE2( "CSchema::InitIndexes: restrictions = %d", cRestrictions );
		return E_INVALIDARG;
	}

	WCHAR* pwszCatalogNameRestr = NULL; 
	WCHAR* pwszSchemaNameRestr = NULL; 
	char szTableNameRestr[MAX_NAME_SIZE]; 
	char szIndexNameRestr[MAX_NAME_SIZE]; 
	int  iIndexNameRestr = -1;
	WORD wIndexTypeRestr = 0;
	bool bCatalogNameRestr = false;
	bool bSchemaNameRestr = false;
	bool bTableNameRestr = false;
	bool bIndexNameRestr = false;
	bool bIndexTypeRestr = false;

	//Only BSTR is accepted
	if ((cRestrictions >= 1 && rgRestrictions[0].vt != VT_BSTR && rgRestrictions[0].vt != VT_EMPTY) ||
		(cRestrictions >= 2 && rgRestrictions[1].vt != VT_BSTR && rgRestrictions[1].vt != VT_EMPTY) || 
		(cRestrictions >= 3 && rgRestrictions[2].vt != VT_BSTR && rgRestrictions[2].vt != VT_EMPTY) ||
		(cRestrictions >= 4 && rgRestrictions[3].vt != VT_BSTR && rgRestrictions[3].vt != VT_EMPTY && rgRestrictions[3].vt != VT_UI4 && rgRestrictions[3].vt != VT_UI2 && rgRestrictions[3].vt != VT_I2 && rgRestrictions[3].vt != VT_I4 ) ||
		(cRestrictions >= 5 && rgRestrictions[4].vt != VT_BSTR && rgRestrictions[4].vt != VT_EMPTY) )
	{
		TRACE( "CSchema::InitIndexes: restr fails" );
		return E_INVALIDARG;
	}

	//Extract CATALOG_NAME restriction
	if (cRestrictions >= 1 && rgRestrictions[0].vt == VT_BSTR)
	{
		pwszCatalogNameRestr = rgRestrictions[0].bstrVal ? rgRestrictions[0].bstrVal : L"";
		bCatalogNameRestr = true;
		TRACE2( "  Catalog [%S]", pwszCatalogNameRestr );
	}

	//Extract SCHEMA_NAME restriction
	if (cRestrictions >= 2 && rgRestrictions[1].vt == VT_BSTR)
	{
		pwszSchemaNameRestr = rgRestrictions[1].bstrVal ? rgRestrictions[1].bstrVal : L"";;
		if( !wcsicmp( pwszSchemaNameRestr, INFORMATION_SCHEMA ) )
			pwszSchemaNameRestr = L"";
		bSchemaNameRestr = true;
		TRACE2( "  Schema [%S]", pwszSchemaNameRestr );
	}

	// Load catalog information if necessary
	if( (bCatalogNameRestr || bSchemaNameRestr) && m_pCDataSource )
	{
		// May fail, Doesn't matter
		m_pCDataSource->CheckAndTrackOneDatabase( pwszCatalogNameRestr, pwszSchemaNameRestr );
	}

	//Extract INDEX_NAME restriction
	if (cRestrictions >= 3 && rgRestrictions[2].vt == VT_BSTR)
	{
		if( rgRestrictions[2].bstrVal == NULL )
			*szIndexNameRestr = '\0';
		else
		{
			WideCharToMultiByte(CP_ACP, 0,
							rgRestrictions[2].bstrVal, -1,
							szIndexNameRestr, MAX_NAME_SIZE,
							NULL, NULL );
			if( *szIndexNameRestr == '#' )
				iIndexNameRestr = atoi( szIndexNameRestr + 1 );
		}
		bIndexNameRestr = true;
		TRACE2( "  Index [%s]", szIndexNameRestr );
	}

	// Extract TYPE restriction
	if (cRestrictions >= 4 && rgRestrictions[3].vt != VT_EMPTY )
	{
		switch (rgRestrictions[3].vt)
		{
		case VT_BSTR:
			wIndexTypeRestr = _wtoi( rgRestrictions[3].bstrVal );
			break;
		
		case VT_UI2:
			wIndexTypeRestr = rgRestrictions[3].uiVal;
			break;

		case VT_UI4:
			wIndexTypeRestr = rgRestrictions[3].ulVal;
			break;

		case VT_I2:
			wIndexTypeRestr = rgRestrictions[3].iVal;
			break;

		case VT_I4:
			wIndexTypeRestr = rgRestrictions[3].lVal;
			break;
		}
		bIndexTypeRestr = true;
		TRACE2( "  Index [%d]", wIndexTypeRestr );
	}

	//Extract TABLE_NAME restriction
	if (cRestrictions >= 5 && rgRestrictions[4].vt == VT_BSTR)
	{
		if( rgRestrictions[4].bstrVal == NULL )
			*szTableNameRestr = '\0';
		else
			WideCharToMultiByte(CP_ACP, 0,
							rgRestrictions[4].bstrVal, -1,
							szTableNameRestr, MAX_NAME_SIZE,
							NULL, NULL );
		bTableNameRestr = true;
		TRACE2( "  Table [%s]", szTableNameRestr );
	}

	TRACE( "CSchema::InitStatistics: before getting data" );

	// Get total number of indexes
	int iTotalIndexes = 0;
	int iTotalMetas;
	CSwstMeta* pSwstMeta;
	if( FAILED( m_pSwstMetaHolder->GetMaxNumber( &iTotalMetas ) ) )
		return E_FAIL;
    if( !bIndexTypeRestr || wIndexTypeRestr == DBPROPVAL_IT_BTREE )
		for( int j = 0; j < iTotalMetas; j++ )
			if( m_pSwstMetaHolder->At( j, &pSwstMeta ) == S_OK &&
				(!bCatalogNameRestr || !wcsicmp( pwszCatalogNameRestr, pSwstMeta->m_CatalogOwner.m_wszCatalog ) ) &&
				(!bSchemaNameRestr || !wcsicmp( pwszSchemaNameRestr, pSwstMeta->m_CatalogOwner.m_wszOwner ) ) )
					iTotalIndexes += pSwstMeta->m_siIndexes;
	
	//Init data
	m_ppIndexes = new SWSTINDEX*[ iTotalIndexes ? iTotalIndexes : 1];
	if( m_ppIndexes == NULL )
		return E_OUTOFMEMORY;

	//Copy index pointers to buffer
	m_cRows = 0;
    if( !bIndexTypeRestr || wIndexTypeRestr == DBPROPVAL_IT_BTREE )
		for( int j = 0; j < iTotalMetas; j++ )
			if( m_pSwstMetaHolder->At( j, &pSwstMeta ) == S_OK &&
			   (!bCatalogNameRestr || !wcsicmp( pwszCatalogNameRestr, pSwstMeta->m_CatalogOwner.m_wszCatalog ) ) &&
			   (!bSchemaNameRestr || !wcsicmp( pwszSchemaNameRestr, pSwstMeta->m_CatalogOwner.m_wszOwner ) ) )
			{
					for (int i = 0; i < pSwstMeta->m_siIndexes; i++)
					{
						// Check INDEX_NAME restriction
						if (bIndexNameRestr)
						{
							if( !PassFilter( pSwstMeta->m_pIndexes + i, szIndexNameRestr, iIndexNameRestr ) )
								continue;
						}

						// Check TABLE_NAME restriction
						if (bTableNameRestr)
						{
							if( stricmp(pSwstMeta->m_pIndexes[i].m_pSwstField->m_pSwstFile->m_szName, szTableNameRestr) )
								continue;
						}

						//Copy file pointer
						m_ppIndexes[m_cRows] = &pSwstMeta->m_pIndexes[i];
						
						// Completed with this row. Go to next one.
						m_cRows++;
					}
			}
	
	//Sort statistics by UNIQUE | (TYPE) | INDEX_CATALOG | INDEX_SCHEMA | INDEX_NAME | ORDINAL_POSITION
	qsort( m_ppIndexes, m_cRows, sizeof(m_ppIndexes[0]), CompareIndexes );
	/*SWSTINDEX* pIndex;
	for (int j = m_cRows - 1; j > 0; j--)
	{
		for (int k = 0; k < j; k++)
		{
			int iUnq = 0;
			if( m_ppIndexes[k + 1]->IsUnique() && !m_ppIndexes[k]->IsUnique() )
				iUnq = 1;
			else if( !m_ppIndexes[k + 1]->IsUnique() && m_ppIndexes[k]->IsUnique() )
				iUnq = -1;
			int iCat = wcsicmp(m_ppIndexes[k]->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszCatalog, 
							   m_ppIndexes[k + 1]->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszCatalog );
			int iOwn = wcsicmp(m_ppIndexes[k]->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszOwner, 
							   m_ppIndexes[k + 1]->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszOwner );
			int iNam = CompareIndexNames( m_ppIndexes[k], m_ppIndexes[k + 1] );
			int iTab = stricmp( m_ppIndexes[k]->m_pSwstField->m_pSwstFile->m_szName, 
							   m_ppIndexes[k + 1]->m_pSwstField->m_pSwstFile->m_szName );
			int iOrd = m_ppIndexes[k]->m_wPart - m_ppIndexes[k + 1]->m_wPart;
				
			if( ( iUnq > 0 ) ||
				( iUnq==0 && iCat > 0 ) || 
				( iUnq==0 && iCat==0 && iOwn > 0 ) || 
				( iUnq==0 && iCat==0 && iOwn==0 && iNam > 0) || 
				( iUnq==0 && iCat==0 && iOwn==0 && iNam==0 && iTab > 0 ) || 
				( iUnq==0 && iCat==0 && iOwn==0 && iNam==0 && iTab==0 && iOrd > 0) )
			{
				pIndex = m_ppIndexes[k];
				m_ppIndexes[k] = m_ppIndexes[k + 1];
				m_ppIndexes[k + 1] = pIndex;
			}
		}
	}*/

	TRACE2( "CSchema::InitIndexes: completing (total %d)", m_cRows );
	return S_OK;
}


// Compare constraint column usage
int __cdecl CompareConstraintColumns(const void *left, const void *right ) 
{
	SWSTINDEX* pLeft = *(SWSTINDEX**)left;
	SWSTINDEX* pRight = *(SWSTINDEX**)right;

	int iCat = wcsicmp(pLeft->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszCatalog, 
					   pRight->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszCatalog );
	if( iCat )
		return iCat;

	int iOwn = wcsicmp(pLeft->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszOwner, 
					   pRight->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszOwner );
	if( iOwn )
		return iOwn;

	int iTab = stricmp( pLeft->m_pSwstField->m_pSwstFile->m_szName, 
					   pRight->m_pSwstField->m_pSwstFile->m_szName );
	if( iTab )
		return iTab;

	int iCol = stricmp( pLeft->m_pSwstField->m_szName, 
					   pRight->m_pSwstField->m_szName );	 
	if( iCol )
		return iCol;

	int iNam = CompareIndexNames( pLeft, pRight );
	return iNam;
}


//Initialize object with columns schema
HRESULT CSchema::InitConstraintColumns
	(
	ULONG			cRestrictions,		//@parm IN	| Count of restriction values
	const VARIANT	rgRestrictions[]	//@parm IN	| Array of restriction values
	)
{
	TRACE( "CSchema::InitConstraintColumns" );

	m_enumSchema = CONSTRAINT_COLUMN_USAGE_SCHEMA;

	//Init columns
	static COLINFO Columns[] = 
	{
		{L"TABLE_CATALOG",	DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"TABLE_SCHEMA",	DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"TABLE_NAME",		DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"COLUMN_NAME",	DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"COLUMN_GUID",	DBTYPE_GUID,	sizeof(GUID),	DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"COLUMN_PROPID",	DBTYPE_UI4,		sizeof(ULONG),	DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"CONSTRAINT_CATALOG",	DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"CONSTRAINT_SCHEMA",	DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"CONSTRAINT_NAME",	DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL}
	};
	m_pColumns = Columns;
	m_cCols = 9;

	//Check restrictions
	if (cRestrictions > 7)
	{
		TRACE2( "CSchema::InitConstraintColumns: restrictions = %d", cRestrictions );
		return E_INVALIDARG;
	}

	WCHAR* pwszCatalogTableNameRestr = NULL; 
	WCHAR* pwszCatalogRestrNameRestr = NULL; 
	WCHAR* pwszSchemaTableNameRestr = NULL; 
	WCHAR* pwszSchemaRestrNameRestr = NULL; 
	char szTableNameRestr[MAX_NAME_SIZE]; 
	char szColumnNameRestr[MAX_NAME_SIZE]; 
	char szRestrNameRestr[MAX_NAME_SIZE]; 
	int  iRestrNameRestr = -1;
	
	bool bCatalogTableNameRestr = false;
	bool bCatalogRestrNameRestr = false;
	bool bSchemaTableNameRestr = false;
	bool bSchemaRestrNameRestr = false;
	bool bTableNameRestr = false;
	bool bColumnNameRestr = false;
	bool bRestrNameRestr = false;

	//Only BSTR is accepted
	if ((cRestrictions >= 1 && rgRestrictions[0].vt != VT_BSTR && rgRestrictions[0].vt != VT_EMPTY) ||
		(cRestrictions >= 2 && rgRestrictions[1].vt != VT_BSTR && rgRestrictions[1].vt != VT_EMPTY) || 
		(cRestrictions >= 3 && rgRestrictions[2].vt != VT_BSTR && rgRestrictions[2].vt != VT_EMPTY) ||
		(cRestrictions >= 4 && rgRestrictions[3].vt != VT_BSTR && rgRestrictions[3].vt != VT_EMPTY) ||
		(cRestrictions >= 5 && rgRestrictions[4].vt != VT_BSTR && rgRestrictions[4].vt != VT_EMPTY) ||
		(cRestrictions >= 6 && rgRestrictions[5].vt != VT_BSTR && rgRestrictions[5].vt != VT_EMPTY) ||
		(cRestrictions >= 7 && rgRestrictions[6].vt != VT_BSTR && rgRestrictions[6].vt != VT_EMPTY) )
	{
		TRACE( "CSchema::InitConstraintColumns: restr fails" );
		return E_INVALIDARG;
	}

	//Extract TABLE_CATALOG restriction
	if (cRestrictions >= 1 && rgRestrictions[0].vt == VT_BSTR)
	{
		pwszCatalogTableNameRestr = rgRestrictions[0].bstrVal ? rgRestrictions[0].bstrVal : L"";
		bCatalogTableNameRestr = true;
		TRACE2( "  Catalog of table [%S]", pwszCatalogTableNameRestr );
	}

	//Extract CONSTRAINT_CATALOG restriction
	if (cRestrictions >= 5 && rgRestrictions[4].vt == VT_BSTR)
	{
		pwszCatalogRestrNameRestr = rgRestrictions[4].bstrVal ? rgRestrictions[4].bstrVal : L"";
		bCatalogRestrNameRestr = true;
		TRACE2( "  Catalog of restr [%S]", pwszCatalogRestrNameRestr );
	}

	//Extract TABLE_SCHEMA restriction
	if (cRestrictions >= 2 && rgRestrictions[1].vt == VT_BSTR)
	{
		pwszSchemaTableNameRestr = rgRestrictions[1].bstrVal ? rgRestrictions[1].bstrVal : L"";;
		if( !wcsicmp( pwszSchemaTableNameRestr, INFORMATION_SCHEMA ) )
			pwszSchemaTableNameRestr = L"";
		bSchemaTableNameRestr = true;
		TRACE2( "  Schema of table [%S]", pwszSchemaTableNameRestr );
	}

	//Extract CONSTRAINT_SCHEMA restriction
	if (cRestrictions >= 6 && rgRestrictions[5].vt == VT_BSTR)
	{
		pwszSchemaRestrNameRestr = rgRestrictions[5].bstrVal ? rgRestrictions[5].bstrVal : L"";;
		if( !wcsicmp( pwszSchemaRestrNameRestr, INFORMATION_SCHEMA ) )
			pwszSchemaRestrNameRestr = L"";
		bSchemaRestrNameRestr = true;
		TRACE2( "  Schema of restr [%S]", pwszSchemaRestrNameRestr );
	}

	// Load catalog information if necessary
	if( (bCatalogTableNameRestr || bSchemaTableNameRestr) && m_pCDataSource )
		m_pCDataSource->CheckAndTrackOneDatabase( pwszCatalogTableNameRestr, pwszSchemaTableNameRestr);
	if( (bCatalogRestrNameRestr || bSchemaRestrNameRestr) && m_pCDataSource )
		m_pCDataSource->CheckAndTrackOneDatabase( pwszCatalogRestrNameRestr, pwszSchemaRestrNameRestr);

	//Extract TABLE_NAME restriction
	if (cRestrictions >= 3 && rgRestrictions[2].vt == VT_BSTR)
	{
		if( rgRestrictions[2].bstrVal == NULL )
			*szTableNameRestr = '\0';
		else
			WideCharToMultiByte(CP_ACP, 0,
							rgRestrictions[2].bstrVal, -1,
							szTableNameRestr, MAX_NAME_SIZE,
							NULL, NULL );
		bTableNameRestr = true;
		TRACE2( "  Table [%s]", szTableNameRestr );
	}

	//Extract COLUMN_NAME restriction
	if (cRestrictions >= 4 && rgRestrictions[3].vt == VT_BSTR)
	{
		if( rgRestrictions[3].bstrVal == NULL )
			*szColumnNameRestr = '\0';
		else
			WideCharToMultiByte(CP_ACP, 0,
							rgRestrictions[3].bstrVal, -1,
							szColumnNameRestr, MAX_NAME_SIZE,
							NULL, NULL );
		bColumnNameRestr = true;
		TRACE2( "  Column [%s]", szColumnNameRestr );
	}

	//Extract CONSTRAINT_NAME restriction
	if (cRestrictions >= 7 && rgRestrictions[6].vt == VT_BSTR)
	{
		if( rgRestrictions[6].bstrVal == NULL )
			*szRestrNameRestr = '\0';
		else
			WideCharToMultiByte(CP_ACP, 0,
							rgRestrictions[6].bstrVal, -1,
							szRestrNameRestr, MAX_NAME_SIZE,
							NULL, NULL );
		if( *szRestrNameRestr == '#' )
			iRestrNameRestr = atoi( szRestrNameRestr + 1 );
		
		bRestrNameRestr = true;
		TRACE2( "  Constraint [%s]", szRestrNameRestr );
	}
	
	TRACE( "CSchema::InitConstraintColumns: before getting data" );

	// Get total number of indexes
	int iTotalIndexes = 0;
	int iTotalMetas;
	CSwstMeta* pSwstMeta;
	if( FAILED( m_pSwstMetaHolder->GetMaxNumber( &iTotalMetas ) ) )
		return E_FAIL;
	for( int j = 0; j < iTotalMetas; j++ )
		if( m_pSwstMetaHolder->At( j, &pSwstMeta ) == S_OK &&
			(!bCatalogTableNameRestr || !wcsicmp( pwszCatalogTableNameRestr, pSwstMeta->m_CatalogOwner.m_wszCatalog ) ) &&
			(!bSchemaTableNameRestr  || !wcsicmp( pwszSchemaTableNameRestr, pSwstMeta->m_CatalogOwner.m_wszOwner ) ) &&
			(!bCatalogRestrNameRestr || !wcsicmp( pwszCatalogRestrNameRestr, pSwstMeta->m_CatalogOwner.m_wszCatalog ) ) &&
			(!bSchemaRestrNameRestr  || !wcsicmp( pwszSchemaRestrNameRestr, pSwstMeta->m_CatalogOwner.m_wszOwner ) ) )
				iTotalIndexes += pSwstMeta->m_siIndexes;
	
	//Init data
	m_ppIndexes = new SWSTINDEX*[ iTotalIndexes ? iTotalIndexes : 1];
	if( m_ppIndexes == NULL )
		return E_OUTOFMEMORY;

	//Copy index pointers to buffer
	m_cRows = 0;
	for( j = 0; j < iTotalMetas; j++ )
		if( m_pSwstMetaHolder->At( j, &pSwstMeta ) == S_OK &&
			(!bCatalogTableNameRestr || !wcsicmp( pwszCatalogTableNameRestr, pSwstMeta->m_CatalogOwner.m_wszCatalog ) ) &&
			(!bSchemaTableNameRestr  || !wcsicmp( pwszSchemaTableNameRestr, pSwstMeta->m_CatalogOwner.m_wszOwner ) ) &&
			(!bCatalogRestrNameRestr || !wcsicmp( pwszCatalogRestrNameRestr, pSwstMeta->m_CatalogOwner.m_wszCatalog ) ) &&
			(!bSchemaRestrNameRestr  || !wcsicmp( pwszSchemaRestrNameRestr, pSwstMeta->m_CatalogOwner.m_wszOwner ) ) )
		{
				for (int i = 0; i < pSwstMeta->m_siIndexes; i++)
				{
					// Check CONSTRAINT_NAME restriction
					if (bRestrNameRestr)
					{
						if( !PassFilter( pSwstMeta->m_pIndexes + i, szRestrNameRestr, iRestrNameRestr ) )
							continue;
					}

					// Check TABLE_NAME restriction
					if (bTableNameRestr)
					{
						if( stricmp(pSwstMeta->m_pIndexes[i].m_pSwstField->m_pSwstFile->m_szName, szTableNameRestr) )
							continue;
					}

					// Check COLUMN_NAME restriction
					if (bColumnNameRestr)
					{
						if( stricmp(pSwstMeta->m_pIndexes[i].m_pSwstField->m_szName, szColumnNameRestr) )
							continue;
					}

					// Only unique constraints and foreign keys
					if( !pSwstMeta->m_pIndexes[i].IsForeignKey() &&
						!pSwstMeta->m_pIndexes[i].IsUnique() )
							continue;

					//Copy file pointer
					m_ppIndexes[m_cRows] = &pSwstMeta->m_pIndexes[i];
					
					// Completed with this row. Go to next one.
					m_cRows++;
				}
		}
	
	//Default Sort Order: TABLE_CATALOG, TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, COLUMN_GUID, COLUMN_PROPID, CONSTRAINT_CATALOG, CONSTRAINT_SCHEMA, CONSTRAINT_NAME
	qsort( m_ppIndexes, m_cRows, sizeof(m_ppIndexes[0]), CompareConstraintColumns );
	/*SWSTINDEX* pIndex;
	for (j = m_cRows - 1; j > 0; j--)
	{
		for (int k = 0; k < j; k++)
		{
			int iCat = wcsicmp(m_ppIndexes[k]->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszCatalog, 
							   m_ppIndexes[k + 1]->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszCatalog );
			int iOwn = wcsicmp(m_ppIndexes[k]->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszOwner, 
							   m_ppIndexes[k + 1]->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszOwner );
			int iTab = stricmp( m_ppIndexes[k]->m_pSwstField->m_pSwstFile->m_szName, 
							   m_ppIndexes[k + 1]->m_pSwstField->m_pSwstFile->m_szName );
			int iCol = stricmp( m_ppIndexes[k]->m_pSwstField->m_szName, 
							   m_ppIndexes[k + 1]->m_pSwstField->m_szName );	 
			int iNam = CompareIndexNames( m_ppIndexes[k], m_ppIndexes[k + 1] );
				
			if( ( iCat > 0 ) ||
				( iCat==0 && iOwn > 0 ) || 
				( iCat==0 && iOwn==0 && iTab > 0 ) || 
				( iCat==0 && iOwn==0 && iTab==0 && iCol > 0) || 
				( iCat==0 && iOwn==0 && iTab==0 && iCol==0 && iNam > 0 ) )
			{
				pIndex = m_ppIndexes[k];
				m_ppIndexes[k] = m_ppIndexes[k + 1];
				m_ppIndexes[k + 1] = pIndex;
			}
		}
	}*/

	TRACE2( "CSchema::InitConstraintColumns: completing (total %d)", m_cRows );
	return S_OK;
}


// Compare constraint table usage
int __cdecl CompareConstraintTables(const void *left, const void *right ) 
{
	SWSTINDEX* pLeft = *(SWSTINDEX**)left;
	SWSTINDEX* pRight = *(SWSTINDEX**)right;

	int iCat = wcsicmp(pLeft->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszCatalog, 
					   pRight->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszCatalog );
	if( iCat )
		return iCat;

	int iOwn = wcsicmp(pLeft->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszOwner, 
					   pRight->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszOwner );
	if( iOwn )
		return iOwn;

	int iTab = stricmp( pLeft->m_pSwstField->m_pSwstFile->m_szName, 
					   pRight->m_pSwstField->m_pSwstFile->m_szName );
	if( iTab )
		return iTab;
	
	int iNam = CompareIndexNames( pLeft, pRight );
	return iNam;	
}


//Initialize object with columns schema
HRESULT CSchema::InitConstraintTables
	(
	ULONG			cRestrictions,		//@parm IN	| Count of restriction values
	const VARIANT	rgRestrictions[]	//@parm IN	| Array of restriction values
	)
{
	TRACE( "CSchema::InitConstraintTables" );

	m_enumSchema = CONSTRAINT_TABLE_USAGE_SCHEMA;

	//Init columns
	static COLINFO Columns[] = 
	{
		{L"TABLE_CATALOG",	DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"TABLE_SCHEMA",	DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"TABLE_NAME",		DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"CONSTRAINT_CATALOG",	DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"CONSTRAINT_SCHEMA",	DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"CONSTRAINT_NAME",	DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL}
	};
	m_pColumns = Columns;
	m_cCols = 6;

	//Check restrictions
	if (cRestrictions > 6)
	{
		TRACE2( "CSchema::InitConstraintTables: restrictions = %d", cRestrictions );
		return E_INVALIDARG;
	}

	WCHAR* pwszCatalogTableNameRestr = NULL; 
	WCHAR* pwszCatalogRestrNameRestr = NULL; 
	WCHAR* pwszSchemaTableNameRestr = NULL; 
	WCHAR* pwszSchemaRestrNameRestr = NULL; 
	char szTableNameRestr[MAX_NAME_SIZE]; 
	char szRestrNameRestr[MAX_NAME_SIZE]; 
	int  iRestrNameRestr = -1;
	
	bool bCatalogTableNameRestr = false;
	bool bCatalogRestrNameRestr = false;
	bool bSchemaTableNameRestr = false;
	bool bSchemaRestrNameRestr = false;
	bool bTableNameRestr = false;
	bool bRestrNameRestr = false;

	//Only BSTR is accepted
	if ((cRestrictions >= 1 && rgRestrictions[0].vt != VT_BSTR && rgRestrictions[0].vt != VT_EMPTY) ||
		(cRestrictions >= 2 && rgRestrictions[1].vt != VT_BSTR && rgRestrictions[1].vt != VT_EMPTY) || 
		(cRestrictions >= 3 && rgRestrictions[2].vt != VT_BSTR && rgRestrictions[2].vt != VT_EMPTY) ||
		(cRestrictions >= 4 && rgRestrictions[3].vt != VT_BSTR && rgRestrictions[3].vt != VT_EMPTY) ||
		(cRestrictions >= 5 && rgRestrictions[4].vt != VT_BSTR && rgRestrictions[4].vt != VT_EMPTY) ||
		(cRestrictions >= 6 && rgRestrictions[5].vt != VT_BSTR && rgRestrictions[5].vt != VT_EMPTY) )
	{
		TRACE( "CSchema::InitConstraintTables: restr fails" );
		return E_INVALIDARG;
	}

	//Extract TABLE_CATALOG restriction
	if (cRestrictions >= 1 && rgRestrictions[0].vt == VT_BSTR)
	{
		pwszCatalogTableNameRestr = rgRestrictions[0].bstrVal ? rgRestrictions[0].bstrVal : L"";
		bCatalogTableNameRestr = true;
		TRACE2( "  Catalog of table [%S]", pwszCatalogTableNameRestr );
	}

	//Extract CONSTRAINT_CATALOG restriction
	if (cRestrictions >= 4 && rgRestrictions[3].vt == VT_BSTR)
	{
		pwszCatalogRestrNameRestr = rgRestrictions[3].bstrVal ? rgRestrictions[3].bstrVal : L"";
		bCatalogRestrNameRestr = true;
		TRACE2( "  Catalog of restr [%S]", pwszCatalogRestrNameRestr );
	}

	//Extract TABLE_SCHEMA restriction
	if (cRestrictions >= 2 && rgRestrictions[1].vt == VT_BSTR)
	{
		pwszSchemaTableNameRestr = rgRestrictions[1].bstrVal ? rgRestrictions[1].bstrVal : L"";;
		if( !wcsicmp( pwszSchemaTableNameRestr, INFORMATION_SCHEMA ) )
			pwszSchemaTableNameRestr = L"";
		bSchemaTableNameRestr = true;
		TRACE2( "  Schema of table [%S]", pwszSchemaTableNameRestr );
	}

	//Extract CONSTRAINT_SCHEMA restriction
	if (cRestrictions >= 5 && rgRestrictions[4].vt == VT_BSTR)
	{
		pwszSchemaRestrNameRestr = rgRestrictions[4].bstrVal ? rgRestrictions[4].bstrVal : L"";;
		if( !wcsicmp( pwszSchemaRestrNameRestr, INFORMATION_SCHEMA ) )
			pwszSchemaRestrNameRestr = L"";
		bSchemaRestrNameRestr = true;
		TRACE2( "  Schema of restr [%S]", pwszSchemaRestrNameRestr );
	}

	// Load catalog information if necessary
	if( (bCatalogTableNameRestr || bSchemaTableNameRestr) && m_pCDataSource )
		m_pCDataSource->CheckAndTrackOneDatabase( pwszCatalogTableNameRestr, pwszSchemaTableNameRestr);
	if( (bCatalogRestrNameRestr || bSchemaRestrNameRestr) && m_pCDataSource )
		m_pCDataSource->CheckAndTrackOneDatabase( pwszCatalogRestrNameRestr, pwszSchemaRestrNameRestr);

	//Extract TABLE_NAME restriction
	if (cRestrictions >= 3 && rgRestrictions[2].vt == VT_BSTR)
	{
		if( rgRestrictions[2].bstrVal == NULL )
			*szTableNameRestr = '\0';
		else
			WideCharToMultiByte(CP_ACP, 0,
							rgRestrictions[2].bstrVal, -1,
							szTableNameRestr, MAX_NAME_SIZE,
							NULL, NULL );
		bTableNameRestr = true;
		TRACE2( "  Table [%s]", szTableNameRestr );
	}

	//Extract CONSTRAINT_NAME restriction
	if (cRestrictions >= 6 && rgRestrictions[5].vt == VT_BSTR)
	{
		if( rgRestrictions[5].bstrVal == NULL )
			*szRestrNameRestr = '\0';
		else
			WideCharToMultiByte(CP_ACP, 0,
							rgRestrictions[5].bstrVal, -1,
							szRestrNameRestr, MAX_NAME_SIZE,
							NULL, NULL );
		if( *szRestrNameRestr == '#' )
			iRestrNameRestr = atoi( szRestrNameRestr + 1 );
		
		bRestrNameRestr = true;
		TRACE2( "  Constraint [%s]", szRestrNameRestr );
	}
	
	TRACE( "CSchema::InitConstraintTables: before getting data" );

	// Get total number of indexes
	int iTotalIndexes = 0;
	int iTotalMetas;
	CSwstMeta* pSwstMeta;
	if( FAILED( m_pSwstMetaHolder->GetMaxNumber( &iTotalMetas ) ) )
		return E_FAIL;
	for( int j = 0; j < iTotalMetas; j++ )
		if( m_pSwstMetaHolder->At( j, &pSwstMeta ) == S_OK &&
			(!bCatalogTableNameRestr || !wcsicmp( pwszCatalogTableNameRestr, pSwstMeta->m_CatalogOwner.m_wszCatalog ) ) &&
			(!bSchemaTableNameRestr  || !wcsicmp( pwszSchemaTableNameRestr, pSwstMeta->m_CatalogOwner.m_wszOwner ) ) &&
			(!bCatalogRestrNameRestr || !wcsicmp( pwszCatalogRestrNameRestr, pSwstMeta->m_CatalogOwner.m_wszCatalog ) ) &&
			(!bSchemaRestrNameRestr  || !wcsicmp( pwszSchemaRestrNameRestr, pSwstMeta->m_CatalogOwner.m_wszOwner ) ) )
				iTotalIndexes += pSwstMeta->m_siIndexes;
	
	//Init data
	m_ppIndexes = new SWSTINDEX*[ iTotalIndexes ? iTotalIndexes : 1];
	if( m_ppIndexes == NULL )
		return E_OUTOFMEMORY;

	//Copy index pointers to buffer
	m_cRows = 0;
	for( j = 0; j < iTotalMetas; j++ )
		if( m_pSwstMetaHolder->At( j, &pSwstMeta ) == S_OK &&
			(!bCatalogTableNameRestr || !wcsicmp( pwszCatalogTableNameRestr, pSwstMeta->m_CatalogOwner.m_wszCatalog ) ) &&
			(!bSchemaTableNameRestr  || !wcsicmp( pwszSchemaTableNameRestr, pSwstMeta->m_CatalogOwner.m_wszOwner ) ) &&
			(!bCatalogRestrNameRestr || !wcsicmp( pwszCatalogRestrNameRestr, pSwstMeta->m_CatalogOwner.m_wszCatalog ) ) &&
			(!bSchemaRestrNameRestr  || !wcsicmp( pwszSchemaRestrNameRestr, pSwstMeta->m_CatalogOwner.m_wszOwner ) ) )
		{
				for (int i = 0; i < pSwstMeta->m_siIndexes; i++)
				{
					// Check CONSTRAINT_NAME restriction
					if (bRestrNameRestr)
					{
						if( !PassFilter( pSwstMeta->m_pIndexes + i, szRestrNameRestr, iRestrNameRestr ) )
							continue;
					}

					// Check TABLE_NAME restriction
					if (bTableNameRestr)
					{
						if( stricmp(pSwstMeta->m_pIndexes[i].m_pSwstField->m_pSwstFile->m_szName, szTableNameRestr) )
							continue;
					}

					// Only unique constraints and foreign keys
					if( !pSwstMeta->m_pIndexes[i].IsForeignKey() &&
						!pSwstMeta->m_pIndexes[i].IsUnique() )
							continue;

					// Only first segments (that is, 1 segment per index)
					if( pSwstMeta->m_pIndexes[i].m_wPart != 0 )
						continue;

					//Copy file pointer
					m_ppIndexes[m_cRows] = &pSwstMeta->m_pIndexes[i];
					
					// Completed with this row. Go to next one.
					m_cRows++;
				}
		}
	
	//Default Sort Order: TABLE_CATALOG, TABLE_SCHEMA, TABLE_NAME, CONSTRAINT_CATALOG, CONSTRAINT_SCHEMA, CONSTRAINT_NAME
	qsort( m_ppIndexes, m_cRows, sizeof(m_ppIndexes[0]), CompareConstraintTables );
	/*SWSTINDEX* pIndex;
	for (j = m_cRows - 1; j > 0; j--)
	{
		for (int k = 0; k < j; k++)
		{
			int iCat = wcsicmp(m_ppIndexes[k]->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszCatalog, 
							   m_ppIndexes[k + 1]->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszCatalog );
			int iOwn = wcsicmp(m_ppIndexes[k]->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszOwner, 
							   m_ppIndexes[k + 1]->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszOwner );
			int iTab = stricmp( m_ppIndexes[k]->m_pSwstField->m_pSwstFile->m_szName, 
							   m_ppIndexes[k + 1]->m_pSwstField->m_pSwstFile->m_szName );
			int iNam = CompareIndexNames( m_ppIndexes[k], m_ppIndexes[k + 1] );
				
			if( ( iCat > 0 ) ||
				( iCat==0 && iOwn > 0 ) || 
				( iCat==0 && iOwn==0 && iTab > 0 ) || 
				( iCat==0 && iOwn==0 && iTab==0 && iNam > 0 ) )
			{
				pIndex = m_ppIndexes[k];
				m_ppIndexes[k] = m_ppIndexes[k + 1];
				m_ppIndexes[k + 1] = pIndex;
			}
		}
	}*/

	TRACE2( "CSchema::InitConstraintTables: completing (total %d)", m_cRows );
	return S_OK;
}

typedef enum {
	eUNDEFINED_ETypeRestr,
	eUNIQUE_ETypeRestr,
	ePRIMARYKEY_ETypeRestr,
	eFOREIGNKEY_ETypeRestr,
	eCHECK_ETypeRestr
} ETypeRestr;

// Compare table constraints
int __cdecl CompareTableConstraints(const void *left, const void *right ) 
{
	SWSTINDEX* pLeft = *(SWSTINDEX**)left;
	SWSTINDEX* pRight = *(SWSTINDEX**)right;

	int iCat = wcsicmp(pLeft->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszCatalog, 
					   pRight->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszCatalog );
	if( iCat )
		return iCat;

	int iOwn = wcsicmp(pLeft->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszOwner, 
					   pRight->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszOwner );
	if( iOwn )
		return iOwn;

	int iNam = CompareIndexNames( pLeft, pRight );
	if( iNam )
		return iNam;

	int iTab = stricmp( pLeft->m_pSwstField->m_pSwstFile->m_szName, 
					   pRight->m_pSwstField->m_pSwstFile->m_szName );
	if( iTab )
		return iTab;

	ETypeRestr iTypeLeft = pLeft->IsForeignKey() ? eFOREIGNKEY_ETypeRestr : 
			( pLeft->IsPrimaryKey() ? ePRIMARYKEY_ETypeRestr : ( pLeft->IsUnique() ? eUNIQUE_ETypeRestr : eUNDEFINED_ETypeRestr ) );
	ETypeRestr iTypeRight = pRight->IsForeignKey() ? eFOREIGNKEY_ETypeRestr : 
			( pRight->IsPrimaryKey() ? ePRIMARYKEY_ETypeRestr : ( pRight->IsUnique() ? eUNIQUE_ETypeRestr : eUNDEFINED_ETypeRestr ) );
	int iTyp = iTypeRight - iTypeLeft;
	return iTyp;
}



//Initialize object with columns schema
HRESULT CSchema::InitTableConstraints
	(
	ULONG			cRestrictions,		//@parm IN	| Count of restriction values
	const VARIANT	rgRestrictions[]	//@parm IN	| Array of restriction values
	)
{
	TRACE( "CSchema::InitTableConstraints" );

	m_enumSchema = TABLE_CONSTRAINTS_SCHEMA;

	//Init columns
	static COLINFO Columns[] = 
	{
		{L"CONSTRAINT_CATALOG",	DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"CONSTRAINT_SCHEMA",	DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"CONSTRAINT_NAME",	DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"TABLE_CATALOG",		DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"TABLE_SCHEMA",		DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"TABLE_NAME",			DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"CONSTRAINT_TYPE",	DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"IS_DEFERRABLE",		DBTYPE_BOOL,	sizeof(VARIANT_BOOL),	DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"INITIALLY_DEFERRED",	DBTYPE_BOOL,	sizeof(VARIANT_BOOL),	DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"DESCRIPTION",		DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
	};
	m_pColumns = Columns;
	m_cCols = 10;

	//Check restrictions
	if (cRestrictions > 7)
	{
		TRACE2( "CSchema::InitTableConstraints: restrictions = %d", cRestrictions );
		return E_INVALIDARG;
	}

	WCHAR* pwszCatalogTableNameRestr = NULL; 
	WCHAR* pwszCatalogRestrNameRestr = NULL; 
	WCHAR* pwszSchemaTableNameRestr = NULL; 
	WCHAR* pwszSchemaRestrNameRestr = NULL; 
	char szTableNameRestr[MAX_NAME_SIZE]; 
	char szRestrNameRestr[MAX_NAME_SIZE]; 
	int  iRestrNameRestr = -1;
	ETypeRestr enumTypeRestr;
	
	bool bCatalogTableNameRestr = false;
	bool bCatalogRestrNameRestr = false;
	bool bSchemaTableNameRestr = false;
	bool bSchemaRestrNameRestr = false;
	bool bTableNameRestr = false;
	bool bRestrNameRestr = false;
	bool bTypeRestr = false;

	//Only BSTR is accepted
	if ((cRestrictions >= 1 && rgRestrictions[0].vt != VT_BSTR && rgRestrictions[0].vt != VT_EMPTY) ||
		(cRestrictions >= 2 && rgRestrictions[1].vt != VT_BSTR && rgRestrictions[1].vt != VT_EMPTY) || 
		(cRestrictions >= 3 && rgRestrictions[2].vt != VT_BSTR && rgRestrictions[2].vt != VT_EMPTY) ||
		(cRestrictions >= 4 && rgRestrictions[3].vt != VT_BSTR && rgRestrictions[3].vt != VT_EMPTY) ||
		(cRestrictions >= 5 && rgRestrictions[4].vt != VT_BSTR && rgRestrictions[4].vt != VT_EMPTY) ||
		(cRestrictions >= 6 && rgRestrictions[5].vt != VT_BSTR && rgRestrictions[5].vt != VT_EMPTY) ||
		(cRestrictions >= 7 && rgRestrictions[6].vt != VT_BSTR && rgRestrictions[6].vt != VT_EMPTY) )
	{
		TRACE( "CSchema::InitTableConstraints: restr fails" );
		return E_INVALIDARG;
	}

	//Extract TABLE_CATALOG restriction
	if (cRestrictions >= 4 && rgRestrictions[3].vt == VT_BSTR)
	{
		pwszCatalogTableNameRestr = rgRestrictions[3].bstrVal ? rgRestrictions[3].bstrVal : L"";
		bCatalogTableNameRestr = true;
		TRACE2( "  Catalog of table [%S]", pwszCatalogTableNameRestr );
	}

	//Extract CONSTRAINT_CATALOG restriction
	if (cRestrictions >= 1 && rgRestrictions[0].vt == VT_BSTR)
	{
		pwszCatalogRestrNameRestr = rgRestrictions[0].bstrVal ? rgRestrictions[0].bstrVal : L"";
		bCatalogRestrNameRestr = true;
		TRACE2( "  Catalog of restr [%S]", pwszCatalogRestrNameRestr );
	}

	//Extract TABLE_SCHEMA restriction
	if (cRestrictions >= 5 && rgRestrictions[4].vt == VT_BSTR)
	{
		pwszSchemaTableNameRestr = rgRestrictions[4].bstrVal ? rgRestrictions[4].bstrVal : L"";
		if( !wcsicmp( pwszSchemaTableNameRestr, INFORMATION_SCHEMA ) )
			pwszSchemaTableNameRestr = L"";
		bSchemaTableNameRestr = true;
		TRACE2( "  Schema of table [%S]", pwszSchemaTableNameRestr );
	}

	//Extract CONSTRAINT_SCHEMA restriction
	if (cRestrictions >= 2 && rgRestrictions[1].vt == VT_BSTR)
	{
		pwszSchemaRestrNameRestr = rgRestrictions[1].bstrVal ? rgRestrictions[1].bstrVal : L"";
		if( !wcsicmp( pwszSchemaRestrNameRestr, INFORMATION_SCHEMA ) )
			pwszSchemaRestrNameRestr = L"";
		bSchemaRestrNameRestr = true;
		TRACE2( "  Schema of restr [%S]", pwszSchemaRestrNameRestr );
	}

	// Load catalog information if necessary
	if( (bCatalogTableNameRestr || bSchemaTableNameRestr) && m_pCDataSource )
		m_pCDataSource->CheckAndTrackOneDatabase( pwszCatalogTableNameRestr, pwszSchemaTableNameRestr);
	if( (bCatalogRestrNameRestr || bSchemaRestrNameRestr) && m_pCDataSource )
		m_pCDataSource->CheckAndTrackOneDatabase( pwszCatalogRestrNameRestr, pwszSchemaRestrNameRestr);

	//Extract TABLE_NAME restriction
	if (cRestrictions >= 6 && rgRestrictions[5].vt == VT_BSTR)
	{
		if( rgRestrictions[5].bstrVal == NULL )
			*szTableNameRestr = '\0';
		else
			WideCharToMultiByte(CP_ACP, 0,
							rgRestrictions[5].bstrVal, -1,
							szTableNameRestr, MAX_NAME_SIZE,
							NULL, NULL );
		bTableNameRestr = true;
		TRACE2( "  Table [%s]", szTableNameRestr );
	}

	//Extract CONSTRAINT_NAME restriction
	if (cRestrictions >= 3 && rgRestrictions[2].vt == VT_BSTR)
	{
		if( rgRestrictions[2].bstrVal == NULL )
			*szRestrNameRestr = '\0';
		else
			WideCharToMultiByte(CP_ACP, 0,
							rgRestrictions[2].bstrVal, -1,
							szRestrNameRestr, MAX_NAME_SIZE,
							NULL, NULL );
		if( *szRestrNameRestr == '#' )
			iRestrNameRestr = atoi( szRestrNameRestr + 1 );
		
		bRestrNameRestr = true;
		TRACE2( "  Constraint [%s]", szRestrNameRestr );
	}
	
	//Extract CONSTRAINT_CATALOG restriction
	if (cRestrictions >= 7 && rgRestrictions[6].vt == VT_BSTR)
	{
		if( rgRestrictions[6].bstrVal == NULL )
			enumTypeRestr = eUNDEFINED_ETypeRestr;
		else if( !wcsicmp( rgRestrictions[6].bstrVal, L"UNIQUE" ) )
			enumTypeRestr = eUNIQUE_ETypeRestr;
		else if( !wcsicmp( rgRestrictions[6].bstrVal, L"PRIMARY KEY" ) )
			enumTypeRestr = ePRIMARYKEY_ETypeRestr;
		else if( !wcsicmp( rgRestrictions[6].bstrVal, L"FOREIGN KEY" ) )
			enumTypeRestr = eFOREIGNKEY_ETypeRestr;
		else if( !wcsicmp( rgRestrictions[6].bstrVal, L"CHECK" ) )
			enumTypeRestr = eCHECK_ETypeRestr;
		bTypeRestr = true;
		TRACE2( "  Type of restr [%s]", rgRestrictions[6].bstrVal );
	}

	TRACE( "CSchema::InitTableConstraints: before getting data" );

	// Get total number of indexes
	int iTotalIndexes = 0;
	int iTotalMetas;
	CSwstMeta* pSwstMeta;
	if( FAILED( m_pSwstMetaHolder->GetMaxNumber( &iTotalMetas ) ) )
		return E_FAIL;
	for( int j = 0; j < iTotalMetas; j++ )
		if( m_pSwstMetaHolder->At( j, &pSwstMeta ) == S_OK &&
			(!bCatalogTableNameRestr || !wcsicmp( pwszCatalogTableNameRestr, pSwstMeta->m_CatalogOwner.m_wszCatalog ) ) &&
			(!bSchemaTableNameRestr  || !wcsicmp( pwszSchemaTableNameRestr, pSwstMeta->m_CatalogOwner.m_wszOwner ) ) &&
			(!bCatalogRestrNameRestr || !wcsicmp( pwszCatalogRestrNameRestr, pSwstMeta->m_CatalogOwner.m_wszCatalog ) ) &&
			(!bSchemaRestrNameRestr  || !wcsicmp( pwszSchemaRestrNameRestr, pSwstMeta->m_CatalogOwner.m_wszOwner ) ) )
				iTotalIndexes += pSwstMeta->m_siIndexes;
	
	//Init data
	m_ppIndexes = new SWSTINDEX*[ iTotalIndexes ? iTotalIndexes : 1];
	if( m_ppIndexes == NULL )
		return E_OUTOFMEMORY;

	//Copy index pointers to buffer
	m_cRows = 0;
	for( j = 0; j < iTotalMetas; j++ )
		if( m_pSwstMetaHolder->At( j, &pSwstMeta ) == S_OK &&
			(!bCatalogTableNameRestr || !wcsicmp( pwszCatalogTableNameRestr, pSwstMeta->m_CatalogOwner.m_wszCatalog ) ) &&
			(!bSchemaTableNameRestr  || !wcsicmp( pwszSchemaTableNameRestr, pSwstMeta->m_CatalogOwner.m_wszOwner ) ) &&
			(!bCatalogRestrNameRestr || !wcsicmp( pwszCatalogRestrNameRestr, pSwstMeta->m_CatalogOwner.m_wszCatalog ) ) &&
			(!bSchemaRestrNameRestr  || !wcsicmp( pwszSchemaRestrNameRestr, pSwstMeta->m_CatalogOwner.m_wszOwner ) ) )
		{
				for (int i = 0; i < pSwstMeta->m_siIndexes; i++)
				{
					// Check CONSTRAINT_NAME restriction
					if (bRestrNameRestr)
					{
						if( !PassFilter( pSwstMeta->m_pIndexes +i, szRestrNameRestr, iRestrNameRestr ) )
							continue;
					}

					// Check TABLE_NAME restriction
					if (bTableNameRestr)
					{
						if( stricmp(pSwstMeta->m_pIndexes[i].m_pSwstField->m_pSwstFile->m_szName, szTableNameRestr) )
							continue;
					}

					// Check CONSTRAINT_TYPE restriction
					if( bTypeRestr )
						switch( enumTypeRestr )
						{
						case eUNIQUE_ETypeRestr:
							if( !pSwstMeta->m_pIndexes[i].IsUnique() )
								continue; // for
							break; // switch
						case ePRIMARYKEY_ETypeRestr:
							if( !pSwstMeta->m_pIndexes[i].IsPrimaryKey() )
								continue; // for
							break; // switch
						case eFOREIGNKEY_ETypeRestr:
							if( !pSwstMeta->m_pIndexes[i].IsForeignKey() )
								continue; // for
							break; // switch
						default:
							continue; // for // does not comply
						}

					// Only unique constraints and foreign keys
					if( !pSwstMeta->m_pIndexes[i].IsForeignKey() &&
						!pSwstMeta->m_pIndexes[i].IsUnique() )
							continue;

					// Only first segments (that is, 1 segment per index)
					if( pSwstMeta->m_pIndexes[i].m_wPart != 0 )
						continue;

					//Copy file pointer
					m_ppIndexes[m_cRows] = &pSwstMeta->m_pIndexes[i];
					
					// Completed with this row. Go to next one.
					m_cRows++;
				}
		}
	
	//Default Sort Order: CONSTRAINT_CATALOG, CONSTRAINT_SCHEMA, CONSTRAINT_NAME, TABLE_CATALOG, TABLE_SCHEMA, TABLE_NAME, CONSTRAINT_TYPE
	qsort( m_ppIndexes, m_cRows, sizeof(m_ppIndexes[0]), CompareTableConstraints );
	/*SWSTINDEX* pIndex;
	for (j = m_cRows - 1; j > 0; j--)
	{
		for (int k = 0; k < j; k++)
		{
			int iCat = wcsicmp(m_ppIndexes[k]->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszCatalog, 
							   m_ppIndexes[k + 1]->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszCatalog );
			int iOwn = wcsicmp(m_ppIndexes[k]->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszOwner, 
							   m_ppIndexes[k + 1]->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszOwner );
			int iNam = CompareIndexNames( m_ppIndexes[k], m_ppIndexes[k + 1] );
			int iTab = stricmp( m_ppIndexes[k]->m_pSwstField->m_pSwstFile->m_szName, 
							   m_ppIndexes[k + 1]->m_pSwstField->m_pSwstFile->m_szName );
			ETypeRestr iTypeLeft = m_ppIndexes[k]->IsForeignKey() ? eFOREIGNKEY_ETypeRestr : 
					( m_ppIndexes[k]->IsPrimaryKey() ? ePRIMARYKEY_ETypeRestr : ( m_ppIndexes[k]->IsUnique() ? eUNIQUE_ETypeRestr : eUNDEFINED_ETypeRestr ) );
			ETypeRestr iTypeRight = m_ppIndexes[k + 1]->IsForeignKey() ? eFOREIGNKEY_ETypeRestr : 
					( m_ppIndexes[k + 1]->IsPrimaryKey() ? ePRIMARYKEY_ETypeRestr : ( m_ppIndexes[k + 1]->IsUnique() ? eUNIQUE_ETypeRestr : eUNDEFINED_ETypeRestr ) );
			int iTyp = iTypeRight - iTypeLeft;
				
			if( ( iCat > 0 ) ||
				( iCat==0 && iOwn > 0 ) || 
				( iCat==0 && iOwn==0 && iNam > 0 ) || 
				( iCat==0 && iOwn==0 && iNam==0 && iTab > 0 ) ||
				( iCat==0 && iOwn==0 && iNam==0 && iTab==0 && iTyp > 0 ) )
			{
				pIndex = m_ppIndexes[k];
				m_ppIndexes[k] = m_ppIndexes[k + 1];
				m_ppIndexes[k + 1] = pIndex;
			}
		}
	}*/

	TRACE2( "CSchema::InitTableConstraints: completing (total %d)", m_cRows );
	return S_OK;
}

// Compare primary keys
int __cdecl ComparePRIMARYKEY_ETypeRestrs(const void *left, const void *right ) 
{
	SWSTINDEX* pLeft = *(SWSTINDEX**)left;
	SWSTINDEX* pRight = *(SWSTINDEX**)right;

	int iCat = wcsicmp(pLeft->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszCatalog, 
					   pRight->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszCatalog );
	if( iCat )
		return iCat;

	int iOwn = wcsicmp(pLeft->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszOwner, 
					   pRight->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszOwner );
	if( iOwn ) 
		return iOwn;

	int iTab = stricmp(pLeft->m_pSwstField->m_pSwstFile->m_szName, 
					   pRight->m_pSwstField->m_pSwstFile->m_szName );
	return iTab;
}



//Initialize object with columns schema
HRESULT CSchema::InitPrimaryKeys
	(
	ULONG			cRestrictions,		//@parm IN	| Count of restriction values
	const VARIANT	rgRestrictions[]	//@parm IN	| Array of restriction values
	)
{
	TRACE( "CSchema::InitPrimaryKeys" );

	m_enumSchema = PRIMARY_KEYS_SCHEMA;

	//Init columns
	static COLINFO Columns[] = 
	{
		{L"PK_NAME",		DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"TABLE_CATALOG",	DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"TABLE_SCHEMA",	DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"TABLE_NAME",		DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"COLUMN_NAME",	DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"COLUMN_GUID",	DBTYPE_GUID,	sizeof(GUID),	DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"COLUMN_PROPID",	DBTYPE_UI4,		sizeof(ULONG),	DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"ORDINAL",		DBTYPE_UI4,		sizeof(ULONG),	DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL}
	};
	m_pColumns = Columns;
	m_cCols = 8;

	//Check restrictions
	if (cRestrictions > 3)
	{
		TRACE2( "CSchema::InitPrimaryKeys: restrictions = %d", cRestrictions );
		return E_INVALIDARG;
	}

	WCHAR* pwszCatalogNameRestr = NULL; 
	WCHAR* pwszSchemaNameRestr = NULL; 
	char szTableNameRestr[MAX_NAME_SIZE]; 
	
	bool bCatalogNameRestr = false;
	bool bSchemaNameRestr = false;
	bool bTableNameRestr = false;

	//Only BSTR is accepted
	if ((cRestrictions >= 1 && rgRestrictions[0].vt != VT_BSTR && rgRestrictions[0].vt != VT_EMPTY) ||
		(cRestrictions >= 2 && rgRestrictions[1].vt != VT_BSTR && rgRestrictions[1].vt != VT_EMPTY) || 
		(cRestrictions >= 3 && rgRestrictions[2].vt != VT_BSTR && rgRestrictions[2].vt != VT_EMPTY) )
	{
		TRACE( "CSchema::InitPrimaryKeys: restr fails" );
		return E_INVALIDARG;
	}

	//Extract TABLE_CATALOG restriction
	if (cRestrictions >= 1 && rgRestrictions[0].vt == VT_BSTR)
	{
		pwszCatalogNameRestr = rgRestrictions[0].bstrVal ? rgRestrictions[0].bstrVal : L"";
		bCatalogNameRestr = true;
		TRACE2( "  Catalog of table [%S]", pwszCatalogNameRestr );
	}

	//Extract TABLE_SCHEMA restriction
	if (cRestrictions >= 2 && rgRestrictions[1].vt == VT_BSTR)
	{
		pwszSchemaNameRestr = rgRestrictions[1].bstrVal ? rgRestrictions[1].bstrVal : L"";;
		if( !wcsicmp( pwszSchemaNameRestr, INFORMATION_SCHEMA ) )
			pwszSchemaNameRestr = L"";
		bSchemaNameRestr = true;
		TRACE2( "  Schema of table [%S]", pwszSchemaNameRestr );
	}

	// Load catalog information if necessary
	if( (bCatalogNameRestr || bSchemaNameRestr) && m_pCDataSource )
		m_pCDataSource->CheckAndTrackOneDatabase( pwszCatalogNameRestr, pwszSchemaNameRestr);

	//Extract TABLE_NAME restriction
	if (cRestrictions >= 3 && rgRestrictions[2].vt == VT_BSTR)
	{
		if( rgRestrictions[2].bstrVal == NULL )
			*szTableNameRestr = '\0';
		else
			WideCharToMultiByte(CP_ACP, 0,
							rgRestrictions[2].bstrVal, -1,
							szTableNameRestr, MAX_NAME_SIZE,
							NULL, NULL );
		bTableNameRestr = true;
		TRACE2( "  Table [%s]", szTableNameRestr );
	}

	TRACE( "CSchema::InitPrimaryKeys: before getting data" );

	// Get total number of indexes
	int iTotalIndexes = 0;
	int iTotalMetas;
	CSwstMeta* pSwstMeta;
	if( FAILED( m_pSwstMetaHolder->GetMaxNumber( &iTotalMetas ) ) )
		return E_FAIL;
	for( int j = 0; j < iTotalMetas; j++ )
		if( m_pSwstMetaHolder->At( j, &pSwstMeta ) == S_OK &&
			(!bCatalogNameRestr || !wcsicmp( pwszCatalogNameRestr, pSwstMeta->m_CatalogOwner.m_wszCatalog ) ) &&
			(!bSchemaNameRestr  || !wcsicmp( pwszSchemaNameRestr, pSwstMeta->m_CatalogOwner.m_wszOwner ) ) )
				iTotalIndexes += pSwstMeta->m_siIndexes;
	
	//Init data
	m_ppIndexes = new SWSTINDEX*[ iTotalIndexes ? iTotalIndexes : 1];
	if( m_ppIndexes == NULL )
		return E_OUTOFMEMORY;

	//Copy index pointers to buffer
	m_cRows = 0;
	for( j = 0; j < iTotalMetas; j++ )
		if( m_pSwstMetaHolder->At( j, &pSwstMeta ) == S_OK &&
			(!bCatalogNameRestr || !wcsicmp( pwszCatalogNameRestr, pSwstMeta->m_CatalogOwner.m_wszCatalog ) ) &&
			(!bSchemaNameRestr  || !wcsicmp( pwszSchemaNameRestr, pSwstMeta->m_CatalogOwner.m_wszOwner ) ) )
		{
				for (int i = 0; i < pSwstMeta->m_siIndexes; i++)
				{
					// Check TABLE_NAME restriction
					if (bTableNameRestr)
					{
						if( stricmp(pSwstMeta->m_pIndexes[i].m_pSwstField->m_pSwstFile->m_szName, szTableNameRestr) )
							continue;
					}

					// Only primary keys
					if( !pSwstMeta->m_pIndexes[i].IsPrimaryKey() )
						continue;

					//Copy file pointer
					m_ppIndexes[m_cRows] = &pSwstMeta->m_pIndexes[i];
					
					// Completed with this row. Go to next one.
					m_cRows++;
				}
		}
	
	//Default Sort Order: TABLE_CATALOG, TABLE_SCHEMA, TABLE_NAME
	qsort( m_ppIndexes, m_cRows, sizeof(m_ppIndexes[0]), ComparePRIMARYKEY_ETypeRestrs );
	/*SWSTINDEX* pIndex;
	for (j = m_cRows - 1; j > 0; j--)
	{
		for (int k = 0; k < j; k++)
		{
			int iCat = wcsicmp(m_ppIndexes[k]->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszCatalog, 
							   m_ppIndexes[k + 1]->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszCatalog );
			int iOwn = wcsicmp(m_ppIndexes[k]->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszOwner, 
							   m_ppIndexes[k + 1]->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszOwner );
			int iTab = stricmp( m_ppIndexes[k]->m_pSwstField->m_pSwstFile->m_szName, 
							   m_ppIndexes[k + 1]->m_pSwstField->m_pSwstFile->m_szName );
				
			if( ( iCat > 0 ) ||
				( iCat==0 && iOwn > 0 ) || 
				( iCat==0 && iOwn==0 && iTab > 0 ) )
			{
				pIndex = m_ppIndexes[k];
				m_ppIndexes[k] = m_ppIndexes[k + 1];
				m_ppIndexes[k + 1] = pIndex;
			}
		}
	}*/

	TRACE2( "CSchema::InitPrimaryKeys: completing (total %d)", m_cRows );
	return S_OK;
}


// Compare referential constraints
int __cdecl CompareReferentialConstraints(const void *left, const void *right ) 
{
	SWSTRELATE* pLeft = *(SWSTRELATE**)left;
	SWSTRELATE* pRight = *(SWSTRELATE**)right;

	int iCatDep = wcsicmp(pLeft->m_ppDependIndex[0]->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszCatalog, 
					   pRight->m_ppDependIndex[0]->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszCatalog );
	if( iCatDep )
		iCatDep;
	
	int iOwnDep = wcsicmp(pLeft->m_ppDependIndex[0]->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszOwner, 
					   pRight->m_ppDependIndex[0]->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszOwner );
	if( iOwnDep )
		return iOwnDep;

	int iNamDep = CompareIndexNames( pLeft->m_ppDependIndex[0], pRight->m_ppDependIndex[0] );
	if( iNamDep )
		return iNamDep;
		
	int iCatPar = wcsicmp(pLeft->m_ppParentIndex[0]->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszCatalog, 
					   pRight->m_ppParentIndex[0]->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszCatalog );
	if( iCatPar )
		return iCatPar;

	int iOwnPar = wcsicmp(pLeft->m_ppParentIndex[0]->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszOwner, 
					   pRight->m_ppParentIndex[0]->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszOwner );
	if( iOwnPar )
		return iOwnPar;
	
	int iNamPar = CompareIndexNames( pLeft->m_ppParentIndex[0], pRight->m_ppParentIndex[0] );
	return iNamPar;
}


//Initialize object with columns schema
HRESULT CSchema::InitReferentialConstraints
	(
	ULONG			cRestrictions,		//@parm IN	| Count of restriction values
	const VARIANT	rgRestrictions[]	//@parm IN	| Array of restriction values
	)
{
	TRACE( "CSchema::InitReferentialConstraints" );

	m_enumSchema = REFERENTIAL_CONSTRAINTS_SCHEMA;

	//Init columns
	static COLINFO Columns[] = 
	{
		{L"CONSTRAINT_CATALOG",	DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"CONSTRAINT_SCHEMA",	DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"CONSTRAINT_NAME",	DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"UNIQUE_CONSTRAINT_CATALOG",	DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"UNIQUE_CONSTRAINT_SCHEMA",	DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"UNIQUE_CONSTRAINT_NAME",		DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"MATCH_OPTION",		DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"UPDATE_RULE",		DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"DELETE_RULE",		DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"DESCRIPTION",		DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL}
	};
	m_pColumns = Columns;
	m_cCols = 10;

	//Check restrictions
	if (cRestrictions > 3)
	{
		TRACE2( "CSchema::InitReferentialConstraints: restrictions = %d", cRestrictions );
		return E_INVALIDARG;
	}

	WCHAR* pwszCatalogNameRestr = NULL; 
	WCHAR* pwszSchemaNameRestr = NULL; 
	char   szRestrNameRestr[MAX_NAME_SIZE]; 
	int    iRestrNameRestr = -1;
	
	bool bCatalogNameRestr = false;
	bool bSchemaNameRestr = false;
	bool bRestrNameRestr = false;

	//Only BSTR is accepted
	if ((cRestrictions >= 1 && rgRestrictions[0].vt != VT_BSTR && rgRestrictions[0].vt != VT_EMPTY) ||
		(cRestrictions >= 2 && rgRestrictions[1].vt != VT_BSTR && rgRestrictions[1].vt != VT_EMPTY) || 
		(cRestrictions >= 3 && rgRestrictions[2].vt != VT_BSTR && rgRestrictions[2].vt != VT_EMPTY) )
	{
		TRACE( "CSchema::InitReferentialConstraints: restr fails" );
		return E_INVALIDARG;
	}

	//Extract CONSTRAINT_CATALOG restriction
	if (cRestrictions >= 1 && rgRestrictions[0].vt == VT_BSTR)
	{
		pwszCatalogNameRestr = rgRestrictions[0].bstrVal ? rgRestrictions[0].bstrVal : L"";
		bCatalogNameRestr = true;
		TRACE2( "  Catalog of restr [%S]", pwszCatalogNameRestr );
	}

	//Extract CONSTRAINT_SCHEMA restriction
	if (cRestrictions >= 2 && rgRestrictions[1].vt == VT_BSTR)
	{
		pwszSchemaNameRestr = rgRestrictions[1].bstrVal ? rgRestrictions[1].bstrVal : L"";
		if( !wcsicmp( pwszSchemaNameRestr, INFORMATION_SCHEMA ) )
			pwszSchemaNameRestr = L"";
		bSchemaNameRestr = true;
		TRACE2( "  Schema of restr [%S]", pwszSchemaNameRestr );
	}

	// Load catalog information if necessary
	if( (bCatalogNameRestr || bSchemaNameRestr) && m_pCDataSource )
		m_pCDataSource->CheckAndTrackOneDatabase( pwszCatalogNameRestr, pwszSchemaNameRestr);

	//Extract CONSTRAINT_NAME restriction
	if (cRestrictions >= 3 && rgRestrictions[2].vt == VT_BSTR)
	{
		if( rgRestrictions[2].bstrVal == NULL )
			*szRestrNameRestr = '\0';
		else
			WideCharToMultiByte(CP_ACP, 0,
							rgRestrictions[2].bstrVal, -1,
							szRestrNameRestr, MAX_NAME_SIZE,
							NULL, NULL );
		if( *szRestrNameRestr == '#' )
			iRestrNameRestr = atoi( szRestrNameRestr + 1 );
		
		bRestrNameRestr = true;
		TRACE2( "  Constraint [%s]", szRestrNameRestr );
	}
	
	TRACE( "CSchema::InitReferentialConstraints: before getting data" );

	// Get total number of indexes
	int iTotalRelations = 0;
	int iTotalMetas;
	CSwstMeta* pSwstMeta;
	if( FAILED( m_pSwstMetaHolder->GetMaxNumber( &iTotalMetas ) ) )
		return E_FAIL;
	for( int j = 0; j < iTotalMetas; j++ )
		if( m_pSwstMetaHolder->At( j, &pSwstMeta ) == S_OK &&
			(!bCatalogNameRestr || !wcsicmp( pwszCatalogNameRestr, pSwstMeta->m_CatalogOwner.m_wszCatalog ) ) &&
			(!bSchemaNameRestr  || !wcsicmp( pwszSchemaNameRestr, pSwstMeta->m_CatalogOwner.m_wszOwner ) ) )
				iTotalRelations += pSwstMeta->m_siRelations;
	
	//Init data
	m_ppRelations = new SWSTRELATE*[ iTotalRelations ? iTotalRelations : 1];
	if( m_ppRelations == NULL )
		return E_OUTOFMEMORY;

	//Copy index pointers to buffer
	m_cRows = 0;
	for( j = 0; j < iTotalMetas; j++ )
		if( m_pSwstMetaHolder->At( j, &pSwstMeta ) == S_OK &&
			(!bCatalogNameRestr || !wcsicmp( pwszCatalogNameRestr, pSwstMeta->m_CatalogOwner.m_wszCatalog ) ) &&
			(!bSchemaNameRestr  || !wcsicmp( pwszSchemaNameRestr, pSwstMeta->m_CatalogOwner.m_wszOwner ) ) )
		{
				for (int i = 0; i < pSwstMeta->m_siRelations; i++)
				{
					// Check CONSTRAINT_NAME restriction
					if (bRestrNameRestr)
					{
						if( !PassFilter( pSwstMeta->m_pRelations[ i ].m_ppDependIndex[ 0 ], szRestrNameRestr, iRestrNameRestr ) )
							continue;
					}

					//Copy file pointer
					m_ppRelations[m_cRows] = &pSwstMeta->m_pRelations[ i ];
					
					// Completed with this row. Go to next one.
					m_cRows++;
				}
		}
	
	//Default Sort Order: CONSTRAINT_CATALOG, CONSTRAINT_SCHEMA, CONSTRAINT_NAME, UNIQUE_CONSTRAINT_CATALOG, UNIQUE_CONSTRAINT_SCHEMA, UNIQUE_CONSTRAINT_NAME
	qsort( m_ppRelations, m_cRows, sizeof(m_ppRelations[0]), CompareReferentialConstraints );
	/*SWSTRELATE* pRelate;
	for (j = m_cRows - 1; j > 0; j--)
	{
		for (int k = 0; k < j; k++)
		{
			int iCatDep = wcsicmp(m_ppRelations[k]->m_ppDependIndex[0]->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszCatalog, 
							   m_ppRelations[k + 1]->m_ppDependIndex[0]->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszCatalog );
			int iOwnDep = wcsicmp(m_ppRelations[k]->m_ppDependIndex[0]->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszOwner, 
							   m_ppRelations[k + 1]->m_ppDependIndex[0]->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszOwner );
			int iNamDep = CompareIndexNames( m_ppRelations[k]->m_ppDependIndex[0], m_ppRelations[k + 1]->m_ppDependIndex[0] );
				
			int iCatPar = wcsicmp(m_ppRelations[k]->m_ppParentIndex[0]->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszCatalog, 
							   m_ppRelations[k + 1]->m_ppParentIndex[0]->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszCatalog );
			int iOwnPar = wcsicmp(m_ppRelations[k]->m_ppParentIndex[0]->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszOwner, 
							   m_ppRelations[k + 1]->m_ppParentIndex[0]->m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszOwner );
			int iNamPar = CompareIndexNames( m_ppRelations[k]->m_ppParentIndex[0], m_ppRelations[k + 1]->m_ppParentIndex[0] );

			if( ( iCatDep > 0 ) ||
				( iCatDep==0 && iOwnDep > 0 ) || 
				( iCatDep==0 && iOwnDep==0 && iNamDep > 0 ) || 
				( iCatDep==0 && iOwnDep==0 && iNamDep==0 && iCatPar > 0 ) ||
				( iCatDep==0 && iOwnDep==0 && iNamDep==0 && iCatPar==0 && iOwnPar > 0 ) ||
				( iCatDep==0 && iOwnDep==0 && iNamDep==0 && iCatPar==0 && iOwnPar==0 && iNamPar > 0 ) )
			{
				pRelate = m_ppRelations[k];
				m_ppRelations[k] = m_ppRelations[k + 1];
				m_ppRelations[k + 1] = pRelate;
			}
		}
	}*/

	TRACE2( "CSchema::InitReferentialConstraints: completing (total %d)", m_cRows );
	return S_OK;
}

// Compare foreign keys
int __cdecl CompareFOREIGNKEY_ETypeRestrs(const void *left, const void *right ) 
{
	FOREIGNKEYINFO* pLeft = (FOREIGNKEYINFO*)left;
	FOREIGNKEYINFO* pRight = (FOREIGNKEYINFO*)right;

	int iCatDep = wcsicmp(pLeft->pRelate->m_ppDependIndex[ pLeft->iIdx ]->
							m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszCatalog, 
						  pRight->pRelate->m_ppDependIndex[ pRight->iIdx ]->
							m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszCatalog );
	if( iCatDep )
		return iCatDep;

	int iOwnDep = wcsicmp(pLeft->pRelate->m_ppDependIndex[ pLeft->iIdx ]->
							m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszOwner, 
						  pRight->pRelate->m_ppDependIndex[ pRight->iIdx ]->
							m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszOwner );
	if( iOwnDep )
		return iOwnDep;

	int iTabDep = stricmp( pLeft->pRelate->m_ppDependIndex[ pLeft->iIdx ]->
							m_pSwstField->m_pSwstFile->m_szName,
						  pRight->pRelate->m_ppDependIndex[ pRight->iIdx ]->
							m_pSwstField->m_pSwstFile->m_szName );
	return iTabDep;
}


//Initialize object with columns schema
HRESULT CSchema::InitForeignKeys
	(
	ULONG			cRestrictions,		//@parm IN	| Count of restriction values
	const VARIANT	rgRestrictions[]	//@parm IN	| Array of restriction values
	)
{
	TRACE( "CSchema::InitForeignKeys" );

	m_enumSchema = FOREIGN_KEYS_SCHEMA;

	//Init columns
	static COLINFO Columns[] = 
	{
		{L"PK_NAME",			DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"PK_TABLE_CATALOG",	DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"PK_TABLE_SCHEMA",	DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"PK_TABLE_NAME",		DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"PK_COLUMN_NAME",		DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"PK_COLUMN_GUID",		DBTYPE_GUID,	sizeof(GUID),	DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"PK_COLUMN_PROPID",	DBTYPE_UI4,		sizeof(ULONG),	DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"FK_NAME",			DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"FK_TABLE_CATALOG",	DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"FK_TABLE_SCHEMA",	DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"FK_TABLE_NAME",		DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"FK_COLUMN_NAME",		DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"FK_COLUMN_GUID",		DBTYPE_GUID,	sizeof(GUID),	DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"FK_COLUMN_PROPID",	DBTYPE_UI4,		sizeof(ULONG),	DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"ORDINAL",			DBTYPE_UI4,		sizeof(ULONG),	DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"UPDATE_RULE",		DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"DELETE_RULE",		DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"DEFERRABILITY",		DBTYPE_I2,		sizeof(SHORT),	DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL}
	};
	m_pColumns = Columns;
	m_cCols = 18;

	//Check restrictions
	if (cRestrictions > 6)
	{
		TRACE2( "CSchema::InitForeignKeys: restrictions = %d", cRestrictions );
		return E_INVALIDARG;
	}

	WCHAR* pwszCatalogParentNameRestr = NULL; 
	WCHAR* pwszCatalogDependNameRestr = NULL; 
	WCHAR* pwszSchemaParentNameRestr = NULL; 
	WCHAR* pwszSchemaDependNameRestr = NULL; 
	char   szParentNameRestr[MAX_NAME_SIZE]; 
	char   szDependNameRestr[MAX_NAME_SIZE]; 
	int    iParentNameRestr = -1;
	int    iDependNameRestr = -1;
	
	bool bCatalogParentNameRestr = false;
	bool bCatalogDependNameRestr = false;
	bool bSchemaParentNameRestr = false;
	bool bSchemaDependNameRestr = false;
	bool bParentNameRestr = false;
	bool bDependNameRestr = false;

	//Only BSTR is accepted
	if ((cRestrictions >= 1 && rgRestrictions[0].vt != VT_BSTR && rgRestrictions[0].vt != VT_EMPTY) ||
		(cRestrictions >= 2 && rgRestrictions[1].vt != VT_BSTR && rgRestrictions[1].vt != VT_EMPTY) || 
		(cRestrictions >= 3 && rgRestrictions[2].vt != VT_BSTR && rgRestrictions[2].vt != VT_EMPTY) ||
		(cRestrictions >= 4 && rgRestrictions[3].vt != VT_BSTR && rgRestrictions[3].vt != VT_EMPTY) ||
		(cRestrictions >= 5 && rgRestrictions[4].vt != VT_BSTR && rgRestrictions[4].vt != VT_EMPTY) || 
		(cRestrictions >= 6 && rgRestrictions[5].vt != VT_BSTR && rgRestrictions[5].vt != VT_EMPTY)	)
	{
		TRACE( "CSchema::InitForeignKeys: restr fails" );
		return E_INVALIDARG;
	}

	//Extract PK_TABLE_CATALOG restriction
	if (cRestrictions >= 1 && rgRestrictions[0].vt == VT_BSTR)
	{
		pwszCatalogParentNameRestr = rgRestrictions[0].bstrVal ? rgRestrictions[0].bstrVal : L"";
		bCatalogParentNameRestr = true;
		TRACE2( "  Catalog of parent restr [%S]", pwszCatalogParentNameRestr );
	}

	//Extract FK_TABLE_CATALOG restriction
	if (cRestrictions >= 4 && rgRestrictions[3].vt == VT_BSTR)
	{
		pwszCatalogDependNameRestr = rgRestrictions[3].bstrVal ? rgRestrictions[3].bstrVal : L"";
		bCatalogDependNameRestr = true;
		TRACE2( "  Catalog of dependent restr [%S]", pwszCatalogDependNameRestr );
	}

	//Extract PK_TABLE_SCHEMA restriction
	if (cRestrictions >= 2 && rgRestrictions[1].vt == VT_BSTR)
	{
		pwszSchemaParentNameRestr = rgRestrictions[1].bstrVal ? rgRestrictions[1].bstrVal : L"";
		if( !wcsicmp( pwszSchemaParentNameRestr, INFORMATION_SCHEMA ) )
			pwszSchemaParentNameRestr = L"";
		bSchemaParentNameRestr = true;
		TRACE2( "  Schema of parent restr [%S]", pwszSchemaParentNameRestr );
	}

	//Extract FK_TABLE_SCHEMA restriction
	if (cRestrictions >= 5 && rgRestrictions[4].vt == VT_BSTR)
	{
		pwszSchemaDependNameRestr = rgRestrictions[4].bstrVal ? rgRestrictions[4].bstrVal : L"";
		if( !wcsicmp( pwszSchemaDependNameRestr, INFORMATION_SCHEMA ) )
			pwszSchemaDependNameRestr = L"";
		bSchemaDependNameRestr = true;
		TRACE2( "  Schema of dependent restr [%S]", pwszSchemaDependNameRestr );
	}

	// Load catalog information if necessary
	if( (bCatalogParentNameRestr || bSchemaParentNameRestr) && m_pCDataSource )
		m_pCDataSource->CheckAndTrackOneDatabase( pwszCatalogParentNameRestr, pwszSchemaParentNameRestr);
	if( (bCatalogDependNameRestr || bSchemaDependNameRestr) && m_pCDataSource )
		m_pCDataSource->CheckAndTrackOneDatabase( pwszCatalogDependNameRestr, pwszSchemaDependNameRestr);

	//Extract PK_TABLE_NAME restriction
	if (cRestrictions >= 3 && rgRestrictions[2].vt == VT_BSTR)
	{
		if( rgRestrictions[2].bstrVal == NULL )
			*szParentNameRestr = '\0';
		else
			WideCharToMultiByte(CP_ACP, 0,
							rgRestrictions[2].bstrVal, -1,
							szParentNameRestr, MAX_NAME_SIZE,
							NULL, NULL );
		if( *szParentNameRestr == '#' )
			iParentNameRestr = atoi( szParentNameRestr + 1 );
		
		bParentNameRestr = true;
		TRACE2( "  parent constraint name [%s]", szParentNameRestr );
	}
	
	//Extract FK_TABLE_NAME restriction
	if (cRestrictions >= 6 && rgRestrictions[5].vt == VT_BSTR)
	{
		if( rgRestrictions[5].bstrVal == NULL )
			*szDependNameRestr = '\0';
		else
			WideCharToMultiByte(CP_ACP, 0,
							rgRestrictions[5].bstrVal, -1,
							szDependNameRestr, MAX_NAME_SIZE,
							NULL, NULL );
		if( *szDependNameRestr == '#' )
			iDependNameRestr = atoi( szDependNameRestr + 1 );
		
		bDependNameRestr = true;
		TRACE2( "  dependent constraint name [%s]", szDependNameRestr );
	}

	TRACE( "CSchema::InitForeignKeys: before getting data" );

	// Get total number of indexes
	int iTotalRelations = 0;
	int iTotalMetas;
	CSwstMeta* pSwstMeta;
	if( FAILED( m_pSwstMetaHolder->GetMaxNumber( &iTotalMetas ) ) )
		return E_FAIL;
	for( int j = 0; j < iTotalMetas; j++ )
		if( m_pSwstMetaHolder->At( j, &pSwstMeta ) == S_OK &&
			(!bCatalogParentNameRestr || !wcsicmp( pwszCatalogParentNameRestr, pSwstMeta->m_CatalogOwner.m_wszCatalog ) ) &&
			(!bSchemaParentNameRestr  || !wcsicmp( pwszSchemaParentNameRestr, pSwstMeta->m_CatalogOwner.m_wszOwner ) )  &&
			(!bCatalogDependNameRestr || !wcsicmp( pwszCatalogDependNameRestr, pSwstMeta->m_CatalogOwner.m_wszCatalog ) ) &&
			(!bSchemaDependNameRestr  || !wcsicmp( pwszSchemaDependNameRestr, pSwstMeta->m_CatalogOwner.m_wszOwner ) ) )
				for( int k = 0; k < pSwstMeta->m_siRelations; k++ )
					iTotalRelations += min( pSwstMeta->m_pRelations[ k ].m_wParentIndexes, 
											pSwstMeta->m_pRelations[ k ].m_wDependIndexes );
	
	//Init data
	m_pForeignKeyInfo = new FOREIGNKEYINFO[ iTotalRelations ? iTotalRelations : 1];
	if( m_pForeignKeyInfo == NULL )
		return E_OUTOFMEMORY;

	//Copy infos to buffer
	m_cRows = 0;
	for( j = 0; j < iTotalMetas; j++ )
		if( m_pSwstMetaHolder->At( j, &pSwstMeta ) == S_OK &&
			(!bCatalogParentNameRestr || !wcsicmp( pwszCatalogParentNameRestr, pSwstMeta->m_CatalogOwner.m_wszCatalog ) ) &&
			(!bSchemaParentNameRestr  || !wcsicmp( pwszSchemaParentNameRestr, pSwstMeta->m_CatalogOwner.m_wszOwner ) )  &&
			(!bCatalogDependNameRestr || !wcsicmp( pwszCatalogDependNameRestr, pSwstMeta->m_CatalogOwner.m_wszCatalog ) ) &&
			(!bSchemaDependNameRestr  || !wcsicmp( pwszSchemaDependNameRestr, pSwstMeta->m_CatalogOwner.m_wszOwner ) ) )
		{
			for (int k = 0; k < pSwstMeta->m_siRelations; k++)
			{
				// All index segments belong to same table!
				
				// Check PK_TABLE_NAME restriction
				if (bParentNameRestr)
				{
					if( stricmp( pSwstMeta->m_pRelations[ k ].m_ppParentIndex[ 0 ]->m_pSwstField->m_pSwstFile->m_szName, szParentNameRestr ) )
						continue;
				}

				// Check FK_TABLE_NAME restriction
				if (bDependNameRestr)
				{
					if( stricmp( pSwstMeta->m_pRelations[ k ].m_ppDependIndex[ 0 ]->m_pSwstField->m_pSwstFile->m_szName, szDependNameRestr ) )
						continue;
				}

				for( int i = 0; i < min( pSwstMeta->m_pRelations[ k ].m_wParentIndexes, pSwstMeta->m_pRelations[ k ].m_wDependIndexes ); i++ )
				{
					// Save an info for every couple of segments
					m_pForeignKeyInfo[m_cRows].pRelate = &pSwstMeta->m_pRelations[ k ];
					m_pForeignKeyInfo[m_cRows].iIdx = i;
					
					// Completed with this row. Go to next one.
					m_cRows++;
				}
			}
		}
	
	//Default Sort Order: FK_TABLE_CATALOG, FK_TABLE_SCHEMA, FK_TABLE_NAME
	qsort( m_pForeignKeyInfo, m_cRows, sizeof(m_pForeignKeyInfo[0]), CompareFOREIGNKEY_ETypeRestrs );
	/*FOREIGNKEYINFO info;
	for (j = m_cRows - 1; j > 0; j--)
	{
		for (int k = 0; k < j; k++)
		{
			int iCatDep = wcsicmp(m_pForeignKeyInfo[k].pRelate->m_ppDependIndex[ m_pForeignKeyInfo[k].iIdx ]->
									m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszCatalog, 
								  m_pForeignKeyInfo[k + 1].pRelate->m_ppDependIndex[ m_pForeignKeyInfo[k + 1].iIdx ]->
								    m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszCatalog );
			int iOwnDep = wcsicmp(m_pForeignKeyInfo[k].pRelate->m_ppDependIndex[ m_pForeignKeyInfo[k].iIdx ]->
									m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszOwner, 
								  m_pForeignKeyInfo[k + 1].pRelate->m_ppDependIndex[ m_pForeignKeyInfo[k + 1].iIdx ]->
								    m_pSwstField->m_pSwstFile->m_pCatalogOwner->m_wszOwner );
			int iTabDep = stricmp( m_pForeignKeyInfo[k].pRelate->m_ppDependIndex[ m_pForeignKeyInfo[k].iIdx ]->
								    m_pSwstField->m_pSwstFile->m_szName,
								  m_pForeignKeyInfo[k + 1].pRelate->m_ppDependIndex[ m_pForeignKeyInfo[k + 1].iIdx ]->
								    m_pSwstField->m_pSwstFile->m_szName );

			if( ( iCatDep > 0 ) ||
				( iCatDep==0 && iOwnDep > 0 ) || 
				( iCatDep==0 && iOwnDep==0 && iTabDep > 0 ) )
			{
				info = m_pForeignKeyInfo[k];
				m_pForeignKeyInfo[k] = m_pForeignKeyInfo[k + 1];
				m_pForeignKeyInfo[k + 1] = info;
			}
		}
	}*/

	TRACE2( "CSchema::InitForeignKeys: completing (total %d)", m_cRows );
	return S_OK;
}

