/* Copyright (C) 2000, 2001  SWsoft, Singapore                                  
 *                                                                              
 *  This program is free software; you can redistribute it and/or modify        
 *  it under the terms of the GNU General Public License as published by        
 *  the Free Software Foundation; either version 2 of the License, or           
 *  (at your option) any later version.                                         
 *                                                                              
 *  This program is distributed in the hope that it will be useful,             
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of              
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               
 *  GNU General Public License for more details.                                
 *                                                                              
 *  You should have received a copy of the GNU General Public License           
 *  along with this program; if not, write to the Free Software                 
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA   
 */

// Includes ------------------------------------------------------------------

#include "hfiles.h"
#include "headers.h"

// CImpIRowsetLocate::Compare
//
// @mfunc	Compares two bookmarks.
//
// @desc	HRESULT
//		@flag	S_OK			| The method succeeded. 
//		@flag	E_FAIL			| A provider-specific error occurred. 
//		@flag	E_INVALIDARG	| cbBookmark1 or cbBookmark2 was zero. 
//		@flag	DB_E_BADBOOKMARK| pBookmark1 or pBookmark2 was zero
//		@flag	E_POINTER		| pComparison was zero

//	DBCOMPARE_LT				The first bookmark is before the second.
//	DBCOMPARE_EQ				The two bookmarks are equal.
//	DBCOMPARE_GT				The first bookmark is after the second.
//	DBCOMPARE_NE				The bookmarks are not equal and not ordered.
//	DBCOMPARE_NOTCOMPARABLE		The two bookmarks cannot be compared.

STDMETHODIMP	CImpIRowsetLocate::Compare(
		HCHAPTER	hReserved, 
		ULONG		cbBookmark1, 
		const BYTE*	pBookmark1,
		ULONG		cbBookmark2,
		const BYTE*	pBookmark2,
		DBCOMPARE*	pComparison)
{
	INTERFACE_METHOD_START( "IRowsetLocate::Compare" );

	ULONG	PropIndex;

	BOOL	bOrderdBookmarks = FALSE;

	// Bookmark length can not be Null
	if (cbBookmark1 == NULL || cbBookmark2 == NULL)
		return E_INVALIDARG;

	// Pointer to comparioson result value can not be Null
	if (pComparison == NULL)
		return E_INVALIDARG;

	// Bookmark length can be either "our custom" or standart
	if ( !(cbBookmark1 == PHYSICAL_POS_BUF_LEN * m_pObj->m_pData->bookmarks() || cbBookmark1 == 1))
		return DB_E_BADBOOKMARK;

	if ( !(cbBookmark2 == PHYSICAL_POS_BUF_LEN * m_pObj->m_pData->bookmarks() || cbBookmark2 == 1))
		return DB_E_BADBOOKMARK;

	// Bookmark value can not be Null
	if (pBookmark1 == NULL || pBookmark2 == NULL)
		return DB_E_BADBOOKMARK;

	// Bookmark value can not be DBBMK_INVALID
	if (*pBookmark1 == DBBMK_INVALID || *pBookmark2 == DBBMK_INVALID)
		return DB_E_BADBOOKMARK;

	// ???? Where cbBookmark1 and cbBookmark2 should be used

	// Get DBPROP_ORDEREDBOOKMARKS property value. Though it is useless for now ;)
	if (m_pObj->m_pUtilProp->GetPropIndex(DBPROP_ORDEREDBOOKMARKS, &PropIndex) == TRUE)
		bOrderdBookmarks = m_pObj->m_pUtilProp->m_rgproperties[PropIndex].boolVal;

	if (bOrderdBookmarks)
	{
		// ordered bookmarks
		if	(*(ULONG *)pBookmark1 == *(ULONG *)pBookmark2)
			*pComparison = DBCOMPARE_EQ;
		else if ((*(ULONG *)pBookmark1 < *(ULONG *)pBookmark2))
			*pComparison = DBCOMPARE_LT;
		else
			*pComparison = DBCOMPARE_GT;
	}
	else
	{
		// unordered bookmarks
		if	(*(ULONG *)pBookmark1 == *(ULONG *)pBookmark2)
			*pComparison = DBCOMPARE_EQ;
		else
			*pComparison = DBCOMPARE_NE;
	}
	
	return S_OK;

	INTERFACE_METHOD_END();
}

// CImpIRowsetLocate::GetRowsAt
//
// @mfunc	Fetches rows starting with the row specified by an 
//			offset from a bookmark.
//
// @desc	HRESULT
//		@flag	S_OK			| The method succeeded. 
//		@flag	E_FAIL			| A provider-specific error occurred. 
//		@flag	E_INVALIDARG	| cbBookmark was zero. 
//		@flag	E_POINTER		| pcRowsObtained or prghRows was zero
//		@flag	DB_E_BADBOOKMARK| pBookmark was zero

STDMETHODIMP	CImpIRowsetLocate::GetRowsAt(
			HWATCHREGION	hReserved1,		// IN
			HCHAPTER		hReserved2,		// IN
			ULONG			cbBookmark,		// IN
			const BYTE*		pBookmark,		// IN
			LONG			lRowsOffset,	// IN
			LONG			cRows,			// IN
			ULONG*			pcRowsObtained,	// OUT
			HROW**			prghRows)		// OUT
{
	INTERFACE_METHOD_START("IRowsetLocate::GetRowsAt");

	ULONG	cRealRowsObtained = 0;
	ULONG	PropIndex;

	BOOL	bCanFetchBackwards = FALSE;
	BOOL	bCanScrollBackwards = FALSE;
	BOOL	bBookmarkSkipped = FALSE;

	HRESULT hr = (S_OK);

	if (m_pObj->m_bNotReentrant)
		return DB_E_NOTREENTRANT;

	if (cbBookmark == NULL)
		return E_INVALIDARG;
	
	if (pBookmark == NULL)
		return DB_E_BADBOOKMARK;
	
	// Bookmark length can be either "our custom" or standard
	if ( !(cbBookmark == PHYSICAL_POS_BUF_LEN * m_pObj->m_pData->bookmarks() || cbBookmark == 1))
		return DB_E_BADBOOKMARK;

	if (pcRowsObtained == NULL || prghRows == NULL)
		return E_INVALIDARG;

	// Get properties. Though it is useless for now ;)
	if (m_pObj->m_pUtilProp->GetPropIndex(DBPROP_CANFETCHBACKWARDS, &PropIndex) == TRUE)
		bCanFetchBackwards = m_pObj->m_pUtilProp->m_rgproperties[PropIndex].boolVal;

	if (m_pObj->m_pUtilProp->GetPropIndex(DBPROP_CANSCROLLBACKWARDS, &PropIndex) == TRUE)
		bCanScrollBackwards = m_pObj->m_pUtilProp->m_rgproperties[PropIndex].boolVal;

	if (m_pObj->m_pUtilProp->GetPropIndex(DBPROP_BOOKMARKSKIPPED, &PropIndex) == TRUE)
		bBookmarkSkipped = m_pObj->m_pUtilProp->m_rgproperties[PropIndex].boolVal;

    // No-op case always succeeds.
	if (cRows == 0 && lRowsOffset == 0)
		return S_OK;

	// even if we have end of rowset status, we will try to locate
	// clear "end of cursor" flag
	m_pObj->m_dwStatus &= ~STAT_ENDOFCURSOR;

	//Moving to bookmark
	hr = m_pObj->MoveToBookmark((void*)pBookmark, cbBookmark);
	if (hr != S_OK)
		return hr;

	// Do not make sense to go on if cRows == 0
	// it is according to specification !
	if (cRows == 0)
		return (S_OK);

	// Get rows
	return m_pObj->m_pIRowset->GetNextRows(false, NULL, lRowsOffset, cRows, pcRowsObtained, prghRows);

	INTERFACE_METHOD_END();
}

//------------------------------------------------------------------------------
// CImpIRowsetLocate::GetRowsByBookmark
//
// @cmember Fetches the rows that match the specified bookmarks.
//
// @desc
//		@flag	S_OK			| The method succeeded. 
//		@flag	E_FAIL			| A provider-specific error occurred. 
//		@flag	E_INVALIDARG	| cbBookmark or rghRows was zero. 
//
STDMETHODIMP	CImpIRowsetLocate::GetRowsByBookmark (
			HCHAPTER		hReserved,
			ULONG			cRows,
			const ULONG		rgcbBookmarks[],
			const BYTE*		rgpBookmarks[],
			HROW			rghRows[],
			DBROWSTATUS		rgRowStatus[]
)
{
	INTERFACE_METHOD_START( "IRowsetLocate::GetRowsByBookmark" );

	ULONG*	pBookmark;

	HRESULT	Status;
	HRESULT	hr;

	ULONG	ulPropIndex;
	BOOL	bCanHoldRows = FALSE;
	BOOL	bRowsetIdentity = FALSE;

	ULONG	irow, krow;
	ULONG	iSlot;
	ULONG	cFetchedRows;

	BOOL	fDoStatus = TRUE;
	BOOL	bRowFound = FALSE;

	PROWBUFF pRowBuff;
	PROWBUFF tmpRowBuff;

	BOOL	bSingleGranularity	= TRUE;
	HROW*	phActivatedRows		= NULL;
	ULONG	ulActivatedRows		= 0;

	if (rghRows == NULL || rgcbBookmarks == NULL || cRows < 0)
		return E_INVALIDARG;

	if (m_pObj->m_bNotReentrant)
		return DB_E_NOTREENTRANT;
	
	if (rgpBookmarks == NULL)
		return DB_E_BADBOOKMARK;

	if (rgRowStatus == NULL)
		fDoStatus = false;

	//Load properties
	if (m_pObj->m_pUtilProp->GetPropIndex(DBPROP_CANHOLDROWS, &ulPropIndex) == TRUE)
		bCanHoldRows = m_pObj->m_pUtilProp->m_rgproperties[ulPropIndex].boolVal;

	if (m_pObj->m_pUtilProp->GetPropIndex(DBPROP_NOTIFICATIONGRANULARITY, &ulPropIndex) == TRUE)
		bSingleGranularity = m_pObj->m_pUtilProp->m_rgproperties[ulPropIndex].boolVal;

	// Check if already allocated rows must be released
	if (!bCanHoldRows && m_pObj->m_ulRowRefCount != 0)
		return DB_E_ROWSNOTRELEASED;

	//Create temporary HROW array
	if (!bSingleGranularity)
	{
		phActivatedRows = new HROW[cRows];
		
		if (phActivatedRows == NULL)
			return E_OUTOFMEMORY;
	}

    assert( m_pObj->m_rgbRowData );

	hr = S_OK;
	cFetchedRows = 0;
    for ( irow = 0; irow < cRows; irow++ )
    {
		pBookmark = (ULONG*)rgpBookmarks[irow];

		// This couldn't be predefined bookmark
		if ( (pBookmark == NULL) || 
			 ((rgcbBookmarks[irow] == 1) && (*((BYTE*)rgpBookmarks[irow]) == DBBMK_FIRST)) || 
			 ((rgcbBookmarks[irow] == 1) && (*((BYTE*)rgpBookmarks[irow]) == DBBMK_LAST )) )
			Status = DBROWSTATUS_E_INVALID;
		else if (rgcbBookmarks[irow] == NULL)
			Status = DBROWSTATUS_E_INVALID;
		else 
		{
			//Check if row is loaded already
			if (bRowsetIdentity)
			{
				bRowFound = FALSE;
				for (krow = 1; krow < ACTIVE_ROWS_LIMIT; krow++)
				{
					if (m_pObj->m_SlotRef[ krow ].Status == 0)
						continue;

					tmpRowBuff = m_pObj->GetRowBuff(m_pObj->m_SlotRef[ krow ].hRow);
					if( !memcmp( tmpRowBuff->dwBmks, pBookmark, tmpRowBuff->cbBmk ) )
					{
						tmpRowBuff->ulRefCount++;
						m_pObj->m_ulRowRefCount++;
						
						rghRows[irow] = m_pObj->m_SlotRef[krow].hRow;

						hr = S_OK;
						bRowFound = TRUE;
						break;
					}
				}
				if (bRowFound)
					continue;
			}

			//Fetch specified row
			if (m_pObj->m_pData->MoveBookmark( pBookmark ) == S_OK)
			{
				if ( m_pObj->IsSlotLimitReached() )
				{
					hr = DB_S_ROWLIMITEXCEEDED;
					break;
				}

				// Find next free row
				iSlot = m_pObj->GetFreeSlot();
				assert (iSlot != 0xFFFFFFFF);

				pRowBuff = m_pObj->GetRowBuff(m_pObj->m_SlotRef[iSlot].hRow);
	
				//Fetch data
				hr = m_pObj->m_pData->GetRow(m_pObj->m_rgdwDataOffsets,	(BYTE*)pRowBuff);
				if (hr == S_OK)
				{
					pRowBuff->ulRefCount = 1;
					pRowBuff->dwBmk = *(ULONG*)pBookmark;
					pRowBuff->cbBmk = BMKS_SIZE * m_pObj->m_pData->bookmarks();
					pRowBuff->dwBmkStatus = BMKS_ROW_IN_BUFFER;

					m_pObj->m_SlotRef[ iSlot ].Status = 1;
					m_pObj->m_ulRowRefCount++;

					rghRows[irow] = m_pObj->m_SlotRef[iSlot].hRow;

					cFetchedRows++;
					m_pObj->m_ulRowRefCount++;
					Status = DBROWSTATUS_S_OK;

					if (bSingleGranularity)
					{
						//Send notification
						m_pObj->OnRowChange
									(
									1, //Count of rows
									&m_pObj->m_SlotRef[iSlot].hRow, //hRow
									DBREASON_ROW_ACTIVATE,
									DBEVENTPHASE_DIDEVENT,
									TRUE);				
					}
					else
					{
						//Store activated HROW to send notification later
						phActivatedRows[ulActivatedRows++] = m_pObj->m_SlotRef[iSlot].hRow;
					}
				}
				else
					Status = DBROWSTATUS_E_INVALID;
			}
	        else
				Status = DBROWSTATUS_E_INVALID;
		}

		if (Status != DBROWSTATUS_S_OK)
			rghRows[irow] = DB_NULL_HROW;

		if (fDoStatus)
			rgRowStatus[irow] = Status;
	}

    m_pObj->m_cRows += cFetchedRows;

	// Send notification on DBREASON_ROW_ACTIVATE
	if (!FAILED(hr) && ulActivatedRows > 0 && !bSingleGranularity)
		m_pObj->OnRowChange(ulActivatedRows, phActivatedRows, DBREASON_ROW_ACTIVATE, DBEVENTPHASE_DIDEVENT, TRUE);			

	delete [] phActivatedRows;

	return hr;

	INTERFACE_METHOD_END();
}

// CImpIRowsetLocate::Hash
//
// @mfunc Returns hash values for the specified bookmarks.
//
// @desc
//

STDMETHODIMP	CImpIRowsetLocate::Hash (
			HCHAPTER		hReserved,
			ULONG			cBookmarks,
			const ULONG		rgcbBookmarks[],
			const BYTE*		rgpBookmarks[],
			DWORD			rgHashedValues[],
			DBROWSTATUS		rgBookmarkStatus[])
{
	INTERFACE_METHOD_START( "IRowsetLocate::Hash " );

	ULONG	cBmks = 0;
	ULONG*	pBookmark;
	bool	fDoStatus = true;

	if (rgcbBookmarks == NULL	 ||
		rgHashedValues == NULL   || 
		rgpBookmarks == NULL)
		return E_INVALIDARG;

	if (rgBookmarkStatus == NULL)
		fDoStatus = false;

	while ( cBookmarks-- )
	{
		if (rgpBookmarks[cBmks] == NULL)
		{
			if (fDoStatus)
				rgBookmarkStatus[cBmks++] = DBROWSTATUS_E_INVALID;
			continue;
		}

		if (rgcbBookmarks[cBmks] == NULL)
		{
			if (fDoStatus)
				rgBookmarkStatus[cBmks++] = DBROWSTATUS_E_INVALID;
			continue;
		}

		pBookmark = (ULONG *)rgpBookmarks[cBmks];

		if ( ((rgcbBookmarks[cBmks] == 1) && (*((BYTE*)rgpBookmarks[cBmks]) == DBBMK_FIRST))  ||  
			 ((rgcbBookmarks[cBmks] == 1) && (*((BYTE*)rgpBookmarks[cBmks]) == DBBMK_LAST )) )
		{
			if (fDoStatus)
				rgBookmarkStatus[cBmks++] = DBROWSTATUS_E_INVALID;
			continue;
		}

		rgHashedValues[cBmks] = (DWORD)*rgpBookmarks[cBmks];
		if (fDoStatus)
			rgBookmarkStatus[cBmks++] = DBROWSTATUS_S_OK;
	}
	return S_OK;

	INTERFACE_METHOD_END();
}


STDMETHODIMP	CImpIRowsetLocate::GetData 
    (
    HROW        hRow,       //@parm IN | Row Handle
    HACCESSOR   hAccessor,  //@parm IN | Accessor to use
    void       *pData       //@parm OUT | Pointer to buffer where data should go.
    )
{
	INTERFACE_METHOD_START( "IRowsetLocate::GetData");
	
	return m_pObj->m_pIRowset->GetData(hRow, hAccessor, pData);

	INTERFACE_METHOD_END();
}

STDMETHODIMP	CImpIRowsetLocate::GetNextRows
    (
    HCHAPTER   hReserved,       //@parm IN  | Reserved for future use. Ingored.
    LONG       cRowsToSkip,     //@parm IN  | Rows to skip before reading
    LONG       cRows,           //@parm IN  | Number of rows to fetch
    ULONG      *pcRowsObtained, //@parm OUT | Number of rows obtained
    HROW       **prghRows       //@parm OUT | Array of Hrows obtained
    )
{
	INTERFACE_METHOD_START( "IRowsetLocate::GetNextRows" );
		
	return m_pObj->m_pIRowset->GetNextRows(hReserved, cRowsToSkip, cRows, pcRowsObtained, prghRows);

	INTERFACE_METHOD_END();
}

STDMETHODIMP	CImpIRowsetLocate::ReleaseRows
    (
    ULONG			cRows,			//@parm IN  | Number of rows to release
    const HROW		rghRows[],		//@parm IN  | Array of handles of rows to be released
	DBROWOPTIONS	rgRowOptions[],	//@parm IN  | Additional Options
    ULONG			rgRefCounts[],	//@parm OUT | Array of refcnts for the rows
	DBROWSTATUS     rgRowStatus[]	//@parm OUT | Array of row status
    )
{
	INTERFACE_METHOD_START( "IRowsetLocate::ReleaseRows" );
	
	return m_pObj->m_pIRowset->ReleaseRows(cRows, rghRows, rgRowOptions, rgRefCounts, rgRowStatus);

	INTERFACE_METHOD_END();
}

STDMETHODIMP	CImpIRowsetLocate::RestartPosition
    (
    HCHAPTER    hReserved        //@parm IN | Reserved for future use.  Ignored.
    )
{
	INTERFACE_METHOD_START( "IRowsetLOcate::RestartPosition" )
	
	return m_pObj->m_pIRowset->RestartPosition(hReserved);

	INTERFACE_METHOD_END();
}

STDMETHODIMP	CImpIRowsetLocate::AddRefRows
    (
    ULONG           cRows,          // @parm IN     | Number of rows to refcount
    const HROW      rghRows[],      // @parm IN     | Array of row handles to refcount
    ULONG           rgRefCounts[],  // @parm OUT    | Array of refcounts
    DBROWSTATUS     rgRowStatus[]   // @parm OUT    | Array of row status
	)
{
	INTERFACE_METHOD_START( "IRowsetLocate::AddRefRows" );
	
	return m_pObj->m_pIRowset->AddRefRows(cRows, rghRows, rgRefCounts, rgRowStatus);

	INTERFACE_METHOD_END();
}
