/* Module:          SQLBindCol.c
 *
 * Description:     Binds application data buffers to columns in the result set.
 *
 * Classes:         
 *
 * API functions:   SQLBindCol
 *
 * Comments:        See "notice.txt" for copyright and license information.
 *
 */

#include "driver.h"

SQLRETURN SQL_API SQLBindCol(
					 SQLHSTMT			hDrvStmt,
					 SQLUSMALLINT		nCol,
					 SQLSMALLINT		nTargetType,
					 SQLPOINTER			pTargetValue,
					 SQLINTEGER			nTargetValueMax,
					 SQLINTEGER			*pnLengthOrIndicator
					 )
					 
{
	StatementClass *stmt = (StatementClass *) hDrvStmt;
	static char *func="SQLBindCol";

	mylog( "%s: entering...\n", func);

	mylog("**** SQLBindCol: stmt = %u, nCol = %d\n", stmt, nCol);

	if ( ! stmt)
	{
		SC_log_error(func, "", NULL);
		return SQL_INVALID_HANDLE;
	}


	SC_clear_error(stmt);
    
	if( stmt->status == STMT_EXECUTING)
	{
		stmt->errormsg = "Can't bind columns while statement is still executing.";
		stmt->errornumber = STMT_SEQUENCE_ERROR;
		SC_log_error(func, "", stmt);
		return SQL_ERROR;
	}

	/*	If the bookmark column is being bound, then just save it */
	if (nCol == 0)
	{
		
		if (pTargetValue == NULL)
		{
			stmt->bookmark.buffer = NULL;
			stmt->bookmark.used = NULL;
		}
		else
		{
			/*	Make sure it is the bookmark data type */
			if ( nTargetType != SQL_C_BOOKMARK)
			{
				stmt->errormsg = "Column 0 is not of type SQL_C_BOOKMARK";
				stmt->errornumber = STMT_PROGRAM_TYPE_OUT_OF_RANGE;
				SC_log_error(func, "", stmt);
				return SQL_ERROR;
			}

			stmt->bookmark.buffer = pTargetValue;
			stmt->bookmark.used = pnLengthOrIndicator;
		}
		return SQL_SUCCESS;
	}

	/*	allocate enough bindings if not already done */
	/*	Most likely, execution of a statement would have setup the  */
	/*	necessary bindings. But some apps call BindCol before any */
	/*	statement is executed. */
	if ( nCol > stmt->bindings_allocated)
	{
		extend_bindings(stmt, nCol);
	}

	/*	check to see if the bindings were allocated */
	if ( ! stmt->bindings)
	{
		stmt->errormsg = "Could not allocate memory for bindings.";
		stmt->errornumber = STMT_NO_MEMORY_ERROR;
		SC_log_error(func, "", stmt);
		return SQL_ERROR;
	}

	nCol--;		/* use zero based col numbers from here out */

	/*	Reset for SQLGetData */
	stmt->bindings[nCol].data_left = -1;

	if (pTargetValue == NULL)
	{
		/* we have to unbind the column */
		stmt->bindings[nCol].buflen = 0;
		stmt->bindings[nCol].buffer = NULL;
		stmt->bindings[nCol].used =   NULL;
		stmt->bindings[nCol].returntype = SQL_C_CHAR;
	} 
	else 
	{
		/* ok, bind that column */
		stmt->bindings[nCol].buflen     = nTargetValueMax;
		stmt->bindings[nCol].buffer     = pTargetValue;
		stmt->bindings[nCol].used       = pnLengthOrIndicator;
		stmt->bindings[nCol].returntype = nTargetType;

		mylog("       bound buffer[%d] = %u\n", nCol, stmt->bindings[nCol].buffer);
	}

	return SQL_SUCCESS;
}


