#include <windows.h>
#include <stdio.h>
#include <assert.h>
#include "main.h"
//------------------------------------------------------------------------------
// Check if the child process is running.
// On NT/2000 the handle must have PROCESS_QUERY_INFORMATION access.
//------------------------------------------------------------------------------
BOOL ConsoleRedirector::IsChildRunning() const
{
	DWORD dwExitCode;
    if (hChildProcess == NULL) 
		return FALSE;
	::GetExitCodeProcess(hChildProcess, &dwExitCode);
	return (dwExitCode == STILL_ACTIVE) ? TRUE: FALSE;
}
//------------------------------------------------------------------------------
ConsoleRedirector::ConsoleRedirector()
{
	hStdIn = NULL;
 	hStdOut = NULL;
 	hStdErr = NULL;
	
	hStdInWrite = NULL;
	hStdOutRead = NULL;
	hStdErrRead = NULL;
	
	hStdOutThread = NULL;
	hStdErrThread = NULL;
}
//------------------------------------------------------------------------------
ConsoleRedirector::~ConsoleRedirector()
{
	Cleanup();
}
//------------------------------------------------------------------------------
#define WIN_EXECPTION(EXP)								\
	char buffer[1024];									\
	sprintf(buffer, "%d : %s", __LINE__, __FILE__);		\
	MessageBox (0, buffer, EXP, MB_ICONEXCLAMATION | MB_OK)
//------------------------------------------------------------------------------
BOOL ConsoleRedirector::CreatePipes()
{
	HANDLE hProcess = ::GetCurrentProcess();
	HANDLE hStdInWriteTmp, hStdOutReadTmp, hStdErrReadTmp;
	SECURITY_ATTRIBUTES sa;

	// Set up the security attributes struct.
	sa.nLength= sizeof(SECURITY_ATTRIBUTES);
	sa.lpSecurityDescriptor = NULL;
	sa.bInheritHandle = TRUE;
	
	try
	{
		// Create the child stdin pipe.
		::CreatePipe(&hStdIn, &hStdInWriteTmp, &sa, 0);

		// Create the child stdout pipe.
		::CreatePipe(&hStdOutReadTmp, &hStdOut, &sa, 0);

		// Create the child stderr pipe.
		::CreatePipe(&hStdErrReadTmp, &hStdErr, &sa, 0);

		// Create new stdin write, stdout and stderr read handles.
		// Set the properties to FALSE. Otherwise, the child inherits the
		// properties and, as a result, non-closeable handles to the pipes
		// are created.

		::DuplicateHandle(hProcess, hStdInWriteTmp,
			hProcess, &hStdInWrite, 0, FALSE, DUPLICATE_SAME_ACCESS);

		::DuplicateHandle(hProcess, hStdOutReadTmp,
			hProcess, &hStdOutRead, 0, FALSE, DUPLICATE_SAME_ACCESS);

		::DuplicateHandle(hProcess, hStdErrReadTmp,
			hProcess, &hStdErrRead, 0, FALSE, DUPLICATE_SAME_ACCESS);

		// Close inheritable copies of the handles you do not want to be
		// inherited.

		::CloseHandle(hStdInWriteTmp);
		::CloseHandle(hStdOutReadTmp);
		::CloseHandle(hStdErrReadTmp);

	}
    catch (WinException e)
    {
        MessageBox (0, e.GetMessage (), MSG_EXCEPTION, MB_ICONEXCLAMATION | MB_OK);
    }
    catch (...)
    {
		WIN_EXECPTION(MSG_EXCEPTION);
        return FALSE;
    }
    return TRUE;
}
//------------------------------------------------------------------------------
BOOL ConsoleRedirector::CloseChildHandles()
{
	try
	{
		if( hStdIn != NULL )
			::CloseHandle(hStdIn);
		hStdIn = NULL;
		
		if( hStdOut != NULL )
			::CloseHandle(hStdOut);
		hStdOut = NULL;
		
		if(hStdErr != NULL)
			::CloseHandle(hStdErr);
		hStdErr = NULL;
		return TRUE;
	}
	catch (WinException e)
	{
		MessageBox (0, e.GetMessage (), MSG_EXCEPTION, MB_ICONEXCLAMATION | MB_OK);
	}
	catch (...)
	{
		WIN_EXECPTION(MSG_EXCEPTION);
        return FALSE;
	}
	return FALSE;
}
//------------------------------------------------------------------------------
BOOL ConsoleRedirector::CloseHandles()
{
	try
	{
		if (hStdInWrite != NULL)
			::CloseHandle(hStdInWrite);
		hStdInWrite = NULL;
		if (hStdOutRead != NULL)
			::CloseHandle(hStdOutRead);
		hStdOutRead = NULL;
		if (hStdErrRead != NULL)
			::CloseHandle(hStdErrRead);
		hStdErrRead = NULL;
	}
	catch (WinException e)
	{
		MessageBox (0, e.GetMessage (), MSG_EXCEPTION, MB_ICONEXCLAMATION | MB_OK);
	}
	catch (...)
	{
		WIN_EXECPTION(MSG_EXCEPTION);
 	    return -1;
	}
}
//------------------------------------------------------------------------------
void ConsoleRedirector::Cleanup()
{
	try
	{
		bRunThread = FALSE;
		if (hStdOutThread != NULL)
		{
	//		if (!::IsWinNT())
			::TerminateThread(hStdOutThread, 1);
			::WaitForSingleObject(hStdOutThread, 1000);
			hStdOutThread = NULL;
		}

		if (hStdErrThread != NULL)
		{
	//		if (!::IsWinNT())
			::TerminateThread(hStdErrThread, 1);
			::WaitForSingleObject(hStdErrThread, 1000);
			hStdErrThread = NULL;
		}
		

		if (hProcessThread != NULL)
		{
			::WaitForSingleObject(hProcessThread, 1000);
			hProcessThread = NULL;
		}

		CloseChildHandles();
		CloseHandles();
	}
	catch (WinException e)
	{
		MessageBox (0, e.GetMessage (), MSG_EXCEPTION, MB_ICONEXCLAMATION | MB_OK);
	}
	catch (...)
	{
		WIN_EXECPTION(MSG_EXCEPTION);
	}

}
//------------------------------------------------------------------------------
int ConsoleRedirector::StdOutThread()
{
	DWORD nBytesRead;
	CHAR lpszBuffer[BUFFER_SIZE+1];

	while (bRunThread)
	{
		if (!::ReadFile(hStdOutRead, lpszBuffer, BUFFER_SIZE,
			&nBytesRead, NULL) || !nBytesRead)
		{
			if (::GetLastError() == ERROR_BROKEN_PIPE)
				break;			// pipe done - normal exit path.
			else
			{
				throw WinException ( MSG_STDOUT_ERR );
			}

		}
		if (nBytesRead)
		{
			// Virtual function to notify derived class that
			// characters are writted to stdout.
			lpszBuffer[nBytesRead] = '\0';
			OnChildStdOut(lpszBuffer);
		}
	}
	hStdOutThread = NULL;
	ExitThread(0);
	return 0;
}
//------------------------------------------------------------------------------
int ConsoleRedirector::StdErrThread()
{
	DWORD nBytesRead;
	CHAR lpszBuffer[BUFFER_SIZE+1];

	while (bRunThread)
	{
		if (!::ReadFile(hStdErrRead, lpszBuffer, BUFFER_SIZE, &nBytesRead, NULL) || !nBytesRead)
		{
			if (::GetLastError() == ERROR_BROKEN_PIPE)
				break;			// pipe done - normal exit path.
			else
			{
				throw WinException ( MSG_STDERR_ERR );
			}
		}
		if (nBytesRead)
		{
			// Virtual function to notify derived class that
			// characters are writted to stderr.
			lpszBuffer[nBytesRead] = '\0';
			OnChildStdErr(lpszBuffer);
		}
	}
	hStdErrThread = NULL;
	ExitThread(0);
	return 0;
}
//------------------------------------------------------------------------------
int ConsoleRedirector::ProcessThread()
{

	while (bRunThread)
	{
		switch (::WaitForSingleObject( hChildProcess, 1000))
		{
			case WAIT_OBJECT_0:	// child process exit
				bRunThread = FALSE;
				break;
		}
	}
	// Virtual function to notify derived class that
	// child process is terminated.
	// Application must call Cleanup()
	// but not direcly from this thread!!!
	hProcessThread = NULL;
	OnChildTerminate();
	ExitThread(0);
	return 0;
}
//------------------------------------------------------------------------------
BOOL ConsoleRedirector::CreateThreads()
{
	DWORD dwThreadID;
	
	try
	{
		// Launch the thread that read the child stdout.
		hStdOutThread = ::CreateThread(NULL, 0, (LPTHREAD_START_ROUTINE)_StdOutThread,
			this, 0, &dwThreadID);

		// Launch the thread that read the child stderr.
		hStdErrThread = ::CreateThread(NULL, 0, (LPTHREAD_START_ROUTINE)_StdErrThread,
			this, 0, &dwThreadID);

		// Launch the thread that monitoring the child process.
		hProcessThread = ::CreateThread(NULL, 0, (LPTHREAD_START_ROUTINE)_ProcessThread,
			this, 0, &dwThreadID);
	}
	catch (WinException e)
	{
		MessageBox (0, e.GetMessage (), MSG_EXCEPTION, MB_ICONEXCLAMATION | MB_OK);
	}
	catch (...)
	{
		WIN_EXECPTION(MSG_EXCEPTION);
		return -1;
	}
}
//------------------------------------------------------------------------------
HANDLE ConsoleRedirector::CreateChildProcess(LPCSTR lpszCmdLine, LPCSTR lpszExecDir,	BOOL bShowChildWindow)
{
	HANDLE hProcess = ::GetCurrentProcess();

	PROCESS_INFORMATION pi;

	// Set up the start up info struct.
	STARTUPINFO si;
	::ZeroMemory(&si, sizeof(STARTUPINFO));
	si.cb = sizeof(STARTUPINFO);
	si.dwFlags = STARTF_USESTDHANDLES|STARTF_USESHOWWINDOW;
	si.hStdOutput = hStdOut;
	si.hStdInput  = hStdIn;
	si.hStdError  = hStdErr;
	
	// Use this if you want to show the console window of the child.
	si.wShowWindow = bShowChildWindow ? SW_SHOW: SW_HIDE;
	// that dwFlags must include STARTF_USESHOWWINDOW to use the wShowWindow flags.

	// Create the NULL security token for the process
	PSECURITY_DESCRIPTOR lpSD = NULL;
	LPSECURITY_ATTRIBUTES lpSA = NULL;

	// On NT/2000 the handle must have PROCESS_QUERY_INFORMATION access.
	// This is made using an empty security descriptor. It is not the same
	// as using a NULL pointer for the security attribute!

	if (IsWinNT())
	{
		lpSD = (PSECURITY_DESCRIPTOR)::GlobalAlloc(GPTR, SECURITY_DESCRIPTOR_MIN_LENGTH);
		::InitializeSecurityDescriptor(lpSD, SECURITY_DESCRIPTOR_REVISION);
		::SetSecurityDescriptorDacl(lpSD, -1, 0, 0);

		lpSA = (LPSECURITY_ATTRIBUTES)::GlobalAlloc(GPTR, sizeof(SECURITY_ATTRIBUTES));
		lpSA->nLength = sizeof(SECURITY_ATTRIBUTES);
		lpSA->lpSecurityDescriptor = lpSD;
		lpSA->bInheritHandle = TRUE;
	}

	// Try to spawn the process.
	BOOL bResult = ::CreateProcess(NULL, 
			(char*)lpszCmdLine, 
			lpSA, 
			NULL, 
			TRUE,
			CREATE_NEW_PROCESS_GROUP | CREATE_NEW_CONSOLE,
      		NULL, 
			lpszExecDir, 
			&si, 
			&pi);

	// Cleanup memory allocation
	if (lpSA != NULL)
		::GlobalFree(lpSA);
	if (lpSD != NULL)
		::GlobalFree(lpSD);

	// Close any unnecessary handles.
	::CloseHandle(pi.hThread);

	// Save global child process handle to cause threads to exit.
	return pi.hProcess;
}
//------------------------------------------------------------------------------
BOOL ConsoleRedirector::StartChildProcess(LPCSTR lpszCmdLine, LPCSTR lpszExecDir, BOOL bShowChildWindow)
{
	CreatePipes();
	hChildProcess = CreateChildProcess(lpszCmdLine, lpszExecDir, bShowChildWindow);
	if (hChildProcess == NULL)
	{
		TCHAR lpszBuffer[BUFFER_SIZE];
		sprintf(lpszBuffer, MSGFMT_CANTRUN_ERR_S, lpszCmdLine);
		OnChildStdOut(lpszBuffer);
		
		CloseChildHandles();
		CloseHandles();
		return FALSE;
	}
	bRunThread = TRUE;
	CreateThreads();
	
	OnChildStarted(lpszCmdLine);
	return TRUE;
}
//------------------------------------------------------------------------------
void ConsoleRedirector::OnChildStdOut(LPCSTR lpszBuffer)
{
	OnChildWrite(lpszBuffer);
}
//------------------------------------------------------------------------------
void ConsoleRedirector::OnChildStdErr(LPCSTR lpszBuffer)
{
	OnChildWrite(lpszBuffer);
}
//------------------------------------------------------------------------------
void ConsoleRedirector::OnChildWrite( LPCSTR lpszOutput)
{
	char *s;
	s = strreplace( (char*)lpszOutput, "\n", "\r\n" );
	control->WriteOutput( s );
	free(s);
}
//------------------------------------------------------------------------------
void ConsoleRedirector::OnChildStarted( LPCSTR lpszOutput)
{
}
//------------------------------------------------------------------------------
void ConsoleRedirector::OnChildTerminate( )
{
	Cleanup();
}
//------------------------------------------------------------------------------
