#ifndef __CONNECTION__
#define __CONNECTION__

#include "odbc.h"
#include "error.h"
#include "common.h"

/* PostgreSQL backend specific information */
#define CATALOG_MAX_LENGTH              64
#define SCHEMA_MAX_LENGTH               64
#define TABLE_MAX_LENGTH                64
#define COLUMN_MAX_LENGTH               64
#define PROCEDURE_MAX_LENGTH            64
#define USER_MAX_LENGTH                 64
#define MAX_COLUMNS_IN_TABLE          1600

#ifdef NEED_LOG
	void Log(const TCHAR* szText);
	
	#define BEGIN(x)    Log(x)
	#define END(x)      return x
#else
	#define BEGIN(x)
	#define END(x)      return x
#endif /* NEED_LOG */


#ifdef _MULTITHREAD_SUPPORT
	#define ENTER_ENV(env, func_name)        BEGIN(func_name); ENTER_CRITICAL_SECTION((Environment*)env); SetError(SQL_HANDLE_ENV, env, ERR_CLEAR, NULL)
	#define ENTER_CONN(conn, func_name)      BEGIN(func_name); ENTER_CRITICAL_SECTION((Connection*)conn); SetError(SQL_HANDLE_DBC, conn, ERR_CLEAR, NULL)
	#define ENTER_STMT(stmt, func_name)      BEGIN(func_name); ENTER_CRITICAL_SECTION((Statement*)stmt);  SetError(SQL_HANDLE_STMT, stmt, ERR_CLEAR, NULL)
	#define ENTER_DESC(desc, func_name)      BEGIN(func_name); ENTER_CRITICAL_SECTION((Descriptor*)desc); SetError(SQL_HANDLE_DESC, desc, ERR_CLEAR, NULL)
	#define LEAVE_ENV(env, ret)              LEAVE_CRITICAL_SECTION((Environment*)env); END(ret)	
	#define LEAVE_CONN(conn, ret)            LEAVE_CRITICAL_SECTION((Connection*)conn); END(ret)
	#define LEAVE_STMT(stmt, ret)            LEAVE_CRITICAL_SECTION((Statement*)stmt); END(ret)
	#define LEAVE_DESC(desc, ret)            LEAVE_CRITICAL_SECTION((Descriptor*)desc); END(ret)

	#define __CRITICAL_SECTION(cs)           _CRITICAL_SECTION(cs)
	#define _INIT_CRITICAL_SECTION(handle)   INIT_CRITICAL_SECTION(handle)
	#define _DELETE_CRITICAL_SECTION(handle) DELETE_CRITICAL_SECTION(handle)
#else
	#define ENTER_ENV(env, func_name)        BEGIN(func_name)
	#define ENTER_CONN(conn, func_name)      BEGIN(func_name)
	#define ENTER_STMT(stmt, func_name)      BEGIN(func_name)
	#define ENTER_DESC(desc, func_name)      BEGIN(func_name)
	#define LEAVE_ENV(env, ret)              END(ret)	
	#define LEAVE_CONN(conn, ret)            END(ret)
	#define LEAVE_STMT(stmt, ret)            END(ret)
	#define LEAVE_DESC(desc, ret)            END(ret)
	
	#define __CRITICAL_SECTION(cs)
	#define _INIT_CRITICAL_SECTION(handle)
	#define _DELETE_CRITICAL_SECTION(handle)
#endif /* _MULTITHREAD_SUPPORT */


#define MAX_ERROR(x, y)    (x == y) ? x : ((SQL_SUCCESS == x) ? y : x)

#define SQLSMALLINT_LENGTH     15 /* number of decimal digits parameter can*/


#define UID_MAX_LENGTH            USER_MAX_LENGTH
#define PWD_MAX_LENGTH            255
#define DATABASE_MAX_LENGTH       UID_MAX_LENGTH /* default database is user name */
#define SERVER_MAX_LENGTH         255
#define PORT_MAX_LENGTH           255
#define DESCRIPTION_MAX_LENGTH    255
#define USEBUFFERING_MAX_LENGTH     1 /* Y or N */
#define CONNECTTIMEOUT_MAX_LENGTH  16
#define COMBOBOX_MAX_LENGTH        20

#define SSL_CERTIFICATE_MAX_LENGTH MAX_PATH
#define SSL_PRIVATE_KEY_MAX_LENGTH MAX_PATH
#define SSL_PWD_MAX_LENGTH         64

#define DSN_MAX_LENGTH             SQL_MAX_DSN_LENGTH

#define MAX_CURSOR_NAME_LENGTH     20
#define ALLOWED_OBJECTS_MAX_LENGTH 512

typedef enum
{
	CS_CONNECTED     = 1,
	CS_READY         = 2,
	CS_SENDING_QUERY = 4,

	CS_DISCONNECTED  = !CS_CONNECTED
} ConnectionState;

typedef enum
{
	TO_NOT_OPENED  = 0x00, /* initial state */

	TO_APPLICATION = 0x11, /* application request */
	TO_DRIVER      = 0x10, /* driver request */

	TO_DRIVER_THE_FIRST = 0x02 /* driver was 'THE FIRST' */
} TransactionOption;

typedef enum
{
	MSG_Unknown = -1,
	MSG_AuthenticationCleartextPassword,
	MSG_AuthenticationCryptPassword,
	MSG_AuthenticationKerberosV4,
	MSG_AuthenticationKerberosV5,
	MSG_AuthenticationMD5Password,
	MSG_AuthenticationOk,
	MSG_AuthenticationSCMCredential,
	MSG_BackendKeyData,
	MSG_Bind,
	MSG_BindComplete,
	MSG_CancelRequest,
	MSG_CloseStatement,
	MSG_ClosePortal,
	MSG_CloseComplete,
	MSG_CommandComplete,
	MSG_CopyData,
	MSG_CopyDone,
	MSG_CopyInResponse,
	MSG_CopyFail,
	MSG_CopyOutResponse,
	MSG_DataRow,
	MSG_Describe,
	MSG_EmptyQueryResponse,
	MSG_ErrorResponse,
	MSG_Execute,
	MSG_Parse,
	MSG_Flush,
	MSG_FunctionCall,
	MSG_FunctionCallResponse,
	MSG_NoData,
	MSG_NoticeResponse,
	MSG_NotificationResponse,
	MSG_ParameterDescription,
	MSG_ParameterStatus,
	MSG_ParseComplete,
	MSG_PasswordMessage,
	MSG_PortalSuspended,
	MSG_Query,
	MSG_ReadyForQuery,
	MSG_RowDescription,
	MSG_SSLRequest,
	MSG_StartupMessage,
	MSG_Sync,
	MSG_Terminate
} MessageType;

/* used once in ParseConnectionString */
typedef enum
{
	PS_KEYWORD_WAITING,
	PS_KEYWORD,
	PS_VALUE_WAITING,
	PS_VALUE,
	PS_VALUE_CLOSE_WAITING,
	PS_DIVISOR_WAITING,
	PS_VALUE_EXT_CHARS_ALLOWED
} ParseState;

typedef struct _message
{
	MessageType  type;
	BYTE         id;
	unsigned int length;
	TCHAR*       message;
} Message;

/* ***** */
typedef struct
{
	const TCHAR* szKeyName;
	const TCHAR* szIdentifier;
	const TCHAR* szDefaultValue;
	unsigned int unMaxValueLength;
	unsigned int unDialogItemID; /* WIN32 configure dialog */
} DSParameter;

#define DESCRIPTION_PARAM          0

#define FIRST_CONNSTR_PARAM        1 /* number of parameters useless in Connection String */ 

#define UID_PARAM                  1
#define SERVER_PARAM               2
#define DATABASE_PARAM             3

#define NECESSARY_CONNSTR_PARAM    4 /* number of first unnecessary parameter in connection string */

#define PWD_PARAM                  4
#define CERTIFICATE_PARAM          5
#define PRIVATE_KEY_PARAM          6
#define PASSPHRASE_PARAM           7
#define PORT_PARAM                 8
#define CONNECTTIMEOUT_PARAM       9
#define ALLOWED_OBJECTS_PARAM     10

#define DS_EDIT_PARAMETERS_NUMBER 11 /* number of parameters, configured in EDITs */

#define USEBUFFERING_PARAM        11

#define DS_CHECKBOX_PARAMETERS_NUMBER 12  /* number of parameters, configured in CHECKBOXes */

#define USESSL_PARAM              12

#define COMBOBOX_LIST_PARAM       13 /* combo box with LIST property */

#define CLIENTENCODING_PARAM      13

#define DS_PARAMETERS_NUMBER      14

#define DS_PARAMETERS_BUFFER_SIZE  UID_MAX_LENGTH + \
                                   PWD_MAX_LENGTH + \
                                   DATABASE_MAX_LENGTH + \
                                   SERVER_MAX_LENGTH + \
                                   PORT_MAX_LENGTH + \
                                   DESCRIPTION_MAX_LENGTH + \
																	 CONNECTTIMEOUT_MAX_LENGTH + \
																	 USEBUFFERING_MAX_LENGTH + \
                                   SSL_CERTIFICATE_MAX_LENGTH + \
                                   SSL_PRIVATE_KEY_MAX_LENGTH + \
                                   SSL_PWD_MAX_LENGTH + \
																	 ALLOWED_OBJECTS_MAX_LENGTH + \
																	 COMBOBOX_MAX_LENGTH*(DS_PARAMETERS_NUMBER-DS_CHECKBOX_PARAMETERS_NUMBER) + \
                                   DS_PARAMETERS_NUMBER + 1 /* null-terminals for every parameter value */

extern const DSParameter c_stDSParameters[DS_PARAMETERS_NUMBER];
/* ***** */

#define MAX(x, y) (x > y ? x:y)
#define MIN(x, y) (x < y ? x:y)

typedef struct _column       Column;
typedef struct _environment  Environment;
typedef struct _connection   Connection;
typedef struct _statement    Statement;
typedef struct _attribute    Attribute;
typedef struct _parameter    Parameter;

/*---------------------------------------------------------
	enum QueryType
---------------------------------------------------------*/
typedef enum
{
	ST_UNKNOWN        = 0x0000,
	ST_SELECT         = 0x0001,
	ST_INSERT         = 0x0002,
	ST_DELETE         = 0x0004,
	ST_UPDATE         = 0x0008,
	ST_MOVE           = 0x0010,
	ST_FETCH          = 0x0020,
	ST_BEGIN          = 0x0040,
	ST_COMMIT         = 0x0080,
	ST_EMPTY          = 0x0100,
	ST_SET            = 0x0200,
	ST_CREATE         = 0x0400,
	ST_ALTER          = 0x0800,
	ST_PREPARE        = 0x1000,
	ST_EXECUTE        = 0x2000,
	ST_DECLARE_CURSOR = 0x4000
} QueryType;


/*---------------------------------------------------------
	struct _conn_attributes
---------------------------------------------------------*/
struct _conn_attributes
{
	SQLUINTEGER connection_dead; /* SQL_CD_TRUE = dead, SQL_CD_FALSE = still alive */
	SQLUINTEGER autocommit;      /* SQL_AUTOCOMMIT_OFF = manual, SQL_AUTOCOMMIT_ON = default */
	SQLUINTEGER unicode;         /* SQL_AA_TRUE = ansi, SQL_AA_FALSE = unicode */
	SQLUINTEGER txn_isolation;   /* */
	SQLUINTEGER login_timeout;
	SQLUINTEGER query_timeout;   /* */
	SQLUINTEGER metadata_id;     /* SQL_TRUE or SQL_FALSE */
	SQLUINTEGER max_rows;
	SQLUINTEGER cursor;
	SQLUINTEGER access_mode;     /* SQL_MODE_READ_WRITE, SQL_MODE_READ_WRITE */
};

/*---------------------------------------------------------
	struct _backend
---------------------------------------------------------*/
struct _backend
{
	int version_first;
	int version_second;
	int version_third;
};

/*---------------------------------------------------------
	struct _restrictions
---------------------------------------------------------*/
typedef struct _restrictions
{
	List list;
	List length;
} Restrictions;

#define PATTERN_LONGER       0x100000

/*---------------------------------------------------------
	struct EnvAttributes
---------------------------------------------------------*/
typedef struct _env_attributes
{
	SQLUINTEGER cp_match;
	SQLUINTEGER odbc_version;
	SQLUINTEGER connection_pooling;
} EnvAttributes;
/*---------------------------------------------------------
	struct Environment
---------------------------------------------------------*/
struct _environment
{
	struct _env_attributes attributes;
	struct _diag           diag;
	struct _list           connections; /* sconnections allocated under this environment */

	__CRITICAL_SECTION(cs);
};


#define CONNECTION_OUT_BUFFER_SIZE        8*1024
#define CONNECTION_IN_BUFFER_SIZE       128*1024

typedef unsigned int ProtocolVersion;
/*---------------------------------------------------------
	struct Connection
---------------------------------------------------------*/
struct _connection
{
#ifdef USE_SSL
	SSL* ssl;
#endif /* USE_SSL */

	unsigned int  state; /* CS_READY, CS_DISCONNECTED */

	struct _diag            diag;
	struct _list            statements;   /* statements allocated under this connection */
	struct _list            descriptors;  /* descriptors             -//-               */
	struct _conn_attributes attributes;
	struct _backend         backend;      /* backend info */
	struct _restrictions    restrictions; /* restriction information about objects user can access to */

	TCHAR* dbms; /* connected dbms information */


	BOOL TransactionState; /* TRUE  - transaction began,
	                        * FALSE - transaction closed or uncreated
	                        */

	BOOL mjet;  /* TRUE  - created by Microsoft Application with all it's features to handle 
	             * FALSE - Unknown application
	             */

	/* SOCKET connection data */
	SOCKET             socket;
	struct sockaddr_in addr;

	/* Backend data */
	unsigned int processId;
	unsigned int secretKey;

	unsigned int cursor_count;

	/* buffer for the outgoing messages to backend */
	BYTE  outBuffer[CONNECTION_OUT_BUFFER_SIZE];
	BYTE* pOutBegin;
	BYTE* pOutFirst;
	BYTE* pOutLast;
	BYTE* pOutEnd;

	/* buffer for the incoming messages from backend */
	BYTE  inBuffer[CONNECTION_IN_BUFFER_SIZE];
	BYTE* pInBegin;
	BYTE* pInFirst;
	BYTE* pInLast;
	BYTE* pInEnd;

	TCHAR dsn[DSN_MAX_LENGTH+1];
	ProtocolVersion version;

	Environment* environment;
	
	/* Data Source parameters */
	TCHAR* parameters[DS_PARAMETERS_NUMBER];
	TCHAR  parameters_buffer[DS_PARAMETERS_BUFFER_SIZE];

	Connection* pNext;   /* next connection in the environment's list */

	__CRITICAL_SECTION(cs);
};

Environment* AllocEnvironment();
SQLRETURN    FreeEnvironment(Environment* pEnvironment);

Connection* AllocConnection(Environment* pEnvironment);
SQLRETURN   FreeConnection(Connection* pConnection, SQLUSMALLINT Option);

SQLRETURN Connect(Connection* pConnection);
SQLRETURN Disconnect(Connection* pConnection);

void ReadFromDS(TCHAR** parameters, TCHAR* DSN, SQLSMALLINT DSNLength);
void WriteToDS(TCHAR** parameters, TCHAR* DSN);

SQLRETURN ParseConnectionString(Connection* pConnection, SQLTCHAR* szConnStrIn, SQLSMALLINT cbConnStrIn);
SQLRETURN GetKeyValue(Connection* pConnection, TCHAR* Key, unsigned int KeyLength, TCHAR* Value, unsigned int ValueLength);
SQLRETURN BeginTransaction(Statement* pStatement, SQLSMALLINT Option);
SQLRETURN EndTransaction(SQLSMALLINT HandleType, SQLHANDLE Handle, SQLSMALLINT CompletionType, SQLSMALLINT Option);
SQLRETURN GetConnectionInfo(Connection* pConnection, SQLUSMALLINT InfoType, SQLPOINTER InfoValue, SQLSMALLINT BufferLength, SQLSMALLINT* StringLength);
SQLRETURN SetConnectAttr(Connection* pConnection, SQLINTEGER Attribute, SQLPOINTER Value, SQLINTEGER StringLength);
SQLRETURN PrepareRestriction(Restrictions* restrictions, TCHAR* text);
SQLRETURN CheckPatterns(TCHAR* begin, TCHAR* end, Restrictions* restrictions);


SQLRETURN sock_connect(Connection* pConnection);
SQLRETURN sock_close(Connection* pConnection);

int sock_send(Connection* pConnection);
int sock_recv(Connection* pConnection);

/* constants definitions */
extern const int c_FIELD_NULL;

extern const TCHAR c_szKeyWords[];
extern const TCHAR c_szODBC_INI[];
extern const TCHAR c_szDriverName[];
extern const TCHAR c_szBookmarkName[];
extern const TCHAR c_szDBMSVersion[12];
extern const TCHAR c_szDriverODBCVer[6];
extern const TCHAR c_szDriverVersion[];
extern const TCHAR c_szDriverFileName[];
extern const TCHAR c_szSpecialCharacter[];


#define TXN_ISOLATION_LEVEL_DEFAULT  SQL_TXN_READ_COMMITTED

#define SQL_STMT_NOSCAN_OFF(stmt)  (SQL_NOSCAN_OFF == stmt->attributes.no_scan)
#define AUTOCOMMIT_MODE_OFF(stmt)  (SQL_AUTOCOMMIT_OFF == stmt->connection->attributes.autocommit)
#define AUTOCOMMIT_MODE_ON(stmt)   (SQL_AUTOCOMMIT_ON  == stmt->connection->attributes.autocommit)
#define STATEMENT_NEEDS_REDECLARATION(stmt)  (0 != (SS_STATEMENT_NEED_REDECLARE & stmt->state))
#define PORTAL_NEEDS_REDECLARATION(stmt)     (0 != (SS_PORTAL_NEED_REDECLARE & stmt->state))

#define SET_PORTAL_NEEDS_REDECLARATION(stmt)    stmt->state |= SS_PORTAL_NEED_REDECLARE
#define SET_PORTAL_UNDECLARED(stmt)             stmt->state ^= SS_PORTAL_DECLARED; \
                                                stmt->state |= SS_PORTAL_NEED_REDECLARE
#define SET_PORTAL_DECLARED(stmt)               stmt->state |= SS_PORTAL_DECLARED; \
                                                stmt->state ^= SS_PORTAL_NEED_REDECLARE; \
                                                stmt->query.cursorState = CS_DECLARED

#define SET_STATEMENT_NEEDS_REDECLARATION(stmt) stmt->state |= SS_STATEMENT_NEED_REDECLARE
#define SET_STATEMENT_UNDECLARED(stmt)          stmt->state ^= SS_STATEMENT_DECLARED; \
                                                stmt->state |= SS_STATEMENT_NEED_REDECLARE
#define SET_STATEMENT_DECLARED(stmt)            stmt->state |= SS_STATEMENT_DECLARED; \
                                                stmt->state ^= SS_STATEMENT_NEED_REDECLARE; \
                                                SET_PORTAL_NEEDS_REDECLARATION(stmt)


#define PORTAL_DECLARED(stmt)    (0 != (SS_PORTAL_DECLARED & stmt->state))
#define STATEMENT_DECLARED(stmt) (0 != (SS_STATEMENT_DECLARED & stmt->state))
#define DBC_USE_BUFFERING(conn)  (_T('Y') == conn->parameters[USEBUFFERING_PARAM][0])
#define STMT_USE_BUFFERING(stmt) (TRUE == stmt->use_buffering)
#define STMT_USE_BOOKMARKS(stmt) (SQL_UB_OFF != stmt->attributes.use_bookmarks)

#define CSOC_OBJ_TABLE           0
#define CSOC_OBJ_PROC            1

#define QUOTER_FOUND             0xFFFF

#define MD5_PASSWD_LEN	35

#endif /* #ifndef __CONNECTION__ */
