#include "connection.h"
#include "statement.h"
#include "protocol.h"

#define PQ_CLOSE                     "CLOSE\""
#define RECV_STRING_BUFFER_SIZE      0x1000
/*-----------------------------------------------------------------------------
 * FUNCTION: RecvByte
 *
 *-----------------------------------------------------------------------------
 */
BOOL RecvByte(Connection* pConnection, BYTE* pByte)
{
	if (pConnection->pInLast - pConnection->pInFirst == 0)
	{
		if (sock_recv(pConnection) < 0)
		{
			/* error in socket communication */
			return 1;
		}
	}
	if (pConnection->pInLast - pConnection->pInFirst == 0)
	{
		/* incoming buffer is empty */
		return 1;
	}
	*pByte = *pConnection->pInFirst;
	if (++pConnection->pInFirst == pConnection->pInLast)
	{
		pConnection->pInFirst = pConnection->pInBegin;
		pConnection->pInLast  = pConnection->pInBegin;
	}
	return 0;
}

/*-----------------------------------------------------------------------------
 * FUNCTION: RecvBytes
 *
 *-----------------------------------------------------------------------------
 */
BOOL RecvBytes(Connection* pConnection, BYTE* pBytes, unsigned int nLength)
{
	unsigned int length;

	if ((unsigned int)(pConnection->pInLast - pConnection->pInFirst) < nLength)
	{
		if (sock_recv(pConnection) < 0)
		{
			/* error in socket communication */
			return 1;
		}
	}

	while(1)
	{
		length = ((unsigned int)(pConnection->pInLast - pConnection->pInFirst) < nLength) ? (unsigned int)(pConnection->pInLast - pConnection->pInFirst) : nLength;
		
		memcpy(pBytes, pConnection->pInFirst, length);
		pConnection->pInFirst += length;
		pBytes                += length;
		nLength               -= length;

		if (pConnection->pInFirst == pConnection->pInLast)
		{
			pConnection->pInFirst = pConnection->pInBegin;
			pConnection->pInLast  = pConnection->pInBegin;
		}

		if (0 == nLength)
			break;

		if (sock_recv(pConnection) < 0)
		{
			/* error in socket communication */
			return 1;
		}
	}

	return 0;
}

/*-----------------------------------------------------------------------------
 * FUNCTION: RecvString
 *
 * DESCRIPTION: returned string is already converted to UTF-8 or UCS-2
 *-----------------------------------------------------------------------------
 */
SQLRETURN RecvString(Connection* pConnection, TCHAR** pString, int Length, int MaxLength)
{
	BYTE buffer[RECV_STRING_BUFFER_SIZE]; /* use this static buffer for most cases */
	BYTE*  text   = NULL;
	TCHAR* result = NULL;

	if (SQL_NTS != Length && Length < sizeof(buffer))
	{/* we know exect length of the string (including terminating zero or NOT)*/
		RecvBytes(pConnection, buffer, Length);
		text = buffer;
	}
	else if (0 != MaxLength && MaxLength < sizeof(buffer))
	{/* we know max possible length and it's less then avaible buffer - we are looking for '\0' */
		BYTE* ptr = buffer-1;
		Length = -1;
		do
		{
			RecvByte(pConnection, ++ptr);
			Length++;
		} while ('\0'!=*ptr);
		text = buffer;
	}
	else
	{/* we can't know the exact length of the incoming string */
		if (SQL_NTS == Length)
		{/* read string - when it will reach the end of buffer - alloc memmory */
			int   freespace = sizeof(buffer);
			BYTE* ptr       = buffer-1;

			Length = -1;
			do
			{
				RecvByte(pConnection, ++ptr);
				Length++;
				freespace--;
			} while ('\0' != *ptr || 0 == freespace);

			if (0 == freespace && '\0' != *ptr)
			{/* string was longer then static buffer - need external allocation */
				int PrevLength = sizeof(buffer);
				int NewLength = (0 < MaxLength) ? MaxLength : PrevLength + sizeof(buffer);
				text = (BYTE*)malloc(NewLength);
				memcpy(text, buffer, PrevLength);
				do
				{/* we will increase buffer utill it could contain full string */
					freespace = NewLength-PrevLength;
					ptr = text + PrevLength;
					do
					{
						RecvByte(pConnection, ++ptr);
						Length++;
						freespace--;
					} while ('\0' != *ptr || 0 == freespace);

					if (0 == freespace && '\0' != *ptr)
					{/* read more! */
						BYTE* temp;
						PrevLength = NewLength;
						temp = (BYTE*)malloc((NewLength += sizeof(buffer)));
						memcpy(temp, text, PrevLength);
						FREE(text);
						text = temp;
					}
				} while ('\0' != *ptr);
			}
			else
			{/* buffer was big enough for string */
				text = buffer;
			}
		}
		else
		{
		/* we know data length - alloc memory for it */
			text = (BYTE*)malloc(Length);
			RecvBytes(pConnection, text, Length);
		}
	}

#ifdef UNICODE
{
	BYTE* ptr    = text;
	int   length = 1;

	/* calculate future length */
	for(;0<Length;length++)
	{
		if (0x80 > *ptr)
		{
			Length -= 1;
			ptr += 1;
		}
		else if (0xE0 == (*ptr & 0xE0))
		{
			Length -= 3;
			ptr += 3;
		}
		else
		{
			Length -= 2;
			ptr += 2;
		}
	}

	/* length here includes zero-terminator */
	if (NULL != (result = (WCHAR*)malloc(length*sizeof(WCHAR))))
	{
		BYTE*  ptr     = text;
		WCHAR* res_ptr = result;

		/* convert string from utf8 to ucs2 */
		for(length--;0<length;res_ptr++,length--)
		{
			if (0x80 > *ptr)
			{
				*res_ptr = (0x007F & *ptr);
				ptr += 1;
			}
			else if (0xE0 == (*ptr & 0xE0))
			{
				*res_ptr = (((uint32)ptr[0] & 0x0F) << 12) | (((uint32)ptr[1] & 0x3F) << 6) | ((uint32)ptr[2] & 0x3F);
				ptr += 3;
			}
			else
			{
				*res_ptr = (((uint32)ptr[0] & 0x1F) << 6) | ((uint32)ptr[1] & 0x3F);
				ptr += 2;
			}
		}
		*res_ptr = L'\0';
	}
	else
	{
		/* no memory! */
		*pString = NULL;
		return SQL_ERROR;
	}
}
#else
	if (NULL != (result = (char*)malloc(Length+1)))
	{
		strncpy(result, (char*)text, Length);
		result[Length] = '\0';
	}
#endif /* UNICODE */
	/* if we were using additional non static buffer - free it! */
	if (text != buffer)
		FREE(text);
	return (*pString = result) ? SQL_SUCCESS : SQL_ERROR;
}

/*-----------------------------------------------------------------------------
 * FUNCTION: RecvInt16
 *
 * RETURN: '\0' - no messages avaible or socket error
 *-----------------------------------------------------------------------------
 */
BOOL RecvInt16(Connection* pConnection, unsigned short* pInt16)
{
	if (pConnection->pInLast - pConnection->pInFirst < 2)
	{
		if (sock_recv(pConnection) < 0)
		{
			/* error in socket communication */
			return 1;
		}
	}
	if (pConnection->pInLast - pConnection->pInFirst < 2)
	{
		return 1;
	}
	*pInt16 = ntohs(*(unsigned short*)pConnection->pInFirst);
	pConnection->pInFirst += 2;
	if (pConnection->pInFirst == pConnection->pInLast)
	{
		pConnection->pInFirst = pConnection->pInBegin;
		pConnection->pInLast  = pConnection->pInBegin;
	}
	return 0;
}

/*-----------------------------------------------------------------------------
 * FUNCTION: RecvInt32
 *
 * RETURN: '\0' - no messages avaible or socket error
 *-----------------------------------------------------------------------------
 */
BOOL RecvInt32(Connection* pConnection, unsigned int* pInt32)
{
	if (pConnection->pInLast - pConnection->pInFirst < 4)
	{
		if (sock_recv(pConnection) < 0)
		{
			/* error in socket communication */
			return 1;
		}
	}
	if (pConnection->pInLast - pConnection->pInFirst < 4)
	{
		if (sock_recv(pConnection) < 0)
			return 1;

		if (pConnection->pInLast - pConnection->pInFirst < 4)
			return 1;
	}

	*pInt32 = ntohl(*(int*)pConnection->pInFirst);
	pConnection->pInFirst += 4;
	if (pConnection->pInFirst == pConnection->pInLast)
	{
		pConnection->pInFirst = pConnection->pInBegin;
		pConnection->pInLast  = pConnection->pInBegin;
	}
	return 0;
}

/*-----------------------------------------------------------------------------
 * SendByte
 *-----------------------------------------------------------------------------
 */
BOOL SendByte(Connection* pConnection, char ch)
{
	while (pConnection->pOutEnd - pConnection->pOutLast == 0)
	{
		if (sock_send(pConnection) < 0)
		{
			/* error in socket communication */
			return 1;
		}	
	}
	*pConnection->pOutLast = ch;
	pConnection->pOutLast++;

	return 0;
}
/*-----------------------------------------------------------------------------
 * SendBytes
 *-----------------------------------------------------------------------------
 */
BOOL SendBytes(Connection* pConnection, BYTE* pBytes, unsigned int nLength)
{
	unsigned int nBufferSize;

	if (pBytes && nLength)
	{
		do
		{
			nBufferSize = pConnection->pOutEnd - pConnection->pOutLast;
			if (nBufferSize > nLength)
			{
				/* output buffer is big enough */
				memcpy(pConnection->pOutLast, pBytes, nLength);
				pConnection->pOutLast += nLength;
				nLength = 0;
			}
			else
			{
				/* fill full output buffer and send data */
				memcpy(pConnection->pOutLast, pBytes, nBufferSize);
				pConnection->pOutLast += nBufferSize;
				pBytes += nBufferSize;
				nLength -= nBufferSize;
				if (sock_send(pConnection) < 0)
				{
					/* error in socket communication */
					return 1;
				}
			}
		} while (nLength); /* we still have something to send */
	}
	return 0;
}

/*-----------------------------------------------------------------------------
 * SendString
 *-----------------------------------------------------------------------------
 */
BOOL SendString(Connection* pConnection, const TCHAR* String, int Length)
{
	BOOL need_terminator = TRUE;
	if (String)
	{
		if (SQL_NTS == Length)
			Length = _tcslen(String);
		else
			need_terminator = FALSE;

#ifdef UNICODE
	{
		int i;

		for(i=0;i<Length;String++,i++)
		{
			if (0 == (0xFF80 & *String))
				/* 1 BYTE */
				SendByte(pConnection, (BYTE)*String);
			else if (0 == (0xF800 & *String))
			{
				/* 2 BYTEs */
				short word = 0x80C0 | ((0x07C0 & *String) >> 6) | ((0x003F & *String) << 8);
				SendBytes(pConnection, (BYTE*)&word, 2);
			}
			else
			{
				/* 3 BYTEs */
				int dword = 0x008080E0 | ((0xF000  & *String) >> 12) | ((0x0FC0  & *String) << 2) | ((0x003F  & *String) << 16);
				SendBytes(pConnection, (BYTE*)&dword, 3);
			}
		}
	}
#else
		do
		{
			/* output buffer is big enough */
			int nBufferSize = pConnection->pOutEnd - pConnection->pOutLast;
			if (nBufferSize > Length)
			{
				memcpy(pConnection->pOutLast, String, Length);
				pConnection->pOutLast += Length;
				Length = 0;
			}
			else
			{
				/* fill full output buffer and send data */
				memcpy(pConnection->pOutLast, String, nBufferSize);
				pConnection->pOutLast += nBufferSize;
				String += nBufferSize;
				Length -= nBufferSize;
				if (sock_send(pConnection) < 0)
				{
					/* error in socket communication */
					return 1;
				}
			}
		} while (Length); /* we still have something to send */
#endif /* UNICODE */
	}

	if (need_terminator)
		SendByte(pConnection, '\0');
	return 0;
}

/*-----------------------------------------------------------------------------
 *
 *-----------------------------------------------------------------------------
 */
BOOL SendInt16(Connection* pConnection, short nInt16)
{
	while (pConnection->pOutEnd - pConnection->pOutLast < sizeof(nInt16))
	{
		if (sock_send(pConnection) < 0)
		{
			/* error in socket communication */
			return 1;
		}	
	}

	*(u_long*) pConnection->pOutLast = htons(nInt16);
	pConnection->pOutLast += sizeof(nInt16);

	return 0;
}

/*-----------------------------------------------------------------------------
 *
 *-----------------------------------------------------------------------------
 */
BOOL SendInt32(Connection* pConnection, int nInt32)
{
	while (pConnection->pOutEnd - pConnection->pOutLast < sizeof(nInt32))
	{
		if (sock_send(pConnection) < 0)
		{
			/* error in socket communication */
			return 1;
		}	
	}

	*(u_long*) pConnection->pOutLast = htonl(nInt32);
	pConnection->pOutLast += sizeof(nInt32);

	return 0;
}

/*-----------------------------------------------------------------------------
 * GetMessageFromBackend
 *
 *
 *-----------------------------------------------------------------------------
 */
SQLRETURN GetMessageFromBackend(Connection* pConnection, Message* msg, Statement* pStatement)
{
	BOOL         read_more;
	SQLRETURN    nRet;
	Descriptor*  pIPD;
	Descriptor*  pIRD;
	ID_REC*      pID_REC;
	TCHAR*       err;

	TCHAR*       pBuffer;
	SQLSMALLINT  i;

	BYTE           msg_byte;
	unsigned int   msg_int32;
	unsigned short msg_int16;
	TCHAR*         msg_string;

	if (RecvByte(pConnection, &msg->id) ||
	    RecvInt32(pConnection, &msg->length)
	   )
		return SQL_ERROR;
	msg->length -= sizeof(uint32);

	nRet = SQL_SUCCESS;
	switch(msg->id)
	{
		case '\0':
			/* no messages avaible */
			break;
		case '1':
			if (0 == msg->length)
				msg->type = MSG_ParseComplete;
			else
				msg->type = MSG_Unknown;
			break;
		case '2':
			if (0 == msg->length)
				msg->type = MSG_BindComplete;
			else
				msg->type = MSG_Unknown;
			break;
		case '3':
			if (0 == msg->length)
				msg->type = MSG_CloseComplete;
			else
				msg->type = MSG_Unknown;
			break;
		case 'c':
			if (0 == msg->length)
				msg->type = MSG_CopyDone;
			else
				msg->type = MSG_Unknown;
			break;
		case 'I':
			if (0 == msg->length)
				msg->type = MSG_EmptyQueryResponse;
			else
				msg->type = MSG_Unknown;
			break;
		case 'n':
			if (0 == msg->length)
				msg->type = MSG_NoData;
			else
				msg->type = MSG_Unknown;
			break;
		case 'R':
			if (RecvInt32(pConnection, &msg_int32))
				return SQL_ERROR;

			switch (msg->length)
			{
				case 4:
					switch(msg_int32)
					{
						case 0:
							msg->type = MSG_AuthenticationOk;
							break;
						case 1:
							msg->type = MSG_AuthenticationKerberosV4;
							break;
						case 2:
							msg->type = MSG_AuthenticationKerberosV5;
							break;
						case 3:
							msg->type = MSG_AuthenticationCleartextPassword;
							break;
						case 6:
							msg->type = MSG_AuthenticationSCMCredential;
							break;
						default:
							msg->type = MSG_Unknown;
					}
					break;
				case 6:
			    /* Int32(4),Byte2 */
					msg->type = MSG_AuthenticationCryptPassword;
					break;
				case 8:
					if (5 == msg_int32)
					{
						msg->message = (TCHAR*) malloc(4);
						if (RecvBytes(pConnection, (BYTE*)msg->message, 4))
						{
							FREE(msg->message);
							return SQL_ERROR;
						}
						msg->type = MSG_AuthenticationMD5Password;					
					}
					else
					{
						msg->type = MSG_Unknown;
					}
					break;
				default:
					break;
			}
			break;
		case 's':
			if (0 == msg->length)
			{
				msg->type = MSG_PortalSuspended;
				if (0 < pStatement->ird->header.rows_affected)
				{
					pStatement->ird->header.rows_affected /= pStatement->ird->header.count;
					PrepareResultset(pStatement, pStatement->ird->header.rows_affected);
				}
			}
			else
				msg->type = MSG_Unknown;
			break;
		case 'K':
			/* Backend Data */
			if (8 == msg->length)
			{
				if (RecvInt32(pConnection, &pConnection->processId) ||
				    RecvInt32(pConnection, &pConnection->secretKey)
				   )
					return SQL_ERROR;
				msg->type = MSG_BackendKeyData;			
			}
			else
			{
				msg->type = MSG_Unknown;
			}
			break;
		/* command completed successfully */
		case 'C':
			if (SQL_ERROR == RecvString(pConnection, &msg_string, msg->length, 0))
				return SQL_ERROR;
			/* parse string to get results */
			switch(msg_string[0])
			{
				/*-------------------------------------------------------------------------------
				 * this messages tell about completing creation of resultset (use 'SaveResultset')
				 *-------------------------------------------------------------------------------
				 */
				case _T('I'): /* INSERT */
				case _T('D'): /* DELETE, DECLARE */
				case _T('U'): /* UPDATE */
					if (NULL == (pIRD = NewResultset(pStatement)))
					{
						SetError(SQL_HANDLE_STMT, pStatement, ERR_NOT_ENOUGH_MEMORY, NULL);
						return SQL_ERROR;
					}
					pIRD = GET_DESCRIPTOR(pIRD);
					SetDescField(pIRD, 0, SQL_DESC_COUNT, (SQLPOINTER)(SQLSMALLINT)0, SQL_IS_SMALLINT);

					if (_T('C') == msg_string[2]) /* DECLARE CURSOR */
					{
						pIRD->header.query_type = ST_DECLARE_CURSOR;
						pIRD->header.rows_affected = -1;
					}
					else switch(msg_string[0])
					{
						case _T('U'): /* UPDATE rows */
							pIRD->header.query_type = ST_UPDATE;
							pIRD->header.rows_affected = _ttoi(msg_string+STR_SIZEOF("UPDATE")+1);
							break;
						case _T('I'): /* INSERT oid rows */
							pIRD->header.rows_affected = _ttoi(_tcsrchr(msg_string, ' ')+1);
							pIRD->header.query_type = ST_INSERT;
							break;
						case _T('D'): /* DELETE rows */
							pIRD->header.rows_affected = _ttoi(msg_string+STR_SIZEOF("DELETE")+1);
							pIRD->header.query_type = ST_DELETE;
					}
					RET_DESCRIPTOR(pIRD);
					break;
				case _T('S'): /* SELECT */
				case _T('E'): /* EXECUTE */
					pIRD = GET_DESCRIPTOR((0 < pStatement->results.irds.used) ? (Descriptor*)pStatement->results.irds.handles[pStatement->results.irds.used-1]: pStatement->ird);
					if ((0 == _tcscmp(msg_string, _T("SELECT"))) ||
					    (0 == _tcscmp(msg_string, _T("SHOW"))) ||
					    (0 == _tcscmp(msg_string, _T("EXECUTE")))
						 )
					{
						pIRD->header.query_type = ST_SELECT;
						pIRD->header.rows_affected /= pIRD->header.count;
						RET_DESCRIPTOR(pIRD);
						PrepareResultset(pStatement, pIRD->header.rows_affected);
					}
					else 
					{
						pIRD->header.query_type = ST_SET;
						pIRD->header.rows_affected = -1;
						RET_DESCRIPTOR(pIRD);
					}
					break;
				/*-------------------------------------------------------------------------------
				 * product no resutset
				 *-------------------------------------------------------------------------------
				 */
				case _T('M'): /* MOVE rows */
					pStatement->ird->header.query_type    = ST_MOVE;
					pStatement->ird->header.rows_affected = _ttoi(msg_string+STR_SIZEOF("MOVE")+1);
					break;
				case _T('F'): /* FETCH rows */
					pStatement->ird->header.query_type    = ST_FETCH;
					pStatement->ird->header.rows_affected = _ttoi(msg_string+STR_SIZEOF("FETCH")+1);
					PrepareResultset(pStatement, pStatement->ird->header.rows_affected);
					break;
				case _T('B'): /* BEGIN */
					pStatement->ird->header.query_type    = ST_BEGIN;
					pStatement->ird->header.rows_affected = -1;
					break;
				case _T('R'): /* ROLLBACK */
				case _T('C'): /* COMMIT */
					/* this operation performs only on connection handle */
					break;
				case _T('P'): /* PREPARE */
					/* just for information */
					break;

				default:
					/* wrong data */
					pStatement->ird->header.query_type    = ST_UNKNOWN;
					pStatement->ird->header.rows_affected = -1;
					break;
			}


			/* translate */


			msg->type = MSG_CommandComplete;
			FREE(msg_string);
			break;
		case 'E':
			/* Error messages */
			err = NULL;
			read_more = 1;
			while (read_more)
			{
				TCHAR** ppString = NULL;

				msg_string = NULL;
				if (RecvByte(pConnection, &msg_byte))
					return SQL_ERROR;
				switch (msg_byte)
				{
					case '\0':
						/* No more messages */
						read_more = 0;
						break;
					case 'S':
						/* Severity: the field contents are ERROR, FATAL, or PANIC (in an error message), or WARNING, NOTICE, DEBUG, INFO, or LOG (in a notice message), or a localized translation of one of these. Always present.*/
						ppString = &msg_string;
						break;
					case 'C':
						/* Code: the SQLSTATE code for the error. Not localizable. Always present. */
						ppString = &msg_string;
						break;
					case 'M':
						/* Message: the primary human-readable error message. This should be accurate but terse (typically one line). Always present.*/
						ppString = &err;
						break;
					case 'D':
						/* Detail: an optional secondary error message carrying more detail about the problem. May run to multiple lines. */
						ppString = &msg_string;
						break;
					case 'H':
						/* Hint: an optional suggestion what to do about the problem. This is intended to differ from Detail in that it offers advice (potentially inappropriate) rather than hard facts. May run to multiple lines. */
						ppString = &msg_string;
						break;
					case 'P':
						/* Position: the field value is a decimal ASCII integer, indicating an error cursor position as an index into the original query string. The first character has index 1, and positions are measured in characters not bytes. */
						ppString = &msg_string;
						break;
					case 'p':
						/* Internal position: this is defined the same as the P field, but it is used when the cursor position refers to an internally generated command rather than the one submitted by the client. The q field will always appear when this field appears. */
						ppString = &msg_string;
						break;
					case 'q':
						/* Internal query: the text of a failed internally-generated command. This could be, for example, a SQL query issued by a PL/pgSQL function. */
						ppString = &msg_string;
						break;
					case 'W':
						/* Where: an indication of the context in which the error occurred. Presently this includes a call stack traceback of active procedural language functions and internally-generated queries. The trace is one entry per line, most recent first. */
						ppString = &msg_string;
						break;
					case 'F':
						/* File: the file name of the source-code location where the error was reported. */
						ppString = &msg_string;
						break;
					case 'L':
						/* Line: the line number of the source-code location where the error was reported. */
						ppString = &msg_string;
						break;
					case 'R':
						/* Routine: the name of the source-code routine reporting the error. */
						ppString = &msg_string;
						break;
					default:
						ppString = &msg_string;
				}
				if (msg_byte)
				{
					if (SQL_ERROR == RecvString(pConnection, ppString, SQL_NTS, 1000))
						return SQL_ERROR;
					FREE(msg_string);
				}
				else
				{
					msg->length--;
					/* prepare message string to return */
					msg->message = err;
				}
			}
			msg->type = MSG_ErrorResponse;
			break;
		case 'Z':
			if (1 == msg->length)
			{
				if (RecvByte(pConnection, &msg_byte))
					return SQL_ERROR;
				if (NULL != pStatement)
					pStatement->results.current = -1;
				msg->type = MSG_ReadyForQuery;
			}
			else
			{
				msg->type = MSG_Unknown;
			}
			break;
		case 'A':
			/* Int32,Int32,String,String  */
			msg->type = MSG_NotificationResponse;
			break;

		/* row data */
		case 'D':
			msg->length -= sizeof(uint16);

			/* Get the number of column values that follow (possibly zero) */
			if (RecvInt16(pConnection, &msg_int16))
				return SQL_ERROR;

			pIRD = GET_DESCRIPTOR((0 < pStatement->results.irds.used) ? (Descriptor*)pStatement->results.irds.handles[pStatement->results.irds.used-1]: pStatement->ird);

			for(i=0;i<msg_int16;i++)
			{
				if(RecvInt32(pConnection, &msg_int32))
					return SQL_ERROR;

				if (-1 == msg_int32)
				{/* null value */
					int* pBuffer = (int*)AddField(pStatement, 0) - 1;
					*pBuffer = FIELD_NULL;
				}
				else
				{
					RecvString(pConnection, &msg_string, msg_int32, 0);
					if (pIRD->id_records[i].max_column_length < (msg_int32 = _tcslen(msg_string)))
						/* get max column length */
						pIRD->id_records[i].max_column_length = msg_int32;

					if (!(pBuffer = AddField(pStatement, msg_int32)) ||
						  !(_tcsncpy(pBuffer, msg_string, msg_int32))
						)
						return SQL_ERROR;
					FREE(msg_string);
				}
			}

			if (0 > pIRD->header.rows_affected)
				pIRD->header.rows_affected = 0;
			
			pIRD->header.rows_affected += msg_int16;
			RET_DESCRIPTOR(pIRD);


			msg->type = MSG_DataRow;
			break;

		/* row description
		 * use this message to populate implementation row descriptor
		 */
		case 'T':
			if (NULL == (pIRD = NewResultset(pStatement)))
			{
				SetError(SQL_HANDLE_STMT, pStatement, ERR_NOT_ENOUGH_MEMORY, NULL);
				return SQL_ERROR;
			}
			pIRD = GET_DESCRIPTOR(pIRD);

			/* Get the number of fields in a row (may be zero) and increase the allocated memory if needed */
			if (RecvInt16(pConnection, &msg_int16) ||
			    SQL_SUCCESS != SetDescField(pIRD, 0, SQL_DESC_COUNT, (SQLPOINTER)(SQLLEN_TO_POINTER) msg_int16, SQL_IS_SMALLINT)
			   )
				nRet = SQL_ERROR;
			else
			{
				/* Read and fill data for an every column */
				for(i=0;i<msg_int16;i++)
				{
					pID_REC = &pIRD->id_records[i];
					if(SQL_ERROR == RecvString(pConnection, (TCHAR**)&pID_REC->name, SQL_NTS, msg->length) || /* column name */
					   RecvInt32(pConnection, &pID_REC->table_oid) ||                     /* object ID of the table or zero */
					   RecvInt16(pConnection, &pID_REC->column_attribute) ||              /* attribute number of the column or zero */
					   RecvInt32(pConnection, &pID_REC->type_oid) ||                      /* object ID of the field's data type */
					   RecvInt16(pConnection, &pID_REC->type_size) ||                     /* data type size, negative values denote variable-width types */
					   RecvInt32(pConnection, (unsigned int*)&pID_REC->type_modifier) ||  /* type modifier */
					   RecvInt16(pConnection, &pID_REC->format_code)                      /* format code being used for the field, currently zero or one */
					  )
					{
						nRet = SQL_ERROR;
						break;
					}
					/* backend always returnes column alias */
					pID_REC->unnamed = SQL_NAMED;
					if (!STMT_USE_BUFFERING(pStatement))
						pID_REC->max_column_length = 213;
				}
				PopulateID(pIRD, pConnection->environment->attributes.odbc_version, pConnection->mjet);
				msg->type = MSG_RowDescription;
				nRet = SQL_SUCCESS;
			}
			RET_DESCRIPTOR(pIRD);
			return nRet;
		case 'S':
			/* run-time parameter status report */
			if (SQL_ERROR == RecvString(pConnection, &msg_string, SQL_NTS, msg->length))
				return SQL_ERROR;
			FREE(msg_string);
			if (SQL_ERROR == RecvString(pConnection, &msg_string, SQL_NTS, msg->length))
				return SQL_ERROR;
			FREE(msg_string);
			msg->type = MSG_ParameterStatus;
			break;

		/* parameter description, 
		 * we can use this message - to determine parameter's info for SQL_ATTR_ENABLE_AUTO_IPD support
		 */
		case 't':
			if (RecvInt16(pConnection, &msg_int16)) /* number of parameters used by the statement */
				return SQL_ERROR;

			pIPD = GET_DESCRIPTOR(pStatement->ipd);
			if (SQL_ERROR == ReallocDescriptorRecords(pIPD, msg_int16))
			{
				SetError(SQL_HANDLE_STMT, pStatement, ERR_NOT_ENOUGH_MEMORY, NULL);
				for(i=0;i<msg_int16;i++)
					RecvInt32(pConnection, &msg_int32); /* we don't need this data */
				nRet = SQL_ERROR;
			}
			else for(i=0;i<msg_int16;i++)
			{
				/* read parameters data */
//				EmptyDescriptorRecord(pIPD, i);
				if (RecvInt32(pConnection, &pIPD->id_records[i].type_oid)) /* the object ID of the parameter data type */
				{
					nRet = SQL_ERROR;
					break;
				}
			}
			RET_DESCRIPTOR(pIPD);
			msg->type = MSG_ParameterDescription;
			break;
		case 'V':
			/* Int32 ... */
			msg->type = MSG_FunctionCallResponse;
			break;
		case 'N':
			/* Int32 ... */
			{
				BYTE buffer[2000];
				RecvBytes(pConnection, buffer, msg->length);
			}
			msg->type = MSG_NoticeResponse;
			break;
		case 'H':
			/* Int32 ... */
			msg->type = MSG_CopyOutResponse;
			break;
		case 'G':
			/* Int32 ... */
			msg->type = MSG_CopyInResponse;
			break;
		case 'd':
			/* Int32,... */
			msg->type = MSG_CopyData;
			break;

		default:
			msg->type = MSG_Unknown;
			break;
	}
	return nRet;
}
/*-----------------------------------------------------------------------------
 * Stmt_SendMessageToBackend
 *-----------------------------------------------------------------------------
 */
SQLRETURN Stmt_SendMessageToBackend(Connection* pConnection, MessageType msgType, void* param)
{
	SQLRETURN nRet;
	ENTER_CRITICAL_SECTION(pConnection);
	nRet = SendMessageToBackend(pConnection, msgType, param);
	LEAVE_CRITICAL_SECTION(pConnection);
	return nRet;
}

/*-----------------------------------------------------------------------------
 * SendMessageToBackend
 *-----------------------------------------------------------------------------
 */
SQLRETURN SendMessageToBackend(Connection* pConnection, MessageType msgType, void* data)
{
	SQLRETURN  nRet = SQL_ERROR;
	ID_REC* pID_REC;
	AD_REC* pAD_REC;
	Message msg;
	
	uint16 j; /* parameters number */
	uint16 parametersCount;
	uint32 unMessageLength;

	union
	{
		Statement* statement;
		TCHAR*     text;
		void*      data;
	} param;

	param.data = data;
	/* prepare frontend's message */
	switch(msgType)
	{
		/* Simple query message, SQL without parameters */
		case MSG_Query:
			if (SendByte(pConnection, 'Q') ||
			    SendInt32(pConnection, (NULL == param.text) ? 5 : SEND_STRLEN(param.text, SQL_NTS) + 5) ||
					SendString(pConnection, param.text, SQL_NTS) ||
			    sock_send(pConnection)
			   )
				nRet = SQL_ERROR;
			else
				nRet = SQL_SUCCESS;
			break;
		/* Reply with password (crypted or uncrypted) on backend's demand */
		case MSG_PasswordMessage:
			if (SendByte(pConnection, 'p') ||
			    SendInt32(pConnection, (unMessageLength = strlen((char*)param.data)) + 5) ||
			    SendBytes(pConnection, (BYTE*)param.data, unMessageLength+1) ||
			    sock_send(pConnection)
			   )
				nRet = SQL_ERROR;
			else
				nRet = SQL_SUCCESS;
			break;
		/* cancel request */
		case MSG_CancelRequest:
			if (SendInt32(pConnection, 16) ||
			    SendInt32(pConnection, 80877102) ||
					SendInt32(pConnection, pConnection->processId) ||
					SendInt32(pConnection, pConnection->secretKey) ||
			    sock_send(pConnection)
			   )
				nRet = SQL_ERROR;
			else
				nRet = SQL_SUCCESS;
			break;
		/* andyk READY */
		case MSG_Flush:
			if (SendByte(pConnection, 'H') ||
			    SendInt32(pConnection, 4) ||
			    sock_send(pConnection)
			   )
				nRet = SQL_ERROR;
			else
				nRet =SQL_SUCCESS;
			break;

		/* Send query to execute prepared portal with given name */
		case MSG_Execute:
		{
			if (SendByte(pConnection, 'E') ||
					SendInt32(pConnection, SEND_STRLEN(param.statement->query.cursorName, SQL_NTS) + 2*sizeof(uint32) + sizeof(uint8)) ||
			    SendString(pConnection, param.statement->query.cursorName, SQL_NTS) ||
					SendInt32(pConnection, (STMT_USE_BUFFERING(param.statement)) ? 0 : param.statement->ard->header.array_size) || /* number of rows to return */
			    sock_send(pConnection)
			   )
				nRet = SQL_ERROR;
			else
				nRet = SQL_SUCCESS;
			break;
		}
		/* andyk READY */
		case MSG_Terminate:
			if (SendByte(pConnection, 'X') ||
			    SendInt32(pConnection, 4) ||
			    sock_send(pConnection)
			   )
				nRet = SQL_ERROR;
			else
				nRet = SQL_SUCCESS;
			break;
		/* copy data message */
		case MSG_CopyData:
			if (SendByte(pConnection, 'd') ||
			    SendInt32(pConnection, *(uint32*) param.data) ||
					SendBytes(pConnection, (BYTE*) param.data + 4, *(uint32*) param.data) ||
			    sock_send(pConnection)
			   )
				nRet = SQL_ERROR;
			else
				nRet =SQL_SUCCESS;
			break;
		/* copy failed message */
		case MSG_CopyFail:
			if (SendByte(pConnection, 'f') ||
			    SendInt32(pConnection, SEND_STRLEN(param.text, SQL_NTS) + 5) ||
					SendString(pConnection, param.text, SQL_NTS) ||
			    sock_send(pConnection)
			   )
				nRet = SQL_ERROR;
			else
				nRet =SQL_SUCCESS;
			break;
		/* copy done message */
		case MSG_CopyDone:
			if (SendByte(pConnection, 'c') ||
			    SendInt32(pConnection, 4) ||
			    sock_send(pConnection)
			   )
				nRet = SQL_ERROR;
			else
				nRet =SQL_SUCCESS;
			break;
		/* andyk READY */
		case MSG_Sync:
			if (SendByte(pConnection, 'S') ||
			    SendInt32(pConnection, 4) ||
			    sock_send(pConnection)
			   )
				nRet = SQL_ERROR;
			else
				nRet =SQL_SUCCESS;
			break;
		/* close prepared statement */
		case MSG_CloseStatement:
		{
			if (SendByte(pConnection, 'C') ||
				  SendInt32(pConnection, SEND_STRLEN(param.statement->query.cursorName, SQL_NTS) + sizeof(uint32) + 2*sizeof(uint8)) ||
					SendByte(pConnection, 'S') ||
					SendString(pConnection, param.statement->query.cursorName, SQL_NTS) ||
			    sock_send(pConnection)
			   )
				nRet = SQL_ERROR;
			else
				nRet =SQL_SUCCESS;
			break;
		}
		/* close prepared portal */
		case MSG_ClosePortal:
		{
			if (SendByte(pConnection, 'C') ||
			    SendInt32(pConnection, SEND_STRLEN(param.statement->query.cursorName, SQL_NTS) + sizeof(uint32) + 2*sizeof(uint8)) ||
					SendByte(pConnection, 'P') ||
					SendString(pConnection, param.statement->query.cursorName, SQL_NTS) ||
			    sock_send(pConnection)
			   )
				nRet = SQL_ERROR;
			else
				nRet =SQL_SUCCESS;
			break;
		}
		/* Send query text to parse for the prepared statement */
		case MSG_Parse:
			/* calculate the message length */
			unMessageLength = sizeof(uint32) +                       /* message length */
			                  SEND_STRLEN(param.statement->query.cursorName, SQL_NTS) + /* name length */
												sizeof(uint8) +                        /* null terminal for name */
												((NULL == param.statement->query.cursorName) ? 1 : SEND_STRLEN(param.statement->query.query, SQL_NTS)) +      /* query length */
												sizeof(uint8) +                        /* null terminal for query */
												sizeof(uint16);                        /* number of parameters */
			if (SendByte(pConnection, 'P') ||
			    SendInt32(pConnection, unMessageLength) ||
					SendString(pConnection, param.statement->query.cursorName, SQL_NTS) ||
					SendString(pConnection, (NULL == param.statement->query.query) ? _T(" ") : param.statement->query.query, SQL_NTS) ||
					SendInt16(pConnection, 0) ||
					sock_send(pConnection)
			   )
				nRet = SQL_ERROR;
			else
				nRet = SQL_SUCCESS;
			break;
		/* describe statement or portal */
		case MSG_Describe:
		{
			if (SendByte(pConnection, 'D') ||
			    SendInt32(pConnection, SEND_STRLEN(param.statement->query.cursorName, SQL_NTS) + sizeof(uint32) + 2*sizeof(uint8)) ||
					SendByte(pConnection, 'S') ||
					SendString(pConnection, param.statement->query.cursorName, SQL_NTS) ||
			    sock_send(pConnection)
			   )
				nRet = SQL_ERROR;
			else
				nRet = SQL_SUCCESS;
			break;
		}
		/* bind parameters */
		case MSG_Bind:
			/* calculate message length */
			pID_REC = param.statement->ipd->id_records;
			pAD_REC = param.statement->apd->ad_records;
			parametersCount = param.statement->ipd->header.count;

			msg.length = sizeof(uint32) +                           /* message length */
			             3*sizeof(uint16) +                         /* 0-text columns, 0-text params, number of params */
			             SEND_STRLEN(param.statement->query.cursorName, SQL_NTS) * 2 + /* statement and portal names */
									 2*sizeof(uint8) +                          /* null terminals for names */
			             parametersCount*sizeof(uint32);
			
			for (j=0;j<parametersCount;j++)
				if (FIELD_NULL != *(int*)pID_REC[j].common.data_ptr)
					msg.length += SEND_STRLEN((TCHAR*)(((int*)pID_REC[j].common.data_ptr)+1), *(int*)pID_REC[j].common.data_ptr); /* data length for every parameter */

			if (SendByte(pConnection, 'B') ||
			    SendInt32(pConnection, msg.length) ||
					SendString(pConnection, param.statement->query.cursorName, SQL_NTS) ||  /* portal name */
					SendString(pConnection, param.statement->query.cursorName, SQL_NTS) ||  /* statement name */
					SendInt16(pConnection, 0) ||
					SendInt16(pConnection, parametersCount) /* all parameters in text */
			   )
				nRet = SQL_ERROR;
			else
			{/* send data for every parameter, -1 means 'null'*/
				for (j=0;j<parametersCount;j++)
				{
					if (FIELD_NULL == *(int*)pID_REC[j].common.data_ptr)
					{
						if (SendInt32(pConnection, -1))
						{
							nRet = SQL_ERROR;
							break;
						}								
					}
					else
					{
						if (SendInt32(pConnection, SEND_STRLEN((TCHAR*)(((int*)pID_REC[j].common.data_ptr)+1), *(int*)pID_REC[j].common.data_ptr)) ||
						    SendString(pConnection, (TCHAR*)(((int*)pID_REC[j].common.data_ptr)+1), *(int*)pID_REC[j].common.data_ptr))
						{
							nRet = SQL_ERROR;
							break;
						}
					}
				}

				if (j == parametersCount)
				{
					if (SendInt16(pConnection, 0) ||
					    sock_send(pConnection)
					   )
						nRet = SQL_ERROR;
					else
						nRet = SQL_SUCCESS;
				}
			}
			break;
		/* */
		case MSG_FunctionCall:
			/* andyk need to be done */
			nRet = SQL_ERROR;
			break;
		default:
			nRet = SQL_ERROR;
	}
	return nRet;
}

/*-----------------------------------------------------------------------------
 * FUNCTION: PemPassword_cb
 *
 * Callback function, returnes passphrase used for private rsa key 
 *-----------------------------------------------------------------------------
 */
int PemPassword_cb(char *buf, int size, int rwflag, void *password)
{
	strncpy(buf, (char *)(password), size);
	buf[size - 1] = '\0';
	return(strlen(buf));
}

/*-----------------------------------------------------------------------------
 * FUNCTION: SendStartupMessage
 *
 *     IN: pConnection with filled:
 *          ->version
 *          ->parameter[UID_PARAM]
 *          ->parameter[DATABASE_PARAM]
 * RETURN: TRUE  - sent,
 *         FALSE - other
 *-----------------------------------------------------------------------------
 */
SQLRETURN SendStartupMessage(Connection* pConnection, Message* msg)
{
	TCHAR     mode = pConnection->parameters[USESSL_PARAM][0];
	SQLRETURN nRet = SQL_ERROR;

	/* this loop enables us to connect maximum twice (one time - using SSL, second - without) */
	while (!SQL_SUCCEEDED(nRet))
	{
		switch(mode)
		{
#ifdef USE_SSL
			case _T('R'): /* require */
			case _T('P'): /* prefer  */
			{/* try to create SSL connection */
				BYTE use_ssl;

#ifdef WIN32
				if (InitSSL())
				{/* impossible to initialize required Open SSL libraries */
					SetError(SQL_HANDLE_DBC, pConnection, ERR_CANT_INIT_OPEN_SSL, NULL);
					nRet = SQL_SUCCESS_WITH_INFO;
				}
				else
#endif /* WIN32 */
				{
					if (SendInt32(pConnection, 8) ||
							SendInt32(pConnection, 80877103) ||
							sock_send(pConnection) ||
							RecvByte(pConnection, &use_ssl)
						 )
						return SQL_ERROR;

					/* check backend response */
					if ('S'     != use_ssl &&
							_T('R') == mode
						 )
					{/* server doesn't support SSL */
						SetError(SQL_HANDLE_DBC, pConnection, ERR_SERVER_REFUSED_SSL_CONNECTION, NULL);
						return SQL_ERROR;
					}

					if ('S' == use_ssl)
					{/* backend supports SSL and ready for begining handeshake */
						SQLRETURN ssl_nRet = SQL_ERROR;
						SSL*      ssl = NULL;
						SSL_CTX*  ctx = NULL;
						int       ret = 1;

						SSL_LIBRARY_INIT();
						SSL_LOAD_ERROR_STRINGS();

						/* process SSL connection */
						if (NULL != (ctx = SSL_CTX_NEW(SSLV23_CLIENT_METHOD())))
						{
							/* check for avaible certificates and try to load them */
							if (_T('\0') != pConnection->parameters[CERTIFICATE_PARAM][0])
							{
								char* certificate;
								char* private_key;
#ifndef UNICODE
								char* passphrase  = pConnection->parameters[PASSPHRASE_PARAM];

								certificate = pConnection->parameters[CERTIFICATE_PARAM];
								private_key = pConnection->parameters[PRIVATE_KEY_PARAM];
#else
								/* ssl library is not unicode unfortunately... */
								char certificate_buff[SSL_CERTIFICATE_MAX_LENGTH+1];
								char private_key_buff[SSL_PRIVATE_KEY_MAX_LENGTH+1];
								char passphrase[SSL_PWD_MAX_LENGTH+1];

								WideCharToMultiByte(0, 0, pConnection->parameters[CERTIFICATE_PARAM], wcslen(pConnection->parameters[CERTIFICATE_PARAM])+1, certificate_buff, sizeof(certificate_buff), NULL, NULL);
								WideCharToMultiByte(0, 0, pConnection->parameters[PRIVATE_KEY_PARAM], wcslen(pConnection->parameters[PRIVATE_KEY_PARAM])+1, private_key_buff, sizeof(private_key_buff), NULL, NULL);
								WideCharToMultiByte(0, 0, pConnection->parameters[PASSPHRASE_PARAM], wcslen(pConnection->parameters[PASSPHRASE_PARAM])+1, passphrase, sizeof(passphrase), NULL, NULL);

								certificate = certificate_buff;
								private_key = private_key_buff;
#endif /* UNICODE */
								if (_T('\0') == private_key[0])
								{/* try to use certificate file for private key */
									private_key = certificate;
								}
								
								SSL_CTX_SET_DEFAULT_PASSWORD_CB(ctx, PemPassword_cb);
								SSL_CTX_SET_DEFAULT_PASSWORD_CB_USERDATA(ctx, passphrase /* WARNING! stack pointer */);

								if ((1 != SSL_CTX_USE_CERTIFICATE_CHAIN_FILE(ctx, certificate)) ||
						        (1 != SSL_CTX_USE_PRIVATEKEY_FILE(ctx, private_key, SSL_FILETYPE_PEM))
								   )
								{/* error while loading certificates */
									SetError(SQL_HANDLE_DBC, pConnection, ERR_SSL_CERT_NOT_LOADED, NULL);
									ssl_nRet = SQL_SUCCESS_WITH_INFO;
								}
							}

							if ((0 != SSL_CTX_CTRL(ctx, SSL_CTRL_MODE, SSL_MODE_AUTO_RETRY, NULL)) &&
							    (0 != SSL_CTX_CTRL(ctx, SSL_CTRL_MODE, SSL_MODE_ENABLE_PARTIAL_WRITE, NULL)) &&
							    (0 != SSL_CTX_CTRL(ctx, SSL_CTRL_MODE, SSL_MODE_ENABLE_PARTIAL_WRITE, NULL)) &&
							    (NULL != (ssl = SSL_NEW(ctx))) &&
							    (0 != SSL_SET_FD(ssl, pConnection->socket)) &&
							    (1 == (ret = SSL_CONNECT(ssl)))
							   )
							{/* SSL connection succeeded */
								ssl_nRet = SQL_SUCCESS;
							}
						}

						if (SQL_ERROR == ssl_nRet)
						{/* error occured */
							pConnection->ssl = NULL;
							ret = SSL_GET_ERROR(ssl,ret);
								
							/* close SSL handles */
							SSL_FREE(ssl);
							SSL_CTX_FREE(ctx);

							/* set error */
							SetError(SQL_HANDLE_DBC, pConnection, ERR_SSL_ERROR, NULL);
							nRet = SQL_SUCCESS_WITH_INFO;
						}
						else
							pConnection->ssl = ssl;
					}
					else
					{/* no SSL will be used */
						SetError(SQL_HANDLE_DBC, pConnection, ERR_SERVER_REFUSED_SSL_CONNECTION, NULL);
						nRet = SQL_SUCCESS_WITH_INFO;
					}
				}
			}

			if (SQL_SUCCESS_WITH_INFO == nRet)
			{
				if (_T('P') != pConnection->parameters[USESSL_PARAM][0])
					return SQL_ERROR;
				mode = _T('I');
				nRet = SQL_ERROR;
				break;
			}

			/* WARNING! NO BREAK */
			case _T('A'): /* allow */
#endif /* USE_SSL */
			case _T('I'): /* ignore - default */
			default:
			{
				unsigned int unMessageLength;
				unsigned int unUserNameLength;
				unsigned int unDatabaseNameLength;

				/* calculating buffer size for the Message */
				if (0 < (unUserNameLength = SEND_STRLEN(pConnection->parameters[UID_PARAM], SQL_NTS)))
					unUserNameLength += SEND_STRLEN(_T("user"), SQL_NTS)+2;
				if (0 < (unDatabaseNameLength = SEND_STRLEN(pConnection->parameters[DATABASE_PARAM], SQL_NTS)))
					unDatabaseNameLength += SEND_STRLEN(_T("database"), SQL_NTS)+2;
				unMessageLength = sizeof(uint32) + sizeof(ProtocolVersion) + unUserNameLength + unDatabaseNameLength + 1;

				if (SendInt32(pConnection, unMessageLength) ||          /* Message length */
						SendInt32(pConnection, pConnection->version) ||     /* Protocol version */
						SendString(pConnection, _T("user"), SQL_NTS) ||     /* User name, required parameter, no default */
						SendString(pConnection, pConnection->parameters[UID_PARAM], SQL_NTS) ||
						SendString(pConnection, _T("database"), SQL_NTS) || /* Database, defaults to the user name */
						SendString(pConnection, pConnection->parameters[DATABASE_PARAM], SQL_NTS) ||
						SendByte(pConnection, '\0') ||                      /* Options not implemented */
						sock_send(pConnection)
					 )
				{
					return SQL_ERROR;
				}
				else
				{/* looks OK, let's get response from backend */
					msg->message = NULL;
					if (SQL_ERROR == GetMessageFromBackend(pConnection, msg, NULL))
					{/* backend closed connection */
						if (_T('A') == mode)
						{/* try to connect using SSL in 'prefer' mode */
							sock_connect(pConnection);
							mode = _T('P');
						}
						else
							return SQL_ERROR;
					}
					else if (MSG_ErrorResponse == msg->type)
					{
						SetError(SQL_HANDLE_DBC, pConnection, ERR_ERROR_DURING_CONNECT, msg->message, NULL);
						FREE(msg->message);
						if (_T('A') == pConnection->parameters[USESSL_PARAM][0])
						{/* try to connect using SSL in 'prefer' mode */
							sock_close(pConnection);
							sock_connect(pConnection);
							mode = _T('P');
						}
						else
							nRet = SQL_SUCCESS_WITH_INFO;
					}
					else
						nRet = SQL_SUCCESS;
				}
			}
		}
	}
	return nRet;
}

SQLRETURN WaitForBackendReply(Connection* pConnection, MessageType msgType, Statement* pStatement)
{
	Message   msg;
	SQLRETURN nRet = SQL_SUCCESS;

	do
	{
		if (SQL_ERROR == GetMessageFromBackend(pConnection, &msg, pStatement))
			return SQL_ERROR;

		if (MSG_ErrorResponse == msg.type)
		{
			/* prepare and set error */
			if (NULL != pStatement)
			{
				SetError(SQL_HANDLE_STMT, pStatement, ERR_SERVER_ERROR, msg.message, NULL);
			}
			else
			{
				SetError(SQL_HANDLE_DBC, pConnection, ERR_SERVER_ERROR, msg.message, NULL);			
			}
			nRet = SQL_ERROR;
		}
		
		if (MSG_ReadyForQuery == msg.type)
		{
			if (SQL_ERROR == nRet  && pStatement && AUTOCOMMIT_MODE_ON(pStatement))
				EndTransaction(SQL_HANDLE_STMT, pStatement, SQL_ROLLBACK, TO_DRIVER);

			return nRet;
		}

	} while (msg.type != msgType);

	return nRet;
}


/*-----------------------------------------------------------------------------
 * FUNCTION: DeclareStatement
 *
 * DESCRIPTION: checks - do we need statement at all, if
 *              no  - close existing (if any)
 *              yes - send declare to backend
 * DESCRIPTION: use cursor name as statement name.
 *-----------------------------------------------------------------------------
 */
SQLRETURN
DeclareStatement(Statement* pStatement, BOOL ForceDeclaration)
{
	SQLRETURN nRet = SQL_SUCCESS;
	
	switch (pStatement->query.prepareType)
	{
		case PT_FR_BK_PROTOCOL:
		case PT_SERVER_SIDE:

			/* close prepared statement if any */
			if (STATEMENT_DECLARED(pStatement))
				CloseDeclared(pStatement, _T('s'));
			else if (SQL_ERROR == SetCursorName(pStatement, NULL, 0, TRUE))
				return SQL_ERROR;

			switch (pStatement->query.prepareType)
			{
				case PT_FR_BK_PROTOCOL:
					/* check - 'do we need statement?' */
					if ((TRUE == ForceDeclaration) ||
							(0 < pStatement->query.nParametersNumber) ||
							((ST_SELECT == pStatement->query.type) &&
							 (!STMT_USE_BUFFERING(pStatement))
						 ))
					{/* statement needed, declare it as new */
						if(SQL_ERROR == Stmt_SendMessageToBackend(pStatement->connection, MSG_Parse, pStatement) ||
							 SQL_ERROR == Stmt_SendMessageToBackend(pStatement->connection, MSG_Describe, pStatement) ||
							 SQL_ERROR == Stmt_SendMessageToBackend(pStatement->connection, MSG_Sync, NULL) ||
							 SQL_ERROR == WaitForBackendReply(pStatement->connection, MSG_ParseComplete, pStatement) ||
							 SQL_ERROR == WaitForBackendReply(pStatement->connection, MSG_ReadyForQuery, pStatement)
							)
							return SQL_ERROR;
						else
						{
							Descriptor* pIPD = GET_DESCRIPTOR(pStatement->ipd);
							if (SQL_TRUE == pStatement->attributes.enable_auto_ipd)
								PopulateID(pIPD, pStatement->connection->environment->attributes.odbc_version, pStatement->connection->mjet); /* populate IPD if SQL_ATTR_ENABLE_AUTO_IPD = SQL_TRUE */
							nRet = SetDescField(pIPD, 0, SQL_DESC_COUNT, (SQLPOINTER)(SQLLEN_TO_POINTER)pStatement->query.nParametersNumber, SQL_IS_SMALLINT);
							RET_DESCRIPTOR(pIPD);

							SET_STATEMENT_DECLARED(pStatement);
						}
					}
					break;
				case PT_SERVER_SIDE:
					nRet = (0 == pStatement->query.nParametersNumber) ? ((SQL_SUCCESS == Stmt_SendMessageToBackend(pStatement->connection, MSG_Query, pStatement->query.query)
						) ? WaitForBackendReply(pStatement->connection, MSG_ReadyForQuery, pStatement) : SQL_ERROR) : SQL_SUCCESS;
					
					SET_STATEMENT_DECLARED(pStatement);
					break;
			}
		case PT_SIMPLE_QUERY:
			/* nothing to declare */
			break;
	}

	return nRet;
}

/*-----------------------------------------------------------------------------
 * FUNCTION: DeclarePortal
 *
 * DESCRIPTION: use cursor name as portal name, we assume that statement is already
 *              declared (and has the same name)
 *-----------------------------------------------------------------------------
 */
SQLRETURN
DeclarePortal(Statement* pStatement)
{
	if (PORTAL_NEEDS_REDECLARATION(pStatement))
	{/* declare statement and free declared portal (if any) */
		if (SQL_ERROR == CloseDeclared(pStatement, _T('p')))
			return SQL_ERROR;

		/* prepare portal, send variables */
		if (SQL_ERROR == BeginTransaction(pStatement, TO_DRIVER) ||
				SQL_ERROR == Stmt_SendMessageToBackend(pStatement->connection, MSG_Bind, pStatement) ||
				SQL_ERROR == Stmt_SendMessageToBackend(pStatement->connection, MSG_Sync, pStatement) ||
				SQL_ERROR == WaitForBackendReply(pStatement->connection, MSG_BindComplete, pStatement) ||
				SQL_ERROR == WaitForBackendReply(pStatement->connection, MSG_ReadyForQuery, pStatement)
			 )
			return SQL_ERROR;

		/* portal successfully declared */
		SET_PORTAL_DECLARED(pStatement);
	}

	return SQL_SUCCESS;
}

/*-----------------------------------------------------------------------------
 * FUNCTION: CloseDeclared
 *
 * DESCRIPTION: closes corresponding prepared statement ('s') or portal ('p')
 *              on backend (if any)
 *
 * RETURN: SQL_SUCCESS - prepared statement/portal closed
 *                     - no statement/portal on backend
 *         SQL_ERROR   - error in backend communication
*-----------------------------------------------------------------------------
 */
SQLRETURN
CloseDeclared(Statement* pStatement, TCHAR chType)
{
	switch(chType)
	{
		/* statement*/
		case _T('s'):
			if (STATEMENT_DECLARED(pStatement))
			{
				if (PT_SERVER_SIDE_CURSOR == pStatement->query.prepareType)
				{
					TCHAR close_query[STR_SIZEOF(PQ_CLOSE) + STR_SIZEOF("\"") + MAX_CURSOR_NAME_LENGTH + 1];
					int   cursor_length = _tcslen(pStatement->query.cursorName);

					/* prepare CLOSE query */
					_tcsncpy(close_query, _T(PQ_CLOSE), STR_SIZEOF(PQ_CLOSE));
					_tcsncpy(close_query + STR_SIZEOF(PQ_CLOSE), pStatement->query.cursorName, cursor_length);
					_tcscpy(close_query + STR_SIZEOF(PQ_CLOSE) + cursor_length, _T("\""));

					if (SQL_ERROR == Stmt_SendMessageToBackend(pStatement->connection, MSG_Query, close_query) ||
							SQL_ERROR == WaitForBackendReply(pStatement->connection, MSG_ReadyForQuery, NULL)
						 )
						 return SQL_ERROR;
				}
				else if (SQL_ERROR == Stmt_SendMessageToBackend(pStatement->connection, MSG_CloseStatement, pStatement) ||
				         SQL_ERROR == Stmt_SendMessageToBackend(pStatement->connection, MSG_Sync, NULL) ||
				         SQL_ERROR == WaitForBackendReply(pStatement->connection, MSG_CloseComplete, NULL) ||
				         SQL_ERROR == WaitForBackendReply(pStatement->connection, MSG_ReadyForQuery, NULL)
			          )
					return SQL_ERROR;

				SET_STATEMENT_UNDECLARED(pStatement);
			}
		/* NO BREAK! closing statement - forces to close associated portal */
		/* portal */
		case _T('p'):
			if (PORTAL_DECLARED(pStatement))
			{
				if (SQL_ERROR == Stmt_SendMessageToBackend(pStatement->connection, MSG_ClosePortal, pStatement) ||
				    SQL_ERROR == Stmt_SendMessageToBackend(pStatement->connection, MSG_Sync, NULL) ||
				    SQL_ERROR == WaitForBackendReply(pStatement->connection, MSG_CloseComplete, NULL) ||
				    SQL_ERROR == WaitForBackendReply(pStatement->connection, MSG_ReadyForQuery, NULL)
			     )
					return SQL_ERROR;
				SET_PORTAL_UNDECLARED(pStatement);
			}
			break;
		default:
			/* impossible - called only from driver with right argument */
			return SQL_ERROR;
	}

	return SQL_SUCCESS;
}

