#include "connection.h"
#include "statement.h"
#include "protocol.h"

Statement* AllocStatement(Connection* pConnection)
{
	Statement* pStatement = (Statement*)malloc(sizeof(Statement));
	if (pStatement)
	{
		memset(pStatement, '\0', sizeof(Statement));

		pStatement->connection = pConnection;
		pStatement->state = SS_ALLOCATED;

		pStatement->use_buffering = (_T('Y') == pConnection->parameters[USEBUFFERING_PARAM][0]) ? TRUE : FALSE;
		/* results */
		pStatement->results.pHead     = NULL;
		pStatement->results.pTail     = NULL;
		pStatement->results.pLastHead = NULL;

		pStatement->results.unFreeSpaceInTailBlock = 0;
		pStatement->fetch_position = -1;

		pStatement->need_data      = _T('\0');
		/* resultsets */
		pStatement->results.irds.allocated =  0;
		pStatement->results.irds.used      =  0;
		pStatement->results.current        = -1; /* use predefined resultset */

		/* default attributes */
		pStatement->attributes.no_scan         = SQL_NOSCAN_OFF;
		pStatement->attributes.use_bookmarks   = SQL_UB_DEFAULT;
		pStatement->attributes.enable_auto_ipd = SQL_FALSE;
		pStatement->attributes.metadata_id     = SQL_FALSE;

		/* query */
		pStatement->query.nParametersNumber = 0;
		pStatement->query.query = NULL;

		/* prepare implicitly allocated descriptors */
		pStatement->d_apd.type = DT_APD;
		pStatement->d_ard.type = DT_ARD;
		pStatement->d_ipd.type = DT_IPD;
		pStatement->d_ird.type = DT_IRD;

		pStatement->d_apd.connection = pConnection;
		pStatement->d_ard.connection = pConnection;
		pStatement->d_ipd.connection = pConnection;
		pStatement->d_ird.connection = pConnection;

		InitDescriptor(pStatement->apd = &pStatement->d_apd, SQL_DESC_ALLOC_AUTO);
		InitDescriptor(pStatement->ard = &pStatement->d_ard, SQL_DESC_ALLOC_AUTO);
		InitDescriptor(pStatement->ipd = &pStatement->d_ipd, SQL_DESC_ALLOC_AUTO);
		InitDescriptor(pStatement->ird = &pStatement->d_ird, SQL_DESC_ALLOC_AUTO);

		/* add to connection's list */
		AddItem(&pConnection->statements, pStatement);

		/* autogenerate cursor name */
		SetCursorName(pStatement, NULL, 0, TRUE);

		InitDiag(&pStatement->diag);
		_INIT_CRITICAL_SECTION(pStatement);
	}

	return pStatement;
}

/*-----------------------------------------------------------------------------
 * FUNCTION: ResetStatement
 *
 * DESCRIPTION: closes result and prepqred statement if any
 *-----------------------------------------------------------------------------
 */
SQLRETURN ResetStatement(Statement* pStatement)
{
	Descriptor* pIRD;

	/* clear statement results */
	pStatement->need_data = _T('\0');
	EmptyStatementResults(pStatement);

	pIRD = GET_DESCRIPTOR(pStatement->ird);
	if (0 < pIRD->header.count)
		SetDescField(pIRD, 0, SQL_DESC_COUNT, (SQLPOINTER)(SQLSMALLINT)0, SQL_IS_SMALLINT);
	RET_DESCRIPTOR(pIRD);

	/* close previously prepared statement if needed  */
	if (0 != (SS_STATEMENT_DECLARED & pStatement->state))
	{
		if (SQL_ERROR == CloseDeclared(pStatement, _T('s')) ||
		    (0 != (SS_PORTAL_DECLARED & pStatement->state) &&
		     SQL_ERROR == CloseDeclared(pStatement, _T('p'))
				)
		   )
	
		return SQL_ERROR;
	}

	pStatement->use_buffering = (_T('Y') == pStatement->connection->parameters[USEBUFFERING_PARAM][0]) ? TRUE : FALSE;
	pStatement->state = SS_ALLOCATED;
	return SQL_SUCCESS;
}

/*-----------------------------------------------------------------------------
 * FUNCTION: BindParameter
 *
 * DESCRIPTION: No need to check ParameterNumber >= 1, Driver Manger does it.
 *-----------------------------------------------------------------------------
 */
SQLRETURN
BindParameter(Statement* pStatement, SQLUSMALLINT ParameterNumber, SQLSMALLINT InputOutputType,
              SQLSMALLINT ValueType, SQLSMALLINT ParameterType, SQLUINTEGER ColumnSize,
              SQLSMALLINT DecimalDigits, SQLPOINTER ParameterValuePtr, SQLINTEGER BufferLength,
							SQLINTEGER* StrLen_or_IndPtr)
{
	SQLRETURN   nRet;
	SQLSMALLINT savedAPDCount;
	SQLSMALLINT savedIPDCount;

	Descriptor* pAPD;
	Descriptor* pIPD;

	/* get descriptors and save there COUNT field */
	pAPD = GET_DESCRIPTOR(pStatement->apd);
	pIPD = GET_DESCRIPTOR(pStatement->ipd);
	savedAPDCount = pAPD->header.count;
	savedIPDCount = pIPD->header.count;
	
	if (SQL_ERROR == ReallocDescriptorRecords(pAPD, ParameterNumber) ||
	    SQL_ERROR == ReallocDescriptorRecords(pIPD, ParameterNumber)
	   )
	{
		SetError(SQL_HANDLE_STMT, pStatement, ERR_NOT_ENOUGH_MEMORY, NULL);
		nRet = SQL_ERROR;
	}
	else
	{
		AD_REC* pAD_REC;
		ID_REC* pID_REC;

		ParameterNumber--;
		nRet = SQL_SUCCESS;
		pAD_REC = &pAPD->ad_records[ParameterNumber];
		pID_REC = &pIPD->id_records[ParameterNumber];
	
		/* set APD's & IPD's requested parameter data */
		pAD_REC->bound               = SQL_TRUE;
		pAD_REC->common.octet_length = BufferLength;
		pAD_REC->common.data_ptr     = ParameterValuePtr;
		pAD_REC->octet_length_ptr    = StrLen_or_IndPtr;
		pAD_REC->indicator_ptr       = StrLen_or_IndPtr;

		pID_REC->common.scale        = DecimalDigits;
		pID_REC->parameter_type      = InputOutputType;

		TranslateType(&pAD_REC->common, ValueType,     DecimalDigits, 0,          C_TYPE);
		TranslateType(&pID_REC->common, ParameterType, DecimalDigits, ColumnSize, SQL_TYPE);
		SQLTypeDescriptor(ValueType, 0, &BufferLength, NULL, &pID_REC->common.scale, NULL, &pID_REC->display_size, NULL, NULL);

		if (SQL_NUMERIC == pAD_REC->common.type)
		{
			pID_REC->common.precision = 0; /* andyk change */
			pID_REC->common.scale = 0;
		}

		/* if an error occured - restore saved COUNT fields */
		if (SQL_ERROR == nRet)
		{
			pAPD->header.count = savedAPDCount;
			pIPD->header.count = savedIPDCount;
		}
	}
	/* end */
	RET_DESCRIPTOR(pAPD);
	RET_DESCRIPTOR(pIPD);
	return nRet;
}

/*-----------------------------------------------------------------------------
 * FreeStatement
 *-----------------------------------------------------------------------------
 */
SQLRETURN FreeStatement(Statement* pStatement, SQLUSMALLINT Option)
{
	Descriptor* pDesc;

	switch (Option)
	{
		/* unbind all bound columns */
		case SQL_UNBIND:
			pDesc = GET_DESCRIPTOR(pStatement->ard);
			pDesc->header.count = 0;
			RET_DESCRIPTOR(pDesc);
			break;
		/* unbind all bound parameters */
		case SQL_RESET_PARAMS:
			pDesc = GET_DESCRIPTOR(pStatement->apd);
			pDesc->header.count = 0;
			RET_DESCRIPTOR(pDesc);
			break;
		/* close cursor on this statement */
		case SQL_CLOSE:
			EmptyStatementResults(pStatement);
			pStatement->state ^= SS_EXECUTED;

			/* we should commit transaction if it was opened only to process SELECT query
			 * when USEBUFFERING_PARAM = N
			 */
			if (STMT_USE_BUFFERING(pStatement) &&
			     0 != (SS_TRANSACTION_OPENED & pStatement->state)
			   )
			{
				EndTransaction(SQL_HANDLE_STMT, pStatement, SQL_COMMIT, TO_DRIVER);
			}

			break;
		default:
			ENTER_CRITICAL_SECTION(pStatement);
			EmptyStatementResults(pStatement);

			/* full close */
			RemoveItem(&pStatement->connection->statements, pStatement);
			/* free results */

			while (NULL != pStatement->results.pHead)
			{
				pStatement->results.pTail = pStatement->results.pHead;
				pStatement->results.pHead = pStatement->results.pHead->next;
				FreeBlock(pStatement->results.pTail);
			}

			free(pStatement->query.query);

			
			FreeDescriptor(pStatement->ird);
			FreeDescriptor(pStatement->ipd);
			FreeDescriptor(pStatement->ard);
			FreeDescriptor(pStatement->apd);

			FreeDiag(&pStatement->diag);
			_DELETE_CRITICAL_SECTION(pStatement);
			free(pStatement);
			return SQL_SUCCESS;
	}

	return SQL_SUCCESS;
}

/*-----------------------------------------------------------------------------
 * EmptyStatementResults
 *-----------------------------------------------------------------------------
 */
void EmptyStatementResults(Statement* pStatement)
{
	int i;
	Descriptor* pIRD;
	/* do not touch here bound parameters */
	if (NULL != pStatement->results.pHead)
	{
		*(int*)pStatement->results.pHead->data = FIELD_TERMINATOR;
		pStatement->results.pLastHead = pStatement->results.pTail = pStatement->results.pHead;
		pStatement->results.pLastHeadData = (int*) pStatement->results.pHead->data;
		pStatement->results.pTailField = (int*) pStatement->results.pTail->data;
		pStatement->results.unFreeSpaceInTailBlock = RESULTED_ROWS_BLOCK_SIZE;
	}

	pIRD = GET_DESCRIPTOR(pStatement->ird);
	if (NULL != pIRD->id_records && NULL != pIRD->id_records[0].common.data_ptr)
	{
		pIRD->header.array_size = 0;
		free(pIRD->id_records[0].common.data_ptr);
		pIRD->id_records[0].common.data_ptr = NULL;
	}
	RET_DESCRIPTOR(pIRD);

	for (i=0;i<pStatement->results.irds.used;i++)
		FreeDescriptor(pStatement->results.irds.handles[i]);
	
	pStatement->ird->header.rows_affected =  0;
	pStatement->results.irds.used  =  0;
	pStatement->fetch_position     = -1;
	pStatement->results.current    = -1;

}

/*-----------------------------------------------------------------------------
 * FUNCTION: PrepareStatement
 *-----------------------------------------------------------------------------
 */
SQLRETURN
PrepareStatement(Statement* pStatement, SQLTCHAR* StatementText, SQLINTEGER TextLength)
{
	SQLRETURN nRet;

	if (0 != (SS_ALLOCATED & pStatement->state))
	{
		PrepareQuery(pStatement, StatementText, TextLength);
		return DeclareStatement(pStatement, TRUE);
	}
	else
		nRet = SQL_ERROR;
	
	/* andyk wrong statement's state */
	return nRet;
}

/*-----------------------------------------------------------------------------
 * FUNCTION: PopulateID
 *
 * DESCRIPTION: converts information about columns, retrived from backend into
 *              data, usefull for application
 *-----------------------------------------------------------------------------
 */
SQLRETURN
PopulateID(Descriptor* pDescriptor)
{
	int i;

	/* fill implementation descriptor fields with data */
	for (i=0;i<pDescriptor->header.count;i++)
	{
		SQLSMALLINT processed = SQL_TRUE;
		SQLSMALLINT type      = pDescriptor->type;
		ID_REC*     pID_REC   = &pDescriptor->id_records[i];

		pID_REC->unnamed        = SQL_UNNAMED;
		pID_REC->is_unsigned    = SQL_FALSE;
		pID_REC->nullable       = SQL_NULLABLE_UNKNOWN;
		pID_REC->parameter_type = SQL_PARAM_INPUT_OUTPUT;
		pID_REC->common.scale   = 0;

		pID_REC->type_name = NULL;
		
		PostgreTypeConverter(pID_REC->type_oid, pID_REC->type_modifier, &pID_REC->type_name, 0,
		                     &pID_REC->common.concise_type, &pID_REC->common.length, &pID_REC->common.precision, &pID_REC->common.scale, NULL, NULL);

		SQLTypeDescriptor(pID_REC->common.concise_type, SQL_FALSE, &pID_REC->common.length, &pID_REC->common.precision, &pID_REC->common.scale, &pID_REC->common.type, &pID_REC->display_size, &pID_REC->common.datetime_interval_precision, &pID_REC->common.num_prec_radix);
	}

	return SQL_SUCCESS;
}

/*-----------------------------------------------------------------------------
 * FUNCTION: AllParametersBound
 *
 * RETURN: SQL_SUCCESS - no parameters, or no need to recend parameters, or some
 *                       parameters were changed and rebind to backend successfully
 *                       completed
 *-----------------------------------------------------------------------------
 */
SQLRETURN AllParametersBound(Statement* pStatement)
{
	int         i    = 0;
	SQLRETURN   nRet = SQL_SUCCESS;
	Descriptor* pIPD = NULL;
	Descriptor* pAPD = GET_DESCRIPTOR(pStatement->apd);

	/* -- check - 'all parameters bound to the driver by the application?' -- */
	if (pAPD->header.count >= pStatement->query.nParametersNumber)
		for (;i<pStatement->query.nParametersNumber;i++)
			if (SQL_FALSE == pAPD->ad_records[i].bound)
				break;

	if (i < pStatement->query.nParametersNumber)
	{/* not all parameters were bound to the driver by the application */
		SetError(SQL_HANDLE_STMT, pStatement, ERR_STMT_NOT_ENOUGH_PARAMETERS, NULL);
		RET_DESCRIPTOR(pAPD);
		return SQL_ERROR;
	}
	
	pIPD = GET_DESCRIPTOR(pStatement->ipd);
	/* -- get SQL_DATA_AT_EXEC parameter -- */
	for (i--;i>=0;i--)
		if (SQL_DATA_AT_EXEC == *pAPD->ad_records[i].indicator_ptr)
		{/* there is still some unprepared parameters */
			pStatement->need_data = _T('p');

			RET_DESCRIPTOR(pAPD);
			RET_DESCRIPTOR(pIPD);
			return SQL_NEED_DATA;
		}

	/* -- check - 'should we resend query text to parse (it changed) to the backend?' -- */
	if (STATEMENT_NEEDS_REDECLARATION(pStatement))
	  nRet = DeclareStatement(pStatement, FALSE);
	
	/* -- check - 'should we resend parameters to the backend?' -- */
	if (SQL_SUCCEEDED(nRet) && PORTAL_NEEDS_REDECLARATION(pStatement))
	{/* we need to reprepare portal because of new query text or parameter values */
		SQLRETURN nRet1 = SQL_SUCCESS;

		if (SQL_ERROR != (nRet1 = ReallocDescriptorRecords(pIPD, pStatement->query.nParametersNumber)))
		{
			for (i=0;i<pStatement->query.nParametersNumber;i++)
			{/* convert parameters from SQL_C_TYPE into backend's specific format(text) */
				CD_REC* common = &pAPD->ad_records[i].common;

				if (NULL == (pIPD->id_records[i].common.data_ptr = (SQLPOINTER)PrepareParameter(pStatement, common->data_ptr, common->octet_length, common->concise_type, pAPD->ad_records[i].indicator_ptr)))
				{/* no more memory */
					nRet1 = SQL_ERROR;
					break;
				}
			}
			pStatement->results.pLastHeadData = pStatement->results.pTailField;
		}

		if (SQL_ERROR == nRet1)
			SetError(SQL_HANDLE_STMT, pStatement, ERR_NOT_ENOUGH_MEMORY, NULL);
		else
			nRet1 = DeclarePortal(pStatement);

		nRet = MAX_ERROR(nRet, nRet1);
	}

	if (!SQL_SUCCEEDED(nRet))
		/* can't bind parameters to the server */
		SetError(SQL_HANDLE_STMT, pStatement, ERR_STMT_CANT_BIND_PARAMETERS, NULL);

	RET_DESCRIPTOR(pIPD);
	RET_DESCRIPTOR(pAPD);
	return nRet;
}

/*-----------------------------------------------------------------------------
 * FUNCTION: ExecuteStatement
 *-----------------------------------------------------------------------------
 */
SQLRETURN ExecuteStatement(Statement* pStatement, BOOL ForceBuffering)
{
	/* first of all - control transaction state */
	SQLRETURN nRet = (AUTOCOMMIT_MODE_OFF(pStatement)) ? BeginTransaction(pStatement, TO_APPLICATION) : SQL_SUCCESS;

	pStatement->use_buffering |= ForceBuffering;
	/* check for the corresponding parameters bound by the application */
	if ((SQL_ERROR != nRet) && (SQL_ERROR != (nRet = AllParametersBound(pStatement))))
	{
		if ((ST_SELECT != pStatement->query.type) ||
		    (STMT_USE_BUFFERING(pStatement))
		   )
		{
			if (PORTAL_DECLARED(pStatement))
			{/* execute prepared portal */
				nRet = ((SQL_SUCCESS == Stmt_SendMessageToBackend(pStatement->connection, MSG_Execute, pStatement)) &&
				        (SQL_SUCCESS == Stmt_SendMessageToBackend(pStatement->connection, MSG_Sync, pStatement))
				       ) ? WaitForBackendReply(pStatement->connection,  MSG_ReadyForQuery, pStatement) : SQL_ERROR;
			}
			else
			{/* use simple query protocol to get resultset */
				nRet = (SQL_SUCCESS == Stmt_SendMessageToBackend(pStatement->connection, MSG_Query, pStatement->query.query)
				       ) ? WaitForBackendReply(pStatement->connection, MSG_ReadyForQuery, pStatement) : SQL_ERROR;
			}
		}

		if (SQL_SUCCESS != nRet)
			SetError(SQL_HANDLE_STMT, pStatement, ERR_STMT_CANT_EXECUTE, NULL);
	}

	/* close transaction if needed */
	if ((STMT_USE_BUFFERING(pStatement) || (ST_SELECT != pStatement->query.type)) && (AUTOCOMMIT_MODE_ON(pStatement)))
	{
		SQLRETURN nRet1 = EndTransaction(SQL_HANDLE_STMT, pStatement, (SQL_SUCCESS == nRet) ? SQL_COMMIT : SQL_ROLLBACK, TO_DRIVER);
		nRet = MAX_ERROR(nRet, nRet1);
	}

	return nRet;
}

/*-----------------------------------------------------------------------------
 * CreateBlock
 *-----------------------------------------------------------------------------
 */
Block* AllocBlock(Block* prev, int size)
{
	Block* pBlock;
	unsigned int resulted = 0;
	if (pBlock = (Block*) malloc((resulted = (RESULTED_ROWS_BLOCK_SIZE < size) ? size : RESULTED_ROWS_BLOCK_SIZE)+sizeof(Block)-RESULTED_ROWS_BLOCK_SIZE))
	{
		pBlock->next = NULL;
		pBlock->prev = prev;
		pBlock->size = resulted;
		if (NULL != prev)
			prev->next = pBlock;
		*(int*) pBlock->data = 0;     /* no more fields avaible */
	}

	return pBlock;
}

/*-----------------------------------------------------------------------------
 * FreeBlock
 *-----------------------------------------------------------------------------
 */
void FreeBlock(Block* block)
{
	free(block);
}

/*-----------------------------------------------------------------------------
 * AddField
 *-----------------------------------------------------------------------------
 */
TCHAR* AddField(Statement* pStatement, unsigned int length)
{
	TCHAR* pRet;
	int    resulted_length = length*sizeof(TCHAR);

	if (NULL == pStatement->results.pTail)
	{
		pStatement->results.pTail = AllocBlock(NULL, resulted_length + 2*sizeof(int));
		pStatement->results.pHead = pStatement->results.pTail;
		pStatement->results.pLastHead = pStatement->results.pTail;
		pStatement->results.pTailField = (int*) pStatement->results.pTail->data;
		pStatement->results.unFreeSpaceInTailBlock = pStatement->results.pTail->size;
		pStatement->results.pLastHeadData = (int*) pStatement->results.pLastHead->data;
	}

	if (pStatement->results.unFreeSpaceInTailBlock < resulted_length + 2*sizeof(int))
	{
		/* add new block*/
		*pStatement->results.pTailField = FIELD_IN_NEXT_BLOCK;
		if (NULL == pStatement->results.pTail->next)
			pStatement->results.pTail = AllocBlock(pStatement->results.pTail, resulted_length + 2*sizeof(int));
		else
			pStatement->results.pTail = pStatement->results.pTail->next;
		pStatement->results.pTailField = (int*) pStatement->results.pTail->data;
		pStatement->results.unFreeSpaceInTailBlock = pStatement->results.pTail->size;
	}

	*pStatement->results.pTailField = length;
	pStatement->results.pTailField += 1;
	pRet = pStatement->results.pTailField_chr;
	pStatement->results.pTailField_chr += length;
	*pStatement->results.pTailField = FIELD_TERMINATOR;
	pStatement->results.unFreeSpaceInTailBlock -= length*sizeof(TCHAR) + sizeof(int);
	return pRet;
}

/*-----------------------------------------------------------------------------
 * FUNCTION: Fetch
 *
 * DESCRIPTION: No need to check pStatement's state, Driver Manager does it.
 *              
 *-----------------------------------------------------------------------------
 */
SQLRETURN
Fetch(Statement* pStatement, SQLSMALLINT FetchOrientation, SQLINTEGER FetchOffset)
{
	SQLRETURN   nRet;
	SQLSMALLINT i;

	Descriptor* pIRD;
	Descriptor* pARD;

	/* check buffering 'OFF' */
	if (!STMT_USE_BUFFERING(pStatement))
	{
		/* empty results buffer */
		EmptyStatementResults(pStatement);
		/* get portal's row */
		if (SQL_ERROR == Stmt_SendMessageToBackend(pStatement->connection, MSG_Execute, pStatement) ||
		    SQL_ERROR == Stmt_SendMessageToBackend(pStatement->connection, MSG_Sync, NULL) ||
		    SQL_ERROR == WaitForBackendReply(pStatement->connection, MSG_ReadyForQuery, pStatement)
		   )
			return SQL_ERROR;
	}
	
	pStatement->fetch_last = pStatement->ird->header.array_size;

	/* set cursor position */
	switch(FetchOrientation)
	{
		case SQL_FETCH_FIRST:
			pStatement->fetch_position = 0;
			break;
		case SQL_FETCH_LAST:
			pStatement->fetch_position = pStatement->fetch_last-1;
			break;
		case SQL_FETCH_PRIOR:
			pStatement->fetch_position = (0 < pStatement->fetch_position) ?	pStatement->fetch_position-1 : -1;
			break;
		case SQL_FETCH_NEXT:
			pStatement->fetch_position = (pStatement->fetch_position < pStatement->fetch_last-1) ? pStatement->fetch_position+1 : pStatement->fetch_last+1;
			break;
		case SQL_FETCH_RELATIVE:
		case SQL_FETCH_ABSOLUTE:
		case SQL_FETCH_BOOKMARK:
			pStatement->fetch_position = 0;
			nRet = SQL_ERROR;
			break;
	}

	if (pStatement->fetch_last < pStatement->fetch_position)
	{
		if (!STMT_USE_BUFFERING(pStatement) && AUTOCOMMIT_MODE_ON(pStatement))
			EndTransaction(SQL_HANDLE_STMT, pStatement, SQL_COMMIT, TO_DRIVER);

		return SQL_NO_DATA;
	}

	pARD = GET_DESCRIPTOR(pStatement->ard);
	pIRD = GET_DESCRIPTOR(pStatement->ird);
	/* fill bound columns */
	for (i=0;i<pIRD->header.count;i++)
	{
		if (NULL != pARD->ad_records && NULL != pARD->ad_records[i].common.data_ptr)
		{
			/* column bound */
			if ((FIELD_NULL == *(int*)pIRD->id_records[i].common.data_ptr[pStatement->fetch_position] ||
					 0          == *(int*)pIRD->id_records[i].common.data_ptr[pStatement->fetch_position]) &&
			    NULL != pARD->ad_records[i].indicator_ptr)
			{
				char* pChr = (char*)pARD->ad_records[i].common.data_ptr;
				*pARD->ad_records[i].indicator_ptr = SQL_NULL_DATA;
				*pChr = '\0';
			}
			else
			{
				SQLINTEGER ret = FillBufferWithValue(pARD->ad_records[i].common.data_ptr,
					                                   pARD->ad_records[i].common.octet_length,
														                 pARD->ad_records[i].common.concise_type,
																						 (int*)pIRD->id_records[i].common.data_ptr[pStatement->fetch_position]+1,
																						 *(int*)pIRD->id_records[i].common.data_ptr[pStatement->fetch_position],
														                 pIRD->id_records[i].common.concise_type
										                        );
				if (NULL != pARD->ad_records[i].octet_length_ptr)
					*pARD->ad_records[i].octet_length_ptr = ret;
			}
		}
		pIRD->id_records[i].bytes_left = *(int*)pIRD->id_records[i].common.data_ptr[pStatement->fetch_position];
		if (pIRD->id_records[i].bytes_left < 0)
			pIRD->id_records[i].bytes_left = 0;
	}

	RET_DESCRIPTOR(pIRD);
	RET_DESCRIPTOR(pARD);
	return SQL_SUCCESS;
}


/*-----------------------------------------------------------------------------
 * PrepareCursor
 *-----------------------------------------------------------------------------
 */
SQLRETURN
PrepareCursor(Statement* pStatement)
{
	TCHAR* pPreparedQuery;

	/* check was the cursor declared in current transaction */
	if (CS_DECLARED == pStatement->query.cursorState)
	{
		SetError(SQL_HANDLE_STMT, pStatement, ERR_STMT_CURSOR_DECLARED, pStatement->query.cursorName, NULL);
		return SQL_ERROR;
	}
	/* create unique cursor name if needed */
	SetCursorName(pStatement, NULL, 0, TRUE);
	/* compile new SELECT statement - with cursor declaration */
	pPreparedQuery = GetText(_T("DECLARE ? CURSOR FOR ?"), pStatement->query.cursorName, pStatement->query.query, NULL);
	free(pStatement->query.query);
	pStatement->query.query = pPreparedQuery;

	return SQL_SUCCESS;
}

/*-----------------------------------------------------------------------------
 * SetCursorName
 *-----------------------------------------------------------------------------
 */
SQLRETURN
SetCursorName(Statement* pStatement, SQLTCHAR* CursorName, SQLSMALLINT NameLength, BOOL bAutoGenerate)
{
	int i;

	/* create unique cursor name if there is no valid name and Auto Generate specified */
	if (TRUE == bAutoGenerate)
	{
		if (_T('\0') == pStatement->query.cursorName[0])
		{
			_tcsncpy(pStatement->query.cursorName, _T("SQL_CUR"), STR_SIZEOF("SQL_CUR"));
			_itot(++pStatement->connection->cursor_count, pStatement->query.cursorName + STR_SIZEOF("SQL_CUR"), 10 /* space economy */);
		}
		return SQL_SUCCESS;
	}

	/* check cursor's presence */
	if (CS_DECLARED == pStatement->query.cursorState)
	{
		SetError(SQL_HANDLE_STMT, pStatement, ERR_STMT_CURSOR_ALREADY_DECLARED, pStatement->query.cursorName, NULL);
		return SQL_ERROR;
	}

	if (SQL_NTS == NameLength)
		NameLength = _tcslen(CursorName);

	/* check cursor name validity */
	if (	(MAX_CURSOR_NAME_LENGTH < NameLength)
	      ||(STR_SIZEOF("SQLCUR") <= NameLength && (0 == _tcsncmp(CursorName, _T("SQLCUR"),  STR_SIZEOF("SQLCUR"))))
	      ||(0 == _tcsncmp(CursorName, _T("SQL_CUR"), STR_SIZEOF("SQL_CUR")))
	   )
	{
		SetError(SQL_HANDLE_STMT, pStatement, ERR_STMT_INVALID_CURSOR_NAME, NULL);
		return SQL_ERROR;
	}

	/* check cursor name originality within connection */
	for (i=pStatement->connection->statements.used-1;i>=0;i--)
	{
		if (0 == _tcsncmp(((Statement*)pStatement->connection->statements.handles[i])->query.cursorName, CursorName, NameLength) &&
			  NameLength == (SQLSMALLINT) _tcslen(((Statement*)pStatement->connection->statements.handles[i])->query.cursorName)
		   )
		{
			SetError(SQL_HANDLE_STMT, pStatement, ERR_STMT_CURSOR_NAME_IN_USE, NULL);
			return SQL_ERROR;
		}
	}

	/* set new name */
	_tcsncpy(pStatement->query.cursorName, CursorName, NameLength);
	pStatement->query.cursorName[NameLength] = _T('\0');
	return SQL_SUCCESS;
}

/*-----------------------------------------------------------------------------
 * FUNCTION: PrepareQuery
 *
 * DESCRIPTION: Checks the type of the query and sets statement's query type,
 *              
 *
 * RETURN: SQL_SUCCESS           - type recognized
 *         SQL_SUCCESS_WITH_INFO - empty query or unknown type, type was set to
 *                                 ST_UNKNOWN
 *-----------------------------------------------------------------------------
 */
SQLRETURN
PrepareQuery(Statement* pStatement, SQLTCHAR *StatementText, SQLINTEGER TextLength)
{
	/* in any case - this function changes the statement's query text */
	SET_STATEMENT_NEEDS_REDECLARATION(pStatement);

	/* do we have a text? */
	if (NULL != StatementText)
	{
		SQLINTEGER   i;
		SQLINTEGER   LengthTemp;
		unsigned int unParametersNumber;
		unsigned int unEscapesNumber;
		int          j, k;
		unsigned int l;
		unsigned int unI;
		unsigned int unParameterLength;
		unsigned int unCurrentLevel;
		unsigned int unParametersTemp;
		unsigned int unParameter;
		unsigned int unPreparedQueryLength;
		TCHAR        parameter[SQLSMALLINT_LENGTH];
		TCHAR*       szPreparedQuery;
		TCHAR*       pText;

		if (SQL_NTS == TextLength)
			TextLength = _tcslen(StatementText);

		/* -- remove leading spaces -- */
		while (_istspace(*StatementText))
		{
			StatementText++;
			TextLength--;
		}

		/* -- remove trailing spaces -- */
		while ((0 < TextLength) && _istspace(StatementText[--TextLength]));
		TextLength++;

		if (0 < TextLength)
		{/* -- replace escape sequencies if SQL_ATTR_NOSCAN = SQL_NOSCAN_OFF -- */
			unParametersNumber = 0;
			unEscapesNumber    = 0;

			if (SQL_STMT_NOSCAN_OFF(pStatement))
			{/* should we replace escape sequences? */
				int escape        = SQL_FALSE;
				int escapeControl = 0;
				int singleQuoter  = 0;
				int doubleQuoter  = 0;

				/* calculate the number of parameters and escape sequences, we count only unquoted escape sequences! */
				for(i=0;i<TextLength;i++)
					switch(StatementText[i])
					{
						case _T('?'): /* parameter to replace */
							if (0 == (singleQuoter || doubleQuoter))
								unParametersNumber++;
							break;
						case _T('{'):
							if (0 == (singleQuoter || doubleQuoter))
							{/* escape sequence BEGIN */
								int j = i;
								int o,begin,end;
								unEscapesNumber++;
								escapeControl++;

								/* skip spaces */
								while (j < TextLength && _istspace(StatementText[++j]));

								/* recognize one of the escape words:
								 * 'd'  - date 'yyyy-mm-dd'
								 * 't'  - time 'hh:mm:ss[,f...]'
								 * 'ts' - timestamp 'yyyy-mm-dd hh:mm:ss[,f...]'
								 * 'fn' - function
								 */
								switch(StatementText[j])
								{
									case _T('d'):	/* date escape */
										escape = SQL_TRUE;
										break;
									case _T('f'):
										if (_T('n') == StatementText[++j] && _istspace(StatementText[++j]))
										{/* function escape */
										
										}
										break;
									case _T('t'):
										if (_istspace(StatementText[++j]))
										{/* time escape */
											escape = SQL_TRUE;
										}
										else if (_T('s') == StatementText[j] && _istspace(StatementText[++j]))
										{/* timestamp escape */
											escape = SQL_TRUE;
										}
										break;
								}
								
								if (SQL_TRUE == escape)
								{
									while (j < TextLength && _istspace(StatementText[++j]));
									/* check data */
									if (_T('\'') == StatementText[j])
									{
										begin = j;
										while (j < TextLength && _T('\'') != StatementText[++j]);
										end = j;
									}
									while (j < TextLength && _istspace(StatementText[++j]));
									if (_T('}') != StatementText[j])
									{
										SetError(SQL_HANDLE_STMT, pStatement, ERR_BAD_QUERY_SYNTAX, NULL);
										return SQL_ERROR;										
									}
									for(o=i;o<begin;o++)
										StatementText[o]=_T(' ');
									for(o=j;o>end;o--)
										StatementText[o]=_T(' ');
									i = j;
									escape = SQL_FALSE;
								}
							}
							break;
						case _T('}'):
							if (0 == (singleQuoter || doubleQuoter) && (0 > --escapeControl))
							{ /* escape sequence END */
								SetError(SQL_HANDLE_STMT, pStatement, ERR_BAD_QUERY_SYNTAX, NULL);
								return SQL_ERROR;
							}
							break;
						case _T('\''):
							singleQuoter ^= QUOTER_FOUND;
							break;
						case _T('\"'):
							doubleQuoter ^= QUOTER_FOUND;
							break;
					}
			}

			if (0 == unParametersNumber)
			{/* no escape sequences to replace - quick copy */
				j = TextLength;
				szPreparedQuery = (TCHAR*) malloc((TextLength+1)*sizeof(TCHAR));
				_tcsncpy(szPreparedQuery, StatementText, TextLength);
			}

			if (0 < unEscapesNumber)
			{/* replace escape sequences */
			
			}

			if (0 < unParametersNumber)
			{
				/* calculate the size of the prepared query text and allocate memory, it depends on
				 * the number of parameters, every '?' must be replaced with it's number - for
				 * example ' $19 '
				 */
				unPreparedQueryLength = TextLength - unParametersNumber + 3;
				unParametersTemp = unParametersNumber;
				unCurrentLevel = 9;
				unI = 4;
				while (unParametersTemp > unCurrentLevel)
				{
					unParametersTemp -= unCurrentLevel;
					unCurrentLevel *= 10;
					unPreparedQueryLength += unCurrentLevel*(unI++);
				}
				unPreparedQueryLength += unParametersTemp*unI;
				szPreparedQuery = (TCHAR*) malloc(unPreparedQueryLength*sizeof(TCHAR));
				/* prepare query */
				for(i=0;i<SQLSMALLINT_LENGTH;i++)
					parameter[i]=_T('0');
				parameter[0]++;
				unParameterLength = 0;
				for(i=0,j=0,unParameter=1;i<TextLength;i++,j++)
				{
					if(_T('?') == StatementText[i])
					{
						szPreparedQuery[j++] = _T(' ');
						szPreparedQuery[j++] = _T('$');
						for (k=unParameterLength;k>=0;k--)
							szPreparedQuery[j++] = parameter[k];
						/* increment parameter's number */
						for (l=0;l<=unParameterLength;l++)
						{
							if (_T(':') != ++parameter[l])
								break;
							else
							{
								parameter[l] = _T('0');
								if (l == unParameterLength)
									unParameterLength++;
							}
						}
						szPreparedQuery[j] = _T(' ');
					}
					else
						szPreparedQuery[j] = StatementText[i];
				}
			}
			szPreparedQuery[j] = _T('\0');

			if (NULL != pStatement->query.query)
				free(pStatement->query.query);
			pStatement->query.query = szPreparedQuery;
			pStatement->query.nParametersNumber = unParametersNumber;
	
			/* -- parse StatementText and set corresponding statement's type -- */
			/* uppercase first word in the query */
			pText = szPreparedQuery;
			LengthTemp = TextLength + 1;
			while(LengthTemp)
			{
				if (_T('a') <= *pText && _T('z') >= *pText)
					*pText += 'A'-'a';
				else
					if (_T('A') > *pText || _T('Z') < *pText)
						break;
				pText++;
				LengthTemp--;
			}

			switch(szPreparedQuery[0])
			{
				case _T('B'): /* BEGIN */
					pStatement->query.type = ST_BEGIN;
					break;
				case _T('C'):	/* COMMIT */
					pStatement->query.type = ST_COMMIT;
					break;
				case _T('D'): /* DELETE */
					pStatement->query.type = ST_DELETE;
					break;
				case _T('F'): /* FETCH */
					pStatement->query.type = ST_FETCH;
					break;
				case _T('I'):	/* INSERT */
					pStatement->query.type = ST_INSERT;
					break;
				case _T('M'): /* MOVE */
					pStatement->query.type = ST_MOVE;
					break;
				case _T('S'): /* SELECT */
					pStatement->query.type = ST_SELECT;
					break;
				case _T('U'): /* UPDATE */
					pStatement->query.type = ST_UPDATE;
					break;
				default:
					pStatement->query.type = ST_UNKNOWN;
			}

			return SQL_SUCCESS;
		}
	}

	/* empty query text to prepare */
	if(NULL != pStatement->query.query)
	{
		free(pStatement->query.query);
		pStatement->query.query = NULL;
		pStatement->query.nParametersNumber = 0;
	}

	pStatement->query.type = ST_EMPTY;
	SetError(SQL_HANDLE_STMT, pStatement, ERR_STMT_EMPTY_QUERY, NULL);
	return SQL_SUCCESS_WITH_INFO;
}

/*-----------------------------------------------------------------------------
 * FUNCTION: SetApplicationDescriptor
 *-----------------------------------------------------------------------------
 */
SQLRETURN
SetApplicationDescriptor(Statement* pStatement, Descriptor** ppDescInUse, Descriptor* pDescImplicit, Descriptor* pDescToSet)
{
	/* set back implicitly allocated descriptor? */
	if (SQL_NULL_HDESC == pDescToSet)
	{
		*ppDescInUse = pDescImplicit;
		return SQL_SUCCESS;
	}

	/* cannot set implicitly allocated descriptor */
	if (SQL_DESC_ALLOC_AUTO == pDescToSet->header.alloc_type)
	{
		SetError(SQL_HANDLE_STMT, pStatement, ERR_STMT_DESC_WRONG_ALLOC_TYPE, NULL);
		return SQL_ERROR;
	}

	/* descriptor and statement must be allocated on the same connection */
	if (pStatement->connection != pDescToSet->connection)
	{
		SetError(SQL_HANDLE_STMT, pStatement, ERR_STMT_DESC_WRONG_CONNECTION, NULL);
		return SQL_ERROR;
	}

	/* ok, set descriptor */
	*ppDescInUse = pDescToSet;
	return SQL_SUCCESS;
}

/*-----------------------------------------------------------------------------
 * FUNCTION: GetData
 *-----------------------------------------------------------------------------
 */
SQLRETURN
GetData(Statement* pStatement, SQLUSMALLINT ColumnNumber, SQLSMALLINT TargetType, SQLPOINTER TargetValue, SQLINTEGER BufferLength, SQLINTEGER *StrLen_or_Ind)
{
	SQLRETURN nRet;

	/* is this a valid column number? */
	if (ColumnNumber <= pStatement->ird->header.count)
	{
		if (0 == ColumnNumber)
		{
			/* bookmark column */
			if (SQL_UB_OFF != pStatement->attributes.use_bookmarks)
			{
				/* andyk get data from bookmark column */
			}
			else
			{
				/* no bookmarks used */
				SetError(SQL_HANDLE_STMT, pStatement, ERR_STMT_BOOKMARKS_UNUSED, NULL);
				nRet = SQL_ERROR;
			}
		}
		else
		{
			/* column exists in result set */
			ColumnNumber--;
			if (0 < pStatement->ird->id_records[ColumnNumber].bytes_left)
			{
				int ret;
				if (0 < (ret = FillBufferWithValue(TargetValue, BufferLength, TargetType, (unsigned int*)pStatement->ird->id_records[ColumnNumber].common.data_ptr[pStatement->fetch_position]+1, pStatement->ird->id_records[ColumnNumber].bytes_left, pStatement->ird->id_records[ColumnNumber].common.concise_type)))
					pStatement->ird->id_records[ColumnNumber].bytes_left -= ret;
				*StrLen_or_Ind = ret;
				nRet = SQL_SUCCESS;
			}
			else
			{
				if (0 == pStatement->ird->id_records[ColumnNumber].bytes_left)
				{
					/* null */
					*StrLen_or_Ind = SQL_NULL_DATA;
					nRet = SQL_SUCCESS;
				}
				else
				{
					/* all avaible data already fetched */
					nRet = SQL_NO_DATA;
				}
			}
		}
	}
	else
	{
		/* column with such number doesn't exist in result set */
		SetError(SQL_HANDLE_STMT, pStatement, ERR_STMT_COLUMN_NUMBER_TOO_BIG, NULL);
		nRet = SQL_ERROR;
	}

	return nRet;
}

/*-----------------------------------------------------------------------------
 * FUNCTION: ColAttribute
 *-----------------------------------------------------------------------------
 */
SQLRETURN
ColAttribute(Statement* pStatement, SQLUSMALLINT ColumnNumber, SQLUSMALLINT FieldIdentifier,
             SQLPOINTER CharacterAttribute, SQLSMALLINT BufferLength, SQLSMALLINT* StringLength,
             SQLPOINTER NumericAttribute)
{
	SQLRETURN   nRet;
	Descriptor* pIRD;

	SQLINTEGER  uInteger;
	SQLSMALLINT ret_type = 0;

	pIRD = GET_DESCRIPTOR(pStatement->ird);
	if (0 == ColumnNumber)
	{
		/* bookmark column */
		/* Driver Manager won't get us here if SQL_ATTR_USE_BOOKMARKS = SQL_UB_OFF */
		switch(FieldIdentifier)
		{
			case SQL_DESC_TYPE:
				uInteger = pIRD->bookmark.type; /* SQL_BINARY or SQL_INTEGER */
				ret_type = SQL_IS_INTEGER;
				break;
			case SQL_DESC_OCTET_LENGTH:
				uInteger = pIRD->bookmark.octet_length;
				ret_type = SQL_IS_INTEGER;
				break;
			/* all other fields here can return underfined values */
		}
		nRet = SQL_SUCCESS;
	}
	else
	{
		BOOL         translate = FALSE;
		SQLPOINTER   value;
		SQLSMALLINT* length;
		SQLSMALLINT  smallint;
		SQLSMALLINT  resLength;
		SQLINTEGER   integer;

		switch(FieldIdentifier)
		{
			/* numeric values */
			case SQL_DESC_AUTO_UNIQUE_VALUE: /* 11 */
			case SQL_DESC_CASE_SENSITIVE:    /* 12 */
			case SQL_DESC_DISPLAY_SIZE:       /* 6 */
			case SQL_DESC_LENGTH:          /* 1003 */
			case SQL_DESC_OCTET_LENGTH:    /* 1013 */
			case SQL_DESC_NUM_PREC_RADIX:  /*   32 */
				value  = (NULL == NumericAttribute) ? &integer : NumericAttribute;
				length = &resLength;
				break;
			case SQL_DESC_CONCISE_TYPE:       /* 2 */
			case SQL_DESC_COUNT:           /* 1001 */
			case SQL_DESC_FIXED_PREC_SCALE:   /* 9 */
			case SQL_DESC_NULLABLE:        /* 1008 */
			case SQL_DESC_PRECISION:       /* 1005 */
			case SQL_DESC_SCALE:           /* 1006 */
			case SQL_DESC_SEARCHABLE:        /* 13 */
			case SQL_DESC_TYPE:            /* 1002 */
			case SQL_DESC_UNNAMED:         /* 1012 */
			case SQL_DESC_UNSIGNED:           /* 8 */
			case SQL_DESC_UPDATABLE:         /* 10 */
				value  = &smallint;
				length = &resLength;
				translate = TRUE;
				break;
			/* string values */
			case SQL_DESC_BASE_COLUMN_NAME:  /* 22 */
			case SQL_DESC_BASE_TABLE_NAME:   /* 23 */
			case SQL_DESC_CATALOG_NAME:      /* 17 */
			case SQL_DESC_LABEL:             /* 18 */
			case SQL_DESC_LITERAL_PREFIX:    /* 27 */
			case SQL_DESC_LITERAL_SUFFIX:    /* 28 */
			case SQL_DESC_LOCAL_TYPE_NAME:   /* 29 */
			case SQL_DESC_NAME:            /* 1011 */
			case SQL_DESC_SCHEMA_NAME:       /* 16 */
			case SQL_DESC_TABLE_NAME:        /* 15 */
			case SQL_DESC_TYPE_NAME:         /* 14 */
				value  = CharacterAttribute;
				length = StringLength;
				break;
		}
		nRet = GetDescField(pIRD, ColumnNumber, FieldIdentifier, value, BufferLength, length, pStatement);
		if (translate)
		{
			integer = smallint;
			memcpy(NumericAttribute, &integer, sizeof(SQLINTEGER));
		}
	}

	if (SQL_IS_INTEGER == ret_type)
		memcpy(NumericAttribute, &uInteger, sizeof(uInteger));

	RET_DESCRIPTOR(pIRD);
	return nRet;
}

/*-----------------------------------------------------------------------------
 * FUNCTION: DescribeCol
 *-----------------------------------------------------------------------------
 */
SQLRETURN
DescribeCol(Statement* pStatement, SQLUSMALLINT ColumnNumber,  SQLTCHAR* ColumnName,
            SQLSMALLINT BufferLength, SQLSMALLINT* NameLength, SQLSMALLINT* DataType,
            SQLUINTEGER* ColumnSize, SQLSMALLINT* DecimalDigits, SQLSMALLINT* Nullable)
{
	SQLRETURN   nRet;
	Descriptor* pIRD;

	pIRD = GET_DESCRIPTOR(pStatement->ird);
	if (ColumnNumber <= pIRD->header.count)
	{
		if (0 < ColumnNumber)
		{
			ID_REC*     pID_REC = &pIRD->id_records[ColumnNumber-1];
			SQLINTEGER  length       = (0 >= pID_REC->max_column_length) ? 213 : pID_REC->max_column_length; /* character or binary types */
			SQLINTEGER  display_size = 0;                          /* date-time types */
			SQLSMALLINT precision    = 0;                          /* numeric types */
			SQLSMALLINT scale        = 0;
			SQLSMALLINT type         = 0;
			
			SQLTypeDescriptor(pID_REC->common.concise_type, 0, &length, &precision, &scale, &type, &display_size, NULL, NULL);

			/* SQL data type */
			if (NULL != DataType)
				*DataType = pID_REC->common.concise_type;

			/* column display size */
			if (NULL != ColumnSize)
				*ColumnSize = (length < (SQLSMALLINT)precision) ? precision : (length < (SQLSMALLINT)display_size) ? display_size : length;

			/* now driver can't determine nullability */
			if (NULL != Nullable)
				*Nullable = pID_REC->nullable;

			/* Decimal digits - this value based on data, retrived from server */
			if (NULL != DecimalDigits)
				*DecimalDigits = (SQL_DATETIME == type || SQL_INTERVAL == type) ? precision : scale;

			/* column name and it's length, we don't call GetDescField - because it sets pDescritor's error */
                        if (NULL != ColumnName) {
	    nRet = ReturnString(ColumnName, BufferLength, NameLength, pID_REC->name, SQL_NTS);
            } else {nRet = SQL_SUCCESS; }
		}
		else
		{
			/* bookmark column */
			if (NULL != ColumnSize)
				*ColumnSize = pIRD->bookmark.octet_length;

			if (NULL != Nullable)
				*Nullable = SQL_NO_NULLS; /* bookmark column can't be NULL */

			if (NULL != DecimalDigits)
				*DecimalDigits = 0;       /* No decimal digits in SQL_BINARY */

			if (NULL != DataType)
				*DataType = pIRD->bookmark.type; /* SQL_BINARY or SQL_INTEGER */

	    nRet = ReturnString(ColumnName, BufferLength, NameLength, c_szBookmarkName, SQL_NTS);
		}

		if (SQL_SUCCESS != nRet)
			SetError(SQL_HANDLE_STMT, pStatement, ERR_TOO_SMALL_BUFFER, _T("ColumnName"), NULL);
	}
	else
	{
		/* column with such number doesn't exist in result set */
		SetError(SQL_HANDLE_STMT, pStatement, ERR_STMT_COLUMN_NUMBER_TOO_BIG, NULL);
		nRet = SQL_ERROR;
	}

	RET_DESCRIPTOR(pIRD);
	return nRet;
}

/*-----------------------------------------------------------------------------
 * FUNCTION: BindCol
 *-----------------------------------------------------------------------------
 */
SQLRETURN 
BindCol(Statement* pStatement, SQLUSMALLINT ColumnNumber, SQLSMALLINT TargetType, SQLPOINTER TargetValue, SQLINTEGER BufferLength, SQLINTEGER* StrLen_or_Ind)
{
	SQLRETURN nRet;

	if (0 < ColumnNumber)
	{
		SQLSMALLINT savedARDCount;
		
		Descriptor* pARD = GET_DESCRIPTOR(pStatement->ard);
		savedARDCount = pARD->header.count;
		if (NULL == TargetValue)
		{
			/* -- unbind column -- */
			/* if this is the last record - just decrement 'count' field, else - empty it */
			nRet = (ColumnNumber == pARD->header.count) ? SetDescField(pARD, 0, SQL_DESC_COUNT, (SQLPOINTER)(SQLSMALLINT)(ColumnNumber-1), SQL_IS_SMALLINT) : EmptyDescriptorRecord(pARD, (SQLSMALLINT)(ColumnNumber-1));
		}
		else
		{
			/* bind or rebind column */
			if (SQL_ERROR == ReallocDescriptorRecords(pARD, ColumnNumber))
			{
				SetError(SQL_HANDLE_STMT, pStatement, ERR_NOT_ENOUGH_MEMORY, NULL);
				nRet = SQL_ERROR;
			}
			else
			{
				AD_REC* pAD_REC	= &pARD->ad_records[--ColumnNumber];

				/* fills ARD's fields with data */
				pAD_REC->bound               = SQL_TRUE;
				pAD_REC->common.octet_length = BufferLength;
				pAD_REC->common.data_ptr     = TargetValue;
				pAD_REC->indicator_ptr       = StrLen_or_Ind;
				pAD_REC->octet_length_ptr    = StrLen_or_Ind;

				TranslateType(&pAD_REC->common, TargetType, 0, BufferLength, C_TYPE);

				nRet = SQL_SUCCESS;
			}
		}
		/* if an error occured - restore saved COUNT field */
		if (SQL_ERROR == nRet)
			pARD->header.count = savedARDCount;
		RET_DESCRIPTOR(pARD);
	}
	else
	{
		/* bind bookmark's column */
		if (SQL_UB_OFF != (pStatement->attributes.use_bookmarks))
		{
			Bookmark*   pBookmark;
			Descriptor* pARD = GET_DESCRIPTOR(pStatement->ard);

			pBookmark = &pARD->bookmark;
			if (NULL == TargetValue)
			{
				/* -- unbind bookmark column -- */
				pBookmark->data_ptr      = NULL;
				pBookmark->indicator_ptr = NULL;
			}
			else
			{
				pBookmark->data_ptr      = TargetValue;
				pBookmark->indicator_ptr = StrLen_or_Ind;
				pBookmark->octet_length  = BufferLength;
				pBookmark->type          = TargetType;
			}
			RET_DESCRIPTOR(pARD);
		}
		else
		{
			/* no bookmarks used */
			SetError(SQL_HANDLE_STMT, pStatement, ERR_STMT_BOOKMARKS_UNUSED, NULL);
			nRet = SQL_ERROR;
		}
	}
	return nRet;
}

/*-----------------------------------------------------------------------------
 * FUNCTION: GetStmtAttr
 *-----------------------------------------------------------------------------
 */
SQLRETURN
GetStmtAttr(Statement* pStatement, SQLINTEGER Attribute, SQLPOINTER Value)
{
	SQLINTEGER  uInteger;
	SQLPOINTER  pointer;
	SQLHDESC    hDesc;
	SQLSMALLINT ret_type;

	if (NULL == Value)
	{
		SetError(SQL_HANDLE_STMT, pStatement, ERR_NULL_POINTER_BUFFER, NULL);
		return SQL_ERROR;
	}
	else
	{
		SQLRETURN nRet = SQL_SUCCESS;
		switch(Attribute)
		{
			case SQL_ATTR_NOSCAN:               /*     2 */
				uInteger = pStatement->attributes.no_scan;
				ret_type = SQL_IS_INTEGER;
				break;
			case SQL_ATTR_CURSOR_TYPE:          /*     6 */
				uInteger = SQL_CURSOR_FORWARD_ONLY;
				ret_type = SQL_IS_INTEGER;
				break;
			case SQL_ATTR_USE_BOOKMARKS:        /*    12 */
				uInteger = pStatement->attributes.use_bookmarks;
				ret_type = SQL_IS_INTEGER;
				break;
			case SQL_ATTR_ENABLE_AUTO_IPD:         /* 15 */
				uInteger = pStatement->attributes.enable_auto_ipd;
				ret_type = SQL_IS_INTEGER;
				break;
			case SQL_ATTR_PARAMS_PROCESSED_PTR: /*    21 */
				pointer  = pStatement->ipd->header.rows_processed_ptr;
				ret_type = SQL_IS_POINTER;
				break;
			case SQL_ATTR_ROWS_FETCHED_PTR:     /*    26 */
				pointer = pStatement->ird->header.rows_processed_ptr;
				ret_type = SQL_IS_POINTER;
				break;
			case SQL_ATTR_ROW_ARRAY_SIZE:       /*    27 */
			/* ODBC 2.0 */
			case SQL_ROWSET_SIZE:               /*     9 */
				uInteger = pStatement->ard->header.array_size;
				ret_type = SQL_IS_INTEGER;
				break;
			case SQL_ATTR_APP_PARAM_DESC:       /* 10011 */
				hDesc    = pStatement->apd;
				ret_type = SQL_IS_SMALLINT;
				break;
			case SQL_ATTR_APP_ROW_DESC:         /* 10010 */
				hDesc    = pStatement->ard;
				ret_type = SQL_IS_SMALLINT;
				break;
			case SQL_ATTR_IMP_PARAM_DESC:       /* 10013 */
				hDesc    = pStatement->ipd;
				ret_type = SQL_IS_SMALLINT;
				break;
			case SQL_ATTR_IMP_ROW_DESC:         /* 10012 */
				hDesc    = pStatement->ird;
				ret_type = SQL_IS_SMALLINT;
				break;
			case SQL_ATTR_ROW_STATUS_PTR:
				return GetDescField(pStatement->ird, 0, SQL_DESC_ARRAY_STATUS_PTR, Value, SQL_IS_USMALLINT, NULL, pStatement);

			case SQL_ATTR_ASYNC_ENABLE:
			case SQL_ATTR_CONCURRENCY:
			case SQL_ATTR_CURSOR_SCROLLABLE:
			case SQL_ATTR_CURSOR_SENSITIVITY:
			case SQL_ATTR_FETCH_BOOKMARK_PTR:
			case SQL_ATTR_KEYSET_SIZE:
			case SQL_ATTR_MAX_LENGTH:
			case SQL_ATTR_MAX_ROWS:
			case SQL_ATTR_METADATA_ID:
			case SQL_ATTR_PARAM_BIND_OFFSET_PTR:
			case SQL_ATTR_PARAM_BIND_TYPE:
			case SQL_ATTR_PARAM_OPERATION_PTR:
			case SQL_ATTR_PARAM_STATUS_PTR:
			case SQL_ATTR_PARAMSET_SIZE:
			case SQL_ATTR_QUERY_TIMEOUT:
			case SQL_ATTR_RETRIEVE_DATA:
			case SQL_ATTR_ROW_BIND_OFFSET_PTR:
			case SQL_ATTR_ROW_BIND_TYPE:
			case SQL_ATTR_ROW_NUMBER:
			case SQL_ATTR_ROW_OPERATION_PTR:
			case SQL_ATTR_SIMULATE_CURSOR:
			default:
				SetError(SQL_HANDLE_STMT, pStatement, ERR_UNKNOWN_ATTRIBUTE, NULL);
				return SQL_ERROR;
		}

		switch (ret_type)
		{
			case SQL_IS_SMALLINT: /* HDESC */
				memcpy(Value, &hDesc, sizeof(hDesc));
				break;
			case SQL_IS_INTEGER:
				memcpy(Value, &uInteger, sizeof(uInteger));
				break;
			case SQL_IS_POINTER:
				memcpy(Value, &pointer, sizeof(pointer));
				break;
		}
		return nRet;
	}
}

