/*--------------------------------------------------------------------
 * FILE:
 *     probe_set_cluster_status.c
 *
 * NOTE:
 *     This file is composed of the probe process 
 *     Low level I/O functions that called by in these functions are 
 *     contained in 'replicate_com.c'.
 *
 *--------------------------------------------------------------------
 */
#include "postgres.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <sys/wait.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/shm.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/file.h>

#ifdef MULTIBYTE
#include "mb/pg_wchar.h"
#endif

#include "pgc_admin.h"

int PGC_Set_Cluster_Status(Probe_Header * header, char * packet);
Cluster_Info * PGC_Get_ClusterDbTbl_Rec(Cluster_Info * key);

static int set_cluster_status_in_pgrp(Cluster_Info * cluster);
static int sync_cluster_status_in_pgrp(HostTbl * host, Cluster_Info * cluster);
static int set_cluster_status_in_pglb(Cluster_Info * cluster);
static int sync_cluster_status_in_pglb(ClusterTbl * host, Cluster_Info * cluster);


int
PGC_Set_Cluster_Status(Probe_Header * header, char * packet)
{
	char * func = "PGC_Set_Cluster_Status()";
	Cluster_Info src;
	int status;

	memset(&src, 0, sizeof(Cluster_Info));
	if (PGC_Set_Packet_2_Cluster_Info(&src, (Cluster_Info *)packet) == NULL)
	{
		show_error("%s:PGC_Set_Packet_2_Cluster_Info failed",func);
		return STATUS_ERROR;
	}
	status = set_cluster_status_in_pgrp(&src);
	status = set_cluster_status_in_pglb(&src);

	return status;
}

Cluster_Info *
PGC_Get_ClusterDbTbl_Rec(Cluster_Info * key)
{
	char * func ="PGC_Get_ClusterDbTbl_Rec()";
	Cluster_Info * replicate = NULL;

	if (ClusterDbTbl == NULL)
	{
		show_error("%s: ClusterDbTbl is NULL",func);
		return NULL;
	}
	replicate = ClusterDbTbl;
	while (replicate->portNumber != 0)
	{
		if (replicate->serverNo == ntohs(key->serverNo))
		{
			return replicate;
		}
		replicate ++;
	}
	return NULL;
}

static int
set_cluster_status_in_pgrp(Cluster_Info * cluster)
{
	char * func ="set_cluster_status_in_pgrp()";
	Pgrp_Info * pgrp = NULL;
	HostTbl * dest = NULL;

	if (PgrpTbl == NULL)
	{
		/* there is no replication server in this physical server */
		return STATUS_OK;
	}
	pgrp = PgrpTbl;
	while (pgrp->replicationPortNumber != 0)
	{
		dest = PGC_Get_Pgrp_Shm(pgrp->workPath);
		if (dest != NULL)
		{
			sync_cluster_status_in_pgrp(dest, cluster);
			shmdt(dest);
			dest = NULL;
		}
		else
		{
			show_error("%s: PGC_Get_Pgrp_Shm failed",func);
			pgrp->status = DATA_ERR;
			PGC_Send_Pgrp_Status(pgrp, NULL );
		}
		pgrp ++;
	}
	return STATUS_OK;
}

static int
sync_cluster_status_in_pgrp(HostTbl * host, Cluster_Info * cluster)
{
	int status = STATUS_ERROR;

	while(host->port != 0)
	{
		if ((!strncmp(host->hostName,cluster->hostName,sizeof(host->hostName))) &&
			(host->port == cluster->portNumber))
		{
			host->useFlag = cluster->status;
			status = STATUS_OK;
			break;
		}
		host ++;
	}
	return status;
}

static int
set_cluster_status_in_pglb(Cluster_Info * cluster)
{
	/* char * func="set_cluster_status_in_pglb()";*/
	Pglb_Info * pglb = NULL;
	ClusterTbl * dest;

	if (PglbTbl == NULL)
	{
		/* there is no replication server in this physical server */
		return STATUS_OK;
	}
	pglb = PglbTbl;
	while (pglb->receivePortNumber != 0)
	{
		dest = PGC_Get_Pglb_Shm(pglb->workPath);
		if (dest != NULL)
		{
			sync_cluster_status_in_pglb(dest, cluster);
			shmdt(dest);
			dest = NULL;
		}
		else
		{
			pglb->status = DATA_ERR;
			PGC_Send_Pglb_Status(pglb, NULL);
		}
		pglb ++;
	}
	return STATUS_OK;
}

static int
sync_cluster_status_in_pglb(ClusterTbl * host, Cluster_Info * cluster)
{
	int status = STATUS_ERROR;

	while(host->port != 0)
	{
		if ((!strncmp(host->hostName,cluster->hostName,sizeof(host->hostName))) &&
			(host->port == cluster->portNumber))
		{
			host->useFlag = cluster->status;
			status = STATUS_OK;
			break;
		}
		host ++;
	}
	return status;
}
	
