/*
 * dbnames.c: Database connection management
 *
 * Copyright (c) 2009-2010, NIPPON TELEGRAPH AND TELEPHONE CORPORATION
 */
 
#include "pg_reporter.h"

static Database *getNextDatabase(FILE *fp, int *lineno, const char *id);
static int tokenize(char *p, char *tokens[], size_t length[], int max_tokens);

/* retrieve all database information */
List *
getDatabases(void)
{
	FILE	   *fp;
	int			lineno;
	Database   *db;
	List	   *dbs = NIL;

	if ((fp = fopen(DBNAMES, "rt")) ==  NULL)
	{
		elog(WARNING, "could not open file: \"%s\": %s",
			DBNAMES, strerror(errno));
		return NIL;
	}

	lineno = 1;
	while ((db = getNextDatabase(fp, &lineno, NULL)) != NULL)
		dbs = lappend(dbs, db);

	fclose(fp);
	return dbs;
}

/* retrieve one database information */
Database *
getDatabase(const char *id)
{
	FILE	   *fp;
	int			lineno;
	Database   *db;

	if ((fp = fopen(DBNAMES, "rt")) ==  NULL)
	{
		elog(WARNING, "could not open file: \"%s\": %s",
			DBNAMES, strerror(errno));
		return NULL;
	}

	lineno = 1;
	db = getNextDatabase(fp, &lineno, id);
	fclose(fp);
	return db;
}

/* free one database information */
void
freeDatabase(Database *db)
{
	if (db == NULL)
		return;

	free(db->id);
	free(db->host);
	free(db->port);
	free(db->database);
	free(db->username);
	free(db->password);
	free(db);
}

/*
 * Returns the next database.
 * If id is not NULL, find the database that is named as 'id'.
 */
static Database *
getNextDatabase(FILE *fp, int *lineno, const char *id)
{
#define LINEBUFSZ		1000
	char		line[LINEBUFSZ];

	for (; fgets(line, LINEBUFSZ, fp) != NULL; (*lineno)++)
	{
		size_t		len;
		char	   *tokens[6];
		size_t		length[6];
		int			ntoken;
		Database   *db;

		len = strlen(line);
		if (line[len - 1] != '\n')
		{
			elog(WARNING, "too long line in file \"%s\": line %d",
				DBNAMES, *lineno);
			break;
		}

		while (len > 0 && IsSpace(line[len - 1]))
		{
			line[len - 1] = '\0';
			len--;
		}

		ntoken = tokenize(line, tokens, length, 6);

		if (ntoken == 0)
			continue;	/* empty line */
		else if (ntoken < 4)
		{
			elog(WARNING, "syntax error in file \"%s\": line %d",
				DBNAMES, *lineno);
			continue;
		}

		if (id != NULL &&
			(strncmp(id, tokens[0], length[0]) != 0 || id[length[0]] != '\0'))
			continue;

		db = pgut_new(Database);
		db->id = strdup_with_len(tokens[0], length[0]);
		db->host = strdup_with_len(tokens[1], length[1]);
		db->port = strdup_with_len(tokens[2], length[2]);
		db->database = strdup_with_len(tokens[3], length[3]);
		db->username = strdup_with_len(tokens[4], length[4]);	/* nulls ok */
		db->password = strdup_with_len(tokens[5], length[5]);	/* nulls ok */

		return db;
	}

	return NULL;
}

/*
 * Split text into tokens with white-spaces.
 * Ignore after comment charaster '#'.
 */
static int
tokenize(char *p, char *tokens[], size_t length[], int max_tokens)
{
	int		n;
	char   *end = p + strlen(p);

	for (n = 0; n < max_tokens; n++)
	{
		/* skip white-spaces */
		for (;;)
		{
			if (p >= end || *p == '#')
				return n;
			if (!IsSpace(*p))
				break;
			p++;
		}

		/* get a token */
		tokens[n] = p;
		for (;;)
		{
			if (p >= end || *p == '#')
			{
				length[n] = p - tokens[n];
				++n;
				goto done;
			}
			if (IsSpace(*p))
				break;
			p++;
		}
		length[n] = p - tokens[n];
	}

done:
	/* fill with zeros remaining fields */
	memset(&tokens[n], 0, sizeof(*tokens) * (max_tokens - n));
	memset(&length[n], 0, sizeof(*length) * (max_tokens - n));

	return n;
}
