/*-------------------------------------------------------------------------
 *
 * pl_exec.c		- Executor for the PL/pgSQL
 *			  procedural language
 *
 * Portions Copyright (c) 1996-2006, PostgreSQL Global Development Group
 * Portions Copyright (c) 1994, Regents of the University of California
 *
 *
 * IDENTIFICATION
 *	  $PostgreSQL: pgsql/src/pl/plpgpsm/src/pl_exec.c,v 1.180 2006/10/04 00:30:13 momjian Exp $
 *
 *-------------------------------------------------------------------------
 */

#include "plpgpsm.h"

#include <ctype.h>

#include "access/transam.h"
#include "catalog/pg_proc.h"
#include "catalog/pg_type.h"
#include "executor/spi_priv.h"
#include "funcapi.h"
#include "nodes/nodeFuncs.h"
#include "parser/scansup.h"
#include "tcop/tcopprot.h"
#include "utils/array.h"
#include "utils/builtins.h"
#include "utils/lsyscache.h"
#include "utils/memutils.h"
#include "utils/snapmgr.h"
#include "utils/typcache.h"

static const char *const raise_skip_msg = "RAISE";

static void exec_signal_not_found(bool state);

static void set_sqlstate(PLpgPSM_execstate *estate);


static bool compare_datum(Datum a, Oid atype, Datum b, Oid btype);


static PLpgPSM_prepared_plan *get_plan(const char *name);
static HTAB *create_plan_hash(void);
static void store_plan(const char *name, PLpgPSM_prepared_plan *plan);

static void forward_set(PLpgPSM_execstate *estate, Portal portal);


static HTAB *planHash = NULL;

typedef struct planHashEnt
{
	char	name[NAMEDATALEN];
	PLpgPSM_prepared_plan *preplan;
} planHashEnt;

#define NUMSTR 16


/*                                                                                                     
 * All plpgpsm function executions within a single transaction share the same                          
 * executor EState for evaluating "simple" expressions.  Each function call                            
 * creates its own "eval_econtext" ExprContext within this estate for                                  
 * per-evaluation workspace.  eval_econtext is freed at normal function exit,                          
 * and the EState is freed at transaction end (in case of error, we assume                             
 * that the abort mechanisms clean it all up).  In order to be sure                                    
 * ExprContext callbacks are handled properly, each subtransaction has to have                         
 * its own such EState; hence we need a stack.  We use a simple counter to                             
 * distinguish different instantiations of the EState, so that we can tell                             
 * whether we have a current copy of a prepared expression.                                            
 *                                                                                                     
 * This arrangement is a bit tedious to maintain, but it's worth the trouble                           
 * so that we don't have to re-prepare simple expressions on each trip through                         
 * a function.  (We assume the case to optimize is many repetitions of a                               
 * function within a transaction.)                                                                     
 */                                                                                                    
typedef struct SimpleEstateStackEntry                                                                  
{                                                                                                      
        EState *xact_eval_estate;                               /* EState for current xact level */    
        long int        xact_estate_simple_id;          /* ID for xact_eval_estate */                  
        SubTransactionId xact_subxid;                   /* ID for current subxact */                   
        struct SimpleEstateStackEntry *next;    /* next stack entry up */                              
} SimpleEstateStackEntry;                                                                              
                                                                                                       
static SimpleEstateStackEntry *simple_estate_stack = NULL;                                             
static long int simple_estate_id_counter = 0;

/************************************************************
 * Local function forward declarations
 ************************************************************/
static void plpgpsm_exec_error_callback(void *arg);
static PLpgPSM_datum *copy_plpgpsm_datum(PLpgPSM_datum *datum);

static int exec_stmt_block(PLpgPSM_execstate *estate,
				PLpgPSM_stmt_block *block);
static int exec_stmts(PLpgPSM_execstate *estate,
		   List *stmts, PLpgPSM_stmt_block *block);
static int exec_stmt(PLpgPSM_execstate *estate,
		  PLpgPSM_stmt *stmt);
static int exec_stmt_assign(PLpgPSM_execstate *estate,
				 PLpgPSM_stmt_assign *stmt);
static int exec_stmt_call(PLpgPSM_execstate *estate,
				  PLpgPSM_stmt_call *stmt);
static int exec_stmt_getdiag(PLpgPSM_execstate *estate,
				  PLpgPSM_stmt_getdiag *stmt);
static int exec_stmt_if(PLpgPSM_execstate *estate,
			 PLpgPSM_stmt_if *stmt);
static int exec_stmt_loop(PLpgPSM_execstate *estate,
			   PLpgPSM_stmt_loop *stmt);
static int exec_stmt_fors(PLpgPSM_execstate *estate,
			   PLpgPSM_stmt_fors *stmt);
static int exec_stmt_open(PLpgPSM_execstate *estate,
			   PLpgPSM_stmt_open *stmt);
static int exec_stmt_fetch(PLpgPSM_execstate *estate,
				PLpgPSM_stmt_fetch *stmt);

static int exec_stmt_prepare(PLpgPSM_execstate *estate,
				PLpgPSM_stmt_prepare *stmt);
				
				
static int exec_stmt_close(PLpgPSM_execstate *estate,
				PLpgPSM_stmt_close *stmt);
static int exec_stmt_leave(PLpgPSM_execstate *estate,
			   PLpgPSM_stmt_leave *stmt);
static int exec_stmt_return(PLpgPSM_execstate *estate,
				 PLpgPSM_stmt_return *stmt);
static int exec_stmt_signal(PLpgPSM_execstate *estate,
				PLpgPSM_stmt_signal *stmt);
static int exec_stmt_print(PLpgPSM_execstate *estate,
				PLpgPSM_stmt_print *stmt);
static int exec_stmt_case(PLpgPSM_execstate *estate,
				PLpgPSM_stmt_case *stmt);
				
static int exec_stmt_sql(PLpgPSM_execstate *estate,
				  PLpgPSM_stmt_sql *stmt);
static int exec_stmt_execute(PLpgPSM_execstate *estate,
					 PLpgPSM_stmt_execute *stmt);

static void plpgpsm_estate_setup(PLpgPSM_execstate *estate,
					 PLpgPSM_function *func,
					 ReturnSetInfo *rsi);
static void exec_eval_cleanup(PLpgPSM_execstate *estate);

static void exec_prepare_plan(PLpgPSM_execstate *estate,
				    PLpgPSM_expr *expr,
				    int cursorOptions);
static bool exec_simple_check_node(Node *node);
static void exec_simple_check_plan(PLpgPSM_expr *expr);
static bool exec_eval_simple_expr(PLpgPSM_execstate *estate,
					  PLpgPSM_expr *expr,
					  Datum *result,
					  bool *isNull,
					  Oid *rettype);
static void exec_assign_expr(PLpgPSM_execstate *estate,
				 PLpgPSM_datum *target,
				 PLpgPSM_expr *expr);
static void exec_assign_value(PLpgPSM_execstate *estate,
				  PLpgPSM_datum *target,
				  Datum value, Oid valtype, bool *isNull);
static void exec_eval_datum(PLpgPSM_execstate *estate,
				PLpgPSM_datum *datum,
				Oid expectedtypeid,
				Oid *typeid,
				Datum *value,
				bool *isnull);
static int exec_eval_integer(PLpgPSM_execstate *estate,
				  PLpgPSM_expr *expr,
				  bool *isNull);
static bool exec_eval_boolean(PLpgPSM_execstate *estate,
				  PLpgPSM_expr *expr,
				  bool *isNull);
static Datum exec_eval_expr(PLpgPSM_execstate *estate,
			   PLpgPSM_expr *expr,
			   bool *isNull,
			   Oid *rettype);
static int exec_run_select(PLpgPSM_execstate *estate,
				PLpgPSM_expr *expr, long maxtuples, Portal *portalP, char *name);
static void exec_move_row(PLpgPSM_execstate *estate,
			  PLpgPSM_rec *rec,
			  PLpgPSM_row *row,
			  HeapTuple tup, TupleDesc tupdesc);
static HeapTuple make_tuple_from_row(PLpgPSM_execstate *estate,
					PLpgPSM_row *row,
					TupleDesc tupdesc);
static char *convert_value_to_string(Datum value, Oid valtype);
static Datum exec_cast_value(Datum value, Oid valtype,
				Oid reqtype,
				FmgrInfo *reqinput,
				Oid reqtypioparam,
				int32 reqtypmod,
				bool isnull);
static Datum exec_simple_cast_value(Datum value, Oid valtype,
					   Oid reqtype, int32 reqtypmod,
					   bool isnull);
static void exec_init_tuple_store(PLpgPSM_execstate *estate);
static bool compatible_tupdesc(TupleDesc td1, TupleDesc td2);
static void exec_set_found(PLpgPSM_execstate *estate, bool state);
static void plpgpsm_create_econtext(PLpgPSM_execstate *estate); 
static void free_var(PLpgPSM_var *var);

static void clean_warning_data(PLpgPSM_execstate *estate);

static bool exception_matches_conditions(int elevel, int sqlerrcode, char *message, 
						PLpgPSM_condition *cond, 
						int pass, bool forErrors);


static PLpgPSM_diag_area *push_diagnostics_error(PLpgPSM_execstate *estate, ErrorData  *edata);


static PLpgPSM_diag_area *pop_and_clean_diagnostics(PLpgPSM_diag_area *current_diag);

#define RESET_DIAGNOSTICS_BEFORE(estate, stmt) \
	if (stmt->cmd_type != PLPGPSM_STMT_GETDIAG && stmt->cmd_type != PLPGPSM_STMT_BLOCK) \
		reset_diagnostic(estate->current_diag)

#define RESET_DIAGNOSTICS_AFTER(estate, stmt) \
	if (stmt->cmd_type == PLPGPSM_STMT_GETDIAG) \
		reset_diagnostic(estate->current_diag)
		
static void reset_diagnostic(PLpgPSM_diag_area *c);



#define SUCCESSFULL_COMPLETION 	MAKE_SQLSTATE('0','0','0','0','0')

/* 
 * Module variables, used for catching warnings and filling sqlstate 
 * variable for any warning signal
 */

int last_sqlstate = SUCCESSFULL_COMPLETION;
PLpgPSM_stmt_block *current_block = NULL;
PLpgPSM_stmt *warning_action	  = NULL;

typedef struct WarningData
{
	int sqlstate;
	char	*message;
	char	*detail;
	char	*hint;
	char	*context;
} WarningData;

WarningData *wdata = NULL;

static WarningData * plpgpsm_warning_filter(int sqlerrcode, char *message, char *detail, char *hint, char *context);
static PLpgPSM_diag_area *push_diagnostics_warning(PLpgPSM_execstate *estate, WarningData *w);


/* ----------
 * plpgpsm_exec_function	Called by the call handler for
 *				function execution.
 * ----------
 */
Datum
plpgpsm_exec_function(PLpgPSM_function *func, FunctionCallInfo fcinfo)
{
	PLpgPSM_execstate estate;
	PLpgPSM_stmt_block	*old_block = current_block;
	ErrorContextCallback plerrcontext;
	int			i;
	int			rc;
	

	/*
	 * Setup the execution state
	 */
	plpgpsm_estate_setup(&estate, func, (ReturnSetInfo *) fcinfo->resultinfo);
	
	estate.fcinfo = fcinfo;
					
	estate.sqlerrcode = SUCCESSFULL_COMPLETION;
	estate.sqlstate_varno = func->sqlstate_varno;


	/*
	 * Setup error traceback support for ereport()
	 */
	plerrcontext.callback = plpgpsm_exec_error_callback;
	plerrcontext.arg = &estate;
	plerrcontext.previous = error_context_stack;
	error_context_stack = &plerrcontext;

	current_block = NULL;

	/*
	 * Make local execution copies of all the datums
	 */
	estate.err_text = gettext_noop("during initialization of execution state");
	for (i = 0; i < estate.ndatums; i++)
		estate.datums[i] = copy_plpgpsm_datum(func->datums[i]);

	/*
	 * Store the actual call argument values into the appropriate variables
	 */
	estate.err_text = gettext_noop("while storing call arguments into local variables");
	for (i = 0; i < func->fn_nargs; i++)
	{
		int			n = func->fn_argvarnos[i];

		switch (estate.datums[n]->dtype)
		{
			case PLPGPSM_DTYPE_VAR:
				{
					PLpgPSM_var *var = (PLpgPSM_var *) estate.datums[n];

					var->value = fcinfo->arg[i];
					var->isnull = fcinfo->argnull[i];
					var->freeval = false;
				}
				break;

			case PLPGPSM_DTYPE_ROW:
				{
					PLpgPSM_row *row = (PLpgPSM_row *) estate.datums[n];
					row->isnull = fcinfo->argnull[i];

					if (!fcinfo->argnull[i])
					{
						HeapTupleHeader td;
						Oid			tupType;
						int32		tupTypmod;
						TupleDesc	tupdesc;
						HeapTupleData tmptup;

						td = DatumGetHeapTupleHeader(fcinfo->arg[i]);
						/* Extract rowtype info and find a tupdesc */
						tupType = HeapTupleHeaderGetTypeId(td);
						tupTypmod = HeapTupleHeaderGetTypMod(td);
						tupdesc = lookup_rowtype_tupdesc(tupType, tupTypmod);
						/* Build a temporary HeapTuple control structure */
						tmptup.t_len = HeapTupleHeaderGetDatumLength(td);
						ItemPointerSetInvalid(&(tmptup.t_self));
						tmptup.t_tableOid = InvalidOid;
						tmptup.t_data = td;
						exec_move_row(&estate, NULL, row, &tmptup, tupdesc);
						ReleaseTupleDesc(tupdesc);
					}
				}
				break;

			default:
				elog(ERROR, "unrecognized dtype: %d", func->datums[i]->dtype);
		}
	}

	estate.err_text = gettext_noop("during function entry");

	/*
	 * Set the magic variable FOUND to false
	 */
	 
	
	exec_set_found(&estate, false);

    
	/* reset sqlstate if exists */
	set_sqlstate(&estate);

	/*
	 * Let the instrumentation plugin peek at this function
	 */
	if (*plugin_ptr && (*plugin_ptr)->func_beg)
		((*plugin_ptr)->func_beg) (&estate, func);

	/*
	 * Now call the toplevel block of statements
	 */
	estate.err_text = NULL;
	estate.err_stmt = (PLpgPSM_stmt *) (func->action);

	estate.outer_stmt = true;
	rc = exec_stmt(&estate, func->action);

	if (rc != PLPGPSM_RC_RETURN)
	{
		estate.err_stmt = NULL;
		estate.err_text = NULL;

		/*
		 * Provide a more helpful message if a CONTINUE has been used outside
		 * a loop.
		 */
		if (rc == PLPGPSM_RC_CONTINUE)
			ereport(ERROR,
					(errcode(ERRCODE_SYNTAX_ERROR),
					 errmsg("ITERATE cannot be used outside a loop")));
		else
			ereport(ERROR,
			   (errcode(ERRCODE_S_R_E_FUNCTION_EXECUTED_NO_RETURN_STATEMENT),
				errmsg("control reached end of function without RETURN")));
	}

	/*
	 * We got a return value - process it
	 */
	estate.err_stmt = NULL;
	estate.err_text = gettext_noop("while casting return value to function's return type");

	fcinfo->isnull = estate.retisnull;

	if (estate.retisset)
	{
		ReturnSetInfo *rsi = estate.rsi;

		/* Check caller can handle a set result */
		if (!rsi || !IsA(rsi, ReturnSetInfo) ||
			(rsi->allowedModes & SFRM_Materialize) == 0)
			ereport(ERROR,
					(errcode(ERRCODE_FEATURE_NOT_SUPPORTED),
					 errmsg("set-valued function called in context that cannot accept a set")));
		rsi->returnMode = SFRM_Materialize;

		/* If we produced any tuples, send back the result */
		if (estate.tuple_store)
		{
			rsi->setResult = estate.tuple_store;
			if (estate.rettupdesc)
			{
				MemoryContext oldcxt;

				oldcxt = MemoryContextSwitchTo(estate.tuple_store_cxt);
				rsi->setDesc = CreateTupleDescCopy(estate.rettupdesc);
				MemoryContextSwitchTo(oldcxt);
			}
		}
		estate.retval = (Datum) 0;
		fcinfo->isnull = true;
	}
	else if (!estate.retisnull)
	{
		if (estate.retistuple)
		{
			/*
			 * We have to check that the returned tuple actually matches the
			 * expected result type.  XXX would be better to cache the tupdesc
			 * instead of repeating get_call_result_type()
			 */
			TupleDesc	tupdesc;

			switch (get_call_result_type(fcinfo, NULL, &tupdesc))
			{
				case TYPEFUNC_COMPOSITE:
					/* got the expected result rowtype, now check it */
					if (estate.rettupdesc == NULL ||
						!compatible_tupdesc(estate.rettupdesc, tupdesc))
						ereport(ERROR,
								(errcode(ERRCODE_DATATYPE_MISMATCH),
								 errmsg("returned record type does not match expected record type")));
					break;
				case TYPEFUNC_RECORD:

					/*
					 * Failed to determine actual type of RECORD.  We could
					 * raise an error here, but what this means in practice is
					 * that the caller is expecting any old generic rowtype,
					 * so we don't really need to be restrictive. Pass back
					 * the generated result type, instead.
					 */
					tupdesc = estate.rettupdesc;
					if (tupdesc == NULL)		/* shouldn't happen */
						elog(ERROR, "return type must be a row type");
					break;
				default:
					/* shouldn't get here if retistuple is true ... */
					elog(ERROR, "return type must be a row type");
					break;
			}

			/*
			 * Copy tuple to upper executor memory, as a tuple Datum. Make
			 * sure it is labeled with the caller-supplied tuple type.
			 */
			estate.retval =
				PointerGetDatum(SPI_returntuple((HeapTuple) (estate.retval),
												tupdesc));
		}
		else
		{
			/* Cast value to proper type */
			estate.retval = exec_cast_value(estate.retval, estate.rettype,
											func->fn_rettype,
											&(func->fn_retinput),
											func->fn_rettypioparam,
											-1,
											fcinfo->isnull);

			/*
			 * If the function's return type isn't by value, copy the value
			 * into upper executor memory context.
			 */
			if (!fcinfo->isnull && !func->fn_retbyval)
			{
				Size		len;
				void	   *tmp;

				len = datumGetSize(estate.retval, false, func->fn_rettyplen);
				tmp = SPI_palloc(len);
				memcpy(tmp, DatumGetPointer(estate.retval), len);
				estate.retval = PointerGetDatum(tmp);
			}
		}
	}

	estate.err_text = gettext_noop("during function exit");

	/*
	 * Let the instrumentation plugin peek at this function
	 */
	if (*plugin_ptr && (*plugin_ptr)->func_end)
		((*plugin_ptr)->func_end) (&estate, func);

	/* Clean up any leftover temporary memory */
	FreeExprContext(estate.eval_econtext);
	estate.eval_econtext = NULL;
	exec_eval_cleanup(&estate);

	/*
	 * Pop the error context stack
	 */
	error_context_stack = plerrcontext.previous;
	current_block = old_block;

	/*
	 * Return the function's result
	 */
	return estate.retval;
}


/* ----------
 * plpgpsm_exec_trigger		Called by the call handler for
 *				trigger execution.
 * ----------
 */
HeapTuple
plpgpsm_exec_trigger(PLpgPSM_function *func,
					 TriggerData *trigdata)
{
	PLpgPSM_execstate estate;
	ErrorContextCallback plerrcontext;
	int			i;
	int			rc;
	PLpgPSM_var *var;
	PLpgPSM_rec *rec_new,
			   *rec_old;
	HeapTuple	rettup;
	PLpgPSM_stmt_block *old_block;

	/*
	 * Setup the execution state
	 */
	plpgpsm_estate_setup(&estate, func, NULL);

	/*
	 * Setup error traceback support for ereport()
	 */
	plerrcontext.callback = plpgpsm_exec_error_callback;
	plerrcontext.arg = &estate;
	plerrcontext.previous = error_context_stack;
	error_context_stack = &plerrcontext;

	old_block = current_block;
	current_block = NULL;

	/*
	 * Make local execution copies of all the datums
	 */
	estate.err_text = gettext_noop("during initialization of execution state");
	for (i = 0; i < estate.ndatums; i++)
		estate.datums[i] = copy_plpgpsm_datum(func->datums[i]);

	/*
	 * Put the OLD and NEW tuples into record variables
	 */
	rec_new = (PLpgPSM_rec *) (estate.datums[func->new_varno]);
	rec_new->freetup = false;
	rec_new->freetupdesc = false;
	rec_old = (PLpgPSM_rec *) (estate.datums[func->old_varno]);
	rec_old->freetup = false;
	rec_old->freetupdesc = false;

	if (TRIGGER_FIRED_FOR_STATEMENT(trigdata->tg_event))
	{
		/*
		 * Per-statement triggers don't use OLD/NEW variables
		 */
		rec_new->tup = NULL;
		rec_new->tupdesc = NULL;
		rec_old->tup = NULL;
		rec_old->tupdesc = NULL;
	}
	else if (TRIGGER_FIRED_BY_INSERT(trigdata->tg_event))
	{
		rec_new->tup = trigdata->tg_trigtuple;
		rec_new->tupdesc = trigdata->tg_relation->rd_att;
		rec_old->tup = NULL;
		rec_old->tupdesc = NULL;
	}
	else if (TRIGGER_FIRED_BY_UPDATE(trigdata->tg_event))
	{
		rec_new->tup = trigdata->tg_newtuple;
		rec_new->tupdesc = trigdata->tg_relation->rd_att;
		rec_old->tup = trigdata->tg_trigtuple;
		rec_old->tupdesc = trigdata->tg_relation->rd_att;
	}
	else if (TRIGGER_FIRED_BY_DELETE(trigdata->tg_event))
	{
		rec_new->tup = NULL;
		rec_new->tupdesc = NULL;
		rec_old->tup = trigdata->tg_trigtuple;
		rec_old->tupdesc = trigdata->tg_relation->rd_att;
	}
	else
		elog(ERROR, "unrecognized trigger action: not INSERT, DELETE, or UPDATE");

	/*
	 * Assign the special tg_ variables
	 */

	var = (PLpgPSM_var *) (estate.datums[func->tg_op_varno]);
	if (TRIGGER_FIRED_BY_INSERT(trigdata->tg_event))
		var->value = DirectFunctionCall1(textin, CStringGetDatum("INSERT"));
	else if (TRIGGER_FIRED_BY_UPDATE(trigdata->tg_event))
		var->value = DirectFunctionCall1(textin, CStringGetDatum("UPDATE"));
	else if (TRIGGER_FIRED_BY_DELETE(trigdata->tg_event))
		var->value = DirectFunctionCall1(textin, CStringGetDatum("DELETE"));
	else
		elog(ERROR, "unrecognized trigger action: not INSERT, DELETE, or UPDATE");
	var->isnull = false;
	var->freeval = true;

	var = (PLpgPSM_var *) (estate.datums[func->tg_name_varno]);
	var->value = DirectFunctionCall1(namein,
							  CStringGetDatum(trigdata->tg_trigger->tgname));
	var->isnull = false;
	var->freeval = true;

	var = (PLpgPSM_var *) (estate.datums[func->tg_when_varno]);
	if (TRIGGER_FIRED_BEFORE(trigdata->tg_event))
		var->value = DirectFunctionCall1(textin, CStringGetDatum("BEFORE"));
	else if (TRIGGER_FIRED_AFTER(trigdata->tg_event))
		var->value = DirectFunctionCall1(textin, CStringGetDatum("AFTER"));
	else
		elog(ERROR, "unrecognized trigger execution time: not BEFORE or AFTER");
	var->isnull = false;
	var->freeval = true;

	var = (PLpgPSM_var *) (estate.datums[func->tg_level_varno]);
	if (TRIGGER_FIRED_FOR_ROW(trigdata->tg_event))
		var->value = DirectFunctionCall1(textin, CStringGetDatum("ROW"));
	else if (TRIGGER_FIRED_FOR_STATEMENT(trigdata->tg_event))
		var->value = DirectFunctionCall1(textin, CStringGetDatum("STATEMENT"));
	else
		elog(ERROR, "unrecognized trigger event type: not ROW or STATEMENT");
	var->isnull = false;
	var->freeval = true;

	var = (PLpgPSM_var *) (estate.datums[func->tg_relid_varno]);
	var->value = ObjectIdGetDatum(trigdata->tg_relation->rd_id);
	var->isnull = false;
	var->freeval = false;

	var = (PLpgPSM_var *) (estate.datums[func->tg_relname_varno]);
	var->value = DirectFunctionCall1(namein,
			CStringGetDatum(RelationGetRelationName(trigdata->tg_relation)));
	var->isnull = false;
	var->freeval = true;

	var = (PLpgPSM_var *) (estate.datums[func->tg_table_name_varno]);
	var->value = DirectFunctionCall1(namein,
			CStringGetDatum(RelationGetRelationName(trigdata->tg_relation)));
	var->isnull = false;
	var->freeval = true;

	var = (PLpgPSM_var *) (estate.datums[func->tg_table_schema_varno]);
	var->value = DirectFunctionCall1(namein,
									 CStringGetDatum(
													 get_namespace_name(
														RelationGetNamespace(
												   trigdata->tg_relation))));
	var->isnull = false;
	var->freeval = true;

	var = (PLpgPSM_var *) (estate.datums[func->tg_nargs_varno]);
	var->value = Int16GetDatum(trigdata->tg_trigger->tgnargs);
	var->isnull = false;
	var->freeval = false;

	/*
	 * Store the trigger argument values into the special execution state
	 * variables
	 */
	estate.err_text = gettext_noop("while storing call arguments into local variables");
	estate.trig_nargs = trigdata->tg_trigger->tgnargs;
	if (estate.trig_nargs == 0)
		estate.trig_argv = NULL;
	else
	{
		estate.trig_argv = palloc(sizeof(Datum) * estate.trig_nargs);
		for (i = 0; i < trigdata->tg_trigger->tgnargs; i++)
			estate.trig_argv[i] = DirectFunctionCall1(textin,
						   CStringGetDatum(trigdata->tg_trigger->tgargs[i]));
	}

	estate.err_text = gettext_noop("during function entry"); 

	/*
	 * Set the magic variable FOUND to false
	 */
	exec_set_found(&estate, false);

	/*
	 * Let the instrumentation plugin peek at this function
	 */
	if (*plugin_ptr && (*plugin_ptr)->func_beg)
		((*plugin_ptr)->func_beg) (&estate, func);

	/*
	 * Now call the toplevel block of statements
	 */
	estate.err_text = NULL;
	estate.err_stmt = (PLpgPSM_stmt *) (func->action);
	estate.outer_stmt = true;

	rc = exec_stmt(&estate, func->action);
	if (rc != PLPGPSM_RC_RETURN)
	{
		estate.err_stmt = NULL;
		estate.err_text = NULL;

		/*
		 * Provide a more helpful message if a CONTINUE has been used outside
		 * a loop.
		 */
		if (rc == PLPGPSM_RC_CONTINUE)
			ereport(ERROR,
					(errcode(ERRCODE_SYNTAX_ERROR),
					 errmsg("ITERATE cannot be used outside a loop")));
		else
			ereport(ERROR,
			   (errcode(ERRCODE_S_R_E_FUNCTION_EXECUTED_NO_RETURN_STATEMENT),
				errmsg("control reached end of trigger procedure without RETURN")));
	}

	estate.err_stmt = NULL;
	estate.err_text = gettext_noop("during function exit");

	if (estate.retisset)
		ereport(ERROR,
				(errcode(ERRCODE_DATATYPE_MISMATCH),
				 errmsg("trigger procedure cannot return a set")));

	/*
	 * Check that the returned tuple structure has the same attributes, the
	 * relation that fired the trigger has. A per-statement trigger always
	 * needs to return NULL, so we ignore any return value the function itself
	 * produces (XXX: is this a good idea?)
	 *
	 * XXX This way it is possible, that the trigger returns a tuple where
	 * attributes don't have the correct atttypmod's length. It's up to the
	 * trigger's programmer to ensure that this doesn't happen. Jan
	 */
	if (estate.retisnull || TRIGGER_FIRED_FOR_STATEMENT(trigdata->tg_event))
		rettup = NULL;
	else
	{
		if (!compatible_tupdesc(estate.rettupdesc,
								trigdata->tg_relation->rd_att))
			ereport(ERROR,
					(errcode(ERRCODE_DATATYPE_MISMATCH),
					 errmsg("returned tuple structure does not match table of trigger event")));
		/* Copy tuple to upper executor memory */
		rettup = SPI_copytuple((HeapTuple) (estate.retval));
	}

	/*
	 * Let the instrumentation plugin peek at this function
	 */
	if (*plugin_ptr && (*plugin_ptr)->func_end)
		((*plugin_ptr)->func_end) (&estate, func);

	/* Clean up any leftover temporary memory */
	FreeExprContext(estate.eval_econtext);
	estate.eval_econtext = NULL;
	exec_eval_cleanup(&estate);

	/*
	 * Pop the error context stack
	 */
	error_context_stack = plerrcontext.previous;

	current_block = old_block;

	/*
	 * Return the trigger's result
	 */
	return rettup;
}


/*
 * error context callback to let us supply a call-stack traceback
 */
static void
plpgpsm_exec_error_callback(void *arg)
{
	PLpgPSM_execstate *estate = (PLpgPSM_execstate *) arg;

	/* safety check, shouldn't happen */
	if (estate->err_func == NULL)
		return;

	/* if we are doing RAISE, don't report its location */
	if (estate->err_text == raise_skip_msg)
		return;

       if (estate->err_text != NULL)                                                                          
        {
                /*
                 * We don't expend the cycles to run gettext() on err_text unless we
                 * actually need it.  Therefore, places that set up err_text should
                 * use gettext_noop() to ensure the strings get recorded in the
                 * message dictionary.
                 *                                                                                                     
                 * If both err_text and err_stmt are set, use the err_text as
                 * description, but report the err_stmt's line number.  When 
                 * err_stmt is not set, we're in function entry/exit, or some such 
                 * place not attached to a specific line number.                                                       
                 */                                                                                                    
                if (estate->err_stmt != NULL)                                                                          
                {                                                                                                      
                        /*                                                                                             
                         * translator: last %s is a phrase such as "during statement                                   
                         * block local variable initialization"                                                        
                         */                                                                                            
                        errcontext("PL/pgPSM function \"%s\" line %d %s", 
                                           estate->err_func->fn_name, 
                                           estate->err_stmt->lineno,
                                           gettext(estate->err_text));
                }                                                                                                      
                else                                                                                                   
                {                                                                                                      
                        /*                                                                                             
                         * translator: last %s is a phrase such as "while storing call                                 
                         * arguments into local variables"                                                             
                         */                                                                                            
                        errcontext("PL/pgPSM function \"%s\" %s",                                                      
                                           estate->err_func->fn_name,                                                  
                                           gettext(estate->err_text));                                                 
                }                                                                                                      
        }                                                                                                              
        else if (estate->err_stmt != NULL)                                                                             
        {                                                                                                              
                /* translator: last %s is a plpgpsm statement type name */                                             
                errcontext("PL/pgPSM function \"%s\" line %d at %s",                                                   
                                   estate->err_func->fn_name,                                                          
                                   estate->err_stmt->lineno,                                                           
                                   plpgpsm_stmt_typename(estate->err_stmt));                                           
        }                                                                                                              
        else                                                                                                           
                errcontext("PL/pgPSM function \"%s\"",                                                                 
                                   estate->err_func->fn_name);                        
}

#define sec_dup(p)	(p) ? pstrdup(p) : NULL;

static WarningData *
plpgpsm_warning_filter(int sqlerrcode, char *message, char *detail, char *hint, char *context)
{
	ListCell	*e;
	int 		pass;
	PLpgPSM_exception *detected_exception;
	int 	elevel = ERROR;

	
	last_sqlstate = sqlerrcode;
	warning_action = NULL;
		
	if (current_block && current_block->exceptions)
	{
		/* I have to set SQLSTATE in compound statement block */

		for(pass = 1; pass <= 3; pass++)
		{
			bool detect_ambiguous = false;

			detected_exception = NULL;
				
			foreach(e, current_block->exceptions->exc_list)
			{
				PLpgPSM_exception *exception = (PLpgPSM_exception*) lfirst(e);
			
				if (exception_matches_conditions(WARNING, sqlerrcode, message,
									exception->conditions, pass, false))
				{
					if (detected_exception)
					{
						if (pass == 3)
							elog(ERROR, "warning handler or not found handler is ambiguous");
						else
						{
							detected_exception = NULL;
							detect_ambiguous = true;
							break;
						}
					}
					detected_exception = exception;
				}
			}
			
			if (detected_exception)
			{
				WarningData *wd = (WarningData*) palloc0(sizeof(WarningData));
				MemoryContext oldcontext = MemoryContextSwitchTo(TopTransactionContext);
				
				wd->sqlstate = sqlerrcode;
				wd->message 	= sec_dup(message); 
				wd->detail	= sec_dup(detail);
				wd->hint	= sec_dup(hint);
				wd->context	= sec_dup(context);

				MemoryContextSwitchTo(oldcontext);                  

				warning_action = detected_exception->action;
			
				return wd;
			}
			else 
			{
				/* 
				 * I can fast detect missing handler or
				 * redundant handler. We can continue in next pass
				 * only if we detected detect_ambiguous handlers
				 */
				if (pass == 1 && !detect_ambiguous)
					/* first pass and didn't find */
					break;
				else if (pass > 1 && !detect_ambiguous)
					elog(ERROR, "warning handler or not found handler is ambiguous");
			}
		}
	}
		
	/* I have to resend signal to error handler */
	if (ERRCODE_TO_CATEGORY(sqlerrcode) == ERRCODE_SUCCESSFUL_COMPLETION)
		elevel = INFO;
	else if (ERRCODE_TO_CATEGORY(sqlerrcode) == ERRCODE_NO_DATA)
		/* implicit handler */
		return NULL;
	else if (ERRCODE_TO_CATEGORY(sqlerrcode) == ERRCODE_WARNING)
		elevel = WARNING;
	else
		elog(ERROR, "warning filter detect error sqlstate '%s'", 
						unpack_sql_state(sqlerrcode));
	ereport(elevel,
		(errcode(sqlerrcode),
	    	    errmsg_internal("%s", message),
	    	    (detail != NULL) ? errdetail("%s", detail) : 0,
	    	    (hint != NULL) ? errhint("%s", hint) :0,
	    	    (context != NULL) ? errcontext("%s", context) : 0));

	return NULL;
}


/* ----------
 * Support function for initializing local execution variables
 * ----------
 */
static PLpgPSM_datum *
copy_plpgpsm_datum(PLpgPSM_datum *datum)
{
	PLpgPSM_datum *result;

	switch (datum->dtype)
	{
		case PLPGPSM_DTYPE_VAR:
			{
				PLpgPSM_var *new = palloc(sizeof(PLpgPSM_var));

				memcpy(new, datum, sizeof(PLpgPSM_var));
				/* Ensure the value is null (possibly not needed?) */
				new->value = 0;
				new->isnull = true;
				new->freeval = false;

				result = (PLpgPSM_datum *) new;
			}
			break;

		case PLPGPSM_DTYPE_REC:
			{
				PLpgPSM_rec *new = palloc(sizeof(PLpgPSM_rec));

				memcpy(new, datum, sizeof(PLpgPSM_rec));
				/* Ensure the value is null (possibly not needed?) */
				new->tup = NULL;
				new->tupdesc = NULL;
				new->freetup = false;
				new->freetupdesc = false;

				result = (PLpgPSM_datum *) new;
			}
			break;

		case PLPGPSM_DTYPE_ROW:
		case PLPGPSM_DTYPE_RECFIELD:
		case PLPGPSM_DTYPE_ARRAYELEM:
		case PLPGPSM_DTYPE_TRIGARG:
		case PLPGPSM_DTYPE_CONDITION:

			/*
			 * These datum records are read-only at runtime, so no need to
			 * copy them
			 */
			result = datum;
			break;

		default:
			elog(ERROR, "unrecognized dtype: %d", datum->dtype);
			result = NULL;		/* keep compiler quiet */
			break;
	}

	return result;
}

#define CONDITION	MAKE_SQLSTATE('P','1','0','0','1')

/* priority match (EXACT >> CATEGORY >> GENERAL */
static bool
exception_matches_conditions(int elevel, int sqlerrcode, char *message, PLpgPSM_condition *cond, int pass, bool forErrors)
{
	/* Never catch ERRCODE_QUERY_CANCELED */
	if (sqlerrcode == ERRCODE_QUERY_CANCELED)
		return false;

	for (; cond != NULL; cond = cond->next)
	{
		int			sqlerrstate = cond->sqlerrstate;

		/* skip warnings, or errors */
		if (cond->handle_errors != forErrors)
			continue;

		/* 
		 * SQLEXCEPTION are not identified by class, has special SQLSTATE '00000',
		 * which has meaning success, but success isn't trapped, and can be used 
		 */   
		if (sqlerrstate == 0 && pass == 1 && elevel == ERROR)
			return true;
		/* Exact match? */
		else if (sqlerrcode == sqlerrstate && pass <= 3)
		{
			/* conditionals has one sqlstate. name is important*/
			if (sqlerrcode == CONDITION)
			{
				if (strcmp(cond->condname, message ? message : "") == 0)
					return true;
			}
			else
				return true;
		}
		/* if possibility for CATEGORY or GENERAL match */
		else if (ERRCODE_TO_CATEGORY(sqlerrcode) == sqlerrstate)
		{
			/* category match */
			if (ERRCODE_IS_CATEGORY(sqlerrstate) && pass <= 2)
				return true;
			
			/* general handler for 01 and 02 classes */
			if (cond->general_handler && pass == 1)
				return true;
		}
	}

	return false;
}


/* ----------
 * exec_stmt_block			Execute a block of statements
 * ----------
 */
static int
exec_stmt_block(PLpgPSM_execstate *estate, PLpgPSM_stmt_block *block)
{
	volatile int rc = -1;
	int			i;
	int			n;

	PLpgPSM_stmt_block	*old_block = current_block;
	int old_sqlstate_varno;
	
	if (!block->generated)
		estate->outer_stmt = false;

	current_block = block;

	old_sqlstate_varno = estate->sqlstate_varno;

	block->opened_cursors = estate->opened_cursors;

	/*
	 * First initialize all variables declared in this block
	 */
	estate->err_text = gettext_noop("during statement block local variable initialization"); 	 

	for (i = 0; i < block->n_initvars; i++)
	{
		n = block->initvarnos[i];
		switch (estate->datums[n]->dtype)
		{
			case PLPGPSM_DTYPE_VAR:
				{
					PLpgPSM_var *var = (PLpgPSM_var *) (estate->datums[n]);
					
					free_var(var);
					
					var->portal = NULL;
					
					if (!var->isconst || var->isnull)
					{
						if (var->default_val == NULL)
						{
							/* initially it contains a NULL */
							var->value = (Datum) 0;
							var->isnull = true;
							/*                                                 
                                                	 * If needed, give the datatype a chance to reject 
                                                	 * NULLs, by assigning a NULL to the variable.     
                                                	 * We claim the value is of type UNKNOWN, not the  
                                                	 * var's datatype, else coercion will be skipped.  
                                                	 * (Do this before the notnull check to be         
                                                	 * consistent with exec_assign_value.)             
                                                	 */                                     
							if (!var->datatype->typinput.fn_strict)
							{
								bool valIsNull = true;	
                                
							        exec_assign_value(estate,                                
                                                            		                  (PLpgPSM_datum *) var, 
                                                                        		  (Datum) 0,             
	                                                                                  UNKNOWNOID,            
        		                                                                  &valIsNull);  
							}
							if (var->notnull)
								ereport(ERROR,
									(errcode(ERRCODE_NULL_VALUE_NOT_ALLOWED),
									 errmsg("variable \"%s\" declared NOT NULL cannot default to NULL",
											var->refname)));
						}
						else
						{
							exec_assign_expr(estate, (PLpgPSM_datum *) var,
											 var->default_val);
						}
					}
				}
				break;

			case PLPGPSM_DTYPE_REC:
				{
					PLpgPSM_rec *rec = (PLpgPSM_rec *) (estate->datums[n]);

					if (rec->freetup)
					{
						heap_freetuple(rec->tup);
						FreeTupleDesc(rec->tupdesc);
						rec->freetup = false;
					}

					rec->tup = NULL;
					rec->tupdesc = NULL;
				}
				break;

			case PLPGPSM_DTYPE_RECFIELD:
			case PLPGPSM_DTYPE_ARRAYELEM:
			case PLPGPSM_DTYPE_CONDITION:
				break;

			default:
				elog(ERROR, "unrecognized dtype: %d",
					 estate->datums[n]->dtype);
		}
	}

	/* local sqlstate overwrite outer sqlstate and need reset */
	if (estate->sqlstate_varno != block->sqlstate_varno)
	{
		estate->sqlstate_varno = block->sqlstate_varno;
		estate->sqlerrcode = SUCCESSFULL_COMPLETION;
		set_sqlstate(estate);
	}

	estate->err_text = gettext_noop("during statement block entry");  

	/* I handle only BEGIN ATOMIC here */
	if (block->atomic)
	{
	
		/*
		 * Execute the statements in the block's body inside a sub-transaction
		 */
		MemoryContext oldcontext = CurrentMemoryContext;
		ResourceOwner oldowner = CurrentResourceOwner;
                ExprContext *old_eval_econtext = estate->eval_econtext;
                EState     *old_eval_estate = estate->eval_estate;
                long int        old_eval_estate_simple_id = estate->eval_estate_simple_id;

		BeginInternalSubTransaction(NULL);
		/* Want to run statements inside function's memory context */
		MemoryContextSwitchTo(oldcontext);
		
		
		PG_TRY();
		{
	                /*                                                                             
                         * We need to run the block's statements with a new eval_econtext              
                         * that belongs to the current subtransaction; if we try to use                
                         * the outer econtext then ExprContext shutdown callbacks will be              
                         * called at the wrong times.                                                  
                         */                 
			plpgpsm_create_econtext(estate); 
			estate->err_text = NULL;

			rc = exec_stmts(estate, block->body, block);
			
			estate->err_text = gettext_noop("during statement block exit"); 

			if (rc != PLPGPSM_RC_UNDO)
			{
				/*
				 * If the block ended with RETURN, we may need to copy the return
				 * value out of the subtransaction eval_context.  This is currently
				 * only needed for scalar result types --- rowtype values will
				 * always exist in the function's own memory context.
				 */
				if (rc == PLPGPSM_RC_RETURN &&
					!estate->retisset &&
					!estate->retisnull &&
					estate->rettupdesc == NULL)
				{
					int16		resTypLen;
					bool		resTypByVal;

					get_typlenbyval(estate->rettype, &resTypLen, &resTypByVal);
					estate->retval = datumCopy(estate->retval,
											   resTypByVal, resTypLen);
				}
			
				/* Commit the inner transaction, return to outer xact context */
				ReleaseCurrentSubTransaction();
				MemoryContextSwitchTo(oldcontext);
				CurrentResourceOwner = oldowner;
				
				/*
				 * AtEOSubXact_SPI() should not have popped any SPI context, but
			         * just in case it did, make sure we remain connected.
			         */
				 SPI_restore_connection();
			}
			else
			{
				/* UNDO any non error signals */
				RollbackAndReleaseCurrentSubTransaction();
				MemoryContextSwitchTo(oldcontext);
				CurrentResourceOwner = oldowner;
				
				SPI_restore_connection();
				
				rc = PLPGPSM_RC_OK;
			}

			/* Revert to outer eval_econtext */ 
                        estate->eval_econtext = old_eval_econtext;
                        estate->eval_estate = old_eval_estate;
                        estate->eval_estate_simple_id = old_eval_estate_simple_id;
		}
		PG_CATCH();
		{
			ErrorData  *edata;
			ListCell   *e;
			int		pass;
			PLpgPSM_exception *detected_exception;
			PLpgPSM_diag_area *current_diag;

			estate->err_text = gettext_noop("during exception cleanup"); 

			/* Save error info */
			MemoryContextSwitchTo(oldcontext);
			edata = CopyErrorData();
			FlushErrorState();

			/* Abort the inner transaction */
			RollbackAndReleaseCurrentSubTransaction();
			MemoryContextSwitchTo(oldcontext);
			CurrentResourceOwner = oldowner;

			/* Revert to outer eval_econtext */ 
                        estate->eval_econtext = old_eval_econtext;
                        estate->eval_estate = old_eval_estate;
                        estate->eval_estate_simple_id = old_eval_estate_simple_id;
			current_block = block;

			/*
			 * If AtEOSubXact_SPI() popped any SPI context of the subxact, it
			 * will have left us in a disconnected state.  We need this hack
			 * to return to connected state.
			 */
			SPI_restore_connection();

			current_diag = push_diagnostics_error(estate, edata);

			/* Look for a matching exception handler */
			
			for(pass = 1; pass <= 3; pass++)
			{
				bool detect_ambiguous = false;
				
				detected_exception = NULL;
				
								
				foreach(e, block->exceptions->exc_list)
				{
					PLpgPSM_exception *exception = (PLpgPSM_exception *) lfirst(e);

					if (exception_matches_conditions(edata->elevel, edata->sqlerrcode, edata->message, 
												    exception->conditions, pass, true))
					{
						if (detected_exception)
						{
							if (pass == 3)
								elog(ERROR, "exception handler is ambiguous");
							else
							{
								detected_exception = NULL;
								detect_ambiguous = true;
								break;
							}
						}
						detected_exception = exception;
					}
				}

				/* only if we found unique handler */
				if (detected_exception)
				{	
					/*
					 * Initialize the magic SQLSTATE and SQLERRM variables for
					 * the exception block. We needn't do this until we have
					 * found a matching exception.
					 */
					if (block->exceptions->sqlstate_varno != PLPGPSM_INVALID_VARNO)
					{
						estate->sqlerrcode = edata->sqlerrcode;
						estate->sqlstate_varno = block->sqlstate_varno;
						set_sqlstate(estate);
					}

					estate->err_text = NULL; 
					 									
					rc = exec_stmt(estate, detected_exception->action);
					
					/* sqlstate over exception handler is '00000' */
					estate->sqlstate_varno = block->sqlstate_varno;
					estate->sqlerrcode = SUCCESSFULL_COMPLETION;
					set_sqlstate(estate);
					
					break;
				}
				else 
				{
					/* 
					 * I can fast detect missing handler or
					 * redundant handler. We can continue in next pass
					 * only if we detected detect_ambiguous handlers
					 */
					if (pass == 1 && !detect_ambiguous)
						/* first pass and didn't find */
						break;
					else if (pass > 1 && !detect_ambiguous)
						elog(ERROR, "exception handler is ambiguous");
				}
				
			}

			/* If no match found, re-throw the error */
			if (!detected_exception)
			{
				current_block = old_block;
				ReThrowError(edata);
			}
			else
				FreeErrorData(edata);

			estate->current_diag = pop_and_clean_diagnostics(current_diag);
		}
		PG_END_TRY();
	}
	else
	{
		/*
		 * Just execute the statements in the block's body
		 */
				 
		estate->err_text = NULL; 

		rc = exec_stmts(estate, block->body, block);
		estate->sqlstate_varno = block->sqlstate_varno;		
	}

	estate->err_text = NULL; 

	/* close all openned unnamed cursors */

	for (i = 0; i < block->n_initvars; i++)
	{
		n = block->initvarnos[i];

		switch (estate->datums[n]->dtype)
		{
			case PLPGPSM_DTYPE_VAR:
				{
					PLpgPSM_var *var = (PLpgPSM_var *) (estate->datums[n]);
					
					if (var->portal && !var->holdable)
					{
						SPI_cursor_close(var->portal);
						var->portal = NULL;
					}
				}
		}
	}

	current_block = old_block;
	estate->sqlstate_varno = old_sqlstate_varno; 
	
	/*
	 * Handle the return code.
	 */
	 	
	switch (rc)
	{
		case PLPGPSM_RC_OK:
		case PLPGPSM_RC_CONTINUE:
		case PLPGPSM_RC_RETURN:
			return rc;

		case PLPGPSM_RC_EXIT:
			if (estate->exitlabel == NULL)
				return PLPGPSM_RC_OK;
			if (block->label == NULL)
				return PLPGPSM_RC_EXIT;
			if (strcmp(block->label, estate->exitlabel))
				return PLPGPSM_RC_EXIT;
			estate->exitlabel = NULL;
			return PLPGPSM_RC_OK;

		default:
			elog(ERROR, "unrecognized rc: %d", rc);
	}


	return PLPGPSM_RC_OK;
}


/* ----------
 * exec_stmts			Iterate over a list of statements
 *				as long as their return code is OK
 * ----------
 */
static int
exec_stmts(PLpgPSM_execstate *estate, List *stmts, PLpgPSM_stmt_block *block)
{
	ListCell   *s;
	PLpgPSM_stmt_block *old_block = current_block;
	PLpgPSM_exception_block *eblock = NULL;

	bool outer_is_atomic = false;
	
	/* sqlstate have to be repeatly reset to origin value */
	int sqlstate_varno = estate->sqlstate_varno; 

	if (stmts == NIL)
	{
		/*
		 * Ensure we do a CHECK_FOR_INTERRUPTS() even though there is no
		 * statement.  This prevents hangup in a tight loop if, for instance,
		 * there is a LOOP construct with an empty body.
		 */
		CHECK_FOR_INTERRUPTS();
		return PLPGPSM_RC_OK;
	}

	if (block)
	{
		outer_is_atomic = block->atomic;
		eblock = block->exceptions;
	}

	foreach(s, stmts)
	{
		int rc;
	
		PLpgPSM_stmt *stmt = (PLpgPSM_stmt *) lfirst(s);
		
		if (eblock && (eblock->has_continue_handler || eblock->has_exit_handler))
		{
			    /*
			     * Execute the statements in the block's body inside a sub-transaction
			     */
			    MemoryContext oldcontext = CurrentMemoryContext;
			    ResourceOwner oldowner = CurrentResourceOwner;
            		    ExprContext *old_eval_econtext = estate->eval_econtext;
            		    EState     *old_eval_estate = estate->eval_estate;
            		    long int        old_eval_estate_simple_id = estate->eval_estate_simple_id;
			    estate->err_text = gettext_noop("during statement entry"); 

			    BeginInternalSubTransaction(NULL);
			    /* Want to run statements inside function's memory context */
			    MemoryContextSwitchTo(oldcontext);
		
			    PG_TRY();
			    {
				    plpgpsm_create_econtext(estate); 
				    estate->err_text = NULL; 

				    rc = exec_stmt(estate, stmt);

				    estate->err_text = gettext_noop("during statement exit"); 
				    
				    /* 
				     * if UNDO WARNING handler, rollback current subtransactio
				     * and return back to rollback outer transaction.
				     */
				    if (rc == PLPGPSM_RC_UNDO)
				    {
					    /* UNDO any non error signals */
					    RollbackAndReleaseCurrentSubTransaction();
					    MemoryContextSwitchTo(oldcontext);
					    CurrentResourceOwner = oldowner;
					    
					    SPI_restore_connection();
				    }
				    else
				    {
					    /* Commit the inner transaction, return to outer xact context */
					    ReleaseCurrentSubTransaction();
					    MemoryContextSwitchTo(oldcontext);
					    CurrentResourceOwner = oldowner;
				    }
				    
				    /*
				     * AtEOSubXact_SPI() should not have popped any SPI context, but
			             * just in case it did, make sure we remain connected.
			             */
				    SPI_restore_connection();

				    /* Revert to outer eval_econtext */ 
                    		    estate->eval_econtext = old_eval_econtext;
                    		    estate->eval_estate = old_eval_estate;
                    		    estate->eval_estate_simple_id = old_eval_estate_simple_id;
			    }
			    PG_CATCH();
			    {
				    ErrorData  *edata;
				    ListCell   *e;
				    int		pass;
				    PLpgPSM_exception *detected_exception;
				    PLpgPSM_diag_area *current_diag;

				    estate->err_text = gettext_noop("during exception cleanup");

				    /* Save error info */
				    MemoryContextSwitchTo(oldcontext);
				    edata = CopyErrorData();
				    FlushErrorState();

				    /* Abort the inner transaction */
				    RollbackAndReleaseCurrentSubTransaction();
				    MemoryContextSwitchTo(oldcontext);
				    CurrentResourceOwner = oldowner;

				    /* Revert to outer eval_econtext */ 
                    		    estate->eval_econtext = old_eval_econtext;
                    		    estate->eval_estate = old_eval_estate;
                    		    estate->eval_estate_simple_id = old_eval_estate_simple_id;			
			
				    current_block = old_block;

				    /*
				     * If AtEOSubXact_SPI() popped any SPI context of the subxact, it
				     * will have left us in a disconnected state.  We need this hack
				     * to return to connected state.
				     */
				    SPI_restore_connection();

				    current_diag = push_diagnostics_error(estate, edata);

				    /* Look for a matching exception handler */
				    for(pass = 1; pass <= 3; pass++)
				    {
					    bool detect_ambiguous = false;
				
					    detected_exception = NULL;
							
					    foreach(e, eblock->exc_list)
					    {
						    PLpgPSM_exception *exception = (PLpgPSM_exception *) lfirst(e);

						    if (exception_matches_conditions(edata->elevel, edata->sqlerrcode, edata->message, 
														exception->conditions, pass, true))
						    {
							    if (detected_exception)
							    {
								    if (pass == 3)
									    elog(ERROR, "exception handler is ambiguous");
								    else
								    {
									    detected_exception = NULL;
									    detect_ambiguous = true;
									    break;
								    }
							    }
							    detected_exception = exception;
						    }
					    }

					    /* only if we found unique handler */
					    if (detected_exception)
					    {	
						    /*
						     * Initialize the magic SQLSTATE and SQLERRM variables for
						     * the exception block. We needn't do this until we have
						     * found a matching exception.
						     */
						    if (block->sqlstate_varno != PLPGPSM_INVALID_VARNO)
						    {
							    estate->sqlerrcode = edata->sqlerrcode;
							    estate->sqlstate_varno = block->sqlstate_varno;
							    set_sqlstate(estate);
						    }

						    estate->err_text = NULL; 
					 									
						    rc = exec_stmt(estate, detected_exception->action);

						    /* sqlstate over exception handler is '00000' */
						    estate->sqlstate_varno = block->sqlstate_varno;
						    estate->sqlerrcode = SUCCESSFULL_COMPLETION;
						    set_sqlstate(estate);

						    /* behave depends on atomicity outer block */
						    
						    switch (detected_exception->handler_type)
						    {
							    case PLPGPSM_HDL_CONTINUE:
								    rc = outer_is_atomic ? PLPGPSM_RC_UNDO : PLPGPSM_RC_OK;
								    break;
								    
							    case PLPGPSM_HDL_EXIT:
								    current_block = old_block;
								    estate->sqlstate_varno = sqlstate_varno;
								    return outer_is_atomic ? PLPGPSM_RC_UNDO : PLPGPSM_RC_OK;
							    
							    case PLPGPSM_HDL_UNDO:
								    current_block = old_block;
								    estate->sqlstate_varno = sqlstate_varno;
								    return PLPGPSM_RC_UNDO;
						    }
						    					
						    break;
					    }
					    else 
					    {
						    /* 
						     * I can fast detect missing handler or
						     * redundant handler. We can continue in next pass
						     * only if we detected detect_ambiguous handlers
						     */
				    		    if (pass == 1 && !detect_ambiguous)
							    /* first pass and didn't find */
							    break;
						    else if (pass > 1 && !detect_ambiguous)
							    elog(ERROR, "exception handler is ambiguous");
					    }
				
				    }

				    /* If no match found, re-throw the error */
				    if (!detected_exception)
				    {
					    current_block = old_block;
					    ReThrowError(edata);
				    }
				    else
					    FreeErrorData(edata);
				    
				    estate->current_diag = pop_and_clean_diagnostics(current_diag);
			}
			PG_END_TRY();
		}
		else
		{
			estate->err_text = NULL; 
			
			rc = exec_stmt(estate, stmt);
		}
		
		estate->err_text = NULL; 
		
		if (rc != PLPGPSM_RC_OK)
			return rc;
	}

	estate->err_text = NULL; 

	return PLPGPSM_RC_OK;
}


/* ----------
 * exec_stmt			Distribute one statement to the statements
 *				type specific execution function.
 * ----------
 */
static int
exec_stmt(PLpgPSM_execstate *estate, PLpgPSM_stmt *stmt)
{
	PLpgPSM_stmt *save_estmt;
	int			rc = -1;
	int 		old_sqlerrcode;

	save_estmt = estate->err_stmt;
	estate->err_stmt = stmt;

	estate->edata = NULL;
	warning_action = NULL;

	/* Let the plugin know that we are about to execute this statement */
	if (*plugin_ptr && (*plugin_ptr)->stmt_beg)
		((*plugin_ptr)->stmt_beg) (estate, stmt);

	CHECK_FOR_INTERRUPTS();
	
	old_sqlerrcode = estate->sqlerrcode;
	estate->sqlerrcode = SUCCESSFULL_COMPLETION;

	RESET_DIAGNOSTICS_BEFORE(estate, stmt);

	switch (stmt->cmd_type)
	{
		case PLPGPSM_STMT_BLOCK:
			rc = exec_stmt_block(estate, (PLpgPSM_stmt_block *) stmt);
			break;

		case PLPGPSM_STMT_ASSIGN:
			rc = exec_stmt_assign(estate, (PLpgPSM_stmt_assign *) stmt);
			break;

		case PLPGPSM_STMT_CALL:
			rc = exec_stmt_call(estate, (PLpgPSM_stmt_call *) stmt);
			break;

		case PLPGPSM_STMT_GETDIAG:
			rc = exec_stmt_getdiag(estate, (PLpgPSM_stmt_getdiag *) stmt);
			break;

		case PLPGPSM_STMT_IF:
			rc = exec_stmt_if(estate, (PLpgPSM_stmt_if *) stmt);
			break;

		case PLPGPSM_STMT_LOOP:
			rc = exec_stmt_loop(estate, (PLpgPSM_stmt_loop *) stmt);
			break;

		case PLPGPSM_STMT_FORS:
			rc = exec_stmt_fors(estate, (PLpgPSM_stmt_fors *) stmt);
			break;

		case PLPGPSM_STMT_LEAVE:
			rc = exec_stmt_leave(estate, (PLpgPSM_stmt_leave *) stmt);
			break;

		case PLPGPSM_STMT_RETURN:
			rc = exec_stmt_return(estate, (PLpgPSM_stmt_return *) stmt);
			break;

		case PLPGPSM_STMT_SIGNAL:
			rc = exec_stmt_signal(estate, (PLpgPSM_stmt_signal *) stmt);
			break;
			
		case PLPGPSM_STMT_PRINT:
			rc = exec_stmt_print(estate, (PLpgPSM_stmt_print *) stmt);
			break;

		case PLPGPSM_STMT_SQL:
			rc = exec_stmt_sql(estate, (PLpgPSM_stmt_sql *) stmt);
			break;

		case PLPGPSM_STMT_EXECUTE:
			rc = exec_stmt_execute(estate, (PLpgPSM_stmt_execute *) stmt);
			break;

		case PLPGPSM_STMT_OPEN:
			rc = exec_stmt_open(estate, (PLpgPSM_stmt_open *) stmt);
			break;

		case PLPGPSM_STMT_FETCH:
			rc = exec_stmt_fetch(estate, (PLpgPSM_stmt_fetch *) stmt);
			break;

		case PLPGPSM_STMT_CLOSE:
			rc = exec_stmt_close(estate, (PLpgPSM_stmt_close *) stmt);
			break;
			
		case PLPGPSM_STMT_CASE:
			rc = exec_stmt_case(estate, (PLpgPSM_stmt_case *) stmt);
			break;
			
		case PLPGPSM_STMT_PREPARE:
			rc = exec_stmt_prepare(estate, (PLpgPSM_stmt_prepare *) stmt);
			break;
			
		default:
			estate->err_stmt = save_estmt;
			elog(ERROR, "unrecognized cmdtype: %d", stmt->cmd_type);
	}

	/* Let the plugin know that we have finished executing this statement */
	if (*plugin_ptr && (*plugin_ptr)->stmt_end)
		((*plugin_ptr)->stmt_end) (estate, stmt);

	estate->err_stmt = save_estmt;

	if (warning_action)
	{
		PLpgPSM_stmt_block *old_block = current_block;
		PLpgPSM_stmt *action = warning_action;
		PLpgPSM_diag_area *old;
		
		current_block = NULL;
		
		set_sqlstate(estate);

		warning_action = NULL;

		old = push_diagnostics_warning(estate, wdata);

		exec_stmt(estate, action);

		estate->current_diag = pop_and_clean_diagnostics(old);

		clean_warning_data(estate);
		
		/* handled warning */		
		estate->sqlerrcode = SUCCESSFULL_COMPLETION;
		estate->current_diag->sqlstate = SUCCESSFULL_COMPLETION;
		set_sqlstate(estate);
		
		current_block = old_block;
		
		switch (estate->htype)
		{
			case PLPGPSM_HDL_EXIT:
				rc = PLPGPSM_RC_EXIT;
				break;
				
			case PLPGPSM_HDL_UNDO:
				rc = PLPGPSM_RC_UNDO;
				break;
		}
	}
	else 
		/* 
		 * some statements cannot change SQLSTATE,
		 * because is more consistent safe origin 
		 * SQLSTATE.
		 */
		switch (stmt->cmd_type)
		{
			case PLPGPSM_STMT_BLOCK:
			case PLPGPSM_STMT_PRINT:
			case PLPGPSM_STMT_LEAVE:
			case PLPGPSM_STMT_RETURN:
			case PLPGPSM_STMT_LOOP:
			case PLPGPSM_STMT_IF:
				/* litle bit nonsens */
				estate->sqlerrcode = old_sqlerrcode;
				break;
			
			default:
				set_sqlstate(estate);
				break;
		}

		/* diag statement cannot raise warning */
		RESET_DIAGNOSTICS_AFTER(estate, stmt);
			
	return rc;
}


static void
set_sqlstate(PLpgPSM_execstate *estate)
{
	PLpgPSM_var *sqlstate_var;

	if (last_sqlstate != SUCCESSFULL_COMPLETION && last_sqlstate != estate->sqlerrcode)
	{
		estate->sqlerrcode = last_sqlstate;
		last_sqlstate = SUCCESSFULL_COMPLETION;
	}

	if (estate->sqlstate_varno != PLPGPSM_INVALID_VARNO)
	{
		sqlstate_var = (PLpgPSM_var *)
				estate->datums[estate->sqlstate_varno];
				
		free_var(sqlstate_var);

		sqlstate_var->value = DirectFunctionCall1(textin,
			    CStringGetDatum(unpack_sql_state(estate->sqlerrcode)));
				    
		sqlstate_var->freeval = true;
		sqlstate_var->isnull = false;	
	}
}




/* ----------
 * exec_stmt_assign			Evaluate an expression and
 *					put the result into a variable.
 * ----------
 */
static int
exec_stmt_assign(PLpgPSM_execstate *estate, PLpgPSM_stmt_assign *stmt)
{
	estate->outer_stmt = false;

	if (stmt->varname)
	{
		Datum value;
		bool isnull;
		Oid restype;
	
		value = exec_eval_expr(estate, stmt->expr, &isnull, &restype);
		
		if (isnull)
			ereport(ERROR,
				(errcode(ERRCODE_NULL_VALUE_NOT_ALLOWED),
				 errmsg("system variable are NOT NULL")));
		
		
		value = exec_simple_cast_value(value, restype, TEXTOID, -1, isnull);
		
		DirectFunctionCall3(set_config_by_name,
					DirectFunctionCall1(textin, CStringGetDatum(stmt->varname)),
					value,
					BoolGetDatum(false));
	}
	else
	{
		ListCell *l;
		
		foreach(l, stmt->items)
		{
			PLpgPSM_assign_item *item = (PLpgPSM_assign_item *) lfirst(l);
			Assert(item->varno >= 0);
			exec_assign_expr(estate, estate->datums[item->varno], item->expr);
		}	
	}

	return PLPGPSM_RC_OK;
}


static int 
exec_stmt_prepare(PLpgPSM_execstate *estate,
				PLpgPSM_stmt_prepare *stmt)
{
	Datum	query;
	bool	isnull = false;
	Oid			restype;
	char	*querystr;
	void *plan;
	MemoryContext	oldctx;
	    
	PLpgPSM_prepared_plan   *prepplan;

	estate->outer_stmt = false;
	query = exec_eval_expr(estate, stmt->expr, &isnull, &restype);
	    
	if (isnull)
	        ereport(ERROR,
			(errcode(ERRCODE_NULL_VALUE_NOT_ALLOWED),
			 errmsg("cannot PREPARE a null querystring")));
				 
	querystr = convert_value_to_string(query, restype);
	    
	exec_eval_cleanup(estate);
	    	    
	plan = SPI_prepare(querystr, stmt->nparams, stmt->params);
	    
	if (!plan)
	{
	        switch (SPI_result)
		{
		        case SPI_ERROR_COPY:
				    ereport(ERROR,
						(errcode(ERRCODE_FEATURE_NOT_SUPPORTED),
						 errmsg("cannot COPY to/from client in PL/pgSQL")));
			case SPI_ERROR_TRANSACTION:
				    ereport(ERROR,
						(errcode(ERRCODE_FEATURE_NOT_SUPPORTED),
						 errmsg("cannot begin/end transactions in PL/pgSQL"),
						 errhint("Use a BEGIN block with an EXCEPTION clause instead.")));
			default:
					elog(ERROR, "SPI_prepare failed for \"%s\": %s",
						querystr, SPI_result_code_string(SPI_result));
		}		    
	}	    
	    
	oldctx = MemoryContextSwitchTo(TopMemoryContext);
	    
	prepplan = palloc0(sizeof(PLpgPSM_prepared_plan) + sizeof(Oid) * stmt->nparams - sizeof(Oid));
	prepplan->query = pstrdup(querystr);

	MemoryContextSwitchTo(oldctx);
	prepplan->plan  = SPI_saveplan(plan);
	prepplan->nparams = stmt->nparams;
	memcpy(prepplan->params, stmt->params, sizeof(Oid)*stmt->nparams);
	    
	store_plan(stmt->name, prepplan);
	    
	SPI_freeplan(plan);
	pfree(querystr);
	
	return PLPGPSM_RC_OK;
}



/* ----------
 * exec_stmt_perform		Evaluate query and discard result (but set
 *							FOUND depending on whether at least one row
 *							was returned).
 * ----------
 */
static int
exec_stmt_call(PLpgPSM_execstate *estate, PLpgPSM_stmt_call *stmt)
{
	PLpgPSM_expr *expr = stmt->expr;

	estate->outer_stmt = false;
	(void) exec_run_select(estate, expr, 0, NULL, NULL);
	exec_set_found(estate, (estate->current_diag->eval_processed != 0));
	exec_eval_cleanup(estate);

	return PLPGPSM_RC_OK;
}

/* ----------
 * exec_stmt_getdiag					Put internal PG information into
 *										specified variables.
 * ----------
 */
static int
exec_stmt_getdiag(PLpgPSM_execstate *estate, PLpgPSM_stmt_getdiag *stmt)
{
	ListCell   *lc;
	int  deep;
	PLpgPSM_diag_area 	*current_diag = estate->current_diag;

	if (stmt->stacked)
	{
		if (stmt->deep_varno != PLPGPSM_INVALID_VARNO)
		{
			Datum value;
			bool isnull;
			Oid  paramtypeid;

			exec_eval_datum(estate, estate->datums[stmt->deep_varno], INT4OID,
						&paramtypeid, &value, &isnull);

			if (isnull)
				ereport(ERROR,
					(errcode(ERRCODE_NULL_VALUE_NOT_ALLOWED),
					errmsg("level of condition cannot be NULL")));

			deep = DatumGetInt32(value);
		}
		else
			deep = stmt->deep;

             if (deep < 1 || deep > 255)                               
                        ereport(ERROR,                                                  
                                    (errcode(ERRCODE_NUMERIC_VALUE_OUT_OF_RANGE),   
                                     errmsg("exception's deep is out of range"),    
                                     errhint("check deep is between 1 and 255")));

	    while (deep--)
	    {
		    current_diag = current_diag->prev;
		    if (!current_diag)
			    ereport(ERROR,
					(errmsg("wrong deep")));
	    }
	}

	estate->outer_stmt = false;
	foreach(lc, stmt->diag_items)
	{
		PLpgPSM_diag_item *diag_item = (PLpgPSM_diag_item *) lfirst(lc);
		PLpgPSM_datum *var;
		bool		isnull = false;
		char	*message;

		if (diag_item->target <= 0)
			continue;

		var = estate->datums[diag_item->target];

		if (var == NULL)
			continue;

		switch (diag_item->kind)
		{
			case PLPGPSM_GETDIAG_ROW_COUNT:

				exec_assign_value(estate, var,
								  UInt32GetDatum(current_diag->eval_processed),
								  INT4OID, &isnull);
				break;

			case PLPGPSM_GETDIAG_MESSAGE_LENGTH:

				exec_assign_value(estate, var,
								  UInt32GetDatum(current_diag->message_length),
								  INT4OID, &isnull);
				break;

			case PLPGPSM_GETDIAG_RESULT_OID:

				exec_assign_value(estate, var,
								  ObjectIdGetDatum(current_diag->eval_lastoid),
								  OIDOID, &isnull);
				break;

			case PLPGPSM_GETDIAG_RETURNED_SQLSTATE:
				exec_assign_value(estate, var,
								  DirectFunctionCall1(textin,
									    CStringGetDatum(unpack_sql_state(current_diag->sqlstate))),
								  TEXTOID, &isnull);
				break;

			case PLPGPSM_GETDIAG_MESSAGE:
			case PLPGPSM_GETDIAG_DETAIL:
			case PLPGPSM_GETDIAG_HINT:
			case PLPGPSM_GETDIAG_CONDITION:
				message = current_diag->fields[diag_item->kind];
				exec_assign_value(estate, var,
								  DirectFunctionCall1(textin,
									    CStringGetDatum(message ? message : "")),
								  TEXTOID, &isnull);
				break;
		    
			default:
				elog(ERROR, "unrecognized attribute request: %d",
					 diag_item->kind);
		}
	}

	return PLPGPSM_RC_OK;
}

/* ----------
 * exec_stmt_if				Evaluate a bool expression and
 *					execute the true or false body
 *					conditionally.
 * ----------
 */
static int
exec_stmt_if(PLpgPSM_execstate *estate, PLpgPSM_stmt_if *stmt)
{
	bool		value;
	bool		isnull;

	estate->outer_stmt = false;
	value = exec_eval_boolean(estate, stmt->cond, &isnull);
	exec_eval_cleanup(estate);

	if (!isnull && value)
	{
		if (stmt->true_body != NIL)
			return exec_stmts(estate, stmt->true_body, NULL);
	}
	else
	{
		if (stmt->false_body != NIL)
			return exec_stmts(estate, stmt->false_body, NULL);
	}

	return PLPGPSM_RC_OK;
}

/* ----------
 * case_stmt_case			
 * ----------
 */
static int
exec_stmt_case(PLpgPSM_execstate *estate, PLpgPSM_stmt_case *stmt)
{
	PLpgPSM_var *t_var = NULL;
	bool		isnull;
	ListCell   *l;

	if (stmt->t_expr != NULL)
	{
		/* simple case */
		Datum	t_val;
		Oid		t_oid;

		t_val = exec_eval_expr(estate, stmt->t_expr, &isnull, &t_oid);

		t_var = (PLpgPSM_var *) estate->datums[stmt->t_varno];

		/*
		 * When expected datatype is different from real, change it.
		 * Note that what we're modifying here is an execution copy
		 * of the datum, so this doesn't affect the originally stored
		 * function parse tree.
		 */
		if (t_var->datatype->typoid != t_oid)
			t_var->datatype = plpgpsm_build_datatype(t_oid, -1);

		/* now we can assign to the variable */
		exec_assign_value(estate,
						  (PLpgPSM_datum *) t_var,
						  t_val,
						  t_oid,
						  &isnull);

		exec_eval_cleanup(estate);
	}

	/* Now search for a successful WHEN clause */
	foreach(l, stmt->case_when_list)
	{
		PLpgPSM_case_when *cwt = (PLpgPSM_case_when *) lfirst(l);
		bool	value;

		value = exec_eval_boolean(estate, cwt->expr, &isnull);
		exec_eval_cleanup(estate);
		if (!isnull && value)
		{
			/* Found it */

			/* We can now discard any value we had for the temp variable */
			if (t_var != NULL)
			{
				free_var(t_var);
				t_var->value = (Datum) 0;
				t_var->isnull = true;
			}

			/* Evaluate the statement(s), and we're done */
			return exec_stmts(estate, cwt->stmts, NULL);
		}
	}

	/* We can now discard any value we had for the temp variable */
	if (t_var != NULL)
	{
		free_var(t_var);
		t_var->value = (Datum) 0;
		t_var->isnull = true;
	}

	/* SQL2003 mandates this error if there was no ELSE clause */
	if (!stmt->have_else)
		ereport(ERROR,
				(errcode(ERRCODE_CASE_NOT_FOUND),
				 errmsg("case not found"),
				 errhint("CASE statement is missing ELSE part.")));

	/* Evaluate the ELSE statements, and we're done */
	return exec_stmts(estate, stmt->else_stmts, NULL);
}


/* ----------
 * exec_stmt_loop		
 *				
 *				
 * ----------
 */
static int
exec_stmt_loop(PLpgPSM_execstate *estate, PLpgPSM_stmt_loop *stmt)
{

	estate->outer_stmt = false;
	for (;;)
	{
		int			rc;
		bool		value;
		bool		isnull;

		/* eval condition before body */
		if (stmt->ctype == PLPGPSM_CYCLE_WHILE)
		{
			value = exec_eval_boolean(estate, stmt->cond, &isnull);
			exec_eval_cleanup(estate);

			if (isnull || !value)
				break;
		}

		rc = exec_stmts(estate, stmt->body, NULL);

		switch (rc)
		{
			case PLPGPSM_RC_OK:
				break;

			case PLPGPSM_RC_EXIT:
				if (stmt->label == NULL)
					return PLPGPSM_RC_EXIT;
				if (strcmp(stmt->label, estate->exitlabel))
					return PLPGPSM_RC_EXIT;
				/* found good cycle */
				estate->exitlabel = NULL;
				return PLPGPSM_RC_OK;

			case PLPGPSM_RC_CONTINUE:
				if (stmt->label != NULL &&
						 strcmp(stmt->label, estate->exitlabel) == 0)
					/* label matches named continue, so re-run loop */
					estate->exitlabel = NULL;
				else
					/* label doesn't match named continue, propagate upward */
					return PLPGPSM_RC_CONTINUE;
				break;

			case PLPGPSM_RC_RETURN:
				return PLPGPSM_RC_RETURN;

			default:
				elog(ERROR, "unrecognized rc: %d", rc);
		}
		
		
		/* eval codition after body */
		if (stmt->ctype == PLPGPSM_CYCLE_REPEAT && rc != PLPGPSM_RC_CONTINUE)
		{
			value = exec_eval_boolean(estate, stmt->cond, &isnull);
			exec_eval_cleanup(estate);

			if (!isnull && value)
				break;
		}
	}

	return PLPGPSM_RC_OK;
}



/* ----------
 * exec_stmt_fors			Execute a query, assign each
 *					tuple to a record or row and
 *					execute a group of statements
 *					for it.
 * ----------
 */
static int
exec_stmt_fors(PLpgPSM_execstate *estate, PLpgPSM_stmt_fors *stmt)
{
	PLpgPSM_rec *rec = NULL;
	PLpgPSM_row *row = NULL;
	SPITupleTable *tuptab;
	Portal		portal;
	bool		found = false;
	int			rc = PLPGPSM_RC_OK;
	int			i;
	int			n;

	estate->outer_stmt = false;
	row = (PLpgPSM_row *) (estate->datums[stmt->row->dno]);

	/*
	 * Open the implicit cursor for the statement and fetch the initial 10
	 * rows.
	 */

	exec_run_select(estate, stmt->query, 0, &portal, stmt->cursor_name);

	SPI_cursor_fetch(portal, true, 10);
	tuptab = SPI_tuptable;
	n = SPI_processed;

	/*
	 * If the query didn't return any rows set  FOUND = false.
	 */
	found = (n != 0);

	/*
	 * Now do the loop
	 */
	while (n > 0)
	{
		for (i = 0; i < n; i++)
		{
			/*
			 * Assign the tuple to the target
			 */
			exec_move_row(estate, rec, row, tuptab->vals[i], tuptab->tupdesc);

			/*
			 * Execute the statements
			 */
			rc = exec_stmts(estate, stmt->body, NULL);
			if (rc != PLPGPSM_RC_OK)
			{
				if (rc == PLPGPSM_RC_EXIT)
				{
					if (estate->exitlabel == NULL)
						/* unlabelled exit, finish the current loop */
						rc = PLPGPSM_RC_OK;
					else if (stmt->label != NULL &&
							 strcmp(stmt->label, estate->exitlabel) == 0)
					{
						/* labelled exit, matches the current stmt's label */
						estate->exitlabel = NULL;
						rc = PLPGPSM_RC_OK;
					}

					/*
					 * otherwise, we processed a labelled exit that does not
					 * match the current statement's label, if any: return
					 * RC_EXIT so that the EXIT continues to recurse upward.
					 */
				}
				else if (rc == PLPGPSM_RC_CONTINUE)
				{
					if (estate->exitlabel == NULL)
					{
						/* anonymous continue, so re-run the current loop */
						rc = PLPGPSM_RC_OK;
						continue;
					}
					else if (stmt->label != NULL &&
							 strcmp(stmt->label, estate->exitlabel) == 0)
					{
						/* label matches named continue, so re-run loop */
						rc = PLPGPSM_RC_OK;
						estate->exitlabel = NULL;
						continue;
					}

					/*
					 * otherwise, we processed a named continue that does not
					 * match the current statement's label, if any: return
					 * RC_CONTINUE so that the CONTINUE will propagate up the
					 * stack.
					 */
				}

				/*
				 * We're aborting the loop, so cleanup and set FOUND. (This
				 * code should match the code after the loop.)
				 */
				SPI_freetuptable(tuptab);
				SPI_cursor_close(portal);
				exec_set_found(estate, found);
				exec_signal_not_found(!found);

				return rc;
			}
		}

		SPI_freetuptable(tuptab);

		/*
		 * Fetch the next 50 tuples
		 */
		SPI_cursor_fetch(portal, true, 50);
		n = SPI_processed;
		tuptab = SPI_tuptable;
	}

	/*
	 * Release last group of tuples
	 */
	SPI_freetuptable(tuptab);

	/*
	 * Close the implicit cursor
	 */
	SPI_cursor_close(portal);

	/*
	 * Set the FOUND variable to indicate the result of executing the loop
	 * (namely, whether we looped one or more times). This must be set here so
	 * that it does not interfere with the value of the FOUND variable inside
	 * the loop processing itself.
	 */

	exec_set_found(estate, found);
	exec_signal_not_found(!found);

	return rc;
}


/* ----------
 * exec_stmt_leave			Implements LEAVE and ITERATE
 *
 * This begins the process of exiting / restarting a loop.
 * ----------
 */
static int
exec_stmt_leave(PLpgPSM_execstate *estate, PLpgPSM_stmt_leave *stmt)
{
	/*
	 * If the exit / continue has a condition, evaluate it
	 */
	estate->outer_stmt = false;
	estate->exitlabel = stmt->label;
	if (stmt->is_leave)
		return PLPGPSM_RC_EXIT;
	else
		return PLPGPSM_RC_CONTINUE;
}


/* ----------
 * exec_stmt_return			Evaluate an expression and start
 *					returning from the function.
 * ----------
 */
static int
exec_stmt_return(PLpgPSM_execstate *estate, PLpgPSM_stmt_return *stmt)
{

	estate->outer_stmt = false;

	/*
	 * If processing a set-returning PL/PgSQL function, the final RETURN
	 * indicates that the function is finished producing tuples.  The rest of
	 * the work will be done at the top level.
	 */
	if (estate->retisset)
	{
		if (stmt->expr != NULL)
		{
			Portal 	portal;

			exec_run_select(estate, stmt->expr, 0, &portal, NULL);
			forward_set(estate, portal);
		}
		
		return PLPGPSM_RC_RETURN;
	}

	/* initialize for null result (possibly a tuple) */
	estate->retval = (Datum) 0;
	estate->rettupdesc = NULL;
	estate->retisnull = true;

	if (stmt->retvarno >= 0)
	{
		PLpgPSM_datum *retvar = estate->datums[stmt->retvarno];

		switch (retvar->dtype)
		{
			case PLPGPSM_DTYPE_VAR:
				{
					PLpgPSM_var *var = (PLpgPSM_var *) retvar;

					estate->retval = var->value;
					estate->retisnull = var->isnull;
					estate->rettype = var->datatype->typoid;
				}
				break;

			case PLPGPSM_DTYPE_REC:
				{
					PLpgPSM_rec *rec = (PLpgPSM_rec *) retvar;

					if (HeapTupleIsValid(rec->tup))
					{
						estate->retval = (Datum) rec->tup;
						estate->rettupdesc = rec->tupdesc;
						estate->retisnull = false;
					}
				}
				break;

			case PLPGPSM_DTYPE_ROW:
				{
					PLpgPSM_row *row = (PLpgPSM_row *) retvar;

					Assert(row->rowtupdesc);
					estate->retval = (Datum) make_tuple_from_row(estate, row,
															row->rowtupdesc);
					if (estate->retval == (Datum) NULL) /* should not happen */
						elog(ERROR, "row not compatible with its own tupdesc");
					estate->rettupdesc = row->rowtupdesc;
					estate->retisnull = false;
				}
				break;

			default:
				elog(ERROR, "unrecognized dtype: %d", retvar->dtype);
		}

		return PLPGPSM_RC_RETURN;
	}

	if (stmt->expr != NULL)
	{
		if (estate->retistuple)
		{
			exec_run_select(estate, stmt->expr, 1, NULL, NULL);
			if (estate->current_diag->eval_processed > 0)
			{
				estate->retval = (Datum) estate->eval_tuptable->vals[0];
				estate->rettupdesc = estate->eval_tuptable->tupdesc;
				estate->retisnull = false;
			}
		}
		else
		{
			/* Normal case for scalar results */
			estate->retval = exec_eval_expr(estate, stmt->expr,
											&(estate->retisnull),
											&(estate->rettype));
		}

		return PLPGPSM_RC_RETURN;
	}

	/*
	 * Special hack for function returning VOID: instead of NULL, return a
	 * non-null VOID value.  This is of dubious importance but is kept for
	 * backwards compatibility.  Note that the only other way to get here is
	 * to have written "RETURN NULL" in a function returning tuple.
	 */
	if (estate->fn_rettype == VOIDOID)
	{
		estate->retval = (Datum) 0;
		estate->retisnull = false;
		estate->rettype = VOIDOID;
	}

	return PLPGPSM_RC_RETURN;
}

static void 
forward_set(PLpgPSM_execstate *estate, Portal portal)
{
	SPITupleTable *tuptab;
	int		n;
	int		i;
	TupleDesc	tupdesc;
	int			natts;
	HeapTuple	tuple;

	if (estate->tuple_store)
		elog(ERROR, "You cannot mix return table and return next statement");

	exec_init_tuple_store(estate);

	tupdesc = estate->rettupdesc;
	natts = tupdesc->natts;

	SPI_cursor_fetch(portal, true, 10);
	n = SPI_processed;
	tuptab = SPI_tuptable;

	if (!compatible_tupdesc(tupdesc, tuptab->tupdesc))
		ereport(ERROR,
				(errcode(ERRCODE_DATATYPE_MISMATCH),
				errmsg("wrong TABLE expression in RETURN statement")));

	while (n > 0)
	{
		for (i = 0; i < n; i++)
		{
			tuple = tuptab->vals[i];

			if (HeapTupleIsValid(tuple))
			{
				MemoryContext oldcxt;

				oldcxt = MemoryContextSwitchTo(estate->tuple_store_cxt);
				tuplestore_puttuple(estate->tuple_store, tuple);
				MemoryContextSwitchTo(oldcxt);
			}
		}

		SPI_freetuptable(tuptab);

		/*
		 * Fetch the next 50 tuples
		 */
		SPI_cursor_fetch(portal, true, 50);
		n = SPI_processed;
		tuptab = SPI_tuptable;
	}

	SPI_freetuptable(tuptab);
	SPI_cursor_close(portal);
}


static void
exec_init_tuple_store(PLpgPSM_execstate *estate)
{
	ReturnSetInfo *rsi = estate->rsi;
	MemoryContext oldcxt;

	/*
	 * Check caller can handle a set result in the way we want
	 */
	if (!rsi || !IsA(rsi, ReturnSetInfo) ||
		(rsi->allowedModes & SFRM_Materialize) == 0 ||
		rsi->expectedDesc == NULL)
		ereport(ERROR,
				(errcode(ERRCODE_FEATURE_NOT_SUPPORTED),
				 errmsg("set-valued function called in context that cannot accept a set")));

	estate->tuple_store_cxt = rsi->econtext->ecxt_per_query_memory;

	oldcxt = MemoryContextSwitchTo(estate->tuple_store_cxt);
	estate->tuple_store = tuplestore_begin_heap(true, false, work_mem);
	MemoryContextSwitchTo(oldcxt);

	estate->rettupdesc = rsi->expectedDesc;
}

/* ----------
 * exec_stmt_signal			Build a message and throw it with elog()
 * ----------
 */
#define sec_free(m) if (m) pfree(m)

static int
exec_stmt_signal(PLpgPSM_execstate *estate, PLpgPSM_stmt_signal *stmt)
{
	Datum value;
	bool paramisnull;
	Oid	paramtypeid;
	
	char	*message = NULL;
	char	*detail = NULL;
	char	*hint = NULL;

	int	sqlstate = 0;
	int	elevel = 0;
	
	estate->outer_stmt = false;	
	estate->err_text = raise_skip_msg;	/* suppress traceback of raise */

	if (stmt->is_resignal)
	{
		PLpgPSM_diag_area *stacked;

		if (!estate->current_diag->prev)
			ereport(ERROR,
					(errcode(MAKE_SQLSTATE('O','K','0','0','0')),
					 errmsg("resignal when handler not active")));

		stacked = estate->current_diag->prev;

		message = stacked->fields[PLPGPSM_GETDIAG_MESSAGE];
		hint = stacked->fields[PLPGPSM_GETDIAG_HINT];
		detail = stacked->fields[PLPGPSM_GETDIAG_DETAIL];

		sqlstate = stacked->sqlstate;
		elevel = stacked->elevel;
	}

	if (stmt->message)
	{
		value = exec_eval_expr(estate, stmt->message,
							    &paramisnull,
							    &paramtypeid);
		if (!paramisnull)
			message = convert_value_to_string(value, paramtypeid);
		exec_eval_cleanup(estate);
	}
	
	if (stmt->hint)
	{
		value = exec_eval_expr(estate, stmt->hint,
							    &paramisnull,
							    &paramtypeid);
		if (!paramisnull)
			hint = convert_value_to_string(value, paramtypeid);
		exec_eval_cleanup(estate);
	}
	
	if (stmt->detail)
	{
		value = exec_eval_expr(estate, stmt->detail,
							    &paramisnull,
							    &paramtypeid);
		if (!paramisnull)
			detail = convert_value_to_string(value, paramtypeid);
		exec_eval_cleanup(estate);
	}

	if (stmt->sqlerrstate)
	{
		sqlstate = stmt->sqlerrstate;
		elevel = stmt->elog_level;
	}

	if (!message)
		message = stmt->condname;

	if (elevel == INFO || elevel == WARNING)
	{
		if (wdata)
		{
			sec_free(wdata->message);
			sec_free(wdata->detail);
			sec_free(wdata->hint);
			sec_free(wdata->context);

			pfree(wdata);
			wdata = NULL;
		}

		wdata = plpgpsm_warning_filter(stmt->sqlerrstate, message, detail, hint, NULL);
	}
	else
	{

		ereport(elevel,
			    (errcode(sqlstate),
			     errmsg_internal("%s", message),
			     (detail != NULL) ? errdetail("%s", detail) : 0,
			     (hint != NULL) ? errhint("%s", hint) :0));
	}

	estate->err_text = NULL;	/* un-suppress... */

	return PLPGPSM_RC_OK;
}


/* ----------
 * exec_stmt_print			Build a message and throw it with elog()
 * ----------
 */
static int
exec_stmt_print(PLpgPSM_execstate *estate, PLpgPSM_stmt_print *stmt)
{
	PLpgPSM_dstring ds;
	bool	first = true;
	
	PLpgPSM_stmt_block *old_block = current_block;

	ListCell   *param;

	estate->outer_stmt = false;

	if (stmt->params == NULL)
		ereport(ERROR,
        			(errcode(ERRCODE_SYNTAX_ERROR),
				 errmsg("too few parameters specified for PRINT")));
	
	plpgpsm_dstring_init(&ds);
	
	foreach(param, stmt->params)
	{
		Oid			paramtypeid;
		Datum		paramvalue;
		bool		paramisnull;
		char	   *extval;
	
		paramvalue = exec_eval_expr(estate,
						  (PLpgPSM_expr *) lfirst(param),
									&paramisnull,
									&paramtypeid);
		if (paramisnull)
			extval = "<NULL>";
		else
			extval = convert_value_to_string(paramvalue, paramtypeid);
			
		if (!first)	
		{
			plpgpsm_dstring_append_char(&ds, ' ');
			plpgpsm_dstring_append(&ds, extval);
		}
		else
		{
			plpgpsm_dstring_append(&ds, extval);
			first = false;
		}

		exec_eval_cleanup(estate);
	}


	/*
	 * Throw the error (may or may not come back)
	 */
	estate->err_text = raise_skip_msg;	/* suppress traceback of raise */
	
	current_block = NULL;	
	
    	ereport(NOTICE,
		 ( 0,  errmsg_internal("%s", plpgpsm_dstring_get(&ds))));
		 
	current_block = old_block;

	estate->err_text = NULL;	/* un-suppress... */

	plpgpsm_dstring_free(&ds);

	return PLPGPSM_RC_OK;
}


/* ----------
 * Initialize a mostly empty execution state
 * ----------
 */
static void
plpgpsm_estate_setup(PLpgPSM_execstate *estate,
					 PLpgPSM_function *func,
					 ReturnSetInfo *rsi)
{
	estate->retval = (Datum) 0;
	estate->retisnull = true;
	estate->rettype = InvalidOid;

	estate->fn_rettype = func->fn_rettype;
	estate->retistuple = func->fn_retistuple;
	estate->retisset = func->fn_retset;

	estate->readonly_func = func->fn_readonly;

	estate->rettupdesc = NULL;
	estate->exitlabel = NULL;

	estate->tuple_store = NULL;
	estate->tuple_store_cxt = NULL;
	estate->rsi = rsi;

	estate->trig_nargs = 0;
	estate->trig_argv = NULL;

	estate->found_varno = func->found_varno;
	estate->ndatums = func->ndatums;
	estate->datums = palloc(sizeof(PLpgPSM_datum *) * estate->ndatums);
	/* caller is expected to fill the datums array */

	estate->eval_tuptable = NULL;

	estate->err_func = func;
	estate->err_stmt = NULL;
	estate->err_text = NULL;

	estate->edata = NULL;

        /*                                                                                             
         * Create an EState and ExprContext for evaluation of simple expressions.                      
         */                                                                                            
        plpgpsm_create_econtext(estate);     

	/*
	 * Let the plugin see this function before we initialize any local
	 * PL/pgSQL variables - note that we also give the plugin a few function
	 * pointers so it can call back into PL/pgSQL for doing things like
	 * variable assignments and stack traces
	 */
	if (*plugin_ptr)
	{
		(*plugin_ptr)->error_callback = plpgpsm_exec_error_callback;
		(*plugin_ptr)->assign_expr = exec_assign_expr;

		if ((*plugin_ptr)->func_setup)
			((*plugin_ptr)->func_setup) (estate, func);
	}
	
	/*
	 * SQL/PSM allow automatic closing opened cursors when we leave block.
	 * This list contains all opened cursors. Every block add and remove
	 * own or unclosed nested cursors
	 */
	estate->opened_cursors = NIL;

	estate->current_diag = palloc0(sizeof(PLpgPSM_diag_area));

	estate->current_diag->eval_processed = 0;
	estate->current_diag->eval_lastoid = InvalidOid;

}

/* ----------
 * Release temporary memory used by expression/subselect evaluation
 *
 * NB: the result of the evaluation is no longer valid after this is done,
 * unless it is a pass-by-value datatype.
 * ----------
 */
static void
exec_eval_cleanup(PLpgPSM_execstate *estate)
{
	/* Clear result of a full SPI_execute */
	if (estate->eval_tuptable != NULL)
		SPI_freetuptable(estate->eval_tuptable);
	estate->eval_tuptable = NULL;

	/* Clear result of exec_eval_simple_expr (but keep the econtext) */
	if (estate->eval_econtext != NULL)
		ResetExprContext(estate->eval_econtext);
}


/* ----------
 * Generate a prepared plan
 * ----------
 */
static void
exec_prepare_plan(PLpgPSM_execstate *estate,
				PLpgPSM_expr *expr,
				int cursorOptions)
{
	int			i;
	SPIPlanPtr	plan;
	Oid		   *argtypes;

	estate->outer_stmt = false;

	/*
	 * We need a temporary argtypes array to load with data. (The finished
	 * plan structure will contain a copy of it.)
	 */
	argtypes = (Oid *) palloc(expr->nparams * sizeof(Oid));

	for (i = 0; i < expr->nparams; i++)
	{
		Datum		paramval;
		bool		paramisnull;

		exec_eval_datum(estate, estate->datums[expr->params[i]],
						InvalidOid,
						&argtypes[i], &paramval, &paramisnull);
	}

	/*
	 * Generate and save the plan
	 */
	plan = SPI_prepare_cursor(expr->query, expr->nparams, argtypes,
							cursorOptions);
	if (plan == NULL)
	{
		/* Some SPI errors deserve specific error messages */
		switch (SPI_result)
		{
			case SPI_ERROR_COPY:
				ereport(ERROR,
						(errcode(ERRCODE_FEATURE_NOT_SUPPORTED),
						 errmsg("cannot COPY to/from client in PL/pgSQL")));
			case SPI_ERROR_TRANSACTION:
				ereport(ERROR,
						(errcode(ERRCODE_FEATURE_NOT_SUPPORTED),
						 errmsg("cannot begin/end transactions in PL/pgSQL"),
						 errhint("Use a BEGIN block with an EXCEPTION clause instead.")));
			default:
				elog(ERROR, "SPI_prepare_cursor failed for \"%s\": %s",
					 expr->query, SPI_result_code_string(SPI_result));
		}
	}
	expr->plan = SPI_saveplan(plan);

	SPI_freeplan(plan);
	plan = expr->plan;
	expr->plan_argtypes = plan->argtypes;
	exec_simple_check_plan(expr);

	pfree(argtypes);
}


/* ----------
 * exec_stmt_execsql			Execute an SQL statement (possibly with INTO).
 * ----------
 */
static int
exec_stmt_sql(PLpgPSM_execstate *estate,
				  PLpgPSM_stmt_sql *stmt)
{
	int			i;
	Datum	   *values;
	char	   *nulls;
	long		tcount;
	int			rc;
	PLpgPSM_expr *expr = stmt->sqlstmt;
	bool outer_stmt = estate->outer_stmt;

	estate->outer_stmt = false;

	/*
	 * On the first call for this statement generate the plan, and detect
	 * whether the statement is INSERT/UPDATE/DELETE
	 */
	if (expr->plan == NULL)
	{
		ListCell   *l;

		exec_prepare_plan(estate, expr, 0);
		stmt->mod_stmt = false;

		foreach(l, expr->plan->plancache_list)
		{
			CachedPlanSource *plansource = (CachedPlanSource *) lfirst(l);
			ListCell   *l2;

			foreach(l2, plansource->plan->stmt_list)
			{
				PlannedStmt *p = (PlannedStmt *) lfirst(l2);

				if (IsA(p, PlannedStmt) &&
					p->canSetTag)
				{
					if (p->commandType == CMD_INSERT ||
						p->commandType == CMD_UPDATE ||
						p->commandType == CMD_DELETE)
						stmt->mod_stmt = true;
				}
			}
		}
	}

	/*
	 * Now build up the values and nulls arguments for SPI_execute_plan()
	 */
	values = (Datum *) palloc(expr->nparams * sizeof(Datum));
	nulls = (char *) palloc(expr->nparams * sizeof(char));

	for (i = 0; i < expr->nparams; i++)
	{
		PLpgPSM_datum *datum = estate->datums[expr->params[i]];
		Oid			paramtypeid;
		bool		paramisnull;

		exec_eval_datum(estate, datum, expr->plan_argtypes[i],
						&paramtypeid, &values[i], &paramisnull);
		if (paramisnull)
			nulls[i] = 'n';
		else
			nulls[i] = ' ';
	}

	/*
	 * If we have INTO, then we only need one row back ... but if we have INTO
	 * STRICT, ask for two rows, so that we can verify the statement returns
	 * only one.  INSERT/UPDATE/DELETE are always treated strictly. Without
	 * INTO, just run the statement to completion (tcount = 0).
	 *
	 * We could just ask for two rows always when using INTO, but there are
	 * some cases where demanding the extra row costs significant time, eg by
	 * forcing completion of a sequential scan.  So don't do it unless we need
	 * to enforce strictness.
	 */
	if (stmt->into)
	{
		if (stmt->strict || stmt->mod_stmt)
			tcount = 2;
		else
			tcount = 1;
	}
	else
		tcount = 0;

	/* function's body is only this statement */
	if (outer_stmt)
	{
		if (estate->retisset)
		{
			Portal	portal;

			if (stmt->into)
				ereport(ERROR, 
					(errcode(ERRCODE_SYNTAX_ERROR),
					 errmsg("INTO cannot be used in this context")));

			portal = SPI_cursor_open(NULL, expr->plan, values, nulls,
								   estate->readonly_func);
			if (portal == NULL)
				elog(ERROR, "could not open implicit cursor for query \"%s\": %s",
					 expr->query, SPI_result_code_string(SPI_result));

			forward_set(estate, portal);

			pfree(values);
			pfree(nulls);

			return PLPGPSM_RC_OK;
		}
	}

	/*
	 * Execute the plan
	 */
	rc = SPI_execute_plan(expr->plan, values, nulls,
						  estate->readonly_func, tcount);
	/*
	 * Check for error, and set FOUND if appropriate (for historical reasons
	 * we set FOUND only for certain query types).	Also Assert that we
	 * identified the statement type the same as SPI did.
	 */
	switch (rc)
	{
		case SPI_OK_SELECT:
			Assert(!stmt->mod_stmt);
			exec_set_found(estate, (SPI_processed != 0));
			exec_signal_not_found(SPI_processed == 0);
			break;

		case SPI_OK_INSERT:
		case SPI_OK_UPDATE:
		case SPI_OK_DELETE:
		case SPI_OK_INSERT_RETURNING:
		case SPI_OK_UPDATE_RETURNING:
		case SPI_OK_DELETE_RETURNING:
			Assert(stmt->mod_stmt);
			exec_set_found(estate, (SPI_processed != 0));
			exec_signal_not_found(SPI_processed == 0);
			break;

		case SPI_OK_SELINTO:
		case SPI_OK_UTILITY:
			Assert(!stmt->mod_stmt);
			exec_signal_not_found(SPI_processed == 0);
			break;

		default:
			elog(ERROR, "SPI_execute_plan failed executing query \"%s\": %s",
				 expr->query, SPI_result_code_string(rc));
	}

	/* All variants should save result info for GET DIAGNOSTICS */
	estate->current_diag->eval_processed = SPI_processed;
	estate->current_diag->eval_lastoid = SPI_lastoid;

	/* Process INTO if present */
	if (stmt->into)
	{
		SPITupleTable *tuptab = SPI_tuptable;
		uint32		n = SPI_processed;
		PLpgPSM_rec *rec = NULL;
		PLpgPSM_row *row = NULL;

		/* If the statement did not return a tuple table, complain */
		if (tuptab == NULL)
			ereport(ERROR,
					(errcode(ERRCODE_SYNTAX_ERROR),
				errmsg("INTO used with a command that cannot return data")));

		/* Determine if we assign to a record or a row */
		if (stmt->rec != NULL)
			rec = (PLpgPSM_rec *) (estate->datums[stmt->rec->dno]);
		else if (stmt->row != NULL)
			row = (PLpgPSM_row *) (estate->datums[stmt->row->dno]);
		else
			elog(ERROR, "unsupported target");

		/*
		 * If SELECT ... INTO specified STRICT, and the query didn't find
		 * exactly one row, throw an error.  If STRICT was not specified, then
		 * allow the query to find any number of rows.
		 */
		if (n == 0)
		{
			if (stmt->strict)
				ereport(ERROR,
						(errcode(ERRCODE_NO_DATA_FOUND),
						 errmsg("query returned no rows")));
			/* set the target to NULL(s) */
			exec_move_row(estate, rec, row, NULL, tuptab->tupdesc);
		}
		else
		{
			if (n > 1 && (stmt->strict || stmt->mod_stmt))
				ereport(ERROR,
						(errcode(ERRCODE_TOO_MANY_ROWS),
						 errmsg("query returned more than one row")));
			/* Put the first result row into the target */
			exec_move_row(estate, rec, row, tuptab->vals[0], tuptab->tupdesc);
		}

		/* Clean up */
		SPI_freetuptable(SPI_tuptable);
	}
	else
	{
		/* If the statement returned a tuple table, complain */
		if (SPI_tuptable != NULL)
			ereport(ERROR,
					(errcode(ERRCODE_SYNTAX_ERROR),
					 errmsg("query has no destination for result data %s", expr->query),
					 (rc == SPI_OK_SELECT) ? errhint("If you want to discard the results of a SELECT, use PERFORM instead.") : 0));
	}

	pfree(values);
	pfree(nulls);

	return PLPGPSM_RC_OK;
}


/* ----------
 * exec_stmt_execute			Execute a dynamic SQL query (execute immediate)
 *					(possibly with INTO).
 * ----------
 */
static int
exec_stmt_execute(PLpgPSM_execstate *estate,
					 PLpgPSM_stmt_execute *stmt)
{
	Datum		query;
	bool		isnull = false;
	Oid			restype;
	char	   *querystr;
	int			exec_res;
	Datum 	*values = NULL;
	char	*nulls = NULL;
	int	tcount;

	estate->outer_stmt = false;

	if (stmt->name)
	{
		int i;
		
		PLpgPSM_prepared_plan *preplan = get_plan(stmt->name);
		
		if (preplan == NULL)
			ereport(ERROR,
					(errmsg("cannot find named prepared plan")));
					
		querystr = pstrdup(preplan->query);
		
		if (preplan->nparams > 0)
		{
			/* have to exists USING */
			if (!stmt->query)
				ereport(ERROR,
					(errmsg("missing USING clause")));
					
			if (stmt->query->nparams != preplan->nparams)
				ereport(ERROR,
					(errmsg("wrong number of arguments in USING clause")));
		
		
			/* args are in query->nparams and query->params */
		
			values = (Datum *) palloc0(preplan->nparams * sizeof(Datum));
			nulls = (char *) palloc0(preplan->nparams * sizeof(char));
		
			for (i = 0; i < preplan->nparams; i++)
			{
				PLpgPSM_datum *	datum = estate->datums[stmt->query->params[i]];
				Oid		paramtypeid;
				bool		paramisnull;
			
				exec_eval_datum(estate, datum, InvalidOid,
							    &paramtypeid, &values[i], &paramisnull);
							    
				values[i] = exec_simple_cast_value(values[i],
								paramtypeid,
								preplan->params[i],
								-1,
								paramisnull);
								
							    
				if (paramisnull)
					nulls[i] = 'n';
				else
					nulls[i] = ' ';
			}
		}

		if (stmt->into)
			tcount	= 1;
		else
			tcount	= 0;
					
		exec_res = SPI_execute_plan(preplan->plan, values, nulls, 
						estate->readonly_func, tcount);
			
	}
	else
	{
		/*
		 * First we evaluate the string expression after the EXECUTE keyword. Its
		 * result is the querystring we have to execute.
		 */
	 
		query = exec_eval_expr(estate, stmt->query, &isnull, &restype);
		if (isnull)
			ereport(ERROR,
				(errcode(ERRCODE_NULL_VALUE_NOT_ALLOWED),
				 errmsg("cannot EXECUTE a null querystring")));

		/* Get the C-String representation */
		querystr = convert_value_to_string(query, restype);

		exec_eval_cleanup(estate);

		/*
		 * Call SPI_execute() without preparing a saved plan.
		 */
		exec_res = SPI_execute(querystr, estate->readonly_func, 0);
	}

	switch (exec_res)
	{
		case SPI_OK_SELECT:
		case SPI_OK_INSERT:
		case SPI_OK_UPDATE:
		case SPI_OK_DELETE:
		case SPI_OK_INSERT_RETURNING:
		case SPI_OK_UPDATE_RETURNING:
		case SPI_OK_DELETE_RETURNING:
		case SPI_OK_UTILITY:
			exec_signal_not_found(SPI_processed == 0);
			break;

		case 0:

			/*
			 * Also allow a zero return, which implies the querystring
			 * contained no commands.
			 */
			exec_signal_not_found(SPI_processed == 0);
			break;

		case SPI_OK_SELINTO:

			/*
			 * We want to disallow SELECT INTO for now, because its behavior
			 * is not consistent with SELECT INTO in a normal plpgpsm context.
			 * (We need to reimplement EXECUTE to parse the string as a
			 * plpgpsm command, not just feed it to SPI_execute.) However,
			 * CREATE AS should be allowed ... and since it produces the same
			 * parsetree as SELECT INTO, there's no way to tell the difference
			 * except to look at the source text.  Wotta kluge!
			 */
			{
				char	   *ptr;

				for (ptr = querystr; *ptr; ptr++)
					if (!scanner_isspace(*ptr))
						break;
				if (*ptr == 'S' || *ptr == 's')
					ereport(ERROR,
							(errcode(ERRCODE_FEATURE_NOT_SUPPORTED),
							 errmsg("EXECUTE of SELECT ... INTO is not implemented yet")));
				break;
			}

			/* Some SPI errors deserve specific error messages */
		case SPI_ERROR_COPY:
			ereport(ERROR,
					(errcode(ERRCODE_FEATURE_NOT_SUPPORTED),
					 errmsg("cannot COPY to/from client in PL/pgSQL")));
		case SPI_ERROR_CURSOR:
			ereport(ERROR,
					(errcode(ERRCODE_FEATURE_NOT_SUPPORTED),
					 errmsg("cannot manipulate cursors directly in PL/pgSQL"),
					 errhint("Use PL/pgSQL's cursor features instead.")));
		case SPI_ERROR_TRANSACTION:
			ereport(ERROR,
					(errcode(ERRCODE_FEATURE_NOT_SUPPORTED),
					 errmsg("cannot begin/end transactions in PL/pgSQL"),
			errhint("Use a BEGIN block with an EXCEPTION clause instead.")));

		default:
			elog(ERROR, "SPI_execute failed executing query \"%s\": %s",
				 querystr, SPI_result_code_string(exec_res));
			break;
	}

	/* Save result info for GET DIAGNOSTICS */
	estate->current_diag->eval_processed = SPI_processed;
	estate->current_diag->eval_lastoid = SPI_lastoid;

	/* Process INTO if present */
	if (stmt->into)
	{
		SPITupleTable *tuptab = SPI_tuptable;
		uint32		n = SPI_processed;
		PLpgPSM_rec *rec = NULL;
		PLpgPSM_row *row = NULL;

		/* If the statement did not return a tuple table, complain */
		if (tuptab == NULL)
			ereport(ERROR,
					(errcode(ERRCODE_SYNTAX_ERROR),
				errmsg("INTO used with a command that cannot return data")));

		/* Determine if we assign to a record or a row */
		if (stmt->rec != NULL)
			rec = (PLpgPSM_rec *) (estate->datums[stmt->rec->dno]);
		else if (stmt->row != NULL)
			row = (PLpgPSM_row *) (estate->datums[stmt->row->dno]);
		else
			elog(ERROR, "unsupported target");

		/*
		 * If SELECT ... INTO specified STRICT, and the query didn't find
		 * exactly one row, throw an error.  If STRICT was not specified, then
		 * allow the query to find any number of rows.
		 */
		if (n == 0)
		{
			if (stmt->strict)
				ereport(ERROR,
						(errcode(ERRCODE_NO_DATA_FOUND),
						 errmsg("query returned no rows")));
			/* set the target to NULL(s) */
			exec_move_row(estate, rec, row, NULL, tuptab->tupdesc);
		}
		else
		{
			if (n > 1 && stmt->strict)
				ereport(ERROR,
						(errcode(ERRCODE_TOO_MANY_ROWS),
						 errmsg("query returned more than one row")));
			/* Put the first result row into the target */
			exec_move_row(estate, rec, row, tuptab->vals[0], tuptab->tupdesc);
		}
	}
	else
	{
		/*
		 * It might be a good idea to raise an error if the query returned
		 * tuples that are being ignored, but historically we have not done
		 * that.
		 */
	}

	/* Release any result from SPI_execute, as well as the querystring */
	SPI_freetuptable(SPI_tuptable);
	pfree(querystr);

	return PLPGPSM_RC_OK;
}


/* ----------
 * exec_stmt_open			Execute an OPEN cursor statement
 * ----------
 */
static int
exec_stmt_open(PLpgPSM_execstate *estate, PLpgPSM_stmt_open *stmt)
{
	PLpgPSM_var *curvar = NULL;
	char	   *curname = NULL;
	PLpgPSM_expr *query = NULL;
	Portal		portal = NULL;
	int			i;
	Datum	   *values = NULL;
	char	   *nulls = NULL;

	estate->outer_stmt = false;
	/* ----------
	 * Get the cursor variable and if it has an assigned name, check
	 * that it's not in use currently.
	 * ----------
	 */
	curvar = (PLpgPSM_var *) (estate->datums[stmt->curvar]);
	
	if (curvar->portal)
		ereport(ERROR,
				(errcode(ERRCODE_DUPLICATE_CURSOR),
				 errmsg("cursor \"%s\" already in use", curvar->refname)));
	
	
	if (!curvar->isnull)
	{
		curname = DatumGetCString(DirectFunctionCall1(textout, curvar->value));

		if (SPI_cursor_find(curname) != NULL)
			ereport(ERROR,
					(errcode(ERRCODE_DUPLICATE_CURSOR),
					 errmsg("cursor \"%s\" already in use", curname)));

		pfree(curname);
	}

	// ToDo: Holdable cursors

	/* static cursors */
	if ((query = curvar->cursor_explicit_expr))
	{
		if (query->plan == NULL)
			exec_prepare_plan(estate, query, curvar->cursor_options);
			
		/* ----------
		 * Here we go if we have a saved plan where we have to put
		 * values into, either from an explicit cursor or from a
		 * refcursor opened with OPEN ... FOR SELECT ...;
		 * ----------
		 */
		values = (Datum *) palloc(query->nparams * sizeof(Datum));
		nulls = (char *) palloc(query->nparams * sizeof(char));

		for (i = 0; i < query->nparams; i++)
		{
			PLpgPSM_datum *datum = estate->datums[query->params[i]];
	    		Oid		paramtypeid;
			bool		paramisnull;

			exec_eval_datum(estate, datum, query->plan_argtypes[i],
							&paramtypeid, &values[i], &paramisnull);
			if (paramisnull)
				nulls[i] = 'n';
			else
				nulls[i] = ' ';
		}

		/* ----------
	    	 * Open the cursor
		 * ----------
		 */
		
		portal = SPI_cursor_open(curvar->holdable ? curvar->refname : NULL, query->plan, values, nulls,
							 estate->readonly_func);

		pfree(values);
		pfree(nulls);
	}
	else
	/* dynamic cursor */
	{
		int i;		
		PLpgPSM_prepared_plan *preplan = get_plan(curvar->prepname);
		
		if (preplan == NULL)
			ereport(ERROR,
					(errmsg("cannot find named prepared plan")));
							
		if (preplan->nparams > 0)
		{
			/* have to exists USING */
			if (!stmt->args)
				ereport(ERROR,
					(errmsg("missing USING clause")));
					
			if (stmt->args->nparams != preplan->nparams)
				ereport(ERROR,
					(errmsg("wrong number of arguments in USING clause")));
		
		
			/* args are in query->nparams and query->params */
		
			values = (Datum *) palloc0(preplan->nparams * sizeof(Datum));
			nulls = (char *) palloc0(preplan->nparams * sizeof(char));
		
			for (i = 0; i < preplan->nparams; i++)
			{
				PLpgPSM_datum *	datum = estate->datums[stmt->args->params[i]];
				Oid		paramtypeid;
				bool		paramisnull;
			
				exec_eval_datum(estate, datum, InvalidOid,
							    &paramtypeid, &values[i], &paramisnull);
							    
				values[i] = exec_simple_cast_value(values[i],
								paramtypeid,
								preplan->params[i],
								-1,
								paramisnull);
								
							    
				if (paramisnull)
					nulls[i] = 'n';
				else
					nulls[i] = ' ';
			}

		}

		portal = SPI_cursor_open(curvar->holdable ? curvar->refname : NULL, preplan->plan, values, nulls,
						estate->readonly_func);

		if (preplan->nparams > 0)
		{
			pfree(values);
			pfree(nulls);
		}
	}

	if (portal == NULL)
		elog(ERROR, "could not open cursor: %s",
			 SPI_result_code_string(SPI_result));

	/* ----------
	 * Store the eventually assigned portal name in the cursor variable
	 * ----------
	 */
	free_var(curvar);
	curvar->value = DirectFunctionCall1(textin, CStringGetDatum(portal->name));
	curvar->isnull = false;
	curvar->freeval = true;
	
	curvar->portal = portal;

	return PLPGPSM_RC_OK;
}


/* ----------
 * exec_stmt_fetch			Fetch from a cursor into a target
 * ----------
 */
static int
exec_stmt_fetch(PLpgPSM_execstate *estate, PLpgPSM_stmt_fetch *stmt)
{
	PLpgPSM_var *curvar = NULL;
	PLpgPSM_rec *rec = NULL;
	PLpgPSM_row *row = NULL;
	SPITupleTable *tuptab;
	Portal		portal;
	char	   *curname;
	int			n;
	int	how_many = stmt->how_many;

	estate->outer_stmt = false;

	/* ----------
	 * Get the portal of the cursor by name
	 * ----------
	 */
	curvar = (PLpgPSM_var *) (estate->datums[stmt->curvar]);
	
	if (curvar->portal)
		portal = curvar->portal;
	else
	{
		/* old path */
		if (curvar->isnull)
			ereport(ERROR,
					(errcode(ERRCODE_NULL_VALUE_NOT_ALLOWED),
					 errmsg("cursor variable \"%s\" is NULL", curvar->refname)));
		curname = DatumGetCString(DirectFunctionCall1(textout, curvar->value));

		portal = SPI_cursor_find(curname);
		if (portal == NULL)
			ereport(ERROR,
					(errcode(ERRCODE_UNDEFINED_CURSOR),
					 errmsg("cursor \"%s\" does not exist", curname)));
		pfree(curname);
	}

	/* calculate position expr FETCH_RELATIVE, FETCH_ABSOLUTE */
	if (stmt->expr)
	{
		bool isnull;
		
		how_many = exec_eval_integer(estate, stmt->expr, &isnull);
		
		if (isnull)
			ereport(ERROR, 
					(errcode(ERRCODE_NULL_VALUE_NOT_ALLOWED),
					 errmsg("cursor position is NULL"),
					 errhint("Relative or absolute cursor position cannot be NULL")));
					 
		exec_eval_cleanup(estate);
	}
	

	/* I have to catch error before postgresql, messy message */
	if (stmt->direction != FETCH_FORWARD || how_many < 0)
		if (!(curvar->cursor_options & CURSOR_OPT_SCROLL))
			ereport(ERROR,
					(errcode(ERRCODE_INVALID_CURSOR_STATE),
					 errmsg("cursor can only scan forward"),
					 errhint("Declare it with SCROLL option to enable scrollable scan")));

	/* ----------
	 * Determine if we fetch into a record or a row
	 * ----------
	 */
	 
	if (!stmt->is_move)
	{
		if (stmt->rec != NULL)
			rec = (PLpgPSM_rec *) (estate->datums[stmt->rec->dno]);
		else if (stmt->row != NULL)
			row = (PLpgPSM_row *) (estate->datums[stmt->row->dno]);
		else
			elog(ERROR, "unsupported target");

		/* ----------
		 * Fetch 1 tuple from the cursor
		 * ----------
		 */
		SPI_scroll_cursor_fetch(portal, stmt->direction, how_many);
		tuptab = SPI_tuptable;
		n = SPI_processed;

		/* ----------
		 * Set the target and the global FOUND variable appropriately.
		 * ----------
		 */
		if (n == 0)
		{
			exec_move_row(estate, rec, row, NULL, tuptab->tupdesc);
			exec_set_found(estate, false);
			exec_signal_not_found(true);
		}
		else
		{
			exec_move_row(estate, rec, row, tuptab->vals[0], tuptab->tupdesc);
			exec_set_found(estate, true);
			exec_signal_not_found(false);
		}

		SPI_freetuptable(tuptab);
	}
	else
	{
		/* Move the cursor */
		SPI_scroll_cursor_move(portal, stmt->direction, how_many);
		n = SPI_processed;
		exec_set_found(estate, n != 0);
		exec_signal_not_found(n == 0);
	}			

	return PLPGPSM_RC_OK;
}


/* ----------
 * exec_stmt_close			Close a cursor
 * ----------
 */
static int
exec_stmt_close(PLpgPSM_execstate *estate, PLpgPSM_stmt_close *stmt)
{
	PLpgPSM_var *curvar = NULL;
	Portal		portal;
	char	   *curname;

	estate->outer_stmt = false;

	/* ----------
	 * Get the portal of the cursor by name
	 * ----------
	 */
	 
	curvar = (PLpgPSM_var *) (estate->datums[stmt->curvar]);
	
	if (curvar->portal)
	{
		SPI_cursor_close(curvar->portal);
		curvar->portal = NULL;
	}
	else
	{
		if (curvar->isnull)
			ereport(ERROR,
				(errcode(ERRCODE_NULL_VALUE_NOT_ALLOWED),
				 errmsg("cursor variable \"%s\" is NULL", curvar->refname)));
		curname = DatumGetCString(DirectFunctionCall1(textout, curvar->value));

		portal = SPI_cursor_find(curname);
		if (portal == NULL)
			ereport(ERROR,
				(errcode(ERRCODE_UNDEFINED_CURSOR),
				 errmsg("cursor \"%s\" does not exist", curname)));
		pfree(curname);

		/* ----------
		 * And close it.
	    	 * ----------
		 */
		SPI_cursor_close(portal);
	}

	return PLPGPSM_RC_OK;
}


/* ----------
 * exec_assign_expr			Put an expression's result into
 *					a variable.
 * ----------
 */
static void
exec_assign_expr(PLpgPSM_execstate *estate, PLpgPSM_datum *target,
				 PLpgPSM_expr *expr)
{
	Datum		value;
	Oid			valtype;
	bool		isnull = false;

	estate->outer_stmt = false;
	value = exec_eval_expr(estate, expr, &isnull, &valtype);
	/* hack, reset valtype if target isn't scalar and isnull */
	if (isnull)
	{
		if (target->dtype != PLPGPSM_DTYPE_VAR)
		{
			valtype = RECORDOID;
		}
			
	}
	
	exec_assign_value(estate, target, value, valtype, &isnull);
	exec_eval_cleanup(estate);
}


/* ----------
 * exec_assign_value			Put a value into a target field
 * ----------
 */
static void
exec_assign_value(PLpgPSM_execstate *estate,
				  PLpgPSM_datum *target,
				  Datum value, Oid valtype, bool *isNull)
{
	switch (target->dtype)
	{
		case PLPGPSM_DTYPE_VAR:
			{
				/*
				 * Target is a variable
				 */
				PLpgPSM_var *var = (PLpgPSM_var *) target;
				Datum		newvalue;

				newvalue = exec_cast_value(value, valtype, var->datatype->typoid,
										   &(var->datatype->typinput),
										   var->datatype->typioparam,
										   var->datatype->atttypmod,
										   *isNull);

				if (*isNull && var->notnull)
					ereport(ERROR,
							(errcode(ERRCODE_NULL_VALUE_NOT_ALLOWED),
							 errmsg("NULL cannot be assigned to variable \"%s\" declared NOT NULL",
									var->refname)));

				/*
				 * If type is by-reference, make sure we have a freshly
				 * palloc'd copy; the originally passed value may not live as
				 * long as the variable!  But we don't need to re-copy if
				 * exec_cast_value performed a conversion; its output must
				 * already be palloc'd.
				 */
				if (!var->datatype->typbyval && !*isNull)
				{
					if (newvalue == value)
						newvalue = datumCopy(newvalue,
											 false,
											 var->datatype->typlen);
				}

				/*
				 * Now free the old value.	(We can't do this any earlier
				 * because of the possibility that we are assigning the var's
				 * old value to it, eg "foo := foo".  We could optimize out
				 * the assignment altogether in such cases, but it's too
				 * infrequent to be worth testing for.)
				 */
				free_var(var);

				var->value = newvalue;
				var->isnull = *isNull;
				if (!var->datatype->typbyval && !*isNull)
					var->freeval = true;
				break;
			}

		case PLPGPSM_DTYPE_ROW:
			{
				/*
				 * Target is a row variable
				 */
				PLpgPSM_row *row = (PLpgPSM_row *) target;

				/* Source must be of RECORD or composite type */
				if (!type_is_rowtype(valtype))
					ereport(ERROR,
							(errcode(ERRCODE_DATATYPE_MISMATCH),
							 errmsg("cannot assign non-composite value to a row variable")));
				row->isnull = *isNull;
				if (!*isNull)
				{
					HeapTupleHeader td;
					Oid			tupType;
					int32		tupTypmod;
					TupleDesc	tupdesc;
					HeapTupleData tmptup;

					/* Else source is a tuple Datum, safe to do this: */
					td = DatumGetHeapTupleHeader(value);
					/* Extract rowtype info and find a tupdesc */
					tupType = HeapTupleHeaderGetTypeId(td);
					tupTypmod = HeapTupleHeaderGetTypMod(td);
					tupdesc = lookup_rowtype_tupdesc(tupType, tupTypmod);
					/* Build a temporary HeapTuple control structure */
					tmptup.t_len = HeapTupleHeaderGetDatumLength(td);
					ItemPointerSetInvalid(&(tmptup.t_self));
					tmptup.t_tableOid = InvalidOid;
					tmptup.t_data = td;
					exec_move_row(estate, NULL, row, &tmptup, tupdesc);
					ReleaseTupleDesc(tupdesc);
				}
				break;
			}

		case PLPGPSM_DTYPE_REC:
			{
				/*
				 * Target is a record variable
				 */
				PLpgPSM_rec *rec = (PLpgPSM_rec *) target;

				/* Source must be of RECORD or composite type */
				if (!type_is_rowtype(valtype))
					ereport(ERROR,
							(errcode(ERRCODE_DATATYPE_MISMATCH),
							 errmsg("cannot assign non-composite value to a row variable")));
				if (*isNull)
				{
					/* If source is null, just assign nulls to the record */
					exec_move_row(estate, rec, NULL, NULL, NULL);
				}
				else
				{
					HeapTupleHeader td;
					Oid			tupType;
					int32		tupTypmod;
					TupleDesc	tupdesc;
					HeapTupleData tmptup;

					/* Else source is a tuple Datum, safe to do this: */
					td = DatumGetHeapTupleHeader(value);
					/* Extract rowtype info and find a tupdesc */
					tupType = HeapTupleHeaderGetTypeId(td);
					tupTypmod = HeapTupleHeaderGetTypMod(td);
					tupdesc = lookup_rowtype_tupdesc(tupType, tupTypmod);
					/* Build a temporary HeapTuple control structure */
					tmptup.t_len = HeapTupleHeaderGetDatumLength(td);
					ItemPointerSetInvalid(&(tmptup.t_self));
					tmptup.t_tableOid = InvalidOid;
					tmptup.t_data = td;
					exec_move_row(estate, rec, NULL, &tmptup, tupdesc);
					ReleaseTupleDesc(tupdesc);
				}
				break;
			}

		case PLPGPSM_DTYPE_RECFIELD:
			{
				/*
				 * Target is a field of a record
				 */
				PLpgPSM_recfield *recfield = (PLpgPSM_recfield *) target;
				PLpgPSM_rec *rec;
				int			fno;
				HeapTuple	newtup;
				int			natts;
				int			i;
				Datum	   *values;
				char	   *nulls;
				void	   *mustfree;
				bool		attisnull;
				Oid			atttype;
				int32		atttypmod;

				rec = (PLpgPSM_rec *) (estate->datums[recfield->recparentno]);

				/*
				 * Check that there is already a tuple in the record. We need
				 * that because records don't have any predefined field
				 * structure.
				 */
				if (!HeapTupleIsValid(rec->tup))
					ereport(ERROR,
						  (errcode(ERRCODE_OBJECT_NOT_IN_PREREQUISITE_STATE),
						   errmsg("record \"%s\" is not assigned yet",
								  rec->refname),
						   errdetail("The tuple structure of a not-yet-assigned record is indeterminate.")));

				/*
				 * Get the number of the records field to change and the
				 * number of attributes in the tuple.
				 */
				fno = SPI_fnumber(rec->tupdesc, recfield->fieldname);
				if (fno == SPI_ERROR_NOATTRIBUTE)
					ereport(ERROR,
							(errcode(ERRCODE_UNDEFINED_COLUMN),
							 errmsg("record \"%s\" has no field \"%s\"",
									rec->refname, recfield->fieldname)));
				fno--;
				natts = rec->tupdesc->natts;

				/*
				 * Set up values/datums arrays for heap_formtuple.	For all
				 * the attributes except the one we want to replace, use the
				 * value that's in the old tuple.
				 */
				values = palloc(sizeof(Datum) * natts);
				nulls = palloc(natts);

				for (i = 0; i < natts; i++)
				{
					if (i == fno)
						continue;
					values[i] = SPI_getbinval(rec->tup, rec->tupdesc,
											  i + 1, &attisnull);
					if (attisnull)
						nulls[i] = 'n';
					else
						nulls[i] = ' ';
				}

				/*
				 * Now insert the new value, being careful to cast it to the
				 * right type.
				 */
				atttype = SPI_gettypeid(rec->tupdesc, fno + 1);
				atttypmod = rec->tupdesc->attrs[fno]->atttypmod;
				attisnull = *isNull;
				values[fno] = exec_simple_cast_value(value,
													 valtype,
													 atttype,
													 atttypmod,
													 attisnull);
				if (attisnull)
					nulls[fno] = 'n';
				else
					nulls[fno] = ' ';

				/*
				 * Avoid leaking the result of exec_simple_cast_value, if it
				 * performed a conversion to a pass-by-ref type.
				 */
				if (!attisnull && values[fno] != value && !get_typbyval(atttype))
					mustfree = DatumGetPointer(values[fno]);
				else
					mustfree = NULL;

				/*
				 * Now call heap_formtuple() to create a new tuple that
				 * replaces the old one in the record.
				 */
				newtup = heap_formtuple(rec->tupdesc, values, nulls);

				if (rec->freetup)
					heap_freetuple(rec->tup);

				rec->tup = newtup;
				rec->freetup = true;

				pfree(values);
				pfree(nulls);
				if (mustfree)
					pfree(mustfree);

				break;
			}

		case PLPGPSM_DTYPE_ARRAYELEM:
			{
				int			nsubscripts;
				int			i;
				PLpgPSM_expr *subscripts[MAXDIM];
				int			subscriptvals[MAXDIM];
				bool		oldarrayisnull;
				Oid			arraytypeid,
							arrayelemtypeid;
				int16		arraytyplen,
							elemtyplen;
				bool		elemtypbyval;
				char		elemtypalign;
				Datum		oldarraydatum,
							coerced_value;
				ArrayType  *oldarrayval;
				ArrayType  *newarrayval;

				/*
				 * Target is an element of an array
				 *
				 * To handle constructs like x[1][2] := something, we have to
				 * be prepared to deal with a chain of arrayelem datums. Chase
				 * back to find the base array datum, and save the subscript
				 * expressions as we go.  (We are scanning right to left here,
				 * but want to evaluate the subscripts left-to-right to
				 * minimize surprises.)
				 */
				nsubscripts = 0;
				do
				{
					PLpgPSM_arrayelem *arrayelem = (PLpgPSM_arrayelem *) target;

					if (nsubscripts >= MAXDIM)
						ereport(ERROR,
								(errcode(ERRCODE_PROGRAM_LIMIT_EXCEEDED),
								 errmsg("number of array dimensions exceeds the maximum allowed, %d",
										MAXDIM)));
					subscripts[nsubscripts++] = arrayelem->subscript;
					target = estate->datums[arrayelem->arrayparentno];
				} while (target->dtype == PLPGPSM_DTYPE_ARRAYELEM);

				/* Fetch current value of array datum */
				exec_eval_datum(estate, target, InvalidOid,
							  &arraytypeid, &oldarraydatum, &oldarrayisnull);

				arrayelemtypeid = get_element_type(arraytypeid);
				if (!OidIsValid(arrayelemtypeid))
					ereport(ERROR,
							(errcode(ERRCODE_DATATYPE_MISMATCH),
							 errmsg("subscripted object is not an array")));

				get_typlenbyvalalign(arrayelemtypeid,
									 &elemtyplen,
									 &elemtypbyval,
									 &elemtypalign);
				arraytyplen = get_typlen(arraytypeid);

				/*
				 * Evaluate the subscripts, switch into left-to-right order.
				 * Like ExecEvalArrayRef(), complain if any subscript is null.
				 */
				for (i = 0; i < nsubscripts; i++)
				{
					bool		subisnull;

					subscriptvals[i] =
						exec_eval_integer(estate,
										  subscripts[nsubscripts - 1 - i],
										  &subisnull);
					if (subisnull)
						ereport(ERROR,
								(errcode(ERRCODE_NULL_VALUE_NOT_ALLOWED),
								 errmsg("array subscript in assignment must not be NULL")));
				}

				/* Coerce source value to match array element type. */
				coerced_value = exec_simple_cast_value(value,
													   valtype,
													   arrayelemtypeid,
													   -1,
													   *isNull);

				/*
				 * If the original array is null, cons up an empty array so
				 * that the assignment can proceed; we'll end with a
				 * one-element array containing just the assigned-to
				 * subscript.  This only works for varlena arrays, though; for
				 * fixed-length array types we skip the assignment.  We can't
				 * support assignment of a null entry into a fixed-length
				 * array, either, so that's a no-op too.  This is all ugly but
				 * corresponds to the current behavior of ExecEvalArrayRef().
				 */
				if (arraytyplen > 0 &&	/* fixed-length array? */
					(oldarrayisnull || *isNull))
					return;

				if (oldarrayisnull)
					oldarrayval = construct_empty_array(arrayelemtypeid);
				else
					oldarrayval = (ArrayType *) DatumGetPointer(oldarraydatum);

				/*
				 * Build the modified array value.
				 */
				newarrayval = array_set(oldarrayval,
										nsubscripts,
										subscriptvals,
										coerced_value,
										*isNull,
										arraytyplen,
										elemtyplen,
										elemtypbyval,
										elemtypalign);

				/*
				 * Avoid leaking the result of exec_simple_cast_value, if it
				 * performed a conversion to a pass-by-ref type.
				 */
				if (!*isNull && coerced_value != value && !elemtypbyval)
					pfree(DatumGetPointer(coerced_value));

				/*
				 * Assign the new array to the base variable.  It's never NULL
				 * at this point.
				 */
				*isNull = false;
				exec_assign_value(estate, target,
								  PointerGetDatum(newarrayval),
								  arraytypeid, isNull);

				/*
				 * Avoid leaking the modified array value, too.
				 */
				pfree(newarrayval);
				break;
			}

		default:
			elog(ERROR, "unrecognized dtype: %d", target->dtype);
	}
}

/*
 * exec_eval_datum				Get current value of a PLpgPSM_datum
 *
 * The type oid, value in Datum format, and null flag are returned.
 *
 * If expectedtypeid isn't InvalidOid, it is checked against the actual type.
 *
 * At present this doesn't handle PLpgPSM_expr or PLpgPSM_arrayelem datums.
 *
 * NOTE: caller must not modify the returned value, since it points right
 * at the stored value in the case of pass-by-reference datatypes.	In some
 * cases we have to palloc a return value, and in such cases we put it into
 * the estate's short-term memory context.
 */





static void
exec_eval_datum(PLpgPSM_execstate *estate,
				PLpgPSM_datum *datum,
				Oid expectedtypeid,
				Oid *typeid,
				Datum *value,
				bool *isnull)
{
	MemoryContext oldcontext;

	switch (datum->dtype)
	{
		case PLPGPSM_DTYPE_VAR:
			{
				PLpgPSM_var *var = (PLpgPSM_var *) datum;

				*typeid = var->datatype->typoid;
				*value = var->value;
				*isnull = var->isnull;
	
				/* check parent null */
				if (var->parent && var->parent->isnull)
					ereport(ERROR,
						  (errcode(ERRCODE_OBJECT_NOT_IN_PREREQUISITE_STATE),
						   errmsg("variable \"%s\" is not assigned yet",
								  var->parent->refname)));

				if (expectedtypeid != InvalidOid && expectedtypeid != *typeid)
					ereport(ERROR,
							(errcode(ERRCODE_DATATYPE_MISMATCH),
							 errmsg("type of \"%s\" does not match that when preparing the plan",
									var->refname)));
				break;
			}

		case PLPGPSM_DTYPE_ROW:
			{
				PLpgPSM_row *row = (PLpgPSM_row *) datum;
				HeapTuple	tup;
				if (!row->rowtupdesc)	/* should not happen */
					elog(ERROR, "row variable has no tupdesc");
				/* Make sure we have a valid type/typmod setting */
				if (row->isnull)
				{
					*value = (Datum) 0;
					*isnull = true;
					*typeid = row->rowtupdesc->tdtypeid;
					break;
				}

				BlessTupleDesc(row->rowtupdesc);
				oldcontext = MemoryContextSwitchTo(estate->eval_econtext->ecxt_per_tuple_memory);
				tup = make_tuple_from_row(estate, row, row->rowtupdesc);
				if (tup == NULL)	/* should not happen */
					elog(ERROR, "row not compatible with its own tupdesc");
				MemoryContextSwitchTo(oldcontext);
				*typeid = row->rowtupdesc->tdtypeid;
				*value = HeapTupleGetDatum(tup);
				*isnull = false;
				if (expectedtypeid != InvalidOid && expectedtypeid != *typeid)
					ereport(ERROR,
							(errcode(ERRCODE_DATATYPE_MISMATCH),
							 errmsg("type of \"%s\" does not match that when preparing the plan",
									row->refname)));
				break;
			}

		case PLPGPSM_DTYPE_REC:
			{
				PLpgPSM_rec *rec = (PLpgPSM_rec *) datum;
				HeapTupleData worktup;

				if (!HeapTupleIsValid(rec->tup))
					ereport(ERROR,
						  (errcode(ERRCODE_OBJECT_NOT_IN_PREREQUISITE_STATE),
						   errmsg("record \"%s\" is not assigned yet",
								  rec->refname),
						   errdetail("The tuple structure of a not-yet-assigned record is indeterminate.")));
				Assert(rec->tupdesc != NULL);
				/* Make sure we have a valid type/typmod setting */
				BlessTupleDesc(rec->tupdesc);

				/*
				 * In a trigger, the NEW and OLD parameters are likely to be
				 * on-disk tuples that don't have the desired Datum fields.
				 * Copy the tuple body and insert the right values.
				 */
				oldcontext = MemoryContextSwitchTo(estate->eval_econtext->ecxt_per_tuple_memory);
				heap_copytuple_with_tuple(rec->tup, &worktup);
				HeapTupleHeaderSetDatumLength(worktup.t_data, worktup.t_len);
				HeapTupleHeaderSetTypeId(worktup.t_data, rec->tupdesc->tdtypeid);
				HeapTupleHeaderSetTypMod(worktup.t_data, rec->tupdesc->tdtypmod);
				MemoryContextSwitchTo(oldcontext);
				*typeid = rec->tupdesc->tdtypeid;
				*value = HeapTupleGetDatum(&worktup);
				*isnull = false;
				if (expectedtypeid != InvalidOid && expectedtypeid != *typeid)
					ereport(ERROR,
							(errcode(ERRCODE_DATATYPE_MISMATCH),
							 errmsg("type of \"%s\" does not match that when preparing the plan",
									rec->refname)));
				break;
			}

		case PLPGPSM_DTYPE_RECFIELD:
			{
				PLpgPSM_recfield *recfield = (PLpgPSM_recfield *) datum;
				PLpgPSM_rec *rec;
				int			fno;

				rec = (PLpgPSM_rec *) (estate->datums[recfield->recparentno]);
				if (!HeapTupleIsValid(rec->tup))
					ereport(ERROR,
						  (errcode(ERRCODE_OBJECT_NOT_IN_PREREQUISITE_STATE),
						   errmsg("record \"%s\" is not assigned yet",
								  rec->refname),
						   errdetail("The tuple structure of a not-yet-assigned record is indeterminate.")));
				fno = SPI_fnumber(rec->tupdesc, recfield->fieldname);
				if (fno == SPI_ERROR_NOATTRIBUTE)
					ereport(ERROR,
							(errcode(ERRCODE_UNDEFINED_COLUMN),
							 errmsg("record \"%s\" has no field \"%s\"",
									rec->refname, recfield->fieldname)));
				*typeid = SPI_gettypeid(rec->tupdesc, fno);
				*value = SPI_getbinval(rec->tup, rec->tupdesc, fno, isnull);
				if (expectedtypeid != InvalidOid && expectedtypeid != *typeid)
					ereport(ERROR,
							(errcode(ERRCODE_DATATYPE_MISMATCH),
							 errmsg("type of \"%s.%s\" does not match that when preparing the plan",
									rec->refname, recfield->fieldname)));
				break;
			}

		case PLPGPSM_DTYPE_TRIGARG:
			{
				PLpgPSM_trigarg *trigarg = (PLpgPSM_trigarg *) datum;
				int			tgargno;

				*typeid = TEXTOID;
				tgargno = exec_eval_integer(estate, trigarg->argnum, isnull);
				if (*isnull || tgargno < 0 || tgargno >= estate->trig_nargs)
				{
					*value = (Datum) 0;
					*isnull = true;
				}
				else
				{
					*value = estate->trig_argv[tgargno];
					*isnull = false;
				}
				if (expectedtypeid != InvalidOid && expectedtypeid != *typeid)
					ereport(ERROR,
							(errcode(ERRCODE_DATATYPE_MISMATCH),
							 errmsg("type of tgargv[%d] does not match that when preparing the plan",
									tgargno)));
				break;
			}

		default:
			elog(ERROR, "unrecognized dtype: %d", datum->dtype);
	}
}

/* ----------
 * exec_eval_integer		Evaluate an expression, coerce result to int4
 *
 * Note we do not do exec_eval_cleanup here; the caller must do it at
 * some later point.  (We do this because the caller may be holding the
 * results of other, pass-by-reference, expression evaluations, such as
 * an array value to be subscripted.  Also see notes in exec_eval_simple_expr
 * about allocation of the parameter array.)
 * ----------
 */
static int
exec_eval_integer(PLpgPSM_execstate *estate,
				  PLpgPSM_expr *expr,
				  bool *isNull)
{
	Datum		exprdatum;
	Oid			exprtypeid;

	exprdatum = exec_eval_expr(estate, expr, isNull, &exprtypeid);
	exprdatum = exec_simple_cast_value(exprdatum, exprtypeid,
									   INT4OID, -1,
									   *isNull);
	return DatumGetInt32(exprdatum);
}

/* ----------
 * exec_eval_boolean		Evaluate an expression, coerce result to bool
 *
 * Note we do not do exec_eval_cleanup here; the caller must do it at
 * some later point.
 * ----------
 */
static bool
exec_eval_boolean(PLpgPSM_execstate *estate,
				  PLpgPSM_expr *expr,
				  bool *isNull)
{
	Datum		exprdatum;
	Oid			exprtypeid;

	exprdatum = exec_eval_expr(estate, expr, isNull, &exprtypeid);
	exprdatum = exec_simple_cast_value(exprdatum, exprtypeid,
									   BOOLOID, -1,
									   *isNull);
	return DatumGetBool(exprdatum);
}

/* ----------
 * exec_eval_expr			Evaluate an expression and return
 *					the result Datum.
 *
 * NOTE: caller must do exec_eval_cleanup when done with the Datum.
 * ----------
 */
static Datum
exec_eval_expr(PLpgPSM_execstate *estate,
			   PLpgPSM_expr *expr,
			   bool *isNull,
			   Oid *rettype)
{
	Datum 		result = 0;
	int			rc;

	/*
	 * If first time through, create a plan for this expression.
	 */
	if (expr->plan == NULL)
		exec_prepare_plan(estate, expr, 0);

	/*
	 * If this is a simple expression, bypass SPI and use the executor
	 * directly
	 */
	if (exec_eval_simple_expr(estate, expr, &result, isNull, rettype))
		return result;

	/*
	 * Else do it hard way via exec_run_select
	 */
	rc = exec_run_select(estate, expr, 2, NULL, NULL);
	if (rc != SPI_OK_SELECT)
		ereport(ERROR,
				(errcode(ERRCODE_WRONG_OBJECT_TYPE),
				 errmsg("query \"%s\" did not return data", expr->query)));

	/*
	 * If there are no rows selected, the result is NULL.
	 */
	if (estate->current_diag->eval_processed == 0)
	{
		*isNull = true;
		return (Datum) 0;
	}

	/*
	 * Check that the expression returned one single Datum
	 */
	if (estate->current_diag->eval_processed > 1)
		ereport(ERROR,
				(errcode(ERRCODE_CARDINALITY_VIOLATION),
				 errmsg("query \"%s\" returned more than one row",
						expr->query)));
	if (estate->eval_tuptable->tupdesc->natts != 1)
		ereport(ERROR,
				(errcode(ERRCODE_SYNTAX_ERROR),
				 errmsg("query \"%s\" returned %d columns", expr->query,
						estate->eval_tuptable->tupdesc->natts)));

	/*
	 * Return the result and its type
	 */
	*rettype = SPI_gettypeid(estate->eval_tuptable->tupdesc, 1);
	return SPI_getbinval(estate->eval_tuptable->vals[0],
						 estate->eval_tuptable->tupdesc, 1, isNull);
}


/* ----------
 * exec_run_select			Execute a select query
 * ----------
 */
static int
exec_run_select(PLpgPSM_execstate *estate,
				PLpgPSM_expr *expr, long maxtuples, Portal *portalP, char *name)
{
	int			i;
	Datum	   *values;
	char	   *nulls;
	int			rc;

	/*
	 * On the first call for this expression generate the plan
	 */
	if (expr->plan == NULL)
		exec_prepare_plan(estate, expr, 0);

	/*
	 * Now build up the values and nulls arguments for SPI_execute_plan()
	 */
	values = (Datum *) palloc(expr->nparams * sizeof(Datum));
	nulls = (char *) palloc(expr->nparams * sizeof(char));

	for (i = 0; i < expr->nparams; i++)
	{
		PLpgPSM_datum *datum = estate->datums[expr->params[i]];
		Oid			paramtypeid;
		bool		paramisnull;

		exec_eval_datum(estate, datum, expr->plan_argtypes[i],
						&paramtypeid, &values[i], &paramisnull);
		if (paramisnull)
			nulls[i] = 'n';
		else
			nulls[i] = ' ';
	}

	/*
	 * If a portal was requested, put the query into the portal
	 */
	if (portalP != NULL)
	{
		*portalP = SPI_cursor_open(name, expr->plan, values, nulls,
								   estate->readonly_func);
		if (*portalP == NULL)
			elog(ERROR, "could not open implicit cursor for query \"%s\": %s",
				 expr->query, SPI_result_code_string(SPI_result));
		pfree(values);
		pfree(nulls);
		return SPI_OK_CURSOR;
	}

	/*
	 * Execute the query
	 */
	rc = SPI_execute_plan(expr->plan, values, nulls,
						  estate->readonly_func, maxtuples);
	exec_signal_not_found(SPI_processed == 0);

	if (rc != SPI_OK_SELECT)
		ereport(ERROR,
				(errcode(ERRCODE_SYNTAX_ERROR),
				 errmsg("query \"%s\" is not a SELECT", expr->query)));

	/* Save query results for eventual cleanup */
	Assert(estate->eval_tuptable == NULL);
	estate->eval_tuptable = SPI_tuptable;
	estate->current_diag->eval_processed = SPI_processed;
	estate->current_diag->eval_lastoid = SPI_lastoid;

	pfree(values);
	pfree(nulls);

	return rc;
}


/* ----------
 * exec_eval_simple_expr -		Evaluate a simple expression returning
 *								a Datum by directly calling ExecEvalExpr().
 *
 * If successful, store results into *result, *isNull, *rettype and return
 * TRUE.  If the expression is not simple (any more), return FALSE.
 *
 * It is possible though unlikely for a simple expression to become non-simple
 * (consider for example redefining a trivial view).  We must handle that for
 * correctness; fortunately it's normally inexpensive to do
 * RevalidateCachedPlan on a simple expression.  We do not consider the other
 * direction (non-simple expression becoming simple) because we'll still give
 * correct results if that happens, and it's unlikely to be worth the cycles
 * to check.
 *
 * Note: if pass-by-reference, the result is in the eval_econtext's
 * temporary memory context.  It will be freed when exec_eval_cleanup
 * is done.
 * ----------
 */
static bool
exec_eval_simple_expr(PLpgPSM_execstate *estate,
					  PLpgPSM_expr *expr,
					  Datum *result,
					  bool *isNull,
					  Oid *rettype)
{
	ExprContext *econtext = estate->eval_econtext;
	CachedPlanSource *plansource;
	CachedPlan *cplan;
	ParamListInfo paramLI;
	int			i;
	MemoryContext oldcontext;

	/*
    	 * Forget it if expression wasn't simple before.
    	 */
    	if (expr->expr_simple_expr == NULL)
    		return false;
    
    	/*
    	 * Revalidate cached plan, so that we will notice if it became stale.
    	 * (We also need to hold a refcount while using the plan.)  Note that
    	 * even if replanning occurs, the length of plancache_list can't change,
    	 * since it is a property of the raw parsetree generated from the query
    	 * text.
    	 */
    	Assert(list_length(expr->plan->plancache_list) == 1);
    	plansource = (CachedPlanSource *) linitial(expr->plan->plancache_list);
    	cplan = RevalidateCachedPlan(plansource, true);
    	if (cplan->generation != expr->expr_simple_generation)
    	{
    		/* It got replanned ... is it still simple? */
    		exec_simple_check_plan(expr);
    		if (expr->expr_simple_expr == NULL)
    		{
    			/* Ooops, release refcount and fail */
    			ReleaseCachedPlan(cplan, true);
    			return false;
    		}
    	}

	/*
	 * Pass back previously-determined result type.
	 */
	*rettype = expr->expr_simple_type;

	/*
	 * Prepare the expression for execution, if it's not been done already in
	 * the current eval_state. (This will be forced to happen if we called
	 * exec_simple_check_plan above.)
	 */
	if (expr->expr_simple_id != estate->eval_estate_simple_id)
	{
		expr->expr_simple_state = ExecPrepareExpr(expr->expr_simple_expr,
												  estate->eval_estate);
		expr->expr_simple_id = estate->eval_estate_simple_id;
	}

	/*
	 * Param list can live in econtext's temporary memory context.
	 *
	 * XXX think about avoiding repeated palloc's for param lists? Beware
	 * however that this routine is re-entrant: exec_eval_datum() can call it
	 * back for subscript evaluation, and so there can be a need to have more
	 * than one active param list.
	 */
	if (expr->nparams > 0)
	{
		/* sizeof(ParamListInfoData) includes the first array element */
		paramLI = (ParamListInfo)
			MemoryContextAlloc(econtext->ecxt_per_tuple_memory,
							   sizeof(ParamListInfoData) +
							   (expr->nparams - 1) *sizeof(ParamExternData));
		paramLI->numParams = expr->nparams;

		for (i = 0; i < expr->nparams; i++)
		{
			ParamExternData *prm = &paramLI->params[i];
			PLpgPSM_datum *datum = estate->datums[expr->params[i]];

			prm->pflags = 0;
			exec_eval_datum(estate, datum, expr->plan_argtypes[i],
							&prm->ptype,
							&prm->value, &prm->isnull);
		}
	}
	else
		paramLI = NULL;

	/*
	 * Now we can safely make the econtext point to the param list.
	 */
	econtext->ecxt_param_list_info = paramLI;

	/*
	 * We have to do some of the things SPI_execute_plan would do, in
	 * particular advance the snapshot if we are in a non-read-only function.
	 * Without this, stable functions within the expression would fail to see
	 * updates made so far by our own function.
	 */
	SPI_push();

	oldcontext = MemoryContextSwitchTo(econtext->ecxt_per_tuple_memory);
	if (!estate->readonly_func)
	{
		CommandCounterIncrement();
		PushActiveSnapshot(GetTransactionSnapshot());
	}

	/*
	 * Finally we can call the executor to evaluate the expression
	 */
	*result = ExecEvalExpr(expr->expr_simple_state,
						   econtext,
						   isNull,
						   NULL);
	MemoryContextSwitchTo(oldcontext);

	if (!estate->readonly_func)
		PopActiveSnapshot();

	SPI_pop();

        /*
    	 * Now we can release our refcount on the cached plan.
    	 */
    	ReleaseCachedPlan(cplan, true);
    
	return true;    	
}


/* ----------
 * exec_move_row			Move one tuple's values into a record or row
 * ----------
 */
static void
exec_move_row(PLpgPSM_execstate *estate,
			  PLpgPSM_rec *rec,
			  PLpgPSM_row *row,
			  HeapTuple tup, TupleDesc tupdesc)
{
	/*
	 * Record is simple - just copy the tuple and its descriptor into the
	 * record variable
	 */
	if (rec != NULL)
	{
		/*
		 * copy input first, just in case it is pointing at variable's value
		 */
		if (HeapTupleIsValid(tup))
			tup = heap_copytuple(tup);
		if (tupdesc)
			tupdesc = CreateTupleDescCopy(tupdesc);

		if (rec->freetup)
		{
			heap_freetuple(rec->tup);
			rec->freetup = false;
		}
		if (rec->freetupdesc)
		{
			FreeTupleDesc(rec->tupdesc);
			rec->freetupdesc = false;
		}

		if (HeapTupleIsValid(tup))
		{
			rec->tup = tup;
			rec->freetup = true;
		}
		else if (tupdesc)
		{
			/* If we have a tupdesc but no data, form an all-nulls tuple */
			char	   *nulls;

			nulls = (char *) palloc(tupdesc->natts * sizeof(char));
			memset(nulls, 'n', tupdesc->natts * sizeof(char));

			rec->tup = heap_formtuple(tupdesc, NULL, nulls);
			rec->freetup = true;

			pfree(nulls);
		}
		else
			rec->tup = NULL;

		if (tupdesc)
		{
			rec->tupdesc = tupdesc;
			rec->freetupdesc = true;
		}
		else
			rec->tupdesc = NULL;

		return;
	}

	/*
	 * Row is a bit more complicated in that we assign the individual
	 * attributes of the tuple to the variables the row points to.
	 *
	 * NOTE: this code used to demand row->nfields == tup->t_data->t_natts,
	 * but that's wrong.  The tuple might have more fields than we expected if
	 * it's from an inheritance-child table of the current table, or it might
	 * have fewer if the table has had columns added by ALTER TABLE. Ignore
	 * extra columns and assume NULL for missing columns, the same as
	 * heap_getattr would do.  We also have to skip over dropped columns in
	 * either the source or destination.
	 *
	 * If we have no tuple data at all, we'll assign NULL to all columns of
	 * the row variable.
	 */
	if (row != NULL)
	{
		int			t_natts;
		int			fnum;
		int			anum;

		if (HeapTupleIsValid(tup))
			t_natts = HeapTupleHeaderGetNatts(tup->t_data);
		else
			t_natts = 0;

		anum = 0;
		row->isnull = false;

		for (fnum = 0; fnum < row->nfields; fnum++)
		{
			PLpgPSM_var *var;
			Datum		value;
			bool		isnull;
			Oid			valtype;

			if (row->varnos[fnum] < 0)
				continue;		/* skip dropped column in row struct */

			var = (PLpgPSM_var *) (estate->datums[row->varnos[fnum]]);

			while (anum < t_natts && tupdesc->attrs[anum]->attisdropped)
				anum++;			/* skip dropped column in tuple */

			if (anum < t_natts)
			{
				value = SPI_getbinval(tup, tupdesc, anum + 1, &isnull);
				valtype = SPI_gettypeid(tupdesc, anum + 1);
				anum++;
			}
			else
			{
				value = (Datum) 0;
				isnull = true;
				valtype = InvalidOid;
			}

			exec_assign_value(estate, (PLpgPSM_datum *) var,
							  value, valtype, &isnull);
		}

		return;
	}

	elog(ERROR, "unsupported target");
}

/* ----------
 * make_tuple_from_row		Make a tuple from the values of a row object
 *
 * A NULL return indicates rowtype mismatch; caller must raise suitable error
 * ----------
 */
static HeapTuple
make_tuple_from_row(PLpgPSM_execstate *estate,
					PLpgPSM_row *row,
					TupleDesc tupdesc)
{
	int			natts = tupdesc->natts;
	HeapTuple	tuple;
	Datum	   *dvalues;
	bool	   *nulls;
	int			i;

	if (natts != row->nfields)
		return NULL;

	dvalues = (Datum *) palloc0(natts * sizeof(Datum));
	nulls = (bool *) palloc(natts * sizeof(bool));

	for (i = 0; i < natts; i++)
	{
		Oid			fieldtypeid;

		if (tupdesc->attrs[i]->attisdropped)
		{
			nulls[i] = true;	/* leave the column as null */
			continue;
		}
		if (row->varnos[i] < 0) /* should not happen */
			elog(ERROR, "dropped rowtype entry for non-dropped column");

		exec_eval_datum(estate, estate->datums[row->varnos[i]],
						InvalidOid, &fieldtypeid, &dvalues[i], &nulls[i]);
		if (fieldtypeid != tupdesc->attrs[i]->atttypid)
			return NULL;
	}

	tuple = heap_form_tuple(tupdesc, dvalues, nulls);

	pfree(dvalues);
	pfree(nulls);

	return tuple;
}

/* ----------
 * convert_value_to_string			Convert a non-null Datum to C string
 *
 * Note: callers generally assume that the result is a palloc'd string and
 * should be pfree'd.  This is not all that safe an assumption ...
 *
 * Note: not caching the conversion function lookup is bad for performance.
 * ----------
 */
static char *
convert_value_to_string(Datum value, Oid valtype)
{
	char	*str;
	Oid			typoutput;
	bool		typIsVarlena;

	getTypeOutputInfo(valtype, &typoutput, &typIsVarlena);

	/*
	 * We do SPI_push to allow the datatype output function to use SPI.
	 * However we do not mess around with CommandCounterIncrement or advancing
	 * the snapshot, which means that a stable output function would not see
	 * updates made so far by our own function.  The use-case for such
	 * scenarios seems too narrow to justify the cycles that would be
	 * expended.
	 */
	SPI_push();

	str = OidOutputFunctionCall(typoutput, value);

	SPI_pop();

	return str;
} 

/* ----------
 * exec_cast_value			Cast a value if required
 * ----------
 */
static Datum
exec_cast_value(Datum value, Oid valtype,
				Oid reqtype,
				FmgrInfo *reqinput,
				Oid reqtypioparam,
				int32 reqtypmod,
				bool isnull)
{
	/*
	 * If the type of the queries return value isn't that of the variable,
	 * convert it.
	 */
	if (valtype != reqtype || reqtypmod != -1)
	{
		if (!isnull)
		{
			char	   *extval;

			extval = convert_value_to_string(value, valtype);

			SPI_push();			

			value = InputFunctionCall(reqinput, extval,
									  reqtypioparam, reqtypmod);
			SPI_pop();

			pfree(extval);
		}
		else
		{
			SPI_push();		    
    
			value = InputFunctionCall(reqinput, NULL,
									  reqtypioparam, reqtypmod);

			SPI_pop();
		}
	}

	return value;
}

/* ----------
 * exec_simple_cast_value			Cast a value if required
 *
 * As above, but need not supply details about target type.  Note that this
 * is slower than exec_cast_value with cached type info, and so should be
 * avoided in heavily used code paths.
 * ----------
 */
static Datum
exec_simple_cast_value(Datum value, Oid valtype,
					   Oid reqtype, int32 reqtypmod,
					   bool isnull)
{
	if (!isnull)
	{
		if (valtype != reqtype || reqtypmod != -1)
		{
			Oid			typinput;
			Oid			typioparam;
			FmgrInfo	finfo_input;

			getTypeInputInfo(reqtype, &typinput, &typioparam);

			fmgr_info(typinput, &finfo_input);

			value = exec_cast_value(value,
									valtype,
									reqtype,
									&finfo_input,
									typioparam,
									reqtypmod,
									isnull);
		}
	}

	return value;
}


/* ----------
 * exec_simple_check_node -		Recursively check if an expression
 *								is made only of simple things we can
 *								hand out directly to ExecEvalExpr()
 *								instead of calling SPI.
 * ----------
 */
static bool
exec_simple_check_node(Node *node)
{
	if (node == NULL)
		return TRUE;

	switch (nodeTag(node))
	{
		case T_Const:
			return TRUE;

		case T_Param:
			return TRUE;

		case T_ArrayRef:
			{
				ArrayRef   *expr = (ArrayRef *) node;

				if (!exec_simple_check_node((Node *) expr->refupperindexpr))
					return FALSE;
				if (!exec_simple_check_node((Node *) expr->reflowerindexpr))
					return FALSE;
				if (!exec_simple_check_node((Node *) expr->refexpr))
					return FALSE;
				if (!exec_simple_check_node((Node *) expr->refassgnexpr))
					return FALSE;

				return TRUE;
			}

		case T_FuncExpr:
			{
				FuncExpr   *expr = (FuncExpr *) node;

				if (expr->funcretset)
					return FALSE;
				if (!exec_simple_check_node((Node *) expr->args))
					return FALSE;

				return TRUE;
			}

		case T_OpExpr:
			{
				OpExpr	   *expr = (OpExpr *) node;

				if (expr->opretset)
					return FALSE;
				if (!exec_simple_check_node((Node *) expr->args))
					return FALSE;

				return TRUE;
			}

		case T_DistinctExpr:
			{
				DistinctExpr *expr = (DistinctExpr *) node;

				if (expr->opretset)
					return FALSE;
				if (!exec_simple_check_node((Node *) expr->args))
					return FALSE;

				return TRUE;
			}

		case T_ScalarArrayOpExpr:
			{
				ScalarArrayOpExpr *expr = (ScalarArrayOpExpr *) node;

				if (!exec_simple_check_node((Node *) expr->args))
					return FALSE;

				return TRUE;
			}

		case T_BoolExpr:
			{
				BoolExpr   *expr = (BoolExpr *) node;

				if (!exec_simple_check_node((Node *) expr->args))
					return FALSE;

				return TRUE;
			}

		case T_FieldSelect:
			return exec_simple_check_node((Node *) ((FieldSelect *) node)->arg);

		case T_FieldStore:
			{
				FieldStore *expr = (FieldStore *) node;

				if (!exec_simple_check_node((Node *) expr->arg))
					return FALSE;
				if (!exec_simple_check_node((Node *) expr->newvals))
					return FALSE;

				return TRUE;
			}

		case T_RelabelType:
			return exec_simple_check_node((Node *) ((RelabelType *) node)->arg);
			
		case T_CoerceViaIO:
            		return exec_simple_check_node((Node *) ((CoerceViaIO *) node)->arg);
			
		case T_ArrayCoerceExpr:
			return exec_simple_check_node((Node *) ((ArrayCoerceExpr *) node)->arg);

		case T_ConvertRowtypeExpr:
			return exec_simple_check_node((Node *) ((ConvertRowtypeExpr *) node)->arg);

		case T_CaseExpr:
			{
				CaseExpr   *expr = (CaseExpr *) node;

				if (!exec_simple_check_node((Node *) expr->arg))
					return FALSE;
				if (!exec_simple_check_node((Node *) expr->args))
					return FALSE;
				if (!exec_simple_check_node((Node *) expr->defresult))
					return FALSE;

				return TRUE;
			}

		case T_CaseWhen:
			{
				CaseWhen   *when = (CaseWhen *) node;

				if (!exec_simple_check_node((Node *) when->expr))
					return FALSE;
				if (!exec_simple_check_node((Node *) when->result))
					return FALSE;

				return TRUE;
			}

		case T_CaseTestExpr:
			return TRUE;

		case T_ArrayExpr:
			{
				ArrayExpr  *expr = (ArrayExpr *) node;

				if (!exec_simple_check_node((Node *) expr->elements))
					return FALSE;

				return TRUE;
			}

		case T_RowExpr:
			{
				RowExpr    *expr = (RowExpr *) node;

				if (!exec_simple_check_node((Node *) expr->args))
					return FALSE;

				return TRUE;
			}
		case T_RowCompareExpr:
			{
				RowCompareExpr *expr = (RowCompareExpr *) node;

				if (!exec_simple_check_node((Node *) expr->largs))
					return FALSE;
				if (!exec_simple_check_node((Node *) expr->rargs))
					return FALSE;

				return TRUE;
			}
		case T_CoalesceExpr:
			{
				CoalesceExpr *expr = (CoalesceExpr *) node;

				if (!exec_simple_check_node((Node *) expr->args))
					return FALSE;

				return TRUE;
			}

		case T_MinMaxExpr:
			{
				MinMaxExpr *expr = (MinMaxExpr *) node;

				if (!exec_simple_check_node((Node *) expr->args))
					return FALSE;

				return TRUE;
			}

		case T_NullIfExpr:
			{
				NullIfExpr *expr = (NullIfExpr *) node;

				if (expr->opretset)
					return FALSE;
				if (!exec_simple_check_node((Node *) expr->args))
					return FALSE;

				return TRUE;
			}

		case T_NullTest:
			return exec_simple_check_node((Node *) ((NullTest *) node)->arg);

		case T_BooleanTest:
			return exec_simple_check_node((Node *) ((BooleanTest *) node)->arg);

		case T_CoerceToDomain:
			return exec_simple_check_node((Node *) ((CoerceToDomain *) node)->arg);

		case T_CoerceToDomainValue:
			return TRUE;

		case T_List:
			{
				List	   *expr = (List *) node;
				ListCell   *l;

				foreach(l, expr)
				{
					if (!exec_simple_check_node(lfirst(l)))
						return FALSE;
				}

				return TRUE;
			}

		default:
			return FALSE;
	}
}


/* ----------
 * exec_simple_check_plan -		Check if a plan is simple enough to
 *								be evaluated by ExecEvalExpr() instead
 *								of SPI.
 * ----------
 */
static void
exec_simple_check_plan(PLpgPSM_expr *expr)
{
	Plan	   *plan;
	TargetEntry *tle;
	PlannedStmt *stmt;
	CachedPlanSource *plansource;
	
	expr->expr_simple_generation = 0;
	expr->expr_simple_expr = NULL;

	/*
	 * 1. We can only evaluate queries that resulted in one single execution
	 * plan
	 */

	if (list_length(expr->plan->plancache_list) != 1)
		return;
		
	plansource = (CachedPlanSource *) linitial(expr->plan->plancache_list);
	expr->expr_simple_generation = plansource->generation;
	if (list_length(plansource->plan->stmt_list) != 1)
		return;
		
	stmt = (PlannedStmt *) linitial(plansource->plan->stmt_list);

	if (!IsA(stmt, PlannedStmt))
		return;
	plan = stmt->planTree;

	/*
	 * 2. It must be a RESULT plan --> no scan's required
	 */
	if (!IsA(plan, Result))
		return;

	/*
	 * 3. Can't have any subplan or qual clause, either
	 */
	if (plan->lefttree != NULL ||
		plan->righttree != NULL ||
		plan->initPlan != NULL ||
		plan->qual != NULL ||
		((Result *) plan)->resconstantqual != NULL)
		return;

	/*
	 * 4. The plan must have a single attribute as result
	 */
	if (list_length(plan->targetlist) != 1)
		return;

	tle = (TargetEntry *) linitial(plan->targetlist);

	/*
	 * 5. Check that all the nodes in the expression are non-scary.
	 */
	if (!exec_simple_check_node((Node *) tle->expr))
		return;

	/*
	 * Yes - this is a simple expression.  Mark it as such, and initialize
	 * state to "not valid in current transaction".
	 */
	expr->expr_simple_expr = tle->expr;
	expr->expr_simple_state = NULL;
	expr->expr_simple_id = -1;

	/* Also stash away the expression result type */
	expr->expr_simple_type = exprType((Node *) tle->expr);
}

/*
 * Check two tupledescs have matching number and types of attributes
 */
static bool
compatible_tupdesc(TupleDesc td1, TupleDesc td2)
{
	int			i;

	if (td1->natts != td2->natts)
		return false;

	for (i = 0; i < td1->natts; i++)
	{
		if (td1->attrs[i]->atttypid != td2->attrs[i]->atttypid)
			return false;
	}

	return true;
}

/* ----------
 * exec_set_found			Set the global found variable
 *					to true/false
 * ----------
 */
static void
exec_set_found(PLpgPSM_execstate *estate, bool state)
{
	PLpgPSM_var *var;

	var = (PLpgPSM_var *) (estate->datums[estate->found_varno]);
	var->value = (Datum) state;
	var->isnull = false;
}



static void
exec_signal_not_found(bool not_found)
{
	if (not_found)
	{
		if (wdata)
		{
			sec_free(wdata->message);
			sec_free(wdata->detail);
			sec_free(wdata->hint);
			sec_free(wdata->context);

			pfree(wdata);
			wdata = NULL;
		}

		wdata = plpgpsm_warning_filter(ERRCODE_NO_DATA, "no data", NULL, NULL, NULL);
	}
}


/*                                                                                                     
 * plpgpsm_create_econtext --- create an eval_econtext for the current function                        
 *                                                                                                     
 * We may need to create a new eval_estate too, if there's not one already                             
 * for the current (sub) transaction.  The EState will be cleaned up at                                
 * (sub) transaction end.                                                                              
 */                                                                                                    
static void                                                                                            
plpgpsm_create_econtext(PLpgPSM_execstate *estate)                                                     
{                                                                                                      
        SubTransactionId my_subxid = GetCurrentSubTransactionId();                                     
        SimpleEstateStackEntry *entry = simple_estate_stack;                                           
                                                                                                       
        /* Create new EState if not one for current subxact */                                         
        if (entry == NULL ||                                                                           
                entry->xact_subxid != my_subxid)                                                       
        {                                                                                              
                MemoryContext oldcontext;                                                              
                                                                                                       
                /* Stack entries are kept in TopTransactionContext for simplicity */                   
                entry = (SimpleEstateStackEntry *)                                                     
                        MemoryContextAlloc(TopTransactionContext,                                      
                                                           sizeof(SimpleEstateStackEntry));            
                                                                                                       
                /* But each EState should be a child of its CurTransactionContext */                   
                oldcontext = MemoryContextSwitchTo(CurTransactionContext);                             
                entry->xact_eval_estate = CreateExecutorState();
                                       
                MemoryContextSwitchTo(oldcontext);                                                     
                                                                                                       
                /* Assign a reasonably-unique ID to this EState */                                     
                entry->xact_estate_simple_id = simple_estate_id_counter++;                             
                entry->xact_subxid = my_subxid;                                                        
                                                                                                       
                entry->next = simple_estate_stack;                                                     
                simple_estate_stack = entry;                                                           
        }                                                                                              
                                                                                                       
        /* Link plpgpsm estate to it */                                                                
        estate->eval_estate = entry->xact_eval_estate;                                                 
        estate->eval_estate_simple_id = entry->xact_estate_simple_id;                                  
                                                                                                       
        /* And create a child econtext for the current function */                                     
        estate->eval_econtext = CreateExprContext(estate->eval_estate);                                
}   

/*                                                                                                     
 * plpgpsm_xact_cb --- post-transaction-commit-or-abort cleanup                                        
 *                                                                                                     
 * If a simple-expression EState was created in the current transaction,                               
 * it has to be cleaned up.                                                                            
 */                                                                                                    
void                                                                                                   
plpgpsm_xact_cb(XactEvent event, void *arg)                                                            
{                                                                                                      
        /*                                                                                             
         * If we are doing a clean transaction shutdown, free the EState (so that                      
         * any remaining resources will be released correctly). In an abort, we                        
         * expect the regular abort recovery procedures to release everything of                       
         * interest.  We don't need to free the individual stack entries since                         
         * TopTransactionContext is about to go away anyway.                                           
         *                                                                                             
         * Note: if plpgpsm_subxact_cb is doing its job, there should be at most                       
         * one stack entry, but we may as well code this as a loop.                                    
         */                                                                                            
        if (event != XACT_EVENT_ABORT)                                                                 
        {                       
                while (simple_estate_stack != NULL)                                                    
                {               
                        FreeExecutorState(simple_estate_stack->xact_eval_estate);                      
                        simple_estate_stack = simple_estate_stack->next;                               
                }                                                                                      
        }                                                                                              
        else                                                                                           
                simple_estate_stack = NULL;
}                                                                                                      

/*                                                                                                     
 * plpgpsm_subxact_cb --- post-subtransaction-commit-or-abort cleanup                                  
 *                                                                                                     
 * If a simple-expression EState was created in the current subtransaction,                            
 * it has to be cleaned up.                                                                            
 */                                                                                                    
void                                                                                                   
plpgpsm_subxact_cb(SubXactEvent event, SubTransactionId mySubid,                                       
                                   SubTransactionId parentSubid, void *arg)                            
{                                                                                                      
        if (event == SUBXACT_EVENT_START_SUB)                                                          
                return;                                                                                
                                                                                                       
        if (simple_estate_stack != NULL &&                                                             
                simple_estate_stack->xact_subxid == mySubid)                                           
        {                                                                                              
                SimpleEstateStackEntry *next;                                                          
                                                                                                       
                if (event == SUBXACT_EVENT_COMMIT_SUB) 
                        FreeExecutorState(simple_estate_stack->xact_eval_estate);                      

                next = simple_estate_stack->next;                                                      
                pfree(simple_estate_stack);                                                            
                simple_estate_stack = next;                                                            
        }                                                                                              
}                                                    

static void
free_var(PLpgPSM_var *var)
{
	if (var->freeval)
	{
		pfree(DatumGetPointer(var->value));
		var->freeval = false;
	}
}



static void
clean_warning_data(PLpgPSM_execstate *estate)
{
	if (estate->edata)
	    FreeErrorData(estate->edata);

	warning_action = NULL;
	if (wdata)
	{
		sec_free(wdata->message);
		sec_free(wdata->detail);
		sec_free(wdata->hint);
		sec_free(wdata->context);

		pfree(wdata);
		wdata = NULL;
	}
}

static PLpgPSM_prepared_plan *
get_plan(const char *name)
{
	planHashEnt *hentry;
	char key[NAMEDATALEN];
	
	if (!planHash)
		planHash = create_plan_hash();
		
	MemSet(key, 0, NAMEDATALEN);
	snprintf(key, NAMEDATALEN - 1, "%s", name);
	
	hentry = (planHashEnt *) hash_search(planHash, key,
							    HASH_FIND, NULL);
	if (hentry)
		return hentry->preplan;
	
	return NULL;
}

static HTAB *
create_plan_hash(void)
{
	HASHCTL		ctl;
	HTAB	*ptr;
	
	ctl.keysize = NAMEDATALEN;
	ctl.entrysize = sizeof(planHashEnt);
	
	ptr = hash_create("plan store", NUMSTR, &ctl, HASH_ELEM);
	
	if (!ptr)
		ereport(ERROR,
			    (errcode(ERRCODE_OUT_OF_MEMORY),
			     errmsg("out of memory")));
	return (ptr);
}

static void 
store_plan(const char *name, PLpgPSM_prepared_plan *plan)
{
	planHashEnt	*hentry;
	bool	found;
	char	key[NAMEDATALEN];
	
	if (!planHash)
		planHash = create_plan_hash();
		
	MemSet(key, 0, NAMEDATALEN);
	snprintf(key, NAMEDATALEN - 1, "%s", name);
	hentry =  hash_search(planHash, key, HASH_ENTER,  &found);
	
	if (!hentry)
		    elog(ERROR, "out of memory");

	/* hangs */
	/* if (found && hentry->preplan)
		SPI_freeplan(hentry->preplan); */
	
	hentry->preplan = plan;
	strncpy(hentry->name, name, NAMEDATALEN - 1);
}


/* do copy diagnostic area and add its to stack */
static PLpgPSM_diag_area *
push_diagnostics_error(PLpgPSM_execstate *estate, ErrorData  *edata)
{
	PLpgPSM_diag_area *c = estate->current_diag;
	PLpgPSM_diag_area *new = palloc0(sizeof(PLpgPSM_diag_area));

	c->sqlstate = edata->sqlerrcode;
	c->elevel = edata->elevel;

	c->fields[PLPGPSM_GETDIAG_MESSAGE] = edata->message;
	c->fields[PLPGPSM_GETDIAG_HINT] = edata->hint;
	c->fields[PLPGPSM_GETDIAG_DETAIL] = edata->detail;

	if (edata->message)
		c->message_length = strlen(edata->message);

	if (edata->sqlerrcode == CONDITION)
		c->fields[PLPGPSM_GETDIAG_CONDITION] = edata->message;

	/* copy diagnostic area to current diagnostic are */
	memcpy(new, c, sizeof(PLpgPSM_diag_area));

	new->prev = c;
	c->next = new;

	estate->current_diag = new;
	    
	return c;
}


static PLpgPSM_diag_area *
push_diagnostics_warning(PLpgPSM_execstate *estate, WarningData *w)
{
	PLpgPSM_diag_area *c = estate->current_diag;
	PLpgPSM_diag_area *new = palloc0(sizeof(PLpgPSM_diag_area));

	c->sqlstate = w->sqlstate;

	if (ERRCODE_TO_CATEGORY(w->sqlstate) == ERRCODE_SUCCESSFUL_COMPLETION)
		c->elevel = INFO;
	else if (ERRCODE_TO_CATEGORY(w->sqlstate) == ERRCODE_WARNING)
		c->elevel = WARNING;

	c->fields[PLPGPSM_GETDIAG_MESSAGE] = w->message;
	c->fields[PLPGPSM_GETDIAG_HINT] = w->hint;
	c->fields[PLPGPSM_GETDIAG_DETAIL] = w->detail;

	if (w->message)
		c->message_length = strlen(w->message);

	if (w->sqlstate == CONDITION)
		c->fields[PLPGPSM_GETDIAG_CONDITION] = w->message;

	/* copy diagnostic area to current diagnostic are */
	memcpy(new, c, sizeof(PLpgPSM_diag_area));

	new->prev = c;
	c->next = new;

	estate->current_diag = new;

	return c;
}

/* free deeper diagnostics areas */
static PLpgPSM_diag_area *
pop_and_clean_diagnostics(PLpgPSM_diag_area *c)
{
	PLpgPSM_diag_area *next = c->next;
	PLpgPSM_diag_area *aux;

	while (next)
	{
		aux = next;
		next = next->next;
		pfree(aux);
	}

	/* after returning exception all have to be right */
	reset_diagnostic(c);
	c->next = NULL;

	return c;
}

static void reset_diagnostic(PLpgPSM_diag_area *c)
{
	c->sqlstate 		= SUCCESSFULL_COMPLETION;
	c->message_length 	= 0;
	c->fields[PLPGPSM_GETDIAG_MESSAGE] 	= NULL;
	c->fields[PLPGPSM_GETDIAG_HINT] 	= NULL;
	c->fields[PLPGPSM_GETDIAG_DETAIL] 	= NULL;
	c->fields[PLPGPSM_GETDIAG_CONDITION] 	= NULL;
	c->elevel 		= 0;
}

