/*-------------------------------------------------------------------------
 *
 * plpgpsm.h		- Definitions for the PL/pgSQL
 *			  procedural language
 *
 * Portions Copyright (c) 1996-2006, PostgreSQL Global Development Group
 * Portions Copyright (c) 1994, Regents of the University of California
 *
 *
 * IDENTIFICATION
 *	  $PostgreSQL: pgsql/src/pl/plpgpsm/src/plpgpsm.h,v 1.81 2006/10/04 00:30:14 momjian Exp $
 *
 *-------------------------------------------------------------------------
 */

#ifndef PLPGPSM_H
#define PLPGPSM_H

#include "postgres.h"

#include "fmgr.h"
#include "miscadmin.h"
#include "commands/trigger.h"
#include "executor/spi.h"
#include "utils/tuplestore.h"


/**********************************************************************
 * Definitions
 **********************************************************************/

/* define our text domain for translations */
#undef TEXTDOMAIN
#define TEXTDOMAIN PG_TEXTDOMAIN("plpgsql")

/* ----------
 * Compiler's namestack item types
 * ----------
 */
enum
{
	PLPGPSM_NSTYPE_LABEL,
	PLPGPSM_NSTYPE_VAR,
	PLPGPSM_NSTYPE_ROW,
	PLPGPSM_NSTYPE_REC,
	PLPGPSM_NSTYPE_CONDITION,
	PLPGPSM_NSTYPE_PLAN
};

/* ----------
 * Datum array node types
 * ----------
 */
enum
{
	PLPGPSM_DTYPE_VAR,
	PLPGPSM_DTYPE_ROW,
	PLPGPSM_DTYPE_REC,
	PLPGPSM_DTYPE_RECFIELD,
	PLPGPSM_DTYPE_ARRAYELEM,
	PLPGPSM_DTYPE_EXPR,
	PLPGPSM_DTYPE_TRIGARG,
	PLPGPSM_DTYPE_CONDITION,
	PLPGPSM_DTYPE_PLAN
};

/* ----------
 * Variants distinguished in PLpgPSM_type structs
 * ----------
 */
enum
{
	PLPGPSM_TTYPE_SCALAR,		/* scalar types and domains */
	PLPGPSM_TTYPE_ROW,			/* composite types */
	PLPGPSM_TTYPE_REC,			/* RECORD pseudotype */
	PLPGPSM_TTYPE_CONDITION,		/* condition type */
	PLPGPSM_TTYPE_PLAN,
	PLPGPSM_TTYPE_PSEUDO		/* other pseudotypes */
};

/* ----------
 * Execution tree node types
 * ----------
 */
enum
{
	PLPGPSM_STMT_BLOCK,
	PLPGPSM_STMT_ASSIGN,
	PLPGPSM_STMT_IF,
	PLPGPSM_STMT_LOOP,
	PLPGPSM_STMT_FORS,
	PLPGPSM_STMT_LEAVE,
	PLPGPSM_STMT_RETURN,
	PLPGPSM_STMT_SIGNAL,
	PLPGPSM_STMT_EXECUTE,
	PLPGPSM_STMT_SQL,
	PLPGPSM_STMT_GETDIAG,
	PLPGPSM_STMT_OPEN,
	PLPGPSM_STMT_FETCH,
	PLPGPSM_STMT_CLOSE,
	PLPGPSM_STMT_PREPARE,
	PLPGPSM_STMT_PRINT,
	PLPGPSM_STMT_WHEN_CLAUSE,
	PLPGPSM_STMT_CASE,
	PLPGPSM_STMT_CALL,
};


enum
{
	PLPGPSM_CYCLE_LOOP,
	PLPGPSM_CYCLE_WHILE,
	PLPGPSM_CYCLE_REPEAT,
};


/* ----------
 * Execution node return codes
 * ----------
 */
enum
{
	PLPGPSM_RC_OK,
	PLPGPSM_RC_EXIT,
	PLPGPSM_RC_RETURN,
	PLPGPSM_RC_CONTINUE,
	PLPGPSM_RC_UNDO
};

/* ----------
 * GET DIAGNOSTICS system attrs
 * ----------
 */
enum
{
	PLPGPSM_GETDIAG_ROW_COUNT,
	PLPGPSM_GETDIAG_RESULT_OID,
	PLPGPSM_GETDIAG_MESSAGE,
	PLPGPSM_GETDIAG_MESSAGE_LENGTH,
	PLPGPSM_GETDIAG_HINT,
	PLPGPSM_GETDIAG_DETAIL,
	PLPGPSM_GETDIAG_CLASS_ORIGIN,
	PLPGPSM_GETDIAG_SUBCLASS_ORIGIN,
	PLPGPSM_GETDIAG_RETURNED_SQLSTATE,
	PLPGPSM_GETDIAG_CONSTRAINT_NAME,
	PLPGPSM_GETDIAG_CONSTRAINT_SCHEMA,
	PLPGPSM_GETDIAG_RESULT,
	PLPGPSM_GETDIAG_EXCEPTION_RW,
	PLPGPSM_GETDIAG_EXCEPTION_RO,
	PLPGPSM_GETDIAG_CURRENT,
	PLPGPSM_GETDIAG_STACKED,
	PLPGPSM_GETDIAG_UNDEF,
	PLPGPMS_COMMAND_FUNCTION,
	PLPGPSM_GETDIAG_CONDITION
};

enum
{
	PLPGPSM_ERRLEVEL_NOTICE,
	PLPGPSM_ERRLEVEL_WARNING,
	PLPGPSM_ERRLEVEL_ERROR,
	PLPGPSM_ERRLEVEL_UNKNOWN
};

enum
{
	PLPGPSM_SINFO_MESSAGE,
	PLPGPSM_SINFO_LEVEL,
	PLPGPSM_SINFO_HINT,
	PLPGPSM_SINFO_DETAIL
};


/*
 * Handler's type
 *
 */
enum
{
	PLPGPSM_HDL_CONTINUE,
	PLPGPSM_HDL_EXIT,
	PLPGPSM_HDL_UNDO
};


enum
{
	PLPGPSM_DECL_CONDITION = 0,
	PLPGPSM_DECL_VARIABLES = 1,
	PLPGPSM_DECL_CURSOR    = 2,
	PLPGPSM_DECL_HANDLER   = 3
};




enum
{
	PLPGPSM_DSQLTYPE_EXEC_PREPARE,
	PLPGPSM_DSQLTYPE_EXEC_IMMEDIATE
};


enum
{
	PLPGPSM_INITPAR_LAST_VARIABLE,
	PLPGPSM_INITPAR_LAST_CURSOR,
	PLPGPSM_INITPAR_LAST_HANDLER,
	PLPGPSM_INITPAR_LAST_CONDITION
};



/**********************************************************************
 * Node and structure definitions
 **********************************************************************/


typedef struct
{								/* Dynamic string control structure */
	int			alloc;
	int			used;			/* Including NUL terminator */
	char	   *value;
} PLpgPSM_dstring;


#define PLPGPSM_INVALID_VARNO	-1
#define PLPGPSM_EXCEPTION_DEEP_INVALID -1


typedef struct
{
	int nparams;
	int mparams;
	int params[1];
} PLpgPSM_init_params;


typedef struct
{								/* Postgres data type */
	char	   *typname;		/* (simple) name of the type */
	Oid			typoid;			/* OID of the data type */
	int			ttype;			/* PLPGPSM_TTYPE_ code */
	int16		typlen;			/* stuff copied from its pg_type entry */
	bool		typbyval;
	Oid			typrelid;
	Oid			typioparam;
	FmgrInfo	typinput;		/* lookup info for typinput function */
	int32		atttypmod;		/* typmod (taken from someplace else) */
} PLpgPSM_type;


/*
 * PLpgPSM_datum is the common supertype for PLpgPSM_expr, PLpgPSM_var,
 * PLpgPSM_row, PLpgPSM_rec, PLpgPSM_recfield, PLpgPSM_arrayelem, and
 * PLpgPSM_trigarg
 */
typedef struct
{								/* Generic datum array item		*/
	int			dtype;
	int			dno;
} PLpgPSM_datum;

/*
 * The variants PLpgPSM_var, PLpgPSM_row, and PLpgPSM_rec share these
 * fields
 */
typedef struct
{								/* Scalar or composite variable */
	int			dtype;
	int			dno;
	char	   *refname;
	int			lineno;
} PLpgPSM_variable;



typedef struct PLpgPSM_expr
{								/* SQL Query to plan and execute	*/
	int			dtype;
	int			dno;
	char	   *query;
	SPIPlanPtr	plan;
	Oid		   *plan_argtypes;
	/* fields for "simple expression" fast-path execution: */
	Expr	   *expr_simple_expr;		/* NULL means not a simple expr */
	int		expr_simple_generation; /* plancache generation we checked */
	Oid		expr_simple_type;	/* result type Oid, if simple */	

        /*                                                                                             
         * if expr is simple AND prepared in current eval_estate,                                      
         * expr_simple_state is valid.  Test validity by seeing if expr_simple_id                      
         * matches eval_estate_simple_id.                                                              
         */                                                                                            
        ExprState  *expr_simple_state;
        long int        expr_simple_id;    

	/* params to pass to expr */
	int			nparams;
	int			params[1];		/* VARIABLE SIZE ARRAY ... must be last */
} PLpgPSM_expr;
    

typedef struct
{							/* Prepared plan variable */
	int			dtype;
	int			dno;
	char	*refname;
	void 	*plan;
	int			nparams;
	int 			params[1];
} PLpgPSM_plan;


typedef struct
{								/* Row variable */
	int			dtype;
	int			dno;
	char	   *refname;
	int			lineno;
	bool		isnull;
	TupleDesc	rowtupdesc;

	/*
	 * Note: TupleDesc is only set up for named rowtypes, else it is NULL.
	 *
	 * Note: if the underlying rowtype contains a dropped column, the
	 * corresponding fieldnames[] entry will be NULL, and there is no
	 * corresponding var (varnos[] will be -1).
	 */
	int			nfields;
	char	  **fieldnames;
	int		   *varnos;
} PLpgPSM_row;


typedef struct
{								/* Scalar variable */
	int			dtype;
	int			dno;
	char	   *refname;
	int			lineno;

	PLpgPSM_type *datatype;
	int			isconst;
	int			notnull;
	PLpgPSM_expr *default_val;
	PLpgPSM_expr *cursor_explicit_expr;
	int			cursor_explicit_argrow;
	int			cursor_options;	
	bool			holdable;
	Portal			portal;
	Datum		value;
	bool		isnull;
	bool		freeval;
	char		*prepname;
	PLpgPSM_row		*parent;
} PLpgPSM_var;


typedef struct
{								/* Record variable (non-fixed structure) */
	int			dtype;
	int			dno;
	char	   *refname;
	int			lineno;

	HeapTuple	tup;
	TupleDesc	tupdesc;
	bool		freetup;
	bool		freetupdesc;
} PLpgPSM_rec;


typedef struct
{								/* Field in record */
	int			dtype;
	int			dno;
	char	   *fieldname;
	int			recparentno;	/* dno of parent record */
} PLpgPSM_recfield;


typedef struct
{								/* Element of array variable */
	int			dtype;
	int			dno;
	PLpgPSM_expr *subscript;
	int			arrayparentno;	/* dno of parent array variable */
} PLpgPSM_arrayelem;


typedef struct
{								/* Positional argument to trigger	*/
	int			dtype;
	int			dno;
	PLpgPSM_expr *argnum;
} PLpgPSM_trigarg;


typedef struct
{
	int		item_type;
	int		ival;
	PLpgPSM_expr	*expr;
} PLpgPSM_signal_info;


typedef struct
{								/* Item in the compilers namestack	*/
	int			itemtype;
	int			itemno;
	char		name[1];
} PLpgPSM_nsitem;


/* XXX: consider adapting this to use List */
typedef struct PLpgPSM_ns
{								/* Compiler namestack level		*/
	int			items_alloc;
	int			items_used;
	PLpgPSM_nsitem **items;
	struct PLpgPSM_ns *upper;
} PLpgPSM_ns;


typedef struct
{								/* Generic execution node		*/
	int			cmd_type;
	int			lineno;
} PLpgPSM_stmt;


typedef struct PLpgPSM_condition
{						/* One condition  */
	int			sqlerrstate;	/* SQLSTATE code */
	char	   *condname;		/* condition name */
	struct PLpgPSM_condition *next;
	bool	general_handler;
	bool	handle_errors;
} PLpgPSM_condition;


typedef struct PLpgPSM_cond_var
{						/* One condition  */
	int	dtype;
	int	cno;		
	int	lineno;		
	int		sqlerrstate;	/* SQLSTATE code */
} PLpgPSM_cond_var;


typedef struct
{
	int		sqlstate_varno;	
	bool	   has_continue_handler;
	bool	   has_undo_handler;
	bool	   has_exit_handler;
	bool	   has_exception_handler;
	List		   *exc_list;		/* List of WHEN clauses */	
} PLpgPSM_exception_block;


typedef struct
{								/* One EXCEPTION ... WHEN clause */
	int			lineno;
	int			handler_type;
	PLpgPSM_condition *conditions;
	PLpgPSM_stmt	   *action;			/* List of statements */
	bool	handle_errors;
} PLpgPSM_exception;


typedef struct
{								/* Block of statements			*/
	int			cmd_type;
	int			lineno;
	bool			generated;			/* true if is generated */
	char	   *label;
	List	   *body;			/* List of statements */
	int			n_initvars;
	int		   *initvarnos;
	bool	atomic;
	int		sqlstate_varno;
	PLpgPSM_exception_block *exceptions;
	List	*opened_cursors;		/* Start of list of opened cursors from this block */
} PLpgPSM_stmt_block;


typedef struct
{								/* Assign statement			*/
	int			lineno;
	int			varno;
	PLpgPSM_expr *expr;
} PLpgPSM_assign_item;


typedef struct
{								/* Assign statement			*/
	int			cmd_type;
	int			lineno;
	List			*items;
	char		*varname;
	PLpgPSM_expr *expr;
} PLpgPSM_stmt_assign;


typedef struct
{								/* PERFORM statement		*/
	int			cmd_type;
	int			lineno;
	PLpgPSM_expr *expr;
} PLpgPSM_stmt_call;


typedef struct
{								/* Get Diagnostics item		*/
	int			kind;			/* id for diagnostic value desired */
	int			target;			/* where to assign it */
} PLpgPSM_diag_item;


typedef struct
{								/* Get Diagnostics statement		*/
	int			cmd_type;
	int			lineno;
	bool	stacked;
	int	deep;
	int	deep_varno;
	List	   *diag_items;		/* List of PLpgPSM_diag_item */
} PLpgPSM_stmt_getdiag;


typedef struct
{
	int		cmd_type;
	int		lineno;
	char	*name;
	PLpgPSM_expr *expr;
	int	nparams;
	Oid	params[1];
} PLpgPSM_stmt_prepare;


typedef struct
{
	char	*query;
	void 	*plan;
	int	nparams;
	Oid	params[1];
} PLpgPSM_prepared_plan;


typedef struct
{								/* IF statement				*/
	int			cmd_type;
	int			lineno;
	PLpgPSM_expr *cond;
	List	   *true_body;		/* List of statements */
	List	   *false_body;		/* List of statements */
} PLpgPSM_stmt_if;


typedef struct
{								/* WHILE cond LOOP statement		*/
	int			cmd_type;
	int			lineno;
	char	   *label;
	int	   ctype;
	PLpgPSM_expr *cond;
	List	   *body;			/* List of statements */
} PLpgPSM_stmt_loop;


typedef struct
{								/* FOR statement running over SELECT	*/
	int			cmd_type;
	int			lineno;
	char	   *label;
	char	   *cursor_name;	
	PLpgPSM_rec *rec;
	PLpgPSM_row *row;
	PLpgPSM_expr *query;
	List	   *body;			/* List of statements */
} PLpgPSM_stmt_fors;


typedef struct
{								/* OPEN a curvar					*/
	int			cmd_type;
	int			lineno;
	int			curvar;
	PLpgPSM_expr *argquery;
	PLpgPSM_expr *args;
} PLpgPSM_stmt_open;


typedef struct
{								/* FETCH curvar INTO statement		*/
	int			cmd_type;
	int			lineno;
	PLpgPSM_rec *rec;					/* target, as record or row */
	PLpgPSM_row *row;
	int			curvar;				/* cursor variable to fetch from */
	FetchDirection		direction;			/* fetch direction */
	int			how_many;		/* count, if constant (expr is NULL) */
	PLpgPSM_expr	*expr;				/* count, if expression */
	bool		is_move;			/* is MOVE statemenent or not */
} PLpgPSM_stmt_fetch;


typedef struct
{								/* CLOSE curvar						*/
	int			cmd_type;
	int			lineno;
	int			curvar;
} PLpgPSM_stmt_close;


typedef struct					/* CASE statement */
{
	int			cmd_type;
	int			lineno;
	PLpgPSM_expr *t_expr;		/* test expression, or NULL if none */
	int			t_varno;		/* var to store test expression value into */
	List	   *case_when_list;	/* List of PLpgSQL_case_when structs */
	bool		have_else;		/* flag needed because list could be empty */
	List	   *else_stmts;		/* List of statements */
} PLpgPSM_stmt_case;

typedef struct					/* one arm of CASE statement */
{
	int			lineno;
	PLpgPSM_expr *expr;			/* boolean expression for this case */
	List	   *stmts;			/* List of statements */
} PLpgPSM_case_when;


typedef struct
{								/* EXIT or CONTINUE statement			*/
	int			cmd_type;
	int			lineno;
	bool		is_leave;		/* Is this an exit or a continue? */
	char	   *label;
} PLpgPSM_stmt_leave;


typedef struct
{								/* RETURN statement			*/
	int			cmd_type;
	int			lineno;
	PLpgPSM_expr *expr;
	int			retvarno;
	bool		is_tblexpr;
} PLpgPSM_stmt_return;


typedef struct
{								/* RETURN NEXT statement */
	int			cmd_type;
	int			lineno;
	PLpgPSM_expr *expr;
	int			retvarno;
} PLpgPSM_stmt_return_next;


typedef struct
{								/* RAISE statement			*/
	int			cmd_type;
	int			lineno;
	int			elog_level;
	bool			is_resignal;
	int			sqlerrstate;
	char	   *condname;
	PLpgPSM_expr   *message;
	PLpgPSM_expr   *hint;
	PLpgPSM_expr   *detail;	
} PLpgPSM_stmt_signal;


typedef struct 
{								/* PRINT statement */
	int			cmd_type;
	int			lineno;
	List		*params;
} PLpgPSM_stmt_print;


typedef struct
{								/* Generic SQL statement to execute */
	int			cmd_type;
	int			lineno;
	PLpgPSM_expr *sqlstmt;
	bool		mod_stmt;		/* is the stmt INSERT/UPDATE/DELETE? */
	/* note: mod_stmt is set when we plan the query */
	bool		into;			/* INTO supplied? */
	bool		strict;			/* INTO STRICT flag */
	PLpgPSM_rec *rec;			/* INTO target, if record */
	PLpgPSM_row *row;			/* INTO target, if row */
} PLpgPSM_stmt_sql;


typedef struct
{								/* Dynamic SQL string to execute */
	int			cmd_type;
	int			lineno;
	int			dtype;
	PLpgPSM_expr *query;		/* string expression */
	bool		into;			/* INTO supplied? */
	bool		strict;			/* INTO STRICT flag */
	PLpgPSM_rec *rec;			/* INTO target, if record */
	PLpgPSM_row *row;			/* INTO target, if row */
	char	*name;
} PLpgPSM_stmt_execute;


typedef struct PLpgPSM_diag_area
{
	char 	*fields[20];
	int	sqlstate;
	int	message_length;
	uint32		eval_processed;
	Oid		eval_lastoid;	
	int	elevel;
	struct PLpgPSM_diag_area 	*next;
	struct PLpgPSM_diag_area	*prev;
} PLpgPSM_diag_area;


typedef struct PLpgPSM_func_hashkey
{								/* Hash lookup key for functions */
	Oid			funcOid;

	/*
	 * For a trigger function, the OID of the relation triggered on is part of
	 * the hashkey --- we want to compile the trigger separately for each
	 * relation it is used with, in case the rowtype is different.	Zero if
	 * not called as a trigger.
	 */
	Oid			trigrelOid;

	/*
	 * We include actual argument types in the hash key to support polymorphic
	 * PLpgPSM functions.  Be careful that extra positions are zeroed!
	 */
	Oid			argtypes[FUNC_MAX_ARGS];
} PLpgPSM_func_hashkey;


typedef struct PLpgPSM_function
{								/* Complete compiled function	  */
	char	   *fn_name;
	Oid			fn_oid;
	TransactionId fn_xmin;
	ItemPointerData fn_tid;
	int			fn_functype;
	PLpgPSM_func_hashkey *fn_hashkey;	/* back-link to hashtable key */
	MemoryContext fn_cxt;

	Oid			fn_rettype;
	int			fn_rettyplen;
	bool		fn_retbyval;
	FmgrInfo	fn_retinput;
	Oid			fn_rettypioparam;
	bool		fn_retistuple;
	bool		fn_retset;
	bool		fn_readonly;

	int			fn_nargs;
	int			fn_argvarnos[FUNC_MAX_ARGS];
	int			out_param_varno;
	int			found_varno;
	int			new_varno;
	int			old_varno;
	int			tg_name_varno;
	int			tg_when_varno;
	int			tg_level_varno;
	int			tg_op_varno;
	int			tg_relid_varno;
	int			tg_relname_varno;
	int			tg_table_name_varno;
	int			tg_table_schema_varno;
	int			tg_nargs_varno;

	int		sqlstate_varno;

	int			ndatums;
	PLpgPSM_datum **datums;
	PLpgPSM_stmt *action;

	unsigned long use_count;
} PLpgPSM_function;


typedef struct
{								/* Runtime execution data	*/
	Datum		retval;
	bool		retisnull;
	Oid			rettype;		/* type of current retval */

	Oid			fn_rettype;		/* info about declared function rettype */
	bool		retistuple;
	bool		retisset;

	bool		readonly_func;

	TupleDesc	rettupdesc;
	char	   *exitlabel;		/* the "target" label of the current EXIT or
								 * CONTINUE stmt, if any */

	Tuplestorestate *tuple_store;		/* SRFs accumulate results here */
	MemoryContext tuple_store_cxt;
	ReturnSetInfo *rsi;

	int			trig_nargs;
	Datum	   *trig_argv;

	int			found_varno;
	int			ndatums;
	PLpgPSM_datum **datums;

	/* temporary state for results from evaluation of query or expr */
	SPITupleTable *eval_tuptable;

        ExprContext *eval_econtext;     /* for executing simple expressions */
        EState     *eval_estate;        /* EState containing eval_econtext */                          
        long int        eval_estate_simple_id;          /* ID for eval_estate */ 

	/* status information for error context reporting */
	PLpgPSM_function *err_func; /* current func */
	PLpgPSM_stmt *err_stmt;		/* current stmt */
	const char *err_text;		/* additional state info */
	void	   *plugin_info;	/* reserved for use by optional plugin */
	int	sqlstate_varno;		/* identify SQLSTATE variable */
	int	sqlerrcode;		/* it's backuped for SQLSTATE */
	ErrorData	*edata;
	int	htype;
	FunctionCallInfo	fcinfo;	/* for compare */
	List	*opened_cursors;
	bool	outer_stmt;
	PLpgPSM_diag_area *current_diag;
} PLpgPSM_execstate;


typedef struct
{
    PLpgPSM_execstate	*estate;
    PLpgPSM_stmt_block  *block;
} PLpgPSM_warning_data;


typedef struct 
{
	int last_decl_type;
	PLpgPSM_init_params	*params;
	PLpgPSM_exception_block	*handlers;
} PLpgPSM_decl_block;


typedef struct
{	
	int sqlstate_varno;
	PLpgPSM_stmt *body;
} PLpgPSM_program;



/*
 * A PLpgPSM_plugin structure represents an instrumentation plugin.
 * To instrument PL/pgSQL, a plugin library must access the rendezvous
 * variable "PLpgPSM_plugin" and set it to point to a PLpgPSM_plugin struct.
 * Typically the struct could just be static data in the plugin library.
 * We expect that a plugin would do this at library load time (_PG_init()).
 * It must also be careful to set the rendezvous variable back to NULL
 * if it is unloaded (_PG_fini()).
 *
 * This structure is basically a collection of function pointers --- at
 * various interesting points in pl_exec.c, we call these functions
 * (if the pointers are non-NULL) to give the plugin a chance to watch
 * what we are doing.
 *
 *	func_setup is called when we start a function, before we've initialized
 *	the local variables defined by the function.
 *
 *	func_beg is called when we start a function, after we've initialized
 *	the local variables.
 *
 *	func_end is called at the end of a function.
 *
 *	stmt_beg and stmt_end are called before and after (respectively) each
 *	statement.
 *
 * Also, immediately before any call to func_setup, PL/pgSQL fills in the
 * error_callback and assign_expr fields with pointers to its own
 * plpgpsm_exec_error_callback and exec_assign_expr functions.	This is
 * a somewhat ad-hoc expedient to simplify life for debugger plugins.
 */

typedef struct
{
	/* Function pointers set up by the plugin */
	void		(*func_setup) (PLpgPSM_execstate *estate, PLpgPSM_function *func);
	void		(*func_beg) (PLpgPSM_execstate *estate, PLpgPSM_function *func);
	void		(*func_end) (PLpgPSM_execstate *estate, PLpgPSM_function *func);
	void		(*stmt_beg) (PLpgPSM_execstate *estate, PLpgPSM_stmt *stmt);
	void		(*stmt_end) (PLpgPSM_execstate *estate, PLpgPSM_stmt *stmt);

	/* Function pointers set by PL/pgSQL itself */
	void		(*error_callback) (void *arg);
	void		(*assign_expr) (PLpgPSM_execstate *estate, PLpgPSM_datum *target,
											PLpgPSM_expr *expr);
} PLpgPSM_plugin;


/**********************************************************************
 * Global variable declarations
 **********************************************************************/

extern bool plpgpsm_DumpExecTree;
extern bool plpgpsm_recompile;
extern bool plpgpsm_SpaceScanned;
extern int	plpgpsm_nDatums;
extern PLpgPSM_datum **plpgpsm_Datums;

extern int	plpgpsm_error_lineno;
extern char *plpgpsm_error_funcname;

/* linkage to the real yytext variable */
extern char *plpgpsm_base_yytext;

#define yytext plpgpsm_base_yytext

extern PLpgPSM_function *plpgpsm_curr_compile;
extern bool plpgpsm_check_syntax;
extern MemoryContext compile_tmp_cxt;

extern PLpgPSM_plugin **plugin_ptr;

/**********************************************************************
 * Function declarations
 **********************************************************************/

/* ----------
 * Functions in pl_comp.c
 * ----------
 */
extern PLpgPSM_function *plpgpsm_compile(FunctionCallInfo fcinfo,
				bool forValidator);
extern int	plpgpsm_parse_word(const char *word);
extern int	plpgpsm_parse_dblword(const char *word);
extern int	plpgpsm_parse_tripword(const char *word);
extern int	plpgpsm_parse_wordtype(char *word);
extern int	plpgpsm_parse_dblwordtype(char *word);
extern int	plpgpsm_parse_tripwordtype(char *word);
extern int	plpgpsm_parse_wordrowtype(char *word);
extern int	plpgpsm_parse_dblwordrowtype(char *word);
extern PLpgPSM_type *plpgpsm_parse_datatype(const char *string);
extern PLpgPSM_type *plpgpsm_build_datatype(Oid typeOid, int32 typmod);
extern PLpgPSM_variable *plpgpsm_build_variable(const char *refname, int lineno,
					   PLpgPSM_type *dtype,
					   bool add2namespace,
					   List **initvars);
extern PLpgPSM_condition *plpgpsm_parse_err_condition(char *condname);
extern void plpgpsm_adddatum(PLpgPSM_datum *new);
extern int	plpgpsm_add_initdatums(int **varnos);
extern void plpgpsm_HashTableInit(void);
extern void plpgpsm_compile_error_callback(void *arg);

/* ----------
 * Functions in pl_handler.c
 * ----------
 */
extern void _PG_init(void);
extern Datum plpgpsm_call_handler(PG_FUNCTION_ARGS);
extern Datum plpgpsm_validator(PG_FUNCTION_ARGS);

/* ----------
 * Functions in pl_exec.c
 * ----------
 */
extern Datum plpgpsm_exec_function(PLpgPSM_function *func,
					  FunctionCallInfo fcinfo);
extern HeapTuple plpgpsm_exec_trigger(PLpgPSM_function *func,
					 TriggerData *trigdata);
extern void plpgpsm_xact_cb(XactEvent event, void *arg);
extern void plpgpsm_subxact_cb(SubXactEvent event, SubTransactionId mySubid,
                                                           SubTransactionId parentSubid, void *arg); 
/* ----------
 * Functions for the dynamic string handling in pl_funcs.c
 * ----------
 */
extern void plpgpsm_dstring_init(PLpgPSM_dstring *ds);
extern void plpgpsm_dstring_free(PLpgPSM_dstring *ds);
extern void plpgpsm_dstring_append(PLpgPSM_dstring *ds, const char *str);
extern void plpgpsm_dstring_append_char(PLpgPSM_dstring *ds, char c);
extern char *plpgpsm_dstring_get(PLpgPSM_dstring *ds);

/* ----------
 * Functions for the namestack handling in pl_funcs.c
 * ----------
 */
extern void plpgpsm_ns_init(void);
extern bool plpgpsm_ns_setlocal(bool flag);
extern void plpgpsm_ns_push(const char *label);
extern void plpgpsm_ns_pop(void);
extern void plpgpsm_ns_additem(int itemtype, int itemno, const char *name);
extern PLpgPSM_nsitem *plpgpsm_ns_lookup(const char *name1, const char *name2,
									    const char *name3, int *names_used);
extern PLpgPSM_nsitem *plpgpsm_ns_lookup_label(const char *name);
extern void plpgpsm_ns_rename(char *oldname, char *newname);

/* ----------
 * Other functions in pl_funcs.c
 * ----------
 */
extern void plpgpsm_convert_ident(const char *s, char **output, int numidents);
extern const char *plpgpsm_stmt_typename(PLpgPSM_stmt *stmt);
extern void plpgpsm_dumptree(PLpgPSM_function *func);

/* ----------
 * Externs in gram.y and scan.l
 * ----------
 */
extern PLpgPSM_expr *plpgpsm_read_expression(int until, const char *expected);
extern int	plpgpsm_yyparse(void);
extern int	plpgpsm_base_yylex(void);
extern int	plpgpsm_yylex(void);
extern void plpgpsm_push_back_token(int token);
extern void plpgpsm_yyerror(const char *message);
extern int	plpgpsm_scanner_lineno(void);
extern void plpgpsm_scanner_init(const char *str, int functype);
extern void plpgpsm_scanner_finish(void);
extern char *plpgpsm_get_string_value(void);
extern int plpgpsm_new_sqlerrstate(void);


#endif   /* PLPGPSM_H */
