/* $Id: call.c,v 1.6 2005/05/12 19:29:35 jwp Exp $
 *
 * Copyright 2005, PostgresPy Project.
 * http://python.projects.postgresql.org
 *
 *//*
 * imp/src/call.c,v 1.8 2004/12/15 13:05:14 flaw
 * imp/src/fcall.c,v 1.4 2004/10/08 20:40:44 flaw
 *//*
 * Base call object providing the necessary implementation for calling
 * created Python functions directly in Python.
 */
#include <setjmp.h>
#include <postgres.h>
#include <access/heapam.h>
#include <catalog/pg_type.h>
#include <catalog/pg_proc.h>
#include <nodes/params.h>
#include <nodes/execnodes.h>
#include <parser/parse_type.h>
#include <parser/parse_oper.h>
#include <tcop/dest.h>
#include <tcop/tcopprot.h>
#include <utils/array.h>
#include <utils/catcache.h>
#include <utils/datum.h>
#include <utils/palloc.h>
#include <utils/builtins.h>
#include <utils/syscache.h>
#include <utils/tuplestore.h>
#include <pypg/environment.h>
#include <pypg/postgres.h>

#include <Python.h>
#include <compile.h>
#include <eval.h>
#include <structmember.h>
#include <pypg/python.h>

#include <pypg/externs.h>
#include <pypg/error.h>
#include <pypg/utils.h>
#include <pypg/tupledesc.h>
#include <pypg/heaptuple.h>
#include <pypg/object.h>
#include <pypg/type.h>
#include <pypg/function.h>
#include <pypg/call.h>
#include <pypg/conv.h>

static PyObj
eval(PyObj self)
{
	PyObj code = PyPgCall_FetchFunctionCode(self);
	Assert(PyCode_Check(code));

	PyObj cargs[3] = {
		PyObj(self), 
		PyPgCall_FetchArguments(self), 
		PyPgCall_FetchKeywords(self)
	};
	PyObj rob;

	rob = PyEval_EvalCodeEx((PyCodeObject *) code,
				__main__, NULL, cargs, 3, NULL, 0, NULL, 0, NULL);

	return(rob);
}

static PyMemberDef PyPgCall_Members[] = {
	{"function", T_OBJECT, offsetof(struct PyPgCall, call_func), RO,
	"call's Postgres.Function"},
	{"arguments", T_OBJECT, offsetof(struct PyPgCall, call_args), RO,
	"call's arguments"},
	{"keywords", T_OBJECT, offsetof(struct PyPgCall, call_kw), RO,
	"call's keyword arguments"},
	{"returned", T_OBJECT, offsetof(struct PyPgCall, call_ret), RO,
	"last object returned from evaluation"},
	{"expects", T_OBJECT, offsetof(struct PyPgCall, call_xtd), RO,
	"Postgres.Call's expected tuple descriptor"},
	{NULL}
};

static PyMethodDef PyPgCall_Methods[] = {
	/*{"name", FunctionRef, METH_NOARGS|METH_O|METH_VARARGS, "docstring"},*/
	{NULL}
};

static int
call_traverse(PyObj self, visitproc visit, void *arg)
{
	int err = 0;
	if (PyPgCall_FetchExpected(self) != Py_None)
	{
		err = visit(PyPgCall_FetchExpected(self), arg);
		if (err) return(err);
	}
	if (PyPgCall_FetchReturned(self) != Py_None)
	{
		err = visit(PyPgCall_FetchReturned(self), arg);
		if (err) return(err);
	}
	err = visit(PyPgCall_FetchArguments(self), arg);
	if (err) return(err);
	err = visit(PyPgCall_FetchKeywords(self), arg);
	if (err) return(err);
	err = visit(PyPgCall_FetchFunction(self), arg);
	return(err);
}

int
call_clear(PyObj self)
{
	PyObj ob;

	ob = PyPgCall_FetchExpected(self);
	if (ob != NULL)
	{
		DECREF(PyPgCall_FetchExpected(self));
		PyPgCall_FixExpected(self, NULL);
	}

	ob = PyPgCall_FetchArguments(self);
	if (ob != NULL)
	{
		DECREF(PyPgCall_FetchArguments(self));
		PyPgCall_FixArguments(self, NULL);
	}

	ob = PyPgCall_FetchKeywords(self);
	if (ob != NULL)
	{
		DECREF(PyPgCall_FetchKeywords(self));
		PyPgCall_FixKeywords(self, NULL);
	}

	ob = PyPgCall_FetchReturned(self);
	if (ob != NULL)
	{
		DECREF(PyPgCall_FetchReturned(self));
		PyPgCall_FixReturned(self, NULL);
	}

	ob = PyPgCall_FetchFunction(self);
	if (ob != NULL)
	{
		DECREF(PyPgCall_FetchFunction(self));
		PyPgCall_FixFunction(self, NULL);
	}

	return(0);
}

static void
call_dealloc(PyObj self)
{
	call_clear(self);
	self->ob_type->tp_free(self);
}

static PyObj
call_call(PyObj self, PyObj args, PyObj kw)
{
	int nargs = PyTuple_GET_SIZE(args);
	PyObj iter, rob = NULL;

	if (nargs > 0)
	{
		PyObj call_args = PyPgCall_FetchArguments(self);
		int i, fnargs = PyList_GET_SIZE(call_args);

		if (nargs != fnargs)
		{
			PyErr_Format(PyExc_TypeError,
				"refixing call arguments requires %d arguments, given %d",
				fnargs, nargs);
			return(NULL);
		}

		/* XXX: check and/or convert */
		for (i = 0; i < nargs; ++i)
		{
			PyList_SetItem(call_args, i, PyTuple_GET_ITEM(args, i));
		}
	}

	iter = PyPgCall_FetchReturned(self);
	if (PyIter_Check(iter))
	{
		rob = PyIter_Next(iter);
		if (rob == NULL)
		{
			rob = Py_None;
			DECREF(PyPgCall_FetchReturned(self));
			PyPgCall_FixReturned(self, rob);
			INCREF(rob);
		}
	}
	else
	{
		rob = eval(self);

		if (rob && PyPgCall_FetchReturned(self) != rob)
		{
			DECREF(PyPgCall_FetchReturned(self));
			PyPgCall_FixReturned(self, rob);
			INCREF(rob);
		}
	}

	return(rob);
}

static PyObj
call_iternext(PyObj self)
{
	PyObj rob = NULL, iter;

	iter = PyPgCall_FetchReturned(self);
	if (PyIter_Check(iter))
	{
		rob = PyIter_Next(iter);
		if (rob == NULL)
		{
			DECREF(PyPgCall_FetchReturned(self));
			PyPgCall_FixReturned(self, Py_None);
			INCREF(Py_None);
		}
	}
	return(rob);
}


/*
 * call_new
 *
 * This is the entry point of all function execution. Signficantly,
 * this fills in PythonLanguageOid and PythonLanguageHandler on the first call.
 */
static PyObj
call_new(PyTypeObject *subtype, PyObj args, PyObj kw)
{
	PyObj src, rob;

	if (!PyArg_ParseTuple(args, "O", &src))
		return(NULL);

	if (PyPgFunction_Check(src))
	{
		PyObj args;
		args = PyMapping_GetItemString(kw, "args");
		if (args == NULL)
			return(NULL);
		rob = subtype->tp_alloc(subtype, 0);
		rob = PyPgCall_Initialize(rob, src, args, Py_None);
	}
	else
	{
		PyErr_Format(PyExc_TypeError,
			"cannot create Postgres.FunctionCall from %s",
			src->ob_type->tp_name);
		return(NULL);
	}

	return(rob);
}

const char PyPgCall_Doc[] =
"Python interface to a Postgres Python function call";

PyTypeObject PyPgCall_Type = {
	PyObject_HEAD_INIT(NULL)
	0,											/* ob_size */
	"Postgres.Call",						/* tp_name */
	sizeof(struct PyPgCall),			/* tp_basicsize */
	0,											/* tp_itemsize */
	call_dealloc,							/* tp_dealloc */
	NULL,										/* tp_print */
	NULL,										/* tp_getattr */
	NULL,										/* tp_setattr */
	NULL,										/* tp_compare */
	NULL,										/* tp_repr */
	NULL,										/* tp_as_number */
	NULL,										/* tp_as_sequence */
	NULL,										/* tp_as_mapping */
	NULL,										/* tp_hash */
	call_call,								/* tp_call */
	NULL,										/* tp_str */
	NULL,										/* tp_getattro */
	NULL,										/* tp_setattro */
	NULL,										/* tp_as_buffer */
	Py_TPFLAGS_DEFAULT|
	Py_TPFLAGS_BASETYPE|
	Py_TPFLAGS_HAVE_GC,					/* tp_flags */
	(char *) PyPgCall_Doc,				/* tp_doc */
	call_traverse,							/* tp_traverse */
	call_clear,								/* tp_clear */
	NULL,										/* tp_richcompare */
	0,											/* tp_weaklistoffset */
	(getiterfunc) Py_RETURN_SELF,		/* tp_iter */
	call_iternext,							/* tp_iternext */
	PyPgCall_Methods,						/* tp_methods */
	PyPgCall_Members,						/* tp_members */
	NULL,										/* tp_getset */
	NULL,										/* tp_base */
	NULL,										/* tp_dict */
	NULL,										/* tp_descr_get */
	NULL,										/* tp_descr_set */
	0,											/* tp_dictoffset */
	NULL,										/* tp_init */
	NULL,										/* tp_alloc */
	call_new,								/* tp_new */
};

PyObj
PyPgCall_Initialize(PyObj self, PyObj func, PyObj args, PyObj expect)
{
	PyObj kw;

	if (self == NULL) return(NULL);
	Assert(func != NULL && args != NULL);

	kw = PyDict_New();
	if (kw == NULL) goto fail;
	PyPgCall_FixKeywords(self, kw);
	PyPgCall_FixateKeywords(kw, args, PyPgFunction_FetchArgNames(func));

	if (expect != NULL)
	{
		if (!PyCallable_Check(expect))
		{
			PyErr_Format(PyExc_TypeError,
				"invalid object for expects, requires a callable, given %s",
				expect->ob_type->tp_name
			);
			goto fail;
		}
	}
	else
	{
		expect = PyPgFunction_FetchReturnType(func);
		if (PyPgFunction_IsSRF(func))
		{
			expect = PyPgType_FetchPyPgTupleDesc(expect);
		}
	}
	INCREF(expect);
	PyPgCall_FixExpected(self, expect);

	INCREF(func);
	PyPgCall_FixFunction(self, func);

	INCREF(args);
	PyPgCall_FixArguments(self, args);

	Py_INCREF(Py_None);
	PyPgCall_FixReturned(self, Py_None);

	return(self);
fail:
	DECREF(self);
	return(NULL);
}

PyObj
PyPgCall_FromPyPgFunctionAndArguments(PyObj self, PyObj args)
{
	int i, given, required;
	PyObj argtypes = PyPgFunction_FetchArgTypes(self);
	PyObj pargs, rob;

	given = PyObject_Length(args);
	required = PyObject_Length(argtypes);
	if (given != required)
	{
		PyErr_Format(PyExc_TypeError,
			"This Postgres.Function requires %d arguments (%d given)",
			required, given);
		return(NULL);
	}

	pargs = PyList_New(given);
	for (i = 0; i < required; ++i)
	{
		PyObj arg, argtype;
		argtype = PyTuple_GET_ITEM(argtypes, i);
		arg = PyList_GET_ITEM(args, i);
		if (!PyPgObject_Check(arg) || PyPgObject_FetchType(arg) != argtype)
			arg = PyObject_CallObject(argtype, arg);
		PyList_SetItem(pargs, i, arg);
	}

	rob = PyPgCall_New(self, pargs, Py_None);
	DECREF(pargs);

	return(rob);
}
/*
 * vim: ts=3:sw=3:noet:
 */
