/*-------------------------------------------------------------------------
 *
 * pg_batch.c
 *
 * Copyright (c) 2010, NIPPON TELEGRAPH AND TELEPHONE CORPORATION
 *
 *-------------------------------------------------------------------------
 */

#include "postgres_fe.h"
#include "pgut/pgut-fe.h"
#include <time.h>

const char *PROGRAM_VERSION	= "1.2.2";
const char *PROGRAM_URL		= NULL;
const char *PROGRAM_EMAIL	= NULL;

typedef enum CommandStatus
{
	COMMAND_READY,
	COMMAND_RUNNING,
	COMMAND_SUCCESS,
	COMMAND_FAILED
} CommandStatus;

typedef struct Command
{
	int				order;
	const char	   *query;
	double			priority;
	time_t			start;
	time_t			end;
	CommandStatus	status;
} Command;

static void batch_all_database(const char *sql);
static void batch_one_database(const char *sql);
static void executeCommands(int ncommands, Command *commands, int nconnections);
static void finishCommand(PGconn *conn, Command *command);
static void printResult(const Command *command, const char *message);
static void noticeReceiver(void *arg, const PGresult *res);
static void pgbatch_cleanup(bool fatal, void *userdata);
static char *toDateStr(time_t time);
static char *toTimeStr(time_t time);

#define MAX_ARGS			100
#define MAX_JOBS			32
#define DEFAULT_PRIORITY	0
#define DEFAULT_JOBS		1
#define DEFAULT_TIMEOUT		0

static const char *script;
static const char *args[MAX_ARGS];
static int			nargs = 0;
static int			jobs = DEFAULT_JOBS;
static int			timeout = DEFAULT_TIMEOUT;
static bool			alldb = false;
static time_t		start_time;
static int			num_failed = 0;
static int			num_skipped = 0;

static pgut_option options[] =
{
	{ 'b', 'a', "all", &alldb },
	{ 'i', 'j', "jobs", &jobs },
	{ 'i', 't', "timeout", &timeout },
	{ 0 },
};

/* ORDER BY Command.priority DESC, Command.order ASC */
static int
Command_cmp(const void *lhs, const void *rhs)
{
	const Command *a = (const Command *)lhs;
	const Command *b = (const Command *)rhs;

	if (a->priority > b->priority)
		return -1;
	else if (a->priority < b->priority)
		return +1;
	else if (a->order < b->order)
		return -1;
	else if (a->order > b->order)
		return +1;
	else
		return 0;
}

int
main(int argc, char *argv[])
{
	FILE		   *fp;
	StringInfoData	str;
	int				i;

	start_time = time(NULL);

	i = pgut_getopt(argc, argv, options);

	/* sanity check */
	if (jobs < 1 || MAX_JOBS < jobs)
		ereport(ERROR,
			(errcode(EINVAL),
			 errmsg("-j, --jobs must be 1..%d (%d passed)", MAX_JOBS, jobs)));
	if (timeout < 0)
		ereport(ERROR,
			(errcode(EINVAL),
			 errmsg("-t, --timeout must be > 0 (%d passed)", timeout)));

	/* script file and arguments */
	for (; i < argc; i++)
	{
		if (script == NULL)
			script = argv[i];
		else if (nargs >= MAX_ARGS)
			ereport(ERROR,
				(errcode(EINVAL),
				 errmsg("too many arguments (maximum is %d) (%d passed)", MAX_ARGS, nargs + 1)));
		else
			args[nargs++] = argv[i];
	}

	if (script == NULL)
	{
		ereport(ERROR,
			(errcode(EINVAL),
			 errmsg("no input file")));
		fp = NULL;
	}
	else if (strcmp(script, "-") == 0)
	{
		/* read script from stdin if not specified */
		fp = stdin;
	}
	else
	{
		/* read script from file */
		fp = pgut_fopen(script, "rt");
	}

	/* read all from input */
	initStringInfo(&str);
	if ((errno = appendStringInfoFile(&str, fp)) != 0)
		ereport(ERROR,
			(errcode_errno(),
			 errmsg("read error at file \"%s\": ", script)));
	if (fp && fp != stdin)
		fclose(fp);

	pgut_atexit_push(&pgbatch_cleanup, NULL);
	if (alldb)
		batch_all_database(str.data);
	else
		batch_one_database(str.data);
	pgut_atexit_pop(&pgbatch_cleanup, NULL);

	/* cleanup */
	termStringInfo(&str);

	return (num_failed > 0 || num_skipped > 0 ? 1 : 0);
}

/*
 * atexit() によりルーチン pgbatch_cleanup() が登録されています。
 * pgbatch_cleanup() は後処理を行うルーチンであり、現行ジョブのキャンセル要求、
 * 未実行ジョブのログ出力、コネクションの切断などの処理を行います。
 * 
 * pgbatch_cleanup() が実行される契機は、正常終了時、タイムアウト時、CTRL+C による
 * 割り込み、エラー発生時であり、実行される契機によって後処理の内容を変えます。
 * 
 * 以下のグローバル変数は pgbatch_cleanup() で行う各処理の実行有無の判断と実行に
 * 必要な情報を保持します。主処理ではこれらの変数を現在のプロセスに合わせて
 * 適宜設定します。
 */

/*
 * PGresult *pg_batch_databases
 * int       pg_batch_currentdb 
 * 
 * 現在処理中のデータベースの情報を示します。
 * pg_batch_databases はデータベースの名前一覧を保持する PGresult です。
 * pg_batch_currentdb は、pg_batch_databases の行番号を保持します。この行番号は、
 * 現在処理中のデータベースを示しています。
 * 
 * pgbatch_cleanup() では、スキップするデータベースのログ出力に当該変数を使用します。
 */
static PGresult	   *pg_batch_databases = NULL;
static int			pg_batch_currentdb = 0;

/*
 * PGconn   *pg_batch_connections[]
 * Command  *pg_batch_curcommands[]
 * 
 * 現在実行中のコマンドの情報を示します。
 * pg_batch_connections と pg_batch_curcommands は配列の要素番号でマッピングします。
 * 
 * pgbatch_cleanup() は、現在実行中のコマンドのキャンセル要求およびコネクションの
 * 切断に当該変数を使用します。
 */
static PGconn	   *pg_batch_connections[MAX_JOBS] = { NULL };
static Command	   *pg_batch_curcommands[MAX_JOBS] = { NULL };

/*
 * Command  *pg_batch_commands
 * int       pg_batch_ncommands
 * 
 * 全てのコマンドの一覧および件数を示します。 
 * 
 * pgbatch_cleanup() は、スキップしたコマンドのログ出力に当該変数を使用します。
 */
static Command	   *pg_batch_commands = NULL;
static int			pg_batch_ncommands = 0;

static void
batch_one_database(const char *sql)
{
	int			nfields;
	int			nrows;
	int			i;
	PGresult   *res;
	Command	   *commands;

	/* retrieve commands */
	reconnect(ERROR);
	res = execute_elevel(sql, nargs, args, alldb ? DEBUG2 : ERROR);
	if (alldb)
	{
		switch (PQresultStatus(res))
		{
			case PGRES_TUPLES_OK:
			case PGRES_COMMAND_OK:
			case PGRES_COPY_IN:
				break;
			default:
				elog(INFO, "SKIP: \"%s\" database", dbname);
				PQclear(res);
				return;
		}
	}
	nfields = PQnfields(res);
	if (nfields < 1 || 2 < nfields)
		ereport(ERROR,
			(errcode(EINVAL),
			 errmsg("must return query (and priority): %d columns returned", nfields)));
	nrows = PQntuples(res);
	elog(INFO, "DATABASE '%s' (%d jobs)", dbname, nrows);
	if (nrows == 0)
	{
		PQclear(res);
		return;
	}

	/* parse tuples into commands */
	commands = pgut_malloc(nrows * sizeof(Command));
	for (i = 0; i < nrows; i++)
	{
		char *endptr;

		commands[i].order = i + 1;
		commands[i].query = PQgetvalue(res, i, 0);
		commands[i].status = COMMAND_READY;
		if (nfields > 1)
		{
			commands[i].priority = strtod(PQgetvalue(res, i, 1), &endptr);
			if (*endptr)
				ereport(ERROR,
					(errcode(EINVAL),
					 errmsg("priority must be numeric ('%s' for row=%d)", PQgetvalue(res, i, 1), i)));
		}
		else
			commands[i].priority = DEFAULT_PRIORITY;
	}
	qsort(commands, nrows, sizeof(Command), Command_cmp);
	for (i = 0; i < nrows; i++)
		commands[i].order = i + 1;

	/* execute commands */
	pg_batch_commands = commands;
	pg_batch_ncommands = nrows;
	executeCommands(nrows, commands, jobs);
	pg_batch_commands = NULL;
	pg_batch_ncommands = 0;

	free(commands);
	PQclear(res);
}

static void
batch_all_database(const char *sql)
{
	int			n;

	pg_batch_databases = NULL;
	pg_batch_currentdb = 0;

	reconnect(ERROR);
	pg_batch_databases = execute(
		"SELECT datname FROM pg_catalog.pg_database "
		"WHERE datallowconn ORDER BY 1", 0, NULL);
	disconnect();

	n = PQntuples(pg_batch_databases);
	for (; pg_batch_currentdb < n; pg_batch_currentdb++)
	{
		dbname = PQgetvalue(pg_batch_databases, pg_batch_currentdb, 0);
		batch_one_database(sql);
		if (password)
		{
			free(password);
			password = NULL;
		}
	}
	PQclear(pg_batch_databases);
	pg_batch_databases = NULL;
}

void
pgut_help(bool details)
{
	printf("%s executes jobs in parallel.\n\n", PROGRAM_NAME);
	printf("Usage:\n");
	printf("  %s [OPTIONS] FILENAME [args...]\n", PROGRAM_NAME);

	if (!details)
		return;

	printf("Options:\n");
	printf("  -a, --all                 batch all databases\n");
	printf("  -j, --jobs=CONNECTIONS    number of connections\n");
	printf("  -t, --timeout=TIMEOUT     timeout of job execution in sec\n");
}

static void
pgbatch_cleanup(bool fatal, void *userdata)
{
	int		  i;

	if (fatal)
		return;	/* do nothing on fatal error */

	/* cancel command and disconnect all. */
	for (i = 0; i < jobs; i++)
	{
		char	  errbuf[256];
		PGcancel *cancel_conn;

		if (!pg_batch_connections[i])
			continue;

		if (PQisBusy(pg_batch_connections[i]))
		{
			cancel_conn = PQgetCancel(pg_batch_connections[i]);
			if (cancel_conn != NULL)
			{
				if (!PQcancel(cancel_conn, errbuf, sizeof(errbuf)))
					elog(WARNING, "cancel request sent failed: %s", errbuf);
				PQfreeCancel(cancel_conn);
			}
		}

		if (pg_batch_curcommands[i])
		{
			if (pg_batch_curcommands[i]->status == COMMAND_RUNNING)
				finishCommand(pg_batch_connections[i], pg_batch_curcommands[i]);
		}
		pgut_disconnect(pg_batch_connections[i]);
		pg_batch_connections[i] = NULL;
	}

	if (pg_batch_commands)
	{
		for (i = 0; i < pg_batch_ncommands; i++)
		{
			if (pg_batch_commands[i].status == COMMAND_READY)
				printResult(&pg_batch_commands[i], NULL);
		}
	}

	/* log unexecuted databases */
	if (pg_batch_databases != NULL)
	{
		int		n;
		n = PQntuples(pg_batch_databases);
		for (pg_batch_currentdb++; pg_batch_currentdb < n; pg_batch_currentdb++)
		{
			elog(INFO, "SKIP: \"%s\" database",
				PQgetvalue(pg_batch_databases, pg_batch_currentdb, 0));
		}
		PQclear(pg_batch_databases);
	}
}

static void
finishCommand(PGconn *conn, Command *command)
{
	PGresult   *res;
	int			status;
	char	   *message;

	Assert(command->status == COMMAND_RUNNING);

	if (command->status != COMMAND_RUNNING)
		return;

	status = COMMAND_SUCCESS;
	message = NULL;
	while ((res = PQgetResult(conn)) != NULL)
	{
		switch (PQresultStatus(res))
		{
			case PGRES_TUPLES_OK:
			case PGRES_COMMAND_OK:
			case PGRES_COPY_IN:
				break;
			default:
				status = COMMAND_FAILED;
				message = pgut_strdup(PQerrorMessage(conn));
				break;
		}
		PQclear(res);
	}

	if (status != COMMAND_SUCCESS)
		num_failed++;

	command->end = time(NULL);
	command->status = status;
	printResult(command, message);
	free(message);
}

static int
waitForCommand(int nconnections, PGconn *connections[])
{
	int		i;

	if (timeout > 0)
	{
		time_t  remain_time;
		struct timeval timeout_remain = { 0, 0 };

		remain_time = start_time + timeout - time(NULL);
		if (remain_time > 0)
			timeout_remain.tv_sec = (long) remain_time;

		elog(DEBUG2, "waitForCommand(): remainder time until time-out (tv_sec=%ld, tv_usec=%ld)",
			timeout_remain.tv_sec, timeout_remain.tv_usec);

		i = pgut_wait(nconnections, connections, &timeout_remain);
	}
	else
	{
		i = pgut_wait(nconnections, connections, NULL);
	}

	if (i < 0)
	{
		if (errno == ENOENT)
			return -1;	/* no more commands */

		/* count skipped or failed commands */
		for (i = 0; i < pg_batch_ncommands; i++)
		{
			switch (pg_batch_commands[i].status)
			{
				case COMMAND_READY:
					if (pg_batch_commands[i].priority >= 0)
						num_skipped++;
					break;
				case COMMAND_FAILED:
					num_failed++;
					break;
				default:
					break;
			}
		}
		if (num_skipped > 0 || num_failed > 0)
		{
			elog(WARNING, "TIMEOUT %s", toDateStr(time(NULL)));
			exit(1);
		}
		else
		{
			elog(INFO, "TIMEOUT %s", toDateStr(time(NULL)));
			exit(0);
		}
	}

	return i;
}

static int
getIdleConnection(int nconnections, PGconn *connections[], Command *commands[])
{
	int		i;

	/* find idle connection */
	for (i = 0; i < nconnections; i++)
	{
		if (connections[i] == NULL)
		{
			reconnect(ERROR);
			connections[i] = connection;
			connection = NULL;
			return i;
		}
		else if (!PQisBusy(connections[i]))
		{
			return i;
		}
	}

	i = waitForCommand(nconnections, connections);
	if (i >= 0)
		finishCommand(connections[i], commands[i]);
	return i;
}

static void
executeCommands(int ncommands, Command *commands, int nconnections)
{
	int			i;

	pg_batch_connections[0] = connection;	/* reuse */
	connection = NULL;

	for (i = 0; i < ncommands; i++)
	{
		PGconn	   *conn;
		Command	   *command;
		int			index;

		index = getIdleConnection(nconnections, pg_batch_connections, pg_batch_curcommands);
		if (index < 0)
			break;

		conn = pg_batch_connections[index];
		command = &commands[i];
		pg_batch_curcommands[index] = command;

		commands[i].start = time(NULL);
		PQsetNoticeReceiver(conn, noticeReceiver, command);
		elog(INFO, "[%d/%d] START: %s",
			commands[i].order, ncommands, toDateStr(command->start));
		elog(INFO, "[%d/%d] QUERY: %s",
			command->order, ncommands, command->query);
		pgut_send(conn, command->query, 0, NULL);
		commands[i].status = COMMAND_RUNNING;
	}

	while ((i = waitForCommand(nconnections, pg_batch_connections)) >= 0)
	{
		finishCommand(pg_batch_connections[i], pg_batch_curcommands[i]);
		pgut_disconnect(pg_batch_connections[i]);
		pg_batch_connections[i] = NULL;
		pg_batch_curcommands[i] = NULL;
	}
}

static void
printResult(const Command *command, const char *message)
{
	switch (command->status)
	{
		case COMMAND_READY:
		case COMMAND_RUNNING:
			elog((command->priority >= 0 ? WARNING : INFO),
				"[%d/%d] SKIP: %s",
				command->order, pg_batch_ncommands, command->query);
			break;
		case COMMAND_SUCCESS:
			elog(INFO, "[%d/%d] SUCCESS: %s (%s)",
				command->order, pg_batch_ncommands, toDateStr(command->end),
				toTimeStr(command->end - command->start));
			break;
		case COMMAND_FAILED:
			elog(WARNING, "[%d/%d] FAILED: %s (%s)",
				command->order, pg_batch_ncommands, toDateStr(command->end),
				toTimeStr(command->end - command->start));
			break;
		default:
			ereport(ERROR,
				(errcode(EINVAL),
				 errmsg("illegal command status (%d)", command->status)));
	}

	if (message != NULL)
		elog(WARNING, "[%d/%d] MESSAGE: %s", command->order, pg_batch_ncommands, message);
}

/* NOTE: this function returns a static buffer. */
static char *
toDateStr(time_t time)
{
	static char		buf[64];
	strftime(buf, sizeof(buf), "%Y-%m-%d %H:%M:%S", localtime(&time));
	return buf;
}

/* NOTE: this function returns a static buffer. */
static char *
toTimeStr(time_t time)
{
	static char		buf[64];	/* GG:MI:SS */

	long	hour, min, sec;

	hour = (long) (time / 3600);
	min = (long) ((time - (hour * 3600)) / 60);
	sec = (long) (time - (hour * 3600) - (min * 60));
	snprintf(buf, sizeof(buf), "%02ld:%02ld:%02ld", hour, min, sec);
	return buf;
}

static void
noticeReceiver(void *arg, const PGresult *res)
{
	StringInfoData message;
	char *message_primary_prefix;
	char *message_primary;
	char *message_detail;
	char *message_hint;

	if (PQresultErrorMessage(res)[0] == '\0')
		return;
	if (pgut_log_level > INFO)
		return;

	initStringInfo(&message);
	message_primary_prefix = PQresultErrorField(res, PG_DIAG_SEVERITY);
	message_primary = PQresultErrorField(res, PG_DIAG_MESSAGE_PRIMARY);
	message_detail = PQresultErrorField(res, PG_DIAG_MESSAGE_DETAIL);
	message_hint = PQresultErrorField(res, PG_DIAG_MESSAGE_HINT);

	appendStringInfo(&message, "%s: [%d/%d] %s\n", message_primary_prefix,
		((Command *) arg)->order, pg_batch_ncommands, message_primary);

	if (message_detail)
		appendStringInfo(&message, "DETAIL: %s\n", message_detail);
	if (message_hint)
		appendStringInfo(&message, "HINT: %s\n", message_hint);

	fwrite(message.data, 1, message.len, stderr);
	termStringInfo(&message);
}
