/****************************************************************************
*                pvtext.c
*
*  This module implements message and message display routines for Windows.
*
*  Copyright  POV-Team 1996-1997. All Rights Reserved.
*  This windows version of POV-Ray is Copyright 1996-1997 Christopher J. Cason.
*  Author : Christopher J. Cason.
*
*  NOTE : As this is Windows code, it was edited using a wide Windows-hosted
*         editor. Accordingly, expect text to exceed 80 columns regularly.
*
*  from Persistence of Vision Raytracer
*  Copyright 1996-1997 Persistence of Vision Team
*---------------------------------------------------------------------------
*  NOTICE: This source code file is provided so that users may experiment
*  with enhancements to POV-Ray and to port the software to platforms other
*  than those supported by the POV-Ray Team.  There are strict rules under
*  which you are permitted to use this file.  The rules are in the file
*  named POVLEGAL.DOC which should be distributed with this file. If
*  POVLEGAL.DOC is not available or for more info please contact the POV-Ray
*  Team Coordinator by leaving a message in CompuServe's POVRAY forum. The
*  The latest version of POV-Ray may be found there as well. POVRAY files can
*  also be found on the world wide web at http://www.povray.org/.
*
* This program is based on the popular DKB raytracer version 2.12.
* DKBTrace was originally written by David K. Buck.
* DKBTrace Ver 2.0-2.12 were written by David K. Buck & Aaron A. Collins.
*
* $Id: //depot/POVRAY/povwin/source/windows/PVTEXT.C#11$
*
*****************************************************************************/

#define POVWIN_FILE
#define _WIN32_IE COMMONCTRL_VERSION

#include <math.h>
#include <setjmp.h>
#include <string.h>
#include <windows.h>
#include "frame.h"
#include "colour.h"
#include "povray.h"
#include "optout.h"
#pragma hdrstop

#include <commctrl.h>
#include "pvengine.h"
#include "pvengine.rh"
#include "pvguiext.h"
#include "pvedit.h"

#define MAX_SEEK_INDEX            1024

typedef struct
{
  unsigned              Line ;
  unsigned short        Offset ;
} seekStruct ;

typedef struct
{
  int         id ;
  int         resid ;
  int         flags ;
  char        *text ;
} toolbarStruct ;

int                     message_xchar ;
int                     message_ychar ;
int                     listbox_xchar ;
int                     listbox_ychar ;
int                     top_message_row ;
int                     message_scroll_pos_x ;
int                     message_scroll_pos_y ;
int                     message_count ;
int                     message_cols ;
int                     message_rows ;
int                     EditFileCount ;
char                    *message_buffer [MAX_MESSAGE] ;
char                    **first_message ;
char                    **next_message ;
char                    **last_message ;
char                    str_buffer [1024] ;
char                    message_font_name [128] ;
char                    listbox_font_name [] = "Courier New" ;
char                    line_buffer [2048] ;
char                    *EditFiles [MAX_EDIT_FILES] ;
unsigned                message_font_size ;
unsigned                message_font_weight ;
unsigned                listbox_font_size = 8 ;
unsigned                listbox_font_weight = FW_NORMAL ;
unsigned                editor_line_number ;
unsigned                editor_char_pos ;
unsigned                editor_offset ;
unsigned                seek_entry_count ;
RECT                    current_rect ;
BOOL                    keep_messages ;
HFONT                   message_font ;
HFONT                   listbox_font ;
HFONT                   tab_font ;
seekStruct              seek_entries [MAX_SEEK_INDEX] ;
HIMAGELIST              ToolImages1 ;
HIMAGELIST              ToolImages2 ;
extern int              statistics_count ;
extern char             ini_path [_MAX_PATH] ;
extern char             EngineIniFileName [_MAX_PATH] ;
extern char             RerunIniPath [_MAX_PATH] ;
extern char             CurrentRerunFileName [_MAX_PATH] ;
extern char             tool_help [MAX_TOOLCMD] [MAX_TOOLHELPTEXT] ;
extern char             requested_render_file [] ;
extern char             PovRenderAnimWinClass [] ;
extern void             *CurrentEditor ;
extern unsigned         background_width ;
extern unsigned         background_height ;
extern unsigned         background_shade ;
extern unsigned         window_count ;
extern unsigned         statusheight ;
extern HWND             main_window ;
extern HWND             message_window ;
extern HWND             hToolComboBox ;
extern HWND             window_list [MAX_WINDOWS] ;
extern HWND             toolbar_combobox ;
extern HWND             aux_toolbar_window ;
extern HWND             renderanim_window ;
extern HWND             StatusWindow ;
extern BOOL             IsWin32 ;
extern BOOL             fast_scroll ;
extern BOOL             tile_background ;
extern BOOL             isMaxiMinimized ;
extern BOOL             IsW95UserInterface ;
extern BOOL             use_toolbar ;
extern BOOL             resizing ;
extern BOOL             quit ;
extern BOOL             exit_after_render ;
extern BOOL             demo_mode ;
extern BOOL             debugging ;
extern BOOL             render_requested ;
extern BOOL             noexec ;
extern BOOL             NoRestore ;
extern jmp_buf          jump_buffer ;
extern HBITMAP          hBmpBackground ;
extern HBITMAP          hBmpRendering ;
extern COLORREF         background_colour ;
extern COLORREF         text_colour ;
extern HPALETTE         hPalApp ;
extern HINSTANCE        hInstance ;
extern CRITICAL_SECTION critical_section ;

int                     toolbar_ids [] = {
                                          CM_FILENEW,
                                          CM_FILEOPEN,
                                          CM_FILESAVE,
                                          CM_FILEQUEUE,
                                          CM_RERUNDIALOG,
                                          CM_RENDERSHOW,
                                          CM_COMMANDLINE,
                                          CM_SOURCEFILE,
                                          CM_FILERENDER,
                                          CM_RENDERSLEEP | 0x8000,
                                         } ;

#define MAX_TOOLS       (sizeof (toolbar_ids) / sizeof (int))

toolbarStruct           maintools [] =
                        {
                          CM_FILENEW,       BMP_TOOLFILENEW,              0x00, "New",
                          CM_FILEOPEN,      BMP_TOOLFILEOPEN,             0x00, "Open",
                          CM_FILESAVE,      BMP_TOOLFILESAVE,             0x00, "Save",
                          CM_FILEQUEUE,     BMP_TOOLFILEQUEUE,            0x00, "Queue",
                          CM_RERUNDIALOG,   BMP_TOOLRERUN,                0x00, "Rerun",
                          CM_RENDERSHOW,    BMP_TOOLRENDERSHOW,           0x00, "Show",
                          CM_RENDERCLOSE,   BMP_TOOLRENDERCLOSE,          0x02, "Hide",
                          CM_COMMANDLINE,   BMP_TOOLCMDLINE,              0x00, "Ini",
                          CM_SOURCEFILE,    BMP_TOOLSOURCEFILE,           0x00, "Sel-Run",
                          CM_FILERENDER,    BMP_TOOLFILERENDER,           0x00, "Run",
                          CM_STOPRENDER,    BMP_TOOLSTOPRENDER,           0x02, "Stop",
                          CM_RENDERSLEEP,   BMP_TOOLSLEEP,                0x01, "Pause",
                          CM_SHOWMAINWINDOW,BMP_TOOLSYSTRAY,              0x00, "Tray",
                          0,                0,                            0x00, NULL
                        } ;

#define MAX_MAIN_TOOLS  (sizeof (maintools) / sizeof (toolbarStruct) - 1)

toolbarStruct           auxtools [] =
                        {
                          CM_HELPCONTENTS,  BMP_TOOLHELPCONT,             0x00, "POV-Win",
                          CM_HELPPOVRAY,    BMP_TOOLHELPPOVRAY,           0x00, "Scene",
                          CM_HELPPOVCD,     BMP_TOOLHELPPOVCD,            0x00, "POV CD",
                          CM_HELPIRTCCD,    BMP_TOOLHELPIRTCCD,           0x00, "IRTC CD",
                          CM_GOPOVRAYORG,   BMP_TOOLGOPOVRAY,             0x00, "POV Site",
                          CM_GOIRTC,        BMP_TOOLGOIRTC,               0x00, "IRTC Site",
                          0,                0,                            0x00, NULL
                        } ;

#define MAX_AUX_TOOLS   (sizeof (auxtools) / sizeof (toolbarStruct) - 1)

BOOL is_horz_line (char *s)
{
  if (strlen (s) < 70)
    return (FALSE) ;

  while (*s)
    if (*s++ != '-')
      return (FALSE) ;

  return (TRUE) ;
}

char *buffer_str (char *s, lftype *lf)
{
  char                  *ptr ;
  static int            x = -1 ;

  if (s == NULL)
  {
    x = -1 ;
    return (NULL) ;
  }
  *lf = None ;
  if (x == -1)
  {
    x = 0 ;
    memset (str_buffer, 0, sizeof (str_buffer)) ;
  }
  ptr = str_buffer + x ;
  while (*s)
  {
    switch (*s)
    {
      case '\r' : x = 0 ;
                  ptr = str_buffer ;
                  *lf = CR ;
                  break ;
      case '\n' : x = -1 ;
                  *lf = LF ;
                  return (++s) ;
      case '\b' : if (x > 0)
                  {
                    --x ;
                    *--ptr = '\0' ;
                  }
                  break ;
      default   : if (isprint (*s) && x < sizeof (str_buffer) - 1)
                  {
                    x++ ;
                    *ptr++ = *s ;
                  }
                  break ;
    }
    s++ ;
  }
  return (s) ;
}

int update_message_display (lftype lf)
{
  RECT        rect ;

  if (message_window == NULL)
    return (0) ;
  GetClientRect (message_window, &rect) ;
  message_cols = rect.right / message_xchar ;
  message_rows = rect.bottom / message_ychar ;
  if (lf == None || lf == LF)
  {
    EnterCriticalSection (&critical_section) ;
    top_message_row = message_count > message_rows ? message_count - message_rows : 0 ;
    LeaveCriticalSection (&critical_section) ;
    SetScrollRange (message_window, SB_HORZ, 0, need_hscroll (), FALSE) ;
    SetScrollPos (message_window, SB_HORZ, message_scroll_pos_x, TRUE) ;
    SetScrollRange (message_window, SB_VERT, 0, top_message_row, FALSE) ;
    SetScrollPos (message_window, SB_VERT, message_scroll_pos_y, TRUE) ;
  }
  if (lf == None)
    return (message_rows) ;
  if (lf == LF)
  {
    EnterCriticalSection (&critical_section) ;
    // is there room for another row ?
    if (current_rect.bottom + message_ychar <= rect.bottom)
    {
      // yes there is
      current_rect.top += message_ychar ;
      current_rect.bottom += message_ychar ;
    }
    else
      ScrollWindow (message_window, 0, -message_ychar, NULL, &rect) ;
    message_scroll_pos_y = top_message_row ;
    SetScrollPos (message_window, SB_VERT, message_scroll_pos_y, TRUE) ;
    LeaveCriticalSection (&critical_section) ;
  }
  PovInvalidateRect (message_window, &current_rect, TRUE) ;
  if (!fast_scroll)
    UpdateWindow (message_window) ;
  return (message_rows) ;
}

void buffer_message (msgtype message_type, char *s)
{
  char                  *s1 ;
  char                  str1 [32] ;
  char                  str2 [128] ;
  lftype                lf ;
  static msgtype        _message_type = All ;

  ExternalBufferMessage (message_type, s) ;

  if (message_type != _message_type)
    _message_type = message_type ;

  while (*s)
  {
    s1 = s ;
    EnterCriticalSection (&critical_section) ;
    s = buffer_str (s, &lf) ;
    LeaveCriticalSection (&critical_section) ;
    if (lf == None)
      continue ;
    if (message_type == mStatistics)
    {
      if (statistics_count != -1)
      {
        sprintf (str1, "StatLn%02d", statistics_count++) ;
        sprintf (str2, "\"%.125s\"", str_buffer) ;
        WritePrivateProfileString ("Statistics", str1, str2, CurrentRerunFileName) ;
        /*
        ** We do this to make sure the total statistics don't get written as well,
        ** since we really don't want them for now. 'Total Time' is the last line.
        */
        if (strncmp (s1, "          Total Time:", 21) == 0)
          statistics_count = -1 ;
      }
    }

    if ((s1 = malloc (strlen (str_buffer) + 2)) == NULL)
    {
      PovMessageBox ("Failed to allocate memory for message string", "Fatal Error") ;
      longjmp (jump_buffer, -1) ;
    }

    if (is_horz_line (str_buffer))
      message_type = mHorzLine ;

    strcpy (s1 + 1, str_buffer) ;
    *s1 = (uchar) message_type ;

    if (lf == CR)
    {
      EnterCriticalSection (&critical_section) ;
      if (*last_message)
        free (*last_message) ;
      *last_message = s1 ;
      LeaveCriticalSection (&critical_section) ;
      update_message_display (CR) ;
    }
    else
    {
      EnterCriticalSection (&critical_section) ;
      if (*next_message)
      {
        free (*next_message) ;
        *next_message = NULL ;
        if (++first_message == message_buffer + MAX_MESSAGE)
          first_message = message_buffer ;
      }
      else
        message_count++ ;
      *next_message = s1 ;
      last_message = next_message ;
      if (++next_message == message_buffer + MAX_MESSAGE)
        next_message = message_buffer ;
      LeaveCriticalSection (&critical_section) ;
      update_message_display (LF) ;
    }
  }
}

void message_printf (char *format, ...)
{
  char                  str [2048] ;
  va_list               arg_ptr ;

  if (strlen (format) > sizeof (str) - 256)
    return ;
  va_start (arg_ptr, format) ;
  vsprintf (str, format, arg_ptr) ;
  va_end (arg_ptr) ;
  buffer_message (mIDE, str) ;
}

void status_printf (int nSection, char *format, ...)
{
  char                  str [256] ;
  va_list               arg_ptr ;

  va_start (arg_ptr, format) ;
  vsprintf (str, format, arg_ptr) ;
  va_end (arg_ptr) ;
  say_status_message (nSection, str) ;
}

char *clean_str (char *s)
{
  char        *s1 ;
  static char str [512] ;

  if (strlen (s) > 511)
    return ("Internal error : string too long in clean_str") ;
  EnterCriticalSection (&critical_section) ;
  for (s1 = str ; *s ; s++)
    if (*s >= ' ')
      *s1++ = *s ;
  *s1 = '\0' ;
  LeaveCriticalSection (&critical_section) ;
  return (str) ;
}

void erase_display_window (HDC hdc, int xoffset, int yoffset)
{
  int         x, y ;
  HDC         hdcMemory ;
  RECT        rect ;
  HBRUSH      hbr ;
  HBITMAP     oldBmp ;

  if (message_window == NULL)
    return ;
  GetClientRect (message_window, &rect) ;
  rect.right++ ;
  rect.bottom++ ;
  if (tile_background)
  {
    hdcMemory = CreateCompatibleDC (hdc) ;
    oldBmp = SelectObject (hdcMemory, hBmpBackground) ;
    xoffset %= background_width ;
    yoffset %= background_height ;
    for (y = -yoffset ; y < rect.bottom ; y += background_height)
      for (x = -xoffset ; x < rect.right ; x += background_width)
        BitBlt (hdc, x, y, background_width, background_height, hdcMemory, 0, 0, SRCCOPY) ;
    SelectObject (hdcMemory, oldBmp) ;
    DeleteDC (hdcMemory) ;
  }
  else
  {
    hbr = CreateSolidBrush (background_colour) ;
    FillRect (hdc, &rect, hbr) ;
    DeleteObject (hbr) ;
  }
}

void paint_display_window (HDC hdc)
{
  int         x, y ;
  int         message_number ;
  int         oldMode ;
  int         dividerStep ;
  int         xoffset ;
  int         yoffset ;
  char        **message = first_message ;
  BOOL        erased = FALSE ;
  RECT        rect ;
  HPEN        hpen1 ;
  HPEN        hpen2 ;
  HPEN        hpenOld ;
  HFONT       oldFont ;
  COLORREF    oldColour ;
  COLORREF    oldBkColour ;
  static RECT oldRect ;

  EnterCriticalSection (&critical_section) ;
  GetClientRect (message_window, &rect) ;
  current_rect.left = 0 ;
  current_rect.right = rect.right ;
  current_rect.top = -message_ychar ;
  current_rect.bottom = 0 ;
  xoffset = (unsigned) message_scroll_pos_x * message_xchar ;
  yoffset = (unsigned) (first_message - message_buffer) * message_ychar ;

  if (*message == NULL || resizing)
  {
    erase_display_window (hdc, xoffset, yoffset) ;
    LeaveCriticalSection (&critical_section) ;
    oldRect = rect ;
    return ;
  }

//if (resizing)
//{
//  if ((oldRect.right != rect.right && oldRect.bottom == rect.bottom) || (oldRect.right == rect.right && oldRect.bottom != rect.bottom))
//    ExcludeClipRect (hdc, oldRect.left, oldRect.top, oldRect.right - message_xchar * 2, oldRect.bottom - message_ychar) ;
//  erase_display_window (hdc, offset) ;
//  erased = TRUE ;
//}

  oldRect = rect ;

  hpen1 = CreatePen (PS_SOLID, 1, RGB (192,192,192)) ;
  hpen2 = CreatePen (PS_SOLID, 1, RGB (64,64,64)) ;
  hpenOld = SelectObject (hdc, hpen2) ;
  oldFont = SelectObject (hdc, message_font) ;
  oldMode = SetBkMode (hdc, TRANSPARENT) ;
  oldColour = SetTextColor (hdc, text_colour) ;
  oldBkColour = SetBkColor (hdc, background_shade) ;
  x = message_scroll_pos_x * -message_xchar + message_xchar ;
//if (x == 0)
//  x = message_xchar ;
  for (message_number = y = 0 ; y < rect.bottom ; message_number++)
  {
    if (*message == NULL)
      break ;
    if (message_number >= message_scroll_pos_y)
    {
      if (!erased)
      {
        erase_display_window (hdc, xoffset, yoffset) ;
        erased++ ;
      }
      current_rect.top += message_ychar ;
      current_rect.bottom += message_ychar ;
      if (**message == mDivider || **message == mHorzLine)
      {
        if (background_shade != RGB (1, 1, 1) && tile_background)
          DRAWFASTRECT (hdc, &current_rect) ;
        dividerStep = **message == mDivider ? message_ychar / 3 : message_ychar / 2 - 1 ;
        MoveToEx (hdc, current_rect.left, y + message_ychar - dividerStep, NULL) ;
        SelectObject (hdc, hpen2) ;
        LineTo (hdc, current_rect.left, y + dividerStep) ;
        LineTo (hdc, current_rect.right - message_xchar, y + dividerStep) ;
        SelectObject (hdc, hpen1) ;
        LineTo (hdc, current_rect.right - message_xchar, y + message_ychar - dividerStep) ;
        LineTo (hdc, current_rect.left, y + message_ychar - dividerStep) ;
      }
      else
        ExtTextOut (hdc, x, y, ETO_CLIPPED, &current_rect, *message + 1, strlen (*message + 1), NULL) ;
      y += message_ychar ;
    }
    if (message == last_message)
      break ;
    if (++message == message_buffer + MAX_MESSAGE)
      message = message_buffer ;
    yoffset += message_ychar ;
  }
  if (!erased)
    erase_display_window (hdc, xoffset, yoffset) ;
  SetTextColor (hdc, oldColour) ;
  SetBkColor (hdc, oldBkColour) ;
  SetBkMode (hdc, oldMode) ;
  SelectObject (hdc, oldFont) ;
  SelectObject (hdc, hpenOld) ;
  DeleteObject (hpen1) ;
  DeleteObject (hpen2) ;
  LeaveCriticalSection (&critical_section) ;
}

void get_logfont (HDC hdc, LOGFONT *lf)
{
  memset (lf, 0, sizeof (LOGFONT)) ;
  lf->lfHeight = -MulDiv (message_font_size, GetDeviceCaps (hdc, LOGPIXELSY), 72) ;
  lf->lfWeight = message_font_weight ;
  lf->lfPitchAndFamily = FIXED_PITCH | FF_MODERN ;
  lf->lfCharSet = DEFAULT_CHARSET ;
  lf->lfQuality = PROOF_QUALITY ;
  strncpy (lf->lfFaceName, message_font_name, sizeof (lf->lfFaceName) - 1) ;
}

void get_lblogfont (HDC hdc, LOGFONT *lf)
{
  memset (lf, 0, sizeof (LOGFONT)) ;
  lf->lfHeight = -MulDiv (listbox_font_size, GetDeviceCaps (hdc, LOGPIXELSY), 72) ;
  lf->lfWeight = listbox_font_weight ;
  lf->lfPitchAndFamily = FIXED_PITCH | FF_MODERN ;
  lf->lfCharSet = DEFAULT_CHARSET ;
  lf->lfQuality = PROOF_QUALITY ;
  strncpy (lf->lfFaceName, listbox_font_name, sizeof (lf->lfFaceName) - 1) ;
}

int create_message_font (HDC hdc, LOGFONT *lf)
{
  HFONT       hfontOld ;
  TEXTMETRIC  tm ;

  if ((message_font = CreateFontIndirect (lf)) == NULL)
    return (1) ;
  hfontOld = SelectObject (hdc, message_font) ;
  GetTextMetrics (hdc, &tm) ;
  message_xchar = tm.tmAveCharWidth ;
  message_ychar = tm.tmHeight + tm.tmExternalLeading ;
  SelectObject (hdc, hfontOld) ;
  return (0) ;
}

int create_listbox_font (HDC hdc, LOGFONT *lf)
{
  HFONT       hfontOld ;
  TEXTMETRIC  tm ;

  if ((listbox_font = CreateFontIndirect (lf)) == NULL)
    return (1) ;
  hfontOld = SelectObject (hdc, listbox_font) ;
  GetTextMetrics (hdc, &tm) ;
  listbox_xchar = tm.tmAveCharWidth ;
  listbox_ychar = tm.tmHeight + tm.tmExternalLeading ;
  SelectObject (hdc, hfontOld) ;
  return (0) ;
}

int initialise_message_display (void)
{
  HDC         hdc ;
  LOGFONT     lf ;

  if (message_window == NULL)
  {
    PovMessageBox ("Message Window does not exist", "Initialize Message Display - Fatal Error") ;
    return (1) ;
  }
  hdc = GetDC (message_window) ;
  get_logfont (hdc, &lf) ;
  if (create_message_font (hdc, &lf) != 0)
  {
    PovMessageBox ("Failed to create message font", "Initialize Message Display - Fatal Error") ;
    ReleaseDC (message_window, hdc) ;
    return (1) ;
  }
  get_lblogfont (hdc, &lf) ;
  if (create_listbox_font (hdc, &lf) != 0)
    PovMessageBox ("Failed to create listbox font", "Initialize Message Display") ;
  first_message = next_message = message_buffer ;
  last_message = NULL ;
  buffer_str (NULL, NULL) ;
  top_message_row = message_count = message_scroll_pos_x = message_scroll_pos_y = 0 ;
  current_rect.left = current_rect.bottom = current_rect.right = 0 ;
  current_rect.top = -message_ychar ;
  ReleaseDC (message_window, hdc) ;
  return (0) ;
}

void clear_messages (void)
{
  int         i ;
  char        **p ;

  EnterCriticalSection (&critical_section) ;
  buffer_str (NULL, NULL) ;
  // free any buffered lines still around from a previous run of the renderer
  for (p = message_buffer, i = 0 ; i < MAX_MESSAGE ; p++, i++)
  {
    if (*p)
      free (*p) ;
    *p = NULL ;
  }
  first_message = next_message = message_buffer ;
  last_message = NULL ;
  top_message_row = message_count = message_scroll_pos_x = message_scroll_pos_y = 0 ;
  current_rect.left = current_rect.bottom = current_rect.right = 0 ;
  current_rect.top = -message_ychar ;
  LeaveCriticalSection (&critical_section) ;
  message_printf ("Messages cleared.\n") ;
}

int need_hscroll (void)
{
  int         x ;
  int         xchars ;
  int         width = 0 ;
  char        **message = first_message ;
  RECT        rect ;

  /* modified to return the scroll range if ANY line is long enough */
  if (message_window == NULL || *message == NULL)
    return (0) ;
  GetClientRect (message_window, &rect) ;
  xchars = rect.right / message_xchar - 1 ;
  while (*message)
  {
    x = strlen (*message + 1) ;
    if (x >= xchars)
      if (x - xchars > width)
        width = x - xchars ;
    if (message == next_message)
      break ;
    if (++message == message_buffer + MAX_MESSAGE)
      message = message_buffer ;
  }
  return (width) ;
}

void dump_pane_to_clipboard (void)
{
  int         y ;
  int         message_number ;
  int         length = 0 ;
  char        **message = first_message ;
  char        *s ;
  RECT        rect ;
  HGLOBAL     hText ;
  static char *_s ;

  if (message_window == NULL || *message == NULL)
    return ;
  if (OpenClipboard (message_window) == FALSE)
  {
    PovMessageBox ("Could not open clipboard", "Error") ;
    return ;
  }
  if ((hText = GlobalAlloc (GMEM_MOVEABLE | GMEM_DDESHARE, 33000)) == NULL)
    return ;
  if ((s = GlobalLock (hText)) == NULL)
    return ;
  _s = s ;
  GetClientRect (message_window, &rect) ;
  for (message_number = y = 0 ; y < rect.bottom ; message_number++)
  {
    if (*message == NULL)
      break ;
    if (message_number >= message_scroll_pos_y)
    {
      length += strlen (*message + 1) + 2 ;
      if (length >= 32767)
        break ;
      s += sprintf (s, "%s\r\n", *message + 1) ;
      y += message_ychar ;
    }
    if (message == next_message)
      break ;
    if (++message == message_buffer + MAX_MESSAGE)
      message = message_buffer ;
  }
  GlobalUnlock (hText) ;
  GlobalReAlloc (hText, length + 1, GMEM_MOVEABLE | GMEM_DDESHARE) ;
  SetClipboardData (CF_TEXT, hText) ;
  CloseClipboard () ;
}

void draw_rerun_listbox (DRAWITEMSTRUCT *d)
{
  int         oldMode ;
  char        str [256] ;
  char        *s ;
  RECT        rect ;
  HFONT       oldFont ;
  COLORREF    oldBackground ;
  COLORREF    oldForeground ;

  if (d->itemID == -1)
    return ;
  oldFont = SelectObject (d->hDC, listbox_font) ;
  oldMode = SetBkMode (d->hDC, TRANSPARENT) ;
  oldForeground = SetTextColor (d->hDC, GetSysColor (d->itemState & ODS_SELECTED ? COLOR_HIGHLIGHTTEXT : COLOR_WINDOWTEXT)) ;
  oldBackground = SetBkColor (d->hDC, GetSysColor (d->itemState & ODS_SELECTED ? COLOR_HIGHLIGHT : COLOR_BTNFACE)) ;
  rect = d->rcItem ;
  ExtTextOut (d->hDC, rect.left, rect.top, ETO_OPAQUE, &rect, "", 0, NULL) ;
  SetBkColor (d->hDC, oldBackground) ;

  SendMessage (d->hwndItem, LB_GETTEXT, d->itemID, (LPARAM) str) ;

  s = strtok (str, "\t") ;
  rect = d->rcItem ;
  rect.left = listbox_xchar ;
  rect.right = d->rcItem.right - listbox_xchar * 46 ;
  ExtTextOut (d->hDC, rect.left, rect.top, ETO_CLIPPED, &rect, s, strlen (s), NULL) ;

  s = strtok (NULL, "\t") ;
  rect = d->rcItem ;
  rect.left = d->rcItem.right - listbox_xchar * 45 ;
  rect.right = d->rcItem.right - listbox_xchar * 26 ;
  ExtTextOut (d->hDC, rect.left, rect.top, ETO_CLIPPED, &rect, s, strlen (s), NULL) ;

  s = strtok (NULL, "\t") ;
  rect = d->rcItem ;
  rect.left = d->rcItem.right - listbox_xchar * 25 ;
  ExtTextOut (d->hDC, rect.left, rect.top, ETO_CLIPPED, &rect, s, strlen (s), NULL) ;

  SetTextColor (d->hDC, oldForeground) ;
  SetBkMode (d->hDC, oldMode) ;
  SelectObject (d->hDC, oldFont) ;
}

void draw_ordinary_listbox (DRAWITEMSTRUCT *d)
{
  int         oldMode ;
  int         dividerStep ;
  char        str [256] ;
  RECT        rect ;
  HPEN        hpen1 ;
  HPEN        hpen2 ;
  HPEN        hpenOld ;
  HFONT       oldFont ;
  COLORREF    oldBackground ;
  COLORREF    oldForeground ;

  if (d->itemID == -1)
    return ;
  hpen1 = CreatePen (PS_SOLID, 1, GetSysColor (COLOR_BTNHIGHLIGHT)) ;
  hpen2 = CreatePen (PS_SOLID, 1, GetSysColor (COLOR_BTNSHADOW)) ;
  hpenOld = SelectObject (d->hDC, hpen2) ;
  oldFont = SelectObject (d->hDC, listbox_font) ;
  oldMode = SetBkMode (d->hDC, TRANSPARENT) ;
  oldForeground = SetTextColor (d->hDC, GetSysColor (d->itemState & ODS_SELECTED ? COLOR_HIGHLIGHTTEXT : COLOR_WINDOWTEXT)) ;
  oldBackground = SetBkColor (d->hDC, GetSysColor (d->itemState & ODS_SELECTED ? COLOR_HIGHLIGHT : COLOR_BTNFACE)) ;
  dividerStep = listbox_ychar / 2 - 1 ;

  rect = d->rcItem ;
  rect.left += listbox_xchar ;
  rect.right -= listbox_xchar ;
  SendMessage (d->hwndItem, LB_GETTEXT, d->itemID, (LPARAM) str) ;
  if (strncmp (str, "----------------", 16) == 0)
  {
    MoveToEx (d->hDC, rect.left, rect.top + listbox_ychar - dividerStep, NULL) ;
    SelectObject (d->hDC, hpen2) ;
    LineTo (d->hDC, rect.left, rect.top + dividerStep) ;
    LineTo (d->hDC, rect.right - listbox_xchar, rect.top + dividerStep) ;
    SelectObject (d->hDC, hpen1) ;
    LineTo (d->hDC, rect.right - listbox_xchar, rect.top + listbox_ychar - dividerStep) ;
    LineTo (d->hDC, rect.left, rect.top + listbox_ychar - dividerStep) ;
  }
  else
    ExtTextOut (d->hDC, rect.left, rect.top, ETO_CLIPPED | ETO_OPAQUE, &rect, str, strlen (str), NULL) ;

  SetBkColor (d->hDC, oldBackground) ;
  SetTextColor (d->hDC, oldForeground) ;
  SetBkMode (d->hDC, oldMode) ;
  SelectObject (d->hDC, oldFont) ;
  SelectObject (d->hDC, hpenOld) ;
  DeleteObject (hpen1) ;
  DeleteObject (hpen2) ;
}

void write_wrapped_text (HDC hdc, RECT *rect, char *text)
{
  int         oldMode ;
  HFONT       hFont ;
  HFONT       hFontOld ;
  LOGFONT     lf ;
  COLORREF    oldForeground ;

  memset (&lf, 0, sizeof (LOGFONT)) ;
  lf.lfHeight = -MulDiv (9, GetDeviceCaps (hdc, LOGPIXELSY), 72) ;
  lf.lfWeight = FW_NORMAL ;
  lf.lfPitchAndFamily = VARIABLE_PITCH ;
  lf.lfCharSet = DEFAULT_CHARSET ;
  strcpy (lf.lfFaceName, "MS Sans Serif") ;
  if ((hFont = CreateFontIndirect (&lf)) != NULL)
  {
    hFontOld = SelectObject (hdc, hFont) ;
    oldMode = SetBkMode (hdc, TRANSPARENT) ;
    oldForeground = SetTextColor (hdc, RGB (0, 0, 0)) ;
    DrawText (hdc, text, -1, rect, DT_WORDBREAK | DT_EXPANDTABS) ;
    SetTextColor (hdc, oldForeground) ;
    SetBkMode (hdc, oldMode) ;
    SelectObject (hdc, hFontOld) ;
    DeleteObject (hFont) ;
  }
}

void tip_of_the_day (HDC hdc, RECT *rect, char *text)
{
  int         oldMode ;
  RECT        rc ;
  HFONT       hFont ;
  HFONT       hFontOld ;
  LOGFONT     lf ;
  COLORREF    oldForeground ;

  rc = *rect ;
  memset (&lf, 0, sizeof (LOGFONT)) ;
  lf.lfHeight = -MulDiv (9, GetDeviceCaps (hdc, LOGPIXELSY), 72) ;
  lf.lfWeight = FW_BOLD ;
  lf.lfPitchAndFamily = VARIABLE_PITCH ;
  lf.lfCharSet = DEFAULT_CHARSET ;
  strcpy (lf.lfFaceName, "MS Sans Serif") ;
  if ((hFont = CreateFontIndirect (&lf)) != NULL)
  {
//  hIcon = LoadIcon (hInstance, MAKEINTRESOURCE (IDI_BULB)) ;
//  DrawIcon (hdc, rc.left, rc.top, hIcon) ;
//  DestroyIcon (hIcon) ;
    hFontOld = SelectObject (hdc, hFont) ;
    oldMode = SetBkMode (hdc, TRANSPARENT) ;
    oldForeground = SetTextColor (hdc, RGB (0, 0, 0)) ;
    ExtTextOut (hdc, rc.left, rc.top + 11, ETO_CLIPPED, &rc, "Did you know ... ? ", 19, NULL) ;
    rc.top += 36 ;
    SetTextColor (hdc, oldForeground) ;
    SetBkMode (hdc, oldMode) ;
    SelectObject (hdc, hFontOld) ;
    DeleteObject (hFont) ;
    write_wrapped_text (hdc, &rc, text) ;
  }
}

void handle_menu_select (WPARAM wParam, LPARAM lParam)
{
  char        str [128] ;
  static int  nLastID = -1 ;

  wParam = LOWORD (wParam) ;
  if (wParam != nLastID)
  {
    nLastID = wParam ;
    if (wParam >= CM_FIRSTGUIEXT && wParam <= CM_LASTGUIEXT)
    {
      say_status_message (StatusMessage, ExternalMenuTip (wParam)) ;
      return ;
    }
    if (wParam < CM_FIRSTTOOL || wParam > CM_LASTTOOL)
    {
      if (LoadString (hInstance, wParam, str, sizeof (str)) != 0)
        say_status_message (StatusMessage, str) ;
      else
        say_status_message (StatusMessage, "") ;
    }
    else
      say_status_message (StatusMessage, tool_help [wParam - CM_FIRSTTOOL]) ;
  }
}

void resize_windows (unsigned left, unsigned top, unsigned width, unsigned height)
{
  MoveWindow (message_window, left, top, width, height, FALSE) ;
}

FILE *editor_stream_init (void)
{
  editor_line_number = 0 ;
  editor_char_pos = 0 ;
  editor_offset = 0 ;
  seek_entry_count = 0 ;
  line_buffer [0] = '\0' ;
  return (POV_INTERNAL_STREAM) ;
}

int editor_ftell (void)
{
  if (seek_entry_count >= MAX_SEEK_INDEX)
  {
    PovMessageBox ("Seek index overflow ; render scene from outside editor", "Error") ;
    quit = TRUE ;
    return (-1) ;
  }
  if (editor_char_pos)
  {
    seek_entries [seek_entry_count].Line = editor_line_number ;
    seek_entries [seek_entry_count++].Offset = editor_char_pos - 1 ;
  }
  seek_entries [seek_entry_count].Line = editor_line_number ;
  seek_entries [seek_entry_count].Offset = editor_char_pos ;
  return (seek_entry_count++) ;
}

int editor_fseek (long offset, int whence)
{
  if (whence != 0)
    return (-1) ;
  editor_line_number = seek_entries [offset].Line ;
  editor_char_pos = seek_entries [offset].Offset ;
// FIXME
//if (Edit.GetLine (CurrentEditor, editor_line_number, line_buffer, sizeof (line_buffer) - 1) == FALSE)
//{
//  PovMessageBox ("Seek index error ; render scene from outside editor", "Error") ;
//  quit = TRUE ;
//  return (-1) ;
//}
  return (0) ;
}

int editor_getc (void)
{
  if (line_buffer [editor_char_pos] == '\0')
  {
// FIXME
//  if (Edit.GetLine (CurrentEditor, ++editor_line_number, line_buffer, sizeof (line_buffer) - 2) == FALSE)
//    return (EOF) ;
    editor_char_pos = 0 ;
    strcat (line_buffer, "\n") ;
  }
  return (line_buffer [editor_char_pos++]) ;
}

FILE *pov_fopen (const char *filename, const char *mode)
{
  if (filename == NULL)
    return (editor_stream_init ()) ;
  else
    return (fopen (filename, mode)) ;
}

int pov_fclose (FILE *stream)
{
  if (stream != POV_INTERNAL_STREAM)
    return (fclose (stream)) ;
  else
    return (0) ;
}

int pov_getc (FILE *stream)
{
  return (stream == POV_INTERNAL_STREAM ? editor_getc () : getc (stream)) ;
}

int pov_fseek (FILE *stream, long offset, int whence)
{
  return (stream == POV_INTERNAL_STREAM ? editor_fseek (offset, whence) : fseek (stream, offset, whence)) ;
}

int pov_ftell (FILE *stream)
{
  return (stream == POV_INTERNAL_STREAM ? editor_ftell () : ftell (stream)) ;
}

void add_edit_file (char *file)
{
  if (strlen (file) == 0)
  {
    PovMessageBox ("Empty filename after /EDIT", "Edit File") ;
    return ;
  }
  if (EditFileCount == MAX_EDIT_FILES)
    return ;
  if (strpbrk (file, "*?") != NULL)
  {
    PovMessageBox ("Filename may not contain wildcards", "Edit File") ;
    return ;
  }
  EditFiles [EditFileCount++] = strdup (file) ;
}

void add_render_file (char *file)
{
  static BOOL first = TRUE ;

  if (strlen (file) == 0)
  {
    PovMessageBox ("Empty filename after /RENDER", "Render File") ;
    return ;
  }
  if (!first)
  {
    PovMessageBox ("Only one /RENDER file may be specified", "Render File") ;
    return ;
  }
  if (strpbrk (file, "*?") != NULL)
  {
    PovMessageBox ("Filename may not contain wildcards", "Render File") ;
    return ;
  }
  first = FALSE ;
  strcpy (requested_render_file, file) ;
  render_requested++ ;
}

char *extract_file (char *filename, char *s)
{
  BOOL        inQuote = FALSE ;

  while (*s == ' ' || *s == '\t')
    s++ ;
  while (*s)
  {
    switch (*s)
    {
      case '"' :
           if (inQuote)
           {
             *filename = '\0' ;
             return (++s) ;
           }
           inQuote = TRUE ;
           break ;

      case ' ' :
      case '\t' :
           if (!inQuote)
           {
             *filename  = '\0' ;
             return (s) ;
           }
           *filename++ = *s ;
           break ;

      default :
           *filename++ = *s ;
           break ;
    }
    s++ ;
  }
  *filename  = '\0' ;
  return (s) ;
}

char *preparse_commandline (char *s)
{
  char        *out ;
  char        *command ;
  char        last = ' ' ;
  char        commandstr [256] ;
  char        filename [_MAX_PATH] ;
  static char outstr [_MAX_PATH * 3] ;

  out = outstr ;
  while (*s)
  {
    if (*s == '/' && (last == ' ' || last == '\t'))
    {
      command = commandstr ;
      while (*++s)
      {
        if (*s == ' ' || *s == '\t')
          break ;
        *command++ = *s ;
      }
      *command = '\0' ;
      last = *s ;
      if (strlen (commandstr) == 0)
      {
        PovMessageBox ("Empty command on commandline", "Commandline processing error") ;
        return (NULL) ;
      }
      if (stricmp (commandstr, "EXIT") == 0)
      {
        exit_after_render++ ;
        while (*s == ' ')
          s++ ;
        continue ;
      }
      if (stricmp (commandstr, "DEMO") == 0)
      {
        demo_mode++ ;
        while (*s == ' ')
          s++ ;
        continue ;
      }
      if (stricmp (commandstr, "DEBUG") == 0)
      {
        debugging++ ;
        while (*s == ' ')
          s++ ;
        continue ;
      }
      if (stricmp (commandstr, "EDIT") == 0)
      {
        s = extract_file (filename, s) ;
        add_edit_file (filename) ;
        while (*s == ' ')
          s++ ;
        continue ;
      }
      if (stricmp (commandstr, "RENDER") == 0)
      {
        s = extract_file (filename, s) ;
        add_render_file (filename) ;
        while (*s == ' ')
          s++ ;
        continue ;
      }
      if (stricmp (commandstr, "NOEXEC") == 0)
      {
        noexec++ ;
        while (*s == ' ')
          s++ ;
        continue ;
      }
      if (stricmp (commandstr, "NORESTORE") == 0 || stricmp (commandstr, "NR") == 0)
      {
        NoRestore = true ;
        while (*s == ' ')
          s++ ;
        continue ;
      }
      sprintf (outstr, "Unrecognized command '%s' on commandline", commandstr) ;
      PovMessageBox (outstr, "Commandline processing error") ;
      return (NULL) ;
    }
    last = *out++ = *s++ ;
  }
  return (outstr) ;
}

HWND create_toolbar (HWND hwndParent)
{
  HWND                  hwnd ;
  RECT                  rect ;
  HBITMAP               hbmp ;
  TBBUTTON              main_tbb [MAX_MAIN_TOOLS] ;
  TBBUTTON              aux_tbb [MAX_AUX_TOOLS] ;
  TBBUTTON              *dt ;
  toolbarStruct         *st ;
  REBARBANDINFO         rbBand ;

  // Initialize REBARBANDINFO for all rebar bands
  rbBand.cbSize = sizeof (REBARBANDINFO) ;
  rbBand.clrFore = GetSysColor (COLOR_BTNTEXT) ;
  rbBand.clrBack = GetSysColor (COLOR_BTNFACE) ;
  rbBand.hbmBack = LoadBitmap (hInstance, MAKEINTRESOURCE (BMP_REBAR)) ;

  hwnd = CreateWindow (TOOLBARCLASSNAME,
                       "",
                       WS_CHILD | WS_VISIBLE | CCS_TOP | TBSTYLE_TOOLTIPS | TBSTYLE_FLAT |
                       WS_CLIPCHILDREN | WS_CLIPSIBLINGS | CCS_NODIVIDER | CCS_NORESIZE,
                       0,
                       0,
                       256,
                       50,
                       hwndParent,
                       NULL,
                       hInstance,
                       NULL) ;
  if (hwnd == NULL)
    return (NULL) ;
  SendMessage (hwnd, TB_BUTTONSTRUCTSIZE, sizeof (TBBUTTON), 0) ;
  SendMessage (hwnd, TB_SETBITMAPSIZE, 0, (LPARAM) MAKELONG (38, 21)) ;
  SendMessage (hwnd, TB_SETBUTTONSIZE, 0, (LPARAM) MAKELONG (38, 21)) ;

  if ((ToolImages1 = ImageList_Create (38, 21, ILC_COLOR4 | ILC_MASK, 0, 16)) == NULL)
    return (NULL) ;
  for (dt = main_tbb, st = maintools ; st->text != NULL ; st++, dt++)
  {
    dt->iBitmap = ImageList_AddMasked (ToolImages1, hbmp = LoadBitmap (hInstance, MAKEINTRESOURCE (st->resid)), RGB (255, 255, 156)) ;
    dt->iString = SendMessage (hwnd, TB_ADDSTRING, 0, (LPARAM) st->text) ;
    dt->idCommand = st->id ;
    dt->fsState = TBSTATE_ENABLED ;
    dt->fsStyle = st->flags & 0x01 ? TBSTYLE_CHECK : TBSTYLE_BUTTON ;
    dt->fsState = st->flags & 0x02 ? TBSTATE_HIDDEN | TBSTATE_ENABLED : TBSTATE_ENABLED ;
    dt->dwData = 0 ;
    DeleteObject (hbmp) ;
  }
  SendMessage (hwnd, TB_SETIMAGELIST, 0, (LPARAM) ToolImages1) ;
  SendMessage (hwnd, TB_ADDBUTTONS, MAX_MAIN_TOOLS, (LPARAM) main_tbb) ;
  SendMessage (hwnd, TB_AUTOSIZE, 0, 0) ;

  rbBand.fMask = RBBIM_COLORS | RBBIM_CHILD | RBBIM_CHILDSIZE | RBBIM_STYLE | RBBIM_BACKGROUND | RBBIM_ID | RBBIM_SIZE ;
  rbBand.fStyle = RBBS_NOVERT | RBBS_CHILDEDGE | RBBS_FIXEDBMP | RBBS_GRIPPERALWAYS ;
  rbBand.hwndChild = hwnd ;
  rbBand.wID = 0 ;
  SendMessage (hwnd, TB_GETITEMRECT, MAX_MAIN_TOOLS - 1, (LPARAM) &rect) ;
  rbBand.cxMinChild = 1 ;
  rbBand.cx = rect.right + 15 ;
  rbBand.cyMinChild = rect.bottom ;
  SendMessage (hwndParent, RB_INSERTBAND, (UINT) -1, (LPARAM) (LPREBARBANDINFO) &rbBand) ;

  DeleteObject (hbmp) ;

  renderanim_window = CreateWindowEx (0,
                                      PovRenderAnimWinClass,
                                      "",
                                      WS_CHILD | WS_VISIBLE | WS_CLIPCHILDREN | WS_CLIPSIBLINGS,
                                      0,
                                      0,
                                      120,
                                      42,
                                      hwndParent,
                                      NULL,
                                      hInstance,
                                      NULL) ;

  rbBand.fMask = RBBIM_COLORS | RBBIM_CHILD | RBBIM_CHILDSIZE | RBBIM_STYLE | RBBIM_ID | RBBIM_SIZE ;
  rbBand.fStyle = RBBS_NOVERT | RBBS_CHILDEDGE | RBBS_FIXEDSIZE ;
  rbBand.hwndChild = renderanim_window ;
  rbBand.wID = 1 ;
  rbBand.cx = 123 ;
  rbBand.cxMinChild = 123 ;
  rbBand.cyMinChild = 42 ;
  SendMessage (hwndParent, RB_INSERTBAND, (UINT) -1, (LPARAM) (LPREBARBANDINFO) &rbBand) ;

  rbBand.fMask = RBBIM_COLORS | RBBIM_CHILDSIZE | RBBIM_STYLE | RBBIM_BACKGROUND | RBBIM_SIZE | RBBIM_TEXT ;
  rbBand.fStyle = RBBS_NOVERT | RBBS_CHILDEDGE | RBBS_FIXEDBMP | RBBS_GRIPPERALWAYS ;
  rbBand.cx = 32 ;
  rbBand.cxMinChild = 1 ;
  rbBand.cyMinChild = 8 ;
  rbBand.lpText = " " ;
  SendMessage (hwndParent, RB_INSERTBAND, (UINT) -1, (LPARAM) (LPREBARBANDINFO) &rbBand) ;

  toolbar_combobox = CreateWindow ("COMBOBOX", //WC_COMBOBOXEX,
                                   "",
                                   WS_CHILD | WS_VISIBLE | CBS_DROPDOWNLIST | WS_VSCROLL | WS_TABSTOP |
                                   WS_CLIPCHILDREN | WS_CLIPSIBLINGS | CCS_NORESIZE,
                                   0,
                                   0,
                                   LOWORD (GetDialogBaseUnits ()) * 20,
                                   HIWORD (GetDialogBaseUnits ()) * 17,
                                   hwndParent,
                                   NULL,
                                   hInstance,
                                   NULL) ;

  SendMessage (toolbar_combobox, WM_SETFONT, SendMessage (hwndParent, WM_GETFONT, 0, 0), TRUE) ;
  rbBand.fMask = RBBIM_COLORS | RBBIM_CHILD | RBBIM_CHILDSIZE | RBBIM_STYLE | RBBIM_BACKGROUND | RBBIM_ID | RBBIM_SIZE ;
  rbBand.fStyle = RBBS_NOVERT | RBBS_CHILDEDGE | RBBS_FIXEDBMP | RBBS_GRIPPERALWAYS | RBBS_BREAK ;
  rbBand.hwndChild = toolbar_combobox ;
  rbBand.wID = 2 ;
  rbBand.cx = LOWORD (GetDialogBaseUnits ()) * 20 ;
  rbBand.cxMinChild = rbBand.cx / 2 ;
  GetClientRect (toolbar_combobox, &rect) ;
  rbBand.cyMinChild = rect.bottom ;
  SendMessage (hwndParent, RB_INSERTBAND, (UINT) -1, (LPARAM) (LPREBARBANDINFO) &rbBand) ;

  aux_toolbar_window = CreateWindow (TOOLBARCLASSNAME,
                                     "",
                                     WS_CHILD | WS_VISIBLE | CCS_TOP | TBSTYLE_TOOLTIPS | TBSTYLE_FLAT | TBSTYLE_LIST |
                                     WS_CLIPCHILDREN | WS_CLIPSIBLINGS | CCS_NODIVIDER | CCS_NORESIZE,
                                     0,
                                     0,
                                     256,
                                     50,
                                     hwndParent,
                                     NULL,
                                     hInstance,
                                     NULL) ;
  if (aux_toolbar_window == NULL)
    return (NULL) ;

  SendMessage (aux_toolbar_window, TB_BUTTONSTRUCTSIZE, sizeof (TBBUTTON), 0) ;
  SendMessage (aux_toolbar_window, TB_SETBITMAPSIZE, 0, (LPARAM) MAKELONG (14, 14)) ;
  SendMessage (aux_toolbar_window, TB_SETBUTTONSIZE, 0, (LPARAM) MAKELONG (14, 14)) ;

  if ((ToolImages2 = ImageList_Create (14, 14, ILC_COLOR4 | ILC_MASK, 0, 8)) == NULL)
    return (hwnd) ;
  for (dt = aux_tbb, st = auxtools ; st->text != NULL ; st++, dt++)
  {
    dt->iBitmap = ImageList_AddMasked (ToolImages2, hbmp = LoadBitmap (hInstance, MAKEINTRESOURCE (st->resid)), RGB (256, 256, 156)) ;
    dt->iString = SendMessage (aux_toolbar_window, TB_ADDSTRING, 0, (LPARAM) st->text) ;
    dt->idCommand = st->id ;
    dt->fsState = TBSTATE_ENABLED ;
    dt->fsStyle = st->flags & 0x01 ? TBSTYLE_CHECK : TBSTYLE_BUTTON ;
    dt->dwData = 0 ;
    DeleteObject (hbmp) ;
  }
  SendMessage (aux_toolbar_window, TB_SETIMAGELIST, 0, (LPARAM) ToolImages2) ;
  SendMessage (aux_toolbar_window, TB_ADDBUTTONS, MAX_AUX_TOOLS, (LPARAM) aux_tbb) ;
  SendMessage (aux_toolbar_window, TB_AUTOSIZE, 0, 0) ;

  rbBand.fMask = RBBIM_COLORS | RBBIM_CHILD | RBBIM_CHILDSIZE | RBBIM_STYLE | RBBIM_BACKGROUND | RBBIM_ID | RBBIM_SIZE ;
  rbBand.fStyle = RBBS_NOVERT | RBBS_CHILDEDGE | RBBS_FIXEDBMP | RBBS_GRIPPERALWAYS ;
  rbBand.hwndChild = aux_toolbar_window ;
  rbBand.wID = 3 ;
  SendMessage (aux_toolbar_window, TB_GETITEMRECT, MAX_AUX_TOOLS - 1, (LPARAM) &rect) ;
  rbBand.cxMinChild = 1 ;
  rbBand.cx = rect.right + 15 ;
  rbBand.cyMinChild = rect.bottom ;
  SendMessage (hwndParent, RB_INSERTBAND, (UINT) -1, (LPARAM) (LPREBARBANDINFO) &rbBand) ;

  return (hwnd) ;
}

HWND create_rebar (HWND hwndParent)
{
  HWND                            hwnd ;
  INITCOMMONCONTROLSEX            icex ;

  icex.dwSize = sizeof (INITCOMMONCONTROLSEX) ;
  icex.dwICC = ICC_COOL_CLASSES ;
  InitCommonControlsEx (&icex) ;
  hwnd = CreateWindowEx (0L,
                         REBARCLASSNAME,
                         NULL,
                         WS_VISIBLE | WS_BORDER | WS_CHILD | WS_CLIPCHILDREN |
                         WS_CLIPSIBLINGS | RBS_VARHEIGHT | RBS_BANDBORDERS | RBS_DBLCLKTOGGLE,
                         0,
                         0,
                         1280,
                         128,
                         hwndParent,
                         (HMENU) ID_REBAR,
                         hInstance,
                         NULL) ;
  return (hwnd) ;
}

HWND CreateStatusbar (HWND hwndParent)
{
  HWND                            hwnd ;
  RECT                            rect ;

  hwnd = CreateWindowEx (0L,
                         STATUSCLASSNAME,
                         NULL,
                         WS_CHILD | SBARS_SIZEGRIP | WS_VISIBLE,
                         0,
                         0,
                         0,
                         0,
                         hwndParent,
                         (HMENU) ID_STATUS,
                         hInstance,
                         NULL) ;
  GetClientRect (hwnd, &rect) ;
  statusheight = rect.bottom - rect.top + 1 ;
  return (hwnd) ;
}

void ResizeStatusBar (HWND hwnd)
{
  int                   parts [] = {0, 0, 0, 0, 0, 0, 0} ;
  int                   i ;
  int                   total = 0 ;
  RECT                  rect ;
  static const int      widths [] = {0, 50, 50, 50, 50, 95, 125} ;
  static const int      all = 420 ;

  GetClientRect (hwnd, &rect) ;
  rect.right -= all ;
  for (i = 0 ; i < sizeof (parts) / sizeof (int) ; i++)
  {
    total += widths [i] ;
    parts [i] = total + rect.right ;
  }
  SendMessage (hwnd, SB_SETPARTS, sizeof (parts) / sizeof (int), (LPARAM) parts) ;
}

void say_status_message (int section, char *message)
{
  char        str [256] = "\t" ;

  switch (section)
  {
    case StatusMessage :
         if (strcmp (message, "       ") == 0)
           return ;
         if (strlen (message) == 10)
           if (message [3] == ':' && message [6] == ':')
             return ;
         if (strncmp (message, " at ", 4) == 0)
           return ;
         SendMessage (StatusWindow, SB_SETTEXT, section, (LPARAM) message) ;
         break ;

    case StatusPPS :
         strncat (str, message, 254) ;
         SendMessage (StatusWindow, SB_SETTEXT, section, (LPARAM) str) ;
         break ;

    case StatusRendertime :
         SendMessage (StatusWindow, SB_SETTEXT, section, (LPARAM) message) ;
         break ;
  }
}


