%{  /* -*- c -*- */
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <stdlib.h>
#include "error.h"
#include "asm.h"

typedef unsigned int loc;
typedef long lineno;

#define STARTPC 0x200
static loc pc = STARTPC;
static short mem[0x10000+4];

static lineno line_number = 1;
static const char *fname = "<stdin>";
char *program_name = "asm";

static struct symbol
{
  struct symbol *next;
  const char *sym;
  lineno defined_line;
  loc pc;
} *htable[1031];

static struct fixup
{
  struct fixup *next;
  loc where;
  enum t_mode kind;
  const char *expression;
  lineno line;
} *fix_chain;

static int
hashit(const char *sym, int symlen)
{
  int i;
  unsigned long hash = 1;
  
  for (i = 0; i < symlen; i++)
    hash = (hash << 4) + sym[i] + (hash >> 20 & 0xF0);
  return hash % (sizeof(htable)/sizeof(htable[0]));
}

static void *
xmalloc(size_t reqd)
{
  void *result;
  if (reqd == 0)
    reqd = 1;
  result = malloc(reqd);
  if (result == NULL)
    error_at_line (1, errno, fname, line_number, "out of memory");
  return result;
}

#define SKIPBLANKS \
  while (exprlen > 0 && (expr[0] == ' ' || expr[0] == '\t'))	\
    {								\
      ++*exprused;						\
      expr++;							\
      exprlen--;				       		\
   }
#define POSTEXPR \
  while (exprused < yyleng)					\
    if (yytext[exprused] != ' ' && yytext[exprused] != '\t')	\
      {								\
	error_at_line(0, 0, fname, line_number,			\
		      "extra stuff after expression, `%.*s'", 	\
		      yyleng-exprused, yytext+exprused);	\
	break;							\
      }								\
    else							\
      exprused++;

static unsigned long
eval_number(const char *expr, int exprlen, int *exprused)
{
  unsigned radix;
  int nlen, i;
  const char *oexpr = expr;
  loc result;

  nlen = 0;
  while (nlen < exprlen && (isalnum(expr[nlen]) || expr[nlen] == ':'))
    nlen++;
  *exprused = nlen;

  if (nlen > 1 && expr[0] == '0')
    {
      expr++;
      nlen--;
      if (isdigit(expr[0]))
	radix = 8;
      else 
	{
	  if (tolower(expr[0]) == 'x')
	    radix = 16;
	  else if (tolower(expr[0]) == 'b')
	    radix = 2;
	  else
	    goto bad;
	  expr++;
	  nlen--;
	}
    }
  else
    radix = 10;

  result = 0;
  for (i = 0; i < nlen; i++)
    {
      int dig;
      
      if (expr[i] >= 'a')
	dig = expr[i] - 'a' + 10;
      else if (expr[i] >= 'A')
	dig = expr[i] - 'A' + 10;
      else
	dig = expr[i] - '0';
      if (dig < 0 || dig > radix)
	goto bad;
      result = result * radix + dig;
    }
  return result;

 bad:
  error_at_line(0, 0, fname, line_number, 
		"invalid number `%.*s'", *exprused, oexpr);
  *exprused = exprlen;
  return (loc)-1;
}

static loc
eval_label(const char *expr, int exprlen, int *exprused, int nofixup)
{
  int nlen;
  struct symbol *htp;
  
  nlen = 0;
  while (nlen < exprlen 
	 && (isalnum(expr[nlen]) || expr[nlen] == '_' || expr[nlen] == '$'))
    nlen++;
  *exprused = nlen;

  if (nlen == 0)
    {
      error_at_line(0, 0, fname, line_number, "unexpected character `%c'", 
		    expr[0]);
      return (loc)-1;
    }

  htp = htable[hashit(expr, nlen)];
  while (htp != NULL && (strncmp(htp->sym, expr, nlen) != 0
			 || htp->sym[nlen] != '\0'))
    htp = htp->next;
  
  if (htp != NULL)
    return htp->pc;
  
  if (!nofixup)
    return (loc)-2;
  
  error_at_line(0, 0, fname, line_number, "undefined symbol `%.*s'", 
		nlen, expr);
  return (loc)-1;
}

/* A simple left-to-right expression evaluator.  */
static enum t_mode
eval_expression(const char *expr, int exprlen, int *exprused, loc where, 
		enum t_mode mode, int nofixup)
{
  loc current;
  int used;
  int overflow, needsfixup;
  
  *exprused = 0;
  SKIPBLANKS;
  
  if (mode == t_inh)
    return t_inh;
  
  if (exprlen > 0 && expr[0] == '#')
    {
      if (mode == t_rmi)
	mode = t_imm;
      else if (mode != t_imm)
	error_at_line(0, 0, fname, line_number, 
		      "immediate operand not allowed");
      ++*exprused;
      expr++;
      exprlen--;
    }
  
  SKIPBLANKS;

  if (exprlen == 0)
  {
    *exprused += exprlen;
    error_at_line(0, 0, fname, line_number, "empty expression");
    return t_dir;
  }

  if (exprlen > 1 && expr[0] == ',' && toupper(expr[1]) == 'X'
      && (mode == t_rmw || mode == t_rmi || mode == t_rm))
    {
      *exprused += 2;
      return t_ix;
    }
  
  if (isdigit(expr[0]))
    current = eval_number(expr, exprlen, &used);
  else
    current = eval_label(expr, exprlen, &used, nofixup);
  expr += used;
  *exprused += used;
  exprlen -= used;
  
  if (current == (loc)-1)
    return t_dir;
  needsfixup = (current == (loc)-2);
  
  while (exprlen > 0 && expr[0] != ',')
    {
      loc next;
      char op;
      
      SKIPBLANKS;
      if (exprlen == 0 || expr[0] == ',')
	break;

      op = expr[0];
      if (op == ',')
	break;
      if (strchr("+-*&|^<>", op) == NULL)
	{
	  *exprused = exprlen;
	  error_at_line(0, 0, fname, line_number, "unknown operator `%c'", op);
	  return t_dir;
	}
      expr++;
      ++*exprused;
      exprlen--;
      if (op == '<' || op == '>')
	{
	  if (exprlen < 1 || expr[0] != op)
	    error_at_line(0, 0, fname, line_number, "operator %c%c missing"
			  " second `%c'", op, op, op);
	  else
	    {
	      expr++;
	      ++*exprused;
	      exprlen--;
	    }
	}

      SKIPBLANKS;
      if (exprlen == 0 || expr[0] == ',')
	{
	  error_at_line(0, 0, fname, line_number, "missing operand");
	  return t_dir;
	}
      
      if (isdigit(expr[0]))
	next = eval_number(expr, exprlen, &used);
      else
	next = eval_label(expr, exprlen, &used, nofixup);
      if (next == (loc)-1)
	return t_dir;
      needsfixup |= (next == (loc)-2);
      expr += used;
      *exprused += used;
      exprlen -= used;
  
      switch (op)
	{
	case '+': current += next; break;
	case '-': current -= next; break;
	case '*': current *= next; break;
	case '&': current &= next; break;
	case '|': current |= next; break;
	case '^': current ^= next; break;
	case '<': current <<= next; break;
	case '>': current >>= next; break;
	}
    }

  if (mode == t_rmw || mode == t_rmi || mode == t_rm)
    {
      SKIPBLANKS;
  
      if (exprlen > 1 && expr[0] == ',' && toupper(expr[1]) == 'X')
	{
	  expr += 2;
	  *exprused += 2;
	  exprlen -= 2;

	  if ((current >= 0x100 || needsfixup) 
	      && (mode == t_rmi || mode == t_rm))
	    mode = t_ix2;
	  else if (current != 0)
	    mode = t_ix1;
	  else
	    mode = t_ix;
	}
      else
	if ((current >= 0x100 || needsfixup)
	    && (mode == t_rmi || mode == t_rm))
	  mode = t_ext;
	else
	  mode = t_dir;
    }

  if (needsfixup)
    {
      struct fixup *old = fix_chain;
      char *xpr;
      fix_chain = xmalloc(sizeof(struct fixup));
      fix_chain->next = old;
      fix_chain->where = where;
      fix_chain->kind = mode;
      fix_chain->line = line_number;
      fix_chain->expression = xpr = xmalloc(*exprused+1);
      memcpy(xpr, expr-*exprused, *exprused);
      xpr[*exprused] = '\0';

      overflow = 0;
    }
  else switch (mode)
    {
    case t_dir: case t_ix1: case t_imm:
      mem[where] = current & 0xFF;
      overflow = (current >= 0x100 && (current & 0xFFFF) < (-0x80 & 0xFFFF));
      break;
    case t_ext: case t_ix2:
      mem[where] = current >> 8 & 0xFF;
      mem[where+1] = current & 0xFF;
      overflow = (current >= 0x10000);
      break;
    case t_rel:
      mem[where] = (current - where - 1) & 0xFF;
      overflow = ((current >= where+1 && current - where - 1 > 0x7F)
		  || (current < where+1 && where+1 - current > 0x80));
      break;
    case t_ix:
      overflow = 0;
      break;
    default:
      abort();
    }
  if (overflow)
    error_at_line(0, 0, fname, line_number, "overflow in expression");
    
  return mode;
}

#define YY_FATAL_ERROR(m) \
  error_at_line(1,errno, fname,line_number,"%s",m)

%}
%option case-insensitive
%option never-interactive
%option noinput
%option nounput
%option noyywrap

IDENT [[:alpha:]_$][[:alnum:]_$]*
COMMENT ("%"|^#)
WHITESPACE [ \t\r\v;]+
OPCODE [[:alnum:]]+
EXPR [ \t]*"#"?[[:alnum:]_$ \t+*&|^<>-]*

%%

{COMMENT}.*\n	|
\n		{ line_number++; }

{WHITESPACE}	/* do nothing */

{IDENT}:	{ 
  int h = hashit(yytext, yyleng-1);
  struct symbol *s;
  char *name;

  name = xmalloc(yyleng);
  memcpy(name, yytext, yyleng-1);
  name[yyleng-1] = '\0';

  

  for (s = htable[h]; s != NULL; s = s->next)
    if (strcmp(s->sym, name) == 0)
      {
	error_at_line(0, 0, fname, line_number, 
		      "symbol `%s' previously defined on line %ld",
		      name, s->defined_line);
	break;
      }
    
  s = xmalloc(sizeof(struct symbol));
  s->next = htable[h];
  s->pc = pc;
  s->defined_line = line_number;
  s->sym = name;
  htable[h] = s;
}

("WORD"|"BYTE"){EXPR} {
  int exprused;
  enum t_mode mode;
  
  mode = (tolower(yytext[0]) == 'w' ? t_ext : t_dir);
  
  eval_expression(yytext+4, yyleng-4, &exprused, pc, mode, 0);
  exprused+=4;
  pc += 1 + (mode == t_ext);
  
  POSTEXPR;
}
"ORG"{EXPR} {
  short m[2];
  int exprused;
  loc npc;

  m[0] = mem[pc];
  m[1] = mem[pc+1];
  eval_expression(yytext+3, yyleng-3, &exprused, pc, t_ext, 1);
  exprused += 3;
  npc = mem[pc] << 8 | mem[pc+1];
  mem[pc] = m[0];
  mem[pc+1] = m[1];
  pc = npc;

  POSTEXPR;
}

BR(CLR|SET)[[:digit:]]{EXPR}","{EXPR} |
{OPCODE}{EXPR}",X"? {
  int oplen;
  const struct opcdata *op;
  enum t_mode mode;
  int exprused;
  
  for (oplen = 0; oplen < yyleng && isalnum(yytext[oplen]); oplen++)
    yytext[oplen] = toupper(yytext[oplen]);
  op = opsearch(yytext, oplen);
  
  if (op == NULL)
    error_at_line(0, 0, fname, line_number, 
		  "unrecognised opcode `%.*s'", oplen, yytext);
  else 
    {
      loc opc = pc;
      
      mem[pc] = op->opnum;
      
      if (op->mode != t_dirrel)
	{
	  mode = eval_expression(yytext + oplen, yyleng - oplen, &exprused,
				 pc+1, op->mode, 0);
	  pc++;
	  if (mode == t_ix2 || mode == t_ext) pc++;
	  if (mode != t_inh && mode != t_ix) pc++;
	}
      else
	{
	  int xu1, xu2;
	  
	  mode = eval_expression(yytext + oplen, yyleng - oplen, &xu1, 
				 pc+1, t_dir, 0);
	  mode = eval_expression(yytext + oplen + xu1 + 1, 
				 yyleng - oplen - xu1 - 1, &xu2, pc+2, t_rel,
				 0);
	  exprused = xu1 + xu2 + 1;
	  pc += 3;
	}
      exprused += oplen;
      POSTEXPR;
      
      if (op->mode == t_rmw || op->mode == t_rmi || op->mode == t_rm)
	switch (mode)
	  {
	  case t_ix:  mem[opc] += 0x50; break;
	  case t_ix1: mem[opc] += 0x40; break;
	  case t_ix2: mem[opc] += 0x30; break;
	  case t_ext: mem[opc] += 0x20; break;
	  case t_dir: mem[opc] += 0x10; break;
	  default: break;
	  }
    }
}

. {
  error_at_line(0, 0, fname, line_number, "unknown character `%c'", yytext[0]);
}

%%

void
output(const char *file)
{
  const int recsize = 32;
  FILE *f = fopen(file, "w");
  loc i;
  int h;
  struct symbol *s;
  
  if (f == NULL)
    error(1, errno, "couldn't write %s", file);

  fprintf(f, "$$ %s\n", fname);
  for (h = 0; h < sizeof(htable)/sizeof(htable[0]); h++)
    for (s = htable[h]; s != NULL; s = s->next)
      fprintf(f, " %s $%X\n", s->sym, s->pc);
  fprintf(f, "$$\n");

  for (i = 0; i < 0x10000; i++)
    {
      int j;
      unsigned l;
      unsigned csum;
      
      if (mem[i] == -1)
	continue;
      for (l = 1; l < recsize && i+l < 0x10000; l++)
	if (mem[i+l] == -1)
	  break;
      
      fprintf(f, "S1%02X%04X", l+3, i);
      csum = (i >> 8) + i + l+3;
      for (j = i; j < i+l; j++)
	{
	  fprintf(f, "%02X", mem[j] & 0xFF);
	  csum += mem[j];
	}
      fprintf(f, "%02X\n", 255-(csum % 256));
      i += l-1;
    }
  
  fprintf(f, "S9%02X%04X%02X\n", 3, 0, 0);
  fclose(f);
}

int
main(int argc, char **argv)
{
  int i;
  
  if (argc != 3)
    error(3, 0, "usage: %s input-file output-file", argv[0]);
  fname = argv[1];
  yyin = fopen(fname, "r");
  if (yyin == NULL)
    error(1, errno, "couldn't open `%s'", fname);

  for (i = 0; i < sizeof(mem)/sizeof(mem[0]); i++)
    mem[i] = -1;
  
  yylex();
  
  while (fix_chain != NULL)
    {
      int junk;
      
      line_number = fix_chain->line;
      (void)eval_expression(fix_chain->expression, 
			    strlen(fix_chain->expression), 
			    &junk, fix_chain->where,
			    fix_chain->kind, 1);
      fix_chain = fix_chain->next;
    }

  output(argv[2]);
  return error_message_count > 0 ? 2 : 0;
}
