-- *************************************************************************
-- DISCLAIMER. THIS SOFTWARE WAS WRITTEN BY EMPLOYEES OF THE U.S.
-- GOVERNMENT AS A PART OF THEIR OFFICIAL DUTIES AND, THEREFORE, IS NOT
-- PROTECTED BY COPYRIGHT. HOWEVER, THIS SOFTWARE CODIFIES THE FINALIST
-- CANDIDATE ALGORITHMS (i.e., MARS, RC6tm, RIJNDAEL, SERPENT, AND
-- TWOFISH) IN THE ADVANCED ENCRYPTION STANDARD (AES) DEVELOPMENT EFFORT
-- SPONSORED BY THE NATIONAL INSTITUTE OF STANDARDS AND TECHNOLOGY (NIST)
-- AND MAY BE PROTECTED BY ONE OR MORE FORMS OF INTELLECTUAL PROPERTY. THE
-- U.S. GOVERNMENT MAKES NO WARRANTY, EITHER EXPRESSED OR IMPLIED,
-- INCLUDING BUT NO LIMITED TO ANY IMPLIED WARRANTIES OF MERCHANTABILITY
-- OR FITNESS FOR A PARTICULAR PURPOSE, REGARDING THIS SOFTWARE. THE U.S.
-- GOVERNMENT FURTHER MAKES NO WARRANTY THAT THIS SOFTWARE WILL NOT
-- INFRINGE ANY OTHER UNITED STATES OR FOREIGN PATENT OR OTHER
-- INTELLECTUAL PROPERTY RIGHT. IN NO EVENT SHALL THE U.S. GOVERNMENT BE
-- LIABLE TO ANYONE FOR COMPENSATORY, PUNITIVE, EXEMPLARY, SPECIAL,
-- COLLATERAL, INCIDENTAL, CONSEQUENTIAL, OR ANY OTHER TYPE OF DAMAGES IN
-- CONNECTION WITH OR ARISING OUT OF COPY OR USE OF THIS SOFTWARE.
-- *************************************************************************

-- ===========================================================================
-- File Name : TWOFISH_Top_Iterative.vhdl
-- Author    : NSA
-- Date      : January 2000
-- Project   : AES Candidate Evaluation - TWOFISH
-- Purpose   : This model is the top level structural model for an
--             iterative implementation of TWOFISH, an Advanced Encryption
--             Standard Candidate. It consists of port mappings among the
--             lower level components.
-- Notes     :
-- ===========================================================================

library ieee;
use ieee.std_logic_1164.all;
use WORK.twofish_pack.all;

-- ===========================================================================
-- =========================== Interface Description =========================
-- ===========================================================================

entity TWOFISH_TOP_ITER is

  port (clock     : in std_logic;
        reset     : in std_logic;

        ENC_DEC_B : in std_logic;  -- '1' = encrypt, '0' = decrypt
        DATA_IN   : in SLV_128;    -- 128-bit input data word (plaintext)
        DATA_LOAD : in std_logic;  -- data valid; load new input data word
        CV_IN     : in SLV_256;    -- 128, 192, 256-bit cv (user supplied key)
        CV_LOAD   : in std_logic;  -- cv_in is valid; load new cryptovariable
        CV_SIZE   : in SLV_2;      -- '00'= 128, '01'= 192, '10'= 256

        DATA_OUT  : out SLV_128;   -- 128-bit output data word (ciphertext)
        DONE      : out std_logic  -- indicates 'data_out' is valid

  );

end TWOFISH_TOP_ITER;

architecture STRUCTURAL of TWOFISH_TOP_ITER is


-- ===========================================================================
-- =========================== Component Definition ==========================
-- ===========================================================================

component INTERFACE_ITERATIVE
  port (clock          :  in std_logic;
        reset          :  in std_logic;

        DATA_LOAD      :  in std_logic;     -- data load pulse
        DATAIN         :  in SLV_128;       -- 128 bit block
        CV_LOAD        :  in std_logic;     -- crypto variable load pulse
        CV_SIZE        :  in SLV_2;         -- '00'= 128, '01'= 192, '10'= 256
        CVIN           :  in SLV_256;
        ENC_DEC_B      :  in std_logic;     -- '1' = encrypt, '0' = decrypt

        CTRL_DATA_LOAD :  out std_logic;    -- data load signal to controller
        CTRL_ENC_DEC_B :  out std_logic;
        ALG_DATA       :  out SLV_128;      -- 128 bit data block to algorithm
        KS_CVLOAD      :  out std_logic;
        KS_CV          :  out SLV_256;
        KS_CVSIZE      :  out SLV_2
  );
end component;


component CONTROL_ITER
  port (clock          :  in std_logic;
        reset          :  in std_logic;

        DATA_LOAD      :  in std_logic;   -- data start signal from interface

        CTRL_ALG_START :  out std_logic;  -- start encryption
        CTRL_KS_START  :  out std_logic   -- start key schedule       
  );
end component;

component ALG_ITERATIVE
 port ( clock              :  in STD_LOGIC;
        reset              :  in STD_LOGIC;

        ALG_DATAIN         :  in SLV_128;
        ALG_CV_LENGTH      :  in SLV_2;
        ALG_S              :  in S_ARRAY_TYPE;
        ALG_ENC            :  in std_logic;
        ALG_START          :  in STD_LOGIC;
        ALG_PREADD_KEY     :  in SLV_128;
        ALG_POSTADD_KEY    :  in SLV_128;
        ALG_ROUND_KEY      :  in SLV_64;

        ALG_DATAOUT        :  out SLV_128;
        ALG_DONE           :  out STD_LOGIC
  );
end component;

component KEY_SCHEDULE_ITERATIVE
  port (clock            :  in std_logic;    -- clock signal
        reset            :  in std_logic;    -- active high reset (asynch)

        KS_CVLOAD        :  in std_logic;    -- new CV load
        KS_START         :  in std_logic;    -- start a new expansion sequence
        KS_CV            :  in SLV_256;      -- cryptovariable input bus
        KS_CVSIZE        :  in SLV_2;        -- cryptovariable size select
        KS_ENC_DECB      :  in std_logic;

        KS_S             :  out S_ARRAY_TYPE;
        KS_PREADD_KEY    :  out SLV_128;
        KS_POSTADD_KEY   :  out SLV_128;
        KS_ROUND_KEY     :  out SLV_64
  );
end component;



-- ===========================================================================
-- =========================== Signal Definition =============================
-- ===========================================================================

signal top_datain     : SLV_128;        -- top level data interconnection
signal top_dataload   : std_logic;      -- start new data connection
signal top_cvload     : std_logic;      -- start new cv
signal top_cvsize     : SLV_2;          -- cryptovariable size interconnect
signal top_cv         : SLV_256;        -- cryptovariable bus interconnect
signal top_enc_decb   : std_logic;      -- encrypt select interconnect
signal top_ks_start   : std_logic;      -- key expansion start signal
signal top_s          : S_ARRAY_TYPE;   -- S values for algorithm
signal top_preadd_key : SLV_128;        -- pre-whiten key interconnect
signal top_postadd_key: SLV_128;        -- post-whiten key interconnect
signal top_round_key  : SLV_64;         -- algorithm round key interconnect
signal top_alg_start  : std_logic;      -- algorithm start connection
signal top_done       : std_logic;      -- done signal connection

begin                                                                                                                                                                                                                             
INTER : INTERFACE_ITERATIVE port map (
         clock,             -- rising edge clock
         reset,             -- active high reset
         DATA_LOAD,         -- ext. load new data
         DATA_IN,           -- ext. data input 
         CV_LOAD,           -- ext. load new cv
         CV_SIZE,           -- ext. cv size select
         CV_IN,             -- ext. cv input bus
         ENC_DEC_B,         -- ext. encrypt select
         top_dataload,      -- start new data
         top_enc_decb,      -- encrypt select intercon.
         top_datain,        -- data interconnect
         top_cvload,        -- load new cv intercon.
         top_cv,            -- cv bus connection
         top_cvsize );      -- 128, 192, 256

CTRL : CONTROL_ITER port map (
         clock,           -- rising edge clock
         reset,           -- active high reset
         top_dataload,    -- process new data
         top_alg_start,   -- start alg processing
         top_ks_start );  -- start key expansion


ALG : ALG_ITERATIVE port map (
         clock,           -- rising edge clock  
         reset,           -- active high reset
         top_datain,      -- input data
         top_cvsize,      -- 128, 192, 256
         top_s,           -- S value output
         top_enc_decb,    -- encrypt select
         top_alg_start,   -- start processing 
         top_preadd_key,  -- key output
         top_postadd_key, -- key output
         top_round_key,   -- key output
         DATA_OUT,        -- processed data output
         top_done );      -- processing complete


KEYSCH : Key_Schedule_Iterative port map (
         clock,            -- rising edge
         reset,            -- active high
         top_cvload,       -- process new cv
         top_ks_start,     -- start key expan
         top_cv,           -- cv input bus
         top_cvsize,       -- 128, 192, 256
         top_enc_decb,     -- encrypt select intercon.
         top_s,            -- S value output
         top_preadd_key,   -- key output
         top_postadd_key,  -- key output
         top_round_key );  -- key output



DONE <= top_done;


end STRUCTURAL;


-- ===========================================================================
-- =========================== Configuration =================================
-- ===========================================================================


configuration CFG_TWOFISH_TOP_ITER of TWOFISH_TOP_ITER is

   for STRUCTURAL

      for CTRL: CONTROL_ITER use
         entity work.CONTROL_ITER(CONTROL_ITER_RTL);
      end for;

      for all: KEY_SCHEDULE_ITERATIVE use
         entity work.KEY_SCHEDULE_ITERATIVE(KEY_SCHEDULE_ITERATIVE_RTL);
      end for;

      for all: ALG_ITERATIVE use
         entity work.ALG_ITERATIVE(ALG_ITERATIVE_RTL);
      end for;

      for INTER: INTERFACE_ITERATIVE USE
         entity work.INTERFACE_ITERATIVE(INTERFACE_ITERATIVE_RTL);
      end for;

   end for;

end CFG_TWOFISH_TOP_ITER;

