-- *************************************************************************
-- DISCLAIMER. THIS SOFTWARE WAS WRITTEN BY EMPLOYEES OF THE U.S.
-- GOVERNMENT AS A PART OF THEIR OFFICIAL DUTIES AND, THEREFORE, IS NOT
-- PROTECTED BY COPYRIGHT. HOWEVER, THIS SOFTWARE CODIFIES THE FINALIST
-- CANDIDATE ALGORITHMS (i.e., MARS, RC6tm, RIJNDAEL, SERPENT, AND
-- TWOFISH) IN THE ADVANCED ENCRYPTION STANDARD (AES) DEVELOPMENT EFFORT
-- SPONSORED BY THE NATIONAL INSTITUTE OF STANDARDS AND TECHNOLOGY (NIST)
-- AND MAY BE PROTECTED BY ONE OR MORE FORMS OF INTELLECTUAL PROPERTY. THE
-- U.S. GOVERNMENT MAKES NO WARRANTY, EITHER EXPRESSED OR IMPLIED,
-- INCLUDING BUT NO LIMITED TO ANY IMPLIED WARRANTIES OF MERCHANTABILITY
-- OR FITNESS FOR A PARTICULAR PURPOSE, REGARDING THIS SOFTWARE. THE U.S.
-- GOVERNMENT FURTHER MAKES NO WARRANTY THAT THIS SOFTWARE WILL NOT
-- INFRINGE ANY OTHER UNITED STATES OR FOREIGN PATENT OR OTHER
-- INTELLECTUAL PROPERTY RIGHT. IN NO EVENT SHALL THE U.S. GOVERNMENT BE
-- LIABLE TO ANYONE FOR COMPENSATORY, PUNITIVE, EXEMPLARY, SPECIAL,
-- COLLATERAL, INCIDENTAL, CONSEQUENTIAL, OR ANY OTHER TYPE OF DAMAGES IN
-- CONNECTION WITH OR ARISING OUT OF COPY OR USE OF THIS SOFTWARE.
-- *************************************************************************

-- ===========================================================================
-- File Name : key_sched_pipe.vhdl
-- Author    : NSA
-- Date      : January 2000
-- Project   : AES Candidate Evaluation - TWOFISH
-- Purpose   : build key schedule for pipelined implementation
-- Notes     :
-- ===========================================================================

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use WORK.twofish_pack.all;

-- ===========================================================================
-- =========================== Interface Description =========================
-- ===========================================================================

entity KEY_SCHEDULE_PIPE is

  port (clock            :  in std_logic;     -- clock signal
        reset            :  in std_logic;     -- active high reset (asynch)

        KS_LOADCV        :  in std_logic;        -- load a new cryptovariable
        KS_CV            :  in SLV_256;          -- cryptovariable input bus
        KS_CVSIZE        :  in CVSIZE_PIPE_TYPE; 
        KS_ENC           :  in std_logic;
        KS_S             :  out S_PIPE_TYPE;     -- S-box keys (pipelined)
        KS_ROUND_KEY_ENC :  out KEY_ARRAY_TYPE;  -- encrypt round key output
        KS_ROUND_KEY_DEC :  out KEY_ARRAY_TYPE   -- decrypt round key output
  );

end KEY_SCHEDULE_PIPE;

architecture KEY_SCHEDULE_PIPE_RTL of KEY_SCHEDULE_PIPE is


-- ===========================================================================
-- =========================== Component Definition ==========================
-- ===========================================================================

component REG32B 

  port (clock      :  in std_logic;   -- clock signal
        reset      :  in std_logic;   -- active high reset (asynchronous)

        DATA_IN    :  in SLV_32;     -- input data bus
        DATA_OUT   :  out SLV_32     -- output data bus

  );

end component;


component REG64B

  port (clock    :  in std_logic;   -- clock signal
        reset    :  in std_logic;   -- active high reset (asynchronous)
    
        DATA_IN  :  in SLV_64;      -- input data bus

        DATA_OUT :  out SLV_64      -- output data bus

  );

end component;


component KS_ROUND is

  port ( round        : in SLV_5;
         m_value_even : in M_PAIR_TYPE;
         m_value_odd  : in M_PAIR_TYPE;
         CV_LENGTH    : in SLV_2;
         dataout      : out SLV_64
  );

end component;


type ROUND_VEC_TYPE is array (0 to 19) of SLV_5;
constant ROUND_VEC : ROUND_VEC_TYPE := (
  "00000", "00001", "00010", "00011",
  "00100", "00101", "00110", "00111",
  "01000", "01001", "01010", "01011",
  "01100", "01101", "01110", "01111",
  "10000", "10001", "10010", "10011" );

type ROUND_IN_TYPE     is array (0 to 15) of SLV_5;
type KS_ROUND_OUT_TYPE is array (0 to 15) of SLV_64;

-- ===========================================================================
-- =========================== Signal Definition =============================
-- ===========================================================================

signal M_EVEN             : M_PAIR_PIPE_TYPE;
signal M_ODD              : M_PAIR_PIPE_TYPE;
signal S_VALUE            : S_PIPE_TYPE;    -- pipeline this one here??
signal ROUND_OUTPUT_0A    : SLV_64;
signal ROUND_OUTPUT_0B    : SLV_64;
signal ROUND_OUTPUT_1A    : SLV_64;
signal ROUND_OUTPUT_1B    : SLV_64;
signal ROUND_OUTPUT_ENC   : PIPE_SUBKEY_TYPE;
signal ROUND_OUTPUT_DEC   : PIPE_SUBKEY_TYPE;
signal POSTADD_SUBKEY_A   : SLV_64;
signal POSTADD_SUBKEY_B   : SLV_64;
signal PIPE_POSTADD_KEY_A : POSTADD_PIPE_TYPE;
signal PIPE_POSTADD_KEY_B : POSTADD_PIPE_TYPE;
signal SUBKEY             : KEY_ARRAY_TYPE;
signal SUBKEY_ENC         : KEY_ARRAY_TYPE;
signal SUBKEY_DEC         : KEY_ARRAY_TYPE;
signal ROUND_NUM          : ROUND_VEC_TYPE;
signal ROUND_IN           : ROUND_IN_TYPE;
signal KS_ROUND_OUT       : KS_ROUND_OUT_TYPE;

begin

-- ===========================================================================
-- ====================== Generate Pipe Structure ============================
-- ===========================================================================
--
-- PURPOSE:  
--
-- The following generate statements create a pipelined architecture for
-- the key expansion. The first two rounds generate 128-bits of key each
-- to be used in the pre-whiten and post-whiten stages. The remaining 16
-- rounds generate 64 bits of key to be used in the TWOFISH algorithm.
--
-- ===========================================================================

-- ---------------------------------------------------------------------------
-- CALCULATIONS

-- Assign round numbers to signal
  ROUND_NUM <= ROUND_VEC;

-- Parse CV for M and S values
  MINST: MP_ROUND(KS_CV,M_EVEN(0),M_ODD(0));
  SINST: S_VALUE_ROUND(KS_CV,S_VALUE(0));

-- Generate round subkeys
  -- Pre-whiten subkeys (post-whiten for decrypt mode)
  ROUND0A: KS_ROUND port map ( ROUND_NUM(0),
                               M_EVEN(0),
                               M_ODD(0),
                               KS_CVSIZE(0),
                               ROUND_OUTPUT_0A );
  ROUND0B: KS_ROUND port map ( ROUND_NUM(1),
                               M_EVEN(0),
                               M_ODD(0),
                               KS_CVSIZE(0),
                               ROUND_OUTPUT_0B );

  -- Post-whiten subkeys (pre-whiten for decrypt mode)
  ROUND1A: KS_ROUND port map ( ROUND_NUM(2),
                               M_EVEN(0),
                               M_ODD(0), 
                               KS_CVSIZE(0),
                               ROUND_OUTPUT_1A );
  ROUND1B: KS_ROUND port map ( ROUND_NUM(3),
                               M_EVEN(0),
                               M_ODD(0),
                               KS_CVSIZE(0),
                               ROUND_OUTPUT_1B );

  -- Regular round subkeys
  G1: for ROUNDKEY_STEP in 4 to 19 generate

    ROUND_IN(ROUNDKEY_STEP-4) <= ROUND_NUM(ROUNDKEY_STEP) when KS_ENC = '1' else
                                 ROUND_NUM(23-ROUNDKEY_STEP);

    ROUND_E: KS_ROUND port map (ROUND_IN(ROUNDKEY_STEP-4), 
                                M_EVEN(ROUNDKEY_STEP-4), 
                                M_ODD(ROUNDKEY_STEP-4), 
                                KS_CVSIZE(ROUNDKEY_STEP-4), 
                                KS_ROUND_OUT(ROUNDKEY_STEP-4));

    ROUND_OUTPUT_ENC(ROUNDKEY_STEP-4) <= KS_ROUND_OUT(ROUNDKEY_STEP-4);
    ROUND_OUTPUT_DEC(ROUNDKEY_STEP-4) <= KS_ROUND_OUT(ROUNDKEY_STEP-4);

  end generate;


-- ---------------------------------------------------------------------------
-- REGISTERS

-- Setup pipeline for M values and S values
  G2: for ROUNDKEY_STEP in FIRST_ROUND to LAST_ROUND generate
    G2a: for i in 0 to 3 generate
      G2b: if ROUNDKEY_STEP < (LAST_ROUND - 1) generate
        MR0: REG32B port map ( clock,
                               reset,
                               M_EVEN(ROUNDKEY_STEP)(i),
                               M_EVEN(ROUNDKEY_STEP+1)(i) );

        MR1: REG32B port map ( clock,
                               reset,
                               M_ODD(ROUNDKEY_STEP)(i),
                               M_ODD(ROUNDKEY_STEP+1)(i) );

        SR0: REG32B port map ( clock,
                               reset,
                               S_VALUE(ROUNDKEY_STEP)(i),
                               S_VALUE(ROUNDKEY_STEP+1)(i) );
      end generate;
    end generate;
  end generate;


-- Set pre-whiten subkey values/Register post-whiten subkey values
  SUBKEY(0)  <= ROUND_OUTPUT_0A(63 downto 32)
                        when KS_ENC = '1' else
                      ROUND_OUTPUT_1A(63 downto 32);
  SUBKEY(1)  <= ROUND_OUTPUT_0A(31 downto 0)
                        when KS_ENC = '1' else
                      ROUND_OUTPUT_1A(31 downto 0);
  SUBKEY(2)  <= ROUND_OUTPUT_0B(63 downto 32)
                        when KS_ENC = '1' else
                      ROUND_OUTPUT_1B(63 downto 32);
  SUBKEY(3)  <= ROUND_OUTPUT_0B(31 downto 0)
                        when KS_ENC = '1' else
                      ROUND_OUTPUT_1B(31 downto 0);


  POSTADD_SUBKEY_A <= ROUND_OUTPUT_1A
                       when KS_ENC = '1' else
                     ROUND_OUTPUT_0A;
  POSTADD_SUBKEY_B <= ROUND_OUTPUT_1B
                       when KS_ENC = '1' else
                     ROUND_OUTPUT_0B;

  G3: for i in 0 to 1 generate

    KR2: REG32B port map ( clock,
                         reset,
                         POSTADD_SUBKEY_A(63-(32*i) downto 32-(32*i)),
                         SUBKEY(i+4) );
    KR3: REG32B port map ( clock,
                         reset,
                         POSTADD_SUBKEY_B(63-(32*i) downto 32-(32*i)),
                         SUBKEY(i+6) );
  end generate;


-- Register round subkeys
  G4: for ROUNDKEY_STEP in 4 to 19 generate
    G4a: for i in 0 to 1 generate
      KR_E: REG32B port map ( clock,
                  reset,
                  ROUND_OUTPUT_ENC(ROUNDKEY_STEP-4)(63-(32*i) downto 32-(32*i)),
                  SUBKEY_ENC(2 * ROUNDKEY_STEP + i) );
      KR_D: REG32B port map ( clock,
                 reset,
                 ROUND_OUTPUT_DEC(19-ROUNDKEY_STEP)(63-(32*i) downto 32-(32*i)),
                 SUBKEY_DEC(2 * ROUNDKEY_STEP + i) );
    end generate;
  end generate;


-- Pipeline registers for post-whiten subkeys
  PIPE_POSTADD_KEY_A(0) <= SUBKEY(4) & SUBKEY(5);
  PIPE_POSTADD_KEY_B(0) <= SUBKEY(6) & SUBKEY(7);

  G5: for ALG_ROUND in 0 to 15 generate

    PR3: REG64B port map ( clock,
                             reset,
                             PIPE_POSTADD_KEY_A(ALG_ROUND),
                             PIPE_POSTADD_KEY_A(ALG_ROUND+1) );

    PR4: REG64B port map ( clock,
                             reset,
                             PIPE_POSTADD_KEY_B(ALG_ROUND),
                             PIPE_POSTADD_KEY_B(ALG_ROUND+1) );
  end generate;




-- ---------------------------------------------------------------------------
-- ASSIGN OUTPUTS

-- Preadd
  G6: for ROUND in 0 to 3 generate
    KS_ROUND_KEY_ENC(ROUND) <= SUBKEY(ROUND);
    KS_ROUND_KEY_DEC(ROUND) <= SUBKEY(ROUND);
  end generate;

-- Postadd
  KS_ROUND_KEY_ENC(4) <= PIPE_POSTADD_KEY_A(16)(63 downto 32);
  KS_ROUND_KEY_ENC(5) <= PIPE_POSTADD_KEY_A(16)(31 downto 0);
  KS_ROUND_KEY_ENC(6) <= PIPE_POSTADD_KEY_B(16)(63 downto 32);
  KS_ROUND_KEY_ENC(7) <= PIPE_POSTADD_KEY_B(16)(31 downto 0);
  KS_ROUND_KEY_DEC(4) <= PIPE_POSTADD_KEY_A(16)(63 downto 32);
  KS_ROUND_KEY_DEC(5) <= PIPE_POSTADD_KEY_A(16)(31 downto 0);
  KS_ROUND_KEY_DEC(6) <= PIPE_POSTADD_KEY_B(16)(63 downto 32);
  KS_ROUND_KEY_DEC(7) <= PIPE_POSTADD_KEY_B(16)(31 downto 0);

-- Round subkeys
  G8: for ROUND in 8 to 39 generate
    KS_ROUND_KEY_ENC(ROUND) <= SUBKEY_ENC(ROUND);
  end generate;

  G7: for ROUND in 0 to 15 generate
    KS_ROUND_KEY_DEC(2*ROUND+8) <= SUBKEY_DEC(38-2*ROUND);
    KS_ROUND_KEY_DEC(2*ROUND+9) <= SUBKEY_DEC(39-2*ROUND);
  end generate;

-- S values
  G9: for ROUND in FIRST_ROUND to (LAST_ROUND-1) generate
    KS_S(ROUND) <= S_VALUE(ROUND);
  end generate;


end KEY_SCHEDULE_PIPE_RTL;

-- ===========================================================================
-- ========================= Configuration ===================================
-- ===========================================================================

configuration CFG_KEY_SCHEDULE_PIPE of KEY_SCHEDULE_PIPE is

   for KEY_SCHEDULE_PIPE_RTL

   end for;

end;
