-- *************************************************************************
-- DISCLAIMER. THIS SOFTWARE WAS WRITTEN BY EMPLOYEES OF THE U.S.
-- GOVERNMENT AS A PART OF THEIR OFFICIAL DUTIES AND, THEREFORE, IS NOT
-- PROTECTED BY COPYRIGHT. HOWEVER, THIS SOFTWARE CODIFIES THE FINALIST
-- CANDIDATE ALGORITHMS (i.e., MARS, RC6tm, RIJNDAEL, SERPENT, AND
-- TWOFISH) IN THE ADVANCED ENCRYPTION STANDARD (AES) DEVELOPMENT EFFORT
-- SPONSORED BY THE NATIONAL INSTITUTE OF STANDARDS AND TECHNOLOGY (NIST)
-- AND MAY BE PROTECTED BY ONE OR MORE FORMS OF INTELLECTUAL PROPERTY. THE
-- U.S. GOVERNMENT MAKES NO WARRANTY, EITHER EXPRESSED OR IMPLIED,
-- INCLUDING BUT NO LIMITED TO ANY IMPLIED WARRANTIES OF MERCHANTABILITY
-- OR FITNESS FOR A PARTICULAR PURPOSE, REGARDING THIS SOFTWARE. THE U.S.
-- GOVERNMENT FURTHER MAKES NO WARRANTY THAT THIS SOFTWARE WILL NOT
-- INFRINGE ANY OTHER UNITED STATES OR FOREIGN PATENT OR OTHER
-- INTELLECTUAL PROPERTY RIGHT. IN NO EVENT SHALL THE U.S. GOVERNMENT BE
-- LIABLE TO ANYONE FOR COMPENSATORY, PUNITIVE, EXEMPLARY, SPECIAL,
-- COLLATERAL, INCIDENTAL, CONSEQUENTIAL, OR ANY OTHER TYPE OF DAMAGES IN
-- CONNECTION WITH OR ARISING OUT OF COPY OR USE OF THIS SOFTWARE.
-- *************************************************************************

-- ===========================================================================
-- File Name : alg_iterative.vhdl
-- Author    : NSA
-- Date      : December 1999
-- Project   : AES Candidate Evaluation - TWOFISH128
-- Purpose   : This block implements the TWOFISH algorithm for an iterative
--             case
-- Notes     :
-- ===========================================================================

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use WORK.twofish_pack.all;

-- ===========================================================================
-- =========================== Interface Description =========================
-- ===========================================================================

entity ALG_ITERATIVE is

  port (clock              :  in STD_LOGIC;
        reset              :  in STD_LOGIC;

        ALG_DATAIN         :  in SLV_128;
        ALG_S              :  in S_ARRAY_TYPE;
        ALG_ENC            :  in std_logic;

        ALG_START          :  in STD_LOGIC;
        ALG_PREADD_KEY     :  in SLV_128;
        ALG_POSTADD_KEY    :  in SLV_128;
        ALG_ROUND_KEY      :  in SLV_64;
        ALG_DATAOUT        :  out SLV_128;
        ALG_DONE           :  out STD_LOGIC
  );

end ALG_ITERATIVE;

architecture ALG_ITERATIVE_RTL of ALG_ITERATIVE is


-- ===========================================================================
-- =========================== Constant Definition ===========================
-- ===========================================================================

constant HOLD : integer := 0;         -- Hold state for controller

-- ===========================================================================
-- =========================== Signal Definition =============================
-- ===========================================================================

signal ROUND             : ROUND_TYPE;     -- round number
signal D_REG             : SLV_128;        -- data out from round i
signal DONE              : std_logic;
constant LAST_ALG_ROUND  : INTEGER := 16;

begin

ALG_DATAOUT <= D_REG when DONE = '1' else 
               (others => '0');


-- ===========================================================================
-- =========================== Data Movement =================================
-- ===========================================================================

DATA_FLOW: process( clock, reset )

begin

if reset = '1' then                        -- check for reset condition

   D_REG    <= ( others => '0' );          -- clear round key outputs
   ALG_DONE <= '0';                        -- clear done signal

elsif clock'event and clock = '1' then     -- rising edge clock

  case ROUND is

    when FIRST_ROUND => 
      D_REG <= D_REG;
      ALG_DONE    <= '0';

    when FIRST_ROUND + 1 => 

      PRE_ADD( ALG_DATAIN,
               ALG_PREADD_KEY(127 downto 96),
               ALG_PREADD_KEY(95 downto 64),
               ALG_PREADD_KEY(63 downto 32),
               ALG_PREADD_KEY(31 downto 0),
               ALG_PREADD_KEY(127 downto 96),
               ALG_PREADD_KEY(95 downto 64),
               ALG_PREADD_KEY(63 downto 32),
               ALG_PREADD_KEY(31 downto 0),
               ALG_ENC,
               D_REG );

      ALG_DONE    <= '0';
 

    when LAST_ROUND + 1 =>
      D_REG <= D_REG;
      ALG_DONE    <= '0';

    when LAST_ROUND + 2 =>

      POST_ADD( D_REG,
                ALG_POSTADD_KEY(127 downto 96),
                ALG_POSTADD_KEY(95 downto 64),
                ALG_POSTADD_KEY(63 downto 32),
                ALG_POSTADD_KEY(31 downto 0),
                ALG_POSTADD_KEY(127 downto 96),
                ALG_POSTADD_KEY(95 downto 64),
                ALG_POSTADD_KEY(63 downto 32),
                ALG_POSTADD_KEY(31 downto 0),
                ALG_ENC,
                D_REG );


  
      ALG_DONE <= '1';
       
    when others =>

      TWOFISH_ROUND(D_REG ,         -- input data 
            	    ALG_ROUND_KEY(63 downto 32), 
             	    ALG_ROUND_KEY(31 downto 0),
               	    ALG_ROUND_KEY(63 downto 32),
              	    ALG_ROUND_KEY(31 downto 0),
                    ALG_S,
                    std_logic_vector(TO_UNSIGNED(ROUND-1, 5)),
                    ALG_ENC,
                    D_REG );   

      ALG_DONE    <= '0';

             
    end case; 

end if; 


end process; -- DATA_FLOW


-- ===========================================================================
-- =========================== State Machine / Controller ====================
-- ===========================================================================

STATE_FLOW: process( clock, reset )

variable active : std_logic;                -- Indicates whether algorithm is
                                            -- active (1) or in hold (0)

begin

if reset = '1' then                         -- Active high reset (asynch)

   ROUND  <= HOLD;                          -- put controller in hold state
   active := '0';                           -- stop process until ALG_START=1

elsif clock'event and clock = '1' then      -- check for rising edge clock

   if  ( ALG_START = '1' or active = '1' )  -- check for inactive  
     and ( ROUND /= LAST_ROUND+2 ) then       -- and completion of algorithm

      active := '1';                        -- enable controller
      ROUND  <= ROUND +1;                  -- active, so increment counter
      DONE   <= '0';

   elsif ROUND = LAST_ROUND+2 then

      active := '0';                        -- disable controller
      ROUND  <= HOLD;                       -- reset counter
      DONE <= '1';

   else

      active := '0';                        -- disable controller
      ROUND  <= HOLD;                       -- reset counter
      DONE   <= '0';

   end if;

end if; -- reset

end process;

end ALG_ITERATIVE_RTL;
