-- *************************************************************************
-- DISCLAIMER. THIS SOFTWARE WAS WRITTEN BY EMPLOYEES OF THE U.S.
-- GOVERNMENT AS A PART OF THEIR OFFICIAL DUTIES AND, THEREFORE, IS NOT
-- PROTECTED BY COPYRIGHT. HOWEVER, THIS SOFTWARE CODIFIES THE FINALIST
-- CANDIDATE ALGORITHMS (i.e., MARS, RC6tm, RIJNDAEL, SERPENT, AND
-- TWOFISH) IN THE ADVANCED ENCRYPTION STANDARD (AES) DEVELOPMENT EFFORT
-- SPONSORED BY THE NATIONAL INSTITUTE OF STANDARDS AND TECHNOLOGY (NIST)
-- AND MAY BE PROTECTED BY ONE OR MORE FORMS OF INTELLECTUAL PROPERTY. THE
-- U.S. GOVERNMENT MAKES NO WARRANTY, EITHER EXPRESSED OR IMPLIED,
-- INCLUDING BUT NO LIMITED TO ANY IMPLIED WARRANTIES OF MERCHANTABILITY
-- OR FITNESS FOR A PARTICULAR PURPOSE, REGARDING THIS SOFTWARE. THE U.S.
-- GOVERNMENT FURTHER MAKES NO WARRANTY THAT THIS SOFTWARE WILL NOT
-- INFRINGE ANY OTHER UNITED STATES OR FOREIGN PATENT OR OTHER
-- INTELLECTUAL PROPERTY RIGHT. IN NO EVENT SHALL THE U.S. GOVERNMENT BE
-- LIABLE TO ANYONE FOR COMPENSATORY, PUNITIVE, EXEMPLARY, SPECIAL,
-- COLLATERAL, INCIDENTAL, CONSEQUENTIAL, OR ANY OTHER TYPE OF DAMAGES IN
-- CONNECTION WITH OR ARISING OUT OF COPY OR USE OF THIS SOFTWARE.
-- *************************************************************************

-- ===========================================================================
-- File Name : key_sched_iterative.vhdl
-- Author    : NSA
-- Date      : January 2000
-- Project   : AES Candidate Evaluation - TWOFISH128
-- Purpose   : build key schedule for iterative implementation
-- Notes     :
-- ===========================================================================

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use WORK.twofish_pack.all;

-- ===========================================================================
-- =========================== Interface Description =========================
-- ===========================================================================

entity KEY_SCHEDULE_ITERATIVE is

  port (clock            :  in std_logic;    -- clock signal
        reset            :  in std_logic;    -- active high reset (asynch)

        KS_CVLOAD        :  in std_logic;    -- new CV load
        KS_START         :  in std_logic;    -- start a new expansion sequence
        KS_CV            :  in SLV_128;      -- cryptovariable input bus
        KS_ENC_DECB      :  in std_logic;

        KS_S             :  out S_ARRAY_TYPE;
        KS_PREADD_KEY    :  out SLV_128;
        KS_POSTADD_KEY   :  out SLV_128;
        KS_ROUND_KEY     :  out SLV_64
  );

end KEY_SCHEDULE_ITERATIVE;

architecture KEY_SCHEDULE_ITERATIVE_RTL of KEY_SCHEDULE_ITERATIVE is


-- ===========================================================================
-- =========================== Type Definition ===============================
-- ===========================================================================

type EXPAND_STATE_TYPE is ( HOLD, CV_EXPAND );


-- ===========================================================================
-- =========================== Signal Definition =============================
-- ===========================================================================

signal M_EVEN             : M_PAIR_TYPE;
signal M_ODD              : M_PAIR_TYPE;
signal S                  : S_ARRAY_TYPE;
signal CV_EXPAND_STEP     : ROUND_TYPE;
signal EXPAND_STEP        : ROUND_TYPE;
signal EXPAND_STATE       : EXPAND_STATE_TYPE;
signal ROUND              : ROUND_TYPE;
signal KS_ROUND_KEY_INT   : SLV_64;
signal KS_PREADD_KEY_INT  : SLV_128;
signal KS_POSTADD_KEY_INT : SLV_128;

begin


-- ===========================================================================
-- =========================== Signal Mapping ================================
-- ===========================================================================
  MP_ROUND(KS_CV, M_EVEN, M_ODD);

  KS_S <= S;

  KS_PREADD_KEY  <= KS_PREADD_KEY_INT;
  KS_POSTADD_KEY <= KS_POSTADD_KEY_INT;

    KS_ROUND(std_logic_vector(TO_UNSIGNED(CV_EXPAND_STEP, 5)), M_EVEN, M_ODD, 
             KS_ROUND_KEY_INT);

-- ===========================================================================
-- =========================== Data Movement =================================
-- ===========================================================================

DATA_FLOW : process( clock, reset )


begin

  if reset = '1' then
    for i in 0 to 1 loop
      S(i)         <= (others => '0');
    end loop;
    KS_ROUND_KEY       <= (others => '0');
    KS_PREADD_KEY_INT  <= (others => '0');
    KS_POSTADD_KEY_INT <= (others => '0');

  elsif clock'event and clock = '1' then

    if KS_CVLOAD = '1' then
      S_VALUE_ROUND(KS_CV, S);
    else
      S      <= S;
    end if;

    KS_ROUND_KEY <= KS_ROUND_KEY_INT;


    if KS_ENC_DECB = '1' then

      case CV_EXPAND_STEP is

        when 0 =>
          KS_PREADD_KEY_INT(127 downto 64)  <= KS_ROUND_KEY_INT;
          KS_PREADD_KEY_INT(63 downto 0)    <= KS_PREADD_KEY_INT(63 downto 0);
        when 1 =>
          KS_PREADD_KEY_INT(127 downto 64)  <= KS_PREADD_KEY_INT(127 downto 64);
          KS_PREADD_KEY_INT(63 downto 0)    <= KS_ROUND_KEY_INT;
        when 2 =>
          KS_POSTADD_KEY_INT(127 downto 64) <= KS_ROUND_KEY_INT;
          KS_POSTADD_KEY_INT(63 downto 0)   <= KS_POSTADD_KEY_INT(63 downto 0);
        when 3 =>
          KS_POSTADD_KEY_INT(127 downto 64) <= KS_POSTADD_KEY_INT(127 downto 64);
          KS_POSTADD_KEY_INT(63 downto 0)   <= KS_ROUND_KEY_INT;
        when others =>
          KS_PREADD_KEY_INT  <= (others => '0');
          KS_POSTADD_KEY_INT <= (others => '0');

      end case;

    else

      case CV_EXPAND_STEP is

        when 2 =>
          KS_PREADD_KEY_INT(127 downto 64)  <= KS_ROUND_KEY_INT;
          KS_PREADD_KEY_INT(63 downto 0)    <= KS_PREADD_KEY_INT(63 downto 0);
        when 3 =>
          KS_PREADD_KEY_INT(127 downto 64)  <= KS_PREADD_KEY_INT(127 downto 64);
          KS_PREADD_KEY_INT(63 downto 0)    <= KS_ROUND_KEY_INT;
        when 0 =>
          KS_POSTADD_KEY_INT(127 downto 64) <= KS_ROUND_KEY_INT;
          KS_POSTADD_KEY_INT(63 downto 0)   <= KS_POSTADD_KEY_INT(63 downto 0);
        when 1 =>
          KS_POSTADD_KEY_INT(127 downto 64) <= KS_POSTADD_KEY_INT(127 downto 64);
          KS_POSTADD_KEY_INT(63 downto 0)   <= KS_ROUND_KEY_INT;
        when others =>
          KS_PREADD_KEY_INT  <= (others => '0');
          KS_POSTADD_KEY_INT <= (others => '0');

      end case;

    end if;

  end if;

end process; -- DATA_FLOW




-- ===========================================================================
-- =========================== State Machine / Controller ====================
-- ===========================================================================

EXPAND_FLOW: process( clock, reset )

begin

  if reset = '1' then                     -- Active high reset (asynchronous)

    CV_EXPAND_STEP <= 0;                 -- put controller in hold state
    EXPAND_STATE   <= HOLD;              -- no activity for expansion

  elsif clock'event and clock = '1' then

    case EXPAND_STATE is

      when HOLD =>


        if KS_START = '1' then
          if KS_ENC_DECB = '1' then
            CV_EXPAND_STEP <= 1;
          else
            CV_EXPAND_STEP <= 3;
          end if;
          EXPAND_STATE   <= CV_EXPAND;
        else
          EXPAND_STATE   <= HOLD;
          if KS_ENC_DECB = '1' then
            CV_EXPAND_STEP <= 0;
          else
            CV_EXPAND_STEP <= 2;
          end if;
        end if;

      when CV_EXPAND =>

        if KS_ENC_DECB = '1' then

          case ( CV_EXPAND_STEP ) is
            when 3 =>
              EXPAND_STATE   <= HOLD;
              CV_EXPAND_STEP <= 0;
            when 1 =>
              EXPAND_STATE   <= EXPAND_STATE;
              CV_EXPAND_STEP <= 4;
            when 19 =>
              EXPAND_STATE   <= EXPAND_STATE;
              CV_EXPAND_STEP <= 2;
            when others =>
              EXPAND_STATE   <= EXPAND_STATE;
              CV_EXPAND_STEP <= CV_EXPAND_STEP + 1;
          end case;

        else

          case ( CV_EXPAND_STEP ) is
            when 1 =>
              EXPAND_STATE   <= HOLD;
              CV_EXPAND_STEP <= 2;
            when 2 =>
              EXPAND_STATE   <= EXPAND_STATE;
              CV_EXPAND_STEP <= 3;
            when 3 =>
              EXPAND_STATE   <= EXPAND_STATE;
              CV_EXPAND_STEP <= 19;
            when 4 =>
              EXPAND_STATE   <= EXPAND_STATE;
              CV_EXPAND_STEP <= 0;
            when 0 =>
              EXPAND_STATE   <= EXPAND_STATE;
              CV_EXPAND_STEP <= 1;
            when others =>
              EXPAND_STATE   <= EXPAND_STATE;
              CV_EXPAND_STEP <= CV_EXPAND_STEP - 1;
          end case;

        end if;

      when others =>

        EXPAND_STATE   <= EXPAND_STATE;
        CV_EXPAND_STEP <= CV_EXPAND_STEP;

    end case;
                                             
  end if;  -- reset = '1'

end process; -- EXPAND_FLOW

end KEY_SCHEDULE_ITERATIVE_RTL;
