-- *************************************************************************
-- DISCLAIMER. THIS SOFTWARE WAS WRITTEN BY EMPLOYEES OF THE U.S.
-- GOVERNMENT AS A PART OF THEIR OFFICIAL DUTIES AND, THEREFORE, IS NOT
-- PROTECTED BY COPYRIGHT. HOWEVER, THIS SOFTWARE CODIFIES THE FINALIST
-- CANDIDATE ALGORITHMS (i.e., MARS, RC6tm, RIJNDAEL, SERPENT, AND
-- TWOFISH) IN THE ADVANCED ENCRYPTION STANDARD (AES) DEVELOPMENT EFFORT
-- SPONSORED BY THE NATIONAL INSTITUTE OF STANDARDS AND TECHNOLOGY (NIST)
-- AND MAY BE PROTECTED BY ONE OR MORE FORMS OF INTELLECTUAL PROPERTY. THE
-- U.S. GOVERNMENT MAKES NO WARRANTY, EITHER EXPRESSED OR IMPLIED,
-- INCLUDING BUT NO LIMITED TO ANY IMPLIED WARRANTIES OF MERCHANTABILITY
-- OR FITNESS FOR A PARTICULAR PURPOSE, REGARDING THIS SOFTWARE. THE U.S.
-- GOVERNMENT FURTHER MAKES NO WARRANTY THAT THIS SOFTWARE WILL NOT
-- INFRINGE ANY OTHER UNITED STATES OR FOREIGN PATENT OR OTHER
-- INTELLECTUAL PROPERTY RIGHT. IN NO EVENT SHALL THE U.S. GOVERNMENT BE
-- LIABLE TO ANYONE FOR COMPENSATORY, PUNITIVE, EXEMPLARY, SPECIAL,
-- COLLATERAL, INCIDENTAL, CONSEQUENTIAL, OR ANY OTHER TYPE OF DAMAGES IN
-- CONNECTION WITH OR ARISING OUT OF COPY OR USE OF THIS SOFTWARE.
-- *************************************************************************

-- ===========================================================================
-- File Name: twofish_pkg.vhdl
-- Author   : NSA
-- Date     : August 1999
-- Project  : AES Candidate Evaluation - TWOFISH128
-- Purpose  : This package defines common types, subtypes, constants,
--            and functions required to implement various VHDL models
--            for the creation of ASIC simulation of TWOFISH, an Advanced
--            Encryption Standard (AES) candidate algorithm.
--
-- ===========================================================================

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;


package twofish_pack is

-- ==========================================================================
--              Type and sub-type declarations for general use             --
-- ==========================================================================

subtype ROUND_TYPE      is integer range 0 to 19;

constant FIRST_ROUND    : ROUND_TYPE := 0;
constant LAST_ROUND     : ROUND_TYPE := 17;


subtype SLV_4           is std_logic_vector(3 downto 0);
subtype SLV_5           is std_logic_vector(4 downto 0);
subtype SLV_8           is std_logic_vector(7 downto 0);
subtype SLV_9           is std_logic_vector(8 downto 0);
subtype SLV_15          is std_logic_vector(14 downto 0);
subtype SLV_16          is std_logic_vector(15 downto 0);
subtype SLV_18          is std_logic_vector(17 downto 0);
subtype SLV_32          is std_logic_vector(31 downto 0);
subtype SLV_33          is std_logic_vector(32 downto 0);
subtype SLV_64          is std_logic_vector(63 downto 0);
subtype SLV_128         is std_logic_vector(127 downto 0);
subtype SLV_256         is std_logic_vector(255 downto 0);


type T_TYPE             is array (0 to 15) of SLV_4;
type S_ARRAY_TYPE       is array (0 to 1) of SLV_32;

type PIPE_SUBKEY_TYPE   is array(FIRST_ROUND to LAST_ROUND-2) of SLV_64;

type KEY_ARRAY_TYPE     is array (0 to 39) of SLV_32;
type M_PAIR_TYPE        is array (0 to 1) of SLV_32;
type M_PAIR_PIPE_TYPE   is array (0 to 17) of M_PAIR_TYPE;


type PIPE_DATA_TYPE     is array(FIRST_ROUND to LAST_ROUND) of SLV_128;
type S_PIPE_TYPE        is array(FIRST_ROUND to LAST_ROUND-1) of S_ARRAY_TYPE;
type POSTADD_PIPE_TYPE  is array(FIRST_ROUND to LAST_ROUND-1) of SLV_64;


constant MDS_GF_FDBK_2 : SLV_8 := "10110100"; -- 180 (0xB4)
constant MDS_GF_FDBK_4 : SLV_8 := "01011010"; -- 90 (0x5A)
constant RS_GF_FDBK    : SLV_9 := "101001101"; -- 333 (0x14d


constant t0_q0 : T_TYPE := (X"8", X"1", X"7", X"D", X"6", X"F", X"3", X"2",
                            X"0", X"B", X"5", X"9", X"E", X"C", X"A", X"4");
constant t1_q0 : T_TYPE := (X"E", X"C", X"B", X"8", X"1", X"2", X"3", X"5", 
                            X"F", X"4", X"A", X"6", X"7", X"0", X"9", X"D");
constant t2_q0 : T_TYPE := (X"B", X"A", X"5", X"E", X"6", X"D", X"9", X"0",
                            X"C", X"8", X"F", X"3", X"2", X"4", X"7", X"1");
constant t3_q0 : T_TYPE := (X"D", X"7", X"F", X"4", X"1", X"2", X"6", X"E",
                            X"9", X"B", X"3", X"0", X"8", X"5", X"C", X"A");

constant t0_q1 : T_TYPE := (X"2", X"8", X"B", X"D", X"F", X"7", X"6", X"E", 
                            X"3", X"1", X"9", X"4", X"0", X"A", X"C", X"5");
constant t1_q1 : T_TYPE := (X"1", X"E", X"2", X"B", X"4", X"C", X"3", X"7",
                            X"6", X"D", X"A", X"5", X"F", X"9", X"0", X"8");
constant t2_q1 : T_TYPE := (X"4", X"C", X"7", X"5", X"1", X"6", X"9", X"A",
                            X"0", X"E", X"D", X"8", X"2", X"B", X"3", X"F");
constant t3_q1 : T_TYPE := (X"B", X"9", X"5", X"1", X"C", X"3", X"D", X"E",
                            X"6", X"4", X"7", X"F", X"2", X"0", X"8", X"A");


-- ==========================================================================
--                        Function Declarations                            --
-- ==========================================================================
function BYTESWAP32_FUNCT( datain  : in SLV_32)
                           return SLV_32;

function BYTESWAP128_FUNCT( datain : in SLV_128)
                            return SLV_128;

procedure BYTESWAP128( datain : in SLV_128;
                signal dataout : out SLV_128 );

function  PRE_ADD_FUNCT( datain      : SLV_128;
                         dec_key0    : SLV_32;
                         dec_key1    : SLV_32;
                         dec_key2    : SLV_32;
                         dec_key3    : SLV_32;
                         enc_dec_b   : std_logic;
                         enc_key0    : SLV_32;
                         enc_key1    : SLV_32;
                         enc_key2    : SLV_32;
                         enc_key3    : SLV_32 )
                         return SLV_128;
 
procedure PRE_ADD( datain      : in SLV_128;
                   enc_key0    : in SLV_32;
                   enc_key1    : in SLV_32;
                   enc_key2    : in SLV_32;
                   enc_key3    : in SLV_32;
                   dec_key0    : in SLV_32;
                   dec_key1    : in SLV_32;
                   dec_key2    : in SLV_32;
                   dec_key3    : in SLV_32;
                   enc_dec_b   : in std_logic;
            signal dataout     : out SLV_128);

function  POST_ADD_FUNCT( datain      : SLV_128;
                          dec_key0    : SLV_32;
                          dec_key1    : SLV_32;
                          dec_key2    : SLV_32;
                          dec_key3    : SLV_32;
                          enc_dec_b   : std_logic;
                          enc_key0    : SLV_32;
                          enc_key1    : SLV_32;
                          enc_key2    : SLV_32;
                          enc_key3    : SLV_32 )
                          return SLV_128;

procedure POST_ADD( datain      : in SLV_128;
                    enc_key0    : in SLV_32;
                    enc_key1    : in SLV_32;
                    enc_key2    : in SLV_32;
                    enc_key3    : in SLV_32;
                    dec_key0    : in SLV_32;
                    dec_key1    : in SLV_32;
                    dec_key2    : in SLV_32;
                    dec_key3    : in SLV_32;
                    enc_dec_b   : in std_logic;
             signal dataout     : out SLV_128);


function F_FUNCT( 
                  datain1   : SLV_32;
                  datain2   : SLV_32;
                  key1      : SLV_32;
                  key2      : SLV_32;
                  S         : S_ARRAY_TYPE)
                  return SLV_64;

procedure KS_ROUND( round        : in SLV_5;
                    m_value_even : in M_PAIR_TYPE;
                    m_value_odd  : in M_PAIR_TYPE;
             signal dataout      : out SLV_64);

function KS_ROUND_FUNCT( 
                         m_value_even : M_PAIR_TYPE;
                         m_value_odd  : M_PAIR_TYPE;
                         round        : SLV_5)
                         return SLV_64;

function PHT_FUNCT( datain1 : SLV_32;
                    datain2 : SLV_32)
                    return SLV_64;

function G_FUNCT( 
                  datain    : SLV_32;
                  S         : S_ARRAY_TYPE) 
                  return SLV_32;

function S_VALUE_FUNCT( datain  : SLV_128 )
                        return S_ARRAY_TYPE;

procedure S_VALUE_ROUND( datain   : in SLV_128;
                  signal S_VALUE  : out S_ARRAY_TYPE);


function M_PAIR_FUNCT( datain      : SLV_128;
                       M_SELECTOR  : std_logic)
                       return  M_PAIR_TYPE;

procedure MP_ROUND( datain      : in  SLV_128;
             signal M_EVEN      : out M_PAIR_TYPE;
             signal M_ODD       : out M_PAIR_TYPE);


function H_FUNCT( 
                  datain    : SLV_32;
                  M_PAIR1   : M_PAIR_TYPE)
                  return SLV_32;

function Q0_FUNCT( datain : SLV_8)
                   return SLV_8;

function Q1_FUNCT( datain : SLV_8)
                   return SLV_8;


function RS_MULTIPLY_FUNCT ( datain : SLV_64 )
                             return SLV_32;

function RS_REM_FUNCT ( datain : SLV_32 )
                             return SLV_32;

function MDS_MULTIPLY_FUNCT ( datain : SLV_32 )
                             return SLV_32;

function Mx_X_FUNCT (datain : SLV_8 )
               return SLV_8;

function Mx_Y_FUNCT (datain : SLV_8 )
               return SLV_8;



function TWOFISH_ROUND_FUNCT( 
                              datain         :  in SLV_128;
                              dec_key1       :  in SLV_32;
                              dec_key2       :  in SLV_32;
                              enc_dec_b      :  in std_logic;
                              enc_key1       :  in SLV_32;
                              enc_key2       :  in SLV_32;
                              round          :  in SLV_5;
                              S              :  in S_ARRAY_TYPE)
                              return SLV_128;

procedure TWOFISH_ROUND( datain         :  in SLV_128;
                         enc_key1       :  in SLV_32;
                         enc_key2       :  in SLV_32;
                         dec_key1       :  in SLV_32;
                         dec_key2       :  in SLV_32;
                         S              :  in S_ARRAY_TYPE;
                         round          :  in SLV_5;
                         enc_dec_b      :  in std_logic;
                  signal dataout        :  out SLV_128  ); 







-- ==========================================================================
--                Declarations for the Key Schedule section                --
-- ==========================================================================

constant HOLD               : integer := 0;


end twofish_pack;




package body twofish_pack is


-- ==========================================================================
--                       Definitions for general use                       --
-- ==========================================================================






-- ==========================================================================
--
--  function BYTESWAP32_FUNCT
--
--  This function swaps bytes in a 32-bit word where the input word is
--  a single 32-bit word. The byte swapping is performed as follows:
--
--     ABCD  -->  DCBA 
--
-- ==========================================================================

function BYTESWAP32_FUNCT( datain  : in SLV_32)
                           return SLV_32 is

-- pragma map_to_operator BYTESWAP32_FUNCT_dw_op
-- pragma return_port_name BYTESWAP32_FUNCT_out

begin

  return (datain(7 downto 0)   & datain(15 downto 8) &
          datain(23 downto 16) & datain(31 downto 24));

end BYTESWAP32_FUNCT;





-- ==========================================================================
--
--  function BYTESWAP128_FUNCT
--
--  This function swaps bytes in a 32-bit (4-byte) word where the input word
--  is a 128-bit (16-byte) word. The byte swapping is performed as follows:
--
--     ABCD  -->  DCBA 
--
-- ==========================================================================

function BYTESWAP128_FUNCT( datain : in SLV_128)
                            return SLV_128 is

-- pragma map_to_operator BYTESWAP128_FUNCT_dw_op
-- pragma return_port_name BYTESWAP128_FUNCT_out

variable dataout : SLV_128;

begin

  for i in 0 to 3 loop
    dataout(32*i+31 downto 32*i) := 
      BYTESWAP32_FUNCT(datain(32*i+31 downto 32*i));
  end loop;

  return dataout;

end BYTESWAP128_FUNCT;


-- ==========================================================================
--
--  procedure BYTESWAP128
--
--  Procedure call for BYTESWAP128_FUNCT 
--
-- ==========================================================================

procedure BYTESWAP128( datain : in SLV_128;
                signal dataout : out SLV_128 ) is

begin

  dataout <= BYTESWAP128_FUNCT(datain);

end BYTESWAP128;



-- ==========================================================================
--
--  function PRE_ADD_FUNCT
--
-- ==========================================================================

function  PRE_ADD_FUNCT( datain      : SLV_128;
                         dec_key0    : SLV_32;
                         dec_key1    : SLV_32;
                         dec_key2    : SLV_32;
                         dec_key3    : SLV_32;
                         enc_dec_b   : std_logic;
                         enc_key0    : SLV_32;
                         enc_key1    : SLV_32;
                         enc_key2    : SLV_32;
                         enc_key3    : SLV_32 )
                         return SLV_128 is
  
-- pragma map_to_operator PRE_ADD_FUNCT_dw_op
-- pragma return_port_name PRE_ADD_FUNCT_out

variable pre_add_out   : SLV_128;
variable key           : SLV_128;
variable data          : SLV_128;

begin
  
  if enc_dec_b = '1' then
    key := enc_key0 & enc_key1 & enc_key2 & enc_key3;
  else
    key := dec_key0 & dec_key1 & dec_key2 & dec_key3;
  end if;

  data := BYTESWAP128_FUNCT(datain);

  pre_add_out := data xor key;

  if enc_dec_b = '1' then
    pre_add_out := pre_add_out;
  else
    pre_add_out := pre_add_out(63 downto 0) & pre_add_out(127 downto 64);
  end if;

  return pre_add_out;

end PRE_ADD_FUNCT;


-- ==========================================================================
--
--  procedure PRE_ADD
--
-- ==========================================================================

procedure PRE_ADD(datain      : in SLV_128;
                  enc_key0    : in SLV_32;
                  enc_key1    : in SLV_32;
                  enc_key2    : in SLV_32;
                  enc_key3    : in SLV_32;
                  dec_key0    : in SLV_32;
                  dec_key1    : in SLV_32;
                  dec_key2    : in SLV_32;
                  dec_key3    : in SLV_32;
                  enc_dec_b   : in std_logic;
            signal dataout    : out SLV_128) is

begin


  dataout <= PRE_ADD_FUNCT(datain, dec_key0, dec_key1, dec_key2, dec_key3, 
                           enc_dec_b, enc_key0, enc_key1, enc_key2, enc_key3);

end PRE_ADD;



-- ==========================================================================
--
--  function POST_ADD_FUNCT
--
-- ==========================================================================

function  POST_ADD_FUNCT( datain      : SLV_128;
                          dec_key0    : SLV_32;
                          dec_key1    : SLV_32;
                          dec_key2    : SLV_32;
                          dec_key3    : SLV_32;
                          enc_dec_b   : std_logic;
                          enc_key0    : SLV_32;
                          enc_key1    : SLV_32;
                          enc_key2    : SLV_32;
                          enc_key3    : SLV_32 )
                          return SLV_128 is
  
-- pragma map_to_operator POST_ADD_FUNCT_dw_op
-- pragma return_port_name POST_ADD_FUNCT_out

variable post_add_out   : SLV_128;
variable key            : SLV_128;

begin
  
  if enc_dec_b = '1' then
    key := enc_key0 & enc_key1 & enc_key2 & enc_key3;
  else
    key := dec_key0 & dec_key1 & dec_key2 & dec_key3;
  end if;

  post_add_out := BYTESWAP128_FUNCT(datain xor key);

  return post_add_out;

end POST_ADD_FUNCT;


-- ==========================================================================
--
--  procedure POST_ADD
--
-- ==========================================================================

procedure POST_ADD(datain      : in SLV_128;
                  enc_key0    : in SLV_32;
                  enc_key1    : in SLV_32;
                  enc_key2    : in SLV_32;
                  enc_key3    : in SLV_32;
                  dec_key0    : in SLV_32;
                  dec_key1    : in SLV_32;
                  dec_key2    : in SLV_32;
                  dec_key3    : in SLV_32;
                  enc_dec_b   : in std_logic;
            signal dataout    : out SLV_128) is

begin


  dataout <= POST_ADD_FUNCT(datain, dec_key0, dec_key1, dec_key2, dec_key3,
                            enc_dec_b, enc_key0, enc_key1, enc_key2, enc_key3);

end POST_ADD;



-- ==========================================================================
--
--  function F_FUNCT
--
-- ==========================================================================
function F_FUNCT( 
                  datain1   : SLV_32;
                  datain2   : SLV_32;
                  key1      : SLV_32;
                  key2      : SLV_32;
                  S         : S_ARRAY_TYPE)
                  return SLV_64 is

-- pragma map_to_operator F_FUNCT_dw_op
-- pragma return_port_name F_FUNCT_out

variable temp1        : SLV_32;
variable temp2        : SLV_32;

variable G_FUNCT_OUT1 : SLV_32;
variable G_FUNCT_OUT2 : SLV_32;
variable PHT_OUT      : SLV_64;
variable PHT_OUT1     : SLV_32;

begin

  temp1 := BYTESWAP32_FUNCT(datain1);

  temp2 :=std_logic_vector(unsigned(datain2) ROL 8);
  temp2 := BYTESWAP32_FUNCT(temp2);

  G_FUNCT_OUT1 := G_FUNCT(temp1,S);
  G_FUNCT_OUT2 := G_FUNCT(temp2,S);

  PHT_OUT := PHT_FUNCT(G_FUNCT_OUT1,G_FUNCT_OUT2);

  PHT_OUT(31 downto 0) := std_logic_vector(unsigned(PHT_OUT(31 downto 0)) +
                          unsigned(key2));

  PHT_OUT(63 downto 32) := std_logic_vector(unsigned(PHT_OUT(63 downto 32)) +
                           unsigned(key1));

  return PHT_OUT;

end F_FUNCT;

-- ==========================================================================
--
-- procedure KS_ROUND
--
-- ==========================================================================

procedure KS_ROUND     ( round        : in SLV_5;
                         m_value_even : in M_PAIR_TYPE;
                         m_value_odd  : in M_PAIR_TYPE;
                  signal dataout      : out SLV_64) is

begin

  dataout <= KS_ROUND_FUNCT( m_value_even,m_value_odd,round );

end KS_ROUND;
                        



-- ==========================================================================
--
-- function KS_ROUND_FUNCT
--
-- ==========================================================================

function KS_ROUND_FUNCT( 
                         m_value_even : M_PAIR_TYPE;
                         m_value_odd  : M_PAIR_TYPE;
                         round        : SLV_5)
                         return SLV_64 is

-- pragma map_to_operator KS_ROUND_FUNCT_dw_op
-- pragma return_port_name KS_ROUND_FUNCT_out

variable temp0       : SLV_32;
variable temp1       : SLV_32;
variable word_input0 : SLV_32;
variable word_input1 : SLV_32;
variable PHT_OUT     : SLV_64;
variable index       : SLV_8;
variable index1      : SLV_8;

begin

  index := "000" & round;
  index := index(6 downto 0) & '0'; -- mult. index by 2
  index1 := std_logic_vector(TO_UNSIGNED(
            (TO_INTEGER(unsigned(index)) + 1 ), 8));

  for counter in 0 to 3 loop

    word_input0(((counter * 8) + 7) downto (counter * 8)) := index ;

    word_input1(((counter * 8) + 7) downto (counter * 8)) := index1;

  end loop;
 
-- below is the H_FUNCT function call 

  temp0 := H_FUNCT( word_input0,m_value_even);
  temp1 := H_FUNCT( word_input1,m_value_odd);
  temp1 := std_logic_vector(unsigned(temp1) ROL 8 );

  PHT_OUT := PHT_FUNCT(temp0,temp1);

  PHT_OUT(31 downto 0) := std_logic_vector(unsigned(
                          PHT_OUT(31 downto 0)) ROL 9);

-- there are no additions after PHT_FUNCT for the Key Scheduler 

  return PHT_OUT;

end KS_ROUND_FUNCT;


-- ==========================================================================
--
-- function PHT_FUNCT
--
-- ==========================================================================


function PHT_FUNCT( datain1 : SLV_32;
                    datain2 : SLV_32)
                    return SLV_64 is

-- pragma map_to_operator PHT_FUNCT_dw_op
-- pragma return_port_name PHT_FUNCT_out

variable temp1 :  SLV_32;
variable temp2 :  SLV_32;
variable PHT_FUNCT_OUT : SLV_64;

begin

temp1 := std_logic_vector(unsigned(datain1) + unsigned(datain2));
temp2 := std_logic_vector(unsigned(datain2) + unsigned(temp1));

PHT_FUNCT_OUT := temp1 & temp2;

return PHT_FUNCT_OUT;

end PHT_FUNCT;

-- ==========================================================================
--
--  function G_FUNCT
--
-- ==========================================================================


function G_FUNCT( 
                  datain       : SLV_32;
                  S            : S_ARRAY_TYPE)
                  return SLV_32 is

-- pragma map_to_operator G_FUNCT_dw_op
-- pragma return_port_name G_FUNCT_out

variable temp1        : SLV_8;
variable temp2        : SLV_8;
variable temp3        : SLV_8;
variable temp4        : SLV_8;
variable temp_concat  :  SLV_32;
variable G_MDS_OUT    : SLV_32;


begin

  temp1 := datain(31 downto 24);
  temp2 := datain(23 downto 16);
  temp3 := datain(15 downto 8);
  temp4 := datain(7 downto 0);

-- 4 Q FUNCTION CALLS -- cv >= 128
  
  temp1 := Q0_FUNCT(temp1);
  temp2 := Q1_FUNCT(temp2);
  temp3 := Q0_FUNCT(temp3);
  temp4 := Q1_FUNCT(temp4);

  temp_concat := (temp1 & temp2 & temp3 & temp4) xor S(0);

  temp1 := temp_concat(31 downto 24);
  temp2 := temp_concat(23 downto 16);
  temp3 := temp_concat(15 downto 8);
  temp4 := temp_concat(7 downto 0);

  temp1 := Q0_FUNCT(temp1);
  temp2 := Q0_FUNCT(temp2);
  temp3 := Q1_FUNCT(temp3);
  temp4 := Q1_FUNCT(temp4);

  temp_concat := (temp1 & temp2 & temp3 & temp4) xor S(1);

  temp1 := temp_concat(31 downto 24);
  temp2 := temp_concat(23 downto 16);
  temp3 := temp_concat(15 downto 8);
  temp4 := temp_concat(7 downto 0);

  temp1 := Q1_FUNCT(temp1);
  temp2 := Q0_FUNCT(temp2);
  temp3 := Q1_FUNCT(temp3);
  temp4 := Q0_FUNCT(temp4);

  temp_concat := (temp1 & temp2 & temp3 & temp4);

  G_MDS_OUT := MDS_MULTIPLY_FUNCT(temp_concat);

  G_MDS_OUT := BYTESWAP32_FUNCT(G_MDS_OUT);

  return G_MDS_OUT;

end G_FUNCT;




-- ==========================================================================
--
-- function S_VALUE_FUNCT
--
-- ==========================================================================

function S_VALUE_FUNCT( datain  : SLV_128 )
                        return S_ARRAY_TYPE is

-- pragma map_to_operator S_VALUE_FUNCT_dw_op
-- pragma return_port_name S_VALUE_FUNCT_out

variable S_out        : S_ARRAY_TYPE;

begin

  for i in 2 to 3 loop

    S_out(i-2) := RS_MULTIPLY_FUNCT( 
                  BYTESWAP32_FUNCT( datain((3-i)*64+31 downto (3-i)*64)) &
                  BYTESWAP32_FUNCT( datain((3-i)*64+63 downto (3-i)*64+32)) );
    S_out(i-2) := BYTESWAP32_FUNCT(S_out(i-2));

  end loop;  -- i (S element calculation complete; repeat)

  return S_out;

end S_VALUE_FUNCT;


-- ==========================================================================
--
-- procedure S_VALUE_ROUND
--
-- ==========================================================================

procedure S_VALUE_ROUND( datain   : in SLV_128;
                  signal S_VALUE  : out S_ARRAY_TYPE) is

begin

S_VALUE <= S_VALUE_FUNCT(datain);

end S_VALUE_ROUND;

-- ==========================================================================
--
-- M_PAIR_FUNCT Function
--
-- ==========================================================================

function M_PAIR_FUNCT ( datain      : SLV_128;
                        M_SELECTOR  : std_logic) --1 is odd, 0 is even
                        return M_PAIR_TYPE is

-- pragma map_to_operator M_PAIR_FUNCT_dw_op
-- pragma return_port_name M_PAIR_FUNCT_out

variable temp : M_PAIR_TYPE;

begin

  if M_SELECTOR = '0' then
    temp(0) := (datain(127 downto 96));
    temp(1) := (datain(63 downto 32));


 else
    temp(0) := (datain(95 downto 64));
    temp(1) := (datain(31 downto 0));
 
  end if;

return temp;

end M_PAIR_FUNCT;

-- ==========================================================================
--
-- procedure MP_ROUND
--
-- ==========================================================================
procedure MP_ROUND( datain      : in  SLV_128;
             signal M_EVEN      : out M_PAIR_TYPE;
             signal M_ODD       : out M_PAIR_TYPE) is

begin

 M_EVEN <= M_PAIR_FUNCT(datain,'0');
 M_ODD  <= M_PAIR_FUNCT(datain,'1');

end MP_ROUND;




-- ==========================================================================
--
-- function H_FUNCT
--
-- ==========================================================================

function H_FUNCT( 
                  datain    : SLV_32;
                  M_PAIR1   : M_PAIR_TYPE)
                  return SLV_32 is

-- pragma map_to_operator H_FUNCT_dw_op
-- pragma return_port_name H_FUNCT_out

variable temp1       : SLV_8;
variable temp2       : SLV_8;
variable temp3       : SLV_8;
variable temp4       : SLV_8;
variable temp_concat : SLV_32;
variable H_MDS_OUT   : SLV_32;

begin

  temp1 := datain(31 downto 24);
  temp2 := datain(23 downto 16);
  temp3 := datain(15 downto 8);
  temp4 := datain(7 downto 0);


-- 4 Q FUNCTION CALLS -- cv >= 128

  temp1 := Q0_FUNCT(temp1);
  temp2 := Q1_FUNCT(temp2);
  temp3 := Q0_FUNCT(temp3);
  temp4 := Q1_FUNCT(temp4);

  temp_concat := (temp1 & temp2 & temp3 & temp4) xor M_PAIR1(1);

  temp1 := temp_concat(31 downto 24);
  temp2 := temp_concat(23 downto 16);
  temp3 := temp_concat(15 downto 8);
  temp4 := temp_concat(7 downto 0);

  temp1 := Q0_FUNCT(temp1);
  temp2 := Q0_FUNCT(temp2);
  temp3 := Q1_FUNCT(temp3);
  temp4 := Q1_FUNCT(temp4);

  temp_concat := (temp1 & temp2 & temp3 & temp4) xor M_PAIR1(0);

  temp1 := temp_concat(31 downto 24);
  temp2 := temp_concat(23 downto 16);
  temp3 := temp_concat(15 downto 8);
  temp4 := temp_concat(7 downto 0);

  temp1 := Q1_FUNCT(temp1);
  temp2 := Q0_FUNCT(temp2);
  temp3 := Q1_FUNCT(temp3);
  temp4 := Q0_FUNCT(temp4);

  temp_concat := (temp1 & temp2 & temp3 & temp4);

  H_MDS_OUT := MDS_MULTIPLY_FUNCT(temp_concat);

  H_MDS_OUT := BYTESWAP32_FUNCT(H_MDS_OUT);

  return H_MDS_OUT;

end H_FUNCT;

-- ==========================================================================
--
-- function Q0_FUNCT
--
-- ==========================================================================


function Q0_FUNCT( datain : SLV_8)
                   return SLV_8 is

-- pragma map_to_operator Q0_FUNCT_dw_op
-- pragma return_port_name Q0_FUNCT_out

variable a0,a1,a2,a3,a4 : SLV_4;
variable b0,b1,b2,b3,b4 : SLV_4;

begin

a0 := datain(7 downto 4);
b0 := datain(3 downto 0);
a1 := a0 xor b0;
b1 := a0 xor (std_logic_vector(unsigned(b0) ror 1)) xor (a0(0)& "000");
a2 := t0_q0(TO_INTEGER(unsigned(a1)));
b2 := t1_q0(TO_INTEGER(unsigned(b1)));
a3 := a2 xor b2;
b3 := a2 xor (std_logic_vector(unsigned(b2) ror 1)) xor (a2(0)& "000");
a4 := t2_q0(TO_INTEGER(unsigned(a3)));
b4 := t3_q0(TO_INTEGER(unsigned(b3)));

return (b4 & a4);

end Q0_FUNCT;


-- ==========================================================================
--
-- function Q1_FUNCT
--
-- ==========================================================================


function Q1_FUNCT( datain : SLV_8)
                   return SLV_8 is

-- pragma map_to_operator Q1_FUNCT_dw_op
-- pragma return_port_name Q1_FUNCT_out

variable a0,a1,a2,a3,a4 : SLV_4;
variable b0,b1,b2,b3,b4 : SLV_4;

begin

a0 := datain(7 downto 4);
b0 := datain(3 downto 0);
a1 := a0 xor b0;
b1 := a0 xor (std_logic_vector(unsigned(b0) ror 1)) xor (a0(0)& "000");
a2 := t0_q1(TO_INTEGER(unsigned(a1)));
b2 := t1_q1(TO_INTEGER(unsigned(b1)));
a3 := a2 xor b2;
b3 := a2 xor (std_logic_vector(unsigned(b2) ror 1)) xor (a2(0)& "000");
a4 := t2_q1(TO_INTEGER(unsigned(a3)));
b4 := t3_q1(TO_INTEGER(unsigned(b3)));

return (b4 & a4);

end Q1_FUNCT;



-- ==========================================================================
--
-- RS_MULTPLY_FUNCT
--
--
-- ==========================================================================

function RS_MULTIPLY_FUNCT ( datain : SLV_64 )
                             return SLV_32 is

-- pragma map_to_operator RS_MULTPLY_FUNCT_dw_op
-- pragma return_port_name RS_MULTPLY_FUNCT_out

variable k1, k0, r : SLV_32;

begin

  k1 := datain(63 downto 32);
  k0 := datain(31 downto 0);

  r := k1;
  for i in 0 to 3 loop
    r := RS_REM_FUNCT(r);
  end loop;

  r := r xor k0;
  for i in 0 to 3 loop
    r := RS_REM_FUNCT(r);
  end loop;

  return (r);

end RS_MULTIPLY_FUNCT;



-- ==========================================================================
--
-- RS_REM_FUNCT
--
--
-- ==========================================================================

function RS_REM_FUNCT ( datain : SLV_32 )
                        return SLV_32 is

-- pragma map_to_operator RS_REM_FUNCT_dw_op
-- pragma return_port_name RS_REM_FUNCT_out

variable result  : SLV_32;
variable b       : SLV_8;
variable g2      : SLV_9;
variable g3      : SLV_9;
variable g2_temp : SLV_9;
variable g3_temp : SLV_9;

begin

  b := datain(31 downto 24);

  if b(7) = '1' then
    g2_temp := RS_GF_FDBK;
  else
    g2_temp := (others => '0');
  end if;

  g2 := (b(7 downto 0) & '0') xor g2_temp;  -- and 0xff??

  if b(0) = '1' then
    g3_temp := '0' & RS_GF_FDBK(8 downto 1);
  else
    g3_temp := (others => '0');
  end if;

  g3 := ("00" & b(7 downto 1)) xor g3_temp xor g2;

  result :=  (datain(23 downto 16) xor g3(7 downto 0) xor ("0000000" & g2(8))) &
            (datain(15 downto 8) xor g2(7 downto 0) xor ("0000000" & g3(8))) &
            (datain(7 downto 0) xor g3(7 downto 0)) &
            b;

  return ( result );

end RS_REM_FUNCT;


-- ==========================================================================
--
-- MDS_MULTPLY_FUNCT
--
-- ==========================================================================

function MDS_MULTIPLY_FUNCT ( datain : SLV_32 )
                              return SLV_32 is

-- pragma map_to_operator MDS_MULTPLY_FUNCT_dw_op
-- pragma return_port_name MDS_MULTPLY_FUNCT_out

variable result : SLV_32;
variable b_0    : SLV_8;
variable b_1    : SLV_8;
variable b_2    : SLV_8;
variable b_3    : SLV_8;

begin

  b_0 := datain(31 downto 24);   -- or is it (31 downto 24) here?
  b_1 := datain(23 downto 16);
  b_2 := datain(15 downto 8);
  b_3 := datain(7 downto 0);

  result(31 downto 24)   := b_0 xor Mx_Y_FUNCT(b_1) xor Mx_X_FUNCT(b_2) xor Mx_X_FUNCT(b_3);
  result(23 downto 16)  := Mx_X_FUNCT(b_0) xor Mx_Y_FUNCT(b_1) xor Mx_Y_FUNCT(b_2) xor b_3;
  result(15 downto 8) := Mx_Y_FUNCT(b_0) xor Mx_X_FUNCT(b_1) xor b_2 xor Mx_Y_FUNCT(b_3);
  result(7 downto 0) := Mx_Y_FUNCT(b_0) xor b_1 xor Mx_Y_FUNCT(b_2) xor Mx_X_FUNCT(b_3);

  return ( result );

end MDS_MULTIPLY_FUNCT;



-- ==========================================================================
--
-- Mx_X_FUNCT
--
-- ==========================================================================

function Mx_X_FUNCT (datain : SLV_8 )
                     return SLV_8 is

-- pragma map_to_operator Mx_X_FUNCT_dw_op
-- pragma return_port_name Mx_X_FUNCT_out

variable result  : SLV_8;
variable term1   : SLV_8;
variable term2   : SLV_8;
variable LFSR2   : SLV_8;


begin

  if datain(1) = '1' then
    term1 := MDS_GF_FDBK_2;
  else
    term1 := (others => '0');
  end if;

  if datain(0) = '1' then
    term2 := MDS_GF_FDBK_4;
  else
    term2 := (others => '0');
  end if;

  LFSR2 := ("00" & datain(7 downto 2)) xor term1 xor term2;

  result := datain xor LFSR2;

  return (result);

end Mx_X_FUNCT;



-- ==========================================================================
--
-- Mx_Y_FUNCT
--
-- ==========================================================================

function Mx_Y_FUNCT (datain : SLV_8 )
                     return SLV_8 is

-- pragma map_to_operator Mx_Y_FUNCT_dw_op
-- pragma return_port_name Mx_Y_FUNCT_out

variable result  : SLV_8;
variable term1   : SLV_8;
variable term2   : SLV_8;
variable term3   : SLV_8;
variable LFSR1   : SLV_8;
variable LFSR2   : SLV_8;

begin

  if datain(1) = '1' then
    term1 := MDS_GF_FDBK_2;
  else
    term1 := (others => '0');
  end if;

  if datain(0) = '1' then
    term2 := MDS_GF_FDBK_4;
    term3 := MDS_GF_FDBK_2;
  else
    term2 := (others => '0');
    term3 := (others => '0');
  end if;

  LFSR1 := ('0' & datain(7 downto 1)) xor term3;
  LFSR2 := ("00" & datain(7 downto 2)) xor term1 xor term2;

  result := datain xor LFSR1 xor LFSR2;

  return (result);

end Mx_Y_FUNCT;




-- ==========================================================================
--
--  function TWOFISH_ROUND_FUNCT
--
-- ==========================================================================

function TWOFISH_ROUND_FUNCT( 
                              datain         :  in SLV_128;
                              dec_key1       :  in SLV_32;
                              dec_key2       :  in SLV_32;
                              enc_dec_b      :  in std_logic;
                              enc_key1       :  in SLV_32;
                              enc_key2       :  in SLV_32;
                              round          :  in SLV_5;
                              S              :  in S_ARRAY_TYPE)
                              return SLV_128 is

-- pragma map_to_operator TWOFISH_ROUND_FUNCT_dw_op
-- pragma return_port_name TWOFISH_ROUND_FUNCT_out

                              
  variable A,B,C,D     : SLV_32;
  variable F0          : SLV_32;
  variable F1          : SLV_32;
  variable F_FUNCT_OUT : SLV_64;
  variable ROUND_OUT   : SLV_128;
  variable F_key1      : SLV_32;
  variable F_key2      : SLV_32;

begin

  if enc_dec_b = '1' then

    A := datain(127 downto 96);
    B := datain(95 downto 64);
    C := datain(63 downto 32);
    D := datain(31 downto 0);

    F_key1 := enc_key1;
    F_key2 := enc_key2;

  else

    if TO_INTEGER(unsigned(round)) = FIRST_ROUND then
      A := datain(127 downto 96);
      B := datain(95 downto 64);
      C := datain(63 downto 32);
      D := datain(31 downto 0);
    else
      A := datain(63 downto 32);
      B := datain(31 downto 0);
      C := datain(127 downto 96);
      D := datain(95 downto 64);
    end if;

    F_key1 := dec_key1;
    F_key2 := dec_key2;

  end if;

  F_FUNCT_OUT := F_FUNCT(A,B,F_key1,F_key2,S); --calls F32(A,B)

  F0 := F_FUNCT_OUT(63 downto 32);  -- F output derived from A
  F1 := F_FUNCT_OUT(31 downto 0);   -- F output derived from B


  if enc_dec_b = '1' then

    C := F0 xor C;
    C := std_logic_vector(unsigned(C) ROR 1);

    D := F1 xor std_logic_vector(unsigned(D) ROL 1);

    if TO_INTEGER(unsigned(round)) = LAST_ROUND - 1 then
      ROUND_OUT := (A & B & C & D);
    else
      ROUND_OUT := (C & D & A & B);
    end if;

  else

    C := F0 xor std_logic_vector(unsigned(C) ROL 1);

    D := F1 xor D;
    D := std_logic_vector(unsigned(D) ROR 1);

    ROUND_OUT := (A & B & C & D);

  end if;

  return (ROUND_OUT); 

end TWOFISH_ROUND_FUNCT;



-- ==========================================================================
--
--  procedure TWOFISH_ROUND
--
--  Performs one round of the TWOFISH block cipher. Encryption or decryption
--  is performed based on the 'encrypt' signal.
--  Result of the round is returned on dataout.
--
-- ==========================================================================

procedure TWOFISH_ROUND( datain         :  in SLV_128;
                         enc_key1       :  in SLV_32;
                         enc_key2       :  in SLV_32;
                         dec_key1       :  in SLV_32;
                         dec_key2       :  in SLV_32;
                         S              :  in S_ARRAY_TYPE;
                         round          :  in SLV_5;
                         enc_dec_b      :  in std_logic;
                  signal dataout        :  out SLV_128  )  is    
              
begin

  dataout <= TWOFISH_ROUND_FUNCT( datain, dec_key1, dec_key2,
                                  enc_dec_b, enc_key1, enc_key2, round, S );

end TWOFISH_ROUND;
-- ==========================================================================
-- ==========================================================================



end twofish_pack;
