-- *************************************************************************
-- DISCLAIMER. THIS SOFTWARE WAS WRITTEN BY EMPLOYEES OF THE U.S.
-- GOVERNMENT AS A PART OF THEIR OFFICIAL DUTIES AND, THEREFORE, IS NOT
-- PROTECTED BY COPYRIGHT. HOWEVER, THIS SOFTWARE CODIFIES THE FINALIST
-- CANDIDATE ALGORITHMS (i.e., MARS, RC6tm, RIJNDAEL, SERPENT, AND
-- TWOFISH) IN THE ADVANCED ENCRYPTION STANDARD (AES) DEVELOPMENT EFFORT
-- SPONSORED BY THE NATIONAL INSTITUTE OF STANDARDS AND TECHNOLOGY (NIST)
-- AND MAY BE PROTECTED BY ONE OR MORE FORMS OF INTELLECTUAL PROPERTY. THE
-- U.S. GOVERNMENT MAKES NO WARRANTY, EITHER EXPRESSED OR IMPLIED,
-- INCLUDING BUT NO LIMITED TO ANY IMPLIED WARRANTIES OF MERCHANTABILITY
-- OR FITNESS FOR A PARTICULAR PURPOSE, REGARDING THIS SOFTWARE. THE U.S.
-- GOVERNMENT FURTHER MAKES NO WARRANTY THAT THIS SOFTWARE WILL NOT
-- INFRINGE ANY OTHER UNITED STATES OR FOREIGN PATENT OR OTHER
-- INTELLECTUAL PROPERTY RIGHT. IN NO EVENT SHALL THE U.S. GOVERNMENT BE
-- LIABLE TO ANYONE FOR COMPENSATORY, PUNITIVE, EXEMPLARY, SPECIAL,
-- COLLATERAL, INCIDENTAL, CONSEQUENTIAL, OR ANY OTHER TYPE OF DAMAGES IN
-- CONNECTION WITH OR ARISING OUT OF COPY OR USE OF THIS SOFTWARE.
-- *************************************************************************
--
-- File Name : alg_iterative.vhdl
-- Author    : NSA
-- Date      : 07 October 99
-- Project   : AES Candidate Evaluation -- RC6
-- Purpose   : This block implements the RC6 algorithm for an iterative case
-- Notes     :
-- ===========================================================================

library ieee;
use ieee.std_logic_1164.all;
use WORK.rc6_pack.all;

-- ===========================================================================
-- =========================== Interface Description =========================
-- ===========================================================================

entity ALG_ITERATIVE is

  port (clock       :  in STD_LOGIC;
        reset       :  in STD_LOGIC;

        ALG_DATAIN  :  in SLV_128;
        ENC_KEY1    :  in SLV_32;
        ENC_KEY2    :  in SLV_32;
        DEC_KEY1    :  in SLV_32;
        DEC_KEY2    :  in SLV_32;     
        ALG_START   :  in STD_LOGIC;
        ALG_ENC     :  in STD_LOGIC;

        ALG_DATAOUT :  out SLV_128;
        ALG_DONE    :  out STD_LOGIC
  );

end ALG_ITERATIVE;

architecture ALG_ITERATIVE_RTL of ALG_ITERATIVE is


-- ===========================================================================
-- =========================== Constant Definition ===========================
-- ===========================================================================

constant HOLD : integer := 0;         -- Hold state for controller

-- ===========================================================================
-- =========================== Signal Definition =============================
-- ===========================================================================

signal ROUND            : ROUND_TYPE;  -- round number
signal BI_REG           : SLV_128;     -- data out from round i
signal ALG_DATA_INT     : SLV_128;     -- initial permutation of alg_datain_swap
signal ALG_DATAIN_SWAP  : SLV_128;     -- byte reordering of alg_datain
signal ALG_DATAOUT_SWAP : SLV_128;     -- byte reordering of alg_dataout


begin

-- ===========================================================================
-- =========================== Data Movement =================================
-- ===========================================================================


DATA_FLOW: process( clock, reset )

begin

if reset = '1' then                        -- check for reset condition

   BI_REG   <= ( others => '0' );          -- clear round key outputs
   ALG_DONE <= '0';                        -- clear done signal
   ALG_DATA_INT <= ( others => '0' );      -- clear interconnection

elsif clock'event and clock = '1' then     -- rising edge clock

   if ALG_START = '1' then

      PRE_WHITEN( ALG_DATAIN_SWAP, ENC_KEY1, ENC_KEY2, DEC_KEY1, DEC_KEY2,
                  ALG_ENC, ALG_DATA_INT );
   end if;

   case ROUND is

    when 1 =>
      
      RC6_ROUND( ALG_DATA_INT,          -- input data 
            	 ENC_KEY1, 
             	 ENC_KEY2,
              	 DEC_KEY1,
               	 DEC_KEY2,
                 ALG_ENC,                             
                 BI_REG );              -- output

      ALG_DONE    <= '0';

    when LAST_ROUND =>

      POST_WHITEN( BI_REG, ENC_KEY1, ENC_KEY2, DEC_KEY1, DEC_KEY2,
                   ALG_ENC, BI_REG );

      ALG_DONE <= '1';
  
    when others =>

       RC6_ROUND( BI_REG,         -- input data 
            	  ENC_KEY1, 
             	  ENC_KEY2,
              	  DEC_KEY1,
               	  DEC_KEY2,
                  ALG_ENC,                             
                  BI_REG );       -- output

      ALG_DONE    <= '0';
    
    end case;        

end if; 

end process; -- DATA_FLOW


-- ===========================================================================
-- =========================== Concurrent Statements =========================
-- ===========================================================================

ALG_DATAOUT <= ALG_DATAOUT_SWAP when ROUND = LAST_ROUND+1-- output data when valid
                          else (others => '0');        -- else output all '0'

SWAPIN : BYTESWAP128( ALG_DATAIN,
                      ALG_DATAIN_SWAP );

SWAPOUT: BYTESWAP128( BI_REG,
                      ALG_DATAOUT_SWAP );



-- ===========================================================================
-- =========================== State Machine / Controller ====================
-- ===========================================================================

STATE_FLOW: process( clock, reset )

variable active : std_logic;                -- Indicates whether algorithm is
                                            -- active (1) or in hold (0)

begin

if reset = '1' then                         -- Active high reset (asynch)

   ROUND  <= HOLD;                          -- put controller in hold state
   active := '0';                           -- stop process until ALG_START=1

elsif clock'event and clock = '1' then      -- check for rising edge clock

   if  ( ALG_START = '1' or active = '1' )  -- check for inactive  
   and ( ROUND /= LAST_ROUND+1 ) then       -- and completion of algorithm

      active := '1';                        -- enable controller
      ROUND  <= ROUND + 1;                  -- active, so increment counter

   else

      active := '0';                        -- disable controller
      ROUND  <= HOLD;                       -- reset counter
             
   end if;

end if; -- reset

end process;

end ALG_ITERATIVE_RTL;
