-- *************************************************************************
-- DISCLAIMER. THIS SOFTWARE WAS WRITTEN BY EMPLOYEES OF THE U.S.
-- GOVERNMENT AS A PART OF THEIR OFFICIAL DUTIES AND, THEREFORE, IS NOT
-- PROTECTED BY COPYRIGHT. HOWEVER, THIS SOFTWARE CODIFIES THE FINALIST
-- CANDIDATE ALGORITHMS (i.e., MARS, RC6tm, RIJNDAEL, SERPENT, AND
-- TWOFISH) IN THE ADVANCED ENCRYPTION STANDARD (AES) DEVELOPMENT EFFORT
-- SPONSORED BY THE NATIONAL INSTITUTE OF STANDARDS AND TECHNOLOGY (NIST)
-- AND MAY BE PROTECTED BY ONE OR MORE FORMS OF INTELLECTUAL PROPERTY. THE
-- U.S. GOVERNMENT MAKES NO WARRANTY, EITHER EXPRESSED OR IMPLIED,
-- INCLUDING BUT NO LIMITED TO ANY IMPLIED WARRANTIES OF MERCHANTABILITY
-- OR FITNESS FOR A PARTICULAR PURPOSE, REGARDING THIS SOFTWARE. THE U.S.
-- GOVERNMENT FURTHER MAKES NO WARRANTY THAT THIS SOFTWARE WILL NOT
-- INFRINGE ANY OTHER UNITED STATES OR FOREIGN PATENT OR OTHER
-- INTELLECTUAL PROPERTY RIGHT. IN NO EVENT SHALL THE U.S. GOVERNMENT BE
-- LIABLE TO ANYONE FOR COMPENSATORY, PUNITIVE, EXEMPLARY, SPECIAL,
-- COLLATERAL, INCIDENTAL, CONSEQUENTIAL, OR ANY OTHER TYPE OF DAMAGES IN
-- CONNECTION WITH OR ARISING OUT OF COPY OR USE OF THIS SOFTWARE.
-- *************************************************************************
--
-- File Name : key_sched_iterative.vhdl
-- Author    : NSA
-- Date      : 04 Oct 1999
-- Project   : AES Candidate Evaluation --RC6 
-- Purpose   : build key schedule for iterative implementation
-- Notes     :
-- ===========================================================================

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use WORK.rc6_pack.all;

-- ===========================================================================
-- =========================== Interface Description =========================
-- ===========================================================================

entity KEY_SCHEDULE_ITERATIVE is

  port (clock            :  in std_logic;    -- clock signal
        reset            :  in std_logic;    -- active high reset (asynch)

        KS_LOADCV        :  in std_logic;    -- load a new cryptovariable
        KS_START         :  in std_logic;    -- start a new expansion sequence
        KS_CV            :  in L_ARRAY_TYPE; -- cryptovariable input bus

        KS_READY         :  out std_logic;   -- '1'=cv expanded, '0'=inprocess
        KS_ROUNDKEY1_ENC :  out SLV_32;      -- first encrypt round key (Si)
        KS_ROUNDKEY2_ENC :  out SLV_32;      -- second encrypt round key (Si)
        KS_ROUNDKEY1_DEC :  out SLV_32;      -- first decrypt round key (Si)
        KS_ROUNDKEY2_DEC :  out SLV_32       -- second decrypt round key (Si)

  );

end KEY_SCHEDULE_ITERATIVE;

architecture KEY_SCHEDULE_ITERATIVE_RTL of KEY_SCHEDULE_ITERATIVE is



component RUNUP_ROUNDKEYS

  port (clock         :  in std_logic;     -- clock signal
        reset         :  in std_logic;     -- active high reset (asynchronous)
    
        KS_LOADCV     :  in std_logic;     -- load a new cryptovariable
        KS_CV         :  in L_ARRAY_TYPE;  -- cryptovariable input bus

        KS_READY      :  out std_logic;    -- '1'=cv expanded, '0'=inprocess
        KS_ROUNDKEYS  :  out S_ARRAY_TYPE  -- output key bus

  );
end component;



-- ===========================================================================
-- =========================== Type Definition ===============================
-- ===========================================================================

type EXPAND_STATE_TYPE is ( HOLD, CV_EXPAND );

-- ===========================================================================
-- =========================== Signal Definition =============================
-- ===========================================================================

signal SUBKEYS        : S_ARRAY_TYPE;       -- current w register expansion
signal CV_EXPAND_STEP : ROUND_TYPE;         -- counter for cv expansion (same)
signal EXPAND_STATE   : EXPAND_STATE_TYPE;
signal KS_CV_SWAP      : L_ARRAY_TYPE;

begin

-- ===========================================================================
-- =========================== Signal Mapping ================================
-- ===========================================================================


KS_ROUNDKEY1_ENC <= 
         SUBKEYS(CV_EXPAND_STEP*2-2) when EXPAND_STATE = CV_EXPAND else
         ( others => '0' );

KS_ROUNDKEY2_ENC <= 
         SUBKEYS(CV_EXPAND_STEP*2+1-2) when EXPAND_STATE = CV_EXPAND else
         ( others => '0' );

KS_ROUNDKEY1_DEC <= 
         SUBKEYS(43-(CV_EXPAND_STEP*2-2)) when EXPAND_STATE = CV_EXPAND else
         ( others => '0' );

KS_ROUNDKEY2_DEC <= 
         SUBKEYS(43-(CV_EXPAND_STEP*2+1-2)) when EXPAND_STATE = CV_EXPAND else
         ( others => '0' );




-- ===========================================================================
-- =========================== State Machine / Controller ====================
-- ===========================================================================


EXPAND_FLOW: process( clock, reset )

begin

   if reset = '1' then                     -- Active high reset (asynchronous)

      CV_EXPAND_STEP <= 0;                 -- put controller in hold state
      EXPAND_STATE   <= HOLD;              -- no activity for expansion

   elsif clock'event and clock = '1' then

      case EXPAND_STATE is

         when HOLD =>

            if KS_START = '1' then

               EXPAND_STATE   <= CV_EXPAND;
               CV_EXPAND_STEP <= 1;
      
            else

               EXPAND_STATE   <= HOLD;
               CV_EXPAND_STEP <= 0;
       
            end if;

         when CV_EXPAND =>

            if ( CV_EXPAND_STEP = LAST_ROUND+1 ) then  -- check for complete

               EXPAND_STATE   <= HOLD;   -- set controller back to hold state
               CV_EXPAND_STEP <= 0;

            else

               EXPAND_STATE   <= EXPAND_STATE;
               CV_EXPAND_STEP <= CV_EXPAND_STEP + 1;

            end if;

      end case;
                                             
   end if;  -- reset = '1'

end process; -- EXPAND_FLOW

G0: for CV_INDEX in 0 to 3 generate

  SWAP: BYTESWAP32( KS_CV(CV_INDEX),
                    KS_CV_SWAP(CV_INDEX) );

end generate;

GEN_NEXT_KEYS: RUNUP_ROUNDKEYS
                        port map( clock,
                                  reset,
                                  KS_LOADCV,
                                  KS_CV_SWAP,
                                  KS_READY,
                                  SUBKEYS );

end KEY_SCHEDULE_ITERATIVE_RTL;
