-- *************************************************************************
-- DISCLAIMER. THIS SOFTWARE WAS WRITTEN BY EMPLOYEES OF THE U.S.
-- GOVERNMENT AS A PART OF THEIR OFFICIAL DUTIES AND, THEREFORE, IS NOT
-- PROTECTED BY COPYRIGHT. HOWEVER, THIS SOFTWARE CODIFIES THE FINALIST
-- CANDIDATE ALGORITHMS (i.e., MARS, RC6tm, RIJNDAEL, SERPENT, AND
-- TWOFISH) IN THE ADVANCED ENCRYPTION STANDARD (AES) DEVELOPMENT EFFORT
-- SPONSORED BY THE NATIONAL INSTITUTE OF STANDARDS AND TECHNOLOGY (NIST)
-- AND MAY BE PROTECTED BY ONE OR MORE FORMS OF INTELLECTUAL PROPERTY. THE
-- U.S. GOVERNMENT MAKES NO WARRANTY, EITHER EXPRESSED OR IMPLIED,
-- INCLUDING BUT NO LIMITED TO ANY IMPLIED WARRANTIES OF MERCHANTABILITY
-- OR FITNESS FOR A PARTICULAR PURPOSE, REGARDING THIS SOFTWARE. THE U.S.
-- GOVERNMENT FURTHER MAKES NO WARRANTY THAT THIS SOFTWARE WILL NOT
-- INFRINGE ANY OTHER UNITED STATES OR FOREIGN PATENT OR OTHER
-- INTELLECTUAL PROPERTY RIGHT. IN NO EVENT SHALL THE U.S. GOVERNMENT BE
-- LIABLE TO ANYONE FOR COMPENSATORY, PUNITIVE, EXEMPLARY, SPECIAL,
-- COLLATERAL, INCIDENTAL, CONSEQUENTIAL, OR ANY OTHER TYPE OF DAMAGES IN
-- CONNECTION WITH OR ARISING OUT OF COPY OR USE OF THIS SOFTWARE.
-- *************************************************************************
--
-- File Name : runup_roundkeys.vhdl
-- Author    : NSA
-- Date      : 04 October 1999
-- Project   : AES Candidate Evaluation --RC6
-- Purpose   : This block performs all iterations of subkey generation and  
--             stores the created array of subkeys for the iterative case
-- Notes     : Used for iterative case only.
-- ===========================================================================

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use WORK.rc6_pack.all;

-- ===========================================================================
-- =========================== Interface Description =========================
-- ===========================================================================

entity RUNUP_ROUNDKEYS is

  port (clock         :  in std_logic;    -- clock signal
        reset         :  in std_logic;    -- active high reset (asynchronous)
    
        KS_LOADCV     :  in std_logic;    -- load a new cryptovariable
        KS_CV         :  in L_ARRAY_TYPE; -- cryptovariable input bus

        KS_READY      :  out std_logic;   -- '1'=cv expanded, '0'=inprocess
        KS_ROUNDKEYS  :  out S_ARRAY_TYPE -- output key bus

  );

end RUNUP_ROUNDKEYS;

architecture RUNUP_ROUNDKEYS_RTL of RUNUP_ROUNDKEYS is


type RUNUP_STATE_TYPE is  ( HOLD, CV_RUNUP, DONE );


-- ===========================================================================
-- =========================== Signal Definition =============================
-- ===========================================================================
type MOD_44_TABLE_TYPE is array (0 to 131) of integer range 0 to 131;

  constant mod44table : MOD_44_TABLE_TYPE := (
0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,
17,18,19,20,21,22,23,24,25,26,27,28,29,30,
31,32,33,34,35,36,37,38,39,40,41,42,43,
0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,
17,18,19,20,21,22,23,24,25,26,27,28,29,30,
31,32,33,34,35,36,37,38,39,40,41,42,43,
0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,
17,18,19,20,21,22,23,24,25,26,27,28,29,30,
31,32,33,34,35,36,37,38,39,40,41,42,43 );

type ROUND_VEC_TYPE is array(0 to 131) of SLV_8;

  constant ROUND_VEC : ROUND_VEC_TYPE := ( 
"00000000", 
"00000001", 
"00000010", 
"00000011", 
"00000100", 
"00000101", 
"00000110", 
"00000111", 
"00001000", 
"00001001", 
"00001010",
 
"00001011", 
"00001100",
"00001101", 
"00001110", 
"00001111", 
"00010000", 
"00010001", 
"00010010", 
"00010011", 
"00010100", 

"00010101", 
"00010110", 
"00010111", 
"00011000", 
"00011001",
"00011010", 
"00011011", 
"00011100", 
"00011101", 
"00011110", 

"00011111", 
"00100000", 
"00100001", 
"00100010", 
"00100011", 
"00100100", 
"00100101", 
"00100110",
"00100111", 
"00101000", 

"00101001", 
"00101010", 
"00101011", --43
"00101100", 
"00101101", 
"00101110", 
"00101111", 
"00110000", 
"00110001", 
"00110010", 
"00110011", 
"00110100", 

"00110101", 
"00110110", 
"00110111", 
"00111000", 
"00111001",
"00111010", 
"00111011", 
"00111100", 
"00111101", 
"00111110", 
"00111111", --63
"01000000",
"01000001", 
"01000010", 
"01000011", 
"01000100", 
"01000101", 
"01000110", 
"01000111", 
"01001000", 
"01001001", 
"01001010",
 
"01001011", 
"01001100",
"01001101", 
"01001110", 
"01001111", 
"01010000", 
"01010001", 
"01010010", 
"01010011", 
"01010100", 

"01010101", 
"01010110", 
"01010111", 
"01011000", 
"01011001",
"01011010", 
"01011011", 
"01011100", 
"01011101", 
"01011110", 

"01011111", 
"01100000", 
"01100001", 
"01100010", 
"01100011", 
"01100100", 
"01100101", 
"01100110",
"01100111", 
"01101000", 

"01101001", 
"01101010", 
"01101011", 
"01101100", 
"01101101", 
"01101110", 
"01101111", 
"01110000", 
"01110001", 
"01110010", 
"01110011", 
"01110100", 

"01110101", 
"01110110", 
"01110111", 
"01111000", 
"01111001",
"01111010", 
"01111011", 
"01111100", 
"01111101", 
"01111110", 
"01111111", --127
"10000000",
"10000001",
"10000010",
"10000011" );


constant INIT_S_VALUE  : S_ARRAY_TYPE := (
X"b7e15163",
X"5618cb1c",
X"f45044d5",
X"9287be8e",
X"30bf3847",
X"cef6b200",
X"6d2e2bb9",
X"0b65a572",
X"a99d1f2b",
X"47d498e4",
X"e60c129d",
X"84438c56",
X"227b060f",
X"c0b27fc8",
X"5ee9f981",
X"fd21733a",
X"9b58ecf3",
X"399066ac",
X"d7c7e065",
X"75ff5a1e",
X"1436d3d7",
X"b26e4d90",
X"50a5c749",
X"eedd4102",
X"8d14babb",
X"2b4c3474",
X"c983ae2d",
X"67bb27e6",
X"05f2a19f",
X"a42a1b58",
X"42619511",
X"e0990eca",
X"7ed08883",
X"1d08023c",
X"bb3f7bf5",
X"5976f5ae",
X"f7ae6f67",
X"95e5e920",
X"341d62d9",
X"d254dc92",
X"708c564b",
X"0ec3d004",
X"acfb49bd",
X"4b32c376" );


signal CV_RUNUP_STEP : integer range 0 to 255; -- counter for cv runup (0 to 31)
signal RUNUP_STATE : RUNUP_STATE_TYPE;
signal S           : S_ARRAY_TYPE;
signal L           : L_ARRAY_TYPE;
signal i           : integer range 0 to 63;
signal j           : integer range 0 to 3;
signal A           : SLV_32;
signal previous_A  : SLV_32;
signal B           : L_ARRAY_TYPE;
signal   TEMP_B    : L_ARRAY_TYPE;
signal round_num   : ROUND_VEC_TYPE;
signal ks_cv_size  : SLV_8;


begin

GEN_KEYS: for i in 0 to 43 generate

 KS_ROUNDKEYS(i) <= S(i);

end generate;

GEN_ROUND_NUM: for i in 0 to 131 generate

        round_num(i) <= ROUND_VEC(i);
 
end generate;



-- ===========================================================================
-- =========================== Data Movement =================================
-- ===========================================================================

GEN_INITIAL_W : process( RUNUP_STATE )

begin

   case RUNUP_STATE is
      when DONE => 
         KS_READY <= '1';
      when others => 
         KS_READY <= '0';
   end case;


end process;  -- GEN_INITIAL_W



DATA_FLOW : process( clock, reset )

variable TEMP_A  : SLV_32;
variable TEMP_L  : L_ARRAY_TYPE;
variable TEMP_S  : S_ARRAY_TYPE;

variable index   : ARRAY_INDEX_TYPE;

begin

  if reset = '1' then

    i <= 0;                              -- Clear index
    j <= 0;                              -- Clear index
         A <= ( others => '0' );              -- Clear temporary variable
         previous_A <= ( others => '0' );

        for i in 0 to 3 loop
         B(i) <= ( others => '0' );              -- Clear temporary variable
         TEMP_B(i) <= ( others => '0' );
        end loop;


  
    for i in 0 to 43 loop                -- Assign computed values to S array
      S(i) <= INIT_S_VALUE(i);
    end loop;  


    for i in 0 to 3 loop                 -- Assign cryptovariable to L array
      L(i) <= KS_CV(i);
    end loop;  


  elsif clock'event and clock = '1' then

   case RUNUP_STATE is
      when HOLD => 
         i <= 0;                         -- Clear index
         j <= 0;                         -- Clear index
         A <= ( others => '0' );              -- Clear temporary variable
         previous_A <= ( others => '0' );

        for i in 0 to 3 loop
         B(i) <= ( others => '0' );              -- Clear temporary variable
         TEMP_B(i) <= ( others => '0' );
        end loop;
 

         if KS_LOADCV = '1' THEN

     for i in 0 to 43 loop                -- Assign computed values to S array
      S(i) <= INIT_S_VALUE(i);
    end loop;


           for i in 0 to 3 loop
             L(i) <= KS_CV(i);             -- Assign cryptovariable to L array
           end loop;  
         else

           for i in 0 to 43 loop                
             S(i) <= S(i);                 -- Hold current S array
           end loop;  

           for i in 0 to 3 loop
             L(i) <= L(i);                 -- Hold current L array
           end loop;  

         end if;

      when DONE => 
         i <= 0;                         -- Clear index
         j <= 0;                         -- Clear index

         A <= ( others => '0' );              -- Clear temporary variable
         previous_A <= ( others => '0' );

        for i in 0 to 3 loop
         B(i) <= ( others => '0' );              -- Clear temporary variable
         TEMP_B(i) <= ( others => '0' );
        end loop;

               

         if KS_LOADCV = '1' THEN

    for i in 0 to 43 loop                -- Assign computed values to S array
      S(i) <= INIT_S_VALUE(i);
    end loop;

           for i in 0 to 3 loop
             L(i) <= KS_CV(i);             -- Assign cryptovariable to L array
           end loop;  

         else

           for i in 0 to 43 loop                
             S(i) <= S(i);                 -- Hold current S array
           end loop;  

           for i in 0 to 3 loop
             L(i) <= L(i);                 -- Hold current L array
           end loop;  

         end if;

      when others => 

        previous_A <= A;
        TEMP_A := A;
        TEMP_B <= B;


          TEMP_A := COMPUTE_KS_A( previous_A,
                                  TEMP_B,
                                  S(mod44table(CV_RUNUP_STEP)),
                                  round_num(CV_RUNUP_STEP)); -- compute A 
                                                             -- and update S(i)

          S((mod44table(CV_RUNUP_STEP))) <= TEMP_A;    -- save new computed S(i)

          TEMP_L := COMPUTE_KS_B( TEMP_B,
                                  L,
                                  TEMP_A,
                                  round_num(CV_RUNUP_STEP) ); -- compute B and
                                                              --  update L(j)

          L          <= TEMP_L;
          TEMP_B     <= TEMP_L;                    -- save previous B value
          previous_A <= TEMP_A;


        A      <= TEMP_A;
        B      <= TEMP_B;

   end case;

  end if;

end process; -- DATA_FLOW




  
-- ===========================================================================
-- =========================== State Machine / Controller ====================
-- ===========================================================================

RUNUP_FLOW: process( clock, reset )

begin

   if reset = '1' then                     -- Active high reset (asynchronous)

      CV_RUNUP_STEP <= 0;                  -- put runup controller in hold
      RUNUP_STATE   <= HOLD;

   elsif clock'event and clock = '1' then

      case RUNUP_STATE is

         when HOLD =>
            CV_RUNUP_STEP <= 0;                  -- put runup controller in hold

            if KS_LOADCV = '1' then
               RUNUP_STATE <= CV_RUNUP;
            else
               RUNUP_STATE <= HOLD;
            end if;

         when CV_RUNUP =>

            if ( CV_RUNUP_STEP /= 131 ) then  -- current activity

               CV_RUNUP_STEP <= CV_RUNUP_STEP + 1;     -- increment counter
               RUNUP_STATE   <= RUNUP_STATE;


            else

               RUNUP_STATE <= DONE;
               CV_RUNUP_STEP <= 0;

            end if;


         when DONE =>

            if KS_LOADCV = '1' then

               RUNUP_STATE   <= CV_RUNUP;
               CV_RUNUP_STEP <= 0;
            else

               RUNUP_STATE   <= DONE;
               CV_RUNUP_STEP <= 0;

            end if;

      end case;
                                             
   end if;  -- reset = '1'

end process; -- RUNUP_FLOW


end RUNUP_ROUNDKEYS_RTL;
