-- *************************************************************************
-- DISCLAIMER. THIS SOFTWARE WAS WRITTEN BY EMPLOYEES OF THE U.S.
-- GOVERNMENT AS A PART OF THEIR OFFICIAL DUTIES AND, THEREFORE, IS NOT
-- PROTECTED BY COPYRIGHT. HOWEVER, THIS SOFTWARE CODIFIES THE FINALIST
-- CANDIDATE ALGORITHMS (i.e., MARS, RC6tm, RIJNDAEL, SERPENT, AND
-- TWOFISH) IN THE ADVANCED ENCRYPTION STANDARD (AES) DEVELOPMENT EFFORT
-- SPONSORED BY THE NATIONAL INSTITUTE OF STANDARDS AND TECHNOLOGY (NIST)
-- AND MAY BE PROTECTED BY ONE OR MORE FORMS OF INTELLECTUAL PROPERTY. THE
-- U.S. GOVERNMENT MAKES NO WARRANTY, EITHER EXPRESSED OR IMPLIED,
-- INCLUDING BUT NO LIMITED TO ANY IMPLIED WARRANTIES OF MERCHANTABILITY
-- OR FITNESS FOR A PARTICULAR PURPOSE, REGARDING THIS SOFTWARE. THE U.S.
-- GOVERNMENT FURTHER MAKES NO WARRANTY THAT THIS SOFTWARE WILL NOT
-- INFRINGE ANY OTHER UNITED STATES OR FOREIGN PATENT OR OTHER
-- INTELLECTUAL PROPERTY RIGHT. IN NO EVENT SHALL THE U.S. GOVERNMENT BE
-- LIABLE TO ANYONE FOR COMPENSATORY, PUNITIVE, EXEMPLARY, SPECIAL,
-- COLLATERAL, INCIDENTAL, CONSEQUENTIAL, OR ANY OTHER TYPE OF DAMAGES IN
-- CONNECTION WITH OR ARISING OUT OF COPY OR USE OF THIS SOFTWARE.
-- *************************************************************************
--
-- File Name: tb_ecb_vt_dec_iter.vhdl
-- Author   : NSA
-- Date     : Oct 1999
-- Project  : AES Candidate Evaluation --RC6
-- Purpose  : This test bench exercises the top level structural model for
--            an iterative implementation of RC6. All vectors from the
--            ECB mode (decrypt) Known Answer Test, Variable Text are
--            tested and verified.
-----------------------------------------------------------------------------

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use WORK.rc6_pack.all;
use ieee.std_logic_arith.all;
use ieee.std_logic_unsigned.all;
use ieee.std_logic_textio.all;
use std.textio.all;


entity TB_ECB_VT_DEC_ITER is
end;

architecture tb of TB_ECB_VT_DEC_ITER is

   type RESULT     is (fail, pass, undefined);

   constant CLK_PER   : time:= 10 ns;

   signal clock       : std_logic := '0';  -- rising edge clock input
   signal reset       : std_logic;  -- asynchronous reset (active high)

   signal enc_dec_b   : std_logic;  -- encrypt select (0=encrypt, 1=decrypt)
   signal data_in     : SLV_128;    -- data input bus 
   signal data_load   : std_logic;  -- data start signal
   signal cv_in       : CV_TYPE;    -- cryptovariable input bus
   signal cv_load     : std_logic;  -- load a new cryptovariable

   signal data_out    : SLV_128;   -- output data bus
   signal done        : std_logic; -- processing done

   signal test_result          : RESULT := undefined;
   signal pass128              : integer := 0;
   signal fail128              : integer := 0;

   signal tmp_data_out         : SLV_128;


   FILE test_vector_file: text is in
	  "./KAT_vectors/ecb_vt_ct.vec";


   component RC6_TOP_PIPE

      port( clock      : in std_logic;
            reset      : in std_logic;

            enc_dec_b  : in std_logic;
            data_in    : in SLV_128; 
            data_load  : in std_logic;
            cv_in      : in CV_TYPE;
            cv_load    : in std_logic;

            data_out   : out SLV_128;
            done       : out std_logic );

   end component;

begin

   top_level : RC6_TOP_PIPE
               PORT MAP (clock, reset, enc_dec_b, data_in, data_load,
                         cv_in, cv_load, data_out, done);

proc_clock  : process

begin

  WAIT for CLK_PER / 2;
  clock <= not clock;

end process;



run_vectors : process

variable L1              : LINE;           -- Predefined in textio
variable tmp_data_in     : SLV_128;

begin

  reset      <= '1';
  enc_dec_b  <= '1';
  data_in    <= (others => '0'); 
  data_load  <= '0';
  cv_in      <= (others => '0');
  cv_load    <= '0';

  wait for 5*CLK_PER;
  reset <= '0';

  enc_dec_b    <= '0';


    cv_in        <= (others => '0');
    cv_load      <= '1';
    wait for CLK_PER;

    cv_in        <= (others => '1');
    cv_load    <= '0';
    wait for NUM_RUNUP_ROUNDS * CLK_PER;


    for j in 1 to 128 loop

      if not (endfile(test_vector_file)) then
        readline(test_vector_file, L1);
        hread(L1, tmp_data_in);
      else
        tmp_data_in := (others => 'X');
      end if;

      data_in   <= tmp_data_in; 
      data_load  <= '1';
      wait for CLK_PER;

      data_load <= '0';
      wait for (LAST_ROUND+3) * CLK_PER;

    end loop;



  wait for 1000*CLK_PER;

end process;




check_output : process (clock, reset)


begin

  if reset = '1' then

    test_result                <= undefined;
    pass128                    <= 0;
    fail128                    <= 0;
    tmp_data_out(127)          <= '1';
    tmp_data_out(126 downto 0) <= (others => '0');

  elsif clock'event and clock = '1' then

    if done = '1' then

        if tmp_data_out = data_out then
          test_result <= pass;
          pass128 <= pass128 + 1;

        else
          test_result <= fail;
          fail128 <= fail128 + 1;

        end if;  -- tmp_data = data_out

        tmp_data_out <= tmp_data_out(0) & tmp_data_out(127 downto 1);

      else

        test_result <= undefined;
        tmp_data_out <= tmp_data_out;

    end if;  -- done = '1'

  end if;  -- reset = '1'

end process;


end tb;



configuration cfg_TB_ECB_VT_DEC_ITER of TB_ECB_VT_DEC_ITER is
   for tb
      for top_level : RC6_TOP_PIPE 
         use entity WORK.RC6_TOP_ITER(STRUCTURAL);
      end for;
   end for;
end;
