-- *************************************************************************
-- DISCLAIMER. THIS SOFTWARE WAS WRITTEN BY EMPLOYEES OF THE U.S.
-- GOVERNMENT AS A PART OF THEIR OFFICIAL DUTIES AND, THEREFORE, IS NOT
-- PROTECTED BY COPYRIGHT. HOWEVER, THIS SOFTWARE CODIFIES THE FINALIST
-- CANDIDATE ALGORITHMS (i.e., MARS, RC6tm, RIJNDAEL, SERPENT, AND
-- TWOFISH) IN THE ADVANCED ENCRYPTION STANDARD (AES) DEVELOPMENT EFFORT
-- SPONSORED BY THE NATIONAL INSTITUTE OF STANDARDS AND TECHNOLOGY (NIST)
-- AND MAY BE PROTECTED BY ONE OR MORE FORMS OF INTELLECTUAL PROPERTY. THE
-- U.S. GOVERNMENT MAKES NO WARRANTY, EITHER EXPRESSED OR IMPLIED,
-- INCLUDING BUT NO LIMITED TO ANY IMPLIED WARRANTIES OF MERCHANTABILITY
-- OR FITNESS FOR A PARTICULAR PURPOSE, REGARDING THIS SOFTWARE. THE U.S.
-- GOVERNMENT FURTHER MAKES NO WARRANTY THAT THIS SOFTWARE WILL NOT
-- INFRINGE ANY OTHER UNITED STATES OR FOREIGN PATENT OR OTHER
-- INTELLECTUAL PROPERTY RIGHT. IN NO EVENT SHALL THE U.S. GOVERNMENT BE
-- LIABLE TO ANYONE FOR COMPENSATORY, PUNITIVE, EXEMPLARY, SPECIAL,
-- COLLATERAL, INCIDENTAL, CONSEQUENTIAL, OR ANY OTHER TYPE OF DAMAGES IN
-- CONNECTION WITH OR ARISING OUT OF COPY OR USE OF THIS SOFTWARE.
-- *************************************************************************

-- ===========================================================================
-- File Name : alg_iterative.vhdl
-- Author    : NSA
-- Date      : 07 September 99
-- Project   : AES Candidate Evaluation
-- Purpose   : This block implements the Serpent algorithm
--             for an iterative case
-- Notes     :
-- ===========================================================================

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use WORK.serpent_pack.all;

-- ===========================================================================
-- =========================== Interface Description =========================
-- ===========================================================================

entity ALG_ITERATIVE is

  port (clock       :  in STD_LOGIC;
        reset       :  in STD_LOGIC;

        ALG_DATAIN  :  in SLV_128;
        ALG_KEY1    :  in SLV_128;
        ALG_START   :  in STD_LOGIC;
        ALG_ENC     :  in STD_LOGIC;

        ALG_DATAOUT :  out SLV_128;
        ALG_DONE    :  out STD_LOGIC
  );

end ALG_ITERATIVE;

architecture ALG_ITERATIVE_RTL of ALG_ITERATIVE is


-- ===========================================================================
-- =========================== Constant Definition ===========================
-- ===========================================================================

constant HOLD : integer := 0;         -- Hold state for controller

-- ===========================================================================
-- =========================== Signal Definition =============================
-- ===========================================================================

signal ROUND          : INTEGER;   -- round number
signal BI_REG         : SLV_128;   -- data out from round i
signal BI_REG_INT     : SLV_128;   -- internal version of bi_reg (readable)
signal ALG_DATAIN_INT : SLV_128;   -- initial permutation of alg_datain

begin

-- ===========================================================================
-- =========================== Data Movement =================================
-- ===========================================================================


IP( ALG_DATAIN, ALG_DATAIN_INT );     -- Initial Permutation
FP( BI_REG, BI_REG_INT );             -- Final Permutation


DATA_FLOW : process( clock, reset )

variable round_input : SLV_128;

begin

if reset = '1' then                        -- check for reset condition

   BI_REG   <= ( others => '0' );          -- clear round key outputs
   ALG_DONE <= '0';                        -- clear done signal

elsif clock'event and clock = '1' then     -- rising edge clock

   if ALG_START = '1' then
      round_input := ALG_DATAIN_INT;
   else
      round_input := BI_REG;
   end if;

   SERPENT_ROUND( round_input,          -- input data
                  ALG_KEY1,             -- encrypt/decrypt subkey
                  ALG_ENC,              -- encrypt/decrypt
                  STD_LOGIC_VECTOR(TO_UNSIGNED(ROUND, 6)),  -- round number
                  BI_REG );             -- round output

   if ROUND = LAST_ROUND+1 then            -- check for algorithm completion
      ALG_DONE <= '1';                     -- indicate algorithm complete
   else
      ALG_DONE <= '0';                     -- indicate algorithm not complete
   end if;

end if; -- reset

end process; -- DATA_FLOW


-- ===========================================================================
-- =========================== Concurrent Statements =========================
-- ===========================================================================

ALG_DATAOUT <= BI_REG_INT when ROUND = LAST_ROUND+2  -- output data when valid
                          else (others => '0');      -- else output all '0'


-- ===========================================================================
-- =========================== State Machine / Controller ====================
-- ===========================================================================

STATE_FLOW : process( clock, reset )

variable active : std_logic;                 -- indicates whether algorithm is
                                             -- active (1) or in hold (0)

begin

if reset = '1' then                          -- active high reset (asynch)

   ROUND  <= HOLD;                           -- put controller in hold state
   active := '0';                            -- stop process until ALG_START=1

elsif clock'event and clock = '1' then       -- check for rising edge clock

   if  ( ALG_START = '1' or active = '1' )   -- check for inactive  
   and ( ROUND /= LAST_ROUND+2 ) then        -- and completion of algorithm

      active := '1';                         -- enable controller
      ROUND  <= ROUND + 1;                   -- active, so increment counter

   else

      active := '0';                         -- disable controller
      ROUND  <= HOLD;                        -- reset counter
             
   end if;

end if; -- reset

end process; -- STATE_FLOW

end ALG_ITERATIVE_RTL;
