-- *************************************************************************
-- DISCLAIMER. THIS SOFTWARE WAS WRITTEN BY EMPLOYEES OF THE U.S.
-- GOVERNMENT AS A PART OF THEIR OFFICIAL DUTIES AND, THEREFORE, IS NOT
-- PROTECTED BY COPYRIGHT. HOWEVER, THIS SOFTWARE CODIFIES THE FINALIST
-- CANDIDATE ALGORITHMS (i.e., MARS, RC6tm, RIJNDAEL, SERPENT, AND
-- TWOFISH) IN THE ADVANCED ENCRYPTION STANDARD (AES) DEVELOPMENT EFFORT
-- SPONSORED BY THE NATIONAL INSTITUTE OF STANDARDS AND TECHNOLOGY (NIST)
-- AND MAY BE PROTECTED BY ONE OR MORE FORMS OF INTELLECTUAL PROPERTY. THE
-- U.S. GOVERNMENT MAKES NO WARRANTY, EITHER EXPRESSED OR IMPLIED,
-- INCLUDING BUT NO LIMITED TO ANY IMPLIED WARRANTIES OF MERCHANTABILITY
-- OR FITNESS FOR A PARTICULAR PURPOSE, REGARDING THIS SOFTWARE. THE U.S.
-- GOVERNMENT FURTHER MAKES NO WARRANTY THAT THIS SOFTWARE WILL NOT
-- INFRINGE ANY OTHER UNITED STATES OR FOREIGN PATENT OR OTHER
-- INTELLECTUAL PROPERTY RIGHT. IN NO EVENT SHALL THE U.S. GOVERNMENT BE
-- LIABLE TO ANYONE FOR COMPENSATORY, PUNITIVE, EXEMPLARY, SPECIAL,
-- COLLATERAL, INCIDENTAL, CONSEQUENTIAL, OR ANY OTHER TYPE OF DAMAGES IN
-- CONNECTION WITH OR ARISING OUT OF COPY OR USE OF THIS SOFTWARE.
-- *************************************************************************

-- ===========================================================================
-- File Name : key_sched_pipe.vhdl
-- Author    : NSA
-- Date      : October 1999
-- Project   : Serpent pipelined key schedule block 
-- Purpose   : build key schedule for pipelined implementation
-- Notes     :
-- ===========================================================================

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use WORK.serpent_pack.all;

-- ===========================================================================
-- =========================== Interface Description =========================
-- ===========================================================================

entity KEY_SCHEDULE_PIPE is

  port (clock        : in std_logic;       -- clock signal
        reset        : in std_logic;       -- active high reset (asynch)

        KS_LOADCV    : in std_logic;       -- load a new cryptovariable
        KS_CV        : in SLV_256;         -- cryptovariable input bus
        KS_ENC       : in std_logic;       -- encrypt select (1=enc,0=dec)
        KS_INITIAL   : in W_TYPE;          -- input from initial expand block

        KS_ROUND_KEY : out PIPE_KEY_TYPE   -- output round key (Ki)

  );

end KEY_SCHEDULE_PIPE;

architecture KEY_SCHEDULE_PIPE_RTL of KEY_SCHEDULE_PIPE is


-- ===========================================================================
-- =========================== Component Definition ==========================
-- ===========================================================================

component REG128B 

  port (clock      : in std_logic;   -- clock signal
        reset      : in std_logic;   -- active high reset (asynchronous)

        DATA_IN    : in SLV_128;     -- input data bus
        DATA_OUT   : out SLV_128     -- output data bus

  );

end component;


component REGISTER_DELAY_W

  port (clock    : in std_logic;   -- clock signal
        reset    : in std_logic;   -- active high reset (asynchronous)

        DATA_IN  : in W_TYPE;      -- input data bus

        DATA_OUT : out W_TYPE      -- output data bus

  );

end component;

-- ===========================================================================
-- =========================== Signal Definition =============================
-- ===========================================================================

signal W                : W_PIPE_TYPE;    -- W register interconnections
signal W_INT            : W_PIPE_TYPE;    -- procedure call to reg interconnect
signal W_ORIG           : W_TYPE;         -- First W reg. connection (from CV )
signal KS_ROUND_KEY_INT : PIPE_KEY_TYPE;  -- procedure call to reg interconnect
signal ROUND_STEP       : ROUND_TYPE;     -- current algorithm step

begin


-- ===========================================================================
-- ====================== Generate Pipe Structure ============================
-- ===========================================================================
--
-- PURPOSE:  
--
-- The following generate statements create a pipelined architecture for 
-- the key expansion.  For each round of expansion, a new set of W values
-- are created, followed by the creation of Ki (Sbox lookup on the new W
-- values that were generated).



G1: for ROUND_STEP in FIRST_ROUND to LAST_ROUND+1 generate

-- =========================== First Round ===================================
-- Generate first round and special case connections- includes W initialization

   G1a: if (ROUND_STEP = FIRST_ROUND) generate


--      Removed from pipeline and created in a separate entity (TLY 3/14/00)
--
--      CVMOD  : INITIAL_EXPAND( KS_CV,                -- CV runup
--                               KS_ENC,
--                               W_ORIG);

      W_ORIG <= KS_INITIAL;

      STEP   : KS_ROUND( KS_ENC,                     -- First round
                         W_ORIG,
                         STD_LOGIC_VECTOR(TO_UNSIGNED(ROUND_STEP, 6)),     
                         STD_LOGIC_VECTOR(TO_UNSIGNED(ROUND_STEP mod 8, 6)),
                         W_INT(ROUND_STEP), 
                         KS_ROUND_KEY_INT(ROUND_STEP) );

      KEYREG : REG128B port map ( clock,             -- Register round output
                                  reset,
                                  KS_ROUND_KEY_INT(ROUND_STEP), 
                                  KS_ROUND_KEY(ROUND_STEP) ); 

      WREG   : REGISTER_DELAY_W port map ( clock,    -- Register W value
                                           reset,
                                           W_INT(ROUND_STEP),
                                           W(ROUND_STEP) );  

   end generate;  -- G1a

-- ===========================  Round i ======================================
-- Generate middle rounds for creating Ki.

   G1b: if (ROUND_STEP <= (LAST_ROUND+1) and ROUND_STEP > 0) generate 

      STEP   : KS_Round( KS_ENC,                     -- Key expansion round
                         W(ROUND_STEP-1),
                         STD_LOGIC_VECTOR(TO_UNSIGNED(ROUND_STEP, 6)),     
                         STD_LOGIC_VECTOR(TO_UNSIGNED(ROUND_STEP mod 8, 6)),
                         W_INT(ROUND_STEP),
                         KS_ROUND_KEY_INT(ROUND_STEP) ); 


      KEYREG : REG128B port map ( clock,             -- Register round output
                                  reset,
                                  KS_ROUND_KEY_INT(ROUND_STEP), 
                                  KS_ROUND_KEY(ROUND_STEP) ); 

      WREG   : REGISTER_DELAY_W port map ( clock,    -- Register W value
                                           reset,
                                           W_INT(ROUND_STEP),
                                           W(ROUND_STEP) );  

   end generate;  -- G1b

end generate;   -- G1



end KEY_SCHEDULE_PIPE_RTL;

-- ===========================================================================
-- ========================= Configuration ===================================
-- ===========================================================================

configuration CFG_KEY_SCHEDULE_PIPE of KEY_SCHEDULE_PIPE is

   for KEY_SCHEDULE_PIPE_RTL

   end for;

end;


