-- *************************************************************************
-- DISCLAIMER. THIS SOFTWARE WAS WRITTEN BY EMPLOYEES OF THE U.S.
-- GOVERNMENT AS A PART OF THEIR OFFICIAL DUTIES AND, THEREFORE, IS NOT
-- PROTECTED BY COPYRIGHT. HOWEVER, THIS SOFTWARE CODIFIES THE FINALIST
-- CANDIDATE ALGORITHMS (i.e., MARS, RC6tm, RIJNDAEL, SERPENT, AND
-- TWOFISH) IN THE ADVANCED ENCRYPTION STANDARD (AES) DEVELOPMENT EFFORT
-- SPONSORED BY THE NATIONAL INSTITUTE OF STANDARDS AND TECHNOLOGY (NIST)
-- AND MAY BE PROTECTED BY ONE OR MORE FORMS OF INTELLECTUAL PROPERTY. THE
-- U.S. GOVERNMENT MAKES NO WARRANTY, EITHER EXPRESSED OR IMPLIED,
-- INCLUDING BUT NO LIMITED TO ANY IMPLIED WARRANTIES OF MERCHANTABILITY
-- OR FITNESS FOR A PARTICULAR PURPOSE, REGARDING THIS SOFTWARE. THE U.S.
-- GOVERNMENT FURTHER MAKES NO WARRANTY THAT THIS SOFTWARE WILL NOT
-- INFRINGE ANY OTHER UNITED STATES OR FOREIGN PATENT OR OTHER
-- INTELLECTUAL PROPERTY RIGHT. IN NO EVENT SHALL THE U.S. GOVERNMENT BE
-- LIABLE TO ANYONE FOR COMPENSATORY, PUNITIVE, EXEMPLARY, SPECIAL,
-- COLLATERAL, INCIDENTAL, CONSEQUENTIAL, OR ANY OTHER TYPE OF DAMAGES IN
-- CONNECTION WITH OR ARISING OUT OF COPY OR USE OF THIS SOFTWARE.
-- *************************************************************************

-- ===========================================================================
-- File Name: serpent_pkg.vhdl
-- Author   : NSA
-- Date     : August 1999
-- Project  : SERPENT
-- Purpose  : This package defines common types, subtypes, constants,
--            and functions required to implement various VHDL models
--            for the creation of ASIC simulation of SERPENT, an Advanced
--            Encryption Standard (AES) candidate algorithm.
--
-- ===========================================================================

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

package serpent_pack is

-- ==========================================================================
-- ======= Type, sub-type and function declarations for general use =========
-- ==========================================================================
type STATE_TYPE         is ( nop, wait4ks, ready, busy );

subtype SLV_2           is std_logic_vector(0 to 1);
subtype SLV_4           is std_logic_vector(0 to 3);
subtype SLV_5           is std_logic_vector(0 to 4);
subtype SLV_32          is std_logic_vector(0 to 31);
subtype SLV_33          is std_logic_vector(0 to 32);
subtype SLV_64          is std_logic_vector(0 to 63);
subtype SLV_128         is std_logic_vector(0 to 127);
subtype SLV_256         is std_logic_vector(0 to 255);

subtype ROUND_TYPE      is std_logic_vector(0 to 5);
subtype SBOX_INDEX_TYPE is std_logic_vector(0 to 5);

constant FIRST_ROUND : integer := 0;
constant LAST_ROUND  : integer := 31;


-- ==========================================================================
-- ============ Declarations for the Encrypt/Decrypt section ================
-- ==========================================================================

type PIPE_DATA_TYPE is array(FIRST_ROUND to LAST_ROUND+1) of SLV_128;
type SBOX_TYPE      is array (0 to 15) of SLV_4;
type SBOX_MATRIX    is array (0 to 7) of SBOX_TYPE;

-- ==========================================================================
-- ================================ SBOX ====================================
-- ==========================================================================

constant SBOX : SBOX_MATRIX := (
--S0:  { 3, 8,15, 1,10, 6, 5,11,14,13, 4, 2, 7, 0, 9,12 }
("0011", "1000", "1111", "0001", "1010", "0110", "0101", "1011",
 "1110", "1101", "0100", "0010", "0111", "0000", "1001", "1100"),

--S1:  {15,12, 2, 7, 9, 0, 5,10, 1,11,14, 8, 6,13, 3, 4 }
("1111", "1100", "0010", "0111", "1001", "0000", "0101", "1010",
 "0001", "1011", "1110", "1000", "0110", "1101", "0011", "0100"),

--S2:  { 8, 6, 7, 9, 3,12,10,15,13, 1,14, 4, 0,11, 5, 2 }
("1000", "0110", "0111", "1001", "0011", "1100", "1010", "1111",
 "1101", "0001", "1110", "0100", "0000", "1011", "0101", "0010"),

--S3:  { 0,15,11, 8,12, 9, 6, 3,13, 1, 2, 4,10, 7, 5,14 }
("0000", "1111", "1011", "1000", "1100", "1001", "0110", "0011",
 "1101", "0001", "0010", "0100", "1010", "0111", "0101", "1110"), 

--S4:  { 1,15, 8, 3,12, 0,11, 6, 2, 5, 4,10, 9,14, 7,13 }
("0001", "1111", "1000", "0011", "1100", "0000", "1011", "0110",
 "0010", "0101", "0100", "1010", "1001", "1110", "0111", "1101"),

--S5:  { 15, 5, 2,11, 4,10, 9,12, 0, 3,14, 8,13, 6, 7, 1 }
("1111", "0101", "0010", "1011", "0100", "1010", "1001", "1100",
 "0000", "0011", "1110", "1000", "1101", "0110", "0111", "0001"),

--S6:  { 7, 2,12, 5, 8, 4, 6,11,14, 9, 1,15,13, 3,10, 0 }
("0111", "0010", "1100", "0101", "1000", "0100", "0110", "1011",
 "1110", "1001", "0001", "1111", "1101", "0011", "1010", "0000"),

--S7:  { 1,13,15, 0,14, 8, 2,11, 7, 4,12,10, 9, 3, 5, 6 }
("0001", "1101", "1111", "0000", "1110", "1000", "0010", "1011",
 "0111", "0100", "1100", "1010", "1001", "0011", "0101", "0110")
);



-- ==========================================================================
-- ============================= INVERSE SBOX ===============================
--  Note: Inverse S-Box is specified in reverse order for ease of indexing
-- ==========================================================================

constant InvSBOX : SBOX_MATRIX := (
--InvS7:  { 3, 0, 6,13, 9,14,15, 8, 5,12,11, 7,10, 1, 4, 2 }
("0011", "0000", "0110", "1101", "1001", "1110", "1111", "1000", 
 "0101", "1100", "1011", "0111", "1010", "0001", "0100", "0010"),

--InvS6:  {15,10, 1,13, 5, 3, 6, 0, 4, 9,14, 7, 2,12, 8,11 }
("1111", "1010", "0001", "1101", "0101", "0011", "0110", "0000", 
 "0100", "1001", "1110", "0111", "0010", "1100", "1000", "1011"),

--InvS5:  { 8,15, 2, 9, 4, 1,13,14,11, 6, 5, 3, 7,12,10, 0 }
("1000", "1111", "0010", "1001", "0100", "0001", "1101", "1110", 
 "1011", "0110", "0101", "0011", "0111", "1100", "1010", "0000"),

--InvS4:  { 5, 0, 8, 3,10, 9, 7,14, 2,12,11, 6, 4,15,13, 1 }
("0101", "0000", "1000", "0011", "1010", "1001", "0111", "1110", 
 "0010", "1100", "1011", "0110", "0100", "1111", "1101", "0001"),

--InvS3:  { 0, 9,10, 7,11,14, 6,13, 3, 5,12, 2, 4, 8,15, 1 }
("0000", "1001", "1010", "0111", "1011", "1110", "0110", "1101", 
 "0011", "0101", "1100", "0010", "0100", "1000", "1111", "0001"),

--InvS2:  {12, 9,15, 4,11,14, 1, 2, 0, 3, 6,13, 5, 8,10, 7 }
("1100", "1001", "1111", "0100", "1011", "1110", "0001", "0010", 
 "0000", "0011", "0110", "1101", "0101", "1000", "1010", "0111"),

--InvS1:  { 5, 8, 2,14,15, 6,12, 3,11, 4, 7, 9, 1,13,10, 0 }
("0101", "1000", "0010", "1110", "1111", "0110", "1100", "0011", 
 "1011", "0100", "0111", "1001", "0001", "1101", "1010", "0000"),

--InvS0:  {13, 3,11, 0,10, 6, 5,12, 1,14, 4, 7,15, 9, 8, 2 }
("1101", "0011", "1011", "0000", "1010", "0110", "0101", "1100", 
 "0001", "1110", "0100", "0111", "1111", "1001", "1000", "0010")
);


function SBOX_LOOKUP (sbox_table : in SBOX_TYPE;
                      vector_in  : in SLV_128 )
                      return SLV_128;

function LINEAR_TRANSFORM (vector_in : in SLV_128)
                           return SLV_128;

function INVERSE_LINEAR_TRANSFORM (vector_in : in SLV_128)
                                   return SLV_128;

function SERPENT_ROUND_FUNCT (datain     :  in SLV_128;
                              encrypt    :  in std_logic;
                              key1       :  in SLV_128;
                              sbox_index :  in SBOX_INDEX_TYPE )
                              return SLV_128;


procedure SERPENT_ROUND(datain     :  in SLV_128;
                        key1       :  in SLV_128;
                        encrypt    :  in std_logic;
                        sbox_index :  in SBOX_INDEX_TYPE;
                 signal dataout    :  out SLV_128  ); 



-- ==========================================================================
-- ============== Declarations for the Key Schedule section =================
-- ==========================================================================

constant PHI                  : SLV_32  := X"9e3779b9";
constant HOLD                 : integer := 0;
constant CV_WIDTH             : integer := 256;

constant LAST_ECVRUNUP_STEP   : integer := 1;  -- Number of steps for cv runup
constant RUNUP_CYCLES_DEC     : integer := 33; -- Total number of operations
                                               -- for decrypt runup
constant RUNUP_OPS_PER_CLK    : integer := 1;  -- Number of runup operations 
                                               -- performed per clock period
constant LAST_DCVRUNUP_STEP_I : integer := RUNUP_CYCLES_DEC/RUNUP_OPS_PER_CLK;


type W_TYPE        is array (-8 to -1) of SLV_32;
type WTEMP_TYPE    is array (-8 to 3) of SLV_32;
type W_PIPE_TYPE   is array (FIRST_ROUND to LAST_ROUND+1) of W_TYPE;
type PIPE_KEY_TYPE is array (FIRST_ROUND to LAST_ROUND+1) of SLV_128;

subtype CV_TYPE    is std_logic_vector(0 to CV_WIDTH-1);


function UPDATE_W_ENCRYPT (round : in ROUND_TYPE;
                           w_in  : in W_TYPE )
                           return W_TYPE;

function UPDATE_W_DECRYPT (round : in ROUND_TYPE;
                           w_in  : in W_TYPE)
                           return W_TYPE;

function CREATE_ROUNDKEY (encrypt    : in STD_LOGIC;
                          sbox_index : in SBOX_INDEX_TYPE;
                          w_in       : in W_TYPE )
                          return SLV_128;

procedure KS_ROUND (encrypt    : in  STD_LOGIC;
                    w_in       : in  W_TYPE;
                    round      : in  ROUND_TYPE;
                    sbox_index : in  SBOX_INDEX_TYPE;
             signal w_out      : out W_TYPE;
             signal roundkey   : out SLV_128 );

function INITIAL_EXPAND_FUNCT (cv_in   : in CV_TYPE;
                               encrypt : in std_logic )
                               return W_TYPE;

procedure INITIAL_EXPAND (cv_in   : in CV_TYPE;
                          encrypt : in std_logic;
                   signal w_out   : out W_TYPE );


function IP_FUNCT (datain : in SLV_128 )
                   return SLV_128;

procedure IP (datain  : in SLV_128;
       signal dataout : out SLV_128 );

function FP_FUNCT (datain : in SLV_128 )
                   return SLV_128;

procedure FP (datain  : in SLV_128;
       signal dataout : out SLV_128 );


end serpent_pack;


-- ==========================================================================
-- ==========================================================================
-- ==========================================================================


package body serpent_pack is

-- ==========================================================================
-- ============= Definitions for the Encrypt/Decrypt section ================
-- ==========================================================================

-- ==========================================================================
--
--  function SBOX_LOOKUP
--
--  Performs the sbox function implemented as a lookup table. The sbox
--  table is passed into the function to allow re-use of the same
--  function for all 8 different sboxes (and 8 inverse sboxes. There
--  are 32 copies of the 4-bit sbox to cover all 128 bits of input/output.
--
-- ==========================================================================

function SBOX_LOOKUP (sbox_table : in SBOX_TYPE;
                      vector_in  : in SLV_128)
                      return SLV_128 is

-- pragma map_to_operator SBOX_LOOKUP_dw_op
-- pragma return_port_name SBOX_LOOKUP_out

variable sboxout   : SLV_128;
variable temp_sbox : SLV_128;

begin

   for i in 0 to 31 loop

      temp_sbox(i*4 to i*4+3) :=
      sbox_table(TO_INTEGER(unsigned(vector_in(i*4 to i*4+3))));

   end loop;
  
   sboxout := temp_sbox;

   return sboxout;

end SBOX_LOOKUP;


-- ==========================================================================
--
--  function LINEAR_TRANSFORM
--
--  Performs the linear transformation implemented as combinational logic 
--  as described in the SERPENT algorithm specification.
--
-- ==========================================================================

function LINEAR_TRANSFORM (vector_in : in SLV_128)
                           return SLV_128 is

-- pragma map_to_operator LINEAR_TRANSFORM_dw_op
-- pragma return_port_name LINEAR_TRANSFORM_out

variable x0, x1, x2, x3 : unsigned(0 to 31);  -- register mapping
variable lt_out         : SLV_128;            -- transform output

begin

-- =========================== Final Permutation ============================

  lt_out := FP_FUNCT( vector_in );

  x3 := unsigned(lt_out(0 to 31));
  x2 := unsigned(lt_out(32 to 63));
  x1 := unsigned(lt_out(64 to 95));
  x0 := unsigned(lt_out(96 to 127));
 
-- =========================== Apply Transform =============================

  x0 := x0 rol 13;                   -- x0 <<< 13
  x2 := x2 rol 3;                    -- x2 <<< 3
  x1 := x1 xor x0 xor x2;            -- x1 + x0 + x2
  x3 := x3 xor x2 xor (x0 sll 3);    -- x3 + x2 + x0 << 3

  x1 := x1 rol 1;                    -- x1 <<< 1
  x3 := x3 rol 7;                    -- x3 <<< 7
  x0 := x0 xor x1 xor x3;            -- x0 + x1 + x3
  x2 := x2 xor x3 xor (x1 sll 7);    -- x2 + x3 + x1 << 7

  x0 := x0 rol 5;                    -- x0 <<< 5
  x2 := x2 rol 22;                   -- x2 <<< 22


-- ========================= Initial Permutation ===========================

  lt_out := IP_FUNCT( std_logic_vector(x3 & x2 & x1 & x0) );

  return lt_out;

end LINEAR_TRANSFORM;


-- ==========================================================================
--
--  function INVERSE_LINEAR_TRANSFORM
--
--  Performs the inverse linear transformation implemented as combinational
--  logic as described in the SERPENT algorithm specification.
--
-- ==========================================================================

function INVERSE_LINEAR_TRANSFORM (vector_in : in SLV_128)
                                   return SLV_128 is

-- pragma map_to_operator INVERSE_LINEAR_TRANSFORM_dw_op
-- pragma return_port_name INVERSE_LINEAR_TRANSFORM_out

variable x0, x1, x2, x3 : unsigned(0 to 31);  -- register mapping
variable ltinv_out      : SLV_128;            -- transform output

begin

-- =========================== Final Permutation ============================

  ltinv_out := FP_FUNCT( vector_in );

  x3 := unsigned(ltinv_out(0 to 31));
  x2 := unsigned(ltinv_out(32 to 63));
  x1 := unsigned(ltinv_out(64 to 95));
  x0 := unsigned(ltinv_out(96 to 127));

-- ======================= Apply Inverse Transform ==========================

  x2 := x2 ror 22;                   -- x2 >>> 22
  x0 := x0 ror 5;                    -- x0 >>> 5

  x2 := x2 xor x3 xor (x1 sll 7);    -- x2 + x3 + x1 << 7
  x0 := x0 xor x1 xor x3;            -- x0 + x1 + x3
  x3 := x3 ror 7;                    -- x3 >>> 7
  x1 := x1 ror 1;                    -- x1 >>> 1

  x3 := x3 xor x2 xor (x0 sll 3);    -- x3 + x2 + x0 << 3
  x1 := x1 xor x0 xor x2;            -- x1 + x0 + x2
  x2 := x2 ror 3;                    -- x2 >>> 3
  x0 := x0 ror 13;                   -- x0 >>> 13

-- ========================= Initial Permutation ===========================

  ltinv_out := IP_FUNCT( std_logic_vector(x3 & x2 & x1 & x0) );

  return ltinv_out;

end INVERSE_LINEAR_TRANSFORM;


-- ==========================================================================
--
--  function SERPENT_ROUND
--
--  Performs one round of the SERPENT block cipher. Encryption or decryption
--  is performed based on the 'encrypt' signal.
--  For encryption, the last round replaces the linear transform with
--  a key addition step using key2; all rounds use key1.
--  For decryption, the first round replaces the inverse linear transform
--  with a key addition step using key1; all rounds use key2.
--  Result of the round is returned on dataout.
--
-- ==========================================================================

function SERPENT_ROUND_FUNCT (datain     : in SLV_128;
                              encrypt    : in std_logic;
                              key1       : in SLV_128;
                              sbox_index : in SBOX_INDEX_TYPE )
                              return SLV_128 is

-- pragma map_to_operator SERPENT_ROUND_FUNCT_dw_op
-- pragma return_port_name SERPENT_ROUND_FUNCT_out

variable step_one : std_logic_vector(0 to 127);
variable sboxout  : std_logic_vector(0 to 127);
variable dataout  : SLV_128;

begin

-- ===========================================================================
-- ============================== Encryption =================================
-- ===========================================================================

   if encrypt = '1' then

      case TO_INTEGER(UNSIGNED(sbox_index)) is
  
         when LAST_ROUND =>      -- last round (step 1)

            step_one := datain xor key1;
            dataout  := SBOX_LOOKUP(SBOX(
                        TO_INTEGER(UNSIGNED(sbox_index)) mod 8), step_one);

         when LAST_ROUND+1 =>    -- last round (step 2)

            dataout := datain xor key1;

         when others =>          -- full encrypt case

            step_one := datain xor key1;
            sboxout  := SBOX_LOOKUP(SBOX(
                        TO_INTEGER(UNSIGNED(sbox_index)) mod 8), step_one);
            dataout  := LINEAR_TRANSFORM(sboxout);

      end case;

-- ===========================================================================
-- ============================== Decryption =================================
-- ===========================================================================

   else

      case TO_INTEGER(UNSIGNED(sbox_index)) is

         when FIRST_ROUND =>      -- first round (step 1)

            dataout := datain xor key1;

         when FIRST_ROUND+1 =>    -- first round (step 2)

            sboxout := SBOX_LOOKUP(InvSBOX(
                      (TO_INTEGER(UNSIGNED(sbox_index)) - 1) mod 8), datain );
            dataout := sboxout xor key1;

         when others =>           -- full decrypt case

            step_one := INVERSE_LINEAR_TRANSFORM (datain);
            sboxout  := SBOX_LOOKUP(InvSBOX(
                        (TO_INTEGER(UNSIGNED(sbox_index))-1) mod 8), step_one);
            dataout  := sboxout xor key1;

      end case;

   end if; -- encrypt = '1'

   return dataout;

end SERPENT_ROUND_FUNCT;


-- ==========================================================================
--
--  procedure SERPENT_ROUND
--
--  Performs one round of the SERPENT block cipher. Encryption or decryption
--  is performed based on the 'encrypt' signal.
--  For encryption, the last round replaces the linear transform with
--  a key addition step using key2; all rounds use key1.
--  For decryption, the first round replaces the inverse linear transform
--  with a key addition step using key1; all rounds use key2.
--  Result of the round is returned on dataout.
--
-- ==========================================================================

procedure SERPENT_ROUND (datain     :  in SLV_128;
                         key1       :  in SLV_128;
                         encrypt    :  in std_logic;
                         sbox_index :  in SBOX_INDEX_TYPE;
                  signal dataout    :  out SLV_128  ) is


begin

   dataout <= SERPENT_ROUND_FUNCT (datain, encrypt, key1, sbox_index);

end SERPENT_ROUND;


-- ==========================================================================
-- =============== Definitions for the Key Schedule section =================
-- ==========================================================================

-- ==========================================================================
--
--  function UPDATE_W_ENCRYPT & function UPDATE_W_DECRYPT
--
-- This function initializes and updates 8- 32 bit words that are used for
-- the key schedule.  It takes as input "w_in", a 12 word x 32 bit array,
-- updates them as appropriate, and returns "w_out", an 8 word x 32 bit array.
-- (The remaining 4 word x 32 bit component is used as a temporary workspace)
-- There are separate functions for encryption and decryption.
--
-- ==========================================================================


function UPDATE_W_ENCRYPT (round : in ROUND_TYPE;
                           w_in  : in W_TYPE)
                           return W_TYPE is

-- pragma map_to_operator UPDATE_W_ENCRYPT_dw_op
-- pragma return_port_name UPDATE_W_ENCRYPT_out

variable w_temp    : WTEMP_TYPE;
variable w_out     : W_TYPE;
variable round_pad : SLV_32;

begin
   
   for i in -8 to -1 loop
      w_temp(i) := w_in (i);
   end loop;
  
   for j in 0 to 3 loop

      -- XOR six words, then rotate left by 11 bits

      w_temp(j) := w_temp(j-1) XOR w_temp(j-3) XOR w_temp(j-5) XOR
                   w_temp(j-8) XOR PHI XOR 
                   std_logic_vector(TO_UNSIGNED(
                   TO_INTEGER(UNSIGNED(round))*4+j, 32));
      w_temp(j) := std_logic_vector(UNSIGNED (w_temp(j)) rol 11);

   end loop;

   for k in -4 to 3 loop
      w_out(k-4) := w_temp(k);
   end loop;

   return w_out;

end UPDATE_W_ENCRYPT;

-- ==========================================================================


function UPDATE_W_DECRYPT (round : in ROUND_TYPE;
                           w_in  : in W_TYPE )
                           return W_TYPE is

-- pragma map_to_operator UPDATE_W_DECRYPT_dw_op
-- pragma return_port_name UPDATE_W_DECRYPT_out

variable w_temp    : WTEMP_TYPE;
variable w_out     : W_TYPE;
variable round_pad : SLV_32;

begin
 
   for i in -4 to 3 loop
      w_temp(i) := w_in (i-4);
   end loop;

   for j in -5 downto -8 loop

      -- rotate right by 11 bits, then XOR six words

      w_temp(j) := std_logic_vector(UNSIGNED (w_temp(j+8)) ror 11);

      w_temp(j) := w_temp(j) XOR w_temp(j+7) XOR w_temp(j+5) XOR
                   w_temp(j+3) XOR PHI XOR
                   std_logic_vector(TO_UNSIGNED(
                   TO_INTEGER(UNSIGNED(round)) * 4 + j + 8 + 8, 32));

   end loop;

   for k in -8 to -1 loop
      w_out(k) := w_temp(k);
   end loop;

   return w_out;

end UPDATE_W_DECRYPT;


-- ==========================================================================
--
--  function CREATE_ROUNDKEY 
--
-- This function takes the current W register and sbox number as input and
-- generates the corresponding round key.
--
-- ==========================================================================


function CREATE_ROUNDKEY (encrypt    : in STD_LOGIC;
                          sbox_index : in SBOX_INDEX_TYPE;
                          w_in       : in W_TYPE )
                          return SLV_128 is

-- pragma map_to_operator CREATE_ROUNDKEY_dw_op
-- pragma return_port_name CREATE_ROUNDKEY_out

variable roundkey : SLV_128;

begin

   if encrypt ='1' then

      for i in 31 downto 0 loop   -- format inputs for Sbox lookup
         roundkey(i*4 to i*4+3)  := w_in(-1)(i) & w_in(-2)(i) &
                                    w_in(-3)(i) & w_in(-4)(i);
      end loop;

                                  -- apply S-box to input 128 bit word
      roundkey := SBOX_LOOKUP(sbox(
                  (3 - TO_INTEGER(UNSIGNED(sbox_index))) mod 8), roundkey);

   else

      for i in 31 downto 0 loop   -- format inputs for Sbox lookup
         roundkey(i*4 to i*4+3)  := w_in(-5)(i) & w_in(-6)(i) &
                                    w_in(-7)(i) & w_in(-8)(i);
      end loop;

                                  -- apply S-box to resulting 128 bit word
      roundkey := SBOX_LOOKUP(sbox(
                  (3 + TO_INTEGER(UNSIGNED(sbox_index))) mod 8), roundkey);

   end if;

   return roundkey;

end CREATE_ROUNDKEY;


-- ==========================================================================
--
--  procedure KS_ROUND
--
--  Performs one round of key schedule for Serpent
--
--  w        := encrypt/decrypt function call
--  roundkey <= use updated w register to create the round key
--
-- ==========================================================================


procedure KS_ROUND (encrypt    : in  STD_LOGIC;
                    w_in       : in  W_TYPE;
                    round      : in  ROUND_TYPE;
                    sbox_index : in  SBOX_INDEX_TYPE;
             signal w_out      : out W_TYPE;
             signal roundkey   : out SLV_128 ) is

variable w : W_TYPE;

begin

   if encrypt ='1' then

      w        := UPDATE_W_ENCRYPT (round, w_in);
      roundkey <= CREATE_ROUNDKEY  (encrypt, sbox_index, w );

   else

     if TO_INTEGER(UNSIGNED(round)) = 0 then 
       w  := w_in;
     else
       w  := UPDATE_W_DECRYPT(STD_LOGIC_VECTOR(TO_UNSIGNED(
             32 - TO_INTEGER(UNSIGNED(round)), 6)), w_in);
     end if;

     roundkey <= CREATE_ROUNDKEY  (encrypt, sbox_index, w );

   end if;

   for i in -8 to -1 loop
      w_out(i) <= w(i);    -- update array "w_out"
   end loop;

end KS_ROUND;


-- ==========================================================================
--
--  function INITIAL_EXPAND_FUNCT
--
--  Creates the W register array to determine the last 8 values
--  of the W expansion.  Used to provide a starting point for the 
--  decryption part of the key schedule.
--
-- ==========================================================================

function INITIAL_EXPAND_FUNCT (cv_in   : in CV_TYPE;
                               encrypt : in std_logic )
                               return W_TYPE is

-- pragma map_to_operator INITIAL_EXPAND_FUNCT_dw_op
-- pragma return_port_name INITIAL_EXPAND_FUNCT_out

variable w     : W_TYPE;
variable w_out : W_TYPE;

begin
  
   for w_index in 0 to 7 loop
      w(-1-w_index) := cv_in( (w_index)*32 to (((w_index+1)*32)-1) );
   end loop;

   if encrypt = '0' then

      for index in 0 to 33 loop
         w := UPDATE_W_ENCRYPT(STD_LOGIC_VECTOR(TO_UNSIGNED(index, 6)), w);
      end loop;

   else

      w := w;
    
   end if;

   for w_index in -8 to -1 loop
      w_out(w_index) := w(w_index);
   end loop;

   return w_out;

end INITIAL_EXPAND_FUNCT;


-- ==========================================================================
--
--  procedure INITIAL_EXPAND
--
--  Creates the W register array to determine the last 8 values
--  of the W expansion.  Used to provide a starting point for the 
--  decryption part of the key schedule.
--
-- ==========================================================================

procedure INITIAL_EXPAND (cv_in   : in CV_TYPE;
                          encrypt : in std_logic;
                   signal w_out   : out W_TYPE ) is

begin
  
   w_out <= INITIAL_EXPAND_FUNCT (cv_in, encrypt);

end INITIAL_EXPAND;


-- ==========================================================================
--
--  function IP_FUNCT
--
--  Performs the Initial Permutation on a 128-bit word
--
-- ==========================================================================

function IP_FUNCT (datain : in SLV_128)
                   return SLV_128 is

-- pragma map_to_operator IP_FUNCT_dw_op
-- pragma return_port_name IP_FUNCT_out

variable dataout : SLV_128;

begin
 
   for i in 0 to 31 loop
      dataout(i*4)   := datain(i);
      dataout(i*4+1) := datain(i+32);
      dataout(i*4+2) := datain(i+64);
      dataout(i*4+3) := datain(i+96);
   end loop;

   return dataout;

end IP_FUNCT;


-- ==========================================================================
--
--  procedure IP
--
--  Performs the Initial Permutation on a 128-bit word
--
-- ==========================================================================

procedure IP (datain  : in SLV_128;
       signal dataout : out SLV_128 ) is

begin

   dataout <= IP_FUNCT (datain); 

end IP;


-- ==========================================================================
--
--  function FP_FUNCT
--
--  Performs the Final Permutation on a 128-bit word
--
-- ==========================================================================

function FP_FUNCT (datain : in SLV_128 )
                   return SLV_128 is

-- pragma map_to_operator FP_FUNCT_dw_op
-- pragma return_port_name FP_FUNCT_out

variable dataout : SLV_128;

begin
 
   for i in 0 to 7 loop

      dataout(i*4)   := datain(i*16+0+0);
      dataout(i*4+1) := datain(i*16+4+0);
      dataout(i*4+2) := datain(i*16+8+0);
      dataout(i*4+3) := datain(i*16+12+0);

      dataout(i*4+0+32) := datain(i*16+0+1);
      dataout(i*4+1+32) := datain(i*16+4+1);
      dataout(i*4+2+32) := datain(i*16+8+1);
      dataout(i*4+3+32) := datain(i*16+12+1);

      dataout(i*4+0+64) := datain(i*16+0+2);
      dataout(i*4+1+64) := datain(i*16+4+2);
      dataout(i*4+2+64) := datain(i*16+8+2);
      dataout(i*4+3+64) := datain(i*16+12+2);

      dataout(i*4+0+96) := datain(i*16+0+3);
      dataout(i*4+1+96) := datain(i*16+4+3);
      dataout(i*4+2+96) := datain(i*16+8+3);
      dataout(i*4+3+96) := datain(i*16+12+3);

   end loop;

   return dataout;

end FP_FUNCT;


-- ==========================================================================
--
--  procedure FP
--
--  Performs the Final Permutation on a 128-bit word (wrapper)
--
-- ==========================================================================

procedure FP (datain  : in SLV_128;
       signal dataout : out SLV_128 ) is

begin
 
   dataout <= FP_FUNCT (datain);

end FP;

-- ==========================================================================

end serpent_pack;
