-- *************************************************************************
-- DISCLAIMER. THIS SOFTWARE WAS WRITTEN BY EMPLOYEES OF THE U.S.
-- GOVERNMENT AS A PART OF THEIR OFFICIAL DUTIES AND, THEREFORE, IS NOT
-- PROTECTED BY COPYRIGHT. HOWEVER, THIS SOFTWARE CODIFIES THE FINALIST
-- CANDIDATE ALGORITHMS (i.e., MARS, RC6tm, RIJNDAEL, SERPENT, AND
-- TWOFISH) IN THE ADVANCED ENCRYPTION STANDARD (AES) DEVELOPMENT EFFORT
-- SPONSORED BY THE NATIONAL INSTITUTE OF STANDARDS AND TECHNOLOGY (NIST)
-- AND MAY BE PROTECTED BY ONE OR MORE FORMS OF INTELLECTUAL PROPERTY. THE
-- U.S. GOVERNMENT MAKES NO WARRANTY, EITHER EXPRESSED OR IMPLIED,
-- INCLUDING BUT NO LIMITED TO ANY IMPLIED WARRANTIES OF MERCHANTABILITY
-- OR FITNESS FOR A PARTICULAR PURPOSE, REGARDING THIS SOFTWARE. THE U.S.
-- GOVERNMENT FURTHER MAKES NO WARRANTY THAT THIS SOFTWARE WILL NOT
-- INFRINGE ANY OTHER UNITED STATES OR FOREIGN PATENT OR OTHER
-- INTELLECTUAL PROPERTY RIGHT. IN NO EVENT SHALL THE U.S. GOVERNMENT BE
-- LIABLE TO ANYONE FOR COMPENSATORY, PUNITIVE, EXEMPLARY, SPECIAL,
-- COLLATERAL, INCIDENTAL, CONSEQUENTIAL, OR ANY OTHER TYPE OF DAMAGES IN
-- CONNECTION WITH OR ARISING OUT OF COPY OR USE OF THIS SOFTWARE.
-- *************************************************************************
-- ===========================================================================
-- ===========================================================================
-- File Name : key_sched_iterative.vhdl
-- Author    : NSA
-- Date      : December 1999
-- Project   : AES Candidate Evaluation - MARS
-- Purpose   : build key schedule for iterative implementation
-- Notes     :
-- ===========================================================================

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use WORK.mars_pack.all;

-- ===========================================================================
-- =========================== Interface Description =========================
-- ===========================================================================

entity KEY_SCHEDULE_ITERATIVE is

  port (clock            :  in std_logic;    -- clock signal
        reset            :  in std_logic;    -- active high reset (asynch)

        KS_LOADCV        :  in std_logic;    -- load a new cryptovariable
        KS_START         :  in std_logic;    -- start a new expansion sequence
        KS_CV            :  in CV_ARRAY_TYPE; -- cryptovariable input bus
        KS_CVSIZE        :  in SLV_2;        -- cryptovariable size select
        KS_ENC_DECB      :  in std_logic;

        KS_READY         :  out std_logic;   -- '1'=cv expanded, '0'=inprocess
        KS_ROUNDKEYS  :  out K_ARRAY_TYPE   -- output key bus

  );

end KEY_SCHEDULE_ITERATIVE;

architecture KEY_SCHEDULE_ITERATIVE_RTL of KEY_SCHEDULE_ITERATIVE is


-- ===========================================================================
-- =========================== Type Definition ===============================
-- ===========================================================================

type EXPAND_STATE_TYPE is ( HOLD, CV_EXPAND );
type RUNUP_STATE_TYPE is  ( HOLD, CV_RUNUP, DONE );
type RUNUP_SUB_STATE_TYPE is ( IDLE,TRANSFORM, STIR1, STIR2, STIR3, STIR4, KEYFIX);

-- ===========================================================================
-- =========================== Signal Definition =============================
-- ===========================================================================

signal KS_CV_SWAP      : CV_ARRAY_TYPE;

signal CV_RUNUP_STEP : integer range 0 to 255; -- counter for cv runup (0 to 31)
signal RUNUP_STATE :RUNUP_STATE_TYPE;
signal RUNUP_SUB_STATE : RUNUP_SUB_STATE_TYPE;

signal K           : K_ARRAY_TYPE;
signal i           : integer range 0 to 63;
signal j           : integer range 0 to 7;
signal T           : T_ARRAY_TYPE;

signal fix_index   : integer range 0 to 63;


begin



-- ===========================================================================
-- =========================== Signal Mapping ================================
-- ===========================================================================

KS_ROUNDKEYS <= K;


-- ===========================================================================
-- =========================== Data Movement =================================
-- ===========================================================================

GEN_INITIAL_W : process( RUNUP_STATE, CV_RUNUP_STEP, KS_ENC_DECB )

begin
  if KS_ENC_DECB = '1' then
    if CV_RUNUP_STEP > KEY_OVERRUN_STEP-1 or RUNUP_STATE = DONE then
      KS_READY <= '1';
    else
      KS_READY <= '0';
    end if; 
  else
    case RUNUP_STATE is
      when DONE => 
         KS_READY <= '1';
      when others => 
         KS_READY <= '0';
    end case;
  end if;

end process;  -- GEN_INITIAL_W


DATA_FLOW : process( clock, reset )

variable w : SLV_32;
variable r : SLV_5;
variable M : SLV_32;
variable p : SLV_32;
variable T_TEMP : T_ARRAY_TYPE;

begin

  if reset = '1' then

    i <= 0;                              -- Clear index
    j <= 0;                              -- Clear index

    fix_index <= 5;

    T <= INIT_T_FUNCT (KS_CV,KS_CVSIZE);

    for i in 0 to 39 loop                -- Assign computed values to S array
      K(i) <= ( others => '0' );
    end loop;  

  elsif clock'event and clock = '1' then

   case RUNUP_STATE is
      when HOLD => 
         i <= 5;                         -- Clear index
         j <= 0;                         -- Clear index
         fix_index <= 5;

         if KS_LOADCV = '1' THEN

           T <= INIT_T_FUNCT (KS_CV,KS_CVSIZE);

         else

           for i in 0 to 14 loop                
             T(i) <= T(i);                 -- Hold current T array
           end loop;  

         end if;


      when DONE => 
         i <= 5;                         -- Clear index
         j <= 0;                         -- Clear index
         fix_index <= 5;

         if KS_LOADCV = '1' THEN

           T <= INIT_T_FUNCT (KS_CV, KS_CVSIZE);

         else

           for i in 0 to 14 loop                
             T(i) <= T(i);                 -- Hold current T array
           end loop;  

         end if;

      when others => 

        case RUNUP_SUB_STATE is
          when TRANSFORM =>
            T <= LINEAR_TRANSFORM(  std_logic_vector(TO_UNSIGNED(j,2)),T );
            j <= j;

          when STIR1 | STIR2 | STIR3 =>
            T <= STIR( T );
            j <= j;
          when STIR4 =>
            T_TEMP := STIR( T );
            for z in 0 to 14 loop
              T(z) <= T_TEMP(z);
            end loop;
            j <= j;
            for z in 0 to 9 loop
              K(10*j+z) <= T_TEMP(4*z mod 15);
            end loop;

          when KEYFIX =>

            w := K(fix_index)(31 downto 2) & "11";
            M := GENERATE_MASK( w );
            r := K(fix_index-1)(4 downto 0);
            p := std_logic_vector(unsigned(
                   B( TO_INTEGER(unsigned( K(fix_index)(1 downto 0)  ))) ) rol TO_INTEGER(unsigned(r)
                   ));
            for z in 0 to 35 loop
              if z = fix_index then
                K(z) <= w xor ( p and M );
              else
                K(z) <= K(z);
              end if;
            end loop;

            if ( fix_index + 2 ) < ((j+1)*10) then
              fix_index <= fix_index + 2;
              j <= j;
            else
              fix_index <= fix_index;
              j <= j+1;
            end if;

           when others =>
             T <= T;
             j <= j;
             fix_index <= 5;

       end case;
     end case;

  end if;

end process; -- DATA_FLOW

-- ===========================================================================
-- =========================== State Machine / Controller ====================
-- ===========================================================================

RUNUP_FLOW: process( clock, reset )

begin

   if reset = '1' then                     -- Active high reset (asynchronous)

      CV_RUNUP_STEP <= 0;                  -- put runup controller in hold
      RUNUP_STATE   <= HOLD;
      RUNUP_SUB_STATE <= TRANSFORM;

   elsif clock'event and clock = '1' then

      case RUNUP_STATE is

         when HOLD =>
 
            if KS_LOADCV = '1' then
               RUNUP_STATE <= CV_RUNUP;
               RUNUP_SUB_STATE <= TRANSFORM;

            else
               RUNUP_STATE <= HOLD;
            end if;

         when CV_RUNUP =>

            if ( CV_RUNUP_STEP /= LAST_CVRUNUP_STEP-1 ) then  -- current activity

               CV_RUNUP_STEP <= CV_RUNUP_STEP + 1;     -- increment counter
               RUNUP_STATE   <= RUNUP_STATE;

              case RUNUP_SUB_STATE is
 
                when IDLE =>
                  RUNUP_SUB_STATE <= IDLE;
                when TRANSFORM =>
                  RUNUP_SUB_STATE <= STIR1;
                when STIR1 =>
                  RUNUP_SUB_STATE <= STIR2;
                when STIR2 =>
                  RUNUP_SUB_STATE <= STIR3;
                when STIR3 =>
                  RUNUP_SUB_STATE <= STIR4;
                when STIR4 =>
                  RUNUP_SUB_STATE <= KEYFIX;
                when KEYFIX =>
                  if ( fix_index + 2 ) < ((j+1)*10) then
                    RUNUP_SUB_STATE <= KEYFIX;
                  else
                    RUNUP_SUB_STATE <= TRANSFORM;
                  end if;
              end case;


            else                              -- disable cvrunup controller

               CV_RUNUP_STEP <= 0;
               RUNUP_STATE   <= DONE;
               RUNUP_SUB_STATE <= IDLE;

            end if;
           
         when DONE =>

            if KS_LOADCV = '1' then

               RUNUP_STATE   <= CV_RUNUP;
               CV_RUNUP_STEP <= 0;
               RUNUP_SUB_STATE <= TRANSFORM;

            else

               RUNUP_STATE   <= DONE;
               CV_RUNUP_STEP <= 0;
               RUNUP_SUB_STATE <= IDLE;

            end if;

      end case;
                                             
   end if;  -- reset = '1'

end process; -- RUNUP_FLOW



-- ===========================================================================
-- =========================== BLock Connections =============================
-- ===========================================================================


G0: for CV_INDEX in 0 to 7 generate

  SWAP: BYTESWAP32( KS_CV(CV_INDEX),
                    KS_CV_SWAP(CV_INDEX) );

end generate;



end KEY_SCHEDULE_ITERATIVE_RTL;
