/**
   DERUtils.java

   Copyright (C) 1999, Claymore Systems, Inc.
   All Rights Reserved.

   ekr@rtfm.com  Sat Jul 24 21:24:06 1999

   This package is a SSLv3/TLS implementation written by Eric Rescorla
   <ekr\@rtfm.com> and licensed by Claymore Systems, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:
   1. Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
   3. All advertising materials mentioning features or use of this software
      must display the following acknowledgement:
      This product includes software developed by Claymore Systems, Inc.
   4. Neither the name of Claymore Systems, Inc. nor the name of Eric
      Rescorla may be used to endorse or promote products derived from this
      software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
   ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
   FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
   OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
   HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
   LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
   OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
   SUCH DAMAGE.


   $Id: DERUtils.java,v 1.4 1999/11/30 20:59:25 ekr Exp $

*/

package COM.claymoresystems.cert;

import java.math.BigInteger;
import java.util.BitSet;
import java.io.*;

/** Simple DER encode utilities. Eventually we'd like to use one
 common ASN.1 engine for everything */
public class DERUtils {
     public static final byte BOOLEAN=2;     
     public static final byte INTEGER=2;
     public static final byte BIT_STRING=3;
     public static final byte OCTET_STRING=4;          
     public static final byte OID=6;
     public static final byte SEQUENCE=(byte)0x30; // Constructed SEQUENCE
     public static final byte IA5STRING=(byte)0x16;

     private static void writeLength(int length, OutputStream os)
       throws IOException {
       if(length<128){
	 os.write((byte)length);
       }
       else{
	 ByteArrayOutputStream bs=new ByteArrayOutputStream();
	 int i;
	 
	 for(i=1;length>0;i++){
	   bs.write((byte)(length & 0xff));
	   length>>=8;
	 }

	 byte[] b=bs.toByteArray();
	 os.write((byte)(0x80 | b.length));

	 for(i=0;i<b.length;i++){
	   os.write(b[b.length-(i+1)]);
	 }
       }
     }

     private static void encodeBytes(byte type, byte[] in, OutputStream os)
       throws IOException{
       os.write(type);
       writeLength(in.length,os);
       os.write(in);
     }
     
     public static void encodeInteger(BigInteger i, OutputStream os)
       throws IOException {
       byte[] i_bytes=i.toByteArray();

       if(i_bytes[0]==0 && i_bytes.length>1 && ((i_bytes[1] & 0x80) == 0)){
	 byte[] n_bytes=new byte[i_bytes.length-1];
	 System.arraycopy(i_bytes,1,n_bytes,0,n_bytes.length);
	 i_bytes=n_bytes;
       }
       encodeBytes(INTEGER,i_bytes,os);
     }

     
     public static void encodeSequence(byte[] in,OutputStream os)
       throws IOException{
       encodeBytes(SEQUENCE,in,os);
     }

     public static void encodeOID(byte[] in,OutputStream os)
       throws IOException {
       encodeBytes(OID,in,os);
     }

     public static void encodeIA5String(String in, OutputStream os)
       throws IOException{
       encodeBytes(IA5STRING,in.getBytes(),os);
     }

     public static void encodeBitString(byte[] in,OutputStream os)
       throws IOException {
       os.write(BIT_STRING);
       writeLength(in.length+1,os);
       os.write(0);
       os.write(in);
     }

     public static byte[] decodeSequence(InputStream is)
       throws IOException {
       return(readTLV(SEQUENCE,is));
     }

     public static byte[] decodeOID(InputStream is)
       throws IOException {
       return(readTLV(OID,is));
     }

     public static byte[] decodeOctetString(InputStream is)
       throws IOException {
       return(readTLV(OCTET_STRING,is));
     }

     public static BigInteger decodeInteger(InputStream is)
       throws IOException {
       byte[] intb=readTLV(INTEGER,is);
       return new BigInteger(1,intb);
     }
       
     public static BitSet decodeBitStringX(InputStream is)
       throws IOException {
       byte[] b=decodeBitString(is);

       int shrt=b[0];
	 
       BitSet ret=new BitSet();
       for(int i=1;i<b.length;i++){
	 int max=(i==b.length-1)?8:8-shrt;
	 byte msk=(byte)0x80;
	   
	 for(int j=0;j<max;j++){
	   if((msk & b[i])!=0){
	     ret.set(((i-1)*8)+j);
	   }
	   msk>>=1;
	 }
       }

       return ret;
     }
	    
       
     public static byte[] decodeBitString(InputStream is)
       throws IOException {
       return(readTLV(BIT_STRING,is));
     }

     public static boolean decodeBoolean(InputStream is)
       throws IOException {
       byte[] v=readTLV(BOOLEAN,is);
       if(v.length!=1)
	 throw new IOException("Bad encoding for boolean");
       // Is this right?
       if(v[0]!=0)
	 return true;
       return false;
     }

     public static byte[] decodeAny(InputStream is)
       throws IOException {
       ByteArrayOutputStream os=new ByteArrayOutputStream();

       int t=decodeTag(is);

       byte[] b=readLV(is);
       encodeBytes((byte)t,b,os);

       return os.toByteArray();
     }
       
     private static byte[] readTLV(int type,InputStream is)
       throws IOException {
       decodeTagOrDie(type,is);

       byte[] ret=readLV(is);

       return ret;
     }

     private static byte[] readLV(InputStream is)
       throws IOException {
       int length=decodeLength(is);

       byte[] ret=new byte[length];

       int actualLength=is.read(ret);

       if(actualLength!=length)
	 throw new IOException("Bad encoding: short read");

       return ret;
     }

  
     private static void decodeTagOrDie(int tag,InputStream is)
       throws IOException {
       if(!decodeTag(tag,is))
	 throw new IOException("Bad encoding: wrong tag");
     }

     private static int decodeTag(InputStream is)
       throws IOException {
       int t=is.read();

       return t;
     }
     
     // This only knows small number form
     private static boolean decodeTag(int tag,InputStream is)
       throws IOException {
       int t=decodeTag(is);

       if(t<0)
	 return false;

       if(t!=tag)
	 return false;

       return true;
     }

     public static boolean isTag(int tag,InputStream is)
       throws IOException {
       is.mark(1);
       boolean result=decodeTag(tag,is);
       is.reset();

       return result;
     }

     private static int decodeLength(InputStream is)
       throws IOException {
       int length=0;

       int l=is.read();

       if(l<0)
	 throw new IOException("Bad encoding: short read");

       if((l & 0x80)==0)
	 return l;
       else{
	 int ct=l & 0x7f;

	 while(ct-->0){
	   if((l=is.read())<0)
	     throw new IOException("Bad encoding: short read");
	   length*=256;
	   length+=l;
	 }

	 return length;
       }
     }     
}
       
       
       
     
