/**
   SSLServerKeyExchange.java

   Copyright (C) 1999, Claymore Systems, Inc.
   All Rights Reserved.

   ekr@rtfm.com  Sat May  8 21:33:02 1999

   This package is a SSLv3/TLS implementation written by Eric Rescorla
   <ekr@rtfm.com> and licensed by Claymore Systems, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:
   1. Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
   3. All advertising materials mentioning features or use of this software
      must display the following acknowledgement:
      This product includes software developed by Claymore Systems, Inc.
   4. Neither the name of Claymore Systems, Inc. nor the name of Eric
      Rescorla may be used to endorse or promote products derived from this
      software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
   ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
   FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
   OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
   HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
   LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
   OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
   SUCH DAMAGE.

   $Id: SSLServerKeyExchange.java,v 1.6 1999/11/16 05:00:26 ekr Exp $

*/


package COM.claymoresystems.ptls;
import java.io.*;
import java.util.*;
import java.math.BigInteger;
import java.security.Signature;
import java.security.PublicKey;
import java.security.PrivateKey;
import cryptix.provider.rsa.RawRSAPublicKey;
import COM.claymoresystems.crypto.DHPublicKey;
import COM.claymoresystems.crypto.DHPrivateKey;
class SSLServerKeyExchange extends SSLPDU
{
     SSLDHParams dh_params;
     SSLRSAParams rsa_params;
     SSLopaque signature=new SSLopaque(-65535);
     SSLPDU par;
     int wb=0;
     String algorithm;
     
     public int encode(SSLConn conn, OutputStream s)
       throws IOException{
       ByteArrayOutputStream kex_os=new ByteArrayOutputStream();
       
       switch(conn.hs.cipher_suite.getKeyExchangeAlg()){
	 case SSLCipherSuite.SSL_KEX_DH:
	   conn.hs.dhEphemeral=conn.ctx.getEphemeralDHPrivateKey();
	   par=dh_params=new SSLDHParams(conn.hs.dhEphemeral);
	   break;
	 case SSLCipherSuite.SSL_KEX_RSA:
	   conn.hs.rsaEphemeral=conn.ctx.getEphemeralRSAPrivateKey();
	   par=rsa_params=new SSLRSAParams(conn.ctx.getEphemeralRSAPublicKey());
	 default:
	   throw new Error("Unknown key exchange algorithm");
       }
       par.encode(conn,kex_os);			
       byte[] kex_enc=kex_os.toByteArray();

       try {
	 // Sign the buffer
	 PrivateKey pk=conn.ctx.getPrivateKey();
	 String alg=pk.getAlgorithm();
	 Signature sigChecker;
	 
	 if(alg.equals("DSA")){
	   sigChecker=Signature.getInstance(alg,
	     LoadProviders.getDSAProvider());
	   sigChecker.setParameter("SecureRandom",
	     conn.hs.rng);
	 }
	 else {
	   sigChecker=Signature.getInstance(alg);
	 }
	 sigChecker.initSign(pk);
	 sigChecker.update(conn.hs.client_random);
	 sigChecker.update(conn.hs.server_random);
	 sigChecker.update(kex_enc);

	 byte[] sig=sigChecker.sign();

	 SSLDebug.debug(SSLDebug.DEBUG_CRYPTO,"Signed Data", kex_enc);
	 SSLDebug.debug(SSLDebug.DEBUG_CRYPTO,"Signature Data", sig);

	 signature.value=sig;
       } catch (Exception e){
	 throw new InternalError(e.toString());
       }

       wb=par.encode(conn,s);
       wb+=signature.encode(conn,s);

       return wb;
     }       
       
     public int decode(SSLConn conn, InputStream s)
       throws Error, java.io.IOException{
       int rb;
       PublicKey tmp_pk=null;
       PublicKey pk=conn.hs.peerSignatureKey;
       String algorithm;
       
       ByteArrayOutputStream kex_os=new ByteArrayOutputStream();
       
       switch(conn.hs.cipher_suite.getKeyExchangeAlg()){
	 case SSLCipherSuite.SSL_KEX_DH:
	   dh_params=new SSLDHParams();
	   rb=dh_params.decode(conn,s);
	   dh_params.encode(conn,kex_os);
	   tmp_pk=new DHPublicKey(
		new BigInteger(1,dh_params.DH_g.value),
		new BigInteger(1,dh_params.DH_p.value),
		new BigInteger(1,dh_params.DH_Ys.value));
	   break;
	 case SSLCipherSuite.SSL_KEX_RSA:
	   rsa_params=new SSLRSAParams();
	   rb=rsa_params.decode(conn,s);
	   rsa_params.encode(conn,kex_os);
	   tmp_pk=new RawRSAPublicKey(
		new BigInteger(1,rsa_params.RSA_modulus.value),
		new BigInteger(1,rsa_params.RSA_exponent.value));
	   break;
	 default:
	   throw new Error("Unknown key exchange algorithm");
       }

       rb+=signature.decode(conn,s);

       byte[] kex_enc=kex_os.toByteArray();

       try {
	 // Check the signature
	 Signature sigChecker=Signature.getInstance(conn.hs.cipher_suite.
	   getSignatureAlgNorm());
	 sigChecker.initVerify(pk);
	 sigChecker.update(conn.hs.client_random);
	 sigChecker.update(conn.hs.server_random);
	 sigChecker.update(kex_enc);

	 
	 SSLDebug.debug(SSLDebug.DEBUG_CRYPTO,"Signed Data", kex_enc);
	 SSLDebug.debug(SSLDebug.DEBUG_CRYPTO,"Signature Data", signature.value);
	 
	 if(!sigChecker.verify(signature.value))
	   conn.alert(SSLAlertX.TLS_ALERT_DECRYPT_ERROR);
       } catch (Exception e){
	 throw new InternalError(e.toString());
       }
       
       conn.hs.peerEncryptionKey=tmp_pk;
       
       return rb;
     }
}
       
       
     
