/**
   SSLSocket.java

   Copyright (C) 1999, Claymore Systems, Inc.
   All Rights Reserved.

   ekr@rtfm.com  Tue May 18 09:30:32 1999

   This package is a SSLv3/TLS implementation written by Eric Rescorla
   <ekr@rtfm.com> and licensed by Claymore Systems, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:
   1. Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
   3. All advertising materials mentioning features or use of this software
      must display the following acknowledgement:
      This product includes software developed by Claymore Systems, Inc.
   4. Neither the name of Claymore Systems, Inc. nor the name of Eric
      Rescorla may be used to endorse or promote products derived from this
      software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
   ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
   FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
   OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
   HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
   LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
   OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
   SUCH DAMAGE.

   $Id: SSLSocket.java,v 1.5 1999/07/27 04:41:48 ekr Exp $

*/


package COM.claymoresystems.ptls;
import COM.claymoresystems.sslg.*;
import COM.claymoresystems.cert.*;
import java.util.Vector;
import java.util.Date;
import java.net.InetAddress;
import java.io.*;

public class SSLSocket extends java.net.Socket
implements SSLSocketXInt {
     SSLConn conn;

     /** Create an SSLSocket and connect it to the server on the
	 specified host and port, doing the SSLHandshake.
	 We need this to support the CSJ interface, which is why
	 it duplicates the "int" version.

	 @param ctx the SSLContext to use to create this socket
	 @param remote_addr the hostname of the remote machine to connect to
	 @param port the port to connect to

	 @exception IOException if something goes wrong in the handshake or making the connection
     */
     public SSLSocket(SSLContext ctx,
       String remote_addr, Integer remote_port)
       throws java.net.UnknownHostException,java.io.IOException {
       this(ctx,remote_addr,remote_port.intValue());
     }

     /** Create an SSLSocket and connect it to the server on the
	 specified host and port, doing the SSLHandshake

	 @param ctx the SSLContext to use to create this socket
	 @param remote_addr the hostname of the remote machine to connect to
	 @param port the port to connect to

	 @exception IOException if something goes wrong in the handshake or making the connection
     */
     public SSLSocket(SSLContext ctx,
       String remote_addr, int remote_port)
       throws java.net.UnknownHostException,java.io.IOException {
       super(remote_addr,remote_port);

       internalSocket(ctx);
     }

     /** Creates a stream socket and connects it to the specified port number at the specified IP address performing the SSL Handshake

	 @param ctx the SSLContext to use to create this socket
	 @param addr the address to connect to
	 @param port the port to connect to 


	 @exception IOException if something goes wrong in the handshake or making the connection performing the SSL Handshake
     */	 
     public SSLSocket(SSLContext ctx,InetAddress addr, int port)
       throws IOException {
       super(addr,port);
       internalSocket(ctx);
     }

     /** Creates a stream socket and connects it to the specified port number at the specified IP address performing the SSL Handshake

	 @param ctx the SSLContext to use to create this socket
	 @param addr the address to connect to
	 @param port the port to connect to
	 @param localAddr the local address to bind to
	 @param localPort the local port to bind to

	 @exception IOException if something goes wrong in the handshake or making the connection
     */	 
     public SSLSocket(SSLContext ctx,InetAddress addr, int port, InetAddress localAddr, int localPort)
       throws IOException {
       super(addr,port,localAddr,localPort);
       internalSocket(ctx);
     }
       

     /** Creates a stream socket and connects it to the specified port number at the specified host, performing the SSL Handshake

	 @param ctx the SSLContext to use to create this socket
	 @param host the address to connect to
	 @param port the port to connect to
	 @param localAddr the local address to bind to
	 @param localPort the local port to bind to

	 @exception IOException if something goes wrong in the handshake or making the connection
     */	 
     public SSLSocket(SSLContext ctx,String host, int port, InetAddress localAddr, int localPort)
       throws IOException {
       super(host,port,localAddr,localPort);
       internalSocket(ctx);
     }
     // No-arg constructor for empty socket
     public SSLSocket(){
     }

     void internalSocket(SSLContext ctx)
       throws IOException {
       // Make the SSL connection
       conn=new SSLConn(this,super.getInputStream(),
	 super.getOutputStream(),ctx,SSLConn.SSL_CLIENT);
       conn.handshake();
     }
	 
     // serverHandshake
     void serverHandshake(SSLContext ctx)
       throws IOException {
       conn=new SSLConn(this,super.getInputStream(),
	 super.getOutputStream(),ctx,SSLConn.SSL_SERVER);
       conn.handshake();
     }
     /** close the connection. This executes the closure procedure
	 and throws an error if the close_notify exchange doesn't
	 succeed.

	 @exception IOException if there is a problem with the close_notify
     */
     public void close()
       throws java.io.IOException {
       conn.close();
     }

     /**
	Get the input stream associated with this socket.
	Data read from this input stream is automatically SSL
	decrypted

     */
     public InputStream getInputStream(){
       return conn.getInStream();
     }
     
     /**
	Get the input stream associated with this socket.
	Data read from this input stream is automatically SSL
	encrypted
     */
     public OutputStream getOutputStream(){
       return conn.getOutStream();
     }

     /** Converts this socket to a string

	 @return a string description of this socket
     */
     public String toString(){
       return "SSL: " + super.toString();
     }

     // SSL extensions
     /** Get the cipherSuite in use on this socket, as an
	 integer

	 @return the ciphersuite in use
     */	 
     public int getCipherSuite()
       throws IOException {
       return conn.getCipherSuite();
     }

     /** get the certificate chain presented by the peer. This is relevant
	 for clients and servers if Client Authentication is being used.

	 @return the certificate chain as a Vector of X509Certs, null if unavailable
     */
     
     public Vector getCertificateChain()
       throws IOException {
       return conn.getCertificateChain();
     }


     /** Get the policy associated with this socket

	 @return the policy
     */
     public SSLPolicyInt getPolicy(){
       return conn.getPolicy();
     }

     /** Get the version of SSL negotiated.

	 @return 768 (0x300) for SSLv3 or 769 (0x301) for TLSv1
     */
     public int getVersion()
       throws IOException {
       return conn.getVersion();
     }

     /** Renegotiate the SSL connection using the given policy
	 <P>
         This is useful (for instance) for a server to renegotiate
         using client authentication

	 @param policy the policy to use
     */
     public void renegotiate(SSLPolicyInt policy)
       throws IOException {
       conn.renegotiate(policy);
     }

     /** Renegotiate the SSL connection using the same policy
	 <P>
	 This is mainly useful when a client is responding to a server's
	 request for renegotiation
     */
     public void renegotiate()
       throws IOException {
       conn.renegotiate(conn.getPolicy());
     }
       
     /** Send our half of the SSL close_notify handshake

	 @exception IOException if the close_notify alert can't be sent
     */
     public void sendClose()
       throws java.io.IOException {
       conn.sendClose();
     }

     /** Wait to receive a close_notify from the other side.

	 @param enforceFinished insist that no more data be present on the connection before the close_notify is received. This ensures that the application has read all the data that the peer sent

	 @exception IOException if the close_notify couldn't be read or if enforceFinished is true and more data was present.
     */
     public void waitForClose(boolean enforceFinished)
       throws java.io.IOException {
       conn.recvClose(enforceFinished);
     }

     /** Don't call this */
     private static void testConn(SSLContext c,String host,String port)
       throws IOException, java.net.UnknownHostException {
       try {
	 SSLSocket s;
	 s=new SSLSocket(c,host,Integer.parseInt(port));

	 Vector cc=s.getCertificateChain();

	 if(cc!=null){
	   System.out.println("Cert chain");

	 
	   for(int i=0;i<cc.size();i++){
	     X509Cert cert=(X509Cert)cc.elementAt(i);

	     System.out.println("Issuer "+cert.getIssuerName().getNameString());
	     System.out.println("Subject "+cert.getSubjectName().getNameString());
	     System.out.println("Serial "+cert.getSerial());
	     System.out.println("Validity "+cert.getValidityNotBefore() +"-"+
	       cert.getValidityNotAfter());

	   }
	 }	   
	 byte buf[]=new byte[4096];

	 InputStreamReader ir=new InputStreamReader(s.getInputStream());
	 BufferedReader br=new BufferedReader(ir);
	 
	 OutputStreamWriter or=new OutputStreamWriter(s.getOutputStream());
	 BufferedWriter bw=new BufferedWriter(or);
	 
	 bw.write("Test string",0,11);
	 bw.newLine();
	 bw.newLine();	 
	 bw.flush();

	 for(;;){
	   String str=br.readLine();

	   if(str==null)
	     break;
	   System.out.println(str);
	 }
	 
	 s.close();
       }       
       catch (SSLAlertException e){
	 throw new Error(e.toString());
       }
     }

     /** Test code */
     public static void main(String args[])
	  throws java.net.UnknownHostException, java.io.IOException {
       String host="localhost";
       String port="4433";
       SSLSocket s;

       if(args.length==2){
	 host=args[0];
	 port=args[1];
       }

       SSLContext c=new SSLContext();
       c.loadRootCertificates("root.b64");
       c.loadEAYKeyFile("bookdsa.pem",
	 "password");
       System.out.println("Trying 1");
       testConn(c,host,port);
       System.out.println("Trying 2");       
       testConn(c,host,port);
     }
}



     
       
       


     
       
       
