#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <time.h>
#include <assert.h>
#include <syslog.h>
#include <pwd.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <netinet/in.h>

#include "prngd.h"

static int randsavefd = -1;
int debug = 0;

int devnull = -1;

int limit_openfd = 0;
int count_openfd = 0;

static void write_seedfile(void);

static void open_seedfile(const char *seedfile)
{
  int bytes, bytes_read_total;
  unsigned char buffer[PRNGD_STATE_SIZE];
  char msg_buf[1024];

  randsavefd = open(seedfile, O_RDWR | O_CREAT | O_SYNC | O_RSYNC, 400);
  if (randsavefd < 0)
  {
    (void)snprintf(msg_buf, sizeof(msg_buf),
	"could not open %s: %s\n", seedfile, strerror(errno));
    msg_buf[sizeof(msg_buf) - 1] = '\0';
    error_fatal(msg_buf, 0);
    exit(EXIT_FAILURE);
  }
  close_on_exec(randsavefd, 1);

  /*
   * The contents of the seedfile may be sensitive, since we start seeding
   * from the file.
   */
  if (fchmod(randsavefd, S_IRUSR | S_IWUSR) < 0)
  {
    (void)snprintf(msg_buf, sizeof(msg_buf),
	"Could not chmod %s: %s\n", seedfile, strerror(errno));
    msg_buf[sizeof(msg_buf) - 1] = '\0';
    error_fatal(msg_buf, 0);
    exit(EXIT_FAILURE);
  }

  /*
   * Do some initial seeding stuff to pre-stir the PRNG.
   */
  seed_internal(NULL);

  /*
   * Now read as much from the file as we can get. 
   */
  bytes_read_total = 0;
  while ((bytes = read(randsavefd, buffer, PRNGD_STATE_SIZE)) > 0)
  {
    rand_add(buffer, bytes, 0.0);
    bytes_read_total += bytes;
  }
  if (bytes < 0)
  {
    (void)snprintf(msg_buf, sizeof(msg_buf),
	"Error reading %s: %s\n", seedfile, strerror(errno));
    msg_buf[sizeof(msg_buf) - 1] = '\0';
    error_fatal(msg_buf, 0);
    exit(EXIT_FAILURE);
  }

  /*
   * Done, let's stir around some more
   */
  seed_internal(NULL);

  if (debug)
    (void)fprintf(stderr, "Read %d bytes\n", bytes_read_total);
  /*
   * Finally, account for the bytes read. We don't know how much entropy
   * was in the bytes we've read, so we only account for the minimum needed
   * and have PRNGD do the rest. If not enough bytes are read, only account
   * the bytes. Don't care for the contents of the buffer, it doesn't hurt
   * to stir the pool.
   */
  if (bytes_read_total > ENTROPY_NEEDED)
    rand_add(buffer, PRNGD_STATE_SIZE, ENTROPY_NEEDED);
  else
    rand_add(buffer, PRNGD_STATE_SIZE, bytes_read_total);

  /*
   * Ok, we don't want to reuse the same seed again, so we will immediately
   * replace the contents of the seed-file.
   */
  write_seedfile();
}

static void write_seedfile(void)
{
  int bytes;
  unsigned char buffer[PRNGD_STATE_SIZE];

  bytes = lseek(randsavefd, 0, SEEK_SET);
  if (bytes < 0)
  {
    error_fatal("Error writing seedfile", errno);
    exit(EXIT_FAILURE);
  }
  bytes = rand_bytes(buffer, PRNGD_STATE_SIZE);
  if (bytes == 0)
  {
    error_error("Info: Random pool not (yet) seeded", 0);
  }
  else
  {
    bytes = write(randsavefd, buffer, PRNGD_STATE_SIZE);
    if (bytes < 0)
    {
      error_fatal("Error writing seedfile", errno);
      exit(EXIT_FAILURE);
    }
  }
  if (debug)
    (void)fprintf(stderr, "Wrote %d bytes back to seedfile\n", bytes);
  bytes = ftruncate(randsavefd, bytes);
}

void close_seedfile(void)
{
  if (randsavefd >= 0) {
    write_seedfile();
    (void)close(randsavefd);
  }
}


static void killpeer(int service_socket)
{
  unsigned char buffer[256];
  char msg_buf[1024];
  int num;
  pid_t peer_id;

  buffer[0] = 0x04;
  if (write(service_socket, buffer, 1) < 1)
  {
    error_fatal("Cannot contact peer", errno);
    exit(EXIT_FAILURE);
  }
  if (read(service_socket, buffer, 1) < 1)
  {
    error_fatal("Cannot read from peer", errno);
    exit(EXIT_FAILURE);
  }
  num = buffer[0];
  if (read(service_socket, buffer, num) < num)
  {
    error_fatal("Cannot read from peer", errno);
    exit(EXIT_FAILURE);
  }
  buffer[num] = '\0';
#ifdef PID_T_IS_LONG_INT
  peer_id = atol((char *) buffer);
  (void)fprintf(stderr, "Killing: %ld\n", peer_id);
#else
  peer_id = atoi((char *) buffer);
  (void)fprintf(stderr, "Killing: %d\n", (int)peer_id);
#endif

  if (kill(peer_id, SIGTERM) < 0)
  {
#ifdef PID_T_IS_LONG_INT
    (void)snprintf(msg_buf, sizeof(msg_buf), "Cannot kill %ld", peer_id);
    msg_buf[sizeof(msg_buf) - 1] = '\0';
    error_fatal(msg_buf, errno);
#else
    (void)snprintf(msg_buf, sizeof(msg_buf), "Cannot kill %d", (int)peer_id);
    msg_buf[sizeof(msg_buf) - 1] = '\0';
    error_fatal(msg_buf, errno);
#endif
    exit(EXIT_FAILURE);
  }
  exit(EXIT_SUCCESS);
}

static void usage(const char *progname)
{
  (void)fprintf(stderr, "prngd %s (%s)\n", PRNGD_VERSION, PRNGD_DATE);
  (void)fprintf(stderr,
	  "Usage: %s [options] /path/to/socket1 [/path/to/socket2 ...]\nOptions:\n\t-d/--debug: debugging on\n\t-c/--cmdfile cmdpath: use cmdpath for entropy commands [%s]\n\t-s/--seedfile seedpath: use seedpath as seedfile [%s]\n\t-n/--no-seedfile: no seedfile, keep pool in memory only\n\t-m/--mode mode: use mode for sockets [%04o]\n\t-k/--kill: kill daemon on other side\n\t-v/--version: print version and exit\n",
	  progname, CONFIGFILE, RANDSAVENAME, SOCKETMODE);
}

int main(int argc, char *argv[])
{
  int argp;
  int ret;
  int statval;
  int killmode = 0;
  time_t seed_stat_interval = SEED_STAT_INTERVAL;
  time_t seed_ext_interval = SEED_EXT_INTERVAL;
  int i, numsock = 0;
  char **randsock = NULL;
  char *cmdpath = CONFIGFILE;
  char *seedpath = RANDSAVENAME;
  char msg_buf[1024];
  unsigned int socketmode = SOCKETMODE;
  int *service_socket, kill_socket;
  struct sockaddr_un sockun, sockun_kill;
  struct stat randsock_stat;
  struct passwd *pwent;
  char *username = "unknown";
  entropy_source_t *entropy_source;

  /*
   * Process arguments 
   */
  argp = 1;
  while (argp < argc)
  {
    if (!strcmp(argv[argp], "-v") || !strcmp(argv[argp], "--version")) {
      (void)fprintf(stdout, "prngd %s (%s)\n", PRNGD_VERSION, PRNGD_DATE);
      exit(EXIT_SUCCESS);
    }
    else if (!strcmp(argv[argp], "-d") || !strcmp(argv[argp], "--debug"))
      debug = 1;
    else if ((!strcmp(argv[argp], "-c") || !strcmp(argv[argp], "--cmdfile"))
	     && (argp + 1 < argc))
      cmdpath = argv[++argp];
    else if ((!strcmp(argv[argp], "-s") || !strcmp(argv[argp], "--seedfile"))
	     && (argp + 1 < argc))
      seedpath = argv[++argp];
    else if ((!strcmp(argv[argp], "-m") || !strcmp(argv[argp], "--mode"))
	     && (argp + 1 < argc))
    {
      if (sscanf(argv[++argp], "%o", &socketmode) == 0)
      {
	(void)snprintf(msg_buf, sizeof(msg_buf),
		"Cannot interpret socket mode '%s'\n", argv[argp]);
	msg_buf[sizeof(msg_buf) - 1] = '\0';
	error_fatal(msg_buf, 0);
	exit(EXIT_FAILURE);
      }
    }
    else if (!strcmp(argv[argp], "-n") || !strcmp(argv[argp], "--no-seedfile"))
      seedpath = NULL;
    else if (!strcmp(argv[argp], "-k") || !strcmp(argv[argp], "--kill"))
      killmode = 1;
    else if (argv[argp][0] != '-') {
      if (numsock == 0)
        randsock = malloc(sizeof(char *));
      else
	randsock = realloc(randsock, (numsock + 1) * sizeof(char *));
      if (randsock == NULL) {
	error_fatal("Could not allocate memory", errno);
	exit(EXIT_FAILURE);
      }
      randsock[numsock++] = argv[argp];
    }
    else
    {
      usage(argv[0]);
      exit(EXIT_FAILURE);
    }
    argp++;
  }
  if (!numsock) {
    usage(argv[0]);
    exit(EXIT_FAILURE);
  }

  if (!killmode)
    parse_configfile(cmdpath, &entropy_source);

  if (!numsock)
  {
    error_fatal("no socket name given, exiting!", 0);
    exit(EXIT_FAILURE);
  }
  if (debug)
    (void)fprintf(stdout, "Debugging enabled\n");

  /*
   * Obtain the maximum number of open filedescriptors and adjust the
   * counter of actual open files. As always we have open stdin, stdout,
   * and stderr.
   */
  limit_openfd = obtain_limit();
  count_openfd = 3;

  /*
   * Open /dev/null
   */
  devnull = open("/dev/null", O_RDWR);
  if (devnull == -1) {
    error_fatal("Couldn't open /dev/null", errno);
    exit(EXIT_FAILURE);
  }
  count_openfd++;

  /*
   * Now, initialize the PRNG (allocate the memory).
   */
  rand_init(PRNGD_STATE_SIZE, ENTROPY_NEEDED);


  /*
   * Start by reading back the seed saved from former runs, if usage of a
   * seedfile is wanted. 
   */
  if (!killmode && seedpath) {
    open_seedfile(seedpath);
    count_openfd++;
  }

  if (!(service_socket = malloc(numsock * sizeof(int)))) {
    error_fatal("Could not allocate memory", errno);
    exit(EXIT_FAILURE);
  }

  /*
   * Prepare the socket used for killing or testing for a running
   * daemon.
   */
  kill_socket = socket(AF_UNIX, SOCK_STREAM, 0);
  if (kill_socket < 0) {
    error_fatal( "Could not create socket", errno);
    exit(EXIT_FAILURE);
  }
  count_openfd++;
  /*
   * Set up the sockets to service entropy requests
   */
  for (i = 0; i < numsock; i++) {

    /*
     * Check out what is already available at the place of the future
     * egd-socket. If we cannot reach the place to be with "stat()",
     * do not continue, as something may be seriously wrong. If the
     * entry does already exist, remove() it, but only if it is of type
     * socket! If removal fails for whatever reason, stop, as something
     * may be seriously wrong.
     */
    statval = stat(randsock[i], &randsock_stat);
    if ((statval < 0) && (errno != ENOENT)) {
      (void)snprintf(msg_buf, sizeof(msg_buf), "Cannot stat socket position %s",
      	randsock[i]);
      msg_buf[sizeof(msg_buf) - 1] = '\0';
      error_fatal(msg_buf, errno);
      exit(EXIT_FAILURE);
    }
    else if (!statval) {
      if (!S_ISSOCK(randsock_stat.st_mode)) {
	(void)snprintf(msg_buf, sizeof(msg_buf), "Will not touch %s: no socket",
		randsock[i]);
	msg_buf[sizeof(msg_buf) - 1] = '\0';
	error_fatal(msg_buf, errno);
	exit(EXIT_FAILURE);
      } else {
	/*
	 * Now that we have found a socket, we must check whether there
	 * is a daemon listining on it. When in kill mode, this is great
	 * and we are going to kill it. When not in kill mode, we cannot
	 * start a new daemon at this address and exit with error.
	 */
	(void)memset(&sockun, 0, sizeof(sockun_kill));
	sockun_kill.sun_family = AF_UNIX;
	if (sizeof(sockun_kill.sun_path) < strlen(randsock[i]) + 1) {
#ifdef SIZEOF_IS_LONG_INT
	(void)snprintf(msg_buf, sizeof(msg_buf),
		"Sockfilename too long: %s, maxlen=%ld\n", randsock[i],
		sizeof(sockun_kill.sun_path));
	msg_buf[sizeof(msg_buf) - 1] = '\0';
	error_error(msg_buf);
#else
	(void)snprintf(msg_buf, sizeof(msg_buf),
		"Sockfilename too long: %s, maxlen=%d\n", randsock[i],
		sizeof(sockun_kill.sun_path));
	msg_buf[sizeof(msg_buf) - 1] = '\0';
	error_error(msg_buf, 0);
#endif /* SIZEOF_IS_LONG_INT */
	  exit(EXIT_FAILURE);
        }

	(void)strcpy(sockun_kill.sun_path, randsock[i]);
	if (connect(kill_socket, (struct sockaddr *) &sockun_kill,
		    sizeof(sockun_kill)) < 0) {
	  if (killmode) {
	    (void)snprintf(msg_buf, sizeof(msg_buf),
	    	"cannot kill peer at %s: no daemon", randsock[i]);
	    msg_buf[sizeof(msg_buf) - 1] = '\0';
	    error_error(msg_buf, 0);
	    /*
	     * Do no exit because there may be more work to do!
	     */
	    continue;
	  }
	}
	else if (killmode) {
	  killpeer(kill_socket);
	  my_shutdown(kill_socket, SHUT_RDWR);
	  count_openfd--;
	  continue;
	}
	else {
	  (void)snprintf(msg_buf, sizeof(msg_buf),
		  "socket %s already used by another process, exiting.\n",
		  randsock[i]);
	  msg_buf[sizeof(msg_buf) - 1] = '\0';
	  error_fatal(msg_buf, 0);
	  exit(EXIT_FAILURE);
	}
	my_shutdown(kill_socket, SHUT_RDWR);
	count_openfd--;

	/*
         * Now there is a socket without service listening on it: we can
	 * remove it.
	 */
	if (remove(randsock[i]) < 0) {
	  (void)snprintf(msg_buf, sizeof(msg_buf),
	  	"Cannot remove already existing entry %s", randsock[i]);
	  msg_buf[sizeof(msg_buf) - 1] = '\0';
	  error_fatal(msg_buf, errno);
	  exit(EXIT_FAILURE);
	}
      }
    }

    /*
     * We can never reach this point in killmode, as for whatever condition
     * not forcing immediate abort the loop was "continue"ed before.
     */
    assert(!killmode);

    service_socket[i] = socket(AF_UNIX, SOCK_STREAM, 0);
    if (service_socket[i] < 0)
    {
      error_fatal("Could not create socket", errno);
      exit(EXIT_FAILURE);
    }
    /*
     * Bind the socket to the actual socket-file
     */
    (void)memset(&sockun, 0, sizeof(sockun));
    sockun.sun_family = AF_UNIX;
    (void)strcpy(sockun.sun_path, randsock[i]);
    if (bind(service_socket[i], (struct sockaddr *) &sockun,
	     sizeof(sockun)) < 0)
    {
      (void)snprintf(msg_buf, sizeof(msg_buf), "Could not bind socket to %s",
      	randsock[i]);
      msg_buf[sizeof(msg_buf) - 1] = '\0';
      error_fatal(msg_buf, errno);
      exit(EXIT_FAILURE);
    }
    if (chmod(randsock[i], socketmode) < 0)
    {
      (void)snprintf(msg_buf, sizeof(msg_buf), "Could not chmod socket to %04o",
      	socketmode);
      msg_buf[sizeof(msg_buf) - 1] = '\0';
      error_fatal(msg_buf, errno);
      exit(EXIT_FAILURE);
    }
    if (listen(service_socket[i], SOMAXCONN) < 0)
    {
      error_fatal("Could not listen on socket", errno);
      exit(EXIT_FAILURE);
    }
    /*
     * Switch the service_socket to non-blocking mode (otherwise it might
     * hang inside accept() if a connection is closed between select() and
     * accept().
     */
    non_blocking(service_socket[i], 1);
    close_on_exec(service_socket[i], 1);
    count_openfd++;
  }

  if (killmode) {
    /*
     * Everything has been done!
     */
    exit(EXIT_SUCCESS);
  }


  /*
   * If not in debugging mode, make myself a daemon 
   */
  if (!debug)
  {
    ret = daemon(0, 0);
    if (ret < 0)
    {
      error_fatal("Could not daemonize", errno);
      exit(EXIT_FAILURE);
    }

    if ((pwent = getpwuid(getuid())) != NULL)
      username = pwent->pw_name;
    /*
     * The setup is finished, and we are daemonized, so we must now switch
     * to syslog...
     */
    (void)snprintf(msg_buf, sizeof(msg_buf),
    	"prngd %s (%s) started up for user %s", PRNGD_VERSION, PRNGD_DATE,
	username);
    msg_buf[sizeof(msg_buf) - 1] = '\0';
    error_notice(msg_buf, 0);
    count_openfd++;
    (void)snprintf(msg_buf, sizeof(msg_buf),
    	"have %d out of %d filedescriptors open", count_openfd, limit_openfd);
    msg_buf[sizeof(msg_buf) - 1] = '\0';
    error_notice(msg_buf, 0);
  }

  /*
   * Maybe we have changed our pid, let's add some more of it. 
   */
  seed_internal(NULL);

  /* Never returns */
  main_loop(service_socket, numsock, sockun, seed_stat_interval,
	    seed_ext_interval, entropy_source, MAX_GATHERER_BYTES);

  /*
   * We never get here, but want to make the compiler happy...
   */
  return(EXIT_SUCCESS);
}
