/* %W% (QUALCOMM) %G% */

/* Header for SOBER:  Seventeen Octet Byte Enabled Register stream cipher. */
/*
This software is free for commercial and non-commercial use in
non-embedded applications subject to the following conditions.

Copyright remains vested in QUALCOMM Incorporated, and Copyright
notices in the code are not to be removed.  If this package is used in
a product, QUALCOMM should be given attribution as the author of the
SOBER encryption algorithm. This can be in the form of a textual
message at program startup or in documentation (online or textual)
provided with the package.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

1. Redistributions of source code must retain the copyright notice,
   this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the
   distribution.

3. All advertising materials mentioning features or use of this
   software must display the following acknowledgement:  This product
   includes software developed by QUALCOMM Incorporated.

4. The software is not embedded in a device intended for a specific
   application, for example but not limited to communication devices
   (including mobile telephones), smart cards, cash registers, teller
   machines, network routers.

THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.

The license and distribution terms for any publically available version
or derivative of this code cannot be changed, that is, this code cannot
simply be copied and put under another distribution license including
the GNU Public License.
*/

#define N 17
#define WORD unsigned char

unsigned char Initial[N]; /* Initial contents of the shift register */
unsigned char R[2*N];     /* Working storage for the shift register */
unsigned char stcnt;      /* stutter control -- when == 0 next output stored */
unsigned char stctrl;     /* used 2 bits at a time */
unsigned char konst;      /* key dependent "constant" for nonlinear function */
int r;                    /* current offset in sliding window buffer */

void t8_key(unsigned char key[], int keylen);
void t8_genbytes(unsigned long frame, unsigned char buf[], int nbytes);

/* multiplication tables */
#include "multab.h"
#include "sbox.h"

/*
 * FOLD is how many register cycles need to be performed after combining the
 * last byte of key and non-linear feedback, before every byte depends on every
 * byte of the key. This depends on the feedback and nonlinear functions, and
 * on where they are combined into the register.
 */
#define FOLD 17
#define FOLDP 4
#define KEYP 15

/* end of SOBER header */

/* cycle the contents of the shift register */

static int cycle(unsigned char *R, int r)
{
    R[r-N] = R[r] = MUL0x63(R[r-N]) ^ R[r-N+4] ^ MUL0xCE(R[r-N+15]);
    if (++r == 2*N)
        r = N;
    return r;
}

/* Return a non-linear function of some parts of the register.
 * The positions of the state bytes form a maximal span full positive
 * difference set, and are 0, 1, 6, 13, 16.
 */

static unsigned char
nltap(unsigned char *R, int r)
{
    unsigned char   t;

    t = R[r-N] + R[r-N+16];
    t = SBox[t];
    t = (t + R[r-N+1] + R[r-N+6]) ^ konst;
    return (unsigned char) (t + R[r-N+13]);
}

/* add a byte of key at the right places */

void addkey(unsigned char k)
{
    if (r-2*N+KEYP < 0)
	R[r-N+KEYP] += k;
    else
	R[r-2*N+KEYP] = R[r-N+KEYP] = R[r-N+KEYP] + k;
}

/* Fold the nonlinear function back in at the right places */

void xornltap(void)
{
    if (r-2*N+FOLDP < 0)
	R[r-N+FOLDP] ^= nltap(R, r);
    else
	R[r-2*N+FOLDP] = R[r-N+FOLDP] = R[r-N+FOLDP] ^ nltap(R, r);
}

/* load some key material into the register */

static void
loadkey(unsigned char key[], int keylen)
{
    int i;

    for (i = 0; i < keylen; ++i)
    {
	addkey(key[i]);
        r = cycle(R, r);
	xornltap();
    }

    /* also fold in the length of the key */
    addkey(keylen);

    /* now diffuse */
    for (i = 0; i < FOLD; ++i)
    {
	r = cycle(R, r);
	xornltap();
    }
}

/* calculate initial contents of the shift register */

void
t8_key(unsigned char key[], int keylen)
{
    int i;

    /* fill the register with fibonacci numbers */

    R[0] = R[1] = 1;
    for (i = 2; i < N; i++)
        R[i] = R[i-1] + R[i-2];

    /* initialise the pointers and start folding in key */

    r = N;
    konst = 0;
    loadkey(key, keylen);

    /* save state and key word for nonlinear function */
    r = cycle(R, r);
    konst = nltap(R, r);
    for (i = 0; i < N; i++)
        Initial[i] = R[r-N+i];
    stcnt = 0;
}

/* Fold in the per-frame key */

void
t8_seckey(unsigned char seckey[], int seckeylength)
{
    register int    i;

    /* copy initial contents */

    for (i = 0; i < N; i++)
        R[i] = Initial[i];
    
    r = N;
    loadkey(seckey, seckeylength);
    stcnt = 0;
}

/* XOR pseudo-random bytes into buffer */

void
t8_gen(unsigned char *buf, int nbytes)
{
    unsigned char   *endbuf;

    endbuf = &buf[nbytes];
    while (buf != endbuf)
    {
        stctrl >>= 2;

        /* reprime stuttering if necessary */

        if (stcnt == 0)
        {
            stcnt = 4;
            r = cycle(R, r);
            stctrl = nltap(R, r);
        }
        stcnt--;

        r = cycle(R, r);
        switch (stctrl & 0x3) {

        case 0: /* just waste a cycle and loop */
            break;

        case 1: /* use the first output from two cycles */
            *buf++ ^= nltap(R, r) ^ 0x69;
            r = cycle(R, r);
            break;

        case 2: /* use the second output from two cycles */
            r = cycle(R, r);
            *buf++ ^= nltap(R, r);
            break;

        case 3: /* return from one cycle */
            *buf++ ^= nltap(R, r) ^ 0x96;
            break;

        }
    }
}

/* encrypt/decrypt a frame of data */

void
t8_genbytes(unsigned long frame, unsigned char *buf, int nbytes)
{
    unsigned char   framebuf[4];

    framebuf[0] = (frame >> 24) & 0xFF;
    framebuf[1] = (frame >> 16) & 0xFF;
    framebuf[2] = (frame >>  8) & 0xFF;
    framebuf[3] = (frame) & 0xFF;
    t8_seckey(framebuf, 4);
    t8_gen(buf, nbytes);
}

#define TEST 1
#ifdef TEST
#include <stdio.h>
#include <string.h>
#include "hexlib.h"

/* test vectors */
typedef unsigned char   uchar;
typedef unsigned long   word32;
uchar   *testkey = (uchar *)"test key";
word32  testframe = 1L;
uchar   testbuf[16];
char    *testout = "1e cd 58 17 f3 86 b3 04 04 03 91 2c a0 d5 24 00";
uchar   bigbuf[1024*1024];

void
test_sober(void)
{
    t8_key(testkey, strlen((char *)testkey));
    t8_genbytes(testframe, testbuf, sizeof testbuf);
    hexprint("testbuf", testbuf, sizeof testbuf);
    hexcheck(testbuf, testout, sizeof testbuf);
}

void
time_sober(void)
{
    word32      i;

    t8_key(testkey, strlen((char *)testkey));
    for (i = 0; i < 10; ++i)
        t8_genbytes(i, bigbuf, sizeof bigbuf);
}

int
main(int ac, char **av)
{
    int         n;
    uchar       key[16];
    int         keysz;
    word32      hook;

    if (ac == 2 && strcmp(av[1], "-test") == 0) {
        test_sober();
        return nerrors;
    }
    if (ac == 2 && strcmp(av[1], "-time") == 0) {
        time_sober();
        return 0;
    }

    if (ac >= 2)
        hexread(key, av[1], keysz = strlen(av[1]) / 2);
    else
        hexread(key, "0000000000000000", keysz = 8);
    sscanf(ac >= 3 ? av[2] : "00000000", "%lx", &hook);
    sscanf(ac >= 4 ? av[3] : "10000", "%d", &n);

    t8_key(key, keysz);
    t8_genbytes(hook, bigbuf, n);
    hexbulk(bigbuf, n);
    return 0;
}
#endif
