/******************************************************************************
 *
 * Copyright (c) 1999-2001 AppGate AB. All Rights Reserved.
 * 
 * This file contains Original Code and/or Modifications of Original Code as
 * defined in and that are subject to the MindTerm Public Source License,
 * Version 1.1, (the 'License'). You may not use this file except in compliance
 * with the License.
 * 
 * You should have received a copy of the MindTerm Public Source License
 * along with this software; see the file LICENSE.  If not, write to
 * AppGate AB, Stora Badhusgatan 18-20, 41121 Goteborg, SWEDEN
 *
 *****************************************************************************/

package com.mindbright.ssh;

import java.awt.*;
import java.awt.event.*;
import java.io.File;
import java.io.OutputStream;
import java.io.IOException;

import com.mindbright.gui.ProgressBar;
import com.mindbright.gui.AWTConvenience;
import com.mindbright.gui.AWTGridBagContainer;

import com.mindbright.sshcommon.SSHSCP1;
import com.mindbright.sshcommon.SSHSCPProgress;

import com.mindbright.ssh2.SSH2SCP1Client;

public final class SSHSCPGUIThread extends Thread implements SSHSCPProgress {
    String           curDir, localFile, remoteFile;
    String           remoteHost;
    int              remotePort;
    boolean          recursive, background, toRemote;
    Frame            parent;
    SSHSCPDialog     browseDialog;

    SSHInteractiveClient client;

    String[]    localFileList;

    Dialog         copyIndicator;
    ProgressBar    progress;
    SSHSCP1        scp1;
    Thread         copyThread;
    Label          srcLbl, dstLbl, sizeLbl, nameLbl, speedLbl;
    Button         cancB;
    long           startTime;
    long           lastTime;
    long           totTransSize;
    long           fileTransSize;
    long           curFileSize;
    long           lastSize;
    int            fileCnt;
    boolean        doneCopying;

    public SSHSCPGUIThread(SSHInteractiveClient client,
			   Frame parent,
			   String curDir, String localFile, String remoteFile,
			   boolean recursive, boolean background,
			   boolean toRemote, SSHSCPDialog browseDialog)
	throws Exception
    {
	try {
	    netscape.security.PrivilegeManager.enablePrivilege("UniversalFileAccess");
	} catch (netscape.security.ForbiddenTargetException e) {
	    // !!!
	}
    
	localFileList = spaceSplit(localFile);

	if(localFileList == null) {
	    throw new Exception("Unbalanced quotes in local files list");
	}

	File lf = new File(localFileList[0]);
	if(lf.isAbsolute()) {
	    curDir = lf.getParent();
	    if(curDir == null)
		curDir = lf.getAbsolutePath();
	}

	localFileList = starExpand(localFileList, curDir);

	if(localFileList.length > 1 && !toRemote) {
	    throw new Exception("Ambiguos local target");
	}

	if(!toRemote) {
	    localFile = localFileList[0];
	}

	this.client        = client;
	this.remoteHost    = client.propsHandler.getSrvHost();
	this.remotePort    = client.propsHandler.getSrvPort();
	this.curDir        = curDir;
	this.parent        = parent;
	this.localFile     = localFile;
	this.remoteFile    = remoteFile;
	this.recursive     = recursive;
	this.background    = background;
	this.toRemote      = toRemote;
	this.fileCnt       = 0;
	this.doneCopying   = false;
	this.startTime     = 0;
	this.lastTime      = 0;
	this.totTransSize  = 0;
	this.fileTransSize = 0;
	this.lastSize      = 0;
	this.browseDialog  = browseDialog;
	this.start();
    }

    public void run() {
	String sourceFile = "localhost:" + unQuote(localFile);
	String destFile   = remoteHost + ":" + unQuote(remoteFile);

	if(!toRemote) {
	    String tmp;
	    tmp        = sourceFile;
	    sourceFile = destFile;
	    destFile   = tmp;
	}

	copyIndicator = new Dialog(parent, "MindTerm - File Transfer", false);
	      
	Label               lbl;
	Button              b;
	AWTGridBagContainer grid  = new AWTGridBagContainer(copyIndicator);

	lbl = new Label("Source:");
	grid.add(lbl, 0, 1);
	  
	srcLbl = new Label(cutName(sourceFile, 32));
	grid.add(srcLbl, 0, 4);

	lbl = new Label("Destination:");
	grid.add(lbl, 1, 1);
	  
	dstLbl = new Label(cutName(destFile, 32));
	grid.add(dstLbl, 1, 4);

	lbl= new Label("Current:");
	grid.add(lbl, 2, 1);

	nameLbl= new Label("connecting...");
	grid.add(nameLbl, 2, 3);

	sizeLbl= new Label("");
	grid.add(sizeLbl, 2, 1);

	grid.getConstraints().fill   = GridBagConstraints.NONE;
	grid.getConstraints().anchor = GridBagConstraints.CENTER;
	grid.getConstraints().insets = new Insets(4, 12, 4, 4);

	progress = new ProgressBar(512, 160, 20);
	grid.add(progress, 3, 3);

	grid.getConstraints().fill   = GridBagConstraints.HORIZONTAL;
	grid.getConstraints().insets = new Insets(4, 4, 4, 4);

	speedLbl = new Label("0.0 kB/sec", Label.CENTER);
	grid.add(speedLbl, 3, GridBagConstraints.REMAINDER);

	cancB = new Button("Cancel");
	cancB.addActionListener(new ActionListener() {
	    public void actionPerformed(ActionEvent e) {
		if(!doneCopying) {
		    if(copyThread != null)
			copyThread.stop();
		    if(scp1 != null)
			scp1.abort();
		}
		copyIndicator.setVisible(false);
	    }
	});

	grid.getConstraints().fill  = GridBagConstraints.NONE;
	grid.getConstraints().ipadx = 2;
	grid.getConstraints().ipady = 2;

	grid.add(cancB, 4, GridBagConstraints.REMAINDER);

	AWTConvenience.setBackgroundOfChildren(copyIndicator);

	Dimension d = speedLbl.getSize();
	d.width += d.width * 2;
	speedLbl.setSize(d);
	sizeLbl.setSize(d);

	copyIndicator.setResizable(true);
	copyIndicator.pack();
	AWTConvenience.placeDialog(copyIndicator);

	copyThread = new Thread(new Runnable() {
	    public void run() {
		try {
		    netscape.security.PrivilegeManager.enablePrivilege("UniversalFileAccess");
		    netscape.security.PrivilegeManager.enablePrivilege("TerminalEmulator");
		} catch (netscape.security.ForbiddenTargetException e) {
		    // !!!
		}

		try {
		    SSHInteractor interactAdapter = new SSHInteractorAdapter() {
			    public void open(SSHClient client) {
				nameLbl.setText("...connected");
			    }
			    public void disconnected(SSHClient client,
						     boolean graceful) {
				scp1.abort();
			    }
			    public void alert(String msg) {
				client.alert(msg);
			    }
			};

		    if(client.isSSH2) {
			OutputStream alertOutput = new OutputStream() {
				public void write(int bb) throws IOException {
				    byte[] buf = new byte[] { (byte)bb };
				    write(buf);
				}
				public void write(byte bb[], int off, int len)
				    throws IOException {
				    client.alert("Remote warning/error: " +
						 new String(bb, off, len));
				}
			    };
			SSH2SCP1Client scpClient2 =
			    new SSH2SCP1Client(new File(curDir),
					       client.connection,
					       alertOutput,
					       recursive, SSH.DEBUG);
			scp1 = scpClient2.scp1();
		    } else {
			SSHSCPClient scpClient =
			    new SSHSCPClient(remoteHost, remotePort,
					     client.propsHandler,
					     interactAdapter,
					     new File(curDir),
					     recursive, SSH.DEBUG);
			scpClient.setClientUser(client.propsHandler);
			scp1 = scpClient.scp1();
		    }

		    scp1.setProgress(SSHSCPGUIThread.this);
		    if(toRemote) {
			scp1.copyToRemote(localFileList, remoteFile);
		    } else {
			scp1.copyToLocal(localFile, remoteFile);
		    }

		    copyThread.setPriority(Thread.NORM_PRIORITY);
		    Toolkit.getDefaultToolkit().beep();
		} catch (Exception e) {
		    client.alert("SCP Error: " + e.getMessage());
		    if(SSH.DEBUGMORE) {
			System.out.println("SCP Error:");
			e.printStackTrace();
		    }
		}
		nameLbl.setText("Copied " + fileCnt + " file" + (fileCnt != 1 ? "s" : "") + ".");
		double kSize = (double)totTransSize / 1024;
		sizeLbl.setText(round(kSize) + " kB");
		doneCopying = true;
		cancB.setLabel("Done");

		browseDialog.refresh();

		AWTConvenience.setKeyListenerOfChildren(copyIndicator,
			   new AWTConvenience.OKCancelAdapter(cancB, cancB),
							null);

	    }
	});

	if(background) {
	    copyThread.setPriority(Thread.MIN_PRIORITY);
	}

	copyThread.start();

	copyIndicator.setVisible(true);
    }

    public static String[] spaceSplit(String str) {
	int  l = 0, r, cnt = 0;
	String[] list = new String[str.length() / 2];
	boolean lastIsQuoted = false;
	str = str.trim();
	while((r = str.indexOf(' ', l)) >= 0) {
	    if(str.charAt(l) == '"') {
		l += 1;
		r = str.indexOf('"', l);
		if(r == -1)
		    return null;
	    }
	    String name = str.substring(l, r);
	    if(name.endsWith(File.separator))
		name = name.substring(0, name.length() - 1);
	    list[cnt++] = name;
	    l = r;
	    do {
		l++;
		if(l == str.length()) {
		    lastIsQuoted = true;
		    break;
		}
	    } while(str.charAt(l) == ' ');
	}

	if(!lastIsQuoted) {
	    if(str.charAt(l) == '"') {
		l += 1;
		r = str.indexOf('"', l);
		if(r == -1)
		    return null;
	    }
	    String name = str.substring(l, r);
	    if(name.endsWith(File.separator))
		name = name.substring(0, name.length() - 1);
	    list[cnt++] = name;
	}

	String[] tmp = list;
	list = new String[cnt];
	System.arraycopy(tmp, 0, list, 0, cnt);

	return list;
    }

    public static String[] starExpand(String[] fileList, String curDir) {
	int i, j, n, cnt = 0;
	String[] newList = new String[4096]; // !!! Ouch...
	String[] curDirList = (new File(curDir)).list();
	String path, curFile;

	for(i = 0; i < fileList.length; i++) {
	    curFile = fileList[i];
	    path    = "";
	    n = curFile.indexOf('*');
	    if(n == -1) {
		cnt = addUnique(newList, curFile, cnt);
		continue;
	    }
	    String[] dirList;
	    File f = new File(curFile);
	    if(!f.isAbsolute()) {
		dirList = curDirList;
	    } else {
		String dir = f.getParent();
		if(dir == null)
		    dir = new String(File.separator); // !!! Ouch...
		dirList = (new File(dir)).list();
		curFile = f.getName();
		path    = dir + File.separator;
		n = curFile.indexOf('*');
	    }

	    String pre  = curFile.substring(0, n);
	    String post = curFile.substring(n + 1);
	    for(j = 0; j < dirList.length; j++) {
		String name = dirList[j];
		if(name.startsWith(pre) && name.endsWith(post)) {
		    cnt = addUnique(newList, path + name, cnt);
		}
	    }
	}
	String[] tmp = newList;
	newList = new String[cnt];
	System.arraycopy(tmp, 0, newList, 0, cnt);
	return newList;
    }

    static int addUnique(String[] list, String str, int last) {
	int i;
	for(i = 0; i < last; i++)
	    if(list[i].equals(str))
		break;
	if(i == last)
	    list[last++] = str;
	return last;
    }

    public void startFile(String file, long size) {
	double kSize = (double)size / 1024;
	sizeLbl.setText(round(kSize) + " kB");
	nameLbl.setText(unQuote(file));
	progress.setMax(size, true);
	lastTime = System.currentTimeMillis();
	if(startTime == 0)
	    startTime = lastTime;
	curFileSize   = size;
	fileTransSize = 0;
	fileCnt++;
    }
    public void startDir(String file) {
	if(startTime == 0)
	    startTime = System.currentTimeMillis();
	if(file.length() > curDir.length())
	    file = file.substring(curDir.length());
	if(toRemote) {
	    srcLbl.setText(cutName("localhost:" + unQuote(file), 32));
	} else {
	    dstLbl.setText(cutName("localhost:" + unQuote(file), 32));
	}
    }
    public void endFile() {
	progress.setValue(curFileSize, true);
    }
    public void endDir() {
    }
    public void progress(int size) {
	totTransSize  += (long)size;
	fileTransSize += (long)size;
	if((curFileSize > 0) &&
	   ((((totTransSize - lastSize) * 100) / curFileSize) >= 1)) {
	    progress.setValue(fileTransSize, !background);
	    long   now      = System.currentTimeMillis();
	    long   elapsed  = (now - startTime);
	    if(elapsed == 0) {
		elapsed = 1;
	    }
	    int curSpeed = (int)((double)totTransSize /
				 ((double)elapsed / 1000));
	    elapsed = (now - lastTime);
	    if(elapsed == 0) {
		elapsed = 1;
	    }
	    curSpeed += (int)((double)(totTransSize - lastSize) /
			      ((double)elapsed / 1000));
	    curSpeed >>>= 1;
	    curSpeed >>>= 10;
	    speedLbl.setText(curSpeed + " kB/sec");
	    lastSize = totTransSize;
	    lastTime = now;
	}
    }
    double round(double val) {
	val = val * 10.0;
	val = Math.floor(val);
	val = val / 10.0;
	return val;
    }

    String cutName(String name, int len) {
	if(name.length() > len) {
	    len -= 3;
	    String pre = name.substring(0, len / 2);
	    String suf = name.substring(name.length() - (len / 2));
	    name = pre + "..." + suf;
	}
	return name;
    }

    String unQuote(String str) {
	if(str.charAt(0) == '"') {
	    str = str.substring(1, str.length() - 1);
	}
	return str;
    }

}
