/******************************************************************************
 *
 * Copyright (c) 1999-2001 AppGate AB. All Rights Reserved.
 * 
 * This file contains Original Code and/or Modifications of Original Code as
 * defined in and that are subject to the MindTerm Public Source License,
 * Version 1.1, (the 'License'). You may not use this file except in compliance
 * with the License.
 * 
 * You should have received a copy of the MindTerm Public Source License
 * along with this software; see the file LICENSE.  If not, write to
 * AppGate AB, Stora Badhusgatan 18-20, 41121 Goteborg, SWEDEN
 *
 *****************************************************************************/

package com.mindbright.ssh2;

import java.io.InputStream;
import java.io.OutputStream;
import java.net.Socket;
import java.math.BigInteger;

import java.io.IOException;

import com.mindbright.jca.security.MessageDigest;
import com.mindbright.jca.security.SecureRandom;
import com.mindbright.jca.security.InvalidKeyException;

import javax.crypto.Cipher;
import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.ShortBufferException;

import com.mindbright.util.SecureRandomAndPad;
import com.mindbright.util.Queue;
import com.mindbright.util.Log;

/*
 * This class implements the transport part of the secure shell version 2 stack.
 */
public final class SSH2Transport {

    private class KeepAliveThread implements Runnable {
	private volatile int     interval;
	private volatile boolean keepRunning;

	protected KeepAliveThread(int interval) {
	    this.interval    = interval;
	    this.keepRunning = true;
	    Thread heartbeat = new Thread(this, "SSH2TransportKeepAlive");
	    heartbeat.setDaemon(true);
	    heartbeat.setPriority(Thread.MIN_PRIORITY);
	    heartbeat.start();
	}

	protected synchronized void setInterval(int interval) {
	    if(interval < 1) {
		stop();
	    } else {
		this.interval = interval;
	    }
	}

	public void run() {
	    while(keepRunning) {
		sendIgnore("heartbeat".getBytes());
		try {
		    Thread.sleep(1000 * interval);
		} catch (InterruptedException e) { /* ignore */ }
	    }
	}

	protected void stop() {
	    keepRunning = false;
	}
    }

    private final static boolean DEBUG_ALL_TX = false;
    private final static boolean DEBUG_ALL_RX = false;

    private boolean weAreAServer;

    private String clientVersion;
    private String serverVersion;

    private SSH2TransportPreferences ourPrefs;
    private SSH2TransportPreferences peerPrefs;

    private SSH2TransportEventHandler eventHandler;

    private SSH2KeyExchanger keyExchanger;
    private SSH2UserAuth     userAuth;
    private SSH2Connection   connection;

    private Log tpLog;

    protected Socket       tpSocket;
    protected InputStream  tpIn;
    protected OutputStream tpOut;

    private Thread transmitter;
    private Thread receiver;
    private Queue  txQueue;

    private SecureRandomAndPad tpRand;

    private KeepAliveThread heartbeat;

    private byte[]           sessionId;
    private volatile boolean keyExchangeInProgress;
    private          boolean keyExchangeOk;
    private          Object  keyExchangeMonitor;
    private SSH2TransportPDU clientKEXINITPkt;
    private SSH2TransportPDU serverKEXINITPkt;

    private int            rxSeqNum;
    private Mac            rxMAC;
    private Cipher         rxCipher;
    private SSH2Compressor rxCompressor;

    private int            txSeqNum;
    private Mac            txMAC;
    private Cipher         txCipher;
    private SSH2Compressor txCompressor;

    private          Object  disconnectMonitor;
    private volatile boolean isConnected;
    private volatile boolean isTxUp;
    private volatile boolean isRxUp;

    // Incompatibility flags (peer's incompatibility of course :-)
    //
    public boolean incompatibleSignature;
    public boolean incompatibleServiceAccept;
    public boolean incompatiblePublicKeyAuth;
    public boolean incompatibleHMACKeyLength;
    public boolean incompatiblePublicKeyUserId;
    public boolean incompatibleChannelOpenFail;
    public boolean incompatibleRijndael;
    public boolean incompatibleCantReKey;


    public SSH2Transport(Socket tpSocket,
			 SSH2TransportPreferences prefs,
			 SSH2TransportEventHandler eventHandler,
			 SecureRandomAndPad rand) {
	this(tpSocket, prefs, eventHandler, rand, new Log(Log.LEVEL_INFO));
    }

    public SSH2Transport(Socket tpSocket,
			 SSH2TransportPreferences prefs,
			 SSH2TransportEventHandler eventHandler,
			 SecureRandomAndPad rand, Log log) {
	this.disconnectMonitor  = new Object();
	this.keyExchangeMonitor = new Object();
	this.isConnected        = false;
	this.isTxUp             = false;
	this.isRxUp             = false;
	this.ourPrefs           = prefs;
	this.eventHandler       = (eventHandler != null ? eventHandler :
				   new SSH2TransportEventAdapter());
	this.tpSocket           = tpSocket;
	this.tpRand             = rand;
	this.tpLog              = log;
	try {
	    tpIn  = tpSocket.getInputStream();
	    tpOut = tpSocket.getOutputStream();
	} catch (IOException e) {
	    // !!! TODO: pathological, fixit!!!
	}
    }

    public SSH2Transport(Socket tpSocket, SSH2TransportPreferences prefs,
			 SecureRandomAndPad rand) {
	this(tpSocket, prefs, null, rand);
    }

    public void boot() throws SSH2Exception {
	synchronized(disconnectMonitor) {
	    if(isConnected) {
		throw new SSH2FatalException("Already booted");
	    }
	    isConnected = true;
	}
	try {
	    negotiateVersion();
	} catch (IOException e) {
	    throw new SSH2FatalException("I/O error in version negotiation", e);
	}

	transmitter = new Thread(new Runnable() {
	    public void run() {
		transportTransmitLoop();
	    }
	}, "SSH2TransportTX");
	txQueue = new Queue();
	transmitter.start();

	// Note we start the receiver AFTER we do startKeyExchange() to avoid
	// race with startKeyExchange() in receiver
	//
	startKeyExchange();

	receiver = new Thread(new Runnable() {
	    public void run() {
		transportReceiveLoop();
	    }
	}, "SSH2TransportRX");
	receiver.start();
    }

    public byte[] getSessionId() {
	byte[] id = sessionId;
	if(!incompatiblePublicKeyUserId) {
	    SSH2DataBuffer buf =
		new SSH2DataBuffer(sessionId.length + 4);
	    buf.writeString(sessionId);
	    id = buf.readRestRaw();
	}
	return id;
    }

    public SSH2TransportPDU getClientKEXINITPDU() {
	return clientKEXINITPkt;
    }

    public SSH2TransportPDU getServerKEXINITPDU() {
	return serverKEXINITPkt;
    }

    public String getClientVersion() {
	return clientVersion;
    }

    public String getServerVersion() {
	return serverVersion;
    }

    public SSH2TransportPreferences getOurPreferences() {
	return ourPrefs;
    }

    public SSH2TransportPreferences getPeerPreferences() {
	return peerPrefs;
    }

    public void setEventHandler(SSH2TransportEventHandler eventHandler) {
	if(eventHandler != null) {
	    this.eventHandler = eventHandler;
	}
    }

    public SSH2TransportEventHandler getEventHandler() {
	return eventHandler;
    }

    public Log getLog() {
	return tpLog;
    }

    public void setLog(Log log) {
	tpLog = log;
    }

    public boolean isServer() {
	return weAreAServer;
    }

    public SecureRandom getSecureRandom() {
	return tpRand;
    }

    public SSH2Compressor getRxCompressor() {
	return rxCompressor;
    }

    public SSH2Compressor getTxCompressor() {
	return txCompressor;
    }

    public void setUserAuth(SSH2UserAuth userAuth, String service) {
	this.userAuth = userAuth;
	SSH2TransportPDU pdu =
	    SSH2TransportPDU.createOutgoingPacket(SSH2.MSG_SERVICE_REQUEST);
	pdu.writeString(service);
	transmit(pdu);
    }

    public void setConnection(SSH2Connection connection) {
	this.connection = connection;
    }

    public void startKeyExchange() throws SSH2Exception {
	this.startKeyExchange(ourPrefs);
    }

    public void startKeyExchange(SSH2TransportPreferences newPrefs)
	throws SSH2Exception
    {
	synchronized(keyExchangeMonitor) {
	    if(!keyExchangeInProgress) {
		if(incompatibleCantReKey && (peerPrefs != null)) {
		    throw new SSH2FatalException("Error, peer '" +
						 (weAreAServer ? clientVersion :
						  serverVersion) +
						 "' doesn't support re-keying");
		}

		ourPrefs              = newPrefs;
		keyExchangeInProgress = true;

		if(incompatibleRijndael) {
		    removeRijndael();
		}

		txQueue.disable();
		txQueue.waitUntilBlocked();

		sendKEXINIT();
	    }
	}
    }

    public boolean waitForKEXComplete() {
	synchronized(keyExchangeMonitor) {
	    if(keyExchangeInProgress) {
		try {
		    keyExchangeMonitor.wait();
		} catch (InterruptedException e) {
		    /* don't care, someone interrupted us on purpose */
		}
	    }
	    return keyExchangeOk;
	}
    }

    public boolean keyExchangeInProgress() {
	return keyExchangeInProgress;
    }

    public boolean isConnected() {
	return isConnected;
    }

    public void sendIgnore(byte[] data) {
	sendIgnore(data, 0, data.length);
    }

    public void sendIgnore(byte[] data, int off, int len) {
	SSH2TransportPDU pdu =
	    SSH2TransportPDU.createOutgoingPacket(SSH2.MSG_IGNORE);
	pdu.writeString(data, off, len);
	transmit(pdu);
    }

    public void sendDebug(boolean alwaysDisp, String message, String language) {
	SSH2TransportPDU pdu =
	    SSH2TransportPDU.createOutgoingPacket(SSH2.MSG_DEBUG);
	pdu.writeBoolean(alwaysDisp);
	pdu.writeString(message);
	pdu.writeString(language);
	transmit(pdu);
    }

    public void enableKeepAlive(int intervalSeconds) {
	if(heartbeat != null) {
	    heartbeat.setInterval(intervalSeconds);
	} else if(intervalSeconds > 0) {
	    heartbeat = new KeepAliveThread(intervalSeconds);
	}
    }

    public void disableKeepAlive() {
	if(heartbeat != null) {
	    heartbeat.stop();
	}
	heartbeat = null;
    }

    private void kexComplete(boolean noError) {
	synchronized(keyExchangeMonitor) {
	    keyExchangeInProgress = false;
	    keyExchangeOk         = noError;
	    keyExchangeMonitor.notifyAll();
	    if(noError) {
		eventHandler.kexComplete(this);
	    }
	}
    }

    private void authTerminate() {
	if(userAuth != null) {
	    userAuth.terminate();
	}
    }

    public void transmit(SSH2TransportPDU pdu) {
	if(isConnected) {
	    txQueue.putLast(pdu);
	}
    }

    public synchronized void transmitInternal(SSH2TransportPDU pdu)
	throws SSH2Exception
    {
	if(DEBUG_ALL_TX) tpLog.debug2("SSH2Transport",
				      "transmitInternal",
				      "sending message of type: " +
				      SSH2.msgTypeString(pdu.pktType),
				      pdu.getData(),
				      pdu.getPayloadOffset(),
				      pdu.getPayloadLength());
	try {
	    pdu.writeTo(tpOut,
			txSeqNum++, txMAC, txCipher, txCompressor, tpRand);
	} catch (ShortBufferException e) {
	    throw new SSH2FatalException("Internal error/bug: " +
					 e.getMessage());
	} catch (IOException e) {
	    throw new SSH2FatalException("Couldn't write packet of type " +
					 SSH2.msgTypeString(pdu.pktType), e);
	}
    }

    public void fatalDisconnect(int reason, String description) {
	disconnectInternal(reason, description,
			   /* !!! TODO: languageTag, from ourPrefs? */ "",
			   false);
    }

    public void normalDisconnect(String description) {
	disconnectInternal(SSH2.DISCONNECT_BY_APPLICATION, description,
			   /* !!! TODO: languageTag, from ourPrefs? */ "",
			   false);
    }

    private void disconnectInternal(int reason, String description,
				    String languageTag, boolean fromPeer) {
	synchronized(disconnectMonitor) {
	    if(!isConnected) {
		return;
	    }
	    isConnected = false;
	}

	if(!fromPeer && isTxUp) {
	    //
	    // !!! Pathological condition: tx may be exiting, will cause bug
	    //
	    txQueue.disable();
	    txQueue.waitUntilBlocked();

	    SSH2TransportPDU pdu =
		SSH2TransportPDU.createOutgoingPacket(SSH2.MSG_DISCONNECT);
	    pdu.writeInt(reason);
	    pdu.writeString(description);
	    pdu.writeString(""); // !!! TODO: Handle the language

	    try {
		transmitInternal(pdu);
	    } catch (SSH2Exception e) {
		tpLog.message(Log.LEVEL_ERROR, "SSH2Transport",
			      "disconnectInternal",
			      "error writing disconnect msg: " + e);
	    }
	}

	disableKeepAlive();

	shutdownRx();
	shutdownTx();

	if(connection != null) {
	    connection.terminate();
	}

	if(fromPeer) {
	    eventHandler.peerDisconnect(this, reason, description, languageTag);
	} else if(reason == SSH2.DISCONNECT_BY_APPLICATION) {
	    eventHandler.normalDisconnect(this, description, languageTag);
	} else {
	    eventHandler.fatalDisconnect(this, reason, description, languageTag);
	}

	tpLog.warning("SSH2Transport", "disconnect: " + description);
    }

    private void negotiateVersion() throws IOException, SSH2Exception {
	String idString;
	String ourVersion = SSH2.getVersionId(ourPrefs.getPackageVersion());

	if(weAreAServer) {
	    serverVersion = ourVersion;
	    idString =  serverVersion + "\r\n";
	    tpOut.write(idString.getBytes());
	    tpOut.flush();
	    clientVersion = idString;
	    tpLog.info("SSH2Transport", "peer's version is '" +
		       clientVersion + "'");
	} else {
	    clientVersion = ourVersion;
	    idString =  clientVersion + "\r\n";
	    tpOut.write(idString.getBytes());
	    tpOut.flush();
	    while(!(idString = readIdString()).startsWith("SSH-")) {
		eventHandler.gotConnectInfoText(this, idString);
	    }
	    serverVersion = idString;
	    tpLog.info("SSH2Transport", "peer's version is '" +
		       serverVersion + "'");
	}

	checkPeerVersion(clientVersion, serverVersion);
    }

    private void checkPeerVersion(String clientVersion, String serverVersion)
	throws SSH2Exception
    {
	String cliPackage = extractPackageVersion(clientVersion);
	String srvPackage = extractPackageVersion(serverVersion);
	int cliMajor = extractMajor(clientVersion);
	int cliMinor = extractMinor(clientVersion);
	int srvMajor = extractMajor(serverVersion);
	int srvMinor = extractMinor(serverVersion);

	if(weAreAServer) {
	    eventHandler.gotPeerVersion(this, clientVersion,
					cliMajor, cliMinor, cliPackage);
	} else {
	    eventHandler.gotPeerVersion(this, serverVersion,
					srvMajor, srvMinor, srvPackage);
	}

	if(cliMajor != srvMajor && !(srvMajor == 1 && srvMinor == 99)) {
	    String msg;
	    if(weAreAServer) {
		msg = "Can't serve a client with version " + clientVersion;
	    } else {
		msg = "Can't connect to a server with version " + serverVersion;
	    }
	    throw new SSH2FatalException(msg);
	}

	String peerPackage = (weAreAServer ? cliPackage : srvPackage);

	if(peerPackage.startsWith("2.0.7 ") ||
	   peerPackage.startsWith("2.0.8 ") ||
	   peerPackage.startsWith("2.0.9 ")) {
	    throw new SSH2FatalException("Peer's version is too old: " + peerPackage);
	}

	incompatibleServiceAccept = peerPackage.startsWith("2.0.11 ") ||
	    peerPackage.startsWith("2.0.12 ") ||
	    peerPackage.startsWith("2.0.13 ");

	incompatiblePublicKeyAuth = incompatibleServiceAccept;

	incompatibleChannelOpenFail = incompatibleServiceAccept;

	incompatibleSignature = peerPackage.startsWith("2.1.0 SSH") ||
	    (peerPackage.startsWith("2.1.0") &&
	     peerPackage.indexOf("F-SECURE") != -1) ||
	    incompatibleServiceAccept;

	incompatibleHMACKeyLength = incompatibleSignature ||
	    peerPackage.startsWith("2.2.0 SSH") ||
	    peerPackage.startsWith("2.3.0 SSH") ||
	    ((peerPackage.startsWith("2.2.0") ||
	      peerPackage.startsWith("2.3.0")) &&
	     peerPackage.indexOf("F-SECURE") != -1);

	incompatiblePublicKeyUserId = incompatibleSignature ||
	    peerPackage.startsWith("OpenSSH_2.0") ||
	    peerPackage.startsWith("OpenSSH_2.1") ||
	    peerPackage.startsWith("OpenSSH_2.2");

	incompatibleRijndael = peerPackage.startsWith("OpenSSH_2.5.1p1") ||
	    peerPackage.startsWith("OpenSSH_2.5.0") ||
	    peerPackage.startsWith("OpenSSH_2.3");

	incompatibleCantReKey = incompatiblePublicKeyUserId ||
	    peerPackage.startsWith("OpenSSH_2.3")   ||
	    peerPackage.startsWith("OpenSSH_2.5.1") ||
	    peerPackage.startsWith("OpenSSH_2.5.2");

	if(incompatibleServiceAccept) {
	    tpLog.notice("SSH2Transport",
			 "enabling draft incompatible SERVICE_ACCEPT");
	    tpLog.notice("SSH2Transport",
			 "enabling draft incompatible publickey method");
	    tpLog.notice("SSH2Transport",
			 "enabling draft incompatible CHANNEL_OPEN_FAILURE");
	}
	if(incompatibleSignature) {
	    tpLog.notice("SSH2Transport",
			 "enabling draft incompatible signature format");
	}
	if(incompatibleHMACKeyLength) {
	    tpLog.notice("SSH2Transport",
			 "enabling rfc incompatible hmac key length");
	}
	if(incompatiblePublicKeyUserId) {
	    tpLog.notice("SSH2Transport",
			 "enabling draft incompatible session id for signature");
	}
	if(incompatibleRijndael) {
	    tpLog.notice("SSH2Transport",
			 "disabling aes/rijndael cipher, peer has buggy implementation");
	}
	if(incompatibleCantReKey) {
	    tpLog.notice("SSH2Transport",
			 "disabling key re-exchange, not implemented in peer");
	}
    }

    public static int extractMajor(String versionStr) throws SSH2Exception {
	try {
	    int r = versionStr.indexOf('.', 4);
	    return Integer.parseInt(versionStr.substring(4, r));
	} catch (NumberFormatException e) {
	    throw new SSH2FatalException("Corrupt version string: " +
					 versionStr);
	}
    }

    public static int extractMinor(String versionStr) throws SSH2Exception {
	try {
	    int l = versionStr.indexOf('.', 4) + 1;
	    int r = versionStr.indexOf('-', l);
	    return Integer.parseInt(versionStr.substring(l, r));
	} catch (NumberFormatException e) {
	    throw new SSH2FatalException("Corrupt version string: " +
					 versionStr);
	}
    }

    public static String extractPackageVersion(String versionStr)
	throws SSH2Exception
    {
	try {
	    int i = versionStr.indexOf('-', 4) + 1;
	    return versionStr.substring(i);
	} catch (Exception e) {
	    throw new SSH2FatalException("Corrupt version string: " +
					 versionStr);
	}
    }

    private String readIdString() throws IOException, SSH2Exception {
	byte[] buf = new byte[256];
	int    len = 0;
	int    c;

	while(true) {
	    c = tpIn.read();
	    if(c == -1) {
		throw new SSH2EOFException("Server closed connection before sending identifaction");
	    }
	    if(c == '\r')
	       continue;
            if(c != '\n') {
                buf[len++] = (byte)c;
            } else {
		return new String(buf, 0, len);
            }
	}
    }

    private void sendKEXINIT() throws SSH2Exception {
	SSH2TransportPDU pdu =
	    SSH2TransportPDU.createOutgoingPacket(SSH2.MSG_KEXINIT);
	byte[] cookie = new byte[16];
	tpRand.nextBytes(cookie);
	pdu.writeRaw(cookie);
	ourPrefs.writeTo(pdu);
	pdu.writeBoolean(false);
	pdu.writeInt(0);

	if(weAreAServer) {
	    serverKEXINITPkt = pdu.makeCopy();
	} else {
	    clientKEXINITPkt = pdu.makeCopy();
	}

	transmitInternal(pdu);
	eventHandler.kexStart(this);
    }

    private void processKEXINIT(SSH2TransportPDU pdu) throws SSH2Exception {
	startKeyExchange();

	if(weAreAServer) {
	    clientKEXINITPkt = pdu;
	} else {
	    serverKEXINITPkt = pdu;
	}

	pdu.readRaw(16); // Cookie, we don't need it
	peerPrefs = new SSH2TransportPreferences();
	peerPrefs.readFrom(pdu);
	boolean firstKEXFollows = pdu.readBoolean();
	pdu.readInt(); // Reserved int, we don't need it

	tpLog.info("SSH2Transport", "peer kex algorithms: " +
	    peerPrefs.listPreference(SSH2TransportPreferences.KEX_ALGORITHMS));
	tpLog.info("SSH2Transport", "peer host key algorithms: " +
	    peerPrefs.listPreference(SSH2TransportPreferences.HOST_KEY_ALG));
	tpLog.info("SSH2Transport", "peer enc. alg. cli2srv: " +
	    peerPrefs.listPreference(SSH2TransportPreferences.CIPHERS_C2S));
	tpLog.info("SSH2Transport", "peer enc. alg. srv2cli: " +
	    peerPrefs.listPreference(SSH2TransportPreferences.CIPHERS_S2C));
	tpLog.info("SSH2Transport", "peer mac alg. cli2srv: " +
	    peerPrefs.listPreference(SSH2TransportPreferences.MACS_C2S));
	tpLog.info("SSH2Transport", "peer mac alg. srv2cli: " +
	    peerPrefs.listPreference(SSH2TransportPreferences.MACS_S2C));
	tpLog.info("SSH2Transport", "peer comp. alg. cli2srv: " +
	    peerPrefs.listPreference(SSH2TransportPreferences.COMP_C2S));
	tpLog.info("SSH2Transport", "peer comp. alg. srv2cli: " +
	    peerPrefs.listPreference(SSH2TransportPreferences.COMP_S2C));
	tpLog.info("SSH2Transport", "our kex algorithms: " +
	    ourPrefs.listPreference(SSH2TransportPreferences.KEX_ALGORITHMS));
	tpLog.info("SSH2Transport", "our host key algorithms: " +
	    ourPrefs.listPreference(SSH2TransportPreferences.HOST_KEY_ALG));
	tpLog.info("SSH2Transport", "our enc. alg. cli2srv: " +
	    ourPrefs.listPreference(SSH2TransportPreferences.CIPHERS_C2S));
	tpLog.info("SSH2Transport", "our enc. alg. srv2cli: " +
	    ourPrefs.listPreference(SSH2TransportPreferences.CIPHERS_S2C));
	tpLog.info("SSH2Transport", "our mac alg. cli2srv: " +
	    ourPrefs.listPreference(SSH2TransportPreferences.MACS_C2S));
	tpLog.info("SSH2Transport", "our mac alg. srv2cli: " +
	    ourPrefs.listPreference(SSH2TransportPreferences.MACS_S2C));
	tpLog.info("SSH2Transport", "our comp. alg. cli2srv: " +
	    ourPrefs.listPreference(SSH2TransportPreferences.COMP_C2S));
	tpLog.info("SSH2Transport", "our comp. alg. srv2cli: " +
	    ourPrefs.listPreference(SSH2TransportPreferences.COMP_S2C));

	keyExchanger = ourPrefs.selectKEXAlgorithm(peerPrefs, weAreAServer);

	tpLog.notice("SSH2Transport", "KEX algorithm chosen: " +
		     ourPrefs.getKEXAlgorithm());
	tpLog.info("SSH2Transport", "same KEX guessed? " +
		   ourPrefs.sameKEXGuess());
	tpLog.info("SSH2Transport", "first KEX follows? " + firstKEXFollows);

	if(!ourPrefs.canAgree(peerPrefs, weAreAServer)) {
	    throw new SSH2FatalException(
			 "Can't agree on transport preferences with peer");
	}

	if(firstKEXFollows && !ourPrefs.sameKEXGuess()) {
	    // Discard next packet which is the incorrectly guessed KEX packet
	    //
	    try {
		receiveInternal();
	    } catch (IOException e) {
		throw new SSH2FatalException("I/O error when reading guessed " +
					     "packet", e);
	    } catch (ShortBufferException e) {
		throw new SSH2FatalException("Internal error/bug: " +
					     e.getMessage());
	    }
	    tpLog.notice("SSH2Transport", "first KEX packet discarded, " +
			 "wrong initial guess");
	}

	eventHandler.kexAgreed(this, ourPrefs, peerPrefs);

	keyExchanger.init(this);
    }

    private void removeRijndael() {
	boolean removedAES = false;
	String l1, l2;
	l1 = ourPrefs.listPreference(SSH2TransportPreferences.CIPHERS_C2S);
	l2 = ourPrefs.listPreference(SSH2TransportPreferences.CIPHERS_S2C);

	int l1l = l1.length();
	int l2l = l2.length();

	l1 = SSH2ListUtil.removeAllPrefixFromList(l1, "aes");
	l1 = SSH2ListUtil.removeAllPrefixFromList(l1, "rijndael");
	l2 = SSH2ListUtil.removeAllPrefixFromList(l2, "aes");
	l2 = SSH2ListUtil.removeAllPrefixFromList(l2, "rijndael");

	if(l1.length() != l1l) {
	    ourPrefs.setPreference(SSH2TransportPreferences.CIPHERS_C2S,
				    l1);
	    removedAES = true;
	}
	if(l2.length() != l2l) {
	    ourPrefs.setPreference(SSH2TransportPreferences.CIPHERS_S2C,
				    l2);
	    removedAES = true;
	}
	if(removedAES) {
	    tpLog.warning("SSH2Transport",
			  "removed AES cipher from our preferences" +
			  " due to bug in peer's implementation");
	}
    }

    public void sendNewKeys() throws SSH2Exception {
	SSH2TransportPDU pdu =
	    SSH2TransportPDU.createOutgoingPacket(SSH2.MSG_NEWKEYS);
	transmitInternal(pdu);
	changeTransmitterKeys();
	txQueue.enable();
    }

    public void authenticateHost(byte[] serverHostKey, byte[] serverSigH,
				 byte[] exchangeHash_H)
	throws SSH2Exception
    {
	tpLog.debug2("SSH2Transport", "authenticateHost",
		     "Server's public host key: ", serverHostKey);
	tpLog.debug2("SSH2Transport", "authenticateHost",
		     "Signature over H: ", serverSigH);
	tpLog.debug2("SSH2Transport", "authenticateHost",
		     "Exchange hash H", exchangeHash_H);

	boolean       verified  = false;
	SSH2Signature signature =
	    SSH2Signature.getInstance(ourPrefs.getHostKeyAlgorithm());

	signature.initVerify(serverHostKey);
	signature.setIncompatibility(this);

	verified = signature.verify(serverSigH, exchangeHash_H);

	if(verified) {
	    tpLog.notice("SSH2Transport", "server's signature verified");
	} else {
	    String msg = "server's signature didn't verify";
	    tpLog.error("SSH2Transport", "authenticateHost", msg);
	    fatalDisconnect(SSH2.DISCONNECT_HOST_KEY_NOT_VERIFIABLE, msg);
	    throw new SSH2FatalException(msg);
	}

	if(!eventHandler.kexAuthenticateHost(this, signature)) {
	    throw new SSH2SignatureException("Host authentication failed");
	}
    }

    private void transportTransmitLoop() {
	isTxUp = true;
	tpLog.debug("SSH2Transport", "transportTransmitLoop",
		    "starting");
	try {
	    SSH2TransportPDU pdu;
	    while((pdu = (SSH2TransportPDU)txQueue.getFirst()) != null) {

		if(DEBUG_ALL_TX) tpLog.debug2("SSH2Transport",
					      "transportTransmitLoop",
					      "sending message of type: " +
					      SSH2.msgTypeString(pdu.pktType),
					      pdu.getData(),
					      pdu.getPayloadOffset(),
					      pdu.getPayloadLength());

		// Note, we don't use transmitInternal since we don't want to
		// loop over the exception handler here
		//
		pdu.writeTo(tpOut, txSeqNum++,
			    txMAC, txCipher, txCompressor, tpRand);
	    }
	} catch (ShortBufferException e) {
	    String msg = "Internal error/bug: " + e.getMessage();
	    tpLog.error("SSH2Transport", "transportTransmitLoop", msg);
	    disconnectInternal(SSH2.DISCONNECT_CONNECTION_LOST, msg,
			       /* !!! TODO: languageTag, from ourPrefs? */ "",
			       false);
	} catch (IOException e) {
	    String msg = "I/O error: " + e.getMessage();
	    if(isTxUp) {
		tpLog.error("SSH2Transport", "transportTransmitLoop", msg);
	    }
	    disconnectInternal(SSH2.DISCONNECT_CONNECTION_LOST, msg,
			       /* !!! TODO: languageTag, from ourPrefs? */ "",
			       false);
	} catch (SSH2CompressionException e) {
	    String msg = "Internal error/bug: " + e.getMessage();
	    tpLog.error("SSH2Transport", "transportTransmitLoop", msg);
	    disconnectInternal(SSH2.DISCONNECT_COMPRESSION_ERROR, msg,
			       /* !!! TODO: languageTag, from ourPrefs? */ "",
			       false);
	} finally {
	    shutdownTx();
	    kexComplete(false);
	    authTerminate();
	}
	tpLog.debug("SSH2Transport", "transportTransmitLoop",
		    "stopping");
    }

    private void transportReceiveLoop() {
	isRxUp = true;
	tpLog.debug("SSH2Transport", "transportReceiveLoop",
		    "starting");
	try {
	    while(isRxUp) {
		SSH2TransportPDU pdu = receiveInternal();

		switch(pdu.pktType) {
		case SSH2.MSG_DISCONNECT: {
		    int    reason      = pdu.readInt();
		    String description = pdu.readJavaString();
		    String languageTag = pdu.readJavaString();
		    // !!! TODO, disconnect ourselves (not trying to send disc...)
		    eventHandler.peerDisconnect(this, reason, description,
						languageTag);
		    break;
		}

		case SSH2.MSG_IGNORE:
		    byte[] data = pdu.readString();
		    eventHandler.msgIgnore(this, data);
		    break;

		case SSH2.MSG_UNIMPLEMENTED:
		    int rejectedSeqNum = pdu.readInt();
		    eventHandler.msgUnimplemented(this, rejectedSeqNum);
		    break;

		case SSH2.MSG_DEBUG: {
		    boolean alwaysDisplay = pdu.readBoolean();
		    String  message       = pdu.readJavaString();
		    String  languageTag   = pdu.readJavaString();
		    eventHandler.msgDebug(this, alwaysDisplay, message,
					  languageTag);
		    break;
		}

		case SSH2.MSG_SERVICE_REQUEST:
		    break;

		case SSH2.MSG_SERVICE_ACCEPT:
		    userAuth.processMessage(pdu);
		    pdu = null;
		    break;

		case SSH2.MSG_KEXINIT:
		    processKEXINIT(pdu);
		    pdu = null;
		    break;

		case SSH2.MSG_NEWKEYS:
		    if(!keyExchangeInProgress)
			throw new SSH2CorruptPacketException(
			"Received MSG_NEWKEYS while not doing key exchange");
		    changeReceiverKeys();
		    break;

		case SSH2.FIRST_KEX_PACKET:
		case 31:
		case 32:
		case 33:
		case 34:
		case 35:
		case 36:
		case 37:
		case 38:
		case 39:
		case 40:
		case 41:
		case 42:
		case 43:
		case 44:
		case 45:
		case 46:
		case 47:
		case 48:
		case SSH2.LAST_KEX_PACKET:
		    if(!keyExchangeInProgress)
			throw new SSH2CorruptPacketException(
			"Received KEX packet while not doing key exchange");
		    keyExchanger.processKEXMethodPDU(pdu);
		    break;

		case SSH2.MSG_USERAUTH_REQUEST:
		case SSH2.MSG_USERAUTH_FAILURE:
		case SSH2.MSG_USERAUTH_SUCCESS:
		case SSH2.MSG_USERAUTH_BANNER:
		case SSH2.FIRST_USERAUTH_METHOD_PACKET:
		case 61:
		case 62:
		case 63:
		case 64:
		case 65:
		case 66:
		case 67:
		case 68:
		case 69:
		case 70:
		case 71:
		case 72:
		case 73:
		case 74:
		case 75:
		case 76:
		case 77:
		case 78:
		case SSH2.LAST_USERAUTH_METHOD_PACKET:
		    userAuth.processMessage(pdu);
		    pdu = null;
		    break;

		case SSH2.MSG_GLOBAL_REQUEST:
		case SSH2.MSG_REQUEST_SUCCESS:
		case SSH2.MSG_REQUEST_FAILURE:
		    connection.processGlobalMessage(pdu);
		    break;
		case SSH2.MSG_CHANNEL_OPEN:
		    connection.processGlobalMessage(pdu);
		    pdu = null;
		    break;

		case SSH2.MSG_CHANNEL_DATA:
		case SSH2.MSG_CHANNEL_EXTENDED_DATA:
		    connection.processChannelMessage(pdu);
		    pdu = null;
		    break;

		case SSH2.MSG_CHANNEL_OPEN_CONFIRMATION:
		case SSH2.MSG_CHANNEL_OPEN_FAILURE:
		case SSH2.MSG_CHANNEL_WINDOW_ADJUST:
		case SSH2.MSG_CHANNEL_EOF:
		case SSH2.MSG_CHANNEL_CLOSE:
		case SSH2.MSG_CHANNEL_REQUEST:
		case SSH2.MSG_CHANNEL_SUCCESS:
		case SSH2.MSG_CHANNEL_FAILURE:
		    connection.processChannelMessage(pdu);
		    break;

		default:
		    tpLog.warning("SSH2Transport",
				  "received packet of unknown type: " +
				  pdu.pktType);
		    SSH2TransportPDU pduUnimp =
			SSH2TransportPDU.createOutgoingPacket(SSH2.MSG_UNIMPLEMENTED);
		    pduUnimp.writeInt(rxSeqNum);
		    if(keyExchangeInProgress) {
			transmitInternal(pduUnimp);
		    } else {
			transmit(pduUnimp);
		    }
		    eventHandler.peerSentUnknownMessage(this, pdu.pktType);
		    break;
		}
		if(pdu != null) {
		    pdu.release();
		}
	    }
	} catch (ShortBufferException e) {
	    String msg = "Internal error/bug: " + e.getMessage();
	    tpLog.error("SSH2Transport", "transportReceiveLoop", msg);
	    disconnectInternal(SSH2.DISCONNECT_CONNECTION_LOST, msg,
			       /* !!! TODO: languageTag, from ourPrefs? */ "",
			       false);
	} catch (SSH2MacCheckException e) {
	    String msg = e.getMessage();
	    tpLog.error("SSH2Transport", "transportReceiveLoop", msg);
	    disconnectInternal(SSH2.DISCONNECT_MAC_ERROR, msg,
			       /* !!! TODO: languageTag, from ourPrefs? */ "",
			       false);
	} catch (SSH2CompressionException e) {
	    String msg = e.getMessage();
	    tpLog.error("SSH2Transport", "transportReceiveLoop", msg);
	    disconnectInternal(SSH2.DISCONNECT_COMPRESSION_ERROR, msg,
			       /* !!! TODO: languageTag, from ourPrefs? */ "",
			       false);
	} catch (SSH2SignatureException e) {
	    String msg = e.getMessage();
	    tpLog.error("SSH2Transport", "transportReceiveLoop", msg);
	    disconnectInternal(SSH2.DISCONNECT_KEY_EXCHANGE_FAILED, msg,
			       /* !!! TODO: languageTag, from ourPrefs? */ "",
			       false);
	} catch (SSH2Exception e) {
	    if(isRxUp) {
		String msg = e.getMessage();
		if(e.getRootCause() != null) {
		    msg += " (rootcause: " + e.getRootCause() + ")";
		}
		tpLog.error("SSH2Transport", "transportReceiveLoop", msg);
		disconnectInternal(SSH2.DISCONNECT_PROTOCOL_ERROR, msg,
				   /* !!! TODO: languageTag, from ourPrefs? */ "",
				   false);
	    }
	} catch (IOException e) {
	    if(isRxUp) {
		String msg = "I/O error: " + e.getMessage();
		tpLog.error("SSH2Transport", "transportReceiveLoop", msg);
		disconnectInternal(SSH2.DISCONNECT_CONNECTION_LOST, msg,
				   /* !!! TODO: languageTag, from ourPrefs? */ "",
				   false);
	    }
	} finally {
	    shutdownRx();
	    kexComplete(false);
	    authTerminate();
	}
	tpLog.debug("SSH2Transport", "transportReceiveLoop",
		    "stopping");
    }

    public SSH2TransportPDU receiveInternal()
	throws SSH2Exception, ShortBufferException, IOException
    {
	SSH2TransportPDU pdu = SSH2TransportPDU.createIncomingPacket();
	pdu.readFrom(tpIn, rxSeqNum++, rxMAC, rxCipher, rxCompressor);

	if(DEBUG_ALL_RX) tpLog.debug2("SSH2Transport",
				      "receiveInternal",
				      "received message of type: " +
				      SSH2.msgTypeString(pdu.pktType),
				      pdu.getData(),
				      pdu.getPayloadOffset(),
				      pdu.getPayloadLength());
	return pdu;
    }

    private void shutdownTx() {
	if(isTxUp) {
	    isTxUp = false;
	    try { tpOut.close(); } catch (IOException e) { /* don't care */ }
	    txQueue.disable();
	    txQueue.setBlocking(false);
	}
    }

    private void shutdownRx() {
	if(isRxUp) {
	    isRxUp = false;
	    try { tpIn.close(); } catch (IOException e) { /* don't care */ }
	}
    }

    private synchronized void changeTransmitterKeys() throws SSH2Exception {
	try {
	    String cipherName = ourPrefs.getTransmitterCipher();
	    String macName    = ourPrefs.getTransmitterMac();
	    String compName   = ourPrefs.getTransmitterCompression();
	    tpLog.info("SSH2Transport", "new transmitter context (" +
		       cipherName + "," + macName + "," + compName + ")");
	    txCipher =
		Cipher.getInstance(ourPrefs.ssh2ToJCECipher(cipherName));
	    txMAC = Mac.getInstance(ourPrefs.ssh2ToJCEMac(macName));
	    initKeys(txCipher, ourPrefs.getCipherKeyLen(cipherName),
		     txMAC,
		     (incompatibleHMACKeyLength ? 16 :
		      ourPrefs.getMacKeyLen(macName)),
		     true);
	    txCompressor =
		SSH2Compressor.getInstance(compName,
					   SSH2Compressor.COMPRESS_MODE,
					   ourPrefs.getCompressionLevel());
	} catch (Exception e) {
	    txCipher = null;
	    txMAC    = null;
	    throw new SSH2FatalException("Error in changeTransmitterKeys", e);
	}
    }

    private synchronized void changeReceiverKeys() throws SSH2Exception {
	try {
	    String cipherName = ourPrefs.getReceiverCipher();
	    String macName    = ourPrefs.getReceiverMac();
	    String compName   = ourPrefs.getReceiverCompression();
	    tpLog.info("SSH2Transport", "new receiver context (" +
		       cipherName + "," + macName + "," + compName + ")");
	    rxCipher =
		Cipher.getInstance(ourPrefs.ssh2ToJCECipher(cipherName));
	    rxMAC = Mac.getInstance(ourPrefs.ssh2ToJCEMac(macName));
	    initKeys(rxCipher, ourPrefs.getCipherKeyLen(cipherName),
		     rxMAC,
		     (incompatibleHMACKeyLength ? 16 :
		      ourPrefs.getMacKeyLen(macName)),
		     false);
	    rxCompressor =
		SSH2Compressor.getInstance(compName,
					   SSH2Compressor.UNCOMPRESS_MODE);
	} catch (Exception e) {
	    rxCipher = null;
	    rxMAC    = null;
	    throw new SSH2FatalException("Error in changeReceiverKeys", e);
	}

	kexComplete(true);
    }

    private void initKeys(Cipher cipher, int ckLen, Mac mac, int mkLen,
			  boolean transmit)
	throws SSH2Exception
    {
	byte[] iv, cKey, mKey;
	char[] ids;
	if(weAreAServer ^ transmit) {
	    ids = new char[] { 'A', 'C', 'E' };
	} else {
	    ids = new char[] { 'B', 'D', 'F' };
	}

	iv   = deriveKey(ids[0], cipher.getBlockSize());
	cKey = deriveKey(ids[1], ckLen);
	mKey = deriveKey(ids[2], mkLen);

	try {
	    if(cipher != null) {
		cipher.init(transmit ? Cipher.ENCRYPT_MODE :
			    Cipher.DECRYPT_MODE,
			    new SecretKeySpec(cKey, cipher.getAlgorithm()),
			    new IvParameterSpec(iv));
	    }
	    if(mac != null) {
		mac.init(new SecretKeySpec(mKey, mac.getAlgorithm()));
	    }
	} catch (InvalidKeyException e) {
	    throw new SSH2FatalException("Invalid key generated in initKeys");
	}
    }

    byte[] deriveKey(char id, int len) {
	byte[] key = new byte[len];

	byte[] sharedSecret_K = keyExchanger.getSharedSecret_K();
	byte[] exchangeHash_H = keyExchanger.getExchangeHash_H();

	if(sessionId == null) {
	    sessionId = new byte[exchangeHash_H.length];
	    System.arraycopy(exchangeHash_H, 0, sessionId, 0,
			     sessionId.length);
	}

	MessageDigest sha1 = keyExchanger.getExchangeHashAlgorithm();

	sha1.update(sharedSecret_K);
	sha1.update(exchangeHash_H);
	sha1.update(new byte[] { (byte)id });
	sha1.update(sessionId);
	byte[] material = sha1.digest();

	int curLen = material.length;
	System.arraycopy(material, 0, key, 0, (curLen < len ? curLen : len));

	while(curLen < len) {
	    sha1.reset();
	    sha1.update(sharedSecret_K);
	    sha1.update(exchangeHash_H);
	    sha1.update(key, 0, curLen);
	    material = sha1.digest();
	    if(len - curLen > material.length)
		System.arraycopy(material, 0, key, curLen, material.length);
	    else
		System.arraycopy(material, 0, key, curLen, len - curLen);
	    curLen += material.length;
	}

	tpLog.debug2("SSH2Transport", "deriveKey", "key id " + id, key);

	return key;
    }

}
