/******************************************************************************
 *
 * Copyright (c) 1999-2001 AppGate AB. All Rights Reserved.
 * 
 * This file contains Original Code and/or Modifications of Original Code as
 * defined in and that are subject to the MindTerm Public Source License,
 * Version 1.1, (the 'License'). You may not use this file except in compliance
 * with the License.
 * 
 * You should have received a copy of the MindTerm Public Source License
 * along with this software; see the file LICENSE.  If not, write to
 * AppGate AB, Stora Badhusgatan 18-20, 41121 Goteborg, SWEDEN
 *
 *****************************************************************************/

package com.mindbright.ssh2;

import java.util.Hashtable;

import com.mindbright.util.Queue;

public final class SSH2UserAuth {
    private SSH2Transport     transport;
    private SSH2Authenticator authenticator;
    private SSH2AuthModule    authModule;

    private volatile boolean isAuthenticated;

    String  currentMethod;

    String service;
    String user;
    String ourMethods;

    Queue procQueue;

    public SSH2UserAuth(SSH2Transport transport,
			SSH2Authenticator authenticator) {
	this.transport       = transport;
	this.authenticator   = authenticator;
	this.isAuthenticated = false;
	this.procQueue       = new Queue();
    }

    public SSH2Transport getTransport() {
	return transport;
    }

    public SSH2Authenticator getAuthenticator() {
	return authenticator;
    }

    public boolean authenticateUser(String service) {
	this.service       = service;
	this.user          = authenticator.getUsername();
	this.currentMethod = null;

	transport.setUserAuth(this, "ssh-userauth");

	String  peerMethods = null;
	boolean retry       = false;
	boolean partial     = false;
	int     lastType    = -1;

    authLoop:
	while(!isAuthenticated) {
	    SSH2TransportPDU pdu = null;

	    if(!retry) {
		pdu = (SSH2TransportPDU)procQueue.getFirst();
		if(pdu == null) {
		    authenticator.authError();
		    break authLoop;
		}
		lastType = pdu.getType();
	    }

	    try {
		switch(lastType) {
		case SSH2.MSG_USERAUTH_FAILURE:
		    if(!retry) {
			peerMethods = new String(pdu.readString());
			partial     = pdu.readBoolean();
			authenticator.peerMethods(peerMethods);
		    }
		    retry = false;

		    transport.getLog().notice("SSH2UserAuth",
					      "failure continuation: " +
					      peerMethods +
					      " (partial: " + partial + ")");

		    if(currentMethod != null) {
			ourMethods =
			    SSH2ListUtil.removeFirstFromList(ourMethods,
							     currentMethod);
			authenticator.authFailure(currentMethod, partial);
		    } else {
			ourMethods = authenticator.getMethods();
		    }

		    transport.getLog().notice("SSH2UserAuth",
					      "our remaining methods: " +
					      (ourMethods.length() > 0 ?
					       ourMethods : "<none>"));

		    currentMethod = SSH2ListUtil.chooseFromList(ourMethods,
								peerMethods);

		    if(currentMethod != null) {
			transport.getLog().notice("SSH2UserAuth",
						  "trying method: " +
						  currentMethod);
			authModule = authenticator.getModule(currentMethod);
			SSH2TransportPDU modPDU =
			    authModule.startAuthentication(this);
			transport.transmit(modPDU);
		    } else {
			authenticator.noMoreMethods();
			transport.getLog().notice("SSH2UserAuth",
				  "no more authentication methods, giving up");
			break authLoop;
		    }
		    break;

		case SSH2.MSG_USERAUTH_SUCCESS:
		    transport.getLog().notice("SSH2UserAuth",
				      "successful authentication with " +
					      currentMethod);
		    isAuthenticated = true;
		    authenticator.authSuccess(currentMethod);
		    break;

		case SSH2.MSG_USERAUTH_BANNER:
		    String msg = new String(pdu.readString());
		    transport.getLog().warning("SSH2UserAuth", "banner: " +
					       msg);
		    authenticator.displayBanner(msg);
		    break;

		case SSH2.MSG_SERVICE_ACCEPT:
		    if(transport.incompatibleServiceAccept) {
			transport.getLog().notice("SSH2UserAuth",
						  "server accepted " +
						  service +
						  " (draft incompatible)");
		    } else {
			transport.getLog().notice("SSH2UserAuth",
						  "server accepted: " +
					  new String(pdu.readString()));
		    }
		    doNoneAuth();
		    break;

		case SSH2.FIRST_USERAUTH_METHOD_PACKET:
		case 61:
		case 62:
		case 63:
		case 64:
		case 65:
		case 66:
		case 67:
		case 68:
		case 69:
		case 70:
		case 71:
		case 72:
		case 73:
		case 74:
		case 75:
		case 76:
		case 77:
		case 78:
		case SSH2.LAST_USERAUTH_METHOD_PACKET:
		    if(authModule != null) {
			SSH2TransportPDU modPDU =
			    authModule.processMethodMessage(this, pdu);
			if(modPDU != null)
			    transport.transmit(modPDU);
		    } else {
			transport.fatalDisconnect(SSH2.DISCONNECT_PROTOCOL_ERROR,
						  "Received userauth method " +
						  "packet when no method selected");
			break authLoop;
		    }
		    break;
		}
	    } catch (SSH2UserCancelException e) {
		String msg = e.getMessage();
		authenticator.moduleCancel(currentMethod, msg);
		transport.fatalDisconnect(
				  SSH2.DISCONNECT_AUTH_CANCELLED_BY_USER,
				  "User cancel: " + msg);
		transport.getLog().
		    notice("SSH2UserAuth",
			   "user canceled authentication: "
			   + msg);
		break authLoop;
	    } catch (SSH2Exception e) {
		transport.getLog().error("SSH2UserAuth",
					 "authenticateUser",
					 "error in module '" +
					 currentMethod + "': " +
					 e.getMessage());
		authenticator.moduleFailure(currentMethod, e);
		partial  = false;
		retry    = true;
		lastType = SSH2.MSG_USERAUTH_FAILURE;
	    }
	    if(pdu != null) {
		pdu.release();
	    }
	}

	return isAuthenticated;
    }

    SSH2TransportPDU createUserAuthRequest(String method) {
	SSH2TransportPDU pdu =
	    SSH2TransportPDU.createOutgoingPacket(SSH2.MSG_USERAUTH_REQUEST);
	pdu.writeString(user);
	pdu.writeString(service);
	pdu.writeString(method);
	return pdu;
    }

    private void doNoneAuth() {
	SSH2TransportPDU pdu = createUserAuthRequest("none");
	transport.transmit(pdu);
    }

    void processMessage(SSH2TransportPDU pdu) {
	procQueue.putLast(pdu);
    }

    void terminate() {
	procQueue.setBlocking(false);
    }

    public boolean isAuthenticated() {
	return isAuthenticated;
    }
}

