/******************************************************************************
 *
 * Copyright (C) 1998 Logi Ragnarsson
 *
 * Adapted 1999 for use in MindTerm by Mats Andersson (mats@mindbright.se)
 * This class is the RandomSpinner class of the Cryptonite library found at:
 *     <http://www.hi.is/~logir/cryptonite/>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 *****************************************************************************
 * $Author: mats $
 * $Date: 1999/07/22 14:30:28 $
 * $Name: rel1-0-1 $
 *****************************************************************************/
package mindbright.security;

import java.util.Random;

/**
 * This class uses the scheduler to generate random numbers. It counts the
 * number of times a loop is repeated before a thread has slept for a
 * specified number of milliseconds. These numbers are then fed to a hash
 * function to mask any possible correlations.
 * <p>
 * This generator is somewhat slower than the SecureRandom generator in the
 * java library, since it spends some time collecting entropy.
 * <p>
 * The <a href="http://www.cs.berkeley.edu/~daw/java-spinner2">helper class</a>
 * which does the actual number generation is by
 * Henry Strickland (<a href="strix@versant.com">strix@versant.com</a>) and
 * Greg Noel (<a href="greg@qualcomm.com">greg@qualcomm.com</a>). It is based on
 * <a href="ftp://ftp.research.att.com/dist/mab/librand.shar">similar C code</a>
 * by Matt Blaze, Jack Lacy, and Don Mitchell.
 *
 * @author <a href="http://www.hi.is/~logir/">Logi Ragnarsson</a> (<a href="mailto:logir@hi.is">logir@hi.is</a>)
 */
public class SecureRandom extends Random {

    MD5State.SubState ss = new MD5State.SubState();
    private int t;
    public Thread updater;

    public static int secureLevel = 0;

    public SecureRandom() {
        t=Spinner.guessTime(1024);

	int paranoia = ((secureLevel > 0) ? 2 : 1);

	for(int i=0; i<paranoia; i++) {
            // Estimate about 4 bits of entropy per call to spinner
            for(int j=ss.buffer.length-1; j>=0; j--) {
                // Fill the buffer with spin-counts from the Spinner class.
                ss.buffer[j] = (byte)Spinner.spin(t);
		if(secureLevel < 2)
		    ss.buffer[--j] = (byte)System.currentTimeMillis();
	    }
            ss.transform(ss.buffer,0);
	}
        unused     = new byte[16];
        unusedPos  = 16;
	unusedLock = new Object();
    }

    public SecureRandom(byte[] seed) {
	try {
	    MD5State md5 = new MD5State();
	    md5.update(seed);
	    ss = md5.state;
	} catch (Exception e) {
	    // !!!
	    System.out.println("Can't operate, MD5 not available...");
	}
        t = Spinner.guessTime(1024);
        unused     = new byte[16];
        unusedPos  = 16;
	unusedLock = new Object();
    }

    /** unused[unusedPos..15] is unused pseudo-random numbers. */
    byte[] unused;
    int    unusedPos;
    Object unusedLock;

    int poolSweep=0;

    /** Get new unused bytes. */
    protected synchronized void update() {
        // Inject entropy into the pool
	//
	if(secureLevel > 1) {
	    ss.buffer[poolSweep++] += Spinner.spin(t) + 1;
	    ss.buffer[poolSweep++] += Spinner.spin(t) + 1;
	} else {
	    ss.buffer[poolSweep++] += Spinner.bogusSpin();
	    ss.buffer[poolSweep]   += ss.buffer[poolSweep - 1];
	    poolSweep++;
	}

        poolSweep %= 64;

	byte[] newUnused = new byte[16];
        ss.transform(ss.buffer,0);
        writeBytes(ss.hash[0], newUnused, 0,4);
        writeBytes(ss.hash[1], newUnused, 4,4);
        writeBytes(ss.hash[2], newUnused, 8,4);
        writeBytes(ss.hash[3], newUnused,12,4);

	synchronized(unusedLock) {
	    unused    = newUnused;
	    unusedPos = 0;
	}
    }
    
    /** Generates the next random number. */
    protected synchronized int next(int bits){
        //System.out.println(bits);
	int r=0;
	synchronized(unusedLock) {
	    for(int b=0; b<bits; b+=8) {
		if(unusedPos==16)
		    update();
		r = (r<<8) + unused[unusedPos++];
	    }
	}
        return r;
    }

    public synchronized void startUpdater() {
	if(updater != null)
	    return;
	updater = (new Thread(new Runnable() {
	    public void run() { 
		SecureRandom.this.updater.setPriority(SecureRandom.this.updater.getPriority() - 1);
		while(true) {
		    try {
			Thread.sleep(10000);
		    } catch (InterruptedException e) {
			// !!!
		    }
		    SecureRandom.this.update();
		}
	    }
	}));
	updater.start();
    }

    public void nextPadBytes(byte[] bytes, int len) {
	byte[] ub;
	int    ui;
	synchronized(unusedLock) {
	    for(int i = 0; i < len; i++) {
		unusedPos %= 16;
		bytes[i] = unused[unusedPos++];
	    }
	}
    }

    public static final void writeBytes(long a, byte[] dest, int i, int length) {
        for (int j=i+length-1; j>=i; j--){
            dest[j]=(byte)a;
            a = a >>> 8;
        }
    }

}
