/******************************************************************************
 *
 * Copyright (c) 1998,99 by Mindbright Technology AB, Stockholm, Sweden.
 *                 www.mindbright.se, info@mindbright.se
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 *****************************************************************************
 * $Author: mats $
 * $Date: 1999/09/26 13:45:40 $
 * $Name: rel1-0-1 $
 *****************************************************************************/
package mindbright.ssh;

import java.awt.*;
import java.awt.event.*;

import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.FileNotFoundException;

import java.net.InetAddress;

import mindbright.application.MindTerm;
import mindbright.security.KeyPair;
import mindbright.security.SecureRandom;
import mindbright.terminal.TerminalWin;
import mindbright.terminal.TerminalMenuListener;

public final class SSHMenuHandler extends Object implements ActionListener, ItemListener,
							    TerminalMenuListener {

  String aboutText = 
"Copyright (c) 1998,99 by Mindbright Technology AB, Stockholm, Sweden.\n" +
"\n" +
"This program is free software; you can redistribute it and/or modify " +
"it under the terms of the GNU General Public License as published by " +
"the Free Software Foundation; either version 2 of the License, or " +
"(at your option) any later version.\n" +
"\n" +
"This program is distributed in the hope that it will be useful, " +
"but WITHOUT ANY WARRANTY; without even the implied warranty of " +
"MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the " +
"GNU General Public License for more details.\n" +
"\n" +
"\tAuthor:\tMats Andersson (mats@mindbright.se)\n" +
"\tWeb:\thttp://www.mindbright.se/mindterm/\n" +
"\tInfo:\tmindterm@mindbright.se\n" +
"\tCVS " + SSH.CVS_NAME + "\n" +
"\tCVS " + SSH.CVS_DATE + "\n" +
"Running on:\n" +
"\tJava vendor:\t" + MindTerm.javaVendor  + "\n" +
"\tJava version:\t" + MindTerm.javaVersion  + "\n" +
"\tOS name:\t" + MindTerm.osName  + "\n" +
"\tOS architecture:\t" + MindTerm.osArch  + "\n" +
"\tOS version:\t" + MindTerm.osVersion  + "\n";

  final class CloseAction implements ActionListener {
      Dialog dialog;
      CloseAction(Dialog dialog) {
	this.dialog = dialog;
      }
      public void actionPerformed(ActionEvent e) {
	dialog.setVisible(false);
      }
  }

  protected final class ProgressBar extends Canvas {
    int   max     = 0;
    int   current = 0;
    int   width   = 0;
    int   height  = 0;
    Color barColor;
    public synchronized void setBarColor(Color c) {
      barColor = c;
    }
    public synchronized void setValue(int v) {
	current = v;
	repaint();
    }
    public Dimension getPreferredSize() {
      return new Dimension(width, height);
    }
    public ProgressBar(int max, int width, int height) {
      super();
      this.max    = max;
      this.width  = width;
      this.height = height;
      barColor    = Color.black;
    }

    public synchronized void paint(Graphics g) {
      FontMetrics fm   = g.getFontMetrics(g.getFont());
      double      perc = ((double)current * 100.0) / (double)max;
      String      p = ((int)perc) + "%";
      int         w = (current * (width - 2)) / max;
      setBackground(Color.white);
      g.drawRect(0, 0, width - 1, height - 1);
      g.drawString(p, (width / 2) - (fm.stringWidth(p) / 2) + 1, (height / 2) + fm.getMaxAscent() + fm.getLeading() - (fm.getHeight() / 2));
      g.setColor(barColor);
      g.setXORMode(getBackground());
      g.fillRect(1, 1, w, height - 2);
    }

  }

  protected class TunnelEditor extends Panel {
    List      list;
    TextField text;

    public TunnelEditor(String head, ActionListener alAdd, ActionListener alDel) {
      super(new BorderLayout(5, 5));

      Panel  pi;
      Button b;

      add(new Label(head), BorderLayout.NORTH);
      add(list = new List(5, false), BorderLayout.CENTER);
      pi = new Panel(new FlowLayout());
      pi.add(text = new TextField("", 26));
      pi.add(b = new Button("Add"));
      b.addActionListener(alAdd);
      pi.add(b = new Button("Delete"));
      b.addActionListener(alDel);
      add(pi, BorderLayout.SOUTH);
      list.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  text.setText(list.getSelectedItem());
	  text.requestFocus();
	}
      });
    }

    public int getItemCount() {
      return list.getItemCount();
    }

    public String getItem(int i) {
      return list.getItem(i);
    }

    public void addToList(String item) {
      list.add(item);
    }

    public int getSelectedIndex() {
      return list.getSelectedIndex();
    }

    public void selectText() {
      text.selectAll();
    }

    public String getText() {
      return text.getText();
    }

    public void removeAll() {
      list.removeAll();
    }
  }

  SSHInteractiveClient client;
  Frame                parent;
  TerminalWin          term;
  MindTerm             mindterm;

  boolean havePopupMenu = false;

  final static int MENU_FILE     = 0;
  final static int MENU_EDIT     = 1;
  final static int MENU_SETTINGS = 2;
  final static int MENU_TUNNELS  = 3;
  final static int MENU_HELP     = 4;

  final static int M_FILE_NEW     = 1;
  final static int M_FILE_CLONE   = 2;
  final static int M_FILE_CONN    = 3;
  final static int M_FILE_DISC    = 4;
  final static int M_FILE_LOAD    = 6;
  final static int M_FILE_SAVE    = 7;
  final static int M_FILE_SAVEAS  = 8;
  final static int M_FILE_CREATID = 10;
  final static int M_FILE_CAPTURE = 12;
  final static int M_FILE_SEND    = 13;
  final static int M_FILE_CLOSE   = 15;
  final static int M_FILE_EXIT    = 16;

  final static int M_EDIT_COPY    = 1;
  final static int M_EDIT_PASTE   = 2;
  final static int M_EDIT_CPPASTE = 3;
  final static int M_EDIT_SELALL  = 4;
  final static int M_EDIT_FIND    = 5;
  final static int M_EDIT_CLS     = 7;
  final static int M_EDIT_CLEARSB = 8;
  final static int M_EDIT_VTRESET = 9;

  final static int M_SET_SSH      = 1;
  final static int M_SET_TERM     = 2;
  final static int M_SET_MISC     = 3;
  final static int M_SET_CMDSH    = 5;
  final static int M_SET_AUTOSAVE = 7;
  final static int M_SET_AUTOLOAD = 8;

  final static int M_TUNL_SIMPLE   = 1;
  final static int M_TUNL_ADVANCED = 2;
  final static int M_TUNL_WIZARD   = 4;
  final static int M_TUNL_CURRENT  = 5;

  final static int M_HELP_TOPICS  = 1;
  final static int M_HELP_ABOUT   = 2;

  final static String[][] menuTexts = {
      { "File", 
	"New Terminal", "Clone Terminal", "Connect...", "Disconnect", null,
	"Load Settings...", "Save Settings", "Save Settings As...", null,
	"Create RSA Identity...", null,
	"_Capture To File...", "Send ASCII File...", null, "Close", "Exit"
      },

      { "Edit",
	"Copy Ctrl+Ins", "Paste Shift+Ins", "Copy & Paste", "Select All", "Find...", null,
	"Clear Screen", "Clear Scrollback", "VT Reset"
      },

      { "Settings",
	"SSH Connection...", "Terminal...", "Terminal Misc...", null,
	"Local Command-Shell", null,
	"_Auto Save Settings", "_Auto Load Settings"
      },

      { "Tunnels",
	"Basic...", "Advanced...", null, "Tunnel Wizard...", "Current Connections..."
      },

      { "Help",
	"Help Topics...", "About MindTerm"
      },
  };

  final static int NO_SHORTCUT = -1;
  final static int[][] menuShortCuts = {
    { NO_SHORTCUT, KeyEvent.VK_N, KeyEvent.VK_O, KeyEvent.VK_C, KeyEvent.VK_D,
      NO_SHORTCUT, NO_SHORTCUT, KeyEvent.VK_S, NO_SHORTCUT, NO_SHORTCUT, NO_SHORTCUT, 
      NO_SHORTCUT, NO_SHORTCUT, NO_SHORTCUT, NO_SHORTCUT, KeyEvent.VK_E, KeyEvent.VK_X },

    { NO_SHORTCUT, NO_SHORTCUT, NO_SHORTCUT, NO_SHORTCUT, KeyEvent.VK_A, KeyEvent.VK_F,
      NO_SHORTCUT, NO_SHORTCUT, NO_SHORTCUT, NO_SHORTCUT, NO_SHORTCUT, NO_SHORTCUT,
      NO_SHORTCUT, NO_SHORTCUT, NO_SHORTCUT, NO_SHORTCUT, NO_SHORTCUT, NO_SHORTCUT },

    { NO_SHORTCUT, KeyEvent.VK_H, KeyEvent.VK_T, KeyEvent.VK_M, NO_SHORTCUT, NO_SHORTCUT, 
      NO_SHORTCUT, NO_SHORTCUT, NO_SHORTCUT, NO_SHORTCUT, NO_SHORTCUT, NO_SHORTCUT, 
      NO_SHORTCUT, NO_SHORTCUT, NO_SHORTCUT, NO_SHORTCUT, NO_SHORTCUT, NO_SHORTCUT },

    { NO_SHORTCUT, NO_SHORTCUT, NO_SHORTCUT, NO_SHORTCUT, NO_SHORTCUT, NO_SHORTCUT, 
      NO_SHORTCUT, NO_SHORTCUT, NO_SHORTCUT, NO_SHORTCUT, NO_SHORTCUT, NO_SHORTCUT, 
      NO_SHORTCUT, NO_SHORTCUT, NO_SHORTCUT, NO_SHORTCUT, NO_SHORTCUT, NO_SHORTCUT },

    { NO_SHORTCUT, NO_SHORTCUT, NO_SHORTCUT, NO_SHORTCUT, NO_SHORTCUT, NO_SHORTCUT, 
      NO_SHORTCUT, NO_SHORTCUT, NO_SHORTCUT, NO_SHORTCUT, NO_SHORTCUT, NO_SHORTCUT, 
      NO_SHORTCUT, NO_SHORTCUT, NO_SHORTCUT, NO_SHORTCUT, NO_SHORTCUT, NO_SHORTCUT },

  };

  Object[][] menuItems;

  public SSHMenuHandler(MindTerm mindterm, SSHInteractiveClient client, Frame parent, TerminalWin term) {
    this.mindterm = mindterm;
    this.client   = client;
    this.parent   = parent;
    this.term     = term;
  }

  int popButtonNum = 3;
  public void setPopupButton(int popButtonNum) {
    term.setPopupButton(popButtonNum);
    this.popButtonNum = popButtonNum;
  }

  public int getPopupButton() {
    return popButtonNum;
  }

  Menu getMenu(int idx) {
    Menu m = new Menu(menuTexts[idx][0]);
    int len = menuTexts[idx].length;
    MenuItem mi;
    String   t;

    if(menuItems == null)
      menuItems = new Object[menuTexts.length][];
    if(menuItems[idx] == null)
      menuItems[idx] = new Object[menuTexts[idx].length];

    for(int i = 1; i < len; i++) {
      t = menuTexts[idx][i];
      if(t == null) {
	m.addSeparator();
	continue;
      }
      if(t.charAt(0) == '_') {
	t = t.substring(1);
	mi = new CheckboxMenuItem(t);
	((CheckboxMenuItem)mi).addItemListener(this);
      } else {
	mi = new MenuItem(t);
	mi.addActionListener(this);
      }

      if(menuShortCuts[idx][i] != NO_SHORTCUT) {
	mi.setShortcut(new MenuShortcut(menuShortCuts[idx][i], true));
      }

      menuItems[idx][i] = mi;
      m.add(mi);
    }
    return m;
  }

  int[] mapAction(String action) {
    int[] id = new int[2];
    int i = 0, j = 0;

    for(i = 0; i < menuTexts.length; i++) {
      for(j = 1; j < menuTexts[i].length; j++) {
	String mt = menuTexts[i][j];
	if(mt != null && action.equals(mt)) {
	    id[0] = i;
	    id[1] = j;
	    i = menuTexts.length;
	    break;
	}
      }
    }
    return id;
  }

  public void actionPerformed(ActionEvent e) {
    int[] id = mapAction(((MenuItem)(e.getSource())).getLabel());
    handleMenuAction(id);
  }

  public void itemStateChanged(ItemEvent e) {
    int[] id = mapAction("_" + (String)e.getItem());
    handleMenuAction(id);
  }

  public void handleMenuAction(int[] id) {
    switch(id[0]) {
    case MENU_FILE:
      switch(id[1]) {
      case M_FILE_NEW:
	mindterm.newWindow();
	break;
      case M_FILE_CLONE:
	mindterm.cloneWindow();
	break;
      case M_FILE_CONN:
	connectDialog();
	break;
      case M_FILE_DISC:
	client.forcedDisconnect();
	client.quiet = false;
	break;
      case M_FILE_LOAD:
	loadFileDialog();
	break;
      case M_FILE_SAVE:
	try {
	  client.saveCurrentProperties();
	} catch (Throwable t) {
	  alertDialog("Error saving settings: " + t.getMessage());
	}
	break;
      case M_FILE_SAVEAS:
	saveAsFileDialog();
	break;
      case M_FILE_CREATID:
	keyGenerationDialog();
	break;
      case M_FILE_CAPTURE:
	if(((CheckboxMenuItem)menuItems[MENU_FILE][M_FILE_CAPTURE]).getState()) {
	  if(!captureToFileDialog()) {
	    ((CheckboxMenuItem)menuItems[MENU_FILE][M_FILE_CAPTURE]).setState(false);
	  }
	} else {
	  endCapture();
	}
	break;
      case M_FILE_SEND:
	sendFileDialog();
	break;
      case M_FILE_CLOSE:
	mindterm.close();
	break;
      case M_FILE_EXIT:
	mindterm.exit();
	break;
      }
      break;

    case MENU_EDIT:
      switch(id[1]) {
      case M_EDIT_COPY:
	term.doCopy();
	break;
      case M_EDIT_PASTE:
	term.doPaste();
	break;
      case M_EDIT_CPPASTE:
	term.doCopy();
	term.doPaste();
	break;
      case M_EDIT_SELALL:
	term.selectAll();
	break;
      case M_EDIT_FIND:
	term.getMenus().findDialog();
	break;
      case M_EDIT_CLS:
	term.clearScreen();
	term.cursorSetPos(0, 0, false);
	break;
      case M_EDIT_CLEARSB:
	term.clearSaveLines();
	break;
      case M_EDIT_VTRESET:
	term.resetInterpreter();
	break;
      }
      break;

    case MENU_SETTINGS:
      switch(id[1]) {
      case M_SET_SSH:
	sshSettingsDialog();
	break;
      case M_SET_TERM:
	term.getMenus().termSettingsDialog();
	break;
      case M_SET_MISC:
	term.getMenus().termSettingsDialog2();
	break;
      case M_SET_CMDSH:
	client.console.println("");
	client.console.println("** hit a key to enter local command-shell **");
	client.sshStdIO.wantCommandShell();
	break;
      case M_SET_AUTOSAVE:
	client.autoSaveProps =
	    ((CheckboxMenuItem)menuItems[MENU_SETTINGS][M_SET_AUTOSAVE]).getState();
	break;
      case M_SET_AUTOLOAD:
	client.autoLoadProps =
	    ((CheckboxMenuItem)menuItems[MENU_SETTINGS][M_SET_AUTOLOAD]).getState();
	break;
      }
      break;

    case MENU_TUNNELS:
      switch(id[1]) {
      case M_TUNL_SIMPLE:
	basicTunnelsDialog();
	break;
      case M_TUNL_ADVANCED:
	advancedTunnelsDialog();
	break;
      case M_TUNL_WIZARD:
	break;
      case M_TUNL_CURRENT:
	currentTunnelsDialog();
	break;
      }
      break;

    case MENU_HELP:
      switch(id[1]) {
      case M_HELP_TOPICS:
	alertDialog("No help available yet, be patient :-)");
	break;
      case M_HELP_ABOUT:
	textDialog("About " + SSH.VER_MINDTERM, aboutText, 15, 60, true);
	break;
      }
      break;
    }
  }

  public void update() {
    if(client.isOpened) {
      ((MenuItem)menuItems[MENU_SETTINGS][M_SET_CMDSH]).setEnabled(client.sshStdIO.hasCommandShell());
      ((MenuItem)menuItems[MENU_TUNNELS][M_TUNL_CURRENT]).setEnabled(true);
    } else {
      ((MenuItem)menuItems[MENU_SETTINGS][M_SET_CMDSH]).setEnabled(false);
      ((MenuItem)menuItems[MENU_TUNNELS][M_TUNL_CURRENT]).setEnabled(false);
    }
    if(client.isConnected) {
      ((MenuItem)menuItems[MENU_FILE][M_FILE_CONN]).setEnabled(false);
      ((MenuItem)menuItems[MENU_FILE][M_FILE_DISC]).setEnabled(true);
      ((MenuItem)menuItems[MENU_FILE][M_FILE_LOAD]).setEnabled(false);
    } else {
      ((MenuItem)menuItems[MENU_FILE][M_FILE_CONN]).setEnabled(true);
      ((MenuItem)menuItems[MENU_FILE][M_FILE_DISC]).setEnabled(false);
      ((MenuItem)menuItems[MENU_FILE][M_FILE_LOAD]).setEnabled(true);
    }

    if(client.getSSHHomeDir() == null) {

    } else {

    }

    ((MenuItem)menuItems[MENU_FILE][M_FILE_SAVE]).setEnabled(client.wantSave());
    ((MenuItem)menuItems[MENU_FILE][M_FILE_SAVEAS]).setEnabled(client.canSaveAs());

    if(client.controller != null)
      ((MenuItem)menuItems[MENU_FILE][M_FILE_SEND]).setEnabled(true);
    else
      ((MenuItem)menuItems[MENU_FILE][M_FILE_SEND]).setEnabled(false);

    if(client.currentPropsFile != null && client.propsHaveChanged()) {
      ((MenuItem)menuItems[MENU_FILE][M_FILE_SAVE]).setEnabled(true);
    } else {
      ((MenuItem)menuItems[MENU_FILE][M_FILE_SAVE]).setEnabled(false);
    }

    ((CheckboxMenuItem)menuItems[MENU_SETTINGS][M_SET_AUTOSAVE]).setState(client.autoSaveProps);
    ((CheckboxMenuItem)menuItems[MENU_SETTINGS][M_SET_AUTOLOAD]).setState(client.autoLoadProps);

    boolean selAvail = client.sshStdIO.selectionAvailable;
    ((MenuItem)menuItems[MENU_EDIT][M_EDIT_COPY]).setEnabled(selAvail);
    ((MenuItem)menuItems[MENU_EDIT][M_EDIT_CPPASTE]).setEnabled(selAvail);
    

    ((MenuItem)menuItems[MENU_TUNNELS][M_TUNL_WIZARD]).setEnabled(false);
  }

  public void prepareMenuBar(MenuBar mb) {
    mb.add(getMenu(0));
    mb.add(getMenu(1));
    mb.add(getMenu(2));
    mb.add(term.getMenus().getOptionsMenu());
    mb.add(getMenu(3));
    mb.setHelpMenu(getMenu(4));

    update();
  }

  public void preparePopupMenu(PopupMenu popupmenu) {
    havePopupMenu = true;
    popupmenu.add(getMenu(0));
    popupmenu.add(getMenu(1));
    popupmenu.add(getMenu(2));
    popupmenu.add(term.getMenus().getOptionsMenu());
    popupmenu.add(getMenu(3));
    popupmenu.addSeparator();
    popupmenu.add(getMenu(4));
    update();
  }

  void placeDialog(Dialog diag) {
    Dimension sDim = Toolkit.getDefaultToolkit().getScreenSize();
    Dimension mDim = diag.getSize();
    int x, y;
    x = ((sDim.width / 2) - (mDim.width / 2));
    y = ((sDim.height / 2) - (mDim.height / 2));
    diag.setLocation(x, y);
  }

  Dialog     settingsDialog = null;
  Choice     choiceCipher, choiceAuthTyp;
  Checkbox   cbX11, cbPrvPrt, cbRemFwd, cbIdHost, cbPortFtp, cbLocHst, cbMTU, cbAlive, cbForcPty;
  TextField  textPort, textUser, textId, textDisp, textMtu, textAlive, textSrv,
      textRealAddr, textAuthList, textLocHost;
  Label      lblAlert;
  String[]   cipher, authtyp, bool;
  FileDialog idFileFD;
  Button     idFileBut, advButton;
  boolean    newServer, advanced = false;
  Panel      ap;
  public final void sshSettingsDialog() {
    int   i;

    if(settingsDialog == null) {
      cipher  = SSH.getCiphers();
      authtyp = SSH.getAuthTypeList();

      settingsDialog = new Dialog(parent, true);

      Label              lbl;
      GridBagLayout      grid  = new GridBagLayout();
      GridBagConstraints gridc = new GridBagConstraints();
      settingsDialog.setLayout(grid);
      Button             b;
      ItemListener       il;

      gridc.fill   = GridBagConstraints.HORIZONTAL;
      gridc.anchor = GridBagConstraints.WEST;
      gridc.gridwidth = 1;
      gridc.insets = new Insets(4, 4, 0, 4);

      gridc.gridy = 0;
      lbl = new Label("Server:");
      grid.setConstraints(lbl, gridc);
      settingsDialog.add(lbl);
      gridc.gridwidth = 3;
      textSrv = new TextField("", 16);
      grid.setConstraints(textSrv, gridc);
      settingsDialog.add(textSrv);
      gridc.gridwidth = 1;
      lbl = new Label("Port:");
      grid.setConstraints(lbl, gridc);
      settingsDialog.add(lbl);
      textPort = new TextField("", 4);
      grid.setConstraints(textPort, gridc);
      settingsDialog.add(textPort);

      gridc.gridy = 1;
      gridc.gridwidth = 2;
      lbl = new Label("Username:");
      grid.setConstraints(lbl, gridc);
      settingsDialog.add(lbl);
      textUser = new TextField("", 10);
      grid.setConstraints(textUser, gridc);
      settingsDialog.add(textUser);
    
      lbl = new Label("Cipher:");
      gridc.gridwidth = 1;
      grid.setConstraints(lbl, gridc);
      settingsDialog.add(lbl);
      choiceCipher = new Choice();
      for(i = 0; i < cipher.length; i++) {
	choiceCipher.add(cipher[i]);
      }
      grid.setConstraints(choiceCipher, gridc);
      settingsDialog.add(choiceCipher);

      gridc.gridy = 4;
      gridc.gridwidth = 2;
      lbl = new Label("Authentication:");
      grid.setConstraints(lbl, gridc);
      settingsDialog.add(lbl);
      choiceAuthTyp = new Choice();
      for(i = 0; i < authtyp.length; i++) {
	choiceAuthTyp.add(authtyp[i]);
      }
      gridc.gridwidth = 1;
      grid.setConstraints(choiceAuthTyp, gridc);
      settingsDialog.add(choiceAuthTyp);
      choiceAuthTyp.addItemListener(il = new ItemListener() {
	public void itemStateChanged(ItemEvent e) {
	  updateChoices();
	}
      });
      gridc.gridwidth = 3;
      textAuthList = new TextField("", 10);
      grid.setConstraints(textAuthList, gridc);
      settingsDialog.add(textAuthList);

      gridc.gridy = 5;
      gridc.gridwidth = 2;
      lbl = new Label("Identity:");
      grid.setConstraints(lbl, gridc);
      settingsDialog.add(lbl);
      gridc.gridwidth = 3;
      textId = new TextField("", 16);
      grid.setConstraints(textId, gridc);
      settingsDialog.add(textId);
      idFileBut = new Button("...");
      idFileBut.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  if(idFileFD == null) {
	    idFileFD = new FileDialog(parent, "MindTerm - Select file with identity (private)", FileDialog.LOAD);
	    idFileFD.setDirectory(client.getSSHHomeDir());
	  }
	  idFileFD.setVisible(true);
	  if(idFileFD.getFile() != null && idFileFD.getFile().length() > 0)
	    textId.setText(idFileFD.getDirectory() + idFileFD.getFile());
	}
      });
      gridc.fill      = GridBagConstraints.NONE;
      gridc.gridwidth = 1;
      grid.setConstraints(idFileBut, gridc);
      settingsDialog.add(idFileBut);

      gridc.gridy = 6;
      gridc.fill      = GridBagConstraints.NONE;
      gridc.gridwidth = GridBagConstraints.REMAINDER;
      gridc.anchor = GridBagConstraints.EAST;
      gridc.insets = new Insets(4, 0, 4, 8);
      advButton = new Button("More options...");
      advButton.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  if(advanced) {
	    advButton.setLabel("More options...");
	    settingsDialog.remove(ap);
	  } else {
	    advButton.setLabel("Less options...");
	    settingsDialog.add(ap);
	  }
	  settingsDialog.pack();
	  advanced = !advanced;
	}
      });
      grid.setConstraints(advButton, gridc);
      settingsDialog.add(advButton);


      ap = new Panel();
      GridBagLayout      grid2  = new GridBagLayout();
      GridBagConstraints gridc2 = new GridBagConstraints();
      gridc2.fill = GridBagConstraints.NONE;
      gridc2.gridwidth = 2;
      gridc2.anchor = GridBagConstraints.WEST;
      gridc2.insets = new Insets(4, 4, 0, 4);
      ap.setLayout(grid2);

      gridc2.gridy = 0;
      gridc2.gridwidth = 1;
      cbX11 = new Checkbox("X11 forward");
      grid2.setConstraints(cbX11, gridc2);
      ap.add(cbX11);
      cbX11.addItemListener(il);
      lbl = new Label("Local X11-display:");
      gridc2.gridwidth = 3;
      grid2.setConstraints(lbl, gridc2);
      ap.add(lbl);
      textDisp = new TextField("", 12);
      gridc2.gridwidth = 1;
      grid2.setConstraints(textDisp, gridc2);
      ap.add(textDisp);

      gridc2.gridy = 1;
      cbMTU = new Checkbox("Set MTU");
      gridc2.gridwidth = 1;
      grid2.setConstraints(cbMTU, gridc2);
      ap.add(cbMTU);
      lbl = new Label("Max. packet size:");
      gridc2.gridwidth = 3;
      grid2.setConstraints(lbl, gridc2);
      ap.add(lbl);
      textMtu = new TextField("", 12);
      gridc2.gridwidth = 1;
      grid2.setConstraints(textMtu, gridc2);
      ap.add(textMtu);
      cbMTU.addItemListener(il);

      gridc2.gridy = 2;
      cbAlive = new Checkbox("Send keep-alive");
      gridc2.gridwidth = 1;
      grid2.setConstraints(cbAlive, gridc2);
      ap.add(cbAlive);
      lbl = new Label("Interval (seconds):");
      gridc2.gridwidth = 3;
      grid2.setConstraints(lbl, gridc2);
      ap.add(lbl);
      textAlive = new TextField("", 12);
      gridc2.gridwidth = 1;
      grid2.setConstraints(textAlive, gridc2);
      ap.add(textAlive);
      cbAlive.addItemListener(il);

      gridc2.gridy = 3;
      cbPortFtp = new Checkbox("Enable ftp PORT");
      gridc2.gridwidth = 1;
      grid2.setConstraints(cbPortFtp, gridc2);
      ap.add(cbPortFtp);
      lbl = new Label("Real sshd address:");
      gridc2.gridwidth = 3;
      grid2.setConstraints(lbl, gridc2);
      ap.add(lbl);
      textRealAddr = new TextField("", 12);
      gridc2.gridwidth = 1;
      grid2.setConstraints(textRealAddr, gridc2);
      ap.add(textRealAddr);
      cbPortFtp.addItemListener(il);

      gridc2.gridy = 4;
      cbLocHst = new Checkbox("Set localhost");
      gridc2.gridwidth = 1;
      grid2.setConstraints(cbLocHst, gridc2);
      ap.add(cbLocHst);
      lbl = new Label("Localhost address:");
      gridc2.gridwidth = 3;
      grid2.setConstraints(lbl, gridc2);
      ap.add(lbl);
      textLocHost = new TextField("", 12);
      gridc2.gridwidth = 1;
      grid2.setConstraints(textLocHost, gridc2);
      ap.add(textLocHost);
      cbLocHst.addItemListener(il);

      gridc2.gridy = 5;
      gridc2.gridwidth = 2;
      gridc2.insets = new Insets(16, 4, 0, 4);

      cbIdHost = new Checkbox("Verify server key");
      grid2.setConstraints(cbIdHost, gridc2);
      ap.add(cbIdHost);

      cbForcPty = new Checkbox("Allocate PTY");
      grid2.setConstraints(cbForcPty, gridc2);
      ap.add(cbForcPty);

      gridc2.gridy = 6;
      cbPrvPrt = new Checkbox("Priv. source port");
      grid2.setConstraints(cbPrvPrt, gridc2);
      ap.add(cbPrvPrt);

      gridc2.gridwidth = 3;
      cbRemFwd = new Checkbox("Allow remote connects");
      grid2.setConstraints(cbRemFwd, gridc2);
      ap.add(cbRemFwd);



      gridc.gridy = 7;
      gridc.insets = new Insets(0, 0, 0, 0);
      gridc.anchor = GridBagConstraints.CENTER;
      gridc.fill = GridBagConstraints.NONE;
      gridc.gridwidth = GridBagConstraints.REMAINDER;
      grid.setConstraints(ap, gridc);


      gridc.insets = new Insets(0, 0, 0, 0);
      lblAlert = new Label("", Label.CENTER);
      gridc.gridy = 12;
      gridc.fill   = GridBagConstraints.HORIZONTAL;
      grid.setConstraints(lblAlert, gridc);
      settingsDialog.add(lblAlert);

      Panel bp = new Panel(new FlowLayout());

      bp.add(b = new Button("OK"));
      b.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  try {
	    String host = null;
	    if(newServer) {
	      host = textSrv.getText();
	      if(host.length() == 0) {
		lblAlert.setText("Please specify a server to connect to");
		return;
	      }
	      client.quiet = true;
	      try {
		client.setPropertyFileAndLoad(host, false);
	      } catch (Throwable t) {
		alertDialog("Error accessing settings file: " + t.getMessage());
	      }
	      client.setProperty("server", host);
	    }

	    String authType = choiceAuthTyp.getSelectedItem();
	    if(authType.equals("custom...")) {
	      client.setProperty("authtyp", textAuthList.getText());
	    } else {
	      client.setProperty("authtyp", authType);
	    }
	    client.setProperty("port", textPort.getText());
	    client.setProperty("usrname", textUser.getText());
	    client.setProperty("cipher", cipher[choiceCipher.getSelectedIndex()]);
	    client.setProperty("idfile", textId.getText());
	    client.setProperty("display", textDisp.getText());
	    client.setProperty("mtu", textMtu.getText());
	    client.setProperty("x11fwd", String.valueOf(cbX11.getState()));
	    client.setProperty("prvport", String.valueOf(cbPrvPrt.getState()));
	    client.setProperty("remfwd", String.valueOf(cbRemFwd.getState()));
	    client.setProperty("idhost", String.valueOf(cbIdHost.getState()));
	    client.setProperty("forcpty", String.valueOf(cbForcPty.getState()));
	    client.setProperty("portftp", String.valueOf(cbPortFtp.getState()));
	    client.setProperty("localhst", String.valueOf(textLocHost.getText()));
	    if(cbPortFtp.getState())
	      client.setProperty("realsrv", textRealAddr.getText());
	    else
	      client.setProperty("realsrv", "");
	    client.setProperty("alive", textAlive.getText());

	    if(newServer)
	      client.sshStdIO.breakPromptLine("Connecting to: " + host);

	    settingsDialog.setVisible(false);
	  } catch (Exception ee) {
	    lblAlert.setText(ee.getMessage());
	  }
	}
      });
      bp.add(b = new Button("Cancel"));
      b.addActionListener(new CloseAction(settingsDialog));

      gridc.gridy = 13;
      grid.setConstraints(bp, gridc);
      settingsDialog.add(bp);

      setBackgroundOfChildren(settingsDialog);

      settingsDialog.setResizable(false);
      settingsDialog.pack();
    }

    String srv = client.getProperty("server");
    if(srv.length() == 0 || !client.isConnected) {
      textSrv.setEnabled(true);
      settingsDialog.setTitle("MindTerm - New Server");
      newServer = true;
      client.clearServerSetting();
    } else {
      textSrv.setEnabled(false);
      newServer = false;
      settingsDialog.setTitle("MindTerm - SSH settings: " + client.currentPropsName);
    }
    textSrv.setText(srv);

    textPort.setText(client.getProperty("port"));

    textUser.setText(client.getProperty("usrname"));

    choiceCipher.select(client.getProperty("cipher"));

    String at = client.getProperty("authtyp");
    if(at.indexOf(',') == -1) {
      choiceAuthTyp.select(at);
    } else {
      choiceAuthTyp.select("custom...");
      textAuthList.setText(at);
    }

    textId.setText(client.getProperty("idfile"));

    textDisp.setText(client.getProperty("display"));
    textMtu.setText(client.getProperty("mtu"));
    textAlive.setText(client.getProperty("alive"));

    InetAddress realAddr = client.getServerRealAddr();
    if(realAddr != null)
      textRealAddr.setText(realAddr.getHostAddress());

    cbX11.setState(Boolean.valueOf(client.getProperty("x11fwd")).booleanValue());
    cbMTU.setState(!client.getProperty("mtu").equals("0"));
    cbAlive.setState(!client.getProperty("alive").equals("0"));

    cbLocHst.setState(!client.getProperty("localhst").equals("0.0.0.0"));
    textLocHost.setEnabled(false);

    cbPrvPrt.setState(Boolean.valueOf(client.getProperty("prvport")).booleanValue());
    cbRemFwd.setState(Boolean.valueOf(client.getProperty("remfwd")).booleanValue());
    cbIdHost.setState(Boolean.valueOf(client.getProperty("idhost")).booleanValue());
    cbPortFtp.setState(Boolean.valueOf(client.getProperty("portftp")).booleanValue());
    cbForcPty.setState(Boolean.valueOf(client.getProperty("forcpty")).booleanValue());

    updateChoices();

    lblAlert.setText("");

    placeDialog(settingsDialog);
    if(textSrv.isEnabled())
      textSrv.requestFocus();
    else
      textUser.requestFocus();
    settingsDialog.setVisible(true);
  }

  void updateChoices() {
    String at = choiceAuthTyp.getSelectedItem();
    if(at.equals("rsa") || at.equals("rhostsrsa") || at.equals("custom...")) {
      textId.setEnabled(true);
      idFileBut.setEnabled(true);
    } else {
      textId.setEnabled(false);
      idFileBut.setEnabled(false);
    }
    if(at.equals("custom...")) {
      textAuthList.setEnabled(true);
    } else {
      textAuthList.setEnabled(false);
      textAuthList.setText("");
    }

    textDisp.setEnabled(cbX11.getState());
    textMtu.setEnabled(cbMTU.getState());
    if(!cbMTU.getState())
      textMtu.setText("0");

    if(!textAlive.isEnabled() && cbAlive.getState()) {
      textAlive.setText("10");
    } else if(!textAlive.isEnabled()) {
      textAlive.setText("0");
    }
    textAlive.setEnabled(cbAlive.getState());

    textRealAddr.setEnabled(cbPortFtp.getState());

    if(cbLocHst.getState()) {
      if(!textLocHost.isEnabled())
	textLocHost.setText(client.getProperty("localhst"));
    } else {
      textLocHost.setText("0.0.0.0");
    }
    textLocHost.setEnabled(cbLocHst.getState());

  }

  Dialog currentTunnelsDialog = null;
  List   currList;
  public final void currentTunnelsDialog() {
    if(currentTunnelsDialog == null) {
      currentTunnelsDialog = new Dialog(parent, "MindTerm - Currently Open Tunnels", false);

      GridBagLayout       grid  = new GridBagLayout();
      GridBagConstraints  gridc = new GridBagConstraints();
      Label               label;
      Button              b;
      ActionListener      al;

      currentTunnelsDialog.setLayout(grid);

      gridc.fill      = GridBagConstraints.NONE;
      gridc.anchor    = GridBagConstraints.WEST;
      gridc.gridy     = 0;
      gridc.insets    = new Insets(10, 10, 0, 10);
      gridc.gridwidth = 2;

      label = new Label("Currently open tunnels:");
      grid.setConstraints(label, gridc);
      currentTunnelsDialog.add(label);

      gridc.fill      = GridBagConstraints.BOTH;
      gridc.anchor    = GridBagConstraints.WEST;
      gridc.insets    = new Insets(10, 10, 10, 10);
      gridc.gridy     = 1;
      gridc.gridwidth = 10;

      currList = new List(8);
      grid.setConstraints(currList, gridc);
      currentTunnelsDialog.add(currList);

      Panel bp = new Panel(new FlowLayout());
      bp.add(b = new Button("Close Tunnel"));
      b.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  int i = currList.getSelectedIndex();
	  if(i == -1) {
	    term.doBell();
	    return;
	  }
	  // !!! Ouch !!!
	  client.controller.closeTunnelFromList(i);
	  Thread.yield();
	  refreshCurrList();
	}
      });
      bp.add(b = new Button("Refresh"));
      b.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  refreshCurrList();
	}
      });
      bp.add(b = new Button("Close Dialog"));
      b.addActionListener(new CloseAction(currentTunnelsDialog));

      gridc.gridy     = 2;
      gridc.gridwidth = GridBagConstraints.REMAINDER;
      gridc.anchor = GridBagConstraints.CENTER;
      grid.setConstraints(bp, gridc);
      currentTunnelsDialog.add(bp);

      setBackgroundOfChildren(currentTunnelsDialog);

      currentTunnelsDialog.setResizable(true);
      currentTunnelsDialog.pack();
    }
    refreshCurrList();

    placeDialog(currentTunnelsDialog);
    currList.requestFocus();
    currentTunnelsDialog.setVisible(true);
  }

  void refreshCurrList() {
    currList.removeAll();
    String[] l = client.controller.listTunnels();
    for(int i = 0; i < l.length; i++) {
      currList.add(l[i]);
    }
    if(l.length > 0)
      currList.select(0);
  }

  Dialog    basicTunnelsDialog = null;
  List      tunnelList;
  TextField remoteHost, remotePort, localPort;
  Choice    protoChoice;
  final static String[] protos = { "general", "ftp", "telnet", "smtp", "http", "pop2", "pop3", "nntp", "imap" };
  final static int[]    servs  = {  0, 21, 23, 25, 80, 109, 110, 119, 143 };
  public final void basicTunnelsDialog() {
    if(basicTunnelsDialog == null) {
      basicTunnelsDialog = new Dialog(parent, "MindTerm - Basic Tunnels Setup", true);

      GridBagLayout       grid  = new GridBagLayout();
      GridBagConstraints  gridc = new GridBagConstraints();
      Label               lbl;
      Button              b;
      ActionListener      al;

      basicTunnelsDialog.setLayout(grid);

      gridc.fill      = GridBagConstraints.NONE;
      gridc.anchor    = GridBagConstraints.WEST;
      gridc.gridy     = 0;
      gridc.insets    = new Insets(4, 4, 0, 4);

      lbl = new Label("Current local tunnels:");
      gridc.gridwidth = 2;
      grid.setConstraints(lbl, gridc);
      basicTunnelsDialog.add(lbl);

      gridc.fill      = GridBagConstraints.BOTH;
      gridc.anchor    = GridBagConstraints.WEST;
      gridc.insets    = new Insets(4, 4, 4, 4);
      gridc.gridwidth = 4;
      gridc.gridy     = 1;

      tunnelList = new List(8);
      grid.setConstraints(tunnelList, gridc);
      basicTunnelsDialog.add(tunnelList);
      tunnelList.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  int i = tunnelList.getSelectedIndex();
	  if(i != -1) {
	    SSHClient.LocalForward fwd = (SSHClient.LocalForward) client.localForwards.elementAt(i);
  	    localPort.setText(String.valueOf(fwd.localPort));
  	    remotePort.setText(String.valueOf(fwd.remotePort));
	    remoteHost.setText(fwd.remoteHost);
	    for(i = 1; i < servs.length; i++) {
	      if(fwd.remotePort == servs[i]) {
		protoChoice.select(protos[i]);
		break;
	      }
	    }
	    if(i == servs.length)
	      protoChoice.select("general");
	  }
	}
      });

      gridc.fill      = GridBagConstraints.NONE;
      gridc.gridy     = 2;
      gridc.gridwidth = 1;

      lbl = new Label("Local port:");
      grid.setConstraints(lbl, gridc);
      basicTunnelsDialog.add(lbl);
      localPort = new TextField("", 5);
      grid.setConstraints(localPort, gridc);
      basicTunnelsDialog.add(localPort);

      lbl = new Label("Protocol:");
      grid.setConstraints(lbl, gridc);
      basicTunnelsDialog.add(lbl);
      protoChoice = new Choice();
      for(int i = 0; i < protos.length; i++) {
	protoChoice.add(protos[i]);
      }
      protoChoice.select("general");
      grid.setConstraints(protoChoice, gridc);
      basicTunnelsDialog.add(protoChoice);
      protoChoice.addItemListener(new ItemListener() {
	public void itemStateChanged(ItemEvent e) {
	  String it = (String)e.getItem();
	  int i;
	  for(i = 0; i < protos.length; i++)
	    if(it.equals(protos[i]))
	      break;
	  if(i > 0) {
	    remotePort.setText(String.valueOf(servs[i]));
	  }
	}
      });

      gridc.gridy     = 3;
      lbl = new Label("Remote host:");
      grid.setConstraints(lbl, gridc);
      basicTunnelsDialog.add(lbl);
      gridc.fill      = GridBagConstraints.HORIZONTAL;
      gridc.gridwidth = 3;
      remoteHost = new TextField("", 16);
      grid.setConstraints(remoteHost, gridc);
      basicTunnelsDialog.add(remoteHost);
      gridc.gridy     = 4;
      gridc.gridwidth = 1;
      lbl = new Label("Remote port:");
      grid.setConstraints(lbl, gridc);
      basicTunnelsDialog.add(lbl);
      remotePort = new TextField("", 5);
      grid.setConstraints(remotePort, gridc);
      basicTunnelsDialog.add(remotePort);

      b = new Button("Add");
      b.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  String rh = remoteHost.getText();
	  String plug = "general";
	  int    lp = -1, rp = -1;
	  try {
	    lp = Integer.valueOf(localPort.getText()).intValue();
	    rp = Integer.valueOf(remotePort.getText()).intValue();
	    if(lp < 1 || lp > 65535) {
	      lp = -1;
	      throw new NumberFormatException();
	    }
	    if(rp < 1 || rp > 65535) {
	      rp = -1;
	      throw new NumberFormatException();
	    }
	  } catch (NumberFormatException ee) {
	    if(lp == -1) {
	      localPort.setText("");
	      localPort.requestFocus();
	    } else {
	      remotePort.setText("");
	      remotePort.requestFocus();
	    }
	    return;
	  }
	  if(protoChoice.getSelectedItem().equals("ftp"))
	    plug = "ftp";
	  try {
	    client.setProperty("local" + client.localForwards.size(),
			       "/" + plug + "/" + lp + ":" + rh + ":" +  rp);
	    if(client.isOpened)
	      alertDialog("Tunnel is now open and operational");
	  } catch (Throwable ee) {
	    alertDialog("Could not open tunnel: " + ee.getMessage());
	  }
	  updateTunnelList();
	}
      });
      gridc.fill = GridBagConstraints.HORIZONTAL;
      grid.setConstraints(b, gridc);
      basicTunnelsDialog.add(b);
      b = new Button("Delete");
      b.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  int i = tunnelList.getSelectedIndex();
	  if(i != -1) {
	    client.removeLocalTunnelAt(i, true);
	  }
	  updateTunnelList();
	}
      });
      grid.setConstraints(b, gridc);
      basicTunnelsDialog.add(b);
      
      b = new Button("Close Dialog");
      b.addActionListener(new CloseAction(basicTunnelsDialog));
      gridc.gridy     = 5;
      gridc.fill      = GridBagConstraints.NONE;
      gridc.gridwidth = GridBagConstraints.REMAINDER;
      gridc.anchor    = GridBagConstraints.CENTER;
      gridc.ipady     = 2;
      gridc.ipadx     = 2;
      grid.setConstraints(b, gridc);
      basicTunnelsDialog.add(b);

      setBackgroundOfChildren(basicTunnelsDialog);

      basicTunnelsDialog.setResizable(false);
      basicTunnelsDialog.pack();
    }
    updateTunnelList();

    placeDialog(basicTunnelsDialog);
    localPort.requestFocus();
    basicTunnelsDialog.setVisible(true);
  }

  void updateTunnelList() {
    tunnelList.removeAll();
    for(int i = 0; i < client.localForwards.size(); i++) {
      SSHClient.LocalForward fwd = (SSHClient.LocalForward) client.localForwards.elementAt(i);
      String plugStr = (fwd.plugin.equals("general") ? "" : " (plugin: " + fwd.plugin + ")");
      tunnelList.add("local: " + fwd.localPort + " -> remote: " + fwd.remoteHost + "/" +
		     fwd.remotePort + plugStr);
    }
  }

  TunnelEditor localEdit    = null;
  TunnelEditor remoteEdit   = null;
  Dialog       tunnelDialog = null;
  public final void advancedTunnelsDialog() {
    if(tunnelDialog == null) {
      tunnelDialog = new Dialog(parent, "MindTerm - Advanced Tunnels Setup", true);

      GridBagLayout       grid  = new GridBagLayout();
      GridBagConstraints  gridc = new GridBagConstraints();
      tunnelDialog.setLayout(grid);

      gridc.fill      = GridBagConstraints.HORIZONTAL;
      gridc.anchor    = GridBagConstraints.WEST;
      gridc.insets    = new Insets(4, 8, 4, 8);

      gridc.gridy     = 0;

      localEdit  = new TunnelEditor("Local: ([/plug/][<loc-host>]:<loc-port>:<rem-host>:<rem-port>)",
				    new ActionListener() {
					public void actionPerformed(ActionEvent e) {
					    try {
						client.setProperty("local" + client.localForwards.size(), localEdit.getText());
						updateAdvancedTunnelLists();
					    } catch (Exception e1) {
						localEdit.selectText();
					    }
					}
				    },
				    new ActionListener() {
					public void actionPerformed(ActionEvent e) {
					    int i = localEdit.getSelectedIndex();
					    if(i != -1) {
						client.removeLocalTunnelAt(i, true);
						updateAdvancedTunnelLists();
					    }
					}
				    });

      grid.setConstraints(localEdit, gridc);
      tunnelDialog.add(localEdit);

      gridc.gridy     = 1;

      remoteEdit = new TunnelEditor("Remote: ([/plug/]<rem-port>:<loc-host>:<loc-port>)",
				    new ActionListener() {
					public void actionPerformed(ActionEvent e) {
					    try {
						client.setProperty("remote" + client.remoteForwards.size(), remoteEdit.getText());
						updateAdvancedTunnelLists();
					    } catch (Exception e2) {
						remoteEdit.selectText();
					    }
					}
				    },
				    new ActionListener() {
					public void actionPerformed(ActionEvent e) {
					    int i = remoteEdit.getSelectedIndex();
					    if(remoteEdit.getItem(i).indexOf(SSHFtpTunnel.TUNNEL_NAME) != -1) {
					      return;
					    }
					    if(i != -1) {
						client.removeRemoteTunnelAt(i);
						updateAdvancedTunnelLists();
					    }
					}
				    });

      grid.setConstraints(remoteEdit, gridc);
      tunnelDialog.add(remoteEdit);

      Button b;
      b = new Button("Close Dialog");
      b.addActionListener(new CloseAction(tunnelDialog));
      gridc.gridy     = 2;
      gridc.anchor = GridBagConstraints.CENTER;
      gridc.fill = GridBagConstraints.NONE;
      grid.setConstraints(b, gridc);
      tunnelDialog.add(b);

      setBackgroundOfChildren(tunnelDialog);

      tunnelDialog.setResizable(false);
      tunnelDialog.pack();
    }

    updateAdvancedTunnelLists();
    placeDialog(tunnelDialog);
    tunnelDialog.setVisible(true);
  }

  void updateAdvancedTunnelLists() {
    String plugStr;
    int    i;
    localEdit.removeAll();
    remoteEdit.removeAll();
    for(i = 0; i < client.localForwards.size(); i++) {
      SSHClient.LocalForward fwd = (SSHClient.LocalForward) client.localForwards.elementAt(i);
      plugStr = (fwd.plugin.equals("general") ? "" : "/" + fwd.plugin + "/");
      localEdit.addToList(plugStr + fwd.localHost + ":" + fwd.localPort + ":" +
			  fwd.remoteHost + ":" + fwd.remotePort);
    }

    for(i = 0; i < client.remoteForwards.size(); i++) {
      SSHClient.RemoteForward fwd = (SSHClient.RemoteForward) client.remoteForwards.elementAt(i);
      plugStr = (fwd.plugin.equals("general") ? "" : "/" + fwd.plugin + "/");
      remoteEdit.addToList(plugStr + fwd.remotePort + ":" + fwd.localHost + ":" + fwd.localPort);
    }
  }

  Dialog  connectDialog = null;
  List    hostList;
  boolean wantToRunSettingsDialog = false;
  public final void connectDialog() {
    if(connectDialog == null) {
      connectDialog = new Dialog(parent, "MindTerm - Connect", true);

      GridBagLayout       grid  = new GridBagLayout();
      GridBagConstraints  gridc = new GridBagConstraints();
      Label               label;
      Button              b;
      ActionListener      al;

      connectDialog.setLayout(grid);

      gridc.fill      = GridBagConstraints.NONE;
      gridc.anchor    = GridBagConstraints.WEST;
      gridc.gridy     = 0;
      gridc.insets    = new Insets(8, 8, 0, 8);

      label = new Label("Available hosts/settings:");
      grid.setConstraints(label, gridc);
      connectDialog.add(label);

      gridc.gridy     = 1;
      label = new Label("(dir: " + client.getSSHHomeDir() + ")");
      grid.setConstraints(label, gridc);
      connectDialog.add(label);

      gridc.fill      = GridBagConstraints.BOTH;
      gridc.anchor    = GridBagConstraints.WEST;
      gridc.insets    = new Insets(8, 8, 8, 8);
      gridc.gridy     = 2;

      hostList = new List(8);
      grid.setConstraints(hostList, gridc);
      connectDialog.add(hostList);

      hostList.addActionListener(al = new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  String host = hostList.getSelectedItem();
	  client.quiet = true;
	  try {
	  client.setPropertyFileAndLoad(host, true);
	  } catch (Throwable t) {
	    alertDialog("Error loading settings: " + t.getMessage());
	  }
	  client.sshStdIO.breakPromptLine("Connecting to: " + host);
	  connectDialog.setVisible(false);
	}
      });

      Panel bp = new Panel(new FlowLayout());
      bp.add(b = new Button("Connect"));
      b.addActionListener(al);
      bp.add(b = new Button("New Server"));
      b.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  connectDialog.setVisible(false);
	  try {
	    client.checkSave();
	  } catch (Throwable t) {
	    alertDialog("Error saving settings: " + t.getMessage());
	  }
	  client.clearServerSetting();
	  wantToRunSettingsDialog = true;
	  connectDialog.setVisible(false);
	}
      });

      bp.add(b = new Button("Cancel"));
      b.addActionListener(new CloseAction(connectDialog));


      gridc.gridy     = 3;
      gridc.gridwidth = GridBagConstraints.REMAINDER;
      gridc.anchor = GridBagConstraints.CENTER;
      grid.setConstraints(bp, gridc);
      connectDialog.add(bp);

      setBackgroundOfChildren(connectDialog);

      connectDialog.setResizable(false);
      connectDialog.pack();
    }
    hostList.removeAll();

    String[] l = client.availablePropertyHosts(client.getSSHHomeDir());
    if(l != null) {
      for(int i = 0; i < l.length; i++) {
	hostList.add(l[i]);
      }
    }
    hostList.select(0);
    connectDialog.pack();

    placeDialog(connectDialog);
    hostList.requestFocus();
    connectDialog.setVisible(true);

    if(wantToRunSettingsDialog) {
      wantToRunSettingsDialog = false;
      sshSettingsDialog();
    }

  }

  FileDialog loadFileDialog = null;
  public final void loadFileDialog() {
    if(loadFileDialog == null) {
      loadFileDialog = new FileDialog(parent, "MindTerm - Select file to load settings from", FileDialog.LOAD);
    }
    loadFileDialog.setDirectory(client.getSSHHomeDir());
    loadFileDialog.setVisible(true);

    String fileName = loadFileDialog.getFile();
    String dirName  = loadFileDialog.getDirectory();
    if(fileName != null && fileName.length() > 0) {
      try {
	client.setPropertyFileAndLoad(dirName + fileName, true);
	client.quiet = false;
      } catch (Throwable t) {
	alertDialog("Error loading settings: " + t.getMessage());
      }
      client.sshStdIO.breakPromptLine("Loaded new settings: " + fileName);
    }
  }

  FileDialog saveAsFileDialog = null;
  public final void saveAsFileDialog() {
    if(saveAsFileDialog == null) {
      saveAsFileDialog = new FileDialog(parent, "MindTerm - Select file to save settings to", FileDialog.SAVE);
    }
    saveAsFileDialog.setDirectory(client.getSSHHomeDir());
    saveAsFileDialog.setFile(client.getProperty("server") + client.PROPS_FILE_EXT);
    saveAsFileDialog.setVisible(true);

    String fileName = saveAsFileDialog.getFile();
    String dirName  = saveAsFileDialog.getDirectory();
    if(fileName != null && fileName.length() > 0) {
      client.setPropertyFile(dirName + fileName);
      client.propsChanged = true;

      try {
	client.saveCurrentProperties();
      } catch (Throwable t) {
	alertDialog("Error saving settings: " + t.getMessage());
      }
    }
  }

  FileDialog sendFileDialog = null;
  public final void sendFileDialog() {
    if(sendFileDialog == null) {
      sendFileDialog = new FileDialog(parent, "MindTerm - Select ASCII-file to send", FileDialog.LOAD);
    }
    sendFileDialog.setVisible(true);

    if(SSH.NETSCAPE_SECURITY_MODEL) {
      try {
	netscape.security.PrivilegeManager.enablePrivilege("UniversalFileAccess");
      } catch (netscape.security.ForbiddenTargetException e) {
	// !!!
      }
    }

    String fileName = sendFileDialog.getFile();
    String dirName  = sendFileDialog.getDirectory();
    if(fileName != null && fileName.length() > 0) {
      try {
	FileInputStream fileIn = new FileInputStream(dirName + fileName);
	byte[] bytes = new byte[fileIn.available()];
	fileIn.read(bytes);
	String fileStr = new String(bytes);
	client.stdinWriteString(fileStr);
      } catch (Throwable t) {
	alertDialog("Error sending file: " + t.getMessage());
      }
    }

  }

  SSHCaptureConsole captureConsole;
  FileDialog captureToFileDialog = null;
  public final boolean captureToFileDialog() {
    if(captureToFileDialog == null) {
      captureToFileDialog = new FileDialog(parent, "MindTerm - Select file to capture to", FileDialog.SAVE);
    }
    captureToFileDialog.setVisible(true);

    if(SSH.NETSCAPE_SECURITY_MODEL) {
      try {
	netscape.security.PrivilegeManager.enablePrivilege("UniversalFileAccess");
      } catch (netscape.security.ForbiddenTargetException e) {
	// !!!
      }
    }

    String fileName = captureToFileDialog.getFile();
    String dirName  = captureToFileDialog.getDirectory();
    if(fileName != null && fileName.length() > 0) {
      try {
	FileOutputStream fileOut = new FileOutputStream(dirName + fileName, true);
	captureConsole = new SSHCaptureConsole(client, fileOut);
	return true;
      } catch (Throwable t) {
	alertDialog("Error in capture: " + t.getMessage());
      }
    }
    return false;
  }

  public void endCapture() {
    if(captureConsole != null) {
      captureConsole.endCapture();
      captureConsole = null;
    }
  }

  final static String keyGenerationHelp =
"This will create your RSA identity to be used with the RSA " +
"authentication method. The identity consists of two parts, " +
"the public and private keys. You will only give one " +
"filename though, the private key is stored in this file, " +
"the public key is stored in a file with the same name and " +
"an extension of '.pub' added to it.\n" +
"\n" +
"The private key is protected by encryption if you give a " +
"password. If you leave the password empty the key will not. " +
"be encrypted, this should only be used in protected " +
"environments when unattended logins are desired. The " +
"'comment' you give is stored with your key, it is displayed " +
"every time you are prompted for the key's password.\n" +
"\n" +
"The key is generated using a random-generator which is " +
"seeded with the mouse-movements you make in this text " +
"area. Please move the mouse around here until the " + 
"progress bar reaches 100%.\n" +
"\n" +
"To use the key you must transfer the file with the public " +
"key to the ssh-server and add the contents of it to the " +
"file 'authorized_keys' in your own ssh-directory " +
"(e.g. ~/.ssh) on the server. For convenience the public " +
"key is put on the clipboard.";

  Dialog keyGenerationDialog;
  FileDialog keyGenFD;
  TextField bitsText;
  TextField fileText;
  TextField pwdText;
  TextField pwdText2;
  TextField commText;
  TextArea  descText;
  Checkbox  useCheck;
  ProgressBar progBar;
  Label     msgLbl;
  Button    okBut;
  int    randCnt = 0;
  int    dummy   = 0;
  byte[] randBytes     = new byte[512];
  public void keyGenerationDialog() {

    if(SSH.NETSCAPE_SECURITY_MODEL) {
      try {
	netscape.security.PrivilegeManager.enablePrivilege("UniversalFileAccess");
      } catch (netscape.security.ForbiddenTargetException e) {
	// !!!
      }
    }

    if(keyGenerationDialog == null) {
      keyGenerationDialog = new Dialog(parent, "MindTerm - RSA Key Generation", true);

      MouseMotionListener ml;
      GridBagLayout       grid  = new GridBagLayout();
      GridBagConstraints  gridc = new GridBagConstraints();
      Label               label;
      Button              b;

      keyGenerationDialog.setLayout(grid);

      gridc.fill      = GridBagConstraints.HORIZONTAL;
      gridc.anchor    = GridBagConstraints.WEST;
      gridc.gridwidth = 12;
      gridc.gridy     = 0;

      gridc.insets    = new Insets(8, 8, 8, 8);

      int rows = 18;
      Dimension sDim = Toolkit.getDefaultToolkit().getScreenSize();
      if(sDim.height < 600)
	  rows = 8;

      descText = new TextArea(keyGenerationHelp, rows, 48, TextArea.SCROLLBARS_VERTICAL_ONLY);
      descText.addMouseMotionListener(ml = new MouseMotionAdapter() {
	  public void mouseMoved(MouseEvent e) {
	    dummy++;
	    if(randCnt < 512) {
	      if((dummy % 2) == 0) {
		randBytes[randCnt++] = (byte)(((randCnt % 2) == 0) ? e.getX() : e.getY());
		progBar.setValue(randCnt);
	      }
	    } else {
	      okBut.setEnabled(true);
	    }
	  }
      });
      grid.setConstraints(descText, gridc);
      keyGenerationDialog.add(descText);
      descText.setEditable(false);

      gridc.gridy     = 1;
      gridc.fill      = GridBagConstraints.NONE;
      gridc.anchor    = GridBagConstraints.CENTER;

      progBar = new ProgressBar(512, 150, 20);
      grid.setConstraints(progBar, gridc);
      keyGenerationDialog.add(progBar);

      gridc.anchor    = GridBagConstraints.WEST;
      gridc.insets    = new Insets(4, 4, 0, 0);
      gridc.gridwidth = 4;

      gridc.gridy     = 2;
      label = new Label("Keylength (bits):");
      grid.setConstraints(label, gridc);
      keyGenerationDialog.add(label);
      bitsText = new TextField("", 5);
      grid.setConstraints(bitsText, gridc);
      keyGenerationDialog.add(bitsText);

      gridc.gridy     = 3;
      gridc.fill      = GridBagConstraints.HORIZONTAL;
      label = new Label("Identity file:");
      grid.setConstraints(label, gridc);
      keyGenerationDialog.add(label);
      fileText = new TextField("", 18);
      grid.setConstraints(fileText, gridc);
      keyGenerationDialog.add(fileText);
      b = new Button("...");
      b.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  if(keyGenFD == null) {
	    keyGenFD = new FileDialog(parent, "MindTerm - Select file to save identity to", FileDialog.SAVE);
	    keyGenFD.setDirectory(client.getSSHHomeDir());
	  }
	  keyGenFD.setVisible(true);
	  if(keyGenFD.getFile() != null && keyGenFD.getFile().length() > 0)
	    fileText.setText(keyGenFD.getDirectory() + keyGenFD.getFile());
	}
      });
      gridc.fill = GridBagConstraints.NONE;
      grid.setConstraints(b, gridc);
      keyGenerationDialog.add(b);

      gridc.gridy     = 4;
      gridc.fill      = GridBagConstraints.HORIZONTAL;
      label = new Label("Password:");
      grid.setConstraints(label, gridc);
      keyGenerationDialog.add(label);
      pwdText = new TextField("", 18);
      pwdText.setEchoChar('*');
      grid.setConstraints(pwdText, gridc);
      keyGenerationDialog.add(pwdText);

      gridc.gridy     = 5;
      label = new Label("Password again:");
      grid.setConstraints(label, gridc);
      keyGenerationDialog.add(label);
      pwdText2 = new TextField("", 18);
      pwdText2.setEchoChar('*');
      grid.setConstraints(pwdText2, gridc);
      keyGenerationDialog.add(pwdText2);

      gridc.gridy     = 6;
      label = new Label("Comment:");
      grid.setConstraints(label, gridc);
      keyGenerationDialog.add(label);
      commText = new TextField("", 18);
      grid.setConstraints(commText, gridc);
      keyGenerationDialog.add(commText);

      gridc.gridy     = 7; 
      gridc.fill      = GridBagConstraints.NONE;
      gridc.gridwidth = 2;
      gridc.insets    = new Insets(8, 4, 0, 0);
      useCheck = new Checkbox("Use key in current session");
      grid.setConstraints(useCheck, gridc);
      keyGenerationDialog.add(useCheck);

      gridc.gridy     = 8;
      gridc.fill      = GridBagConstraints.HORIZONTAL;
      gridc.insets    = new Insets(0, 0, 0, 0);
      gridc.gridwidth = 4;
      msgLbl = new Label("", Label.CENTER);
      gridc.fill  = GridBagConstraints.HORIZONTAL;
      gridc.gridwidth = GridBagConstraints.REMAINDER;
      gridc.anchor = GridBagConstraints.CENTER;
      grid.setConstraints(msgLbl, gridc);
      keyGenerationDialog.add(msgLbl);

      gridc.gridy     = 9;
      Panel bp = new Panel(new FlowLayout());
      bp.add(okBut = new Button("Generate"));
      okBut.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  int    bits;
	  String passwd   = pwdText.getText();
	  String fileName = fileText.getText();
	  String comment  = commText.getText();

	  try {
	    bits = Integer.valueOf(bitsText.getText()).intValue();
	    if(bits < 512 || bits > 2048) {
	      msgLbl.setText("Keylength must be in interval 512-2048");
	      return;
	    }
	  } catch (Exception ee) {
	    msgLbl.setText("Keylength must be an integer");
	    return;
	  }
	  if(!passwd.equals(pwdText2.getText())) {
	    msgLbl.setText("Please give same password twice");
	    return;
	  }
	  try {
	    if(fileName.length() == 0)
	      throw new Exception();
	    if(fileName.indexOf(File.separator) == -1)
	      fileName = client.getSSHHomeDir() + fileName;
	    FileOutputStream f = new FileOutputStream(fileName);
	    f.close();
	  } catch (Exception eee) {
	    msgLbl.setText("File can't be written to, please check");
	    return;
	  }
	  if(comment.length() == 0) {
	    msgLbl.setText("Please enter a non-empty comment text");
	    return;
	  }

	  // To make the label show up in win32/jview?!
	  //
	  keyGenerationDialog.remove(msgLbl);
	  msgLbl.setText("Please wait while generating key...");
	  keyGenerationDialog.add(msgLbl);
	  okBut.setEnabled(false);
	  try {
	    String pks = SSH.generateKeyFiles(SSH.generateRSAKeyPair(bits, new SecureRandom(randBytes)), fileName, passwd, comment);
	    client.sshStdIO.setSelection(pks);
	    client.sshStdIO.selectionAvailable(true);
	    msgLbl.setText("");
	    alertDialog("Key is now generated and saved to file and clipboard");
	    if(useCheck.getState())
	      client.setProperty("idfile", fileName);
	  } catch (IOException ee) {
	    alertDialog("Error saving identity: " + ee.getMessage());
	    msgLbl.setText("An error occured while saving identity");
	  }

	  pwdText.setText("");
	  pwdText2.setText("");
	  randCnt = 0;
	  dummy   = 0;
	  progBar.setValue(0);
	  setDefaultFileName();
	}
      });
      bp.add(b = new Button("Close"));
      b.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  randCnt = 0;
	  dummy   = 0;
	  keyGenerationDialog.setVisible(false);
	  pwdText.setText("");
	  pwdText2.setText("");
	}
      });

      gridc.gridwidth = GridBagConstraints.REMAINDER;
      gridc.anchor = GridBagConstraints.CENTER;
      grid.setConstraints(bp, gridc);
      keyGenerationDialog.add(bp);

      setBackgroundOfChildren(keyGenerationDialog);

      keyGenerationDialog.setResizable(false);
      keyGenerationDialog.pack();
    }
    useCheck.setState(true);
    msgLbl.setText("");
    okBut.setEnabled(false);
    bitsText.setText("1024");
    progBar.setValue(0);
    setDefaultFileName();

    placeDialog(keyGenerationDialog);
    bitsText.requestFocus();
    keyGenerationDialog.setVisible(true);
  }

  void setDefaultFileName() {
    try {
      String fn = client.getSSHHomeDir() + SSHInteractiveClient.DEF_IDFILE;
      File   f  = new File(fn);
      int    fi = 0;
      while(f.exists()) {
	fn = client.getSSHHomeDir() + SSHInteractiveClient.DEF_IDFILE + fi;
	f  = new File(fn);
	fi++;
      }
      fi--;
      fileText.setText(SSHInteractiveClient.DEF_IDFILE + (fi >= 0 ? String.valueOf(fi) : ""));
    } catch (Throwable t) {
      // !!!
      // Don't care...
    }
  }

  Dialog alertDialog = null;
  Label  alertLabel;
  Button okAlertBut;
  public final void alertDialog(String message) {

    if(alertDialog == null) {
      alertDialog = new Dialog(parent, "MindTerm - Alert", true);

      GridBagLayout       grid  = new GridBagLayout();
      GridBagConstraints  gridc = new GridBagConstraints();

      alertDialog.setLayout(grid);

      gridc.fill = GridBagConstraints.HORIZONTAL;
      gridc.gridwidth = GridBagConstraints.REMAINDER;
      gridc.anchor    = GridBagConstraints.CENTER;
      gridc.insets    = new Insets(8, 4, 4, 8);

      gridc.gridy = 0;
      alertLabel = new Label();
      grid.setConstraints(alertLabel, gridc);
      alertDialog.add(alertLabel);

      okAlertBut = new Button("OK");
      okAlertBut.addActionListener(new CloseAction(alertDialog));
      gridc.fill = GridBagConstraints.NONE;
      gridc.gridy = 1;
      grid.setConstraints(okAlertBut, gridc);
      alertDialog.add(okAlertBut);

      setBackgroundOfChildren(alertDialog);

      alertDialog.setResizable(false);
    }

    alertDialog.remove(alertLabel);
    alertLabel.setText(message);
    alertDialog.add(alertLabel);
    alertDialog.pack();

    placeDialog(alertDialog);
    okAlertBut.requestFocus();
    alertDialog.setVisible(true);
  }

  Dialog  confirmDialog = null;
  Label   confirmLabel;
  boolean confirmRet;
  Button  yesBut, noBut;
  public final boolean confirmDialog(String message, boolean defAnswer) {

    if(confirmDialog == null) {
      confirmDialog = new Dialog(parent, "MindTerm - Confirmation", true);

      GridBagLayout       grid  = new GridBagLayout();
      GridBagConstraints  gridc = new GridBagConstraints();
      ActionListener      al;
      confirmDialog.setLayout(grid);

      gridc.fill = GridBagConstraints.HORIZONTAL;
      gridc.gridwidth = GridBagConstraints.REMAINDER;
      gridc.anchor    = GridBagConstraints.CENTER;
      gridc.insets    = new Insets(8, 4, 4, 8);

      gridc.gridy = 0;
      confirmLabel = new Label();
      grid.setConstraints(confirmLabel, gridc);
      confirmDialog.add(confirmLabel);

      Panel bp = new Panel(new FlowLayout());

      bp.add(yesBut = new Button("Yes"));

      yesBut.addActionListener(al = new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  if(e.getActionCommand().equals("Yes"))
	    confirmRet = true;
	  else
	    confirmRet = false;
	  confirmDialog.setVisible(false);
	}
      });

      bp.add(noBut = new Button("No"));
      noBut.addActionListener(al);

      gridc.gridy = 1;
      gridc.gridwidth = GridBagConstraints.REMAINDER;
      grid.setConstraints(bp, gridc);
      confirmDialog.add(bp);

      setBackgroundOfChildren(confirmDialog);

      confirmDialog.setResizable(false);
    }

    confirmDialog.remove(confirmLabel);
    confirmLabel.setText(message);
    confirmDialog.add(confirmLabel);
    confirmDialog.pack();

    placeDialog(confirmDialog);

    if(defAnswer)
      yesBut.requestFocus();
    else
      noBut.requestFocus();

    confirmDialog.setVisible(true);

    return confirmRet;
  }

  public final void textDialog(String head, String text, int rows, int cols, boolean scrollbar) {
    Dialog   textDialog = null;
    TextArea textArea;
    Button   okTextBut;

    textDialog = new Dialog(parent, head, true);

    GridBagLayout       grid  = new GridBagLayout();
    GridBagConstraints  gridc = new GridBagConstraints();

    textDialog.setLayout(grid);

    gridc.fill      = GridBagConstraints.NONE;
    gridc.gridwidth = GridBagConstraints.REMAINDER;
    gridc.anchor    = GridBagConstraints.CENTER;
    gridc.insets    = new Insets(4, 4, 4, 4);

    textArea = new TextArea(text, rows, cols,
			    scrollbar ? TextArea.SCROLLBARS_VERTICAL_ONLY : TextArea.SCROLLBARS_NONE);
    grid.setConstraints(textArea, gridc);
    textDialog.add(textArea);
    textArea.setEditable(false);

    okTextBut = new Button("OK");
    okTextBut.addActionListener(new CloseAction(textDialog));
    gridc.fill = GridBagConstraints.NONE;
    grid.setConstraints(okTextBut, gridc);
    textDialog.add(okTextBut);

    setBackgroundOfChildren(textDialog);

    textDialog.setResizable(false);
    textDialog.pack();

    placeDialog(textDialog);
    okTextBut.requestFocus();
    textDialog.setVisible(true);
  }

  final static void setBackgroundOfChildren(Dialog dialog) {
    Component[] children = dialog.getComponents();
    dialog.setBackground(SystemColor.menu);
    for(int i = 0; i < children.length; i++) {
      if(children[i] instanceof Choice)
	continue;
      children[i].setBackground(SystemColor.menu);
    }
  }

}
