/*
 *  Sshtools - Application Framework
 *
 *  Copyright (C) 2002 Lee David Painter.
 *
 *  Written by: 2002 Lee David Painter <lee@sshtools.com>
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package com.sshtools.daemon.windows;

import com.sshtools.j2ssh.SshThread;

import com.sshtools.j2ssh.configuration.ConfigurationLoader;
import com.sshtools.j2ssh.configuration.PlatformConfiguration;

import com.sshtools.j2ssh.platform.NativeAuthenticationProvider;

import org.apache.log4j.Logger;

import java.io.IOException;


public class WindowsAuthentication extends NativeAuthenticationProvider {
    private static Logger log = Logger.getLogger(WindowsAuthentication.class);

    static {
        System.loadLibrary("sshtools-daemon-win32");
    }

    public WindowsAuthentication() {
    }

    public String getHomeDirectory(String username) throws IOException {

      String dir = null;

      if(Thread.currentThread() instanceof SshThread) {
        dir = (String)SshThread.getCurrentThread().getProperty("user.home");
      }

      if(dir==null) {
        // To get the user home directory under windows we will first look
        // to see if they have one set in the control panel
        dir = getNativeHomeDirectory(getDomain(), username);

        if (dir == null) {
          // The user does not have one set so lets look for the profile directory,
          // this is the nasty bit as we can only get it with a logon token, so if
          // if the user is not logged in, were going to have to create a token
          // to get it.
          if (Thread.currentThread()instanceof SshThread) {
            SshThread thread = SshThread.getCurrentThread();

            int handle;

            if (!thread.containsProperty("Win32UserAuthInfo")) {
              handle = createToken(username, getDomain());
              dir = getNativeProfileDirectory(handle);
              closeHandle(handle);
            }
            else {
              WindowsAuthenticationInfo info = (WindowsAuthenticationInfo) thread.
                  getProperty("Win32UserAuthInfo");
              dir = getNativeProfileDirectory(info.getLogonToken());
            }

            if (dir == null)
              dir = ConfigurationLoader.getPlatformConfiguration().getSetting(
                  "DefaultHomeDir");
          }
        }

        if (Thread.currentThread()instanceof SshThread) {
          SshThread thread = SshThread.getCurrentThread();
          thread.setProperty("user.home", dir);
        }
      }

      return dir;
    }

    public void logoffUser() throws IOException {


      SshThread thread = SshThread.getCurrentThread();

      log.info("Logging off user " + thread.getUsername()
               + " from session "
               + String.valueOf(thread.getSessionIdString()));

      nativeLogoffUser((WindowsAuthenticationInfo)
                      thread.getProperty("Win32UserAuthInfo"));


    }

    public boolean logonUser(String username, String password)
        throws IOException {
        WindowsAuthenticationInfo tokens = nativeLogonUser(username, password, getDomain());



        if (tokens != null) {

            log.debug("Authenticated handle is " + String.valueOf(tokens.getLogonToken()));

            if (!(Thread.currentThread() instanceof SshThread)) {
                log.error(
                    "Calling process is not an instance of SshThread, cannot set token handle");

                return false;
            }


            ((SshThread) Thread.currentThread()).setProperty("Win32UserAuthInfo",
                tokens);


            return true;
        } else {
            log.error("nativeLogonUser returned a null handle!");

            return false;
        }
    }

    public boolean changePassword(String username, String oldpassword,
        String newpassword) {
        return false;
    }


    public boolean logonUser(String username) throws IOException {
      WindowsAuthenticationInfo tokens = nativeLogonUserWOPassword(username, getDomain());


              if (tokens != null) {

                  log.debug("Authenticated handle is " + String.valueOf(tokens.getLogonToken()));

                  if (!(Thread.currentThread() instanceof SshThread)) {
                      log.error(
                          "Calling process is not an instance of SshThread, cannot set token handle");

                      return false;
                  }


                  ((SshThread) Thread.currentThread()).setProperty("Win32UserAuthInfo",
                      tokens);

                  return true;
              } else {
                  log.error("nativeLogonUser returned a null handle!");

                  return false;
              }

    }

    protected String getDomain() throws IOException {
        String domain = "."; // Local only by default
        PlatformConfiguration platform = ConfigurationLoader
            .getPlatformConfiguration();

        if (platform.containsSetting("AuthenticateOnDomain")) {
            domain = platform.getSetting("AuthenticateOnDomain");

            if (domain.trim().length() == 0) {
                domain = ".";
            }
        }

        return domain;
    }

    private native String getNativeHomeDirectory(String domain, String username);

    public native String getNativeProfileDirectory(int handle);

    private native WindowsAuthenticationInfo nativeLogonUser(String username, String password,
        String domain);

    private native WindowsAuthenticationInfo nativeLogonUserWOPassword(String username,
        String domain);

    private native void nativeLogoffUser(WindowsAuthenticationInfo info);

    private native int createToken(String username, String domain);

    private native void closeHandle(int handle);


}
