 /*
 *  Sshtools - Java SSH2 API
 *
 *  Copyright (C) 2002 Lee David Painter.
 *
 *  Written by: 2002 Lee David Painter <lee@sshtools.com>
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package com.sshtools.j2ssh;

import com.sshtools.j2ssh.authentication.AuthenticationProtocolClient;
import com.sshtools.j2ssh.authentication.AuthenticationProtocolState;
import com.sshtools.j2ssh.authentication.PublicKeyAuthenticationClient;
import com.sshtools.j2ssh.authentication.SshAuthenticationClient;

import com.sshtools.j2ssh.configuration.SshConnectionProperties;

import com.sshtools.j2ssh.connection.Channel;
import com.sshtools.j2ssh.connection.ChannelEventListener;
import com.sshtools.j2ssh.connection.ConnectionProtocol;

import com.sshtools.j2ssh.forwarding.ForwardingClient;

import com.sshtools.j2ssh.session.SessionChannelClient;

import com.sshtools.j2ssh.connection.ChannelFactory;

import com.sshtools.j2ssh.transport.ConsoleHostKeyVerification;
import com.sshtools.j2ssh.transport.HostKeyVerification;
import com.sshtools.j2ssh.transport.ServiceOperationException;
import com.sshtools.j2ssh.transport.TransportProtocolClient;
import com.sshtools.j2ssh.transport.TransportProtocolEventHandler;
import com.sshtools.j2ssh.transport.TransportProtocolState;

import com.sshtools.j2ssh.transport.publickey.SshPublicKey;

import com.sshtools.j2ssh.sftp.SftpSubsystemClient;

import com.sshtools.j2ssh.util.State;

import org.apache.log4j.Logger;

import java.io.IOException;

import java.net.InetSocketAddress;
import java.net.Socket;

import java.util.List;
import java.util.Properties;
import java.util.Vector;
import java.util.Iterator;

public class SshClient {
    private static Logger log = Logger.getLogger(SshClient.class);
    protected AuthenticationProtocolClient authentication;
    protected ConnectionProtocol connection;
    protected ForwardingClient forwarding;
    protected List authMethods;
    protected Socket socket;
    protected String authenticationBanner;
    protected TransportProtocolClient transport;
    protected int authenticationState = AuthenticationProtocolState.READY;
    protected int connectTimeout = 10000;
    protected int socketTimeout = 0;
    protected TransportProtocolEventHandler eventHandler = null;
    protected Vector activeChannels = new Vector();
    protected ActiveChannelEventListener eventListener = new ActiveChannelEventListener();
    protected boolean useDefaultForwarding = true;

    public SshClient() {
    }

    public String getAuthenticationBanner(int timeout) {
        if (authentication == null) {
            return "";
        } else {
            return authentication.getBannerMessage(timeout);
        }
    }

    public List getAvailableAuthMethods(String username)
        throws IOException {
        if (authentication != null) {
            return authentication.getAvailableAuths(username,
                connection.getServiceName());
        } else {
            return null;
        }
    }

    public boolean isConnected() {
        State state = (transport == null) ? null : transport.getState();
        int value = (state == null) ? TransportProtocolState.DISCONNECTED
                                    : state.getValue();

        return ((value == TransportProtocolState.CONNECTED)
        || (value == TransportProtocolState.PERFORMING_KEYEXCHANGE));
    }

    public String getServerId() {
        return transport.getRemoteId();
    }

    public SshPublicKey getServerHostKey() {
        return transport.getServerHostKey();
    }

    public TransportProtocolState getConnectionState() {
        return transport.getState();
    }

    public ForwardingClient getForwardingClient() {
        return forwarding;
    }

    public int getRemoteEOL() {
        return transport.getRemoteEOL();
    }

    public void setEventHandler(TransportProtocolEventHandler eventHandler) {
        // If were connected then add, otherwise store for later connection
        if(transport!=null)
          transport.addEventHandler(eventHandler);
        else
          this.eventHandler = eventHandler;
    }

    public void setConnectTimeout(int milliseconds) {
        this.connectTimeout = milliseconds;
    }

    public void setSocketTimeout(int milliseconds) {
        this.socketTimeout = milliseconds;
    }

    public String getRemoteEOLString() {
        return ((transport.getRemoteEOL() == TransportProtocolClient.EOL_CRLF)
        ? "\r\n" : "\n");
    }

    public int authenticate(SshAuthenticationClient auth)
        throws IOException {
        // Do the authentication
        authenticationState = authentication.authenticate(auth, connection);

        return authenticationState;
    }

    public boolean acceptsKey(String username, SshPublicKey key) throws IOException {
      if(authenticationState!=AuthenticationProtocolState.COMPLETE) {
        PublicKeyAuthenticationClient pk = new PublicKeyAuthenticationClient();

        return pk.acceptsKey(authentication,
                             username,
                             connection.getServiceName(),
                             key);
      } else
        throw new SshException("Authentication has been completed!");

    }

    public void connect(String hostname) throws IOException {
        connect(hostname, 22, new ConsoleHostKeyVerification());
    }

    public void connect(String hostname, HostKeyVerification hosts)
        throws IOException {
        connect(hostname, 22, hosts);
    }

    public void connect(String hostname, int port) throws IOException {
        connect(hostname, port, new ConsoleHostKeyVerification());
    }

    public void connect(String hostname, int port, HostKeyVerification hosts)
        throws IOException {
        SshConnectionProperties properties = new SshConnectionProperties();
        properties.setHost(hostname);
        properties.setPort(port);

        connect(properties, hosts);
    }

    public void connect(SshConnectionProperties properties)
        throws IOException {
        connect(properties, new ConsoleHostKeyVerification());
    }

    public void connect(SshConnectionProperties properties,
        HostKeyVerification hostVerification) throws IOException {
        socket = new Socket();
        socket.setSoTimeout(socketTimeout);
        socket.connect(new InetSocketAddress(properties.getHost(),
                properties.getPort()), connectTimeout);

        // Start the transport protocol
        transport = new TransportProtocolClient(hostVerification);
        transport.addEventHandler(eventHandler);
        transport.startTransportProtocol(socket, properties);

        // Start the authentication protocol
        authentication = new AuthenticationProtocolClient();
        transport.requestService(authentication);

        connection = new ConnectionProtocol();

        if(useDefaultForwarding)
          forwarding = new ForwardingClient(connection);

    }

    public SshConnectionProperties getProperties() {
      return transport.getProperties();
    }

    public void setKexTimeout(long seconds) throws IOException {
        transport.setKexTimeout(seconds);
    }

    public void setKexTransferLimit(long kilobytes) throws IOException {
        transport.setKexTransferLimit(kilobytes);
    }

    public void setSendIgnore(boolean sendIgnore) {
        transport.setSendIgnore(sendIgnore);
    }

    public void setUseDefaultForwarding(boolean useDefaultForwarding) {
      this.useDefaultForwarding = useDefaultForwarding;
    }

    public void disconnect() {
        if (connection != null) {
            connection.stop();
        }

        if (transport != null) {
            transport.disconnect("Terminating connection");
        }
    }

    public long getOutgoingByteCount() {
        return transport.getOutgoingByteCount();
    }

    public long getIncomingByteCount() {
        return transport.getIncomingByteCount();
    }

    public int getActiveChannelCount() {
        synchronized (activeChannels) {
            return activeChannels.size();
        }
    }

    public List getActiveChannels() {
        synchronized (activeChannels) {
            return (List) activeChannels.clone();
        }
    }

    public boolean hasActiveSession(String type) {
      Iterator it = activeChannels.iterator();
      Object obj;
      while(it.hasNext()) {
        obj = it.next();
        if(obj instanceof SessionChannelClient) {
          if(((SessionChannelClient)obj).getSessionType().equals(type))
            return true;
        }
      }

      return false;
    }

    public SessionChannelClient getActiveSession(String type) throws IOException {
      Iterator it = activeChannels.iterator();
      Object obj;
      while(it.hasNext()) {
        obj = it.next();
        if(obj instanceof SessionChannelClient) {
          if(((SessionChannelClient)obj).getSessionType().equals(type))
            return (SessionChannelClient) obj;
        }
      }

      throw new IOException("There are no active " + type + " sessions");

    }


    public SessionChannelClient openSessionChannel() throws IOException {
        if (authenticationState != AuthenticationProtocolState.COMPLETE) {
            throw new SshException("Authentication has not been completed!");
        }

        SessionChannelClient session = new SessionChannelClient();

        if (!connection.openChannel(session, eventListener)) {
            throw new SshException("The server refused to open a session");
        }

        return session;
    }

    public SftpSubsystemClient openSftpChannel() throws IOException {
      SessionChannelClient session = openSessionChannel();
      SftpSubsystemClient sftp = new SftpSubsystemClient();
      if(!session.startSubsystem(sftp))
        throw new SshException("The SFTP subsystem failed to start");

      return sftp;
    }

    public boolean openChannel(Channel channel) throws IOException {
      if (authenticationState != AuthenticationProtocolState.COMPLETE) {
            throw new SshException("Authentication has not been completed!");
        }

        // Open the channel providing our channel listener so we can track
        return connection.openChannel(channel, eventListener);
    }

    public void allowChannelOpen(String channelName, ChannelFactory cf) throws IOException {
      connection.addChannelFactory(channelName, cf);
    }

    public boolean sendGlobalRequest(String requestName,
                                     boolean wantReply,
                                     byte[] requestData)
        throws IOException {
      return connection.sendGlobalRequest(requestName, wantReply, requestData);
    }


    class ActiveChannelEventListener implements ChannelEventListener {
        public void onChannelOpen(Channel channel) {
            synchronized (activeChannels) {
                activeChannels.add(channel);
            }
        }

        public void onChannelEOF(Channel channel) {

        }

        public void onChannelClose(Channel channel) {
            synchronized (activeChannels) {
                activeChannels.remove(channel);
            }
        }

        public void onDataReceived(Channel channel, int numBytes) {

        }

        public void onDataSent(Channel channel, int numBytes) {

        }
    }
}
